/***************************************************************************

    chdcodec.h

    Codecs used by the CHD format

****************************************************************************

    Copyright Aaron Giles
    All rights reserved.

    Redistribution and use in source and binary forms, with or without
    modification, are permitted provided that the following conditions are
    met:

        * Redistributions of source code must retain the above copyright
          notice, this list of conditions and the following disclaimer.
        * Redistributions in binary form must reproduce the above copyright
          notice, this list of conditions and the following disclaimer in
          the documentation and/or other materials provided with the
          distribution.
        * Neither the name 'MAME' nor the names of its contributors may be
          used to endorse or promote products derived from this software
          without specific prior written permission.

    THIS SOFTWARE IS PROVIDED BY AARON GILES ''AS IS'' AND ANY EXPRESS OR
    IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
    WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
    DISCLAIMED. IN NO EVENT SHALL AARON GILES BE LIABLE FOR ANY DIRECT,
    INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
    (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
    SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
    HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
    STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING
    IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
    POSSIBILITY OF SUCH DAMAGE.

***************************************************************************/

#pragma once

#ifndef __CHDCODEC_H__
#define __CHDCODEC_H__

#include "osdcore.h"


#define CHDCODEC_VERIFY_COMPRESSION 0


//**************************************************************************
//  MACROS
//**************************************************************************

#define CHD_MAKE_TAG(a,b,c,d)		(((a) << 24) | ((b) << 16) | ((c) << 8) | (d))



//**************************************************************************
//  TYPE DEFINITIONS
//**************************************************************************

// forward references
class chd_file;

// base types
typedef UINT32 chd_codec_type;


// ======================> chd_codec

// common base class for all compressors and decompressors
class chd_codec
{
protected:
	// can't create these directly
	chd_codec(chd_file &file, bool lossy);

public:
	// allow public deletion
	virtual ~chd_codec();

	// accessors
	chd_file &chd() const { return m_chd; }
	bool lossy() const { return m_lossy; }

	// implementation
	virtual void configure(int param, void *config);

private:
	// internal state
	chd_file &			m_chd;
	bool				m_lossy;
};


// ======================> chd_compressor

// base class for all compressors
class chd_compressor : public chd_codec
{
protected:
	// can't create these directly
	chd_compressor(chd_file &file, bool lossy);

public:
	// implementation
	virtual UINT32 compress(const UINT8 *src, UINT32 srclen, UINT8 *dest) = 0;
};


// ======================> chd_decompressor

// base class for all decompressors
class chd_decompressor : public chd_codec
{
protected:
	// can't create these directly
	chd_decompressor(chd_file &file, bool lossy);

public:
	// implementation
	virtual void decompress(const UINT8 *src, UINT32 complen, UINT8 *dest, UINT32 destlen) = 0;
};


// ======================> chd_codec_list

// wrapper to get at the list of codecs
class chd_codec_list
{
public:
	// create compressors or decompressors
	static chd_compressor *new_compressor(chd_codec_type type, chd_file &file);
	static chd_decompressor *new_decompressor(chd_codec_type type, chd_file &file);

	// utilities
	static bool codec_exists(chd_codec_type type) { return (find_in_list(type) != NULL); }
	static const char *codec_name(chd_codec_type type);

private:
	// an entry in the list
	struct codec_entry
	{
		chd_codec_type		m_type;
		bool				m_lossy;
		const char *		m_name;
		chd_compressor *	(*m_construct_compressor)(chd_file &, bool);
		chd_decompressor *	(*m_construct_decompressor)(chd_file &, bool);
	};

	// internal helper functions
	static const codec_entry *find_in_list(chd_codec_type type);

	template<class _CompressorClass>
	static chd_compressor *construct_compressor(chd_file &chd, bool lossy) { return new _CompressorClass(chd, lossy); }

	template<class _DecompressorClass>
	static chd_decompressor *construct_decompressor(chd_file &chd, bool lossy) { return new _DecompressorClass(chd, lossy); }

	// the static list
	static const codec_entry s_codec_list[];
};


// ======================> chd_compressor_group

// helper class that wraps several compressors
class chd_compressor_group
{
public:
	// construction/destruction
	chd_compressor_group(chd_file &file, chd_codec_type compressor_list[4]);
	~chd_compressor_group();

	// find the best compressor
	INT8 find_best_compressor(const UINT8 *src, UINT8 *compressed, UINT32 &complen);

private:
	// internal state
	UINT32					m_hunkbytes;		// number of bytes in a hunk
	chd_compressor *		m_compressor[4];	// array of active codecs
	dynamic_buffer			m_compress_test;	// test buffer for compression
#if CHDCODEC_VERIFY_COMPRESSION
	chd_decompressor *		m_decompressor[4];	// array of active codecs
	dynamic_buffer			m_decompressed;		// verification buffer
#endif
};



//**************************************************************************
//  CONSTANTS
//**************************************************************************

// currently-defined codecs
const chd_codec_type CHD_CODEC_NONE 	= 0;
const chd_codec_type CHD_CODEC_ZLIB 	= CHD_MAKE_TAG('z','l','i','b');
const chd_codec_type CHD_CODEC_LZMA 	= CHD_MAKE_TAG('l','z','m','a');
const chd_codec_type CHD_CODEC_HUFFMAN	= CHD_MAKE_TAG('h','u','f','f');
const chd_codec_type CHD_CODEC_FLAC		= CHD_MAKE_TAG('f','l','a','c');
const chd_codec_type CHD_CODEC_CD_FLAC	= CHD_MAKE_TAG('c','d','f','l');
const chd_codec_type CHD_CODEC_AVHUFF	= CHD_MAKE_TAG('a','v','h','u');

// A/V codec configuration parameters
enum
{
	AVHUFF_CODEC_DECOMPRESS_CONFIG = 1
};


#endif // __CHDCODEC_H__
