/*
 * Copyright 2013 Yuichiro Moriguchi
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package net.morilib.awk.expr;

import java.math.BigInteger;

import net.morilib.awk.AwkArithmeticException;
import net.morilib.awk.io.AwkFiles;
import net.morilib.awk.namespace.AwkNamespace;
import net.morilib.awk.value.AwkFloat;
import net.morilib.awk.value.AwkInteger;
import net.morilib.awk.value.AwkUndefined;
import net.morilib.awk.value.AwkValue;

public class AwkRemainderOp extends AwkExpression {

	private AwkExpression e1, e2;

	public AwkRemainderOp(AwkExpression e1, AwkExpression e2) {
		this.e1 = e1;
		this.e2 = e2;
	}

	public AwkValue eval(AwkNamespace ns, AwkFiles o) {
		AwkValue r1 = e1.eval(ns, o), r2 = e2.eval(ns, o);
		BigInteger e, f;

		if(r1 instanceof AwkUndefined) {
			return AwkInteger.ZERO;
		} else if(r2 instanceof AwkUndefined) {
			throw new ArithmeticException();
		} else if(r1.isFloat() || r2.isFloat()) {
			return AwkFloat.valueOf(
					Math.IEEEremainder(r1.toFloat(), r2.toFloat()));
		} else if(r1.isIntegerValue() && r2.isIntegerValue()) {
			try {
				e = r1.toInteger();
				f = r2.toInteger();
				return AwkInteger.valueOf(e.remainder(f));
			} catch(ArithmeticException z) {
				throw new AwkArithmeticException(z.getMessage());
			}
		} else {
			return AwkFloat.valueOf(
					Math.IEEEremainder(r1.toFloat(), r2.toFloat()));
		}
	}

}
