/*
 * Copyright 2013 Yuichiro Moriguchi
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package net.morilib.awk.namespace;

import java.util.HashMap;
import java.util.Map;

import net.morilib.awk.value.AwkFunction;
import net.morilib.awk.value.AwkValue;

/**
 * 単純な名前空間です。
 *
 * @author MORIGUCHI, Yuichiro 2013/03/08
 */
public class AwkSimpleNamespace implements AwkNamespace {

	/**
	 * 親の名前空間です。
	 */
	protected AwkNamespace root;

	private Map<String, AwkValue> values;
	private Map<String, AwkFunction> functions;
	private Map<String, AwkNamespace> namespaces;

	//
	AwkSimpleNamespace() {
		values = new HashMap<String, AwkValue>();
		functions = new HashMap<String, AwkFunction>();
		namespaces = new HashMap<String, AwkNamespace>();
		this.root = null;
	}

	//
	private AwkSimpleNamespace(
			Map<String, AwkValue> values,
			Map<String, AwkFunction> functions,
			Map<String, AwkNamespace> namespaces,
			AwkNamespace root) {
		this.values = values;
		this.functions = functions;
		this.namespaces = namespaces;
		this.root = root;
	}

	/**
	 * 名前空間を生成します。
	 * 
	 * @param root 親の名前空間
	 */
	public AwkSimpleNamespace(AwkNamespace root) {
		this();
		this.root = root;
	}

	/* (non-Javadoc)
	 * @see net.morilib.awk.namespace.AwkNamespace#assign(java.lang.String, net.morilib.awk.value.AwkValue)
	 */
	public void assign(String s, AwkValue v) {
		values.put(s, v);
	}

	/* (non-Javadoc)
	 * @see net.morilib.awk.namespace.AwkNamespace#assignArgument(java.lang.String, net.morilib.awk.value.AwkValue)
	 */
	public void assignArgument(String s, AwkValue v) {
		values.put(s, v);
	}

	/* (non-Javadoc)
	 * @see net.morilib.awk.namespace.AwkNamespace#find(java.lang.String)
	 */
	public AwkValue find(String s) {
		return values.get(s);
	}

	/* (non-Javadoc)
	 * @see net.morilib.awk.namespace.AwkNamespace#findFunction(java.lang.String)
	 */
	public AwkFunction findFunction(String s) {
		return functions.get(s);
	}

	/* (non-Javadoc)
	 * @see net.morilib.awk.namespace.AwkNamespace#bindFunction(java.lang.String, net.morilib.awk.value.AwkFunction)
	 */
	public void bindFunction(String s, AwkFunction v) {
		functions.put(s, v);
	}

	/* (non-Javadoc)
	 * @see net.morilib.awk.namespace.AwkNamespace#isReadonly(java.lang.String)
	 */
	public boolean isReadonly(String s) {
		return false;
	}

	/* (non-Javadoc)
	 * @see net.morilib.awk.namespace.AwkNamespace#subnamespace(java.lang.String)
	 */
	public synchronized AwkNamespace subnamespace(String s) {
		AwkNamespace ns;

		if((ns = namespaces.get(s)) == null) {
			ns = new AwkSimpleNamespace(this);
			namespaces.put(s, ns);
		}
		return ns;
	}

	/* (non-Javadoc)
	 * @see net.morilib.awk.namespace.AwkNamespace#referNamespace(java.lang.String)
	 */
	public AwkNamespace referNamespace(String s) {
		return namespaces.get(s);
	}

	/* (non-Javadoc)
	 * @see net.morilib.awk.namespace.AwkNamespace#referField(int)
	 */
	public AwkValue referField(int n) {
		return root.referField(n);
	}

	/* (non-Javadoc)
	 * @see net.morilib.awk.namespace.AwkNamespace#sizeOfField()
	 */
	public int sizeOfField() {
		return root.sizeOfField();
	}

	/* (non-Javadoc)
	 * @see net.morilib.awk.namespace.AwkNamespace#setField(java.lang.String)
	 */
	public void setField(String s) {
		root.setField(s);
	}

	/* (non-Javadoc)
	 * @see net.morilib.awk.namespace.AwkNamespace#setField(int, java.lang.String)
	 */
	public void setField(int n, String s) {
		root.setField(n, s);
	}

	/* (non-Javadoc)
	 * @see net.morilib.awk.namespace.AwkNamespace#getFS()
	 */
	public String getFS() {
		return root.getFS();
	}

	/* (non-Javadoc)
	 * @see net.morilib.awk.namespace.AwkNamespace#getRS()
	 */
	public String getRS() {
		return root.getRS();
	}

	/* (non-Javadoc)
	 * @see net.morilib.awk.namespace.AwkNamespace#getOFS()
	 */
	public String getOFS() {
		return root.getOFS();
	}

	/* (non-Javadoc)
	 * @see net.morilib.awk.namespace.AwkNamespace#getORS()
	 */
	public String getORS() {
		return root.getORS();
	}

	/* (non-Javadoc)
	 * @see net.morilib.awk.namespace.AwkNamespace#getRoot()
	 */
	public AwkNamespace getRoot() {
		return root.getRoot();
	}

	/* (non-Javadoc)
	 * @see net.morilib.awk.namespace.AwkNamespace#isRoot()
	 */
	public boolean isRoot() {
		return false;
	}

	/* (non-Javadoc)
	 * @see net.morilib.awk.namespace.AwkNamespace#newInstance()
	 */
	public AwkNamespace newInstance() {
		return new AwkSimpleNamespace(
				new HashMap<String, AwkValue>(values),
				functions,
				namespaces,
				root);
	}

}
