/*
 * Copyright 2013 Yuichiro Moriguchi
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package net.morilib.awk.stat.dist.continuous;

import net.morilib.awk.stat.dist.AbstractContinuousDistribution;

/**
 * ロジスティック分布です。
 *
 * @author MORIGUCHI, Yuichiro 2013/04/14
 */
public class LogisticDistribution
extends AbstractContinuousDistribution {

	private double mu, s;

	/**
	 * 確率分布を生成します。
	 * 
	 * @param mu 位置
	 * @param s  尺度
	 */
	public LogisticDistribution(double mu, double s) {
		if(s <= 0)  throw new IllegalArgumentException();
		this.mu = mu;
		this.s  = s;
	}

	/**
	 * 尺度sを得ます。
	 * 
	 * @return s
	 */
	public double getScale() {
		return s;
	}

	/* (non-Javadoc)
	 * @see net.morilib.awk.stat.dist.AbstractContinuousDistribution#invCdf(double)
	 */
	public double invCdf(double x) {
		if(x < 0.0 || x > 1.0)  return Double.NaN;
		if(x == 0.0)  return 0.0;
		if(x == 1.0)  return 1.0;
		return mu - s * Math.log(1.0 / x - 1.0);
	}

	public boolean isInSupport(double n) {
		return true;
	}

	public double supportInfimum() {
		return Double.NEGATIVE_INFINITY;
	}

	public double supportSupremum() {
		return Double.POSITIVE_INFINITY;
	}

	public double cdf(double x) {
		return 1.0 / (1.0 + Math.exp(-(x - mu) / s));
	}

	public double expectedValue() {
		return mu;
	}

	public double kurtosis() {
		return 1.2;
	}

	public double mode() {
		return mu;
	}

	public double skewness() {
		return 0;
	}

	public double variance() {
		return (Math.PI * s) * (Math.PI * s) / 3.0;
	}

	public double f(double x) {
		double k, m;

		k = Math.exp(-(x - mu) / s);
		m = Math.exp( (x - mu) / s);
		return 1.0 / (2.0 + k + m) / s;
	}

}
