/*
 * Copyright 2009-2010 Yuichiro Moriguchi
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package net.morilib.awk.stat.dist;

import net.morilib.awk.misc.Hashes;
import net.morilib.awk.stat.DistributionUtils;
import net.morilib.awk.stat.special.Erf;

/**
 * 対数正規分布です。
 * 
 *
 * @author MORIGUCHI, Yuichiro 2013/04/20
 */
public class LogNormalDistribution
extends AbstractContinuousDistribution {

	//
	private double mu, sigma2;

	/**
	 * 確率分布を生成します。
	 * 
	 * @param mu     μ
	 * @param sigma2 σ^2
	 */
	public LogNormalDistribution(double mu, double sigma2) {
		if(sigma2 <= 0)  throw new IllegalArgumentException();
		this.mu = mu;
		this.sigma2 = sigma2;
	}

	/**
	 * μを得ます。
	 * 
	 * @return μ
	 */
	public double getMu() {
		return mu;
	}

	/**
	 * σ<sup>2</sup>を得ます。
	 * 
	 * @return σ^2
	 */
	public double getSigma2() {
		return sigma2;
	}

	/* (non-Javadoc)
	 * @see net.morilib.math.stat.dist.ProbabilityDistribution#expectedValue()
	 */
	public double expectedValue() {
		return Math.exp(mu + sigma2 / 2);
	}

	/* (non-Javadoc)
	 * @see net.morilib.math.stat.dist.ProbabilityDistribution#variance()
	 */
	public double variance() {
		return Math.exp(2 * mu + sigma2) * (Math.exp(sigma2) - 1);
	}

	/* (non-Javadoc)
	 * @see net.morilib.math.stat.dist.ProbabilityDistribution#mode()
	 */
	public double mode() {
		return Math.exp(mu - sigma2);
	}

	/* (non-Javadoc)
	 * @see net.morilib.math.stat.dist.ProbabilityDistribution#skewness()
	 */
	public double skewness() {
		return (Math.sqrt(Math.exp(sigma2) - 1) *
				(Math.exp(sigma2) + 2));
	}

	/* (non-Javadoc)
	 * @see net.morilib.math.stat.dist.ProbabilityDistribution#kurtosis()
	 */
	public double kurtosis() {
		return (Math.exp(4 * sigma2) +
				2 * Math.exp(3 * sigma2) +
				3 * Math.exp(2 * sigma2) - 6);
	}

	/* (non-Javadoc)
	 * @see net.morilib.math.stat.dist.ContinuousDistribution#f(double)
	 */
	public double f(double x) {
		double k, a, l;

		if(x <= 0)  return 0.0;
		l = Math.log(x);
		k = -((l - mu) * (l - mu) / 2.0 / sigma2);
		a = 1.0 / Math.sqrt(2 * Math.PI * sigma2) / x;
		return a * Math.exp(k);
	}

	/* (non-Javadoc)
	 * @see net.morilib.math.stat.dist.ContinuousDistribution#isInSupport(double)
	 */
	public boolean isInSupport(double n) {
		return true;
	}

	/* (non-Javadoc)
	 * @see net.morilib.math.stat.dist.ContinuousDistribution#cdf(double)
	 */
	public double cdf(double x) {
		double l;

		if(x <= 0)  return 0.0;
		l = Math.log(x);
		return DistributionUtils.trimCdf(
				0.5 * Erf.erfc(-(l - mu) / Math.sqrt(2 * sigma2)),
				1e-13);
	}

	/* (non-Javadoc)
	 * @see net.morilib.math.stat.dist.ContinuousDistribution#supportSupremum()
	 */
	public double supportSupremum() {
		return Double.POSITIVE_INFINITY;
	}

	/* (non-Javadoc)
	 * @see net.morilib.math.stat.dist.ContinuousDistribution#supportInfimum()
	 */
	public double supportInfimum() {
		return 0;
	}

	/* (non-Javadoc)
	 * @see net.morilib.math.stat.dist.AbstractContinuousDistribution#median()
	 */
	@Override
	public double median() {
		return Math.exp(mu);
	}

	/* (non-Javadoc)
	 * @see java.lang.Object#hashCode()
	 */
	public int hashCode() {
		int r = Hashes.INIT;

		r = Hashes.A * (r + (int)Double.doubleToLongBits(mu));
		r = Hashes.A * (r + (int)Double.doubleToLongBits(sigma2));
		return r;
	}

	/* (non-Javadoc)
	 * @see java.lang.Object#equals(java.lang.Object)
	 */
	public boolean equals(Object o) {
		LogNormalDistribution n;

		if(o instanceof LogNormalDistribution) {
			n = (LogNormalDistribution)o;
			return mu == n.mu && sigma2 == n.sigma2;
		}
		return false;
	}

}
