/*
 * Copyright 2013 Yuichiro Moriguchi
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package net.morilib.sh;

import java.util.EnumMap;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Properties;

public class ShFunctionEnvironment implements ShEnvironment {

	private Map<String, String> map;
	private Map<String, ShTree> functions;
	private Map<String, Attributes> attrs;
	private ShEnvironment env;
	private EnumMap<ShSignal, String> traps;

	public ShFunctionEnvironment(ShEnvironment env) {
		this.env  = env;
		this.map  = new HashMap<String, String>();
		functions = new HashMap<String, ShTree>();
		this.attrs = new HashMap<String, Attributes>();
	}

	public void bind(String name, String value) {
		if(isReadonly(name)) {
			throw new ShRuntimeException();
		} else if(map.containsKey(name)) {
			map.put(name, value);
		} else {
			env.bind(name, value);
		}
	}

	public void bind(String name, ShTree f) {
		if(functions.containsKey(name)) {
			functions.put(name, f);
		} else {
			env.bind(name, f);
		}
	}

	public boolean contains(String name) {
		return map.containsKey(name) || env.contains(name);
	}

	public void export(String name) {
		env.export(name);
		if(map.containsKey(name)) {
			env.bind(name, map.get(name));
			map.remove(name);
		}
	}

	public String find(String name) {
		String v;

		return (v = map.get(name)) != null ? v : env.find(name);
	}

	public ShTree findFunction(String name) {
		ShTree v = functions.get(name);

		return v != null ? v : env.findFunction(name);
	}

	public ShEnvironment getEnvironment() {
		return env.getEnvironment();
	}

	public List<String> getPath() {
		return env.getPath();
	}

	public boolean isEnvironment() {
		return false;
	}

	public boolean isSet(String name) {
		return env.isSet(name);
	}

	public void put(String name, String value) {
		map.put(name, value);
	}

	public void put(String name, ShTree function) {
		functions.put(name, function);
	}

	public void reset(String name) {
		env.reset(name);
	}

	public void set(String name) {
		env.set(name);
	}

	public void set(String name, boolean value) {
		env.set(name, value);
	}

	public void unbind(String name) {
		if(map.containsKey(name)) {
			map.remove(name);
		} else {
			env.unbind(name);
		}
	}

	public boolean isReadonly(String name) {
		Attributes a;

		return (isSystem(name) ||
				(map.containsKey(name) &&
						(a = attrs.get(name)) != null &&
						a.isReadonly()) ||
				(!map.containsKey(name) && env.isReadonly(name)));
	}

	public boolean isSystem(String name) {
		return (name.matches("[0-9]+") ||
				name.equals("@")       ||
				name.equals("*"));
	}

	public void setReadonly(String name) {
		if(map.containsKey(name)) {
			attrs.put(name, READONLY);
		} else {
			env.setReadonly(name);
		}
	}

	public Properties toProperties() {
		Properties p = env.toProperties();

		for(Map.Entry<String, String> e : map.entrySet()) {
			p.setProperty(e.getKey(), e.getValue());
		}
		return p;
	}

	/* (non-Javadoc)
	 * @see net.morilib.sh.ShEnvironment#getTrap(java.lang.String)
	 */
	public String getTrap(ShSignal signal) {
		String s;

		if(!signal.isInherit()) {
			return traps.get(signal);
		} else if((s = traps.get(signal)) != null) {
			return s;
		} else {
			return env.getTrap(signal);
		}
	}

	/* (non-Javadoc)
	 * @see net.morilib.sh.ShEnvironment#setTrap(java.lang.String, java.lang.String)
	 */
	public void setTrap(ShSignal signal, String cmd) {
		traps.put(signal, cmd);
	}

}
