/*
 * Copyright 2013 Yuichiro Moriguchi
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package net.morilib.sh.file;

import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Properties;

import net.morilib.sh.ShEnvironment;
import net.morilib.sh.ShSignal;
import net.morilib.sh.ShTree;

public class ShUnixEnvironment implements ShEnvironment {

	private static final String PS =
		System.getProperty("path.separator");

	private Map<String, String> map;
	private List<String> path;
	private Map<String, ShTree> functions;
	private Map<String, Attributes> attrs;

	ShUnixEnvironment() {
		functions  = new HashMap<String, ShTree>();
		this.attrs = new HashMap<String, Attributes>();
		try {
			map = new HashMap<String, String>(System.getenv());
			setpath();
		} catch(SecurityException e) {
			map = new HashMap<String, String>();
		}
	}

	private void setpath() {
		String s;

		if((s = map.get("PATH")) != null) {
			s = s.replaceFirst("^" + PS + "+", "");
			s = s.replaceFirst(PS + "+$", "");
			path = Arrays.asList(s.split(PS + "+"));
		}
	}

	public void bind(String name, String value) {
		map.put(name, value);
		setpath();
	}

	public boolean contains(String name) {
		return map.containsKey(name);
	}

	public String find(String name) {
		return map.containsKey(name) ? map.get(name) : "";
	}

	public List<String> getPath() {
		return path;
	}

	public ShEnvironment getEnvironment() {
		return null;
	}

	public boolean isEnvironment() {
		return true;
	}

	public void put(String name, String value) {
		map.put(name, value);
	}

	public Properties toProperties() {
		Properties r = new Properties();

		for(Map.Entry<String, String> e : map.entrySet()) {
			r.setProperty(e.getKey(), e.getValue());
		}
		return r;
	}

	public void bind(String name, ShTree function) {
		functions.put(name, function);
	}

	public void put(String name, ShTree function) {
		functions.put(name, function);
	}

	public ShTree findFunction(String name) {
		return functions.get(name);
	}

	public void export(String name) {
		map.put(name, "");
	}

	public boolean isSet(String name) {
		return false;
	}

	public void set(String name) {
	}

	public void reset(String name) {
	}

	public void set(String name, boolean value) {
	}

	public void unbind(String name) {
		map.remove(name);
	}

	public boolean isReadonly(String name) {
		Attributes a;

		return (a = attrs.get(name)) != null && a.isReadonly();
	}

	public boolean isSystem(String name) {
		return false;
	}

	public void setReadonly(String name) {
		if(map.containsKey(name))  attrs.put(name, READONLY);
	}

	/* (non-Javadoc)
	 * @see net.morilib.sh.ShEnvironment#getTrap(net.morilib.sh.ShSignal)
	 */
	public String getTrap(ShSignal signal) {
		// OS signals cannot trap by Java VM
		return null;
	}

	/* (non-Javadoc)
	 * @see net.morilib.sh.ShEnvironment#setTrap(net.morilib.sh.ShSignal, java.lang.String)
	 */
	public void setTrap(ShSignal signal, String cmd) {
		// OS signals cannot trap by Java VM
	}

}
