/*
 * Copyright 2013 Yuichiro Moriguchi
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package net.morilib.awk;

import net.morilib.awk.printf.Printf;
import net.morilib.awk.value.AwkFloat;
import net.morilib.awk.value.AwkInteger;
import net.morilib.awk.value.AwkString;
import junit.framework.TestCase;

public class PrintfTest extends TestCase {

	static final Printf P = Printf.getInstance();

	void eqs(String f, double a, String r) {
		assertEquals(r, P.format(f, AwkFloat.valueOf(a)));
	}

	void eqs(String f, long a, String r) {
		assertEquals(r, P.format(f, AwkInteger.valueOf(a)));
	}

	void eqs(String f, long a, long b, String r) {
		assertEquals(r, P.format(f,
				AwkInteger.valueOf(a),
				AwkInteger.valueOf(b)));
	}

	void eqs(String f, long a, long b, long c, String r) {
		assertEquals(r, P.format(f,
				AwkInteger.valueOf(a),
				AwkInteger.valueOf(b),
				AwkInteger.valueOf(c)));
	}

	void eqs(String f, long a, double b, String r) {
		assertEquals(r, P.format(f,
				AwkInteger.valueOf(a),
				AwkFloat.valueOf(b)));
	}

	void eqs(String f, long a, long b, double c, String r) {
		assertEquals(r, P.format(f,
				AwkInteger.valueOf(a),
				AwkInteger.valueOf(b),
				AwkFloat.valueOf(c)));
	}

	void eqs(String f, String a, String r) {
		assertEquals(r, P.format(f, AwkString.valueOf(a)));
	}

	public void testAsterisk() {
		eqs("%*d", 5, 0, "    0");
		eqs("%.*f", 2, 1.1, "1.10");
		eqs("%+.*f", 2, 1.1, "+1.10");
		eqs("% .*f", 2, 1.1, " 1.10");
		eqs("%*.*f", 10, 2,  1.1, "      1.10");
		eqs("%*.*f", 10, 2, -1.1, "     -1.10");
	}

	public void testG() {
		eqs("%.2g", 1.1, "1.1");
		eqs("%+.2g", 1.1, "+1.1");
		eqs("% .2g", 1.1, " 1.1");
		eqs("%10.5g",    1.1, "       1.1");
		eqs("%10.5g",   -1.1, "      -1.1");
		eqs("%+10.5g",   1.1, "      +1.1");
		eqs("%+10.5g",  -1.1, "      -1.1");
		eqs("% 10.5g",   1.1, "       1.1");
		eqs("% 10.5g",  -1.1, "      -1.1");
		eqs("%010.5g",   1.1, "00000001.1");
		eqs("%010.5g",  -1.1, "-0000001.1");
		eqs("%-10.5g",   1.1, "1.1       ");
		eqs("%-10.5g",  -1.1, "-1.1      ");
		eqs("%-010.5g",  1.1, "1.1       ");
		eqs("%-010.5g", -1.1, "-1.1      ");
		eqs("%- 10.5g",  1.1, " 1.1      ");
		eqs("%- 10.5g", -1.1, "-1.1      ");

		eqs("%10.5g",    1.0, "         1");
		eqs("%10.5g",   -1.0, "        -1");
		eqs("%+10.5g",   1.0, "        +1");
		eqs("%+10.5g",  -1.0, "        -1");
		eqs("% 10.5g",   1.0, "         1");
		eqs("% 10.5g",  -1.0, "        -1");
		eqs("%010.5g",   1.0, "0000000001");
		eqs("%010.5g",  -1.0, "-000000001");
		eqs("%-10.5g",   1.0, "1         ");
		eqs("%-10.5g",  -1.0, "-1        ");
		eqs("%-010.5g",  1.0, "1         ");
		eqs("%-010.5g", -1.0, "-1        ");
		eqs("%- 10.5g",  1.0, " 1        ");
		eqs("%- 10.5g", -1.0, "-1        ");
	}

	public void testF() {
		eqs("%.2f", 1.1, "1.10");
		eqs("%+.2f", 1.1, "+1.10");
		eqs("% .2f", 1.1, " 1.10");
		eqs("%10.2f",    1.1, "      1.10");
		eqs("%10.2f",   -1.1, "     -1.10");
		eqs("%+10.2f",   1.1, "     +1.10");
		eqs("%+10.2f",  -1.1, "     -1.10");
		eqs("% 10.2f",   1.1, "      1.10");
		eqs("% 10.2f",  -1.1, "     -1.10");
		eqs("%010.2f",   1.1, "0000001.10");
		eqs("%010.2f",  -1.1, "-000001.10");
		eqs("%-10.2f",   1.1, "1.10      ");
		eqs("%-10.2f",  -1.1, "-1.10     ");
		eqs("%-010.2f",  1.1, "1.10      ");
		eqs("%-010.2f", -1.1, "-1.10     ");
		eqs("%- 10.2f",  1.1, " 1.10     ");
		eqs("%- 10.2f", -1.1, "-1.10     ");

		eqs("%7.5f",   111.1, "111.10000");
		eqs("%7.5f",  -111.1, "-111.10000");
		eqs("% 7.5f",  111.1, " 111.10000");
		eqs("% 7.5f", -111.1, "-111.10000");
		eqs("%07.5f",  111.1, "111.10000");
		eqs("%07.5f", -111.1, "-111.10000");
	}

	public void testE() {
		eqs("%.2e", 1.1, "1.10e+00");
		eqs("%+.2e", 1.1, "+1.10e+00");
		eqs("% .2e", 1.1, " 1.10e+00");
		eqs("%10.2e", 1.1, "  1.10e+00");
		eqs("%10.2e", -1.1, " -1.10e+00");
		eqs("%+10.2e", 1.1, " +1.10e+00");
		eqs("%+10.2e", -1.1, " -1.10e+00");
		eqs("% 10.2e", 1.1, "  1.10e+00");
		eqs("% 10.2e", -1.1, " -1.10e+00");
		eqs("%010.2e", 1.1, "001.10e+00");
		eqs("%010.2e", -1.1, "-01.10e+00");
		eqs("%-10.2e", 1.1, "1.10e+00  ");
		eqs("%-10.2e", -1.1, "-1.10e+00 ");
		eqs("%-010.2e", 1.1, "1.10e+00  ");
		eqs("%-010.2e", -1.1, "-1.10e+00 ");
		eqs("%- 10.2e", 1.1, " 1.10e+00 ");
		eqs("%- 10.2e", -1.1, "-1.10e+00 ");

		eqs("%10.5e", 1.1, "1.10000e+00");
		eqs("%10.5e", -1.1, "-1.10000e+00");
		eqs("% 10.5e", 1.1, " 1.10000e+00");
		eqs("% 10.5e", -1.1, "-1.10000e+00");
		eqs("%010.5e", 1.1, "1.10000e+00");
		eqs("%010.5e", -1.1, "-1.10000e+00");
	}

	public void testS() {
		eqs("%s", "aaa", "aaa");
		eqs("%s", "", "");

		eqs("%5s", "aaa", "  aaa");
		eqs("%5s", "", "     ");
		eqs("%-5s", "aaa", "aaa  ");
		eqs("%-5s", "", "     ");

		eqs("%.5s", "1234567", "12345");
		eqs("%.5s", "123", "  123");
		eqs("%.5s", "", "     ");
		eqs("%-.5s", "123", "123  ");
		eqs("%-.5s", "", "     ");

		eqs("%3.5s", "1234567", "12345");
		eqs("%3.5s", "1234", "1234");
		eqs("%3.5s", "12", " 12");
		eqs("%3.5s", "", "   ");
		eqs("%-3.5s", "1234567", "12345");
		eqs("%-3.5s", "1234", "1234");
		eqs("%-3.5s", "12", "12 ");
		eqs("%-3.5s", "", "   ");

		eqs("%5.3s", "1234567", "  123");
		eqs("%5.3s", "12345", "  123");
		eqs("%5.3s", "12", "   12");
		eqs("%5.3s", "", "     ");
		eqs("%-5.3s", "1234567", "123  ");
		eqs("%-5.3s", "12345", "123  ");
		eqs("%-5.3s", "12", "12   ");
		eqs("%-5.3s", "", "     ");

		eqs("%0s", "aaa", "aaa");
		eqs("%-0s", "aaa", "aaa");
		eqs("%.0s", "", "");
		eqs("%-.0s", "", "");
		eqs("%0.5s", "aaa", "  aaa");
		eqs("%-0.5s", "aaa", "aaa  ");
		eqs("%5.0s", "aaa", "     ");
		eqs("%-5.0s", "aaa", "     ");
	}

	public void testD() {
		eqs("%d", 123, "123");
		eqs("%+d", 123, "+123");
		eqs("% d", 123, " 123");
		eqs("%d", -123, "-123");
		eqs("%+d", -123, "-123");
		eqs("% d", -123, "-123");

		eqs("%5d", 0, "    0");
		eqs("%5d", 123, "  123");
		eqs("%5d", -123, " -123");
		eqs("%5d", 12345, "12345");
		eqs("%5d", 1234567, "1234567");

		eqs("%05d", 0, "00000");
		eqs("%05d", 123, "00123");
		eqs("%05d", -123, "-0123");
		eqs("%05d", 12345, "12345");
		eqs("%05d", 1234567, "1234567");

		eqs("%+5d", 123, " +123");
		eqs("%+5d", 1234, "+1234");
		eqs("%+5d", 12345, "+12345");
		eqs("%+5d", 0, "   +0");
		eqs("%+5d", -123, " -123");
		eqs("%+5d", -1234, "-1234");
		eqs("%+5d", -12345, "-12345");

		eqs("%+05d", 123, "+0123");
		eqs("%+05d", 1234, "+1234");
		eqs("%+05d", 12345, "+12345");
		eqs("%+05d", 0, "+0000");
		eqs("%+05d", -123, "-0123");
		eqs("%+05d", -1234, "-1234");
		eqs("%+05d", -12345, "-12345");

		eqs("% 5d", 123, "  123");
		eqs("% 5d", 1234, " 1234");
		eqs("% 5d", 12345, " 12345");
		eqs("% 5d", 0, "    0");
		eqs("% 5d", -123, " -123");
		eqs("% 5d", -1234, "-1234");
		eqs("% 5d", -12345, "-12345");

		eqs("% 05d", 123, " 0123");
		eqs("% 05d", 1234, " 1234");
		eqs("% 05d", 12345, " 12345");
		eqs("% 05d", 0, " 0000");
		eqs("% 05d", -123, "-0123");
		eqs("% 05d", -1234, "-1234");
		eqs("% 05d", -12345, "-12345");
		eqs("%0 5d", 123, " 0123");

		eqs("%-5d", 0, "0    ");
		eqs("%-5d", 123, "123  ");
		eqs("%-5d", 12345, "12345");
		eqs("%-5d", 1234567, "1234567");

		eqs("%-05d", 0, "0    ");
		eqs("%-05d", 123, "123  ");
		eqs("%-05d", 12345, "12345");
		eqs("%-05d", 1234567, "1234567");

		eqs("%- 5d", 0, " 0   ");
		eqs("%- 5d", 123, " 123 ");
		eqs("%- 5d", 12345, " 12345");
		eqs("%- 5d", 1234567, " 1234567");
	}

	public void testO() {
		eqs("%o", 0123, "123");
		eqs("%+o", 0123, "+123");
		eqs("% o", 0123, " 123");
		eqs("%o", -0123, "-123");
		eqs("%+o", -0123, "-123");
		eqs("% o", -0123, "-123");

		eqs("%5o", 0, "    0");
		eqs("%5o", 0123, "  123");
		eqs("%5o", -0123, " -123");
		eqs("%5o", 012345, "12345");
		eqs("%5o", 01234567, "1234567");

		eqs("%05o", 0, "00000");
		eqs("%05o", 0123, "00123");
		eqs("%05o", -0123, "-0123");
		eqs("%05o", 012345, "12345");
		eqs("%05o", 01234567, "1234567");

		eqs("%+5o", 0123, " +123");
		eqs("%+5o", 01234, "+1234");
		eqs("%+5o", 012345, "+12345");
		eqs("%+5o", 0, "   +0");
		eqs("%+5o", -0123, " -123");
		eqs("%+5o", -01234, "-1234");
		eqs("%+5o", -012345, "-12345");

		eqs("%+05o", 0123, "+0123");
		eqs("%+05o", 01234, "+1234");
		eqs("%+05o", 012345, "+12345");
		eqs("%+05o", 0, "+0000");
		eqs("%+05o", -0123, "-0123");
		eqs("%+05o", -01234, "-1234");
		eqs("%+05o", -012345, "-12345");

		eqs("% 5o", 0123, "  123");
		eqs("% 5o", 01234, " 1234");
		eqs("% 5o", 012345, " 12345");
		eqs("% 5o", 0, "    0");
		eqs("% 5o", -0123, " -123");
		eqs("% 5o", -01234, "-1234");
		eqs("% 5o", -012345, "-12345");

		eqs("% 05o", 0123, " 0123");
		eqs("% 05o", 01234, " 1234");
		eqs("% 05o", 012345, " 12345");
		eqs("% 05o", 0, " 0000");
		eqs("% 05o", -0123, "-0123");
		eqs("% 05o", -01234, "-1234");
		eqs("% 05o", -012345, "-12345");
		eqs("%0 5o", 0123, " 0123");

		eqs("%-5o", 0, "0    ");
		eqs("%-5o", 0123, "123  ");
		eqs("%-5o", 012345, "12345");
		eqs("%-5o", 01234567, "1234567");

		eqs("%-05o", 0, "0    ");
		eqs("%-05o", 0123, "123  ");
		eqs("%-05o", 012345, "12345");
		eqs("%-05o", 01234567, "1234567");
	}

	public void testX() {
		eqs("%x", 0x123, "123");
		eqs("%+x", 0x123, "+123");
		eqs("% x", 0x123, " 123");
		eqs("%x", -0x123, "-123");
		eqs("%+x", -0x123, "-123");
		eqs("% x", -0x123, "-123");
		eqs("%x", 0xabc, "abc");
		eqs("%X", 0xabc, "ABC");

		eqs("%5x", 0, "    0");
		eqs("%5x", 0x123, "  123");
		eqs("%5x", -0x123, " -123");
		eqs("%5x", 0x12345, "12345");
		eqs("%5x", 0x1234567, "1234567");

		eqs("%05x", 0, "00000");
		eqs("%05x", 0x123, "00123");
		eqs("%05x", -0x123, "-0123");
		eqs("%05x", 0x12345, "12345");
		eqs("%05x", 0x1234567, "1234567");

		eqs("%+5x", 0x123, " +123");
		eqs("%+5x", 0x1234, "+1234");
		eqs("%+5x", 0x12345, "+12345");
		eqs("%+5x", 0, "   +0");
		eqs("%+5x", -0x123, " -123");
		eqs("%+5x", -0x1234, "-1234");
		eqs("%+5x", -0x12345, "-12345");

		eqs("%+05x", 0x123, "+0123");
		eqs("%+05x", 0x1234, "+1234");
		eqs("%+05x", 0x12345, "+12345");
		eqs("%+05x", 0, "+0000");
		eqs("%+05x", -0x123, "-0123");
		eqs("%+05x", -0x1234, "-1234");
		eqs("%+05x", -0x12345, "-12345");

		eqs("% 5x", 0x123, "  123");
		eqs("% 5x", 0x1234, " 1234");
		eqs("% 5x", 0x12345, " 12345");
		eqs("% 5x", 0, "    0");
		eqs("% 5x", -0x123, " -123");
		eqs("% 5x", -0x1234, "-1234");
		eqs("% 5x", -0x12345, "-12345");

		eqs("% 05x", 0x123, " 0123");
		eqs("% 05x", 0x1234, " 1234");
		eqs("% 05x", 0x12345, " 12345");
		eqs("% 05x", 0, " 0000");
		eqs("% 05x", -0x123, "-0123");
		eqs("% 05x", -0x1234, "-1234");
		eqs("% 05x", -0x12345, "-12345");
		eqs("%0 5x", 0x123, " 0123");

		eqs("%-5x", 0, "0    ");
		eqs("%-5x", 0x123, "123  ");
		eqs("%-5x", 0x12345, "12345");
		eqs("%-5x", 0x1234567, "1234567");

		eqs("%-05x", 0, "0    ");
		eqs("%-05x", 0x123, "123  ");
		eqs("%-05x", 0x12345, "12345");
		eqs("%-05x", 0x1234567, "1234567");
	}

	public void testC() {
		eqs("%c", 32, " ");
	}

	public void testArg2() {
		assertEquals("aaa", P.format("aaa"));
		assertEquals("aaa bbb", P.format("%s %s",
				AwkString.valueOf("aaa"),
				AwkString.valueOf("bbb")));
	}

}
