/*
 * Copyright 2013 Yuichiro Moriguchi
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package net.morilib.awk;

import java.io.IOException;
import java.io.StringReader;

import junit.framework.TestCase;

import net.morilib.awk.io.AwkFiles;
import net.morilib.awk.namespace.AwkNamespace;
import net.morilib.awk.namespace.AwkRootNamespace;
import net.morilib.awk.parser.AwkLexer;
import net.morilib.awk.parser.AwkParser;
import net.morilib.awk.value.AwkUndefined;

public class AwkExpressionTest extends TestCase {

	static AwkNamespace ns0;

	static AwkLexer lex(String s) {
		try {
			return new AwkLexer(new StringReader(s));
		} catch (IOException e) {
			throw new RuntimeException(s);
		}
	}

	static AwkNamespace ns0() {
		return new AwkRootNamespace();
	}

	static AwkFiles fs0() {
		return new AwkFiles();
	}

	protected void setUp() {
		ns0 = ns0();
	}

	static void eqs(String l, String s) {
		try {
			assertEquals(AwkParser.parseExpression(lex(l)).eval(ns0, fs0()).toString(), s);
		} catch (IOException e) {
			throw new RuntimeException(e);
		}
	}

	static void eqi(String l, long s) {
		try {
			assertEquals(AwkParser.parseExpression(lex(l)).eval(ns0, fs0()).toInteger().longValue(), s);
		} catch (IOException e) {
			throw new RuntimeException(e);
		}
	}

	static void eqf(String l, double s) {
		try {
			assertEquals(AwkParser.parseExpression(lex(l)).eval(ns0, fs0()).toFloat(), s);
		} catch (IOException e) {
			throw new RuntimeException(e);
		}
	}

	static void eqr(String l, String s) {
		try {
			assertEquals(AwkParser.parseExpression(lex(l)).eval(ns0, fs0()).toRegex().pattern(), s);
		} catch (IOException e) {
			throw new RuntimeException(e);
		}
	}

	static void und(String l) {
		try {
			assertEquals(AwkParser.parseExpression(lex(l)).eval(ns0, fs0()), AwkUndefined.UNDEF);
		} catch (IOException e) {
			throw new RuntimeException(e);
		}
	}

	void eqz(String nm, long v) {
		assertEquals(ns0.find(nm).toInteger().longValue(), v);
	}

	void eqz(String nm, double v) {
		assertEquals(ns0.find(nm).toFloat(), v);
	}

	void eqz(String nm, String v) {
		assertEquals(ns0.find(nm).toString(), v);
	}

	public void testParen() {
		eqi("3-(2-1)", 2);
	}

	public void testField() {
		eqs("FS = \" \t\"", " \t");
		ns0.setField("aaa bbb  ccc\t\tddd");
		eqs("$0", "aaa bbb  ccc\t\tddd");
		eqs("$1", "aaa");
		eqs("$2", "bbb");
		eqs("$3", "ccc");
		eqs("$4", "ddd");
		eqs("$NF", "ddd");
		und("$5");

		ns0.setField("aaa bbb  ccc\t\tddd  ");
		eqs("$0", "aaa bbb  ccc\t\tddd  ");
		eqs("$1", "aaa");
		eqs("$2", "bbb");
		eqs("$3", "ccc");
		eqs("$4", "ddd");
		und("$5");
	}

	public void testIncdec() {
		eqi("a++", 0);  eqz("a", 1);
		eqi("a++", 1);  eqz("a", 2);
		eqi("++a", 3);  eqz("a", 3);
		eqi("a--", 3);  eqz("a", 2);
		eqi("--a", 1);  eqz("a", 1);
		eqf("a = 1.5", 1.5);
		eqf("a++", 1.5);  eqz("a", 2.5);
		eqf("++a", 3.5);  eqz("a", 3.5);
		eqf("a--", 3.5);  eqz("a", 2.5);
		eqf("--a", 1.5);  eqz("a", 1.5);
		eqs("a = \"1\"", "1");
		eqs("a++", "1");  eqz("a", 2);
		eqs("a = \"1\"", "1");
		eqi("++a", 2);  eqz("a", 2);
		eqs("a = \"1.5\"", "1.5");
		eqs("a++", "1.5");  eqz("a", 2.5);
		eqs("a = \"1.5\"", "1.5");
		eqf("++a", 2.5);  eqz("a", 2.5);
		eqs("a = \"1\"", "1");
		eqs("a--", "1");  eqz("a", 0);
		eqs("a = \"1\"", "1");
		eqi("--a", 0);  eqz("a", 0);
		eqs("a = \"1.5\"", "1.5");
		eqs("a--", "1.5");  eqz("a", 0.5);
		eqs("a = \"1.5\"", "1.5");
		eqf("--a", 0.5);  eqz("a", 0.5);
		eqf("b++", 0);  eqz("b", 1);
		eqf("++c", 1);  eqz("c", 1);
		eqf("d--", 0);  eqz("d", -1);
		eqf("--e", -1);  eqz("e", -1);
	}

	public void testPow() {
		eqi("2^4", 16);
		eqi("2^3^2", 512);
		eqi("-2^4", -16);
	}

	public void testUnaries() {
		eqi("-1", -1);
		eqi("+1",  1);
		eqi("!0",  1);
		eqi("-1+2", 1);
	}

	public void testId() {
		eqi("1", 1);
		eqf("3.0", 3.0);
		eqs("\"aaa\"", "aaa");
		eqs("\"\"", "");
		eqr("/aaa/", "aaa");
		eqr("/=v=/", "=v=");
		eqr("/   /", "   ");
		eqr("//", "");
	}

	public void testFactor() {
		eqi("2*3", 6);
		eqi("2*3*4", 24);
		eqi("5%3", 2);
		eqi("4/2", 2);
		eqi("24/4/3", 2);
		eqi("24/4*3", 18);
		eqi("5%3%2", 0);
		eqf("2*3.0", 6.0);
		eqf("4/2.0", 2.0);
		eqi("2*\"3\"", 6);
		eqi("4/\"2\"", 2);
		eqi("5%\"3\"", 2);
		eqf("2*\"3.0\"", 6.0);
		eqf("4/\"2.0\"", 2.0);
		eqf("3/2", 1.5);
	}

	public void testTerm() {
		eqi("3+2", 5);
		eqi("3-2", 1);
		eqi("3-2-1", 0);
		eqi("3-2+1", 2);
		eqi("3*5+18/3", 21);
		eqf("3+2.0", 5.0);
		eqf("3-2.0", 1.0);
		eqi("3+\"2\"", 5);
		eqi("3-\"2\"", 1);
		eqf("3+\"2.0\"", 5.0);
		eqf("3-\"2.0\"", 1.0);
	}

	public void testConcat() {
		eqs("\"1\" \"2\"", "12");
		eqs("\"1\" \"2\" \"3\"", "123");
		eqs("1 + 2 \"2\"", "32");
		eqs("1 + 2 \".\" 7 * 2", "3.14");
		eqs("1 2", "12");
		eqs("1.0 2.0", "1.02.0");
	}

	public void testRelop() {
		eqi("7 >  2", 1);
		eqi("7 >= 2", 1);
		eqi("7 <  2", 0);
		eqi("7 <= 2", 0);
		eqi("7 == 2", 0);
		eqi("7 != 2", 1);
		eqi("7 >  7", 0);
		eqi("7 >= 7", 1);
		eqi("7 <  7", 0);
		eqi("7 <= 7", 1);
		eqi("7 == 7", 1);
		eqi("7 != 7", 0);
		eqi("7 >  8", 0);
		eqi("7 >= 8", 0);
		eqi("7 <  8", 1);
		eqi("7 <= 8", 1);
		eqi("7 == 8", 0);
		eqi("7 != 8", 1);
		eqi("\"aa\" >  \"ab\"", 0);
		eqi("\"aa\" >= \"ab\"", 0);
		eqi("\"aa\" <  \"ab\"", 1);
		eqi("\"aa\" <= \"ab\"", 1);
		eqi("\"aa\" == \"ab\"", 0);
		eqi("\"aa\" != \"ab\"", 1);
		eqi("\"aa\" >  \"aa\"", 0);
		eqi("\"aa\" >= \"aa\"", 1);
		eqi("\"aa\" <  \"aa\"", 0);
		eqi("\"aa\" <= \"aa\"", 1);
		eqi("\"aa\" == \"aa\"", 1);
		eqi("\"aa\" != \"aa\"", 0);
		eqi("\"ab\" >  \"aa\"", 1);
		eqi("\"ab\" >= \"aa\"", 1);
		eqi("\"ab\" <  \"aa\"", 0);
		eqi("\"ab\" <= \"aa\"", 0);
		eqi("\"ab\" == \"aa\"", 0);
		eqi("\"ab\" != \"aa\"", 1);
		eqi("7 >  2.0", 1);
		eqi("7 >= 2.0", 1);
		eqi("7 <  2.0", 0);
		eqi("7 <= 2.0", 0);
		eqi("7 == 2.0", 0);
		eqi("7 != 2.0", 1);
		eqi("7 >  7.0", 0);
		eqi("7 >= 7.0", 1);
		eqi("7 <  7.0", 0);
		eqi("7 <= 7.0", 1);
		eqi("7 == 7.0", 1);
		eqi("7 != 7.0", 0);
		eqi("7 >  8.0", 0);
		eqi("7 >= 8.0", 0);
		eqi("7 <  8.0", 1);
		eqi("7 <= 8.0", 1);
		eqi("7 == 8.0", 0);
		eqi("7 != 8.0", 1);
		eqi("72 >  \"8\"", 0);
		eqi("72 >= \"8\"", 0);
		eqi("72 <  \"8\"", 1);
		eqi("72 <= \"8\"", 1);
		eqi("72 == \"8\"", 0);
		eqi("72 != \"8\"", 1);
		eqi("8  >  \"72\"", 1);
		eqi("8  >= \"72\"", 1);
		eqi("8  <  \"72\"", 0);
		eqi("8  <= \"72\"", 0);
		eqi("8  == \"72\"", 0);
		eqi("8  != \"72\"", 1);
		eqi("72 >  \"72\"", 0);
		eqi("72 >= \"72\"", 1);
		eqi("72 <  \"72\"", 0);
		eqi("72 <= \"72\"", 1);
		eqi("72 == \"72\"", 1);
		eqi("72 != \"72\"", 0);
		eqi("1+6 >  2", 1);
		eqi("1+6 >= 2", 1);
		eqi("1+6 <  2", 0);
		eqi("1+6 <= 2", 0);
		eqi("1+6 == 2", 0);
		eqi("1+6 != 2", 1);
		eqi("72 >  \"7\" \"2\"", 0);
		eqi("72 >= \"7\" \"2\"", 1);
		eqi("72 <  \"7\" \"2\"", 0);
		eqi("72 <= \"7\" \"2\"", 1);
		eqi("72 == \"7\" \"2\"", 1);
		eqi("72 != \"7\" \"2\"", 0);
	}

	public void testMatchop() {
		eqi("/a+/ ~  \"aaa\"", 1);
		eqi("/a+/ !~ \"aaa\"", 0);
		eqi("/a+/ ~  \"bbb\"", 0);
		eqi("/a+/ !~ \"bbb\"", 1);
		eqi("/1+/ ~  111", 1);
		eqi("/1+/ !~ 111", 0);
		eqi("/1+/ ~  111.0", 1);
		eqi("/1+/ !~ 111.0", 0);
		eqi("/a+/ ~  \"bb\" \"a\"", 1);
		eqi("/a+/ !~ \"bb\" \"b\"", 1);
	}

	public void testLogand() {
		eqi("1 && 1", 1);
		eqi("0 && 1", 0);
		eqi("1 && 0", 0);
		eqi("0 && 0", 0);
		eqi("\"0\" && \"0\"", 1);
		eqi("\"\"  && \"0\"", 0);
		eqi("\"0\" && \"\"",  0);
		eqi("\"\"  && \"\"",  0);
		eqi("2 + 0 && 1 + 3", 1);
		eqi("2 == 2 && 1 == 1", 1);
		eqi("2 != 2 && 1 == 1", 0);
		eqi("2 == 2 && 1 != 1", 0);
		eqi("2 != 2 && 1 != 1", 0);
		eqi("1 && 1 && 0", 0);
		eqi("1 && 1 && 1", 1);
	}

	public void testLogor() {
		eqi("1 || 1", 1);
		eqi("0 || 1", 1);
		eqi("1 || 0", 1);
		eqi("0 || 0", 0);
		eqi("\"0\" || \"0\"", 1);
		eqi("\"\"  || \"0\"", 1);
		eqi("\"0\" || \"\"",  1);
		eqi("\"\"  || \"\"",  0);
		eqi("2 + 0 || 1 + 3", 1);
		eqi("2 == 2 || 1 == 1", 1);
		eqi("2 != 2 || 1 == 1", 1);
		eqi("2 == 2 || 1 != 1", 1);
		eqi("2 != 2 || 1 != 1", 0);
		eqi("0 || 0 || 0", 0);
		eqi("1 || 0 || 0", 1);
		eqi("1 && 1 || 1 && 0", 1);
	}

	public void testCond() {
		eqi("1 ? 3 : 4", 3);
		eqi("0 ? 3 : 4", 4);
		eqi("1 ? 3 + 4 : 4 + 8", 7);
		eqi("0 ? 3 + 4 : 4 + 5", 9);
		eqi("1 ? 1 || 1 : 0 || 0", 1);
		eqi("0 ? 1 || 1 : 0 || 0", 0);
	}

	public void testAssignop() {
		eqi("a  = 1", 1);  eqz("a", 1);
		eqi("a += 3", 4);  eqz("a", 4);
		eqi("a -= 2", 2);  eqz("a", 2);
		eqi("a *= 3", 6);  eqz("a", 6);
		eqi("a /= 2", 3);  eqz("a", 3);
	}

}
