/*
Wrapper of reference part in xoper/xloper12.
// AXL3 - An XLL Library
// Wrapper class library for Microsft Excel's xll add-in which written by C++.
//
// Copyright(c) 2009-2009, yukimi_sake@users.sourceforge.jp. All rights reserved.

xlVar and xlRange is twins.
A class which manipurates Excel's worksheet like an Excel's Range object.
*/

#pragma once

namespace xl{
	class xlRange : public xlBase{
		friend class xlAPI;
		friend class xlVar;
		friend class xlSheet;

	public:
		// constructors
		xlRange(){ XL_CALL_FUNC_1(f_createDefault, xltypeRef); }
		xlRange(IDSHEET sheetId){createAsSheetId(sheetId);}
		xlRange(void* op, UINT bitFree = 0){ XL_CALL_FUNC_2(f_createVoidPtr, op, bitFree); }
		xlRange(xlVar& op){createFromxlVar(op);} // op as xltypeRef or xltypeSRef
		xlRange(int row, int col){createSref(row,col,row,col);} //as xltypeSRef
		xlRange(int row, int col, xlRange* sheetId){createRef(row,col,row,col, sheetId);}//as xltypeRef
		xlRange(int rFirst, int cFirst, int rLast, int cLast){createSref(rFirst,cFirst,rLast,cLast);} //as xltypeSRef
		xlRange(int rFirst, int cFirst, int rLast, int cLast, xlRange* sheetId){createRef(rFirst,cFirst,rLast,cLast,sheetId);} //xltypeRef

		// copy constructor
		xlRange(const xlRange& from){ 
			m_excel12 ? f_copyThis12(&from) : f_copyThis4(&from);
		}

		//destructor
		~xlRange(){
			*m_refcount > 0 ? *m_refcount -= 1 : XL_CALL_FUNC_0(f_deleteSubstance); 
		}

		// static methods
		static xlRange ActiveCell(){ // returns ActiveCell on ActiveSheet.
			xlRange res; 
			API.Call(xlfActiveCell, &res, 0); 
			return res;
		}

		static xlRange Selection(){ // returns selection on ActiveSheet.
			xlRange res; 
			API.Call(xlfSelection, &res, 0); 
			return res;
		}

		// properties
		Property(xlVar, Value);
		Property(xlVar, Formula);
		Property(xlBorder, Border);
		Property(xlInterior, Interior);
		Property(xlFont, Font);
		Property(xlAlignment, Alignment);
		Property(std::string, NumberFormat);
		Property(bool, Locked);
		Property(bool, Hidden);

		PrReader(int, Row);
		PrReader(int, Column);

		// access methods
		xlVar getValue(){xlVar val; m_excel12 ? f_getValue12(&val) : f_getValue4(&val); return val;}
		void putValue(const xlVar& val){XL_CALL_FUNC_1(f_putValue, &val);}
		void putValue(const double val){xlVar op(val); XL_CALL_FUNC_1(f_putValue, &op);}
		void putValue(const int val){xlVar op(val); XL_CALL_FUNC_1(f_putValue, &op);}
		void putValue(const char* val){xlVar op(val); XL_CALL_FUNC_1(f_putValue, &op);}
		void putValue(const wchar_t* val){xlVar op(val); XL_CALL_FUNC_1(f_putValue, &op);}
		void putValue(const std::string val){xlVar op(val); XL_CALL_FUNC_1(f_putValue, &op);}
		void putValue(const bool val){xlVar op(val); XL_CALL_FUNC_1(f_putValue, &op);}

		xlVar getFormula(){xlVar val; XL_CALL_FUNC_1(f_getFormula, &val);return val;}
		void putFormula(const xlVar& val){XL_CALL_FUNC_1(f_putFormula, &val);}
		void putFormula(const char* val){xlVar op(val); XL_CALL_FUNC_1(f_putFormula, &op);}
		void putFormula(const std::string val){xlVar op(val); XL_CALL_FUNC_1(f_putFormula, &op);}

		xlBorder getBorder(){xlBorder bd; bd.putRange(this); return bd;}
		void putBorder(xlBorder border){border.set(this);}

		xlInterior getInterior(){xlInterior pt; pt.putRange(this); return pt;}
		void putInterior(xlInterior pt){pt.set(this);}

		xlFont getFont(){xlFont fn; fn.putRange(this); return fn;}
		void putFont(xlFont fn){fn.set(this);}

		xlAlignment getAlignment(){xlAlignment al; al.putRange(this); return al;}
		void putAlignment(xlAlignment al){al.set(this);}

		std::string getNumberFormat(){xlVar v; API.Call(xlfGetCell, &v, "IP", 7, this); return v.String;}
		void putNumberFormat(const char* fm){ API.Call(xlcFormatNumber, 0, "C", fm);}
		void putNumberFormat(const std::string fm){ API.Call(xlcFormatNumber, 0, "S", fm);}

		bool getLocked(){xlVar v; API.Call(xlfGetCell, &v, "IP", 14, this); return v.Bool;}
		void putLocked(bool lock){API.Call(xlcSelect, 0, "P", this); API.Call(xlcCellProtection, 0, "B", lock);}

		bool getHidden(){xlVar v; API.Call(xlfGetCell, &v, "IP", 15, this); return v.Bool;}
		void putHidden(bool hidden){API.Call(xlcSelect, 0, "P", this); API.Call(xlcCellProtection, 0, "_B", hidden);}

		int getRow(){return XL_CALL_FUNC_0(f_getRow);}
		int getColumn(){return XL_CALL_FUNC_0(f_getColumn);}

		// operators
		void operator =(const xlVar& val){putValue(val);}
		void operator =(const double val){putValue(val);}
		void operator =(const int val){putValue(val);}
		void operator =(const char* val){putValue(val);}
		void operator =(const wchar_t* val){putValue(val);}
		void operator =(const std::string val){putValue(val);}
		void operator =(const bool val){putValue(val);}

		// public methods
		xlRange& ChangeRef(const int row, const int col){
			if(m_excel12)f_ChangeRef<xloper12>(row, col, row, col); 
			else         f_ChangeRef<xloper4>(row, col, row, col); 
			return *this;
		}

		xlRange& ChangeRef(const int rFirst, const int cFirst, const int rLast, const int cLast){
			if(m_excel12)f_ChangeRef<xloper12>(rFirst, cFirst, rLast, cLast); 
			else         f_ChangeRef<xloper4>(rFirst, cFirst, rLast, cLast); 
			return *this;
		}

		int CountColumns(){ return XL_CALL_FUNC_0(f_CountColumns); }
		int CountRows(){ return XL_CALL_FUNC_0(f_CountRows); }

		void Select(){ API.Call(xlcSelect,0, "P", this);}
		void ClearAll(){ Select(); API.Call(xlcClear, 0, "I", 1);}
		void ClearFormats(){ Select(); API.Call(xlcClear, 0, "I", 2);}
		void ClearContents(){ Select(); API.Call(xlcClear, 0, "I", 3);}
		void ClearComments(){ Select(); API.Call(xlcClear, 0, "I", 4);}
		void Delete(){ Select(); API.Call(xlcEditDelete, 0, 0);}
		void Delete(int shiftNum){ Select(); API.Call(xlcEditDelete, 0, "I", shiftNum);}
		void Cut(){ API.Call(xlcCut, 0, "P", this);}
		void Cut(xlRange& pasteTo){ API.Call(xlcCut, 0, "PP", this, &pasteTo);}
		void Copy(){ API.Call(xlcCopy, 0, "P", this);}
		void Copy(xlRange& pasteTo){ API.Call(xlcCopy, 0, "PP", this, &pasteTo);}
		void Paste(){ API.Call(xlcPaste, 0, "P", this);}
		void PasteSpecial(const int pasteNum, const int operationNum, const bool skipBlanks, const bool transpose){
			Select(); API.Call(xlcPasteSpecial, 0, "IIBB", pasteNum, operationNum, skipBlanks, transpose);
		}
		void Insert(){ Select(); API.Call(xlcInsert, 0, 0);}
		void Insert(int shiftNum){ Select(); API.Call(xlcInsert, 0, "I", shiftNum);}
		/* 
			shiftNum :
				1 Shift cells right 
				2 Shift cells down 
				3 Shift entire row 
				4 Shift entire column

			pasteNum :
				1 All 
				2 Formulas 
				3 Values 
				4 Formats 
				5 Comments 
				6 All except borders

			operationNum :
				1 None 
				2 Add 
				3 Subtract 
				4 Multiply 
				5 Divide 
		*/
    private:

		// constructor implements

		void createSref(const int rFirst, const int cFirst, const int rLast, const int cLast){
			if(m_excel12) {
				f_setSubstance<xloper12>();
				f_setSref<xloper12>(rFirst,cFirst,rLast,cLast);
			}else{
				f_setSubstance<xloper4>();
				f_setSref<xloper>(rFirst,cFirst,rLast,cLast);
			}
		}

		void createRef(const int rFirst, const int cFirst, const int rLast, const int cLast,const xlRange* sheetId){
			if(m_excel12){
				f_setSubstance<xloper12>();
				f_setRef<xloper12, XLMREF12>(rFirst,cFirst,rLast,cLast, sheetId);
			}else{
				f_setSubstance<xloper4>();
				f_setRef<xloper, XLMREF>(rFirst,cFirst,rLast,cLast, sheetId);
			}
		}

		void createAsSheetId(IDSHEET id){
			if(m_excel12) {
				f_setSubstance<xloper12>();
				f_setRef<xloper12, XLMREF12>(0, 0, 0, 0, id);
			}else{
				f_setSubstance<xloper4>();
				f_setRef<xloper, XLMREF>(0, 0, 0, 0, id);
			}
		}

		void createFromxlVar(xlVar& var){
			// need error check
			if (var.XlType & (xltypeRef | xltypeSRef)){
				m_xloper = var.m_xloper4;
				m_refcount = var.m_refcount;
				*m_refcount += 1;
			}else{
				m_excel12 ? f_createErr<xloper12>(xlerrRef) : f_createErr<xloper4>(xlerrRef) ;
			}
		}

		//property imprements

		template <class P, class F> void f_getValue(xlVar* val){
			P tmp; 
			tmp.xltype = xltypeInt; 
			tmp.val.w = 5;
			P* arg[2] = {&tmp, (P*)m_xloper};
			P pret; pret.xltype = xltypeMissing;
			int r = ((F)(m_xlproc))(xlfGetCell, 2, &arg[0], &pret);
			m_excel12 ? f_clone12(val->m_xloper12, (xloper12*)&pret) : f_clone4(val->m_xloper4, (xloper*)&pret);
			f_xlfree<P, F>(&pret);
		}

		void f_getValue4(xlVar* val){
			xloper4 tmp; 
			tmp.xltype = xltypeInt; 
			tmp.val.w = 5;
			xloper4* arg[2] = {&tmp, m_xloper4};
			xloper4 pret;
			pret.xltype = xltypeMissing;
			int r = ((XLPROC4)(m_xlproc))(xlfGetCell, 2, &arg[0], &pret);
			f_clone4(val->m_xloper4, &pret);
			f_xlfree<xloper4, XLPROC4>(&pret);
		}

		void f_getValue12(xlVar* val){
			xloper12 tmp; 
			tmp.xltype = xltypeInt; 
			tmp.val.w = 5;
			xloper12* arg[2] = {&tmp, m_xloper12};
			xloper12 pret;
			pret.xltype = xltypeMissing;
			int r = ((XLPROC12)(m_xlproc))(xlfGetCell, 2, &arg[0], &pret);
			f_clone12(val->m_xloper12, &pret);
			f_xlfree<xloper12, XLPROC12>(&pret);
		}

		template <class P, class F> void f_putValue(const xlVar* val){
			P* arg[2] = {(P*)m_xloper, (P*)(val->m_xloper)};
			((F)m_xlproc)(xlSet, 2, &arg[0], 0);
		}

		template <class P, class F> void f_getFormula(xlVar* val){
			P* arg[1]={(P*)m_xloper};
			P pret; pret.xltype = xltypeMissing;
			((F)(m_xlproc))(xlfGetFormula, 1, &arg[0], (P*)&pret);
			m_excel12 ? f_clone12(val->m_xloper12, (xloper12*)&pret) : f_clone4(val->m_xloper4, (xloper*)&pret);
			f_xlfree<P, F>(&pret);
		}

		template <class P, class F> void f_putFormula(const xlVar* val){
			P* arg[2] = {(P*)(val->m_xloper), (P*)m_xloper};
			((F)m_xlproc)(xlcFormulaFill, 2, &arg[0], 0);
		}
		template <class P, class F> int f_getRow(){
			if(((P*)m_xloper)->xltype == xltypeSRef) return ((P*)m_xloper)->val.sref.ref.rwFirst;
			else return ((P*)m_xloper)->val.mref.lpmref->reftbl->rwFirst;
		}

		template <class P, class F> int f_getColumn(){
			if(((P*)m_xloper)->xltype == xltypeSRef) return ((P*)m_xloper)->val.sref.ref.colFirst;
			else return ((P*)m_xloper)->val.mref.lpmref->reftbl->colFirst;
		}

		//internal methods

		template <class P> void f_setSref(const int rFirst, const int cFirst, const int rLast, const int cLast){
			((P*)m_xloper)->xltype = xltypeSRef;
			((P*)m_xloper)->val.sref.ref.rwFirst = rFirst;
			((P*)m_xloper)->val.sref.ref.colFirst = cFirst;
			((P*)m_xloper)->val.sref.ref.rwLast = rLast;
			((P*)m_xloper)->val.sref.ref.colLast = cLast;
			((P*)m_xloper)->val.sref.count = 1;
		}

		template <class P, class M> void f_setRef(const int rFirst, const int cFirst, const int rLast, const int cLast, const xlRange* sheetId){
			((P*)m_xloper)->xltype = xltypeRef;
			((P*)m_xloper)->val.mref.idSheet = ((P*)(sheetId->m_xloper))->val.mref.idSheet;
			((P*)m_xloper)->val.mref.lpmref = new M;
			((P*)m_xloper)->val.mref.lpmref->count = 1;
			((P*)m_xloper)->val.mref.lpmref->reftbl->rwFirst = rFirst;
			((P*)m_xloper)->val.mref.lpmref->reftbl->colFirst = cFirst;
			((P*)m_xloper)->val.mref.lpmref->reftbl->rwLast = rLast;
			((P*)m_xloper)->val.mref.lpmref->reftbl->colLast = cLast;
		}

		template <class P, class M> void f_setRef(const int rFirst, const int cFirst, const int rLast, const int cLast, const IDSHEET id){
			((P*)m_xloper)->xltype = xltypeRef;
			((P*)m_xloper)->val.mref.idSheet = id;
			((P*)m_xloper)->val.mref.lpmref = new M;
			((P*)m_xloper)->val.mref.lpmref->count = 1;
			((P*)m_xloper)->val.mref.lpmref->reftbl->rwFirst = rFirst;
			((P*)m_xloper)->val.mref.lpmref->reftbl->colFirst = cFirst;
			((P*)m_xloper)->val.mref.lpmref->reftbl->rwLast = rLast;
			((P*)m_xloper)->val.mref.lpmref->reftbl->colLast = cLast;
		}

		template <class P> void f_ChangeRef(const int rFirst, const int cFirst,const int rLast,const int cLast){
			if(((P*)m_xloper)->xltype == xltypeSRef) {
				((P*)m_xloper)->val.sref.ref.rwFirst = rFirst;
				((P*)m_xloper)->val.sref.ref.colFirst = cFirst;
				((P*)m_xloper)->val.sref.ref.rwLast = rLast;
				((P*)m_xloper)->val.sref.ref.colLast = cLast;
			}else {
				((P*)m_xloper)->val.mref.lpmref->reftbl->rwFirst = rFirst;
				((P*)m_xloper)->val.mref.lpmref->reftbl->colFirst = cFirst;
				((P*)m_xloper)->val.mref.lpmref->reftbl->rwLast = rLast;
				((P*)m_xloper)->val.mref.lpmref->reftbl->colLast = cLast;
			}
		}

		template <class P, class F> int f_CountColumns(){
			if(((P*)m_xloper)->xltype == xltypeSRef) 
				return ((P*)m_xloper)->val.sref.ref.colLast-((P*)m_xloper)->val.sref.ref.colFirst + 1;
			else
				return ((P*)m_xloper)->val.mref.lpmref->reftbl->colLast-((P*)m_xloper)->val.mref.lpmref->reftbl->colFirst + 1;
		}

		template <class P, class F> int f_CountRows(){
			if(((P*)m_xloper)->xltype == xltypeSRef) 
				return ((P*)m_xloper)->val.sref.ref.rwLast-((P*)m_xloper)->val.sref.ref.rwFirst + 1;
			else
				return ((P*)m_xloper)->val.mref.lpmref->reftbl->rwLast-((P*)m_xloper)->val.mref.lpmref->reftbl->rwFirst + 1;
		}
	};
}