package cerot.blight.view.sequence;

import java.awt.Color;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;

import javax.swing.JComponent;

import cerot.blight.btrace.MethodExpression;
import cerot.blight.view.sequence.entity.Connector;
import cerot.blight.view.sequence.entity.ConnectorType;
import cerot.blight.view.sequence.entity.EndPoint;
import cerot.blight.view.sequence.entity.Node;

import com.mxgraph.swing.mxGraphComponent;
import com.mxgraph.util.mxConstants;
import com.mxgraph.util.mxRectangle;
import com.mxgraph.util.mxUtils;
import com.mxgraph.view.mxGraph;

/**
 * シーケンス図を作成するクラス。
 * @author cero-t
 * 
 */
public class SequenceCreator {
	/** 呼び出し線のスタイル */
	private static final String STYLE_CALL;

	/** 戻り線のスタイル */
	private static final String STYLE_RETURN;

	/** 生存線のスタイル */
	private static final String STYLE_LIFELINE;

	/** ノードの幅 */
	private int nodeWidth = 80;

	/** ノードの高さ */
	private int nodeHeight = 20;

	/** ノードの間隔 */
	private int nodeInterval = 10;

	/** 結線の間隔 */
	private int edgeInterval = 25;

	/** グラフ */
	private mxGraph graph;

	/** ノードセルのマップ */
	private Map<Node, List<Object>> endPointMap = new LinkedHashMap<Node, List<Object>>();

	/** 端点のY座標のマップ */
	private Map<Object, Integer> yposMap = new HashMap<Object, Integer>();

	/** 結線の数 */
	private int edgeCount;

	static {
		String styleCall = mxUtils.setStyle(null, mxConstants.STYLE_ENDARROW,
				mxConstants.ARROW_OPEN);
		styleCall = mxUtils.setStyle(styleCall,
				mxConstants.STYLE_VERTICAL_ALIGN, mxConstants.ALIGN_BOTTOM);
		STYLE_CALL = styleCall;

		String styleReturn = mxUtils.setStyle(null, mxConstants.STYLE_ENDARROW,
				mxConstants.ARROW_OPEN);
		styleReturn = mxUtils.setStyle(styleReturn,
				mxConstants.STYLE_VERTICAL_ALIGN, mxConstants.ALIGN_BOTTOM);
		styleReturn = mxUtils.setStyle(styleReturn, mxConstants.STYLE_DASHED,
				"1");
		STYLE_RETURN = styleReturn;

		String styleLifeLine = mxUtils.setStyle(null,
				mxConstants.STYLE_ENDARROW, mxConstants.NONE);
		STYLE_LIFELINE = styleLifeLine;
	}

	/**
	 * コンストラクタ。モデルの変換を行います。
	 * @param methodExpressionList メソッド表現のリスト
	 */
	public SequenceCreator(List<MethodExpression> methodExpressionList) {
		List<Connector> connectorList = SequenceConverter
				.toConnectorList(methodExpressionList);
		this.graph = new mxGraph();
		this.graph.setAllowDanglingEdges(false);
		this.graph.setEdgeLabelsMovable(false);
		this.graph.setKeepInsideParentOnMove(false);

		for (Connector connector : connectorList) {
			createConnectorCell(connector);
		}

		int count = 0;
		for (Map.Entry<Node, List<Object>> entry : this.endPointMap.entrySet()) {
			Node node = entry.getKey();
			Object eolCell = createEndPointCell(node);

			Object[] endPoints = entry.getValue().toArray();
			Object group = createNodeGroup(count, node, endPoints);

			recalculateEndPoint(entry.getValue());

			// 生存線の描画
			this.graph.insertEdge(group, null, null, group, eolCell,
					STYLE_LIFELINE);

			count++;
		}
	}

	/**
	 * ノード（端点のグループ）を作成します。
	 * @param count ノード数
	 * @param node ノード
	 * @param endPoints 端点の配列
	 * @return グループ
	 */
	private Object createNodeGroup(int count, Node node, Object[] endPoints) {
		Object result = this.graph.group(null, 0, endPoints);

		// X座標 = 左端からの余白 + ノード数 * (ノード幅 + ノード間隔)
		int x = this.nodeInterval + count
				* (this.nodeWidth + this.nodeInterval);

		// Y座標 = 上端からの余白
		int y = this.nodeInterval;

		this.graph.resize(result, new mxRectangle(x, y, this.nodeWidth,
				this.nodeHeight));
		this.graph.getModel().setValue(result, node.getName());
		return result;
	}

	/**
	 * 端点の座標計算を再計算します。<br>
	 * グループ化により座標が変化するため、グループ化後に再計算します。
	 * @param endPointList 端点のリスト
	 */
	private void recalculateEndPoint(List<Object> endPointList) {
		for (Object endPoint : endPointList) {
			double dx = (double) this.nodeWidth / 2;
			double dy = this.yposMap.get(endPoint).doubleValue();
			this.graph.move(new Object[] { endPoint }, dx, dy);
		}
	}

	/**
	 * グラフの描画を行います。
	 * @return グラフコンポーネント
	 */
	public JComponent createComponent() {
		mxGraphComponent component = new mxGraphComponent(this.graph);
		component.getViewport().setBackground(Color.WHITE);
		return component;
	}

	/**
	 * 結線を作成します。
	 * @param connector 結線モデル
	 */
	private void createConnectorCell(Connector connector) {
		EndPoint from = connector.getFrom();
		EndPoint to = connector.getTo();

		Object fromCell = createEndPointCell(from.getParentNode());
		Object toCell = createEndPointCell(to.getParentNode());

		Object parent = this.graph.getDefaultParent();

		if (connector.getConnectorType() == ConnectorType.METHOD_RETURN) {
			this.graph.insertEdge(parent, null, connector.getName(), fromCell,
					toCell, STYLE_RETURN);
		} else {
			this.graph.insertEdge(parent, null, connector.getName(), fromCell,
					toCell, STYLE_CALL);
		}

		this.edgeCount++;
	}

	/**
	 * 端点のセルを作成します。
	 * @param node 端点が所属するノード
	 * @return 端点セルオブジェクト
	 */
	private Object createEndPointCell(Node node) {
		Object parent = this.graph.getDefaultParent();
		Object result = this.graph.insertVertex(parent, null, null, 0, 0, 0, 0);

		// Y座標 = 上端からの余白 + ノードの高さ + (結線の数 + 1) * 結線の間隔
		int y = this.nodeInterval + this.nodeHeight + (this.edgeCount + 1)
				* this.edgeInterval;
		this.yposMap.put(result, Integer.valueOf(y));

		// 端点をノードグループに追加
		List<Object> endPointList = this.endPointMap.get(node);
		if (endPointList == null) {
			endPointList = new ArrayList<Object>();
			this.endPointMap.put(node, endPointList);
		}
		endPointList.add(result);

		return result;
	}
}
