package cerot.blight.sequence;

import java.awt.Color;
import java.awt.geom.Rectangle2D;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import cerot.blight.sequence.entity.Connector;
import cerot.blight.sequence.entity.ConnectorType;
import cerot.blight.sequence.entity.EndPoint;
import cerot.blight.sequence.entity.Node;

import org.jgraph.JGraph;
import org.jgraph.graph.DefaultEdge;
import org.jgraph.graph.DefaultGraphCell;
import org.jgraph.graph.DefaultGraphModel;
import org.jgraph.graph.GraphConstants;
import org.jgraph.graph.GraphModel;

public class SequenceCreator {
	/** 破線のパターン */
	private static final float[] DASH_PATTERN = { 5, 5 };

	/** ノードの幅 */
	private int nodeWidth = 80;

	/** ノードの高さ */
	private int nodeHeight = 20;

	/** ノードの間隔 */
	private int nodeInterval = 10;

	/** 結線の間隔 */
	private int edgeInterval = 25;

	/** ノードセルのマップ */
	private Map<Node, DefaultGraphCell> nodeMap = new HashMap<Node, DefaultGraphCell>();

	/** 全オブジェクトのリスト */
	List<DefaultGraphCell> cellList = new ArrayList<DefaultGraphCell>();

	/** 結線の数 */
	private int edgeCount;

	/**
	 * コンストラクタ。モデルの変換を行います。
	 * @param connectorList 結線のリスト
	 */
	public SequenceCreator(List<Connector> connectorList) {
		for (Connector connector : connectorList) {
			createConnectorCell(connector);
		}

		// 活性線の作成
		for (DefaultGraphCell nodeCell : this.nodeMap.values()) {
			DefaultGraphCell endCell = createEndPointCell(nodeCell);

			DefaultEdge edge = new DefaultEdge();
			edge.setSource(nodeCell.getChildAt(0));
			edge.setTarget(endCell.getChildAt(0));
			this.cellList.add(edge);
		}
	}

	/**
	 * グラフの描画を行います。
	 */
	public JGraph createGraph() {
		GraphModel model = new DefaultGraphModel();
		JGraph graph = new JGraph(model);
		graph.setJumpToDefaultPort(true);
		graph.getGraphLayoutCache().insert(this.cellList.toArray());

		return graph;
	}

	private void createConnectorCell(Connector connector) {
		EndPoint from = connector.getFrom();
		EndPoint to = connector.getTo();

		DefaultGraphCell fromNodeCell = findParentNodeCell(from);
		DefaultGraphCell fromCell = createEndPointCell(fromNodeCell);

		DefaultGraphCell toNodeCell = findParentNodeCell(to);
		DefaultGraphCell toCell = createEndPointCell(toNodeCell);

		DefaultEdge edge = new DefaultEdge(connector.getName());
		edge.setSource(fromCell.getChildAt(0));
		edge.setTarget(toCell.getChildAt(0));
		GraphConstants.setLineEnd(edge.getAttributes(),
				GraphConstants.ARROW_CLASSIC);
		GraphConstants.setLabelAlongEdge(edge.getAttributes(), true);

		if (connector.getConnectorType() == ConnectorType.METHOD_RETURN) {
			GraphConstants.setDashPattern(edge.getAttributes(), DASH_PATTERN);
		}

		this.cellList.add(edge);
		this.edgeCount++;
	}

	private DefaultGraphCell findParentNodeCell(EndPoint endPoint) {
		Node node = endPoint.getParentNode();
		DefaultGraphCell result = findNodeCell(node);

		return result;
	}

	private DefaultGraphCell createEndPointCell(DefaultGraphCell nodeCell) {
		DefaultGraphCell result = new DefaultGraphCell();

		// X座標 = ノードの中央値
		Rectangle2D rectangle = (Rectangle2D) nodeCell.getAttributes().get(
				GraphConstants.BOUNDS);
		double x = rectangle.getCenterX();

		// Y座標 = 上端からの余白 + ノードの高さ + (結線の数 + 1) * 結線の間隔
		int y = this.nodeInterval + this.nodeHeight + (this.edgeCount + 1)
				* edgeInterval;

		GraphConstants.setBounds(result.getAttributes(),
				new Rectangle2D.Double(x, y, 0, 0));
		result.addPort();

		this.cellList.add(result);
		return result;
	}

	private DefaultGraphCell findNodeCell(Node node) {
		DefaultGraphCell result = this.nodeMap.get(node);

		if (result == null) {
			result = new DefaultGraphCell(node.getName());

			// X座標 = 左端からの余白 + ノード数 * (ノード幅 + ノード間隔)
			int x = this.nodeInterval + this.nodeMap.size()
					* (this.nodeWidth + this.nodeInterval);

			// Y座標 = 上端からの余白
			int y = this.nodeInterval;

			GraphConstants.setBounds(result.getAttributes(),
					new Rectangle2D.Double(x, y, nodeWidth, nodeHeight));
			GraphConstants.setBorderColor(result.getAttributes(), Color.black);

			this.nodeMap.put(node, result);
			this.cellList.add(result);
			result.addPort();
		}

		return result;
	}
}
