/*
**  bif_vm.h
**  bif-c
**
**  Created by Joel Rees on 2009/07/22.
**  Copyright 2009 __Reiisi_Kenkyuu__. All rights reserved.
**
** Translated to C from BIFDP/A, as mechanically as possible.
**
*/


#if !defined BIF_VM_H
#define BIF_VM_H


#include "bifu_i.h"
#include "bif_m.h"


/* Since this is where most of the inner interpreter got defined, 
** I'll define the virtual machine registers here, too.
** Notes from BIFDOC.TXT:
*******************************************************************************
                        The BIF Virtual Machine

fig     6809
UP      [DP]    pointer to the per-USER variable table (USER Pointer)
IP      Y       pointer to the next definition (Instruction Pointer)
RP      S       return/control stack pointer
SP      U       parameter/data stack pointer
W       [S]     pointer to executing definition's parameter field
*/


extern cell_u	UP;	/* [DP]    pointer to the per-USER variable table (USER Pointer) */
extern cell_u	* IP;	/* Y       Pointer to the next definition to execute (Instruction Pointer). */
extern cell_u	* RP;	/* S       return/control stack pointer */
extern cell_u	* SP;	/* U       parameter/data stack pointer */
/* W was originally (ephemeral) on the 6809 stack because there weren't enough registers. */
extern cell_u	W;	/* [S]     pointer to executing definition's parameter field */
extern volatile cell_u	sysSIG;	/* Added as a way to break out of the inner interpreter. */
#define	ICODE_LIST_INITIALIZATION_ERROR	-0x4000L
#define	ICODE_LIST_NULL_EXECUTION	-0x2000L
#define	ICODE_LIST_RETURN_ERROR	-256
#define	ICODE_LIST_NOT_AVAILABLE_ERROR	-2
#define	ICODE_LIST_GENERAL_ERROR	-1
#define	ICODE_LIST_CONTINUE	0
#define	ICODE_LIST_END	1
/* Positive is to quit n levels of list.
** Negative is to throw a hard exception (beyond ERROR). */


#define SPARE_REG_COUNT	8	/* Don't know if these will be useful. */
extern cell_u spares[ SPARE_REG_COUNT ];


#if defined DBG_TRACE_NEXT
extern int isTracing( void );
extern void dumpState( char * tag );

extern definition_header_s hTRACEON;
extern void TRACEON( void );
extern definition_header_s hTRACEOFF;
extern void TRACEOFF( void );
#endif


/* The inner interpreter.
XNEXT    ( --- )         
	Initiates an inner interpretation loop.
** The inner interpretor performs a C call, 
** so the effective jump back to NEXT is the end of the loop block.
*/
/* Pulled back out of the WARM boot code.
*/
extern inline void XNEXT (void);

/* Nest into an icode list.
XCOL   ( *** IP )
        Characteristic of a colon (:) definition.  Begins execution of a
        non-leaf definition, i. e., nests the definition and begins
        processing icodes. 
*/
extern void XCOL(void);

/* Push the first cell from the parameter field as a constant.
DOCON   ( --- n )
        Characteristic of a CONSTANT.  A CONSTANT simply loads its value
        from its parameter field and pushes it on the stack.
*/
extern void XCON(void);

/* Push the address of the first cell of the parameter field as a variable.
DOVAR   ( --- vadr )    jsr <XVAR (bif.m, bifdp.a)
        Characteristic of a VARIABLE.  A VARIABLE pushes its PFA address
        on the stack.  The parameter field of a VARIABLE is the actual
        allocation of the variable, so that pushing its address allows
        its contents to be @ed (fetched).  Ordinary arrays and strings
        that do not subscript themselves may be allocated by defining a
        variable and immediately ALLOTting the remaining space.
        VARIABLES are global to all users, and thus should have been
        hidden in resource monitors, but aren't.
*/
extern void XVAR(void);

/* Push the address of a the per-task variable, base of USER PAGE + offset.
*/
extern void XUSER(void);

/* Push the address of the element of a 1 dimensional bounded array.
*/
#define LINEARRAY_DATAOFFSET	3	/* Except this really shouldn't be visible, I guess. */
extern void X1ARR(void);

/* Push the value of a user variable as a constant.
*/
extern void XUCON(void);

/* Stores a pointer to itself in the search context root pointer.
*/
extern void XVOC(void);

/* Start into a characteristic that is defined high-level.
*/
extern void XDOES(void);

/* Push the first two cells from the parameter field as a double-width constant.
*/
extern void XDCON(void);

/* C code to call a definition through its header, setting and restoring W.
** But, use mERROR(), the macro wrapped EXEC(), instead.
void callDefinition( definition_header_s * headerp );
*/

extern definition_header_s hLIT;
extern void LIT(void);

extern definition_header_s hDLIT;
extern void DLIT(void);

#define mCALLcell( cell )	( ( * ( W = cell ).definitionp ).codeLink.icode )()
#define mCALLdefp( headerp )	( ( * ( W.definitionp = headerp ) ).codeLink.icode )()
#define mCALLdef( header )	mCALLdefp( &header )

extern definition_header_s hEXEC;
extern void EXEC(void);
/* Doesn't really make sense to go to the extra effort of calling EXEC, 
** and then there is the recursion problem when EXEC calls ERROR.
** On the other hand, if we don't know whether we are trying to call out to NULL pointers, these might be useful.
** If so, I'll un-comment them. 
#define	mEXECdef( def )	\
{	( * --SP ).definitionp = def;	\
	EXEC();	\
}
#define	mEXECcell( defdell )	\
{	( * --SP ).definitionp = defcell.definitionp;	\
	EXEC();	\
}
*/

extern definition_header_s hTBR;
extern void TBR(void);

extern definition_header_s hBRANCH;
extern void BRANCH(void);

extern definition_header_s hZBR;
extern void ZBR(void);

extern definition_header_s hXLOOP;
extern void XLOOP(void);

extern definition_header_s hXPLOOP;
extern void XPLOOP(void);

extern definition_header_s hXDO;
extern void XDO(void);

extern definition_header_s hXSCODE;
extern void XSCODE(void);

extern definition_header_s hMOVE;
extern void MOVE(void);

extern definition_header_s hCMOVE;
extern void CMOVE(void);

#define LASTinVM	hCMOVE


#endif /* !defined BIF_VM_H */
