/*
**  makecelltype.c
**  bif-c
**
**  Created by Joel Rees on 2013/06/18.
**  Copyright 2013 __Reiisi_Kenkyuu__. All rights reserved.
**
**  Generate type information for cell proto-type.
*/


#include <limits.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>


typedef char * prCharPtr_p;
typedef void * prVoidPtr_p;
typedef void (* icode_f)(void);
/* The above type declarations as strings, MUST BE IDENTICAL: */
char * prPtrTpDecs[] = 
{
"char * prCharPtr_p",
"void * prVoidPtr_p",
"void (* icode_f )( void )"
};
/* The above type names as strings, MUST BE IDENTICAL: */
char * prPtrTypes[] = 
{
"prCharPtr_p",
"prVoidPtr_p",
"icode_f"
};
#define PRCHARPTR 0
#define PRVOIDPTR 1
#define PRVFPTR 2
/* Tricky macro substitution might eliminate the duplication, 
// but we only need the above three.
*/


typedef struct typeBits_s
{  int byteSize;
   int shortSize;
   int intSize;
   int longSize;
   int llongSize;
   int pointerSize;
   int cellSize;
} typeBits_t;


#if defined ULLONG_MAX

#endif /* defined ULLONG_MAX */



typedef unsigned char ubyte_t;
typedef signed char sbyte_t;

int probe_byte( FILE * outfile, typeBits_t * bits )
{  ubyte_t uch = (ubyte_t) -1;
   sbyte_t schmax = (sbyte_t) (uch >> 1);
   unsigned long memory = 0;
   int ui = 0;
   int status = 0;

   /* Must be identical to the two types declared above: */
   fputs( "typedef unsigned char ubyte_t;\n", outfile );
   fputs( "typedef signed char sbyte_t;\n", outfile );
   fputs( "/* For now, assuming two's complement: */\n", outfile );
   fprintf( outfile, "#define UBYTE_MAX ((ubyte_t) 0x%x)\n", uch );
   for ( uch = (ubyte_t) -1, ui = 0; 
         ( ui <= 1024 ) && ( uch != 0 ); 
         )
   {  memory = uch;
      uch <<= 1;
      ++ui;
   }
   if ( ui != CHAR_BIT )
   {  status = -1;
      fprintf( outfile, "#error \"*** C CHAR_BIT is %d but actual byte width is %d. ***\"\n", CHAR_BIT, ui ); 
   }
   bits->byteSize = ui;
   fprintf( outfile, "#define BITSPERBYTE %d\n", ui );
   fprintf( outfile, "#define BYTE_HIGH_BIT ((ubyte_t) 0x%x)\n", 
            (ubyte_t) memory );
   if ( ( schmax != ((ubyte_t) ~memory) )
        || ( ( ((ubyte_t) schmax) + 1 ) != memory ) )
   {  status = -1;
      fputs( "#error \"*** Not two's complement! Needs fixing. ***\"\n",
             outfile );
      fprintf( outfile, "#error \"filled bit shifted right: %x, rollover: %x\"\n",
               schmax, ((ubyte_t) schmax) + 1 );
      fprintf( outfile, "#error \"1 max shifted left: %x, inverted: %x\"\n",
               ((ubyte_t) memory), ((ubyte_t) ~memory) );
   }
   fprintf( outfile, "#define SBYTE_MAX ((sbyte_t) 0x%x)\n", 
            (ubyte_t) schmax );
   fputs( "#define SBYTE_MIN ((sbyte_t) BYTE_HIGH_BIT)\n\n\n", outfile );
   return status;
}


typedef unsigned short ushortw_t;
typedef signed short sshortw_t;

int probe_shortw( FILE * outfile, typeBits_t * bits )
{  ushortw_t ush = (ushortw_t) -1;
   sshortw_t sshmax = (sshortw_t) (ush >> 1);
   unsigned long memory = 0;
   int ui = 0;
   int status = 0;

   /* Must be identical to the two types declared above: */
   fputs( "typedef unsigned short ushortw_t;\n", outfile );
   fputs( "typedef signed short sshortw_t;\n", outfile );
   fputs( "/* For now, assuming two's complement: */\n", outfile );
   fprintf( outfile, "#define USHORTW_MAX ((ushortw_t) 0x%x)\n", ush );
   for ( ush = (ushortw_t) -1, ui = 0; 
         ( ui <= 1024 ) && ( ush != 0 ); 
         )
   {  memory = ush;
      ush <<= 1;
      ++ui;
   }
   bits->shortSize = ui;
   fprintf( outfile, "#define BITSPERSHORTW %d\n", ui );
   fprintf( outfile, "#define SHORTW_HIGH_BIT ((ushortw_t) 0x%x)\n", 
            (ushortw_t) memory );
   if ( ( sshmax != ((ushortw_t) ~memory) )
        || ( ( ((ushortw_t) sshmax) + 1 ) != memory ) )
   {  status = -1;
      fputs( "#error \"*** Not two's complement! Needs fixing. ***\"\n",
             outfile );
      fprintf( outfile, "#error \"filled bit shifted right: %x, rollover: %x\"\n",
               sshmax, ((ushortw_t) sshmax) + 1 );
      fprintf( outfile, "#error \"1 max shifted left: %x, inverted: %x\"\n",
               ((ushortw_t) memory), ((ushortw_t) ~memory) );
   }
   fprintf( outfile, "#define SSHORTW_MAX ((sshortw_t) 0x%x)\n", 
            (ushortw_t) sshmax );
   fputs( "#define SSHORTW_MIN ((sshortw_t) SHORTW_HIGH_BIT)\n\n\n", outfile );
   return status;
}


typedef unsigned int uintw_t;
typedef signed int sintw_t;

int probe_intw( FILE * outfile, typeBits_t * bits )
{  uintw_t uiw = (uintw_t) -1;
   sintw_t siwmax = (sintw_t) (uiw >> 1);
   unsigned long memory = 0;
   int ui = 0;
   int status = 0;

   /* Must be identical to the two types declared above: */
   fputs( "typedef unsigned int uintw_t;\n", outfile );
   fputs( "typedef signed int sintw_t;\n", outfile );
   fputs( "/* For now, assuming two's complement: */\n", outfile );
   fprintf( outfile, "#define UINTW_MAX ((uintw_t) 0x%x)\n", uiw );
   for ( uiw = (uintw_t) -1, ui = 0; 
         ( ui <= 1024 ) && ( uiw != 0 ); 
         )
   {  memory = uiw;
      uiw <<= 1;
      ++ui;
   }
   bits->intSize = ui;
   fprintf( outfile, "#define BITSPERINTW %d\n", ui );
   fprintf( outfile, "#define INTW_HIGH_BIT ((uintw_t) 0x%x)\n", 
            (uintw_t) memory );
   if ( ( siwmax != ((uintw_t) ~memory) )
        || ( ( ((uintw_t) siwmax) + 1 ) != memory ) )
   {  status = -1;
      fputs( "#error \"*** Not two's complement! Needs fixing. ***\"\n",
             outfile );
      fprintf( outfile, "#error \"filled bit shifted right: %x, rollover: %x\"\n",
               siwmax, ((uintw_t) siwmax) + 1 );
      fprintf( outfile, "#error \"1 max shifted left: %x, inverted: %x\"\n",
               ((uintw_t) memory), ((uintw_t) ~memory) );
   }
   fprintf( outfile, "#define SINTW_MAX ((sintw_t) 0x%x)\n", 
            (uintw_t) siwmax );
   fputs( "#define SINTW_MIN ((sintw_t) INTW_HIGH_BIT)\n\n\n", outfile );
   return status;
}


typedef unsigned long ulongw_t;
typedef signed long slongw_t;

int probe_longw( FILE * outfile, typeBits_t * bits )
{  ulongw_t ulw = (ulongw_t) -1;
   slongw_t slwmax = (slongw_t) (ulw >> 1);
   unsigned long memory = 0;
   int ui = 0;
   int status = 0;

   /* Must be identical to the two types declared above: */
   fputs( "typedef unsigned long ulongw_t;\n", outfile );
   fputs( "typedef signed long slongw_t;\n", outfile );
   fputs( "/* For now, assuming two's complement: */\n", outfile );
   fprintf( outfile, "#define ULONGW_MAX ((ulongw_t) 0x%lxL)\n", ulw );
   for ( ulw = (ulongw_t) -1, ui = 0; 
         ( ui <= 1024 ) && ( ulw != 0 ); 
         )
   {  memory =  ulw;
      ulw <<= 1;
      ++ui;
   }
   bits->longSize = ui;
   fprintf( outfile, "#define BITSPERLONGW %d\n", ui );
   fprintf( outfile, "#define LONGW_HIGH_BIT ((ulongw_t) 0x%lxL)\n", 
            (ulongw_t) memory );
   if ( ( slwmax != ((ulongw_t) ~memory) )
        || ( ( ((ulongw_t) slwmax) + 1 ) != memory ) )
   {  status = -1;
      fputs( "#error \"*** Not two's complement! Needs fixing. ***\"\n",
             outfile );
      fprintf( outfile, "#error \"filled bit shifted right: %lx, rollover: %lx\"\n",
               slwmax, ((ulongw_t) slwmax) + 1 );
      fprintf( outfile, "#error \"1 max shifted left: %lx, inverted: %lx\"\n",
               ((ulongw_t) memory), ((ulongw_t) ~memory) );
   }
   fprintf( outfile, "#define SLONGW_MAX ((slongw_t) 0x%lxL)\n", 
            (ulongw_t) slwmax );
   fputs( "#define SLONGW_MIN ((slongw_t) LONGW_HIGH_BIT)\n\n\n", outfile );
   return status;
}


#if defined ULLONG_MAX
typedef unsigned long long ullongw_t;
typedef signed long long sllongw_t;

typedef union cell_order_probe_u
{  ulongw_t singleParts[ 2 ];
   ullongw_t doublePart;
} cell_order_probe_u;

int probe_llongw( FILE * outfile, typeBits_t * bits )
{
   cell_order_probe_u cellOrder = { { 0, 1 } };
   bits->llongSize = 0;
   ullongw_t udlw = (ullongw_t) -1;
   sllongw_t sdlwmax = (sllongw_t) (udlw >> 1);
   unsigned long long memory = 0;
   int ui = 0;
   int status = 0;

   /* Must be identical to the two types declared above: */
   fputs( "/* *** long long type supported on this architecture. *** */\n", 
          outfile );
   fputs( "typedef unsigned long long ullongw_t;\n", outfile );
   fputs( "typedef signed long long sllongw_t;\n", outfile );
   fputs( "/* For now, assuming two's complement: */\n", outfile );
   fprintf( outfile, "#define ULLONGW_MAX ((ullongw_t) 0x%llxLL)\n", udlw );
   for ( udlw = (ullongw_t) -1, ui = 0; 
         ( ui <= 1024 ) && ( udlw != 0 ); 
         )
   {  memory =  udlw;
      udlw <<= 1;
      ++ui;
   }
   bits->llongSize = ui;
   fprintf( outfile, "#define BITSPERLLONGW %d\n", ui );
   fprintf( outfile, "#define LLONGW_HIGH_BIT ((ullongw_t) 0x%llxLL)\n", 
            (ullongw_t) memory );
   if ( ( sdlwmax != ((ullongw_t) ~memory) )
        || ( ( ((ullongw_t) sdlwmax) + 1 ) != memory ) )
   {  status = -1;
      fputs( "#error \"*** Not two's complement! Needs fixing. ***\"\n",
             outfile );
      fprintf( outfile, "#error \"filled bit shifted right: %llx, rollover: %llx\"\n",
               sdlwmax, ((ullongw_t) sdlwmax) + 1 );
      fprintf( outfile, "#error \"1 max shifted left: %llx, inverted: %llx\"\n",
               ((ullongw_t) memory), ((ullongw_t) ~memory) );
   }
   fprintf( outfile, "#define SLLONGW_MAX ((sllongw_t) 0x%llxLL)\n", 
            (ullongw_t) sdlwmax );
   fputs( "#define SLLONGW_MIN ((sllongw_t) LLONGW_HIGH_BIT)\n\n\n", outfile );
   /* Add the cell order information: */
   if ( sizeof (ulongw_t) == sizeof (ullongw_t) )
   {  fputs( "#define SINGLE_C_CELL_DOUBLE\n\n\n", outfile );
   }
   else if ( ( sizeof (ulongw_t) * 2 ) != sizeof (ullongw_t) )
   {  status = -1;
      fprintf( outfile, "#error \"*** sizeof ullongw_t == %d, sizeof ulongw_t == %d, not double! ***\"\n",
               sizeof (ullongw_t), sizeof (ulongw_t) );
   }
   else if ( cellOrder.doublePart == 1 )
   {  fputs( "#define HIGH_C_CELL_FIRST\n\n\n", outfile );
   }
   else if ( cellOrder.doublePart == ( (ullongw_t) ULONG_MAX ) + 1 )
   {  fputs( "#define LOW_C_CELL_FIRST\n\n\n", outfile );
   }
   else
   {  status = -1;
      fputs( "#error \"*** Undecipherable double long cell order!!! ***\"\n\n\n", outfile );
   }
   return status;
}
#endif /* defined ULLONG_MAX */


char * headerName = "celltype.h";
char * sourceName;
char * headerMacro;

char * header[] = 
{
"/*",
"**  %s",
"**  bif-c",
"**",
"**  Template created by Joel Rees on 2013/06/18.",
"**  Copyright 2013 __Reiisi_Kenkyuu__. All rights reserved.",
"**",
"**  Type information for cell proto-type.",
"*/\n\n"
};

char * macroBlock[] =
{
"#if !defined %s \t/* recursive inclusion blocker */",
"#define %s\n\n"
};

/* includes are not so easy, don't benefit, anyway. */

char * macroEnd[] =
{
"#endif /* defined %s \trecursive inclusion blocker */\n"
};


/* argv[ 1 ] is optional header file name, including ".h".
*/
int main( int argc, char *argv[] )
{
   FILE * headerFile;
   FILE * sourceFile;
   int fileNameLength;
   int ptrSize = sizeof (icode_f);
   int ptrSizeTypeX = PRVFPTR;
   char * cellSizeType = "longw_t";
   char * cellSizeString = "LONGW";
   int cellSize = sizeof (long);
   int i;
   int exitCode = EXIT_SUCCESS;
   typeBits_t typeBits = { 0 };

   /* Open the files. */
   if ( argc > 1 )
      headerName = argv[ 1 ];	/* No error checking. */
   fileNameLength = strlen( headerName );
   if ( ( sourceName = malloc( fileNameLength + 1 ) ) == NULL )
      return EXIT_FAILURE;
   strncpy( sourceName, headerName, fileNameLength + 1 );
   sourceName[ fileNameLength - 1 ] = 'c';
   if ( ( headerMacro = malloc( fileNameLength + 1 ) ) == NULL )
      return EXIT_FAILURE;
   for ( i = 0; i < fileNameLength; ++i )
      headerMacro[ i ] = toupper( headerName[ i ] );
   headerMacro[ fileNameLength ] = '\0';
   headerMacro[ fileNameLength - 2 ] = '_';
   if ( ( headerFile = fopen( headerName, "w" ) ) == NULL )
      return EXIT_FAILURE;
   if ( ( sourceFile = fopen( sourceName, "w" ) ) == NULL )
   {  fclose( headerFile );
      return EXIT_FAILURE;
   }

   /* Write the file headers */
   for ( i = 0; i < sizeof header / sizeof header[ 0 ] ; ++i )
   {  fprintf( headerFile, header[ i ], headerName );
      fputc( '\n', headerFile );
      fprintf( sourceFile, header[ i ], sourceName );
      fputc( '\n', sourceFile );
   }
   for ( i = 0; i < sizeof macroBlock / sizeof macroBlock[ 0 ] ; ++i )
   {  fprintf( headerFile, macroBlock[ i ], headerMacro );
      fputc( '\n', headerFile );
   }

   /* The source file must include the header file: */
   fprintf( sourceFile, "#include %s\n", headerName );

   /* Add the probed pointer types to the header: */
   for ( i = 0; 
         i < sizeof prPtrTpDecs / sizeof prPtrTpDecs[ 0 ]; 
         ++i )
   {  fprintf( headerFile, "typedef %s;\n", prPtrTpDecs[ i ] );
   }

   /* Basis for a size for addresses: */
   if ( sizeof (prCharPtr_p) > sizeof (icode_f) ) 
   {  if ( sizeof (prVoidPtr_p) > sizeof (prCharPtr_p) )
      {  ptrSize = sizeof (prVoidPtr_p);
         ptrSizeTypeX = PRVOIDPTR;
      }
      else
      {  ptrSize = sizeof (prCharPtr_p);
         ptrSizeTypeX = PRCHARPTR;
      }
   }
   else 
   {  if ( sizeof (prVoidPtr_p) > sizeof (icode_f) ) 
      {  ptrSize = sizeof (prVoidPtr_p);
         ptrSizeTypeX = PRVOIDPTR;
      }
   }
   fprintf( headerFile, "#define PROBE_POINTER_BYTE %d\n", ptrSize );
   /* Assume that the pointer, in memory, is a byte multiple size. */
   typeBits.pointerSize = ptrSize * CHAR_BIT;
   fprintf( headerFile, "#define PROBE_POINTER_BIT %d\n", 
            typeBits.pointerSize );
   fprintf( headerFile, "#define PROBE_POINTER_TYPE %s\n\n\n", 
            prPtrTypes[ ptrSizeTypeX ] );

   /* Make the fundamental cell size available. 
   // Usually don't want to use int, use either short or long instead.
   */
   if ( ( ptrSize <= sizeof (ushortw_t) ) 
        && ( ( sizeof (ushortw_t) * 2 ) <= sizeof (ulongw_t) ) )
   {  cellSizeType = "shortw_t";
      cellSizeString = "SHORTW";
      cellSize = sizeof (ushortw_t);
   }
   else if ( ( ptrSize <= sizeof (uintw_t) )
        && ( ( sizeof (uintw_t) * 2 ) <= sizeof (ulongw_t) ) )
   {  cellSizeType = "intw_t";
      cellSizeString = "INTW";
      cellSize = sizeof (uintw_t);
   }
   else if ( ptrSize <= sizeof (ulongw_t) ) 
   {  cellSizeType = "longw_t";	/* This should be the usual case. */
      cellSizeString = "LONGW";
      cellSize = sizeof (ulongw_t);
   }
#if defined ULLONG_MAX
   else if ( ptrSize <= sizeof (ullongw_t) ) 
   {  cellSizeType = "llongw_t";
      cellSizeString = "LLONGW";
      cellSize = sizeof (ullongw_t);
      fputs( "#error \"*** Pointers are supposed to fit in unsigned long! ***\"\n", headerFile );
   }
#endif /* defined ULLONG_MAX */
   else
   {
      fprintf( headerFile, "#error \"*** Probed pointer size %d unsatisfiable! ***\"\n", ptrSize );
      fprintf( sourceFile, "#error \"*** Probed pointer size %d unsatisfiable! ***\"\n", ptrSize );
      printf( "*** Probed pointer size %d unsatisfiable! ***\n", ptrSize );
      exitCode = EXIT_FAILURE;
   }
   typeBits.cellSize = cellSize * CHAR_BIT;

   probe_byte( headerFile, &typeBits );
   probe_shortw( headerFile, &typeBits );
   probe_intw( headerFile, &typeBits );
   probe_longw( headerFile, &typeBits );
#if defined ULLONG_MAX
   probe_llongw( headerFile, &typeBits );
#endif /* defined ULLONG_MAX */

   fputs( "/* This is the primitive cell, \n", headerFile );
   fputs( "** not the target cell.\n*/\n", headerFile );
   fprintf( headerFile, "typedef s%s probe_scell_t;\n", cellSizeType );
   fprintf( headerFile, "typedef u%s probe_ucell_t;\n", cellSizeType );
   fprintf( headerFile, "#define PROBE_CELL_BYTE %d\n", cellSize );
   fprintf( headerFile, "#define PROBE_CELL_BIT BITSPER%s\n", 
            cellSizeString );
   fprintf( headerFile, "#define PROBE_CELL_HIGH_BIT ((probe_ucell_t) %s_HIGH_BIT)\n", 
            cellSizeString );
   fprintf( headerFile, "#define PROBE_UCELL_MAX ((probe_ucell_t) U%s_MAX)\n", 
            cellSizeString );
   fprintf( headerFile, "#define PROBE_SCELL_MAX ((probe_scell_t) S%s_MAX)\n", 
            cellSizeString );
   fprintf( headerFile, "#define PROBE_SCELL_MIN ((probe_scell_t) S%s_MIN)\n\n\n", 
            cellSizeString );

   fputs( "/* A manufactured double has to be constructed on the target.\n", headerFile );
   fputs( "** All we can do here is provide the context.\n**\n", headerFile );
   fputs( "** Cells are never smaller than C int.\n*/\n", headerFile );
   if ( typeBits.longSize >= 2 * typeBits.cellSize )
   {  fputs( "typedef ulongw_t udoublew_t;\n", headerFile );
      fputs( "typedef slongw_t sdoublew_t;\n", headerFile );
      fprintf( headerFile, "#define PROBE_DOUBLE_BYTE %d\n",
               (int) sizeof (ulongw_t) );
      fputs( "#define PROBE_DOUBLE_BIT BITSPERLONGW\n", headerFile );
      fputs( "#define PROBE_DOUBLE_HIGH_BIT ((udoublew_t) LONGW_HIGH_BIT)\n",
             headerFile );
      fputs( "#define PROBE_UDOUBLE_MAX ((udoublew_t) ULONGW_MAX)\n",
             headerFile );
      fputs( "#define PROBE_SDOUBLE_MAX ((sdoublew_t) SLONGW_MAX)\n",
             headerFile );
      fputs( "#define PROBE_SDOUBLE_MIN ((sdoublew_t) SLONGW_MIN)\n",
             headerFile );
   }
#if defined ULLONG_MAX
   else if ( typeBits.llongSize >= 2 * typeBits.cellSize )
   {  fputs( "typedef ullongw_t udoublew_t;\n", headerFile );
      fputs( "typedef sllongw_t sdoublew_t;\n", headerFile );
      fprintf( headerFile, "#define PROBE_DOUBLE_BYTE %d\n", 
               (int) sizeof (ullongw_t) );
      fputs( "#define PROBE_DOUBLE_BIT BITSPERLLONGW\n", headerFile );
      fputs( "#define PROBE_DOUBLE_HIGH_BIT ((udoublew_t) LLONGW_HIGH_BIT)\n",
             headerFile );
      fputs( "#define PROBE_UDOUBLE_MAX ((udoublew_t) ULLONGW_MAX)\n",
             headerFile );
      fputs( "#define PROBE_SDOUBLE_MAX ((sdoublew_t) SLLONGW_MAX)\n",
             headerFile );
      fputs( "#define PROBE_SDOUBLE_MIN ((sdoublew_t) SLLONGW_MIN)\n",
             headerFile );
   }
#endif /* defined ULLONG_MAX */



   /* probeByteOrder() */


/*  Byte order, high bits,
    mask patterns?
    Set the double_t synthetic math flag!!!!

   {
      signed char sprobe;
      int charsize
   }
*/
   /* Absolute byte order, and basis for cell order in double_t: */
/*
   for ( i = sizeof (unsigned long long) - 1; i > 0; --i )
   {  bolonglong |= i;
      bolonglong <<= CHAR_BIT; 
   }
   boPtr = (unsigned char *) &bolonglong;
   printf( "unsigned char byteOrder[ %ld ] = { ", sizeof (unsigned long long) );
   for ( i = 0; i < sizeof (unsigned long long); ++i )
   {  printf( "%d%c ", boPtr[ i ], 
              ( i < sizeof (unsigned long long) - 1 ) ? ',' : ' ' );
   }
   printf( "};\n" );
*/
   fprintf( headerFile, macroEnd[ 0 ], headerMacro );
   fclose( headerFile );
   fclose( sourceFile );
   return exitCode;
}
