#include "stdafx.h"
#include "resource.h"
#include "BITSExplorerView.h"
#include "BITSItem.h"
#include "BITSUtil.h"
#include "MiscUtil.h"


const HEADER_COLUMN_INFO JOB_COLUMN_TABLE[] = {
	{CBITSExplorerView::DISPLAY_NAME_COLUMN_INDEX,	CBITSExplorerView::DISPLAY_NAME_COLUMN_WIDTH,	IDS_COLUMN_JOB_DISPLAY_NAME},
	{CBITSExplorerView::TYPE_COLUMN_INDEX,			CBITSExplorerView::TYPE_COLUMN_WIDTH,			IDS_COLUMN_JOB_TYPE},
	{CBITSExplorerView::PRIORITY_COLUMN_INDEX,		CBITSExplorerView::PRIORITY_COLUMN_WIDTH,		IDS_COLUMN_JOB_PRIORITY},
	{CBITSExplorerView::STATE_COLUMN_INDEX,			CBITSExplorerView::STATE_COLUMN_WIDTH,			IDS_COLUMN_JOB_STATE},
	{CBITSExplorerView::SIZE_COLUMN_INDEX,			CBITSExplorerView::SIZE_COLUMN_WIDTH,			IDS_COLUMN_JOB_SIZE},
	{CBITSExplorerView::TRANSFERED_COLUMN_INDEX,	CBITSExplorerView::TRANSFERED_COLUMN_WIDTH,		IDS_COLUMN_JOB_TRANSFERED},
	{CBITSExplorerView::PROGRESS_COLUMN_INDEX,		CBITSExplorerView::PROGRESS_COLUMN_WIDTH,		IDS_COLUMN_JOB_PROGRESS},
	{CBITSExplorerView::OWNER_COLUMN_INDEX,			CBITSExplorerView::OWNER_COLUMN_WIDTH,			IDS_COLUMN_JOB_OWNER},
};


CBITSExplorerView::CBITSExplorerView():
	m_bDisplyAllUsersJobs(true)
{
	NULL;
}


BOOL CBITSExplorerView::PreTranslateMessage(__in MSG* /*pMsg*/)
{
	return FALSE;
}


LRESULT CBITSExplorerView::OnCreate(__in CREATESTRUCT* /*lpCreateStruct*/)
{
	HRESULT hr = m_spBITSManager.CoCreateInstance(CLSID_BackgroundCopyManager);
	ATLVERIFY(SUCCEEDED(hr));

	SetMsgHandled(FALSE);

	return 0;
}


LRESULT CBITSExplorerView::OnDestroy(void)
{
	ATLVERIFY(DeleteAllItems());

	return 1;
}


LRESULT CBITSExplorerView::OnReflectedNMDBClick(__in LPNMHDR pnmh)
{
	if (NULL == pnmh) {
		ATLASSERT(false);
		return 0;
	}

	if (m_hWnd != pnmh->hwndFrom) {
		return 0;
	}

	CComPtr<IBackgroundCopyJob> spJob = GetSelectedJob();
	if (!spJob) {
		return 0;
	}

	::SendMessage(GetParent(), WM_COMMAND, ID_JOB_PROPERTY, 0);

	return 0;
}


LRESULT CBITSExplorerView::OnReflectedNMRClick(__in LPNMHDR pnmh)
{
	if (NULL == pnmh) {
		ATLASSERT(false);
		return 0;
	}

	if (m_hWnd != pnmh->hwndFrom) {
		return 0;
	}

	::SendMessage(GetParent(), WM_COMMAND, ID_VIEW_REFRESH, 0);
	ATLVERIFY(TrackJobPopupMenu());

	return 1;
}


bool CBITSExplorerView::CreateHeaderColumns(void)
{
	for (int nIndex = 0; nIndex < _countof(JOB_COLUMN_TABLE); nIndex++) {
		ATLVERIFY(AddHeaderColumn(JOB_COLUMN_TABLE[nIndex]));
	}

	return true;
}


bool CBITSExplorerView::InitJobStateImageList(void)
{
	CImageList JobStateImageList;
	ATLVERIFY(JobStateImageList.Create(JOB_STATE_IMAGE_LIST_SIZE_X, JOB_STATE_IMAGE_LIST_SIZE_Y,
		ILC_COLORDDB | ILC_MASK, 8, 8));

	HBITMAP hBitmap = ::LoadBitmap(ModuleHelper::GetResourceInstance(), MAKEINTRESOURCE(IDB_JOB_STATE));
	if (NULL == hBitmap) {
		ATLASSERT(false);
		return false;
	}
	JobStateImageList.Add(hBitmap, LIST_VIEW_BMP_MASK);
	SetImageList(JobStateImageList, LVSIL_SMALL);
	JobStateImageList.Detach();

	return true;
}


bool CBITSExplorerView::AddHeaderColumn(__in const HEADER_COLUMN_INFO& columnInfo)
{
	CString strColumnName;
	if (FALSE == strColumnName.LoadString(columnInfo.unStringId)) {
		ATLASSERT(false);
		return false;
	}
	int nResult = InsertColumn(columnInfo.dwColumnIndex, strColumnName, LVCFMT_LEFT, columnInfo.dwColumnWidth);

	return (-1 != nResult)?true :false;
}


bool CBITSExplorerView::Refresh(bool bCleanup)
{
	if (!m_spBITSManager) {
		ATLASSERT(false);
		return false;
	}

	DWORD dwFlags = 0;
	if (true == m_bDisplyAllUsersJobs) {
		dwFlags = BG_JOB_ENUM_ALL_USERS;
	}

	CComPtr<IEnumBackgroundCopyJobs> spEnumBITSJobs;
	HRESULT hr = m_spBITSManager->EnumJobs(dwFlags, &spEnumBITSJobs);
	if (FAILED(hr) || !spEnumBITSJobs) {
		ATLASSERT(false);
		return false;
	}

	CSimpleArray<GUID> ActiveJobIdArray;
	for ( ; ; ) {
		CComPtr<IBackgroundCopyJob> spJob;
		hr = spEnumBITSJobs->Next(1, &spJob, NULL);
		if (FAILED(hr)) {
			ATLASSERT(false);
			return false;
		}
		if (S_OK != hr) {
			break;
		}

		GUID JobId = {0};
		hr = spJob->GetId(&JobId);
		if (FAILED(hr)) {
			ATLASSERT(false);
			continue;
		}
		ActiveJobIdArray.Add(JobId);

		int nItemIndex = CreateItem(JobId, spJob);
		if (-1 == nItemIndex) {
			ATLASSERT(false);
			continue;
		}
	}

	ATLVERIFY(UpdateAllItems());

	if (true == bCleanup) {
		ATLVERIFY(CleanupOldItems(ActiveJobIdArray));
	}

	return true;
}


int CBITSExplorerView::CreateItem(__in const GUID& jobId, __in const CComPtr<IBackgroundCopyJob>& spJob)
{
	int nItemIndex = FindItem(jobId);
	if (-1 == nItemIndex) {
		nItemIndex = GetItemCount();
		nItemIndex = InsertItem(nItemIndex, L"");
		if (-1 == nItemIndex) {
			ATLASSERT(false);
			return -1;
		}

		CBITSItem* pBITSItem = new CBITSItem;
		pBITSItem->SetJobId(jobId);
		pBITSItem->SetJobIF(spJob);
		pBITSItem->SetBITSManagerIF(m_spBITSManager);
		SetItemData(nItemIndex, reinterpret_cast<DWORD_PTR>(pBITSItem));
	}

	return nItemIndex;
}


bool CBITSExplorerView::DeleteItem(__in int nItemIndex)
{
	if (-1 == nItemIndex) {
		ATLASSERT(false);
		return false;
	}

	CBITSItem* pItem = reinterpret_cast<CBITSItem*>(GetItemData(nItemIndex));
	if (NULL != pItem) {
		delete pItem;
		pItem = NULL;
	} else {
		ATLASSERT(false);
	}

	BOOL bResult = CListViewCtrl::DeleteItem(nItemIndex);

	return (0 != bResult)?true : false;
}


bool CBITSExplorerView::UpdateAllItems(void)
{
	int nItemCount = GetItemCount();

	for (int nIndex = 0; nIndex < nItemCount; nIndex++) {
		CBITSItem* pItem = reinterpret_cast<CBITSItem*>(GetItemData(nIndex));
		if (NULL == pItem) {
			ATLASSERT(false);
			continue;
		}
		ATLVERIFY(UpdateItem(nIndex, pItem->GetJobIF()));
	}

	return true;
}


bool CBITSExplorerView::UpdateItem(__in int nItemIndex, __in const CComPtr<IBackgroundCopyJob>& spJob)
{
#pragma warning(disable: 6031)

	if (!spJob || -1 == nItemIndex) {
		ATLASSERT(false);
		return false;
	}

	CString strDisplayName;
	HRESULT hr = BITSUtil::GetJobDisplayName(spJob, &strDisplayName);
	if (FAILED(hr)) {
		ATLASSERT(false);
		return false;
	}
	UpdateItemText(nItemIndex, DISPLAY_NAME_COLUMN_INDEX, strDisplayName);

	CString strSubItemText;
	BG_JOB_TYPE jobType = BG_JOB_TYPE_DOWNLOAD;
	hr = BITSUtil::GetJobType(spJob, &jobType);
	ATLASSERT(SUCCEEDED(hr));
	strSubItemText = MiscUtil::GetJobTypeString(jobType);
	ATLASSERT(!strSubItemText.IsEmpty());
	UpdateItemText(nItemIndex, TYPE_COLUMN_INDEX, strSubItemText);

	BG_JOB_PRIORITY jobPriority = BG_JOB_PRIORITY_FOREGROUND;
	hr = BITSUtil::GetJobPriority(spJob, &jobPriority);
	ATLASSERT(SUCCEEDED(hr));
	strSubItemText = MiscUtil::GetJobPriorityString(jobPriority);
	ATLASSERT(!strSubItemText.IsEmpty());
	UpdateItemText(nItemIndex, PRIORITY_COLUMN_INDEX, strSubItemText);

	BG_JOB_STATE jobState = BG_JOB_STATE_QUEUED;
	hr = BITSUtil::GetJobState(spJob, &jobState);
	ATLASSERT(SUCCEEDED(hr));
	strSubItemText = MiscUtil::GetJobStateString(jobState);
	ATLASSERT(!strSubItemText.IsEmpty());
	UpdateItemText(nItemIndex, STATE_COLUMN_INDEX, strSubItemText);
	UpdateItemImageIcon(nItemIndex, jobState);

	BG_JOB_PROGRESS progress = {0};
	hr = BITSUtil::GetJobProgress(spJob, &progress);
	ATLASSERT(SUCCEEDED(hr));
	if (BG_SIZE_UNKNOWN == progress.BytesTotal || 0 == progress.BytesTotal) {
		progress.BytesTotal = 0;
		strSubItemText = L"0 %";
	} else {
		if (progress.BytesTotal == progress.BytesTransferred) {
			strSubItemText = L"100 %";
		} else {
			float fProgress = progress.BytesTransferred / static_cast<float>(progress.BytesTotal) * 100;
			strSubItemText.Format(L"%.1f %%", fProgress);
		}
	}
	UpdateItemText(nItemIndex, PROGRESS_COLUMN_INDEX, strSubItemText);

	strSubItemText = MiscUtil::FormatNumber(progress.BytesTotal);
	UpdateItemText(nItemIndex, SIZE_COLUMN_INDEX, strSubItemText);

	strSubItemText = MiscUtil::FormatNumber(progress.BytesTransferred);
	UpdateItemText(nItemIndex, TRANSFERED_COLUMN_INDEX, strSubItemText);

	CString strJobOwner;
	hr = BITSUtil::GetJobOwner(spJob, &strJobOwner);
	ATLASSERT(SUCCEEDED(hr));
	UpdateItemText(nItemIndex, OWNER_COLUMN_INDEX, strJobOwner);

#pragma warning(default: 6031)

	return true;
}


bool CBITSExplorerView::UpdateItemText(__in int nItemIndex, __in int nSubItem, __in LPCWSTR lpszText)
{
	if (-1 == nItemIndex || -1 == nSubItem || NULL == lpszText) {
		ATLASSERT(false);
		return false;
	}

	CComBSTR bstrText;
	GetItemText(nItemIndex, nSubItem, bstrText.m_str);

	if (bstrText == lpszText) {
		return true;
	}

	return (TRUE == SetItemText(nItemIndex, nSubItem, lpszText))? true: false;
}


bool CBITSExplorerView::UpdateItemImageIcon(__in int nItemIndex, __in BG_JOB_STATE jobState)
{
	LVITEM item = {0};
	item.mask	= LVIF_IMAGE;
	item.iImage	= GetImageIndexFromJobState(jobState);
	item.iItem	= nItemIndex;

	ATLVERIFY(ListView_SetItem(m_hWnd, &item));

	return true;
}


int CBITSExplorerView::GetImageIndexFromJobState(__in BG_JOB_STATE jobState)
{
	int nImageIndex = PAUSE_ICON_IMAGE_INDEX;

	switch (jobState) {
		case BG_JOB_STATE_QUEUED:
		case BG_JOB_STATE_SUSPENDED:
			nImageIndex = PAUSE_ICON_IMAGE_INDEX;
			break;

		case BG_JOB_STATE_CONNECTING:
		case BG_JOB_STATE_TRANSFERRING:
			nImageIndex = RUNNING_ICON_IMAGE_INDEX;
			break;

		case BG_JOB_STATE_ERROR:
			nImageIndex = ERROR_ICON_IMAGE_INDEX;
			break;

		case BG_JOB_STATE_TRANSIENT_ERROR:
			nImageIndex = WARNING_ICON_IMAGE_INDEX;
			break;

		case BG_JOB_STATE_TRANSFERRED:
		case BG_JOB_STATE_ACKNOWLEDGED:
			nImageIndex = COMPLETE_ICON_IMAGE_INDEX;
			break;

		case BG_JOB_STATE_CANCELLED:
			nImageIndex = CANCEL_ICON_IMAGE_INDEX;
			break;

		default:
			ATLASSERT(false);
			break;
	}

	return nImageIndex;
}


int	CBITSExplorerView::FindItem(__in const GUID& jobId)
{
	int nItemCount = GetItemCount();
	if (0 == nItemCount) {
		return -1;
	}

	for (int nItemIndex = 0; nItemIndex < nItemCount; nItemIndex++) {
		CBITSItem* pBITSItem = reinterpret_cast<CBITSItem*>(GetItemData(nItemIndex));
		if (NULL == pBITSItem) {
			ATLASSERT(false);
			continue;
		}
		if (pBITSItem->GetJobId() == jobId) {
			return nItemIndex;
		}
	}

	return -1;
}


bool CBITSExplorerView::DeleteAllItems(void)
{
	int nItemCount = GetItemCount();
	for (int nItemIndex = nItemCount - 1; 0 <= nItemIndex; nItemIndex--) {
		DeleteItem(nItemIndex);
	}

	return true;
}


bool CBITSExplorerView::CleanupOldItems(__in const CSimpleArray<GUID>& activeJobIdArray)
{
	int nItemCount = GetItemCount();
	if (0 == nItemCount) {
		return true;
	}

	for (int nItemIndex = nItemCount - 1; 0 <= nItemIndex; nItemIndex--) {
		CBITSItem* pItem = reinterpret_cast<CBITSItem*>(GetItemData(nItemIndex));
		if (NULL == pItem) {
			ATLASSERT(false);
			ATLVERIFY(CListViewCtrl::DeleteItem(nItemIndex));
			continue;
		}

		int nFindIndex = activeJobIdArray.Find(pItem->GetJobId());
		if (-1 == nFindIndex) {
			ATLVERIFY(DeleteItem(nItemIndex));
			int nItemCountAfter = GetItemCount();
			if (nItemCountAfter <= 0) {
				continue;
			}

			// 폜ꂽƂ́Ä̃ACeIԂɂ
			if (nItemCountAfter <= nItemIndex) {
				ATLVERIFY(SetItemState(nItemCountAfter - 1, LVIS_SELECTED | LVIS_FOCUSED, LVIS_SELECTED | LVIS_FOCUSED));
			} else {
				ATLVERIFY(SetItemState(nItemIndex, LVIS_SELECTED | LVIS_FOCUSED, LVIS_SELECTED | LVIS_FOCUSED));
			}
		
			continue;
		}
	}

	return true;
}


__checkReturn CComPtr<IBackgroundCopyJob> CBITSExplorerView::GetSelectedJob(void)
{
	LVITEM lvItem = {0};
	if (FALSE == GetSelectedItem(&lvItem)) {
		return NULL;
	}

	CBITSItem* pBITSItem = reinterpret_cast<CBITSItem*>(GetItemData(lvItem.iItem));
	if (NULL == pBITSItem) {
		ATLASSERT(false);
		return NULL;
	}

	return pBITSItem->GetJobIF();
}


bool CBITSExplorerView::TrackJobPopupMenu(void)
{
	CMenu menu;
	BOOL bResult = menu.LoadMenu(IDR_MAINFRAME);
	if (FALSE == bResult) {
		ATLASSERT(false);
		return false;
	}

	CMenuHandle SubMenu = menu.GetSubMenu(2);
	if (NULL == SubMenu.m_hMenu) {
		ATLASSERT(false);
		return false;
	}

	POINT currentPoint = {0};
	ATLVERIFY(::GetCursorPos(&currentPoint));
	bResult = SubMenu.TrackPopupMenu(TPM_LEFTALIGN | TPM_TOPALIGN | TPM_RIGHTBUTTON, currentPoint.x,
		currentPoint.y,	GetParent());

	return (0 == bResult? false: true);
}

