
/* ------------------------
MathInput.js
*/

/* ====================
** inputフォーム管理オブジェクト（<mi><mo><mn>相当）
*/
function MathInput(par, v) {
    this.wrap = par;
    this.exp = par.exp;
    this.objType = 'input';
    this.obj = document.createElement('textarea');
    this.obj.className = 'MathInput ' + (this.wrap.level == 1) ? 'level1' : 'level2';
    this.level = this.wrap.level;
    this.exp.issueID(this);
    this.selection = new Selection(this.obj);
    this.isHidden = (v == 'visible') ? false : true;
    var self = this;
    this.optimizeSize();
    
    // イベントハンドラ指定
    this.obj.onkeydown = function(evt) {
	self._myKeyDown(evt);
    }
    this.obj.onkeypress = function(evt) {
        return self._myKeyPress(evt);
    }
    this.obj.onkeyup = function(evt) {
        return self._myKeyUp(evt);
    }
    this.obj.onfocus = function() {
        self.exp.submitFocus(self, self.obj.id);
    }
/*
    this.obj.onblur = function() {
    }
*/
}

// ---- クラス変数

// オブジェクトに置き換える文字列
MathInput.insertStrings = {
    'sin': function(w) {return new MathString(w, 'sin')},
    'cos': function(w) {return new MathString(w, 'cos')},
    'tan': function(w) {return new MathString(w, 'tan')},
    'log': function(w) {return new MathString(w, 'log')},
    'sqrt': function(w) {return new MathSqrt(w)},
    'frac': function(w) {return new MathFrac(w)},
    'lim': function(w) {return new MathLimit(w)}
}

// ---- global

// 指定位置にカーソルを置く
MathInput.prototype.setCursor = function(num) {
    if(num == -1) num = this.obj.value.length;
    this.selection.setCursor(num);
}

// inputフォーム中にフォーカスがある状態で機能ボタンが押された時の動作
MathInput.prototype.insert = function(insertObj) {
    var selpos = this.selection.create();
    var txt = this.obj.value;
    var prefix = txt.substring(0, selpos.start);
    var content = txt.substring(selpos.start, selpos.end);
    var postfix = txt.substring(selpos.end, txt.length);
    var postobj = new MathInput(this.wrap, ((postfix != '') ? 'visible' : ''));
    var thispos = this.wrap.getPos(this.obj.id);
    this.wrap.add(postobj, thispos + 1);
    this.wrap.add(insertObj, thispos + 1);
    this.obj.value = prefix;
    postobj.obj.value = postfix;
    postobj.optimizeSize();
    this.optimizeSize();
    insertObj.setInitialValue(content);
}

// 文字を囲ってコンテナを作った場合
MathInput.prototype.setInitialValue = function(val) {
    this.obj.value = val;
    this.selection.selectAll();
    this.layout();
}

// 隠すかどうかを設定
MathInput.prototype.setIsHide = function() {
    var pos = this.wrap.getPos(this.obj.id);
    this.isHidden =
	((this.wrap.array.length > 1) // 行に2つ以上要素があって
	 && (this.obj.value.length == 0) // 中身がなくて
	 // (右隣があって必要とされていて、なおかつ左隣がない) の逆ならば隠す
	 && ((pos > 1) || (pos == this.wrap.array.length - 1) || (!this.wrap.array[pos + 1].isNeedPrev)));
}

// 親への伝播なしの再配置
MathInput.prototype.optimizeSize = function() {
    var s = 'MathInput';
    s += (this.wrap.level == 1) ? ' level1' : ' level2';
    s += (this.isHidden) ? ' hidden' : '';
    s += (this.obj.id == this.exp.focusID) ? ' selected' : '';
    this.obj.className = s;
    this.obj.style.width = this._calcWidth() + 'px';
}

// 再配置
MathInput.prototype.layout = function() {
    this.optimizeSize();
    this.wrap.layout();
}

// 必要サイズを応答
MathInput.prototype.getPreferredSize = function() {
    var h = Const.tokenHeight[this.level] + 4;
    return {
        top: h / 2,
        bottom: h / 2,
        height: h,
        width: this._calcWidth() + 4
    };
}

// MathMLコード書き出し
MathInput.prototype.outputMathML = function(expand) {
    var v = this.obj.value;
    if(v == '') return null;
    v = v.replace(/</, "&lt;");
    v = v.replace(/>/, "&gt;");
    v = v.replace(/\//g,"//"); // スラッシュ記号が判別不可能になるので一時退避
    v = v.replace(/([a-zA-Z]+)/g,"<mi>$1</mi>");
    v = v.replace(/(\d+(\.\d+)?)/g, "<mn>$1</mn>");
    v = v.replace(/([-=+\*]|\/\/)/g, "<mo>$1</mo>");
    v = v.replace(/\/\//g, "/"); //スラッシュ記号を戻す
    v = v.replace(/(<\w+>[^<]+?<\/\w+>$)/, ""); // 最後の要素を分離
    //alert(this.obj.value + ' /// ' + v + ' /// ' +  RegExp.$1);
    return {
        front: v, 
        rear: RegExp.$1
    };
}

// LaTeXコード書き出し
MathInput.prototype.outputLatex = function() {
    return this.obj.value;
}

// 表示部品削除
MathInput.prototype.removeObj = function() {
    this.obj.parentNode.removeChild(this.obj);
}

// ----private

// inputフォーム内でキー入力があった時の動作
MathInput.prototype._myKeyDown = function(evt) {
    evt = (evt) ? evt : ((window.event) ? event : null);
    if(this._onKeyCode[evt.keyCode] != null) {
	this.selpos = this.selection.create();
	this._onKeyCode[evt.keyCode](evt, this);
    }
}

// キー入力時の操作関数配列
MathInput.prototype._onKeyCode = new Array();

// BackSpaceキー（キーコード8）
MathInput.prototype._onKeyCode[BUtil.key.bs] = function(evt, input) {
    if(input.selpos.end == 0) {
	BUtil.event.stop(evt);
	var pos = input.wrap.getPos(input.obj.id) - 1;
	if(pos >= 0) input.wrap.del(pos);
    }
}

// 「←」キー（キーコード37）
MathInput.prototype._onKeyCode[BUtil.key.left] = function(evt, input) {
    if(input.selpos.end == 0) {
	BUtil.event.stop(evt);
	input.wrap.backCursor(input.obj.id);
    }
}

// 「→」キー（キーコード39）
MathInput.prototype._onKeyCode[BUtil.key.right] = function(evt, input) {
    if(input.selpos.start == input.obj.value.length) {
	BUtil.event.stop(evt);
	input.wrap.fwdCursor(input.obj.id);
    }
}

// 「↑」キー（キーコード38）
MathInput.prototype._onUpArrow = function() {
    // とりあえず殺しておく
}

// 「↓」キー（キーコード40）
MathInput.prototype._onDownArrow = function() {
    // とりあえず殺しておく
}


// inputフォーム中で文字が入力された時の動作
MathInput.prototype._myKeyPress = function(evt) {
    evt = (evt) ? evt : ((window.event) ? event : null);
    if(evt && (!this.isKeyProcessed)) {
	// Firefox以外ではkeyCodeで文字コードが渡るので対処
        var charCode = (evt.charCode) ? (evt.charCode) : (evt.keyCode);
        // Enter の場合
        switch(charCode) {
            // 改行されると困るので殺す
            case 13:
                return false;
            // 「（」で括弧を作る
            case 40:
                this.insert(new MathFenced(this.wrap));
                return false;
            // 「）」でも括弧を作る
            case 41:
                this.insert(new MathFenced(this.wrap));
                return false;
            // 「＾」で累乗を作る
            case 94:
                this.insert(new MathSup(this.wrap));
                return false;
            // 「＿」で添字を作る
            case 95:
                this.insert(new MathSub(this.wrap));
                return false;
        }
        // 文字入力でオブジェクトを作成する
        for(var str in MathInput.insertStrings) {
            if(charCode == str.charCodeAt(str.length - 1)) {
                var selpos = this.selection.create();
                if(this.obj.value.substring(selpos.end - str.length + 1, selpos.end) == str.substring(0, str.length - 1)) {
                    this.obj.value = this.obj.value.substring(0, selpos.end - str.length + 1);
                    this.insert(MathInput.insertStrings[str](this.wrap));
                    // this.optimizeSize();
                    return false;
                }
            }
        }
    }
}

// inputフォーム中でキーを離した時の動作
MathInput.prototype._myKeyUp = function(evt) {
    this.layout();
}

// 幅の計算
MathInput.prototype._calcWidth = function() {
    if(this.isHidden) {
	return 4;
    }
    else {
	return Math.floor((this.obj.value.length + 2) * Const.tokenWidth[this.level] );
    }
}

// スタイル設定
BUtil.style.add(
    [['textarea.MathInput', "position: absolute; border: 1px solid gray; font-family: 'Courier New', monospace; width: 30px; overflow: hidden;"],
     ['textarea.level1', 'font-size: 16px; height: 22px;'],
     ['textarea.level2', 'font-size: 12px; height: 18px;'],
     ['textarea.selected', 'border-color: black; background-color: #fff8ee;'],
     ['textarea.hidden', 'width: 4px; border-color: #ddd; background-color: #ddd;'],
     ['textarea.hidden:hover', 'border-color: gray;']]
);

