
/**
 * @class数式内のコンテナオブジェクトの抽象クラス
 * @extends Bredima.Exp.Rowfactor
 * @constructor
 * @param {Object} param オブジェクト構築共通パラメタ
 * @param {Array} rowLevelMap mrowを初期化する際の表示レベルを示した配列
 * 
 * mrowの初期化
 */
Bredima.Container = function(param) {
    arguments.callee.superClass.apply(this, arguments);
    this.exp = this.row.container.exp;
    this.parts = {};

    // rowdownにその他のプロパティがあるか確認
    if(param.rundown && (param.rundown.length > this.rowCount))
	this.config = param.rundown.splice(0, (param.rundown.length - this.rowCount));
    
    // 指定分だけmrowを用意
    this.rows = new Array;
    var p = {};
    for(var i = 0; i < this.rowCount; i++) {
	p = {
	    parent: this,
	    rundown: (param.rundown) ? param.rundown[i] : undefined,
	    value: (param.value && (i == this.primaryRow)) ? param.value : undefined,
	    level: (this.rowLevels && (this.rowLevels[i] > 0)) ? this.rowLevels[i] : this.row.getLevel()
	};
	this.rows.push(new Bredima.Exp.Row(p));
	this.obj.appendChild(this.rows[i].getDom());
    }
    this.content = this.rows[this.primaryRow];
}

Bredima.util.inherit(Bredima.Container, Bredima.Exp.Rowfactor);

Bredima.util.setProperty(Bredima.Container, {
    // Containerが使う属性
			     rowCount: 1,
			     padding: {top:0, right:0, bottom:0, left:0},
    primaryRow: 0
			 });

/* ===========================
 Bredima.Containerで定義
   -----------------------------
 配列管理
*/

/**
 * 管理しているrowの数を返す
 */
Bredima.Container.prototype.getLength = function() { return this.rows.length; }

/**
 * 指定した位置のrowを返す
 * @param {Number} order 位置
 */
Bredima.Container.prototype.getRow = function(order) {return this.rows[order]; }

/**
 * コンテナオブジェクト内のmrowの順序を返す
 */
Bredima.Container.prototype.orderOf = function(row) {
    for(var i = 0; i < this.rows.length; i++) {
	if(this.rows[i] == row) return i;
    }
    return null;
}

/**
 * mrowの縦方向の位置を返す
 * liftCursorの処理に使用
 * @private
 */
Bredima.Container.prototype._verticalOrderOf = function(row) {
    var order = row.getOrder();
    for(var i = 0; i < this.verticalOrder.length; i++) {
	if(order == this.verticalOrder[i]) return i;
    }
    return null;
}

/* ----------------------
 カーソル移動
*/

/**
 * コンテナオブジェクト内でのmrow間のカーソル横移動
 * Expからの呼び出し専用
 * @param {Object} row mrowオブジェクト
 * @param {String} dir 移動方向 back / fwd
 */
Bredima.Container.prototype.slideCursorFrom = function(row, dir) {
    var order = this.orderOf(row);
    var c = (dir == 'back');
    if((c && order == 0) || (!c && order == this.rows.length - 1)) return false;
    this.rows[order + (c ? -1 : 1)].setCursorToRow(c ? 'tail' : 'head');
    return true;
}

/**
 * コンテナオブジェクト内でのmrow間のカーソル縦移動
 * Expからの呼び出し専用
 * @param {Object} rowfactor rowfactorオブジェクト
 * @param {String} dir 移動方向 up/down
 */
Bredima.Container.prototype.liftCursorFrom = function(rowfactor, dir) {
    if(!this.verticalOrder) return false;
    var order = this._verticalOrderOf(rowfactor.row);
    var c = (dir == 'up');
    if((c && order == 0) || (!c && order == this.rows.length - 1)) return false;
    var row = this.rows[this.verticalOrder[order + (c ? -1 : 1)]];
    var rf2 = row.getRowfactor(Math.floor(row.getLength() * rowfactor.getOrder() / rowfactor.row.getLength()));
    if(!rf2.setCursorTo('head'))
	this.exp.slideCursorFrom(rf2, 'back');
    return true;
}

/* ---------------------------
 その他
*/

/**
 * （継承先で使用）構成パーツを登録
 * @param {Object} parts 登録するパーツの連想配列{登録名: 登録オブジェクト}
 */
Bredima.Container.prototype.add = function(key, parts) {
    this.parts[key] = parts;
    this.obj.appendChild(parts.getDom());
}

/**
 * 概要オブジェクト先頭の特殊属性を返す
 */
Bredima.Container.prototype._toRundownContentParam = function() {
}

/* ====================
 Exp.Rowfactorの実装
*/
/**
 * コンテナオブジェクトのheadかtailにカーソルを置く
 * @param {String} loc カーソルを置く位置 head / tail
 */
Bredima.Container.prototype.setCursorTo = function(loc) {
    this.rows[((loc == 'head') ? 0 : (this.rows.length - 1))].setCursorToRow(loc);
    return true;
}

Bredima.Container.prototype.getIncomingInput = function() {
    return this.content.getRowfactor(0).getIncomingInput();
}

/* ====================
 Expの実装
*/
/**
 * 概要オブジェクトの中身を返す
 */
Bredima.Container.prototype._toRundownContent = function() {
    var ret = new Array;
    var param = this._toRundownContentParam();
    if(param) ret = ret.concat(param);
    for(var i = 0; i < this.rowCount; i++) {
	ret.push(this.rows[i].toRundown());
    }
    return ret;
}

Bredima.Container.prototype._toMMLContent = function(expand) {
    var out = '';
    for(var i = 0; i < this.rows.length; i++)
	out += this.rows[i].toMML(expand);
    return out;
}

Bredima.Container.prototype._toLatexContent = function() {
    var out = '';
    for(var i = 0; i < this.rows.length; i++)
	out += this.rows[i].toLatex();
    return out;
}


/* ------------------------
 レイアウト管理
*/

Bredima.Container.prototype._reposition = function() {
    var prefer = this.content.getPreferredSize();
    this.top = prefer.top + this.padding.top;
    this.bottom = prefer.bottom + this.padding.bottom;
    this.height = this.top + this.bottom;
    this.width = prefer.width + this.padding.left + this.padding.right;

    this._resize();
    this.content.setPos(this.padding.left, this.padding.top);
}

Bredima.Container.prototype.repositionAll = function() {
    for(var i = 0; i < this.rows.length; i++) {
	this.rows[i].repositionAll();
    }
    this._reposition();
}

Bredima.style.add(
    ['.container', 'border: 1px dashed orange;']
);
