
/**
  @fileoverview
  コンテナオブジェクト（レイアウトスキーマ）

  Bredima.Container.
    Frac
    
 */
/* ===============================
*/
/**
 * @class 分数
 * @extends Bredima.Container
 * @constructor
 * @param {Object} param オブジェクト構築共通パラメタ
 */
Bredima.Container.Frac = function(param) {
    this.__super__(param);
    this.add('separator', new Bredima.Parts.Separator());
}

Bredima.util.inherit(Bredima.Container.Frac, Bredima.Container);

Bredima.util.setProperty(Bredima.Container.Frac, {
			     rowCount : 2,
			     MML : 'mfrac',
			     tex : 'frac',
			     verticalOrder : [0, 1]
			 });

Bredima.Exp.register(Bredima.Container.Frac, 'container', 'frac');

Bredima.Container.Frac.prototype._reposition = function() {
    var num = this.rows[0].getPreferredSize();
    var den = this.rows[1].getPreferredSize();
    this.width = Math.max(num.width, den.width);
    this.top = num.height;
    this.bottom = den.height + 1;
    this.height = this.top + this.bottom;
    
    this._resize();
    this.rows[0].setPos((this.width - num.width) / 2, 0);
    this.rows[1].setPos((this.width - den.width) / 2, num.height + 1);
    this.parts.separator.setPos(0, num.height);
}

/* ===============================
*/
/**
 * @class 括弧
 * @extends Bredima.Container
 * @constructor
 * @param {Object} param オブジェクト構築共通パラメタ
 */
Bredima.Container.Fenced = function(param, type) {
    this.__super__(param);
    this.type = (this.config) ? this.config : ((type == null) ? 0 : type);
    var menu = new Bredima.Parts.Popup(this.exp.bredima, 'fenced', this);
    this.add('open', new Bredima.Parts.Image(this.types[this.type].name + '_open.png'));
    this.add('close', new Bredima.Parts.Image(this.types[this.type].name + '_close.png'));
    this.add('menu', menu);
    this.parts.open.setPos(0, 0);
}

Bredima.util.inherit(Bredima.Container.Fenced, Bredima.Container);

Bredima.util.setProperty(Bredima.Container.Fenced, {
    MML: 'mfenced',
    padding: {top:0, right:16, bottom:0, left:8}
});

Bredima.Exp.register(Bredima.Container.Fenced, 'container', 'fenced');

Bredima.Container.Fenced.prototype.types = [
    {name: 'parenthesis', character: ['(', ')'], certified: true},
    {name: 'brace', character: ['{', '}']},
    {name: 'bracket', character: ['[', ']']}
];

/* ------------------
 Containerの実装
*/
Bredima.Container.Fenced.prototype._toRundownContentParam = function() {
    if(this.type != 0) return this.type;
}

/* ------------------
 Expの実装
*/
/**
 * MathML表記を返す
 * 開きタグが特殊なので全て構築
 */
Bredima.Container.Fenced.prototype.toMML = function() {
    var attr = this.types[this.type];
    var out = (!attr.certified) ? ('open="' + attr.character[0] + '" close="' + attr.character[1] + '"') : '';
    return Bredima.util.addTag(this.MML, this.content.toMML(), out);
}

/**
 * LaTeX表記を返す
 * 閉じコマンドも必要なので全て構築
 */
Bredima.Container.Fenced.prototype.toLatex = function() {
    var out = '\\left' + this.types[this.type].character[0];
    out += this.content.toLatex('noFence');
    out += '\\right' + this.types[this.type].character[1];
    return out;
}

/* ------------------
 private
*/
/**
 * ポップアップメニューの選択を受信して括弧の種類を変更
 * @privete
 */
Bredima.Container.Fenced.prototype.menuClickPerformed = function(arg) {
    if(this.type != arg) {
	this.exp.startRowRec(this.row);
	this.type = arg;
	this.parts.open.setURI(this.types[this.type].name + '_open.png');
	this.parts.close.setURI(this.types[this.type].name + '_close.png');
	this.exp.endRowRec();
    }
}

Bredima.Container.Fenced.prototype._reposition = function() {
    Bredima.util.applySuper('_reposition', this);
    this.parts.open.setSize(10, this.height);
    this.parts.close.setSize(10, this.height);
    this.parts.close.setPos(this.width - 16, 0);
    this.parts.menu.setPos(this.width - 10, this.height - 10);
}

/* ===============================
*/
/**
 * @class 平方根
 * @extends Bredima.Container
 * @constructor
 * @param {Object} param オブジェクト構築共通パラメタ
 */
Bredima.Container.Sqrt = function(param) {
    this.__super__(param);
    this.add('radical', new Bredima.Parts.Image('radical.png'));
    this.add('line', new Bredima.Parts.Separator());
    this.parts.radical.setPos(0, 2);
    this.parts.line.setPos(10, 2);
}

Bredima.util.inherit(Bredima.Container.Sqrt, Bredima.Container);

Bredima.util.setProperty(Bredima.Container.Sqrt, {
			     MML: 'msqrt',
			     tex: 'sqrt',
			     padding: {top: 4, right: 0, bottom: 0, left: 10}
			 });

Bredima.Exp.register(Bredima.Container.Sqrt, 'container', 'sqrt');

Bredima.Container.Sqrt.prototype._reposition = function() {
    Bredima.util.applySuper('_reposition', this);
    this.parts.radical.setSize(10, this.height - 2);
    this.parts.line.setSize(this.width - 10, 1);
}

/* ===============================
*/
/**
 * @class 累乗根
 * @extends Bredima.Container
 * @constructor
 * @param {Object} param オブジェクト構築共通パラメタ
 */
Bredima.Container.Root = function(param) {
    this.__super__(param);
    this.padding = {top: 4, right: 0, bottom: 0, left: 10}; // repositionで変更するのでインスタンス毎に持つ
    this.add('radical', new Bredima.Parts.Image('radical.png'));
    this.add('line', new Bredima.Parts.Separator());
    this.rows[0].setPos(0, 0);
}

Bredima.util.inherit(Bredima.Container.Root, Bredima.Container);

Bredima.util.setProperty(Bredima.Container.Root, {
			     rowCount: 2,
			     rowLevels: [2, 0],
			     primaryRow: 1,
			     MML: 'mroot',
			     tex: 'sqrt'
			 });

Bredima.Exp.register(Bredima.Container.Root, 'container', 'root');

Bredima.Container.Root.prototype._toMMLContent = function(expand) {
    return this.rows[1].toMML(expand) + this.rows[0].toMML(expand);
}

Bredima.Container.Root.prototype._toLatexContent = function() {
    return '[' + this.rows[0].toLatex('nofence') + ']' + this.rows[1].toLatex();
}

Bredima.Container.Root.prototype._reposition = function() {
    var exp = this.rows[0].getPreferredSize();
    var offset = Math.max((exp.height - this.content.getPreferredSize().top), 4);
    this.padding.top = offset;
    this.padding.left = exp.width + 8;
    Bredima.util.applySuper('_reposition', this);
    
    this.parts.radical.setSize(10, this.height - offset + 2);
    this.parts.radical.setPos(exp.width - 2, offset - 2);
    this.parts.line.setSize(this.width - exp.width - 8, 1);
    this.parts.line.setPos(exp.width + 8, offset - 2);
}

/* ===============================
*/
/**
 * @class 累乗
 * @extends Bredima.Container
 * @constructor
 * @param {Object} param オブジェクト構築共通パラメタ
 */
Bredima.Container.Sup = function(param) {
    this.__super__(param);
    this.content.setPos(0, 0);
}

Bredima.util.inherit(Bredima.Container.Sup, Bredima.Container);

Bredima.util.setProperty(Bredima.Container.Sup, {
			     MML: 'msup',
			     isMMLNeedPrev: true,
			     rowLevels: [2]
			 });

Bredima.Exp.register(Bredima.Container.Sup, 'container', 'sup');

Bredima.Container.Sup.prototype._toMMLContent = function(expand, prev) {
    if(prev == '') prev = '<mo></mo>';
    return prev + this.content.toMML(expand);
}

Bredima.Container.Sup.prototype.toLatex = function() {
    return '^' + this.content.toLatex();
}

Bredima.Container.Sup.prototype._reposition = function() {
    var prefer = this.content.getPreferredSize();
    this.top = prefer.height;
    this.bottom = 0;
    this.height = this.top + this.bottom;
    this.width = prefer.width;

    this._resize();
}

/* ===============================
*/
/**
 * @class 添字
 * @extends Bredima.Container
 * @constructor
 * @param {Object} param オブジェクト構築共通パラメタ
 */
Bredima.Container.Sub = function(param) {
    this.__super__(param);
    this.content.setPos(0, 0);
}

Bredima.util.inherit(Bredima.Container.Sub, Bredima.Container);

Bredima.util.setProperty(Bredima.Container.Sub, {
			     MML: 'msub',
			     isMMLNeedPrev: true,
			     rowLevels: [2]
			 });

Bredima.Exp.register(Bredima.Container.Sub, 'container', 'sub');

Bredima.Container.Sub.prototype._toMMLContent = function(expand, prev) {
    if(prev == '') prev = '<mo></mo>';
    return prev + this.content.toMML(expand);
}

Bredima.Container.Sub.prototype.toLatex = function() {
    return '_' + this.content.toLatex();
}

Bredima.Container.Sub.prototype._reposition = function() {
    var prefer = this.content.getPreferredSize();
    this.top = 0;
    this.bottom = prefer.height;
    this.height = this.top + this.bottom;
    this.width = prefer.width;

    this._resize();
}
