
/**
 * @class 数式全体を管理するオブジェクト　数式のrootに存在
 * @constructor
 */
Bredima.ExpCtrl = function(bredima){
    this.bredima = bredima;
    this.root = new Bredima.Exp.Root(this);
    this.obj = this.root.getDom();
    this.focus = ''; // 現在DOM的にフォーカスがあるオブジェクト
    this.current = ''; // 現在選択されていると見なしているオブジェクト
    this.history = this.bredima.history;
    this.history.observer.attach(this);
    this.observer = new Bredima.Observer();

    var self = this;
    this.obj.onkeydown = function(evt) {
	if(self.focus) {
	    //self.record.commit();  // 前のキーのkeyupの前にkeydownが来ると困るのでやめ
	    self.record.cache();
	    self.handleControlCharacter(evt);
	}
    }
    this.obj.onkeypress = function(evt) {
	if(self.focus) {
	    self.handleControlCharacter(evt);
	    self.handleNormalCharacter(evt);
	}
    }
    this.obj.onkeyup = function(evt) {
	if(self.focus) {
	    if(self.isProcessing) {
		self.endRowRec();
		self._startRec();
	    }
	    else {
		 // 前のcommitと連続している可能性あり
		if(self.record.commit('temporary'))
		    self.history.ackRecordArise();
		self.record.cache();
	    }

	    // 再配置
	    self.focus.repositionToRoot();
	}
    }
}

/**
 * 起動時の初期化
 */
Bredima.ExpCtrl.prototype.init = function(rundown) {
    if(rundown) this.root.row.initFromRundown(rundown);

    this.root.repositionAll();
    this.root.setCursorTo('head');
}

/* ====================
 オブジェクト挿入処理
 */
/**
 * 数式内のフォーカスがある位置にオブジェクトを挿入
 * 数式内にフォーカスがない状態での処理
 * @param {Function} func コンストラクタ実行関数
 */
Bredima.ExpCtrl.prototype.insElement = function(func) {
    this.startRowRec();

    this._insert(func);

    this.endRowRec();
    this._startRec();
}

/**
 * オブジェクト挿入の中身
 * 1つの履歴の中でオブジェクト挿入をしたい場合はこっち
 * @private
 */
Bredima.ExpCtrl.prototype._insert = function(func) {
    var origin = this.current;
    var ret = origin.split();
    var newelem = func({parent: origin.row, value: ret.selectedValue});
    origin.row.add(newelem, origin.getOrder() + 1);
    
    var ipt;
    if((ipt = newelem.getIncomingInput())) {
	ipt.selectAll();
    }
    else {
	ipt = ret.addedInput.getIncomingInput();
	ipt.setCursorTo('head');
    }
    this.current = ipt;

    // 再配置
    origin.repositionAll();
    newelem.repositionAll();
    ret.addedInput.repositionAll();
    newelem.repositionToRoot();
}

/* ====================
 数式内のフォーカス管理
 */

/**
 * 数式内のフォーカスを設定
 * @param {Object} フォーカスを置くBredima.Exp.Inputインスタンス
 */
Bredima.ExpCtrl.prototype.setFocus = function(input) {
    if(this.focus === input) return; // ウィンドウがアクティブになったときにも飛んでくるっぽいので対処
    this.focus = input;
    if(!this.isProcessing) {
	var prev = this.current;
	this.current = input;
	// 前のinputがhidden表示になるかもしれないので再配置
	if((prev) && (prev != input)) prev.repositionToRoot();

	this._startRec();

	// hiddenからの復帰があるので再配置
	input.repositionToRoot();
    }
}

/**
 * 設定したinput要素からフォーカスが離れたことを連絡
 */
Bredima.ExpCtrl.prototype.leaveFocus = function() {
    this.focus = '';
    if(!this.isProcessing) {
	this._endRec('noselection');
    }
}
    
/**
 * アクセサ - 現在対象となっているinput要素を返す
 */
Bredima.ExpCtrl.prototype.getCurrent = function() {return this.current; }

/* ====================
 数式内のカーソル処理
 */
/**
 * カーソルの横移動
 * @param {Object} rf 横移動元のRowfactor要素
 * @param {String} dir 移動方向 back/fwd
 */
Bredima.ExpCtrl.prototype.slideCursorFrom = function(rf, dir) {
    while(1) {
	var order = rf.getOrder();
	var c = (dir == 'back');
	var loc = (c ? 'tail' : 'head');
	if((c && order == 0) || (!c && order == rf.row.getLength() - 1)) {
	    // カーソルがrowをはみ出す場合
	    if(rf.row.container.constructor.classID == 'exproot')
		return false;
	    else if(rf.row.container.slideCursorFrom(rf.row, dir)) // コンテナオブジェクト内で済んだ場合
	    return true;
	    else
		rf = rf.row.container;
	}
	else{
	    // row内で済む場合
	    var neighbor = rf.row.getRowfactor(order + (c ? -1 : 1))
	    if(neighbor.setCursorTo(loc))
		return true;
	    else
		rf = neighbor;
	}
    }
    return false;
}

/**
 * カーソルの縦移動
 * @param {Object} rf 移動元のRowfactor要素
 * @param {String} dir 移動方向 up/down
 */
Bredima.ExpCtrl.prototype.liftCursorFrom = function(rf, dir) {
    while(1) {
	if(rf.row.container.constructor.classID == 'exproot')
	    return false;
	else if(rf.row.container.liftCursorFrom(rf, dir))
	    return true;
	else
	    rf = rf.row.container;
    }
}

/* ====================
 履歴管理
 */

/**
 * input内の記録を開始
 */
Bredima.ExpCtrl.prototype._startRec = function() {
    this.record = new Bredima.History.InputRecord(this.current);
    this.record.cache();
}

/**
 * input内の記録を終了
 */
Bredima.ExpCtrl.prototype._endRec = function(option) {
    this.record.commit(option);
    if(this.record.close()) this.history.add(this.record);
}

/**
 * rowの記録を開始
 */
Bredima.ExpCtrl.prototype.startRowRec = function() {
    this.isProcessing = true;
    this.record = new Bredima.History.RowRecord(this.current.row);
    this.record.commit();
}

/**
 * rowの記録を終了
 */
Bredima.ExpCtrl.prototype.endRowRec = function() {
    this.record.commit();
    this.record.close();
    this.history.add(this.record);
    var self = this;
    setTimeout(function(){self.isProcessing = false;}, 0);
}

/* ====================
 その他
 */

/**
 * 履歴の更新を受信して自身の更新とする
 * @private
 */
Bredima.ExpCtrl.prototype.onHistoryChange = function() {
    this.observer.notify('ExpressionChange');
}


/* ================================
 キーイベント処理
*/

/**
 * 制御キーの処理
 * ブラウザ依存の関係でkeydownとkeypressの両方で呼ばれる
 * inputオブジェクトが使用
 * @param {Object} input 処理元のinputオブジェクト
 * @param {Event} evt inputが受け取ったイベントオブジェクト
 */
Bredima.ExpCtrl.prototype.handleControlCharacter = function(evt) {
    evt = (evt) ? evt : ((window.event) ? event : null);
    if(!evt) return false;
    // keydownで既にハンドリング済だった場合
    if(evt.type == 'keypress' && this.isHandledInKeydown) {
	if(this.isPreventOnKeydown) Bredima.util.stopEvent(evt);
	this.isHandledInKeydown = false;
	return true;
    }
    this.isPreventOnKeydown = false;
    // 処理対象のキーなら処理
    if(Bredima.ExpCtrl._ctrlCharHandler[evt.keyCode]) {
	var h = Bredima.ExpCtrl._ctrlCharHandler[evt.keyCode];
	if(h.addCnd(evt)) {
	    this.isHandledInKeydown = (evt.type == 'keydown');
	    if(h.position != '') var selpos = this.focus.getSelection();
	    if((h.position == '')
		|| ((h.position == 'head') && (selpos.end == 0))
	       || ((h.position == 'tail') && (selpos.start == this.focus.obj.value.length))) {
		Bredima.util.stopEvent(evt);
		this.isPreventOnKeydown = true;
		h.func.call(this, this.focus);
	    }
	    return true;
	}
    }
    return false;
}

/*
 制御キーのキー別処理内容
*/
Bredima.ExpCtrl._ctrlCharHandler = new Array;

// BackSpace;
Bredima.ExpCtrl._ctrlCharHandler[Bredima.consts.keys.bs] = {
    addCnd : function(){ return true;},
    position : 'head',
    func : function(input) {
	var order = input.getOrder() - 1;
	if(order >= 0) {
	    if(!this.isProcessing) {
		this._endRec();
		this.startRowRec();
	    }
	    input.row.remove(input.row.getRowfactor(order));
	    if(order > 0) input.mergeWithPrevInput();
	}
    }
};

// ←
Bredima.ExpCtrl._ctrlCharHandler[Bredima.consts.keys.left] = {
    // Operaで%が誤爆するのでshiftKeyの判定で弾く
    addCnd : function(evt) { return (!evt.shiftKey); }, 
    position : 'head',
    func : function(input) {
	this.slideCursorFrom(input, 'back');
    }
};

// →
Bredima.ExpCtrl._ctrlCharHandler[Bredima.consts.keys.right] = {
    // 「'」が誤爆するのでOperaのみkeypressで判定
    addCnd : function(evt) { return ((evt.which == undefined) // IEを逃がす
				  || (evt.charCode != undefined) // Mozilla, Safariを逃がす
				  || ((evt.type == 'keypress') && (evt.which != Bredima.consts.keys.right)));},
    position: 'tail',
    func: function(input) {
	this.slideCursorFrom(input, 'fwd');
    }
};

// ↑
Bredima.ExpCtrl._ctrlCharHandler[Bredima.consts.keys.up] = {
    // Operaで&が誤爆するのでshiftKeyの判定で弾く
    addCnd: function(evt) { return (!evt.shiftKey); },
    position: '',
    func: function(input) {
	this.liftCursorFrom(input, 'up');
    }
};

// ↓
Bredima.ExpCtrl._ctrlCharHandler[Bredima.consts.keys.down] = {
    // Operaで「)」が誤爆するのでshiftKeyの判定で弾く
    addCnd: function(evt) { return (!evt.shiftKey); },
    position: '',
    func: function(input) {
	this.liftCursorFrom(input, 'down');
    }
};

/**
 * 通常の文字キーの処理
 * inputのkeypressハンドラから呼び出し
 * @param {Object} input 処理元のinputオブジェクト
 * @param {Event} evt inputが受け取ったイベントオブジェクト
 */
Bredima.ExpCtrl.prototype.handleNormalCharacter = function(evt) {
    evt = (evt) ? evt : ((window.event) ? event: null);
    if(!evt) return false;
    var code = (evt.charCode) ? evt.charCode : evt.keyCode;
    //1文字ハンドリング
    var classid = Bredima.ExpCtrl._normalChar[code];
    if(classid) {
	Bredima.util.stopEvent(evt);
	if(classid != 'noaction') {
	    var func = Bredima.Exp.getClassById(classid);
	    if(!this.isProcessing) {
		// inputの履歴を閉じてrowの履歴を開く
		this._endRec();
		this.startRowRec();
	    }
	    this._insert(function(rd){return new func(rd);});
	}
	return true;
    }
    // 文字列ハンドリング
    else {
	var index = 0;
	var selpos = '';
	// 入力文字と文字列の末尾が一致
	while((index = Bredima.ExpCtrl._strSuffix.indexOf(String.fromCharCode(code), index)) >= 0) {
	    var k = Bredima.ExpCtrl._replaceStrings[index];
	    if(!selpos) selpos = this.current.getSelection();
	    var str = this.current.toString();
	    var start = selpos.end - k.str.length + 1;
	    // 全部一致
	    if(str.substring(start, selpos.end) == k.strShort) {
		Bredima.util.stopEvent(evt);
		this.current.init({
				      value: str.substring(0, start) + str.substring(selpos.end, str.length),
				      selection: new Bredima.Range(start, start)
				  });
		// 機能文字列を削除した状態で履歴に記録
		this._endRec('temporary');
		this.startRowRec();
		if(k.op == 'string')
		    this._insert(function(rd){return new Bredima.Token.String(rd, k.str)});
		else
		    this._insert(function(rd){return new (Bredima.Exp.getClassById(k.str))(rd)});
		return true;
	    }
	    index++;
	}
    }
    return false;
}

// 1文字ハンドリング一覧
Bredima.ExpCtrl._normalCharFeatures = [
    {code: 13, make: 'noaction'}, // 殺すだけ
    {code: '(', make: 'fenced'},
    {code: ')', make: 'fenced'},
    {code: '^', make: 'sup'},
    {code: '_', make: 'sub'}
];

Bredima.ExpCtrl._normalChar = new Array;
// 処理時に扱いやすいように整形
(function() {
     for(var i = 0; i < Bredima.ExpCtrl._normalCharFeatures.length; i++) {
	 var k = Bredima.ExpCtrl._normalCharFeatures[i];
	 Bredima.ExpCtrl._normalChar[(typeof k.code == 'number') ? k.code : k.code.charCodeAt(0)] =
	     k.make;
     }
 })();

// 文字列ハンドリング一覧
Bredima.ExpCtrl._replaceStrings = [
    {str: 'sin', op: 'string'},
    {str: 'cos', op: 'string'},
    {str: 'tan', op: 'string'},
    {str: 'log', op: 'string'},
    {str: 'frac', op: 'frac'},
    {str: 'sqrt', op: 'sqrt'}
];

Bredima.ExpCtrl._strSuffix = '';
// 処理時に扱いやすいように整形
(function() {
     for(var i = 0; i < Bredima.ExpCtrl._replaceStrings.length; i++) {
	 var k = Bredima.ExpCtrl._replaceStrings[i];
	 Bredima.ExpCtrl._strSuffix += k.str.charAt(k.str.length - 1);
	 k.strShort = k.str.substring(0, k.str.length - 1);
     }
 })();
