
/**
 * @class mrow管理オブジェクト
 * @constructor
 * @extends Bredima.Exp
 * @param {Object} container このmrowを含むコンテナオブジェクトインスタンス
 * @param {Undef} def 初期値　文字列か概要オブジェクト
 * @param {Number} level mrow表示サイズのレベル(1/2)
 */
Bredima.Exp.Row = function(param) {
    this.container = param.parent;
    this.level = param.level;
    this.__super__(param);
    this.rowfactors = new Array;

    if(param.rundown)
	this.initFromRundown(param.rundown);
    else
	this.add(new Bredima.Token.Input({parent: this, value: param.value}));
}

Bredima.util.inherit(Bredima.Exp.Row, Bredima.Exp);

Bredima.util.setProperty(Bredima.Exp.Row, {
    MML: 'mrow',
    margin: 2
});

Bredima.Exp.register(Bredima.Exp.Row, 'row');

/* ====================
 Exp.Rowで定義
 */
/**
 * 概要オブジェクトより初期化
 * @param {Array} rundown 概要オブジェクト
 */
Bredima.Exp.Row.prototype.initFromRundown = function(rundown) {
    var t, rowfactor;
    for(var i = this.rowfactors.length - 1; i >= 0; i--) {
	this.remove(this.rowfactors[i]);
    }
    for(var i = 0; i < rundown.length; i++) {
	t = rundown[i];
	if(typeof t == 'number' || typeof t == 'string') {
	    this.add(new Bredima.Token.Input({parent: this, value: t}));
	}
	else if(typeof t == 'object'){
	    var func = Bredima.Exp.getClassById(t[0]);
	    if(func && (func.objType == 'container' || func.objType == 'token')) {
		this.add(new func({parent: this, rundown: t.slice(1)}));
	    }
	}
    }
}

/* ----------------------------
 配列要素操作関係
 */
/**
 * rowfactors配列に渡されたオブジェクトを追加する だけ 内部側からの呼び出し専用
 * @param {Object} rowfactor 配置するオブジェクト
 */
Bredima.Exp.Row.prototype.add = function(rowfactor, order) {
    if(!order)
	this.rowfactors.push(rowfactor);
    else {
	var at = this.rowfactors.splice(0, order);
	this.rowfactors = at.concat(rowfactor, this.rowfactors);
    }
    this.obj.appendChild(rowfactor.getDom());
}

/**
 * mrow内の指定したオブジェクトを削除する だけ 内部側からの呼び出し専用
 * @param {Object} rowfactor 削除するオブジェクト
 */
Bredima.Exp.Row.prototype.remove = function(rowfactor) {
    this.obj.removeChild(rowfactor.getDom());
    this.rowfactors.splice(this.orderOf(rowfactor), 1);
}

/**
 * オブジェクトが配置されている位置を返す
 * @param {Object} rowfactor 位置を調べたいオブジェクト
 */
Bredima.Exp.Row.prototype.orderOf = function(rowfactor) {
    for(var i = 0; i < this.rowfactors.length; i++) {
	if(this.rowfactors[i] == rowfactor) return i;
    }
    return null;
}

/**
 * row内の配列の要素数を返す
 */
Bredima.Exp.Row.prototype.getLength = function() { return this.rowfactors.length; }

/**
 * row内の指定位置の要素を返す
 * @param {Number} order 位置
 */
Bredima.Exp.Row.prototype.getRowfactor = function(order) { return this.rowfactors[order]; }

/* -----------------------------
 他
 */
/**
 * mrowオブジェクトのheadかtailにカーソルを置く
 * @param {String} loc カーソルを置く位置 head / tail
 */
Bredima.Exp.Row.prototype.setCursorToRow = function(loc) {
    this.rowfactors[((loc == 'head') ? 0 : (this.rowfactors.length - 1))].setCursorTo(loc);
}

Bredima.Exp.Row.prototype.getOrder = function() { return this.container.orderOf(this);}

Bredima.Exp.Row.prototype.getLevel = function() { return this.level; }

/* ====================
 Expの実装
   -----------------------------
 オブジェクト情報入出力関係
 */

Bredima.Exp.Row.prototype.toRundown = function() {
    var ret = new Array;
    for(var i = 0; i < this.rowfactors.length; i++) {
	ret.push(this.rowfactors[i].toRundown());
    }
    return ret;
}

Bredima.Exp.Row.prototype._toMMLContent = function(expand) {
    var out = new Array;
    var rf;
    var mml;
    for(var i = 0; i < this.rowfactors.length; i++) {
	rf = this.rowfactors[i];
	mml = rf.toMML(expand, (rf.isMMLNeedPrev) ? out.pop() : null);
	if(mml != '') out = out.concat(mml);
    }
    return out.join('');
}

Bredima.Exp.Row.prototype.toLatex = function(nofence) {
    var out = new Array;
    for(var i = 0; i < this.rowfactors.length; i++)
	out.push(this.rowfactors[i].toLatex());
    if(nofence)
	return out.join(' ');
    else
	return '{' + out.join(' ') + '}';
}

/* -----------------------------
 レイアウト関係
 */

Bredima.Exp.Row.prototype.repositionAll = function() {
    for(var i = 0; i < this.rowfactors.length; i++) {
	this.rowfactors[i].repositionAll();
    }
    this._reposition();
}

Bredima.Exp.Row.prototype.repositionToRoot = function() {
    this._reposition();
    this.container.repositionToRoot();
}

Bredima.Exp.Row.prototype._reposition = function() {
    var tmpsize = new Array;
    this.width = 0;
    this.top = 0;
    this.bottom = 0;
    for(var i = 0; i < this.rowfactors.length; i++) {
	var s = this.rowfactors[i].getPreferredSize();
	this.top = Math.max(this.top, s.top);
	this.bottom = Math.max(this.bottom, s.bottom);
	this.width += s.width;
	tmpsize[i] = s;
    }
    this.width += (this.rowfactors.length - 1) * 4 // 内容間の隙間
    this.height = this.top + this.bottom;
    this._resize();
    var left = 0;
    for(i = 0; i < this.rowfactors.length; i++) {
	this.rowfactors[i].setPos(left, this.top - tmpsize[i].top);
	left += (4 + tmpsize[i].width);
    }
}

Bredima.style.add(
    ['.row', 'margin: 2px;']
);