
/**
  @fileoverview
  コンテナオブジェクト（レイアウトスキーマ）

  Bredima.Container.
    Frac
    
 */
/* ===============================
*/
/**
 * @class 分数
 * @extends Bredima.Container
 * @constructor
 * @param {Object} param オブジェクト構築共通パラメタ
 */
Bredima.Container.Frac = function(param) {
    this.__super__(param);
    this.add('separator', new Bredima.Parts.Separator());
}

Bredima.util.inherit(Bredima.Container.Frac, Bredima.Container);

Bredima.util.setProperty(Bredima.Container.Frac, {
			     rowCount : 2,
			     MML : 'mfrac',
			     tex : 'frac',
			     verticalOrder : [0, 1]
			 });

Bredima.Exp.register(Bredima.Container.Frac, 'container', 'frac');

Bredima.Container.Frac.prototype._repositionContent = function(ps) {
    this.width = Math.max(ps[0].width, ps[1].width);
    this.top = ps[0].height;
    this.bottom = ps[1].height + 1;
    
    this._resize();
    this.rows[0].setPos((this.width - ps[0].width) / 2, 0);
    this.rows[1].setPos((this.width - ps[1].width) / 2, ps[0].height + 1);
    this.parts.separator.setPos(0, ps[0].height);
}

/* ===============================
*/
/**
 * @class 括弧
 * @extends Bredima.Container
 * @constructor
 * @param {Object} param オブジェクト構築共通パラメタ
 * @param {Number} type 括弧の種類 0: (), 1: {}, 2: []
 */
Bredima.Container.Fenced = function(param, type) {
    this.__super__(param);
    this.type = (this.config) ? this.config : ((type == null) ? 0 : type);
    //var menu = new Bredima.Parts.Popup(this.exp.bredima, 'fenced', this);
    this.add('open', new Bredima.Parts.Image(this.types[this.type].name + '_open.png'));
    this.add('close', new Bredima.Parts.Image(this.types[this.type].name + '_close.png'));
    this.add('menu', new Bredima.Parts.Popup(this.exp.bredima, 'fenced', this));
    this.parts.open.setPos(0, 0);
}

Bredima.util.inherit(Bredima.Container.Fenced, Bredima.Container);

Bredima.util.setProperty(Bredima.Container.Fenced, {
    MML: 'mfenced',
    padding: {top:0, right:16, bottom:0, left:8}
});

Bredima.Exp.register(Bredima.Container.Fenced, 'container', 'fenced');

Bredima.Container.Fenced.prototype.types = [
    {name: 'parenthesis', character: ['(', ')'], certified: true},
    {name: 'brace', character: ['{', '}']},
    {name: 'bracket', character: ['[', ']']}
];

/* ------------------
 Containerの実装
*/
Bredima.Container.Fenced.prototype._toRundownContentParam = function() {
    if(this.type != 0) return this.type;
}

/* ------------------
 Expの実装
*/
/**
 * MathML表記を返す
 * 開きタグが特殊なので全て構築
 */
Bredima.Container.Fenced.prototype.toMML = function(expand) {
    var attr = this.types[this.type];
    var out = (!attr.certified) ? ('open="' + attr.character[0] + '" close="' + attr.character[1] + '"') : '';
    return Bredima.util.addTag(this.MML, this.content.toMML(expand), out);
}

/**
 * LaTeX表記を返す
 * 閉じコマンドも必要なので全て構築
 */
Bredima.Container.Fenced.prototype.toLatex = function() {
    var out = '\\left' + this.types[this.type].character[0];
    out += this.content.toLatex('noFence');
    out += '\\right' + this.types[this.type].character[1];
    return out;
}

Bredima.Container.Fenced.prototype._repositionContent = function() {
    Bredima.util.applySuper('_repositionContent', this, arguments);
    this.parts.open.setSize(10, this.height);
    this.parts.close.setSize(10, this.height);
    this.parts.close.setPos(this.width - 16, 0);
    this.parts.menu.setPos(this.width - 10, this.height - 10);
}

/* ------------------
 PopupListenerの実装
*/
/**
 * ポップアップメニューの選択を受信して括弧の種類を変更
 * @privete
 */
Bredima.Container.Fenced.prototype.menuClickPerformed = function(arg) {
    if(this.type != arg) {
	this.exp.startRowRec(this.row);
	this.type = arg;
	this.parts.open.setURI(this.types[this.type].name + '_open.png');
	this.parts.close.setURI(this.types[this.type].name + '_close.png');
	this.exp.endRowRec('nofocus');
    }
}

/* ===============================
*/
/**
 * @class 平方根
 * @extends Bredima.Container
 * @constructor
 * @param {Object} param オブジェクト構築共通パラメタ
 */
Bredima.Container.Sqrt = function(param) {
    this.__super__(param);
    this.add('radical', new Bredima.Parts.Image('radical.png'));
    this.add('line', new Bredima.Parts.Separator());
    this.parts.radical.setPos(0, 2);
    this.parts.line.setPos(10, 2);
}

Bredima.util.inherit(Bredima.Container.Sqrt, Bredima.Container);

Bredima.util.setProperty(Bredima.Container.Sqrt, {
			     MML: 'msqrt',
			     tex: 'sqrt',
			     padding: {top: 4, right: 0, bottom: 0, left: 10}
			 });

Bredima.Exp.register(Bredima.Container.Sqrt, 'container', 'sqrt');

Bredima.Container.Sqrt.prototype._repositionContent = function() {
    Bredima.util.applySuper('_repositionContent', this, arguments);
    this.parts.radical.setSize(10, this.height - 2);
    this.parts.line.setSize(this.width - 10, 1);
}

/* ===============================
*/
/**
 * @class 累乗根
 * @extends Bredima.Container
 * @constructor
 * @param {Object} param オブジェクト構築共通パラメタ
 */
Bredima.Container.Root = function(param) {
    this.__super__(param);
    this.padding = {top: 4, right: 0, bottom: 0, left: 10}; // repositionで変更するのでインスタンス毎に持つ
    this.add('radical', new Bredima.Parts.Image('radical.png'));
    this.add('line', new Bredima.Parts.Separator());
    this.rows[0].setPos(0, 0);
}

Bredima.util.inherit(Bredima.Container.Root, Bredima.Container);

Bredima.util.setProperty(Bredima.Container.Root, {
			     rowCount: 2,
			     rowLevels: [2, 0],
			     primaryRow: 1,
			     MML: 'mroot',
			     tex: 'sqrt'
			 });

Bredima.Exp.register(Bredima.Container.Root, 'container', 'root');

Bredima.Container.Root.prototype._toMMLContent = function(expand) {
    return this.rows[1].toMML(expand) + this.rows[0].toMML(expand);
}

Bredima.Container.Root.prototype._toLatexContent = function() {
    return '[' + this.rows[0].toLatex('nofence') + ']' + this.rows[1].toLatex();
}

Bredima.Container.Root.prototype._repositionContent = function(ps) {
    var offset = Math.max((ps[0].height - ps[1].top), 4);
    this.top = offset + ps[1].top;
    this.bottom = ps[1].bottom;
    this.width = ps[0].width + 8 + ps[1].width;

    this._resize();
    this.rows[1].setPos(ps[0].width + 8, offset);
    this.parts.radical.setSize(10, ps[1].height + 2);
    this.parts.radical.setPos(ps[0].width - 2, offset - 2);
    this.parts.line.setSize(ps[1].width, 1);
    this.parts.line.setPos(ps[0].width + 8, offset - 2);
}

/* ===============================
*/
/**
 * @class 累乗
 * @extends Bredima.Container
 * @constructor
 * @param {Object} param オブジェクト構築共通パラメタ
 */
Bredima.Container.Sup = function(param) {
    this.__super__(param);
    this.content.setPos(0, 0);
}

Bredima.util.inherit(Bredima.Container.Sup, Bredima.Container);

Bredima.util.setProperty(Bredima.Container.Sup, {
			     MML: 'msup',
			     isMMLNeedPrev: true,
			     rowLevels: [2]
			 });

Bredima.Exp.register(Bredima.Container.Sup, 'container', 'sup');

Bredima.Container.Sup.prototype._toMMLContent = function(expand, prev) {
    if(prev == '') prev = '<mo></mo>';
    return prev + this.content.toMML(expand);
}

Bredima.Container.Sup.prototype.toLatex = function() {
    return '^' + this.content.toLatex();
}

Bredima.Container.Sup.prototype._repositionContent = function(ps) {
    this.top = ps.height;
    this.bottom = 0;
    this.width = ps.width;

    this._resize();
}

/* ===============================
*/
/**
 * @class 添字
 * @extends Bredima.Container
 * @constructor
 * @param {Object} param オブジェクト構築共通パラメタ
 */
Bredima.Container.Sub = function(param) {
    this.__super__(param);
    this.content.setPos(0, 0);
}

Bredima.util.inherit(Bredima.Container.Sub, Bredima.Container);

Bredima.util.setProperty(Bredima.Container.Sub, {
			     MML: 'msub',
			     isMMLNeedPrev: true,
			     rowLevels: [2]
			 });

Bredima.Exp.register(Bredima.Container.Sub, 'container', 'sub');

Bredima.Container.Sub.prototype._toMMLContent = function(expand, prev) {
    if(prev == '') prev = '<mo></mo>';
    return prev + this.content.toMML(expand);
}

Bredima.Container.Sub.prototype.toLatex = function() {
    return '_' + this.content.toLatex();
}

Bredima.Container.Sub.prototype._repositionContent = function(ps) {
    this.top = 0;
    this.bottom = ps.height;
    this.width = ps.width;

    this._resize();
}
