/*
 * $HeadURL: https://svn.apache.org/repos/asf/jakarta/httpcomponents/httpclient/tags/4.0-alpha2/module-client/src/main/java/org/apache/http/client/methods/HttpOptions.java $
 * $Revision: 569634 $
 * $Date: 2007-08-25 09:27:56 +0200 (Sat, 25 Aug 2007) $
 *
 * ====================================================================
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 * ====================================================================
 *
 * This software consists of voluntary contributions made by many
 * individuals on behalf of the Apache Software Foundation.  For more
 * information on the Apache Software Foundation, please see
 * <http://www.apache.org/>.
 *
 */

package org.apache.http.client.methods;

import java.net.URI;
import java.net.URISyntaxException;
import java.util.Collections;
import java.util.HashSet;
import java.util.Set;

import org.apache.http.Header;
import org.apache.http.HeaderElement;
import org.apache.http.HttpResponse;
import org.apache.http.ParseException;

/**
 * HTTP OPTIONS method.
 * <p>
 * The HTTP OPTIONS method is defined in section 9.2 of 
 * <a href="http://www.ietf.org/rfc/rfc2616.txt">RFC2616</a>:
 * <blockquote>
 *  The OPTIONS method represents a request for information about the
 *  communication options available on the request/response chain
 *  identified by the Request-URI. This method allows the client to
 *  determine the options and/or requirements associated with a resource,
 *  or the capabilities of a server, without implying a resource action
 *  or initiating a resource retrieval.
 * </blockquote>
 * </p>
 * 
 * @version $Revision: 569634 $
 * 
 * @since 4.0
 */
public class HttpOptions extends HttpRequestBase {

    public final static String METHOD_NAME = "OPTIONS";
    
    public HttpOptions() {
        super();
    }

    public HttpOptions(final URI uri) {
        super();
        setURI(uri);
    }

    public HttpOptions(final String uri) throws URISyntaxException {
        super();
        setURI(new URI(uri));
    }

    public String getMethod() {
        return METHOD_NAME;
    }
    
    public Set getAllowedMethods(final HttpResponse response)
        throws ParseException {

        if (response == null) {
            throw new IllegalArgumentException("HTTP response may not be null");
        }
        Header header = response.getFirstHeader("Allow");
        if (header == null) {
            return Collections.EMPTY_SET;
        }
        HeaderElement[] elements = header.getElements();
        Set methods = new HashSet(elements.length);
        for (int i = 0; i < elements.length; i++) {
            methods.add(elements[i].getName());
        }
        return methods;
    }
    
}
