﻿/*
 * FormMainUtil.cs
 * Copyright (c) 2008-2009 kbinani
 *
 * This file is part of Boare.Cadencii.
 *
 * Boare.Cadencii is free software; you can redistribute it and/or
 * modify it under the terms of the BSD License.
 *
 * Boare.Cadencii is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 */
using System;
using System.Collections.Generic;
using System.Drawing;
using System.Drawing.Drawing2D;
using System.IO;
using System.Media;
using System.Threading;
using System.Windows.Forms;

using Boare.Lib.AppUtil;
using Boare.Lib.Vsq;

namespace Boare.Cadencii {

    partial class FormMain {
        private void UpdateLayout() {
            int width = panel1.Width;
            int height = panel1.Height;

            picturePositionIndicator.Width = width;
            picturePositionIndicator.Height = _PICT_POSITION_INDICATOR_HEIGHT;

            hScroll.Width = width - pictureBox2.Width - pictureBox3.Width - trackBar.Width;
            hScroll.Height = _SCROLL_WIDTH;

            vScroll.Width = _SCROLL_WIDTH;
            vScroll.Height = height - _PICT_POSITION_INDICATOR_HEIGHT - _SCROLL_WIDTH;

            pictPianoRoll.Width = width - _SCROLL_WIDTH;
            pictPianoRoll.Height = height - _PICT_POSITION_INDICATOR_HEIGHT - _SCROLL_WIDTH;

            pictureBox3.Width = AppManager._KEY_LENGTH;
            pictureBox3.Height = _SCROLL_WIDTH;
            pictureBox2.Height = _SCROLL_WIDTH;
            trackBar.Height = _SCROLL_WIDTH;

            picturePositionIndicator.Top = 0;
            picturePositionIndicator.Left = 0;

            pictPianoRoll.Top = _PICT_POSITION_INDICATOR_HEIGHT;
            pictPianoRoll.Left = 0;

            vScroll.Top = _PICT_POSITION_INDICATOR_HEIGHT;
            vScroll.Left = width - _SCROLL_WIDTH;

            pictureBox3.Top = height - _SCROLL_WIDTH;
            pictureBox3.Left = 0;

            hScroll.Top = height - _SCROLL_WIDTH;
            hScroll.Left = pictureBox3.Width;

            trackBar.Top = height - _SCROLL_WIDTH;
            trackBar.Left = width - _SCROLL_WIDTH - trackBar.Width;

            pictureBox2.Top = height - _SCROLL_WIDTH;
            pictureBox2.Left = width - _SCROLL_WIDTH;
        }

        /// <summary>
        /// 現在の編集データを全て破棄する。DirtyCheckは行われない。
        /// </summary>
        private void ClearExistingData() {
            m_manager.ClearCommandBuffer();
            m_manager.ClearSelectedBezier();
            m_manager.ClearSelectedEvent();
            m_manager.ClearSelectedTempo();
            m_manager.ClearSelectedTimesig();
        }

        /// <summary>
        /// 保存されていない編集内容があるかどうかチェックし、必要なら確認ダイアログを出す。
        /// </summary>
        /// <returns>保存されていない保存内容などない場合、または、保存する必要がある場合で（保存しなくてよいと指定された場合または保存が行われた場合）にtrueを返す</returns>
        private bool DirtyCheck() {
            if ( m_edited ) {
                string file = m_manager.FileName;
                if ( file == "" ) {
                    file = "Untitled";
                } else {
                    file = Path.GetFileName( file );
                }
                DialogResult dr = MessageBox.Show( _( "Save this sequence?" ),
                                                   _( "Affirmation" ),
                                                   MessageBoxButtons.YesNoCancel,
                                                   MessageBoxIcon.Question );
                switch ( dr ) {
                    case DialogResult.Yes:
                        if ( m_manager.FileName == "" ) {
                            if ( saveVsqDialog.ShowDialog() == DialogResult.OK ) {
                                m_manager.SaveTo( saveVsqDialog.FileName );
                                return true;
                            } else {
                                return false;
                            }
                        } else {
                            m_manager.SaveTo( m_manager.FileName );
                            return true;
                        }
                        break;
                    case DialogResult.No:
                        return true;
                    default:
                        return false;
                }
            } else {
                return true;
            }
        }

        /// <summary>
        /// waveView用のwaveファイルを読込むスレッドで使用する
        /// </summary>
        /// <param name="arg"></param>
        private void LoadWaveThreadProc( object arg ) {
            string file = (string)arg;
            waveView.LoadWave( file );
        }

        /// <summary>
        /// menuVisualWaveform.Checkedの値をもとに、splitterContainer2の表示状態を更新します
        /// </summary>
        private void UpdateSplitContainer2Size() {
            if ( menuVisualWaveform.Checked ) {
                splitContainer2.Panel2MinSize = _SPL2_PANEL2_MIN_HEIGHT;
                splitContainer2.IsSplitterFixed = false;
                splitContainer2.SplitterWidth = _SPL_SPLITTER_WIDTH;
                if ( m_last_splitcontainer2_split_distance <= 0 || m_last_splitcontainer2_split_distance > splitContainer2.Height ) {
                    splitContainer2.SplitterDistance = (int)(splitContainer2.Height * 0.9);
                } else {
                    splitContainer2.SplitterDistance = m_last_splitcontainer2_split_distance;
                }
            } else {
                m_last_splitcontainer2_split_distance = splitContainer2.SplitterDistance;
                splitContainer2.Panel2MinSize = 0;
                splitContainer2.SplitterWidth = 0;
                splitContainer2.SplitterDistance = splitContainer2.Height;
                splitContainer2.IsSplitterFixed = true;
            }
        }

        /// <summary>
        /// trackSelectorに表示するコントロールのカーブの種類を、AppManager.EditorConfigの設定に応じて更新します
        /// </summary>
        private void UpdateTrackSelectorVisibleCurve() {
            if ( AppManager.EditorConfig.CurveVisibleVelocity ) {
                trackSelector.AddViewingCurve( CurveType.VEL );
            }
            if ( AppManager.EditorConfig.CurveVisibleAccent ) {
                trackSelector.AddViewingCurve( CurveType.Accent );
            }
            if ( AppManager.EditorConfig.CurveVisibleDecay ) {
                trackSelector.AddViewingCurve( CurveType.Decay );
            }
            if ( AppManager.EditorConfig.CurveVisibleVibratoRate ) {
                trackSelector.AddViewingCurve( CurveType.VibratoRate );
            }
            if ( AppManager.EditorConfig.CurveVisibleVibratoDepth ) {
                trackSelector.AddViewingCurve( CurveType.VibratoDepth );
            }
            if ( AppManager.EditorConfig.CurveVisibleDynamics ) {
                trackSelector.AddViewingCurve( CurveType.DYN );
            }
            if ( AppManager.EditorConfig.CurveVisibleBreathiness ) {
                trackSelector.AddViewingCurve( CurveType.BRE );
            }
            if ( AppManager.EditorConfig.CurveVisibleBrightness ) {
                trackSelector.AddViewingCurve( CurveType.BRI );
            }
            if ( AppManager.EditorConfig.CurveVisibleClearness ) {
                trackSelector.AddViewingCurve( CurveType.CLE );
            }
            if ( AppManager.EditorConfig.CurveVisibleOpening ) {
                trackSelector.AddViewingCurve( CurveType.OPE );
            }
            if ( AppManager.EditorConfig.CurveVisibleGendorfactor ) {
                trackSelector.AddViewingCurve( CurveType.GEN );
            }
            if ( AppManager.EditorConfig.CurveVisiblePortamento ) {
                trackSelector.AddViewingCurve( CurveType.POR );
            }
            if ( AppManager.EditorConfig.CurveVisiblePitchbend ) {
                trackSelector.AddViewingCurve( CurveType.PIT );
            }
            if ( AppManager.EditorConfig.CurveVisiblePitchbendSensitivity ) {
                trackSelector.AddViewingCurve( CurveType.PBS );
            }
            splitContainer1.Panel2MinSize = trackSelector.PreferredMinSize;
            this.MinimumSize = GetWindowMinimumSize();
        }

        /// <summary>
        /// ウィンドウの表示内容に応じて、ウィンドウサイズの最小値を計算します
        /// </summary>
        /// <returns></returns>
        private Size GetWindowMinimumSize() {
            Size current_minsize = this.MinimumSize;
            Size client = this.ClientSize;
            Size current = this.Size;
            return new Size( current_minsize.Width,
                             splitContainer1.Panel2MinSize +
                                _SCROLL_WIDTH + _PICT_POSITION_INDICATOR_HEIGHT + pictPianoRoll.MinimumSize.Height +
                                toolStripContainer.TopToolStripPanel.Height +
                                menuStripMain.Height + statusStrip1.Height +
                                (current.Height - client.Height) +
                                20 );
        }

        /// <summary>
        /// 現在のm_input_textboxの状態を元に、歌詞の変更を反映させるコマンドを実行します
        /// </summary>
        private void ExecuteLyricChangeCommand() {
            if ( !m_input_textbox.Enabled ){
                return;
            }
            if ( m_input_textbox.IsDisposed ) {
                return;
            }
            int selected = m_manager.Selected;
            string original_phrase = m_manager.SelectedEvent.LastSelected.Original.ID.LyricHandle.L0.Phrase;
            string original_symbol = m_manager.SelectedEvent.LastSelected.Original.ID.LyricHandle.L0.PhoneticSymbol;
            bool symbol_protected = m_manager.SelectedEvent.LastSelected.Original.ID.LyricHandle.L0.PhoneticSymbolProtected;
            bool phonetic_symbol_edit_mode = ((TagLyricTextBox)m_input_textbox.Tag).PhoneticSymbolEditMode;
#if DEBUG
                Common.DebugWriteLine( "    original_phase,symbol=" + original_phrase + "," + original_symbol );
                Common.DebugWriteLine( "    phonetic_symbol_edit_mode=" + phonetic_symbol_edit_mode );
                Common.DebugWriteLine( "    m_input_textbox.Text=" + m_input_textbox.Text );
#endif
            if ( (phonetic_symbol_edit_mode && m_input_textbox.Text != original_symbol) ||
                 (!phonetic_symbol_edit_mode && m_input_textbox.Text != original_phrase) ) {
                string phrase, phonetic_symbol;
                TagLyricTextBox kvp = (TagLyricTextBox)m_input_textbox.Tag;
                if ( phonetic_symbol_edit_mode ) {
                    phrase = kvp.BufferText;
                    phonetic_symbol = m_input_textbox.Text;
                    string[] spl = phonetic_symbol.Split( " ".ToCharArray(), StringSplitOptions.RemoveEmptyEntries );
                    List<string> list = new List<string>();
                    foreach ( string s in spl ) {
                        if ( VsqPhoneticSymbol.IsValidSymbol( s ) ) {
                            list.Add( s );
                        }
                    }
                    phonetic_symbol = "";
                    bool first = true;
                    foreach ( string s in list ) {
                        if ( first ) {
                            phonetic_symbol += s;
                            first = false;
                        } else {
                            phonetic_symbol += " " + s;
                        }
                    }
                    symbol_protected = true;
                } else {
                    phrase = m_input_textbox.Text;
                    if ( !symbol_protected ) {
                        SymbolTable.Attatch( phrase, out phonetic_symbol );
                    } else {
                        phonetic_symbol = original_symbol;
                    }
                }
#if DEBUG
                    Common.DebugWriteLine( "    phrase,phonetic_symbol=" + phrase + "," + phonetic_symbol );
#endif
                CadenciiCommand run = new CadenciiCommand(
                    VsqCommand.GCommandEventChangeLyric( selected,
                                                      m_manager.SelectedEvent.LastSelectedID,
                                                      phrase,
                                                      phonetic_symbol,
                                                      symbol_protected ) );
                m_manager.Register( m_manager.VsqFile.Execute( run ) );
                Edited = true;
            }
        }

        private void ReloadGameControler() {
            try {
                bool init_success = false;
                int num_joydev = VSTiProxy.JoyInit();
                if ( num_joydev <= 0 ) {
                    init_success = false;
                } else {
                    init_success = true;
                }
                if ( init_success ) {
                    m_game_mode = GameControlMode.Normal;
                    stripLblGameCtrlMode.Image = null;
                    stripLblGameCtrlMode.Text = m_game_mode.ToString();
                    m_timer = new System.Windows.Forms.Timer();
                    m_timer.Interval = 10;
                    m_timer.Tick += new EventHandler( m_timer_Tick );
                    m_timer.Start();
                }
            } catch ( Exception ex ) {
#if DEBUG
                Common.DebugWriteLine( "FormMain+ReloadGameControler" );
                Common.DebugWriteLine( "    ex=" + ex );
#endif
            }
        }

        /// <summary>
        /// 現在のゲームコントローラのモードに応じてstripLblGameCtrlModeの表示状態を更新します。
        /// </summary>
        private void UpdateGameStatus() {
            if ( m_game_mode == GameControlMode.Disabled ) {
                stripLblGameCtrlMode.Text = _( "Disabled" );
                stripLblGameCtrlMode.Image = (Bitmap)Properties.Resources.slash.Clone();
            } else if ( m_game_mode == GameControlMode.Cursor ) {
                stripLblGameCtrlMode.Text = _( "Cursor" );
                stripLblGameCtrlMode.Image = null;
            } else if ( m_game_mode == GameControlMode.Keyboard ) {
                stripLblGameCtrlMode.Text = _( "Keyboard" );
                stripLblGameCtrlMode.Image = (Bitmap)Properties.Resources.piano.Clone();
            } else if ( m_game_mode == GameControlMode.Normal ) {
                stripLblGameCtrlMode.Text = _( "Normal" );
                stripLblGameCtrlMode.Image = null;
            }
        }

        /// <summary>
        /// スクリプトフォルダ中のスクリプトへのショートカットを作成する
        /// </summary>
        private void UpdateScriptShortcut() {
            Dictionary<string, ScriptInvoker> old = new Dictionary<string, ScriptInvoker>();
            foreach ( ToolStripItem item in menuScript.DropDownItems ) {
                if ( !(item is ToolStripMenuItem) ) {
                    continue;
                }
                ToolStripMenuItem tsmi = (ToolStripMenuItem)item;
                if ( tsmi.DropDownItems.Count <= 0 ) {
                    continue;
                }
                if ( tsmi.DropDownItems[0].Tag != null && tsmi.DropDownItems[0].Tag is ScriptInvoker ) {
                    ScriptInvoker si = (ScriptInvoker)tsmi.DropDownItems[0].Tag;
                    old.Add( si.ScriptFile, si );
                }
            }
            menuScript.DropDownItems.Clear();
            string script_path = Path.Combine( AppManager.ApplicationDataPath, "script" );
            if ( !Directory.Exists( script_path ) ) {
                Directory.CreateDirectory( script_path );
            }

            DirectoryInfo current = new DirectoryInfo( script_path );
            int count = 0;
            List<FileInfo> files = new List<FileInfo>( current.GetFiles( "*.txt" ) );
            files.AddRange( current.GetFiles( "*.cs" ) );
            foreach ( FileInfo fi in files ) {
                count++;
                string fname = fi.FullName;
                ToolStripMenuItem item = new ToolStripMenuItem( Path.GetFileNameWithoutExtension( fname ) );
                ToolStripMenuItem dd_run = new ToolStripMenuItem( _( "Run" ) + "(&R)" );
                if ( old.ContainsKey( fname ) && old[fname] != null ) {
                    dd_run.Tag = old[fname];
                } else {
                    ScriptInvoker si2 = new ScriptInvoker();
                    si2.FileTimestamp = DateTime.MinValue;
                    si2.ScriptFile = fname;
                    dd_run.Tag = si2;
                } 
                dd_run.Click += new EventHandler( dd_run_Click );
                item.DropDownItems.Add( dd_run );
                menuScript.DropDownItems.Add( item );
            }
            old.Clear();
            if ( count > 0 ) {
                menuScript.DropDownItems.Add( new ToolStripSeparator() );
            }
            menuScript.DropDownItems.Add( menuScriptUpdate );
        }

        private void dd_run_Click( object sender, EventArgs e ) {
#if DEBUG
            Common.DebugWriteLine( "dd_run_Click" );
#endif
            ScriptInvoker si = (ScriptInvoker)((ToolStripMenuItem)sender).Tag;// m_manager.LoadScript( script_file );
            string script_file = si.ScriptFile;
#if DEBUG
            Common.DebugWriteLine( "    si.FileTimestamp=" + si.FileTimestamp );
            Common.DebugWriteLine( "    File.GetLastWriteTimeUtc( script_file )=" + File.GetLastWriteTimeUtc( script_file ) );
#endif
            if ( si.FileTimestamp != File.GetLastWriteTimeUtc( script_file ) || si.ScriptType == null || si.Serializer == null || si.EditVsqDelegate == null ) {
                si = m_manager.LoadScript( script_file );
                ((ToolStripMenuItem)sender).Tag = si;
#if DEBUG
                Common.DebugWriteLine( "    err_msg=" + si.ErrorMessage );
#endif
            }
            if ( si.EditVsqDelegate != null ) {
                m_manager.BgmPlayer.UnLoad();
                VsqFileEx old = (VsqFileEx)m_manager.VsqFile.Clone();
                if ( m_manager.InvokeScript( si ) ) {
                    VsqFileEx new_one = (VsqFileEx)m_manager.VsqFile.Clone();
                    m_manager.SetVsqFile( old, m_manager.FileName );
                    CadenciiCommand run = VsqFileEx.GCommandReplace( new_one );
                    m_manager.Register( m_manager.VsqFile.Execute( run ) );
                    Edited = true;
                    UpdateDrawObjectList();
                    RefreshScreen();
                }
            } else {
                using ( FormCompileResult dlg = new FormCompileResult( _( "Failed loading script." ), si.ErrorMessage ) ) {
                    dlg.ShowDialog();
                }
            }
        }

        /// <summary>
        /// プレイカーソルが見えるようスクロールする
        /// </summary>
        private void EnsureCursorVisible() {
            // カーソルが画面内にあるかどうか検査
            int clock_left = ClockFromXCoord( AppManager._KEY_LENGTH );
            int clock_right = ClockFromXCoord( pictPianoRoll.Width );
            int uwidth = clock_right - clock_left;
            if ( m_manager.CurrentClock < clock_left || clock_right < m_manager.CurrentClock ) {
                int cl_new_center = (m_manager.CurrentClock / uwidth) * uwidth + uwidth / 2;
                float f_draft = cl_new_center - (pictPianoRoll.Width / 2 + 34 - 70) / m_manager.ScaleX;
                if ( f_draft < 0f ) {
                    f_draft = 0;
                }
                int draft = (int)(f_draft);
                if ( draft < hScroll.Minimum ) {
                    draft = hScroll.Minimum;
                } else if ( hScroll.Maximum < draft ) {
                    draft = hScroll.Maximum;
                }
                if ( hScroll.Value != draft ) {
                    m_draw_start_index[m_manager.Selected - 1] = 0;
                    hScroll.Value = draft;
#if DEBUG
                    Common.DebugWriteLine( "EnsureCursorvisible" );
                    Common.DebugWriteLine( "    CurrentClock=" + m_manager.CurrentClock );
                    Common.DebugWriteLine( "    hScroll.Value=" + hScroll.Value );
#endif
                }
            }
        }

        private void ProcessSpecialShortcutKey( PreviewKeyDownEventArgs e ) {
#if DEBUG
            Common.DebugWriteLine( "ProcessSpecialShortcutKey" );
            Common.DebugWriteLine( "    e.KeyCode=" + e.KeyCode );
#endif
            if ( !m_input_textbox.Enabled ) {
                if ( e.KeyCode == Keys.Return ) {
                    m_manager.Playing = !m_manager.Playing;
                } else if ( e.KeyCode == Keys.Space ) {
                    //m_
                } else if ( (e.KeyCode & Keys.OemPeriod) == Keys.OemPeriod ) {
                    if ( m_manager.Playing ) {
                        m_manager.Playing = false;
                    } else {
                        if ( !m_manager.StartMarkerEnabled ) {
                            m_manager.CurrentClock = 0;
                        } else {
                            m_manager.CurrentClock = m_manager.StartMarker;
                        }
                        RefreshScreen();
                    }
                } else if ( (e.KeyCode & Keys.Add) == Keys.Add || (e.KeyCode & Keys.Oemplus) == Keys.Oemplus || (e.KeyCode & Keys.Right) == Keys.Right ) {
                    Forward();
                } else if ( (e.KeyCode & Keys.Subtract) == Keys.Subtract || (e.KeyCode & Keys.OemMinus) == Keys.OemMinus || (e.KeyCode & Keys.Left) == Keys.Left ) {
                    Rewind();
                }
            }
            return;
            #region OLD CODES DO NOT REMOVE
            /*if ( AppManager.EditorConfig.Platform == Platform.Macintosh ) {
                if ( AppManager.EditorConfig.CommandKeyAsControl ) {
                    #region menuStripMain
                    if ( e.Alt && e.KeyCode == Keys.N && menuFileNew.Enabled ) {
                        this.menuFileNew_Click( menuFileNew, null );
                        return;
                    } else if ( e.Alt && e.KeyCode == Keys.O && menuFileOpen.Enabled ) {
                        this.menuFileOpen_Click( menuFileOpen, null );
                        return;
                    } else if ( e.Alt && e.KeyCode == Keys.S && menuFileSave.Enabled ) {
                        this.menuFileSave_Click( menuFileSave, null );
                        return;
                    } else if ( e.Alt && e.KeyCode == Keys.Q && menuFileQuit.Enabled ) {
                        this.menuFileQuit_Click( menuFileQuit, null );
                        return;
                    } else if ( e.Alt && e.KeyCode == Keys.Z && menuEditUndo.Enabled ) {
                        this.menuEditUndo_Click( menuEditUndo, null );
                        return;
                    } else if ( e.Alt && e.Shift && e.KeyCode == Keys.Z && menuEditRedo.Enabled ) {
                        this.menuEditRedo_Click( this.menuEditRedo, null );
                        return;
                    } else if ( e.Alt && e.KeyCode == Keys.X && this.menuEditCut.Enabled ) {
                        this.menuEditCut_Click( this.menuEditCut, null );
                        return;
                    } else if ( e.Alt && e.KeyCode == Keys.C && this.menuEditCopy.Enabled ) {
                        this.menuEditCopy_Click( this.menuEditCopy, null );
                        return;
                    } else if ( e.Alt && e.KeyCode == Keys.V && this.menuEditPaste.Enabled ) {
                        this.menuEditPaste_Click( this.menuEditPaste, null );
                        return;
                    } else if ( e.Alt && e.KeyCode == Keys.A && this.menuEditSelectAll.Enabled ) {
                        this.menuEditSelectAll_Click( this.menuEditSelectAll, null );
                        return;
                    } else if ( e.Alt && e.Shift && this.menuEditSelectAllEvents.Enabled ) {
                        this.menuEditSelectAllEvents_Click( this.menuEditSelectAllEvents, null );
                        return;
                    } else if ( e.Alt && e.KeyCode == Keys.V && this.menuHiddenEditPaste.Enabled ) {
                        this.menuHiddenEditPaste_Click( this.menuHiddenEditPaste, null );
                        return;
                    } else if ( e.Alt && e.KeyCode == Keys.W && this.menuHiddenEditFlipToolPointerPencil.Enabled ) {
                        this.menuHiddenEditFlipToolPointerPencil_Click( this.menuHiddenEditFlipToolPointerPencil, null );
                        return;
                    } else if ( e.Alt && e.KeyCode == Keys.E && this.menuHiddenEditFlipToolPointerEraser.Enabled ) {
                        this.menuHiddenEditFlipToolPointerEraser_Click( this.menuHiddenEditFlipToolPointerEraser, null );
                        return;
                    } else if ( (e.KeyCode & Keys.Clear) == Keys.Clear && e.Alt && e.Shift && this.menuHiddenVisualForwardParameter.Enabled ) {
                        this.menuHiddenVisualForwardParameter_Click( this.menuHiddenVisualForwardParameter, null );
                        return;
                    } else if ( (e.KeyCode & Keys.LButton) == Keys.LButton && (e.KeyCode & Keys.LineFeed) == Keys.LineFeed && e.Alt && e.Shift && this.menuHiddenVisualBackwardParameter.Enabled ) {
                        this.menuHiddenVisualBackwardParameter_Click( this.menuHiddenVisualBackwardParameter, null );
                        return;
                    } else if ( (e.KeyCode & Keys.Clear) == Keys.Clear && e.Alt && this.menuHiddenTrackNext.Enabled ) {
                        this.menuHiddenTrackNext_Click( this.menuHiddenTrackNext, null );
                        return;
                    } else if ( (e.KeyCode & Keys.LButton) == Keys.LButton && (e.KeyCode & Keys.LineFeed) == Keys.LineFeed && e.Alt && this.menuHiddenTrackBack.Enabled ) {
                        this.menuHiddenTrackBack_Click( this.menuHiddenTrackBack, null );
                        return;
                    }
                    #endregion

                    #region cMenuPiano
                    if ( e.Alt && e.KeyCode == Keys.Z && cMenuPianoUndo.Enabled ) {
                        this.cMenuPianoUndo_Click( this.cMenuPianoUndo, null );
                        return;
                    } else if ( e.Alt && e.Shift && e.KeyCode == Keys.Z && this.cMenuPianoRedo.Enabled ) {
                        this.cMenuPianoRedo_Click( this.cMenuPianoRedo, null );
                        return;
                    } else if ( e.Alt && e.KeyCode == Keys.X && this.cMenuPianoCut.Enabled ) {
                        this.cMenuPianoCut_Click( this.cMenuPianoCut, null );
                        return;
                    } else if ( e.Alt && e.KeyCode == Keys.C && this.cMenuPianoCopy.Enabled ) {
                        this.cMenuPianoCopy_Click( this.cMenuPianoCopy, null );
                        return;
                    } else if ( e.Alt && e.KeyCode == Keys.A && cMenuPianoSelectAll.Enabled ) {
                        this.cMenuPianoSelectAll_Click( this.cMenuPianoSelectAll, null );
                        return;
                    } else if ( e.Alt && e.Shift && e.KeyCode == Keys.A && cMenuPianoSelectAllEvents.Enabled ) {
                        this.cMenuPianoSelectAllEvents_Click( this.cMenuPianoSelectAllEvents, null );
                        return;
                    }
                    #endregion

                    #region cMenuTrackSelector
                    if ( e.Alt && e.KeyCode == Keys.Z && cMenuTrackSelectorUndo.Enabled ) {
                        this.cMenuTrackSelectorUndo_Click( this.cMenuTrackSelectorUndo, null );
                        return;
                    } else if ( e.Alt && e.Shift && e.KeyCode == Keys.Z && this.cMenuTrackSelectorRedo.Enabled ) {
                        this.cMenuTrackSelectorRedo_Click( this.cMenuTrackSelectorRedo, null );
                        return;
                    } else if ( e.Alt && e.KeyCode == Keys.X && this.cMenuTrackSelectorCut.Enabled ) {
                        this.cMenuTrackSelectorCut_Click( this.cMenuTrackSelectorCut, null );
                        return;
                    } else if ( e.Alt && e.KeyCode == Keys.C && this.cMenuTrackSelectorCopy.Enabled ) {
                        this.cMenuTrackSelectorCopy_Click( this.cMenuTrackSelectorCopy, null );
                        return;
                    } else if ( e.Alt && e.KeyCode == Keys.V && this.cMenuTrackSelectorPaste.Enabled ) {
                        this.cMenuTrackSelectorPaste_Click( this.cMenuTrackSelectorPaste, null );
                        return;
                    } else if ( e.Alt && e.Shift && e.KeyCode == Keys.A && this.cMenuTrackSelectorSelectAll.Enabled ) {
                        this.cMenuTrackSelectorSelectAll_Click( this.cMenuTrackSelectorSelectAll, null );
                        return;
                    }
                    #endregion
                } else {
                    bool RButton = (e.KeyCode & Keys.RButton) == Keys.RButton;
                    bool Clear = (e.KeyCode & Keys.Clear) == Keys.Clear;
                    bool Return = (e.KeyCode & Keys.Return) == Keys.Return;
                    bool Pause = (e.KeyCode & Keys.Pause) == Keys.Pause;
                    bool FinalMode = (e.KeyCode & Keys.FinalMode) == Keys.FinalMode;
                    bool Cancel = (e.KeyCode & Keys.Cancel) == Keys.Cancel;
                    bool CapsLock = (e.KeyCode & Keys.CapsLock) == Keys.CapsLock;
                    bool LButton = (e.KeyCode & Keys.LButton) == Keys.LButton;
                    bool JunjaMode = (e.KeyCode & Keys.JunjaMode) == Keys.JunjaMode;
                    bool LineFeed = (e.KeyCode & Keys.LineFeed) == Keys.LineFeed;
                    bool ControlKey = (e.KeyCode & Keys.ControlKey) == Keys.ControlKey;
                    bool XButton1 = (e.KeyCode & Keys.XButton1) == Keys.XButton1;

                    #region menuStripMain
                    if ( RButton && Clear && (e.KeyCode & Keys.N) == Keys.N && menuFileNew.Enabled ) {
                        this.menuFileNew_Click( menuFileNew, null );
                        return;
                    } else if ( RButton && Return && (e.KeyCode & Keys.O) == Keys.O && menuFileOpen.Enabled ) {
                        this.menuFileOpen_Click( menuFileOpen, null );
                        return;
                    } else if ( Pause && (e.KeyCode & Keys.S) == Keys.S && menuFileSave.Enabled ) {
                        this.menuFileSave_Click( menuFileSave, null );
                        return;
                    } else if ( ControlKey && (e.KeyCode & Keys.Q) == Keys.Q && menuFileQuit.Enabled ) {
                        this.menuFileQuit_Click( menuFileQuit, null );
                        return;
                    } else if ( RButton && FinalMode && (e.KeyCode & Keys.Z) == Keys.Z && menuEditUndo.Enabled ) {
                        this.menuEditUndo_Click( menuEditUndo, null );
                        return;
                    } else if ( RButton && FinalMode && e.Shift && (e.KeyCode & Keys.Z) == Keys.Z && menuEditRedo.Enabled ) {
                        this.menuEditRedo_Click( this.menuEditRedo, null );
                        return;
                    } else if ( FinalMode && (e.KeyCode & Keys.X) == Keys.X && this.menuEditCut.Enabled ) {
                        this.menuEditCut_Click( this.menuEditCut, null );
                        return;
                    } else if ( Cancel && (e.KeyCode & Keys.C) == Keys.C && this.menuEditCopy.Enabled ) {
                        this.menuEditCopy_Click( this.menuEditCopy, null );
                        return;
                    } else if ( RButton && CapsLock && (e.KeyCode & Keys.V) == Keys.V && this.menuEditPaste.Enabled ) {
                        this.menuEditPaste_Click( this.menuEditPaste, null );
                        return;
                    } else if ( LButton && (e.KeyCode & Keys.A) == Keys.A && this.menuEditSelectAll.Enabled ) {
                        this.menuEditSelectAll_Click( this.menuEditSelectAll, null );
                        return;
                    } else if ( LButton && e.Shift && (e.KeyCode & Keys.A) == Keys.A && this.menuEditSelectAllEvents.Enabled ) {
                        this.menuEditSelectAllEvents_Click( this.menuEditSelectAllEvents, null );
                        return;
                    } else if ( RButton && CapsLock && (e.KeyCode & Keys.V) == Keys.V && this.menuHiddenEditPaste.Enabled ) {
                        this.menuHiddenEditPaste_Click( this.menuHiddenEditPaste, null );
                        return;
                    } else if ( JunjaMode && (e.KeyCode & Keys.W) == Keys.W && this.menuHiddenEditFlipToolPointerPencil.Enabled ) {
                        this.menuHiddenEditFlipToolPointerPencil_Click( this.menuHiddenEditFlipToolPointerPencil, null );
                        return;
                    } else if ( XButton1 && (e.KeyCode & Keys.E) == Keys.E && this.menuHiddenEditFlipToolPointerEraser.Enabled ) {
                        this.menuHiddenEditFlipToolPointerEraser_Click( this.menuHiddenEditFlipToolPointerEraser, null );
                        return;
                    } else if ( Clear && e.Control && e.Shift && this.menuHiddenVisualForwardParameter.Enabled ) {
                        this.menuHiddenVisualForwardParameter_Click( this.menuHiddenVisualForwardParameter, null );
                        return;
                    } else if ( LButton && LineFeed && e.Control && e.Shift && this.menuHiddenVisualBackwardParameter.Enabled ) {
                        this.menuHiddenVisualBackwardParameter_Click( this.menuHiddenVisualBackwardParameter, null );
                        return;
                    } else if ( Clear && e.Control && this.menuHiddenTrackNext.Enabled ) {
                        this.menuHiddenTrackNext_Click( this.menuHiddenTrackNext, null );
                        return;
                    } else if ( LButton && LineFeed && e.Control && this.menuHiddenTrackBack.Enabled ) {
                        this.menuHiddenTrackBack_Click( this.menuHiddenTrackBack, null );
                        return;
                    }
                    #endregion

                    #region cMenuPiano
                    if ( RButton && FinalMode && (e.KeyCode & Keys.Z) == Keys.Z && cMenuPianoUndo.Enabled ) {
                        this.cMenuPianoUndo_Click( this.cMenuPianoUndo, null );
                        return;
                    } else if ( RButton && FinalMode && e.Shift && (e.KeyCode & Keys.Z) == Keys.Z && this.cMenuPianoRedo.Enabled ) {
                        this.cMenuPianoRedo_Click( this.cMenuPianoRedo, null );
                        return;
                    } else if ( FinalMode && (e.KeyCode & Keys.X) == Keys.X && this.cMenuPianoCut.Enabled ) {
                        this.cMenuPianoCut_Click( this.cMenuPianoCut, null );
                        return;
                    } else if ( Cancel && (e.KeyCode & Keys.C) == Keys.C && this.cMenuPianoCopy.Enabled ) {
                        this.cMenuPianoCopy_Click( this.cMenuPianoCopy, null );
                        return;
                    } else if ( LButton && (e.KeyCode & Keys.A) == Keys.A && cMenuPianoSelectAll.Enabled ) {
                        this.cMenuPianoSelectAll_Click( this.cMenuPianoSelectAll, null );
                        return;
                    } else if ( LButton && e.Shift && (e.KeyCode & Keys.A) == Keys.A && cMenuPianoSelectAllEvents.Enabled ) {
                        this.cMenuPianoSelectAllEvents_Click( this.cMenuPianoSelectAllEvents, null );
                        return;
                    }
                    #endregion

                    #region cMenuTrackSelector
                    if ( RButton && FinalMode && (e.KeyCode & Keys.Z) == Keys.Z && cMenuTrackSelectorUndo.Enabled ) {
                        this.cMenuTrackSelectorUndo_Click( this.cMenuTrackSelectorUndo, null );
                        return;
                    } else if ( RButton && FinalMode && e.Shift && (e.KeyCode & Keys.Z) == Keys.Z && this.cMenuTrackSelectorRedo.Enabled ) {
                        this.cMenuTrackSelectorRedo_Click( this.cMenuTrackSelectorRedo, null );
                        return;
                    } else if ( FinalMode && (e.KeyCode & Keys.X) == Keys.X && this.cMenuTrackSelectorCut.Enabled ) {
                        this.cMenuTrackSelectorCut_Click( this.cMenuTrackSelectorCut, null );
                        return;
                    } else if ( Cancel && (e.KeyCode & Keys.C) == Keys.C && this.cMenuTrackSelectorCopy.Enabled ) {
                        this.cMenuTrackSelectorCopy_Click( this.cMenuTrackSelectorCopy, null );
                        return;
                    } else if ( RButton && CapsLock && (e.KeyCode & Keys.V) == Keys.V && this.cMenuTrackSelectorPaste.Enabled ) {
                        this.cMenuTrackSelectorPaste_Click( this.cMenuTrackSelectorPaste, null );
                        return;
                    } else if ( LButton && e.Shift && (e.KeyCode & Keys.A) == Keys.A && this.cMenuTrackSelectorSelectAll.Enabled ) {
                        this.cMenuTrackSelectorSelectAll_Click( this.cMenuTrackSelectorSelectAll, null );
                        return;
                    }
                    #endregion
                }
            } else {
                #region menuStripMain
                if ( e.Control && e.KeyCode == Keys.N && menuFileNew.Enabled ) {
                    this.menuFileNew_Click( menuFileNew, null );
                    return;
                } else if ( e.Control && e.KeyCode == Keys.O && menuFileOpen.Enabled ) {
                    this.menuFileOpen_Click( menuFileOpen, null );
                    return;
                } else if ( e.Control && e.KeyCode == Keys.S && menuFileSave.Enabled ) {
                    this.menuFileSave_Click( menuFileSave, null );
                    return;
                } else if ( e.Control && e.KeyCode == Keys.Q && menuFileQuit.Enabled ) {
                    this.menuFileQuit_Click( menuFileQuit, null );
                    return;
                } else if ( e.Control && e.KeyCode == Keys.Z && menuEditUndo.Enabled ) {
                    this.menuEditUndo_Click( menuEditUndo, null );
                    return;
                } else if ( e.Control && e.Shift && e.KeyCode == Keys.Z && menuEditRedo.Enabled ) {
                    this.menuEditRedo_Click( this.menuEditRedo, null );
                    return;
                } else if ( e.Control && e.KeyCode == Keys.X && this.menuEditCut.Enabled ) {
                    this.menuEditCut_Click( this.menuEditCut, null );
                    return;
                } else if ( e.Control && e.KeyCode == Keys.C && this.menuEditCopy.Enabled ) {
                    this.menuEditCopy_Click( this.menuEditCopy, null );
                    return;
                } else if ( e.Control && e.KeyCode == Keys.V && this.menuEditPaste.Enabled ) {
                    this.menuEditPaste_Click( this.menuEditPaste, null );
                    return;
                } else if ( e.Control && e.KeyCode == Keys.A && this.menuEditSelectAll.Enabled ) {
                    this.menuEditSelectAll_Click( this.menuEditSelectAll, null );
                    return;
                } else if ( e.Control && e.Shift && this.menuEditSelectAllEvents.Enabled ) {
                    this.menuEditSelectAllEvents_Click( this.menuEditSelectAllEvents, null );
                    return;
                } else if ( e.Control && e.KeyCode == Keys.V && this.menuHiddenEditPaste.Enabled ) {
                    this.menuHiddenEditPaste_Click( this.menuHiddenEditPaste, null );
                    return;
                } else if ( e.Control && e.KeyCode == Keys.W && this.menuHiddenEditFlipToolPointerPencil.Enabled ) {
                    this.menuHiddenEditFlipToolPointerPencil_Click( this.menuHiddenEditFlipToolPointerPencil, null );
                    return;
                } else if ( e.Control && e.KeyCode == Keys.E && this.menuHiddenEditFlipToolPointerEraser.Enabled ) {
                    this.menuHiddenEditFlipToolPointerEraser_Click( this.menuHiddenEditFlipToolPointerEraser, null );
                    return;
                } else if ( e.Control && e.Alt && (e.KeyCode & Keys.PageDown) == Keys.PageDown && this.menuHiddenVisualForwardParameter.Enabled ) {
                    this.menuHiddenVisualForwardParameter_Click( this.menuHiddenVisualForwardParameter, null );
                    return;
                } else if ( e.Control && e.Alt && (e.KeyCode & Keys.PageUp) == Keys.PageUp && this.menuHiddenVisualBackwardParameter.Enabled ) {
                    this.menuHiddenVisualBackwardParameter_Click( this.menuHiddenVisualBackwardParameter, null );
                    return;
                } else if ( e.Control && (e.KeyCode & Keys.PageDown) == Keys.PageDown && this.menuHiddenTrackNext.Enabled ) {
                    this.menuHiddenTrackNext_Click( this.menuHiddenTrackNext, null );
                    return;
                } else if ( e.Control && (e.KeyCode & Keys.PageUp) == Keys.PageUp && this.menuHiddenTrackBack.Enabled ) {
                    this.menuHiddenTrackBack_Click( this.menuHiddenTrackBack, null );
                    return;
                }
                #endregion

                #region cMenuPiano
                if ( e.Control && e.KeyCode == Keys.Z && cMenuPianoUndo.Enabled ) {
                    this.cMenuPianoUndo_Click( this.cMenuPianoUndo, null );
                    return;
                } else if ( e.Control && e.Shift && e.KeyCode == Keys.Z && this.cMenuPianoRedo.Enabled ) {
                    this.cMenuPianoRedo_Click( this.cMenuPianoRedo, null );
                    return;
                } else if ( e.Control && e.KeyCode == Keys.X && this.cMenuPianoCut.Enabled ) {
                    this.cMenuPianoCut_Click( this.cMenuPianoCut, null );
                    return;
                } else if ( e.Control && e.KeyCode == Keys.C && this.cMenuPianoCopy.Enabled ) {
                    this.cMenuPianoCopy_Click( this.cMenuPianoCopy, null );
                    return;
                } else if ( e.Control && e.KeyCode == Keys.A && cMenuPianoSelectAll.Enabled ) {
                    this.cMenuPianoSelectAll_Click( this.cMenuPianoSelectAll, null );
                    return;
                } else if ( e.Control && e.Shift && e.KeyCode == Keys.A && cMenuPianoSelectAllEvents.Enabled ) {
                    this.cMenuPianoSelectAllEvents_Click( this.cMenuPianoSelectAllEvents, null );
                    return;
                }
                #endregion

                #region cMenuTrackSelector
                if ( e.Control && e.KeyCode == Keys.Z && cMenuTrackSelectorUndo.Enabled ) {
                    this.cMenuTrackSelectorUndo_Click( this.cMenuTrackSelectorUndo, null );
                    return;
                } else if ( e.Control && e.Shift && e.KeyCode == Keys.Z && this.cMenuTrackSelectorRedo.Enabled ) {
                    this.cMenuTrackSelectorRedo_Click( this.cMenuTrackSelectorRedo, null );
                    return;
                } else if ( e.Control && e.KeyCode == Keys.X && this.cMenuTrackSelectorCut.Enabled ) {
                    this.cMenuTrackSelectorCut_Click( this.cMenuTrackSelectorCut, null );
                    return;
                } else if ( e.Control && e.KeyCode == Keys.C && this.cMenuTrackSelectorCopy.Enabled ) {
                    this.cMenuTrackSelectorCopy_Click( this.cMenuTrackSelectorCopy, null );
                    return;
                } else if ( e.Control && e.KeyCode == Keys.V && this.cMenuTrackSelectorPaste.Enabled ) {
                    this.cMenuTrackSelectorPaste_Click( this.cMenuTrackSelectorPaste, null );
                    return;
                } else if ( e.Control && e.Shift && e.KeyCode == Keys.A && this.cMenuTrackSelectorSelectAll.Enabled ) {
                    this.cMenuTrackSelectorSelectAll_Click( this.cMenuTrackSelectorSelectAll, null );
                    return;
                }
                #endregion
            }*/
            #endregion
        }

        private void SetHScrollRange( int draft_length ) {
            const int _ARROWS = 40; // 両端の矢印の表示幅px（おおよその値）
            draft_length += 240;
            if ( draft_length > hScroll.Maximum ) {
                hScroll.Maximum = draft_length;
            }
            int large_change = (int)((pictPianoRoll.Width - AppManager._KEY_LENGTH) / (float)m_manager.ScaleX);
            int box_width = (int)((hScroll.Width - _ARROWS) * large_change / (float)(hScroll.Maximum + large_change));
            if ( box_width < AppManager.EditorConfig.MinimumScrollHandleWidth ) {
                box_width = AppManager.EditorConfig.MinimumScrollHandleWidth;
                large_change = (int)((float)hScroll.Maximum * (float)box_width / (float)(hScroll.Width - _ARROWS - box_width));
            }
            if ( large_change > 0 ) {
                hScroll.LargeChange = large_change;
            }
        }

        private void SetVScrollRange( int draft_length ) {
            const int _ARROWS = 40; // 両端の矢印の表示幅px（おおよその値）
            if ( draft_length > vScroll.Maximum ) {
                vScroll.Maximum = draft_length;
            }
            int large_change = (int)pictPianoRoll.Height;
            int box_width = (int)((vScroll.Height - _ARROWS) * large_change / (float)(vScroll.Maximum + large_change));
            if ( box_width < AppManager.EditorConfig.MinimumScrollHandleWidth ) {
                box_width = AppManager.EditorConfig.MinimumScrollHandleWidth;
                large_change = (int)((float)vScroll.Maximum * (float)box_width / (float)(vScroll.Width - _ARROWS - box_width));
            }
            if ( large_change > 0 ) {
                vScroll.LargeChange = large_change;
            }
        }

        private static void ApplyRendererRecurse( Control control, ToolStripProfessionalRenderer renderer ) {
            ToolStrip ts = control as ToolStrip;
            if ( ts != null ) {
                if ( renderer == null ) {
                    ts.RenderMode = ToolStripRenderMode.System;
                } else {
                    ts.Renderer = renderer;
                }
            }
            MenuStrip ms = control as MenuStrip;
            if ( ms != null ) {
                if ( renderer == null ) {
                    ms.RenderMode = ToolStripRenderMode.System;
                } else {
                    ms.Renderer = renderer;
                }
            }
            foreach ( Control c in control.Controls ) {
                ApplyRendererRecurse( c, renderer );
            }
        }

        private void RefreshScreenCore() {
            pictPianoRoll.Refresh();
            picturePositionIndicator.Refresh();
            trackSelector.Refresh();
            if ( menuVisualWaveform.Checked ) {
                waveView.Draw();
                waveView.Refresh();
            }
        }

        private void RefreshScreen() {
            if ( !bgWorkScreen.IsBusy ) {
                bgWorkScreen.RunWorkerAsync();
            }
        }

        public void FlipMixerDialogVisible( bool visible ) {
            m_mixer_dlg.Visible = visible;
            AppManager.EditorConfig.MixerVisible = visible;
            menuVisualMixer.Checked = visible;
        }

        /// <summary>
        /// メニューのショートカットキーを、プラットフォームに応じて変更します
        /// </summary>
        private void ApplyPlatformDependentShortCut() {
            if ( AppManager.EditorConfig.Platform == Platform.Macintosh ) {
                string _CO = "";
                //if ( AppManager.EditorConfig.CommandKeyAsControl ) {
                _CO = new string( '\x2318', 1 );
                //} else {
                //_CO = "^";
                //}
                string _SHIFT = "⇧";
                //if ( AppManager.EditorConfig.CommandKeyAsControl ) {
                #region menuStripMain
                menuFileNew.ShortcutKeys = Keys.Alt | Keys.N;
                menuFileOpen.ShortcutKeys = Keys.Alt | Keys.O;
                menuFileSave.ShortcutKeys = Keys.Alt | Keys.S;
                menuFileQuit.ShortcutKeys = Keys.Alt | Keys.Q;

                menuEditUndo.ShortcutKeys = Keys.Alt | Keys.Z;
                menuEditRedo.ShortcutKeys = Keys.Alt | Keys.Shift | Keys.Z;
                menuEditCut.ShortcutKeys = Keys.Alt | Keys.X;
                menuEditCopy.ShortcutKeys = Keys.Alt | Keys.C;
                menuEditPaste.ShortcutKeys = Keys.Alt | Keys.V;
                menuEditSelectAll.ShortcutKeys = Keys.Alt | Keys.A;
                menuEditSelectAllEvents.ShortcutKeys = Keys.Alt | Keys.Shift | Keys.A;

                menuHiddenEditPaste.ShortcutKeys = Keys.Alt | Keys.V;
                menuHiddenEditFlipToolPointerPencil.ShortcutKeys = Keys.Alt | Keys.W;
                menuHiddenEditFlipToolPointerEraser.ShortcutKeys = Keys.Alt | Keys.E;
                menuHiddenVisualForwardParameter.ShortcutKeys = Keys.Clear | Keys.Alt | Keys.Shift;
                menuHiddenVisualBackwardParameter.ShortcutKeys = Keys.LButton | Keys.LineFeed | Keys.Alt | Keys.Shift;
                menuHiddenTrackNext.ShortcutKeys = Keys.Clear | Keys.Alt;
                menuHiddenTrackBack.ShortcutKeys = Keys.LButton | Keys.LineFeed | Keys.Alt;
                #endregion

                #region cMenuPiano
                cMenuPianoUndo.ShortcutKeys = Keys.Alt | Keys.Z;
                cMenuPianoRedo.ShortcutKeys = Keys.Alt | Keys.Shift | Keys.Z;
                cMenuPianoCut.ShortcutKeys = Keys.Alt | Keys.X;
                cMenuPianoCopy.ShortcutKeys = Keys.Alt | Keys.C;
                cMenuPianoSelectAll.ShortcutKeys = Keys.Alt | Keys.A;
                cMenuPianoSelectAllEvents.ShortcutKeys = Keys.Alt | Keys.Shift | Keys.A;
                #endregion

                #region cMenuTrackSelector
                cMenuTrackSelectorUndo.ShortcutKeys = Keys.Alt | Keys.Z;
                cMenuTrackSelectorRedo.ShortcutKeys = Keys.Alt | Keys.Shift | Keys.Z;
                cMenuTrackSelectorCut.ShortcutKeys = Keys.Alt | Keys.X;
                cMenuTrackSelectorCopy.ShortcutKeys = Keys.Alt | Keys.C;
                cMenuTrackSelectorPaste.ShortcutKeys = Keys.Alt | Keys.V;
                cMenuTrackSelectorSelectAll.ShortcutKeys = Keys.Alt | Keys.Shift | Keys.A;
                #endregion
                /*} else {
                    #region menuStripMain
                    menuFileNew.ShortcutKeys = Keys.RButton | Keys.Clear | Keys.N;
                    menuFileOpen.ShortcutKeys = Keys.RButton | Keys.Return | Keys.O;
                    menuFileSave.ShortcutKeys = Keys.Pause | Keys.S;
                    menuFileQuit.ShortcutKeys = Keys.ControlKey | Keys.Q;

                    menuEditUndo.ShortcutKeys = Keys.RButton | Keys.FinalMode | Keys.Z;
                    menuEditRedo.ShortcutKeys = Keys.RButton | Keys.FinalMode | Keys.Shift | Keys.Z;
                    menuEditCut.ShortcutKeys = Keys.FinalMode | Keys.X;
                    menuEditCopy.ShortcutKeys = Keys.Cancel | Keys.C;
                    menuEditPaste.ShortcutKeys = Keys.RButton | Keys.CapsLock | Keys.V;
                    menuEditSelectAll.ShortcutKeys = Keys.LButton | Keys.A;
                    menuEditSelectAllEvents.ShortcutKeys = Keys.LButton | Keys.Shift | Keys.A;

                    menuHiddenEditPaste.ShortcutKeys = Keys.RButton | Keys.CapsLock | Keys.V;
                    menuHiddenEditFlipToolPointerPencil.ShortcutKeys = Keys.JunjaMode | Keys.W;
                    menuHiddenEditFlipToolPointerEraser.ShortcutKeys = Keys.XButton1 | Keys.E;
                    menuHiddenVisualForwardParameter.ShortcutKeys = Keys.Clear | Keys.Control | Keys.Shift;
                    menuHiddenVisualBackwardParameter.ShortcutKeys = Keys.LButton | Keys.LineFeed | Keys.Control | Keys.Shift;
                    menuHiddenTrackNext.ShortcutKeys = Keys.Clear | Keys.Control;
                    menuHiddenTrackBack.ShortcutKeys = Keys.LButton | Keys.LineFeed | Keys.Control;
                    #endregion

                    #region cMenuPiano
                    cMenuPianoUndo.ShortcutKeys = Keys.RButton | Keys.FinalMode | Keys.Z;
                    cMenuPianoRedo.ShortcutKeys = Keys.RButton | Keys.FinalMode | Keys.Shift | Keys.Z;
                    cMenuPianoCut.ShortcutKeys = Keys.FinalMode | Keys.X;
                    cMenuPianoCopy.ShortcutKeys = Keys.Cancel | Keys.C;
                    cMenuPianoSelectAll.ShortcutKeys = Keys.LButton | Keys.A;
                    cMenuPianoSelectAllEvents.ShortcutKeys = Keys.LButton | Keys.Shift | Keys.A;
                    #endregion

                    #region cMenuTrackSelector
                    cMenuTrackSelectorUndo.ShortcutKeys = Keys.RButton | Keys.FinalMode | Keys.Z;
                    cMenuTrackSelectorRedo.ShortcutKeys = Keys.RButton | Keys.FinalMode | Keys.Shift | Keys.Z;
                    cMenuTrackSelectorCut.ShortcutKeys = Keys.FinalMode | Keys.X;
                    cMenuTrackSelectorCopy.ShortcutKeys = Keys.Cancel | Keys.C;
                    cMenuTrackSelectorPaste.ShortcutKeys = Keys.RButton | Keys.CapsLock | Keys.V;
                    cMenuTrackSelectorSelectAll.ShortcutKeys = Keys.LButton | Keys.Shift | Keys.A;
                    #endregion
                }*/
                menuFileNew.ShortcutKeyDisplayString = _CO + "N";
                menuFileOpen.ShortcutKeyDisplayString = _CO + "O";
                menuFileSave.ShortcutKeyDisplayString = _CO + "S";
                menuFileQuit.ShortcutKeyDisplayString = _CO + "Q";

                menuEditUndo.ShortcutKeyDisplayString = _CO + "Z";
                menuEditRedo.ShortcutKeyDisplayString = _SHIFT + _CO + "Z";
                menuEditCut.ShortcutKeyDisplayString = _CO + "X";
                menuEditCopy.ShortcutKeyDisplayString = _CO + "C";
                menuEditPaste.ShortcutKeyDisplayString = _CO + "V";
                menuEditSelectAll.ShortcutKeyDisplayString = _CO + "A";
                menuEditSelectAllEvents.ShortcutKeyDisplayString = _SHIFT + _CO + "A";

                cMenuPianoUndo.ShortcutKeyDisplayString = _CO + "Z";
                cMenuPianoRedo.ShortcutKeyDisplayString = _SHIFT + _CO + "Z";
                cMenuPianoCut.ShortcutKeyDisplayString = _CO + "X";
                cMenuPianoCopy.ShortcutKeyDisplayString = _CO + "C";
                cMenuPianoSelectAll.ShortcutKeyDisplayString = _CO + "A";
                cMenuPianoSelectAllEvents.ShortcutKeyDisplayString = _SHIFT + _CO + "A";

                cMenuTrackSelectorUndo.ShortcutKeyDisplayString = _CO + "Z";
                cMenuTrackSelectorRedo.ShortcutKeyDisplayString = _SHIFT + _CO + "Z";
                cMenuTrackSelectorCut.ShortcutKeyDisplayString = _CO + "X";
                cMenuTrackSelectorCopy.ShortcutKeyDisplayString = _CO + "C";
                cMenuTrackSelectorPaste.ShortcutKeyDisplayString = _CO + "V";
                cMenuTrackSelectorSelectAll.ShortcutKeyDisplayString = _SHIFT + _CO + "A";
            } else {
                #region menuStripMain
                menuFileNew.ShortcutKeys = Keys.Control | Keys.N;
                menuFileOpen.ShortcutKeys = Keys.Control | Keys.O;
                menuFileSave.ShortcutKeys = Keys.Control | Keys.S;
                menuFileQuit.ShortcutKeys = Keys.Control | Keys.Q;

                menuEditUndo.ShortcutKeys = Keys.Control | Keys.Z;
                menuEditRedo.ShortcutKeys = Keys.Control | Keys.Shift | Keys.Z;
                menuEditCut.ShortcutKeys = Keys.Control | Keys.X;
                menuEditCopy.ShortcutKeys = Keys.Control | Keys.C;
                menuEditPaste.ShortcutKeys = Keys.Control | Keys.V;
                menuEditSelectAll.ShortcutKeys = Keys.Control | Keys.A;
                menuEditSelectAllEvents.ShortcutKeys = Keys.Control | Keys.Shift | Keys.A;

                menuHiddenEditPaste.ShortcutKeys = Keys.Control | Keys.V;
                menuHiddenEditFlipToolPointerPencil.ShortcutKeys = Keys.Control | Keys.W;
                menuHiddenEditFlipToolPointerEraser.ShortcutKeys = Keys.Control | Keys.E;
                menuHiddenVisualForwardParameter.ShortcutKeys = Keys.Control | Keys.Alt | Keys.PageDown;
                menuHiddenVisualBackwardParameter.ShortcutKeys = Keys.Control | Keys.Alt | Keys.PageUp;
                menuHiddenTrackNext.ShortcutKeys = Keys.Control | Keys.PageDown;
                menuHiddenTrackBack.ShortcutKeys = Keys.Control | Keys.PageUp;
                #endregion

                #region cMenuPiano
                cMenuPianoUndo.ShortcutKeys = Keys.Control | Keys.Z;
                cMenuPianoRedo.ShortcutKeys = Keys.Control | Keys.Shift | Keys.Z;
                cMenuPianoCut.ShortcutKeys = Keys.Control | Keys.X;
                cMenuPianoCopy.ShortcutKeys = Keys.Control | Keys.C;
                cMenuPianoSelectAll.ShortcutKeys = Keys.Control | Keys.A;
                cMenuPianoSelectAllEvents.ShortcutKeys = Keys.Control | Keys.Shift | Keys.A;
                #endregion

                #region cMenuTrackSelector
                cMenuTrackSelectorUndo.ShortcutKeys = Keys.Control | Keys.Z;
                cMenuTrackSelectorRedo.ShortcutKeys = Keys.Control | Keys.Shift | Keys.Z;
                cMenuTrackSelectorCut.ShortcutKeys = Keys.Control | Keys.X;
                cMenuTrackSelectorCopy.ShortcutKeys = Keys.Control | Keys.C;
                cMenuTrackSelectorPaste.ShortcutKeys = Keys.Control | Keys.V;
                cMenuTrackSelectorSelectAll.ShortcutKeys = Keys.Control | Keys.Shift | Keys.A;
                #endregion
            }
        }

        /// <summary>
        /// ソングポジションを1小節進めます
        /// </summary>
        private void Forward() {
            bool playing = m_manager.Playing;
            if ( playing ) {
                return;
            }
            int current = m_manager.VsqFile.GetBarCountFromClock( m_manager.CurrentClock ) + 1;
            int new_clock = m_manager.VsqFile.GetClockFromBarCount( current );
            if ( new_clock <= hScroll.Maximum + (pictPianoRoll.Width - AppManager._KEY_LENGTH) / m_manager.ScaleX ) {
                m_manager.CurrentClock = new_clock;
                EnsureCursorVisible();
                m_manager.Playing = playing;
                RefreshScreen();
            }
        }

        /// <summary>
        /// ソングポジションを1小節戻します
        /// </summary>
        private void Rewind() {
            bool playing = m_manager.Playing;
            if ( playing ) {
                return;
            }
            int current = m_manager.VsqFile.GetBarCountFromClock( m_manager.CurrentClock );
            if ( current > 0 ) {
                current--;
            }
            int new_clock = m_manager.VsqFile.GetClockFromBarCount( current );
            m_manager.CurrentClock = new_clock;
            EnsureCursorVisible();
            m_manager.Playing = playing;
            RefreshScreen();
        }

        /// <summary>
        /// cMenuPianoの固定長音符入力の各メニューのチェック状態をm_pencil_modeを元に更新します
        /// </summary>
        public void UpdateCMenuPianoFixed() {
            cMenuPianoFixed01.CheckState = CheckState.Unchecked;
            cMenuPianoFixed02.CheckState = CheckState.Unchecked;
            cMenuPianoFixed04.CheckState = CheckState.Unchecked;
            cMenuPianoFixed08.CheckState = CheckState.Unchecked;
            cMenuPianoFixed16.CheckState = CheckState.Unchecked;
            cMenuPianoFixed32.CheckState = CheckState.Unchecked;
            cMenuPianoFixed64.CheckState = CheckState.Unchecked;
            cMenuPianoFixed128.CheckState = CheckState.Unchecked;
            cMenuPianoFixedOff.CheckState = CheckState.Unchecked;
            cMenuPianoFixedTriplet.CheckState = CheckState.Unchecked;
            cMenuPianoFixedDotted.CheckState = CheckState.Unchecked;
            switch ( m_pencil_mode.Mode ) {
                case PencilModeEnum.L1:
                    cMenuPianoFixed01.CheckState = CheckState.Checked;
                    break;
                case PencilModeEnum.L2:
                    cMenuPianoFixed02.CheckState = CheckState.Checked;
                    break;
                case PencilModeEnum.L4:
                    cMenuPianoFixed04.CheckState = CheckState.Checked;
                    break;
                case PencilModeEnum.L8:
                    cMenuPianoFixed08.CheckState = CheckState.Checked;
                    break;
                case PencilModeEnum.L16:
                    cMenuPianoFixed16.CheckState = CheckState.Checked;
                    break;
                case PencilModeEnum.L32:
                    cMenuPianoFixed32.CheckState = CheckState.Checked;
                    break;
                case PencilModeEnum.L64:
                    cMenuPianoFixed64.CheckState = CheckState.Checked;
                    break;
                case PencilModeEnum.L128:
                    cMenuPianoFixed128.CheckState = CheckState.Checked;
                    break;
                case PencilModeEnum.Off:
                    cMenuPianoFixedOff.CheckState = CheckState.Checked;
                    break;
            }
            cMenuPianoFixedTriplet.CheckState = (m_pencil_mode.Triplet) ? CheckState.Checked : CheckState.Unchecked;
            cMenuPianoFixedDotted.CheckState = (m_pencil_mode.Dot) ? CheckState.Checked : CheckState.Unchecked;
        }

        public bool DetermineCaptureForcus() {
            if ( (m_versioninfo == null || (m_versioninfo != null && m_versioninfo.IsDisposed)) &&
                 !m_mixer_dlg.Visible ) {
                return true;
            }
            return false;
        }

        private void ClearTempWave() {
            string tmppath = m_manager.TempWaveDir;

            for ( int i = 1; i <= 16; i++ ) {
                string file = Path.Combine( tmppath, i + ".wav" );
                if ( File.Exists( file ) ) {
                    for ( int error = 0; error < 100; error++ ) {
                        try {
                            File.Delete( file );
                            break;
                        } catch ( Exception ex ) {
#if DEBUG
                            bocoree.debug.push_log( "FormMain+ClearTempWave()" );
                            bocoree.debug.push_log( "    ex=" + ex.ToString() );
                            bocoree.debug.push_log( "    error_count=" + error );
#endif
                            System.Threading.Thread.Sleep( 100 );
                        }
                    }
                }
            }
        }

        private void Render( int[] tracks ) {
            string tmppath = m_manager.TempWaveDir;
            if ( !Directory.Exists( tmppath ) ) {
                Directory.CreateDirectory( tmppath );
            }
            string[] files = new string[tracks.Length];
            for ( int i = 0; i < tracks.Length; i++ ) {
                files[i] = Path.Combine( tmppath, tracks[i] + ".wav" );
            }
            using ( FormSynthesize dlg = new FormSynthesize( m_manager, m_manager.VsqFile, AppManager.EditorConfig.PreSendTime, tracks, files, m_manager.VsqFile.TotalClocks + 240 ) ) {
                if ( dlg.ShowDialog() == DialogResult.OK ) {
                    m_manager.VsqFile.Tracks[m_manager.Selected].ResetEditedArea();
                }
                int[] finished = dlg.Finished;
                for ( int i = 0; i < finished.Length; i++ ) {
                    m_manager.setRenderRequired( finished[i], false );
                }
            }
        }

        private void PlayPreviewSound( int note ) {
                KeySoundPlayer.Play( note );
        }

        private void MouseHoverEventGenerator( object arg ) {
            int note = (int)arg;
            if ( AppManager.EditorConfig.MouseHoverTime > 0 ) {
                Thread.Sleep( AppManager.EditorConfig.MouseHoverTime );
            }
            KeySoundPlayer.Play( note );
        }

        public void WaitForPreviewSinglePrepared() {
#if DEBUG
            Common.DebugWriteLine( "WaitForPreviewSinglePrepared" );
#endif
            DateTime wait_start = DateTime.Now;
            while ( !m_preview_single_prepared ) {
                double elapsed = DateTime.Now.Subtract( wait_start ).TotalSeconds;
                if ( elapsed > _WAIT_LIMIT ) {
                    break;
                }
                Application.DoEvents();
            }
            if ( m_preview_single_prepared ) {
                SoundPlayer sp = new SoundPlayer( m_preview_single_file );
                sp.Play();
            }
            m_preview_single_prepared = false;
        }

        public static string _( string id ) {
            return Messaging.GetMessage( id );
        }

        public void ApplyLanguage() {
            try {
                openVsqDialog.Filter = _( "VSQ Format(*.vsq)|*.vsq" ) + "|" + _( "Original Format(*.evsq)|*.evsq" ) + "|" + _( "All Files(*.*)|*.*" );
            } catch {
                openVsqDialog.Filter = "VSQ Format(*.vsq)|*.vsq|Original Format(*.evsq)|*.evsq|All Files(*.*)|*.*";
            }
            try {
                saveVsqDialog.Filter = _( "VSQ Format(*.vsq)|*.vsq" ) + "|" + _( "Original Format(*.evsq)|*.evsq" ) + "|" + _( "MIDI Format(*.mid)|*.mid" );
            } catch {
                saveVsqDialog.Filter = "VSQ Format(*.vsq)|*.vsq|Original Format(*.evsq)|*.evsq|MIDI Format(*.mid)|*.mid";
            }
            try {
                openUstDialog.Filter = _( "UTAU Script Format(*.ust)|*.ust" ) + "|" + _( "All Files(*.*)|*.*" );
            } catch {
                openUstDialog.Filter = "UTAU Script Format(*.ust)|*.ust|All Files(*.*)|*.*";
            }

            stripLblGameCtrlMode.ToolTipText = _( "Game Controler" );
            this.Invoke( new VoidDelegate( UpdateGameStatus ) );

            stripBtnPointer.Text = _( "Pointer" );
            stripBtnPointer.ToolTipText = _( "Pointer" );
            stripBtnPencil.Text = _( "Pencil" );
            stripBtnPencil.ToolTipText = _( "Pencil" );
            stripBtnLine.Text = _( "Line" );
            stripBtnLine.ToolTipText = _( "Line" );
            stripBtnEraser.Text = _( "Eraser" );
            stripBtnEraser.ToolTipText = _( "Eraser" );
            stripBtnCurve.Text = _( "Curve" );
            stripBtnCurve.ToolTipText = _( "Curve" );
            stripBtnGrid.Text = _( "Grid" );
            stripBtnGrid.ToolTipText = _( "Grid" );

            #region main menu
            menuFile.Text = _( "File" ) + "(&F)";
            menuFileNew.Text = _( "New" ) + "(&N)";
            menuFileOpen.Text = _( "Open" ) + "(&O)";
            menuFileSave.Text = _( "Save" ) + "(&S)";
            menuFileSaveNamed.Text = _( "Save As" ) + "(&A)";
            menuFileImport.Text = _( "Import" ) + "(&I)";
            menuFileImportUst.Text = _( "UTAU Project File" );
            menuFileExport.Text = _( "Export" ) + "(&E)";
            menuFileRecent.Text = _( "Recent Files" ) + "(&R)";
            menuFileQuit.Text = _( "Quit" ) + "(&Q)";

            menuEdit.Text = _( "Edit" ) + "(&E)";
            menuEditUndo.Text = _( "Undo" ) + "(&U)";
            menuEditRedo.Text = _( "Redo" ) + "(&R)";
            menuEditCut.Text = _( "Cut" ) + "(&T)";
            menuEditCopy.Text = _( "Copy" ) + "(&C)";
            menuEditPaste.Text = _( "Paste" ) + "(&P)";
            menuEditDelete.Text = _( "Delete" ) + "(&D)";
            menuEditAutoNormalizeMode.Text = _( "Auto Normalize Mode" ) + "(&N)";
            menuEditSelectAll.Text = _( "Select All" ) + "(&A)";
            menuEditSelectAllEvents.Text = _( "Select All Events" ) + "(&E)";

            menuVisual.Text = _( "View" ) + "(&V)";
            menuVisualControlTrack.Text = _( "Control Track" ) + "(&C)";
            menuVisualMixer.Text = _( "Mixer" ) + "(&X)";
            menuVisualWaveform.Text = _( "Waveform" ) + "(&W)";
            menuVisualGridline.Text = _( "Grid Line" ) + "(&G)";
            menuVisualStartMarker.Text = _( "Start Marker" ) + "(&S)";
            menuVisualEndMarker.Text = _( "End Marker" ) + "(&E)";
            menuVisualLyrics.Text = _( "Lyrics/Phoneme" ) + "(&L)";
            menuVisualNoteProperty.Text = _( "Note Expression/Vibrato" ) + "(&N)";
            
            menuJob.Text = _( "Job" ) + "(&J)";
            menuJobNormalize.Text = _( "Normalize Notes" ) + "(&N)";
            menuJobInsertBar.Text = _( "Insert Bars" ) + "(&I)";
            menuJobDeleteBar.Text = _( "Delete Bars" ) + "(&D)";
            menuJobRandomize.Text = _( "Randomize" ) + "(&R)";
            menuJobConnect.Text = _( "Connect Notes" ) + "(&C)";
            menuJobLyric.Text = _( "Insert Lyrics" ) + "(&L)";
            menuJobRewire.Text = _( "Import ReWire Host Tempo" ) + "(&T)";
            menuJobReloadVsti.Text = _( "Reload VSTi" ) + "(&R)";

            menuTrack.Text = _( "Track" ) + "(&T)";
            menuTrackOn.Text = _( "Track On" ) + "(&K)";
            menuTrackAdd.Text = _( "Add Track" ) + "(&A)";
            menuTrackCopy.Text = _( "Copy Track" ) + "(&C)";
            menuTrackChangeName.Text = _( "Rename Track" ) + "(&R)";
            menuTrackDelete.Text = _( "Delete Track" ) + "(&D)";
            menuTrackRenderCurrent.Text = _( "Render Current Track" ) + "(&T)";
            menuTrackRenderAll.Text = _( "Render All Tracks" ) + "(&S)";
            menuTrackOverlay.Text = _( "Overlay" ) + "(&O)";

            menuLyric.Text = _( "Lyrics" ) + "(&L)";
            menuLyricExpressionProperty.Text = _( "Note Expression Property" ) + "(&E)";
            menuLyricVibratoProperty.Text = _( "Note Vibrato Property" ) + "(&V)";
            menuLyricSymbol.Text = _( "Phoneme Transformation" ) + "(&T)";
            menuLyricDictionary.Text = _( "User Word Dictionary" ) + "(&C)";

            menuScript.Text = _( "Script" ) + "(&C)";
            menuScriptUpdate.Text = _( "Update Script List" ) + "(&U)";

            menuSetting.Text = _( "Setting" ) + "(&S)";
            menuSettingPreference.Text = _( "Preference" ) + "(&P)";
            menuSettingGameControler.Text = _( "Game Controler" ) + "(&G)";
            menuSettingGameControlerReload.Text = _( "Reload" ) + "(&R)";
            menuSettingGameControlerSetting.Text = _( "Setting" ) + "(&S)";
            menuSettingDefaultSingerStyle.Text = _( "Singing Style Defaults" ) + "(&D)";
            menuSettingPositionQuantize.Text = _( "Quantize" ) + "(&Q)";
            menuSettingPositionQuantizeOff.Text = _( "Off" );
            menuSettingPositionQuantizeTriplet.Text = _( "Triplet" );
            menuSettingLengthQuantize.Text = _( "Length" ) + "(&L)";
            menuSettingLengthQuantizeOff.Text = _( "Off" );
            menuSettingLengthQuantizeTriplet.Text = _( "Triplet" );
            menuSettingSingerProperty.Text = _( "Singer Properties" ) + "(&S)";
            menuSettingBgmSetSound.Text = _( "Set Sound File" ) + "(&S)";
            menuSettingBgmMute.Text = _( "Mute" ) + "(&M)";

            menuHelp.Text = _( "Help" ) + "(&H)";
            menuHelpAbout.Text = _( "About Cadencii" ) + "(&A)";
            #endregion

            #region cMenuPiano
            cMenuPianoPointer.Text = _( "Arrow" ) + "(&A)";
            cMenuPianoPencil.Text = _( "Pencil" ) + "(&W)";
            cMenuPianoEraser.Text = _( "Eraser" ) + "(&E)";
            cMenuPianoCurve.Text = _( "Curve" ) + "(&V)";

            cMenuPianoFixed.Text = _( "Note Fixed Length" ) + "(&N)";
            cMenuPianoFixedTriplet.Text = _( "Triplet" );
            cMenuPianoFixedOff.Text = _( "Off" );
            cMenuPianoFixedDotted.Text = _( "Dot" );
            cMenuPianoQuantize.Text = _( "Quantize" ) + "(&Q)";
            cMenuPianoQuantizeTriplet.Text = _( "Triplet" );
            cMenuPianoQuantizeOff.Text = _( "Off" );
            cMenuPianoLength.Text = _( "Length" ) + "(&L)";
            cMenuPianoLengthTriplet.Text = _( "Triplet" );
            cMenuPianoLengthOff.Text = _( "Off" );
            cMenuPianoGrid.Text = _( "Show/Hide Grid Line" ) + "(&S)";

            cMenuPianoUndo.Text = _( "Undo" ) + "(&U)";
            cMenuPianoRedo.Text = _( "Redo" ) + "(&R)";

            cMenuPianoCut.Text = _( "Cut" ) + "(&T)";
            cMenuPianoPaste.Text = _( "Paste" ) + "(&P)";
            cMenuPianoCopy.Text = _( "Copy" ) + "(&C)";
            cMenuPianoDelete.Text = _( "Delete" ) + "(&D)";

            cMenuPianoSelectAll.Text = _( "Select All" ) + "(&A)";
            cMenuPianoSelectAllEvents.Text = _( "Select All Events" ) + "(&E)";

            cMenuPianoExpressionProperty.Text = _( "Note Expression Property" ) + "(&P)";
            cMenuPianoVibratoProperty.Text = _( "Note Vibrato Property" );
            cMenuPianoImportLyric.Text = _( "Insert Lyrics" ) + "(&P)";
            #endregion

            #region cMenuTrackTab
            cMenuTrackTabTrackOn.Text = _( "Track On" ) + "(&K)";
            cMenuTrackTabAdd.Text = _( "Add Track" ) + "(&A)";
            cMenuTrackTabCopy.Text = _( "Copy Track" ) + "(&C)";
            cMenuTrackTabChangeName.Text = _( "Rename Track" ) + "(&R)";
            cMenuTrackTabDelete.Text = _( "Delete Track" ) + "(&D)";

            cMenuTrackTabRenderCurrent.Text = _( "Render Current Track" ) + "(&T)";
            cMenuTrackTabRenderAll.Text = _( "Render All Tracks" ) + "(&S)";
            cMenuTrackTabOverlay.Text = _( "Overlay" ) + "(&O)";
            #endregion

            #region cMenuTrackSelector
            cMenuTrackSelectorPointer.Text = _( "Arrow" ) + "(&A)";
            cMenuTrackSelectorPencil.Text = _( "Pencil" ) + "(&W)";
            cMenuTrackSelectorLine.Text = _( "Line" ) + "(&L)";
            cMenuTrackSelectorEraser.Text = _( "Eraser" ) + "(&E)";
            cMenuTrackSelectorCurve.Text = _( "Curve" ) + "(&V)";

            cMenuTrackSelectorUndo.Text = _( "Undo" ) + "(&U)";
            cMenuTrackSelectorRedo.Text = _( "Redo" ) + "(&R)";

            cMenuTrackSelectorCut.Text = _( "Cut" ) + "(&T)";
            cMenuTrackSelectorCopy.Text = _( "Copy" ) + "(&C)";
            cMenuTrackSelectorPaste.Text = _( "Paste" ) + "(&P)";
            cMenuTrackSelectorDelete.Text = _( "Delete" ) + "(&D)";

            cMenuTrackSelectorSelectAll.Text = _( "Select All Events" ) + "(&E)";
            #endregion
        }

        private void ImportLyric() {
#if DEBUG
            Common.DebugWriteLine( "ImportLyric" );
#endif
            int start = 0;
            int selectedid = m_manager.SelectedEvent.LastSelectedID;
            for ( int i = 0; i < m_manager.VsqFile.Tracks[m_manager.Selected].Events.Count; i++ ) {
                if ( selectedid == m_manager.VsqFile.Tracks[m_manager.Selected].Events[i].InternalID ) {
                    start = i;
                    break;
                }
            }
            int count = m_manager.VsqFile.Tracks[m_manager.Selected].Events.Count - 1 - start + 1;
#if DEBUG
            Common.DebugWriteLine( "    count=" + count );
#endif
            using ( FormImportLyric dlg = new FormImportLyric( count ) ) {
                if ( dlg.ShowDialog() == DialogResult.OK ) {
                    string[] phrases = dlg.GetLetters();
#if DEBUG
                    for ( int i = 0; i < phrases.Length; i++ ) {
                        Common.DebugWriteLine( "    " + phrases[i] );
                    }
#endif
                    int min = Math.Min( count, phrases.Length );
                    string[] new_phrases = new string[min];
                    string[] new_symbols = new string[min];
                    for ( int i = 0; i < min; i++ ) {
                        new_phrases[i] = phrases[i];
                        SymbolTable.Attatch( phrases[i], out new_symbols[i] );
                    }
                    VsqID[] new_ids = new VsqID[min];
                    int[] ids = new int[min];
                    for ( int i = start; i < start + min; i++ ) {
                        new_ids[i - start] = (VsqID)m_manager.VsqFile.Tracks[m_manager.Selected].Events[i].ID.Clone();
                        new_ids[i - start].LyricHandle.L0.Phrase = new_phrases[i - start];
                        new_ids[i - start].LyricHandle.L0.PhoneticSymbol = new_symbols[i - start];
                        ids[i - start] = m_manager.VsqFile.Tracks[m_manager.Selected].Events[i].InternalID;
                    }
                    CadenciiCommand run = new CadenciiCommand(
                        VsqCommand.GCommandEventChangeIDContaintsRange( m_manager.Selected, ids, new_ids ) );
                    m_manager.Register( m_manager.VsqFile.Execute( run ) );
                    Edited = true;
                    this.Refresh();
                }
            }
        }

        private void NoteVibratoProperty() {
            SelectedEventEntry item = m_manager.SelectedEvent.LastSelected;
            if ( item == null ) {
                return;
            }

            VsqEvent ev = item.Original;
            using ( FormVibratoConfig dlg = new FormVibratoConfig( ev.ID.VibratoHandle, ev.ID.Length, AppManager.EditorConfig.DefaultVibratoLength ) ) {
                if ( dlg.ShowDialog() == DialogResult.OK ) {
                    VsqEvent edited = (VsqEvent)ev.Clone();
                    if ( dlg.VibratoHandle != null ) {
                        edited.ID.VibratoHandle = (VibratoHandle)dlg.VibratoHandle.Clone();
                        edited.ID.VibratoDelay = ev.ID.Length - dlg.VibratoHandle.Length;
                    } else {
                        edited.ID.VibratoHandle = null;
                    }
                    CadenciiCommand run = new CadenciiCommand(
                        VsqCommand.GCommandEventChangeIDContaints( m_manager.Selected, ev.InternalID, (VsqID)edited.ID.Clone() ) );
                    m_manager.Register( m_manager.VsqFile.Execute( run ) );
                    Edited = true;
                    RefreshScreen();
                }
            }
        }

        private void NoteExpressionProperty() {
            SelectedEventEntry item = m_manager.SelectedEvent.LastSelected;
            if ( item == null ) {
                return;
            }

            VsqEvent ev = item.Original;
            using ( FormSingerStyleConfig dlg = new FormSingerStyleConfig( _( "Expression control property" ), false ) ) {
                dlg.PMBendDepth = ev.ID.PMBendDepth;
                dlg.PMBendLength = ev.ID.PMBendLength;
                dlg.PMbPortamentoUse = ev.ID.PMbPortamentoUse;
                dlg.DEMdecGainRate = ev.ID.DEMdecGainRate;
                dlg.DEMaccent = ev.ID.DEMaccent;
                if ( dlg.ShowDialog() == DialogResult.OK ) {
                    VsqEvent edited = (VsqEvent)ev.Clone();
                    edited.ID.PMBendDepth = dlg.PMBendDepth;
                    edited.ID.PMBendLength = dlg.PMBendLength;
                    edited.ID.PMbPortamentoUse = dlg.PMbPortamentoUse;
                    edited.ID.DEMdecGainRate = dlg.DEMdecGainRate;
                    edited.ID.DEMaccent = dlg.DEMaccent;
                    CadenciiCommand run = new CadenciiCommand(
                        VsqCommand.GCommandEventChangeIDContaints( m_manager.Selected, ev.InternalID, (VsqID)edited.ID.Clone() ) );
                    m_manager.Register( m_manager.VsqFile.Execute( run ) );
                    Edited = true;
                    RefreshScreen();
                }
            }
        }

        private int NewHScrollValueFromWheelDelta( int delta ) {
            double new_val = (double)hScroll.Value - delta * AppManager.EditorConfig.WheelOrder / (5.0 * m_manager.ScaleX);
            if ( new_val < 0.0 ) {
                new_val = 0;
            }
            int draft = (int)new_val;
            if ( draft > hScroll.Maximum ) {
                draft = hScroll.Maximum;
            } else if ( draft < hScroll.Minimum ) {
                draft = hScroll.Minimum;
            }
            return draft;
        }

        #region 音符の編集関連
        private void SelectAll() {
            m_manager.ClearSelectedEvent();
            m_manager.ClearSelectedTempo();
            m_manager.ClearSelectedTimesig();
            trackSelector.DisableSelectedRegion();
            int min = int.MaxValue;
            int max = int.MinValue;
            int premeasure = m_manager.VsqFile.PreMeasureClocks;
            foreach ( VsqEvent ve in m_manager.VsqFile.Tracks[m_manager.Selected].Events ) {
                if ( premeasure <= ve.Clock ) {
                    m_manager.AddSelectedEvent( ve.InternalID );
                    min = Math.Min( min, ve.Clock );
                    max = Math.Max( max, ve.Clock + ve.ID.Length );
                }
            }
            foreach ( CurveType vct in CurveType.GetEnumerator() ) {
                if ( vct.IsScalar || vct.IsAttachNote ) {
                    continue;
                }
                int count = m_manager.VsqFile.Tracks[m_manager.Selected][vct.Name].Count;
                if ( count >= 1 ) {
                    int max_key = m_manager.VsqFile.Tracks[m_manager.Selected][vct.Name].Keys[count - 1];
                    max = Math.Max( max, m_manager.VsqFile.Tracks[m_manager.Selected][vct.Name][max_key] );
                    for ( int i = 0; i < count; i++ ) {
                        int key = m_manager.VsqFile.Tracks[m_manager.Selected][vct.Name].Keys[i];
                        if ( premeasure <= key ) {
                            min = Math.Min( min, key );
                            break;
                        }
                    }
                }
            }
            if ( min < premeasure ) {
                min = premeasure;
            }
            if ( min < max ) {
                int stdx = m_manager.StartToDrawX;
                min = XCoordFromClocks( min ) + stdx;
                max = XCoordFromClocks( max ) + stdx;
                m_manager.SelectedRegion = new SelectedRegion( min );
                m_manager.SelectedRegion.SetEnd( max );
                m_manager.SelectedRegionEnabled = true;
            }
        }

        private void SelectAllEvent() {
            m_manager.ClearSelectedTempo();
            m_manager.ClearSelectedTimesig();
            m_manager.ClearSelectedEvent();
            trackSelector.DisableSelectedRegion();
            foreach ( VsqEvent ev in m_manager.VsqFile.Tracks[m_manager.Selected].Events ) {
                m_manager.AddSelectedEvent( ev.InternalID );
            }
            RefreshScreen();
        }

        private void DeleteEvent() {
#if DEBUG
            Common.DebugWriteLine( "DeleteEvent()" );
            Common.DebugWriteLine( "    m_input_textbox.Enabled=" + m_input_textbox.Enabled );
#endif
            if ( !m_input_textbox.Enabled ) {
                if ( m_manager.SelectedEvent.Count > 0 ) {
                    List<int> ids = new List<int>();
                    foreach ( SelectedEventEntry ev in m_manager.SelectedEvent.GetEnumerator() ) {
                        ids.Add( ev.Original.InternalID );
                    }
                    VsqCommand run = VsqCommand.GCommandEventDeleteRange( m_manager.Selected, ids.ToArray() );
                    if ( m_manager.SelectedRegionEnabled ) {
                        VsqFileEx work = (VsqFileEx)m_manager.VsqFile.Clone();
                        work.Execute( run );
                        int stdx = m_manager.StartToDrawX;
                        int start_clock = ClockFromXCoord( m_manager.SelectedRegion.Start - stdx );
                        int end_clock = ClockFromXCoord( m_manager.SelectedRegion.End - stdx );
                        List<List<BPPair>> curves = new List<List<BPPair>>();
                        List<CurveType> types = new List<CurveType>();
                        foreach ( CurveType vct in CurveType.GetEnumerator() ) {
                            if ( vct.IsScalar || vct.IsAttachNote ) {
                                continue;
                            }
                            List<BPPair> t = new List<BPPair>();
                            t.Add( new BPPair( start_clock, work.Tracks[m_manager.Selected][vct.Name][start_clock] ) );
                            t.Add( new BPPair( end_clock, work.Tracks[m_manager.Selected][vct.Name][end_clock] ) );
                            curves.Add( t );
                            types.Add( vct );
                        }
                        string[] strs = new string[types.Count];
                        for ( int i = 0; i < types.Count; i++ ) {
                            strs[i] = types[i].Name;
                        }
                        VsqCommand delete_curve = VsqCommand.GCommandTrackEditCurveRange( m_manager.Selected,
                                                                                          strs,
                                                                                          curves.ToArray() );
                        work.Execute( delete_curve );
                        CadenciiCommand run2 = new CadenciiCommand( VsqCommand.GCommandReplace( work ) );
                        m_manager.Register( m_manager.VsqFile.Execute( run2 ) );
                        Edited = true;
                    } else {
                        CadenciiCommand run2 = new CadenciiCommand( run );
                        m_manager.Register( m_manager.VsqFile.Execute( run2 ) );
                        Edited = true;
                        m_manager.ClearSelectedEvent();
                    }
                    this.Refresh();
                } else if ( m_manager.SelectedTempo.Count > 0 ) {
                    List<int> clocks = new List<int>();
                    foreach ( KeyValuePair<int, SelectedTempoEntry> ev in m_manager.SelectedTempo ) {
                        clocks.Add( ev.Key );
                    }
                    int[] dum = new int[clocks.Count];
                    for ( int i = 0; i < dum.Length; i++ ) {
                        dum[i] = -1;
                    }
                    CadenciiCommand run = new CadenciiCommand( VsqCommand.GCommandUpdateTempoRange( clocks.ToArray(), clocks.ToArray(), dum ) );
                    m_manager.Register( m_manager.VsqFile.Execute( run ) );
                    Edited = true;
                    m_manager.ClearSelectedTempo();
                    this.Refresh();
                } else if ( m_manager.SelectedTimesig.Count > 0 ) {
#if DEBUG
                    Common.DebugWriteLine( "    Timesig" );
#endif
                    int[] barcounts = new int[m_manager.SelectedTimesig.Count];
                    int[] numerators = new int[m_manager.SelectedTimesig.Count];
                    int[] denominators = new int[m_manager.SelectedTimesig.Count];
                    int count = -1;
                    foreach ( KeyValuePair<int, SelectedTimesigEntry> ev in m_manager.SelectedTimesig ) {
                        count++;
                        barcounts[count] = ev.Key;
                        numerators[count] = -1;
                        denominators[count] = -1;
                    }
                    CadenciiCommand run = new CadenciiCommand(
                        VsqCommand.GCommandUpdateTimesigRange( barcounts, barcounts, numerators, denominators ) );
                    m_manager.Register( m_manager.VsqFile.Execute( run ) );
                    Edited = true;
                    m_manager.ClearSelectedTimesig();
                    this.Refresh();
                }
                if ( trackSelector.SelectedRegionEnabled ) {
#if DEBUG
                    Common.DebugWriteLine( "    Curve" );
#endif
                    KeyValuePair<int, int> t = trackSelector.SelectedRegion;
                    int start = t.Key;
                    int end = t.Value;
                    List<BPPair> edit = new List<BPPair>();
                    string curve;
                    if ( CurveType.TryGetOriginal( trackSelector.SelectedCurve, out curve ) ) {
                        int value = m_manager.VsqFile.Tracks[m_manager.Selected][curve][start];
                        edit.Add( new BPPair( start, value ) );
                        edit.Add( new BPPair( end, value ) );
                        CadenciiCommand run = new CadenciiCommand(
                            VsqCommand.GCommandTrackEditCurve( m_manager.Selected, curve, edit ) );
                        m_manager.Register( m_manager.VsqFile.Execute( run ) );
                        Edited = true;
                    } else {
                        //todo: FormMain+DeleteEvent; VibratoDepth, VibratoRateの場合
                    }
                    trackSelector.DisableSelectedRegion();
                    RefreshScreen();
                }
            }
        }

        private void PasteEvent() {
            int clock = m_manager.CurrentClock;
            int unit = AppManager.GetPositionQuantizeClock();
            int odd = clock % unit;
            clock -= odd;
            if ( odd > unit / 2 ) {
                clock += unit;
            }
            int dclock = clock - AppManager.CopyStartedClock;

            VsqCommand add_event = null; // VsqEventを追加するコマンド

            if ( AppManager.CopiedEvent != null ) {
                // VsqEventのペーストを行うコマンドを発行
                if ( clock >= m_manager.VsqFile.PreMeasureClocks ) {
                    List<VsqEvent> paste = new List<VsqEvent>();
                    for ( int i = 0; i < AppManager.CopiedEvent.Count; i++ ) {
                        VsqEvent item = (VsqEvent)AppManager.CopiedEvent[i].Clone();
                        item.Clock = AppManager.CopiedEvent[i].Clock + dclock;
                        paste.Add( item );
                    }
                    add_event = VsqCommand.GCommandEventAddRange( m_manager.Selected, paste.ToArray() );
                }
            } else if ( AppManager.CopiedTempo != null ) {
                // テンポ変更の貼付けを実行
                int[] clocks = new int[AppManager.CopiedTempo.Count];
                int[] tempos = new int[AppManager.CopiedTempo.Count];
                for ( int i = 0; i < AppManager.CopiedTempo.Count; i++ ) {
                    clocks[i] = AppManager.CopiedTempo[i].Clock + dclock;
                    tempos[i] = AppManager.CopiedTempo[i].Tempo;
                }
                CadenciiCommand run = new CadenciiCommand( VsqCommand.GCommandUpdateTempoRange( clocks, clocks, tempos ) );
                m_manager.Register( m_manager.VsqFile.Execute( run ) );
                Edited = true;
                RefreshScreen();
                return;
            } else if ( AppManager.CopiedTimesig != null ) {
                // 拍子変更の貼付けを実行
                int bar_count = m_manager.VsqFile.GetBarCountFromClock( clock );
                int min_barcount = AppManager.CopiedTimesig[0].BarCount;
                foreach ( TimeSigTableEntry tste in AppManager.CopiedTimesig ) {
                    min_barcount = Math.Min( min_barcount, tste.BarCount );
                }
                int dbarcount = bar_count - min_barcount;
                int[] barcounts = new int[AppManager.CopiedTimesig.Count];
                int[] numerators = new int[AppManager.CopiedTimesig.Count];
                int[] denominators = new int[AppManager.CopiedTimesig.Count];
                for ( int i = 0; i < AppManager.CopiedTimesig.Count; i++ ) {
                    barcounts[i] = AppManager.CopiedTimesig[i].BarCount + dbarcount;
                    numerators[i] = AppManager.CopiedTimesig[i].Numerator;
                    denominators[i] = AppManager.CopiedTimesig[i].Denominator;
                }
                CadenciiCommand run = new CadenciiCommand(
                    VsqCommand.GCommandUpdateTimesigRange( barcounts, barcounts, numerators, denominators ) );
                m_manager.Register( m_manager.VsqFile.Execute( run ) );
                Edited = true;
                RefreshScreen();
                return;
            }

            // BPPairの貼付け
            VsqCommand edit_bpcurve = null; // BPListを変更するコマンド
            if ( AppManager.CopiedCurve != null && AppManager.CopiedCurve.Count > 0 ) {
                KeyValuePair<int, int> kvp = trackSelector.SelectedRegion;
                int max = Math.Max( kvp.Key, kvp.Value );
                
                Dictionary<string, List<BPPair>> target = new Dictionary<string, List<BPPair>>();
                foreach ( CurveType curve in AppManager.CopiedCurve.Keys ) {
                    if ( curve.IsScalar ) {
                        continue;
                    }
                    if ( curve.IsAttachNote ) {
                        //todo: FormMain+PasteEvent; VibratoRate, VibratoDepthカーブのペースト処理
                    } else {
                        int end = m_manager.VsqFile.Tracks[m_manager.Selected][curve.Name][max + dclock];
                        List<BPPair> edit = new List<BPPair>();
                        for ( int i = 0; i < AppManager.CopiedCurve[curve].Count; i++ ) {
                            edit.Add( new BPPair( AppManager.CopiedCurve[curve][i].Clock + dclock, AppManager.CopiedCurve[curve][i].Value ) );
                        }
                        edit[edit.Count - 1].Value = end;
                        bool changed = true;
                        while ( changed ) {
                            changed = false;
                            for ( int i = 0; i < edit.Count - 1; i++ ) {
                                if ( edit[i].Clock > edit[i + 1].Clock ) {
                                    BPPair tmp = new BPPair( edit[i].Clock, edit[i].Value );
                                    edit[i].Clock = edit[i + 1].Clock;
                                    edit[i].Value = edit[i + 1].Value;
                                    edit[i + 1].Clock = tmp.Clock;
                                    edit[i + 1].Value = tmp.Value;
                                    changed = true;
                                }
                            }
                        }
                        target.Add( curve.Name, edit );
                    }
                }
                if ( target.Count > 0 ) {
                    string[] curves = new string[target.Count];
                    List<BPPair>[] bplists = new List<BPPair>[target.Count];
                    int count = -1;
                    foreach ( string s in target.Keys ) {
                        count++;
                        curves[count] = s;
                        bplists[count] = target[s];
                    }
                    edit_bpcurve = VsqCommand.GCommandTrackEditCurveRange( m_manager.Selected, curves, bplists );
                }
                trackSelector.DisableSelectedRegion();
            }

            // ベジエ曲線の貼付け
            CadenciiCommand edit_bezier = null;
            if ( AppManager.CopiedBezier != null && AppManager.CopiedBezier.Count > 0 ) {
                AttachedCurve attached_curve = (AttachedCurve)m_manager.VsqFile.AttachedCurves[m_manager.Selected - 1].Clone();
                Dictionary<CurveType, List<BezierChain>> command_arg = new Dictionary<CurveType, List<BezierChain>>();
                foreach ( CurveType curve in AppManager.CopiedBezier.Keys ) {
                    if ( curve.IsScalar ) {
                        continue;
                    }
                    List<BezierChain> src = new List<BezierChain>();
                    foreach ( BezierChain bc in AppManager.CopiedBezier[curve] ) {
                        BezierChain bc_copy = (BezierChain)bc.Clone();
                        foreach ( BezierPoint bp in bc_copy.points ) {
                            bp.Base = new PointD( bp.Base.X + dclock, bp.Base.Y );
                        }
                        attached_curve.MergeBezierChain( curve, bc_copy );
                    }
                    List<BezierChain> arg = new List<BezierChain>();
                    foreach ( BezierChain bc in attached_curve[curve].Values ) {
                        arg.Add( bc );
                    }
                    command_arg.Add( curve, arg );
                }
                edit_bezier = VsqFileEx.GCommandReplaceAttachedCurveRange( m_manager.Selected, command_arg );
            }

            int commands = 0;
            commands += (add_event != null) ? 1 : 0;
            commands += (edit_bpcurve != null) ? 1 : 0;
            commands += (edit_bezier != null) ? 1 : 0;

            if ( commands == 1 ) {
                if ( add_event != null ) {
                    CadenciiCommand run = new CadenciiCommand( add_event );
                    m_manager.Register( m_manager.VsqFile.Execute( run ) );
                } else if ( edit_bpcurve != null ) {
                    CadenciiCommand run = new CadenciiCommand( edit_bpcurve );
                    m_manager.Register( m_manager.VsqFile.Execute( run ) );
                } else if ( edit_bezier != null ) {
                    m_manager.Register( m_manager.VsqFile.Execute( edit_bezier ) );
                }
                Edited = true;
                RefreshScreen();
            } else if ( commands > 1 ) {
                VsqFileEx work = (VsqFileEx)m_manager.VsqFile.Clone();
                if ( add_event != null ) {
                    work.Execute( add_event );
                }
                if ( edit_bpcurve != null ) {
                    work.Execute( edit_bpcurve );
                }
                if ( edit_bezier != null ) {
                    work.Execute( edit_bezier );
                }
                CadenciiCommand run = VsqFileEx.GCommandReplace( work );
                m_manager.Register( m_manager.VsqFile.Execute( run ) );
                Edited = true;
                RefreshScreen();
            }
        }

        /// <summary>
        /// アイテムのコピーを行います
        /// </summary>
        private void CopyEvent() {
#if DEBUG
            Common.DebugWriteLine( "CopyEvent" );
#endif
            AppManager.CopiedBezier = null;
            AppManager.CopiedCurve = null;
            AppManager.CopiedEvent = null;
            AppManager.CopiedTempo = null;
            AppManager.CopiedTimesig = null;
            int min = int.MaxValue; // コピーされたアイテムの中で、最小の開始クロック
            if ( m_manager.SelectedEvent.Count > 0 ) {
                AppManager.CopiedEvent = new List<VsqEvent>();
                foreach ( SelectedEventEntry item in m_manager.SelectedEvent.GetEnumerator() ) {
                    min = Math.Min( item.Original.Clock, min );
                    AppManager.CopiedEvent.Add( (VsqEvent)item.Original.Clone() );
                }
                if ( min != int.MaxValue ) {
                    AppManager.CopyStartedClock = min;
                }
            }

            if ( m_manager.SelectedRegionEnabled ) {
                int stdx = m_manager.StartToDrawX;
                int start_clock = ClockFromXCoord( m_manager.SelectedRegion.Start - stdx );
                int end_clock = ClockFromXCoord( m_manager.SelectedRegion.End - stdx );
                AppManager.CopyStartedClock = start_clock;
                AppManager.CopiedCurve = new Dictionary<CurveType, List<BPPair>>();
                AppManager.CopiedBezier = new Dictionary<CurveType, List<BezierChain>>();
                foreach ( CurveType vct in CurveType.GetEnumerator() ) {
                    if ( vct.IsScalar || vct.IsAttachNote ) {
                        continue;
                    }
                    List<BPPair> tmp = new List<BPPair>();
                    List<BezierChain> tmp_bezier = new List<BezierChain>();
                    CopyCurveCor( m_manager.Selected,
                                  vct,
                                  start_clock,
                                  end_clock,
                                  out tmp_bezier,
                                  out tmp );
                    AppManager.CopiedBezier.Add( vct, tmp_bezier );
                    AppManager.CopiedCurve.Add( vct, tmp );
                }
            } else if ( m_manager.SelectedTempo.Count > 0 ) {
                AppManager.CopiedTempo = new List<TempoTableEntry>();
                foreach ( KeyValuePair<int, SelectedTempoEntry> item in m_manager.SelectedTempo ) {
                    min = Math.Min( item.Value.Original.Clock, min );
                    AppManager.CopiedTempo.Add( (TempoTableEntry)item.Value.Original.Clone() );
                }
                if ( min != int.MaxValue ) {
                    AppManager.CopyStartedClock = min;
                }
            } else if ( m_manager.SelectedTimesig.Count > 0 ) {
                AppManager.CopiedTimesig = new List<TimeSigTableEntry>();
                foreach ( KeyValuePair<int, SelectedTimesigEntry> item in m_manager.SelectedTimesig ) {
                    min = Math.Min( item.Value.Original.Clock, min );
                    AppManager.CopiedTimesig.Add( (TimeSigTableEntry)item.Value.Original.Clone() );
                }
                if ( min != int.MaxValue ) {
                    AppManager.CopyStartedClock = min;
                }
            } else if ( trackSelector.SelectedRegionEnabled ) {
                AppManager.CopiedCurve = new Dictionary<CurveType,List<BPPair>>();
                AppManager.CopiedBezier = new Dictionary<CurveType,List<BezierChain>>();
                KeyValuePair<int, int> t = trackSelector.SelectedRegion;
                int start = t.Key;
                int end = t.Value;
                AppManager.CopyStartedClock = start;
                List<BezierChain> tmp_bezier;
                List<BPPair> tmp_bppair;
                CopyCurveCor( m_manager.Selected,
                              trackSelector.SelectedCurve,
                              start,
                              end,
                              out tmp_bezier,
                              out tmp_bppair );
                AppManager.CopiedCurve.Add( trackSelector.SelectedCurve, tmp_bppair );
                AppManager.CopiedBezier.Add( trackSelector.SelectedCurve, tmp_bezier );
            }
        }

        private void CutEvent() {
            // まずコピー
            CopyEvent();

            // 選択されたノートイベントがあれば、まず、削除を行うコマンドを発行
            VsqCommand delete_event = null;
            bool other_command_executed = false;
            if ( m_manager.SelectedEvent.Count > 0 ) {
                List<int> ids = new List<int>();
                foreach( SelectedEventEntry item in m_manager.SelectedEvent.GetEnumerator() ){
                    ids.Add( item.Original.InternalID );
                }
                delete_event = VsqCommand.GCommandEventDeleteRange( m_manager.Selected, ids.ToArray() );
            }

            // Ctrlキーを押しながらドラッグしたか、そうでないかで分岐
            if ( m_manager.SelectedRegionEnabled || trackSelector.SelectedRegionEnabled ) {
                int stdx = m_manager.StartToDrawX;
                int start_clock, end_clock;
                if ( m_manager.SelectedRegionEnabled ) {
                    start_clock = ClockFromXCoord( m_manager.SelectedRegion.Start - stdx );
                    end_clock = ClockFromXCoord( m_manager.SelectedRegion.End - stdx );
                } else {
                    start_clock = trackSelector.SelectedRegion.Key;
                    end_clock = trackSelector.SelectedRegion.Value;
                }

                // クローンを作成
                VsqFileEx work = (VsqFileEx)m_manager.VsqFile.Clone();
                if ( delete_event != null ) {
                    // 選択されたノートイベントがあれば、クローンに対して削除を実行
                    work.Execute( delete_event );
                }

                // BPListに削除処理を施す
                List<List<BPPair>> curves = new List<List<BPPair>>();
                List<string> types = new List<string>();
                foreach ( CurveType vct in CurveType.GetEnumerator() ) {
                    if ( vct.IsScalar || vct.IsAttachNote ) {
                        continue;
                    }
                    List<BPPair> t = new List<BPPair>();
                    t.Add( new BPPair( start_clock, work.Tracks[m_manager.Selected][vct.Name][start_clock] ) );
                    t.Add( new BPPair( end_clock, work.Tracks[m_manager.Selected][vct.Name][end_clock] ) );
                    curves.Add( t );
                    types.Add( vct.Name );
                }
                VsqCommand delete_curve = VsqCommand.GCommandTrackEditCurveRange( m_manager.Selected, types.ToArray(), curves.ToArray() );
                work.Execute( delete_curve );

                // ベジエ曲線に削除処理を施す
                List<CurveType> target_curve = new List<CurveType>();
                if ( m_manager.SelectedRegionEnabled ) {
                    // ctrlによる全選択モード
                    foreach ( CurveType ct in CurveType.GetEnumerator() ) {
                        if ( ct.IsScalar || ct.IsAttachNote ) {
                            continue;
                        }
                        target_curve.Add( ct );
                    }
                } else {
                    // 普通の選択モード
                    target_curve.Add( trackSelector.SelectedCurve );
                }
                work.AttachedCurves[m_manager.Selected - 1].DeleteBeziers( target_curve, start_clock, end_clock );

                // コマンドを発行し、実行
                CadenciiCommand run = VsqFileEx.GCommandReplace( work );
                m_manager.Register( m_manager.VsqFile.Execute( run ) );
                this.Edited = true;

                other_command_executed = true;
            } else if ( m_manager.SelectedTempo.Count > 0 ) {
                // テンポ変更のカット
                int count = -1;
                int[] dum = new int[m_manager.SelectedTempo.Count];
                int[] clocks = new int[m_manager.SelectedTempo.Count];
                foreach ( KeyValuePair<int, SelectedTempoEntry> item in m_manager.SelectedTempo ) {
                    count++;
                    dum[count] = -1;
                    clocks[count] = item.Value.Original.Clock;
                }
                CadenciiCommand run = new CadenciiCommand( VsqCommand.GCommandUpdateTempoRange( clocks, clocks, dum ) );
                m_manager.Register( m_manager.VsqFile.Execute( run ) );
                Edited = true;
                other_command_executed = true;
            } else if ( m_manager.SelectedTimesig.Count > 0 ) {
                // 拍子変更のカット
                int[] barcounts = new int[m_manager.SelectedTimesig.Count];
                int[] numerators = new int[m_manager.SelectedTimesig.Count];
                int[] denominators = new int[m_manager.SelectedTimesig.Count];
                int count = -1;
                foreach ( KeyValuePair<int, SelectedTimesigEntry> item in m_manager.SelectedTimesig ) {
                    count++;
                    barcounts[count] = item.Value.Original.BarCount;
                    numerators[count] = -1;
                    denominators[count] = -1;
                }
                CadenciiCommand run = new CadenciiCommand(
                    VsqCommand.GCommandUpdateTimesigRange( barcounts, barcounts, numerators, denominators ) );
                m_manager.Register( m_manager.VsqFile.Execute( run ) );
                Edited = true;
                other_command_executed = true;
            }

            // 冒頭で作成した音符イベント削除以外に、コマンドが実行されなかった場合
            if ( delete_event != null && !other_command_executed ) {
                CadenciiCommand run = new CadenciiCommand( delete_event );
                m_manager.Register( m_manager.VsqFile.Execute( run ) );
                Edited = true;
            }

            RefreshScreen();
        }

        private void CopyCurveCor(
            int track,
            CurveType curve_type, 
            int start, 
            int end,
            out List<BezierChain> copied_chain, 
            out List<BPPair> copied_curve 
        ) {
            copied_curve = new List<BPPair>();
            copied_curve.Add( new BPPair( start, m_manager.VsqFile.Tracks[track][curve_type.Name][start] ) );
            foreach ( int clock in m_manager.VsqFile.Tracks[track][curve_type.Name].GetKeyClockEnumerator() ) {
                if ( start < clock && clock <= end ) {
                    copied_curve.Add( new BPPair( clock, m_manager.VsqFile.Tracks[track][curve_type.Name][clock] ) );
                }
            }
            copied_chain = new List<BezierChain>();
            foreach ( BezierChain bc in m_manager.VsqFile.AttachedCurves[track - 1][curve_type].Values ) {
                int len = bc.points.Count;
                if ( len < 2 ) {
                    continue;
                }
                int chain_start = (int)bc.points[0].Base.X;
                int chain_end = (int)bc.points[len - 1].Base.X;
                if ( start < chain_start && chain_start < end && end < chain_end ) {
                    // (1) chain_start ~ end をコピー
                    copied_chain.Add( bc.ExtractPartialBezier( chain_start, end ) );
                } else if ( chain_start <= start && end <= chain_end ) {
                    // (2) start ~ endをコピー
                    copied_chain.Add( bc.ExtractPartialBezier( start, end ) );
                } else if ( chain_start < start && start < chain_end && chain_end <= end ) {
                    // (3) start ~ chain_endをコピー
                    copied_chain.Add( bc.ExtractPartialBezier( start, chain_end ) );
                } else if ( start <= chain_start && chain_end <= end ) {
                    // (4) 全部コピーでOK
                    copied_chain.Add( (BezierChain)bc.Clone() );
                }
            }
        }
        #endregion

        #region トラックの編集関連
        private void TrackCopyCore() {
            VsqTrack track = (VsqTrack)m_manager.VsqFile.Tracks[m_manager.Selected].Clone();
            track.Name += " (1)";
            CadenciiCommand run = new CadenciiCommand(
                VsqCommand.GCommandAddTrack( track, m_manager.VsqFile.Mixer.Slave[m_manager.Selected - 1], m_manager.VsqFile.Tracks.Count ) );
            m_manager.Register( m_manager.VsqFile.Execute( run ) );
            Edited = true;
            m_mixer_dlg.UpdateStatus();
            RefreshScreen();
        }

        private void ChangeTrackNameCore() {
            if ( m_txtbox_track_name != null ) {
                if ( !m_txtbox_track_name.IsDisposed ) {
                    m_txtbox_track_name.Dispose();
                }
                m_txtbox_track_name = null;
            }
            m_txtbox_track_name = new TextBoxEx();
            m_txtbox_track_name.Visible = false;
            int selector_width = trackSelector.SelectorWidth;
            int x = AppManager._KEY_LENGTH + (m_manager.Selected - 1) * selector_width;
            m_txtbox_track_name.Location = new Point( x, trackSelector.Height - TrackSelector._OFFSET_TRACK_TAB + 1 );
            m_txtbox_track_name.BorderStyle = BorderStyle.None;
            m_txtbox_track_name.Text = m_manager.VsqFile.Tracks[m_manager.Selected].Name;
            m_txtbox_track_name.KeyUp += new KeyEventHandler( m_txtbox_track_name_KeyUp );
            m_txtbox_track_name.Size = new Size( selector_width, TrackSelector._OFFSET_TRACK_TAB );
            m_txtbox_track_name.Parent = trackSelector;
            m_txtbox_track_name.Visible = true;
            m_txtbox_track_name.Focus();
            m_txtbox_track_name.SelectAll();
        }

        private void DeleteTrackCore() {
            int selected = m_manager.Selected;
            if ( MessageBox.Show(
                    string.Format( _( "Do you wish to remove track? {0} : \"{1}\"" ), selected, m_manager.VsqFile.Tracks[selected].Name ),
                    _APP_NAME,
                    MessageBoxButtons.YesNo,
                    MessageBoxIcon.Question ) == DialogResult.Yes ) {
                VsqFileEx temp = (VsqFileEx)m_manager.VsqFile.Clone();
                VsqCommand run = VsqCommand.GCommandDeleteTrack( m_manager.Selected );
                temp.Execute( run );
                temp.AttachedCurves.RemoveAt( m_manager.Selected - 1 );
                CadenciiCommand run2 = VsqFileEx.GCommandReplace( temp );
                if ( m_manager.Selected >= 2 ) {
                    m_manager.Selected -= 1;
                }
                m_manager.Register( m_manager.VsqFile.Execute( run2 ) );
                UpdateDrawObjectList();
                Edited = true;
                m_mixer_dlg.UpdateStatus();
                RefreshScreen();
            }
        }

        private void AddTrackCore() {
            int i = m_manager.VsqFile.Tracks.Count;
            string name = "Voice" + i;
            string singer = "Miku";
            VsqFileEx temp = (VsqFileEx)m_manager.VsqFile.Clone();
            VsqCommand run = VsqCommand.GCommandAddTrack( new VsqTrack( name, singer ), 
                                                          new VsqMixerEntry( 0, 0, 0, 0 ), 
                                                          i );
            temp.Execute( run );
            temp.AttachedCurves.Add( new AttachedCurve() );
            CadenciiCommand run2 = VsqFileEx.GCommandReplace( temp );
            m_manager.Register( m_manager.VsqFile.Execute( run2 ) );
            UpdateDrawObjectList();
            Edited = true;
            m_manager.Selected = i;
            m_mixer_dlg.UpdateStatus();
            RefreshScreen();
        }
        #endregion

        /// <summary>
        /// length, positionの各Quantizeモードに応じて、表示状態を更新します
        /// </summary>
        private void ApplyQuantizeMode() {
            cMenuPianoQuantize04.CheckState = CheckState.Unchecked;
            cMenuPianoQuantize08.CheckState = CheckState.Unchecked;
            cMenuPianoQuantize16.CheckState = CheckState.Unchecked;
            cMenuPianoQuantize32.CheckState = CheckState.Unchecked;
            cMenuPianoQuantize64.CheckState = CheckState.Unchecked;
            cMenuPianoQuantize128.CheckState = CheckState.Unchecked;
            cMenuPianoQuantizeOff.CheckState = CheckState.Unchecked;

            stripDDBtnQuantize04.CheckState = CheckState.Unchecked;
            stripDDBtnQuantize08.CheckState = CheckState.Unchecked;
            stripDDBtnQuantize16.CheckState = CheckState.Unchecked;
            stripDDBtnQuantize32.CheckState = CheckState.Unchecked;
            stripDDBtnQuantize64.CheckState = CheckState.Unchecked;
            stripDDBtnQuantize128.CheckState = CheckState.Unchecked;
            stripDDBtnQuantizeOff.CheckState = CheckState.Unchecked;

            menuSettingPositionQuantize04.CheckState = CheckState.Unchecked;
            menuSettingPositionQuantize08.CheckState = CheckState.Unchecked;
            menuSettingPositionQuantize16.CheckState = CheckState.Unchecked;
            menuSettingPositionQuantize32.CheckState = CheckState.Unchecked;
            menuSettingPositionQuantize64.CheckState = CheckState.Unchecked;
            menuSettingPositionQuantize128.CheckState = CheckState.Unchecked;
            menuSettingPositionQuantizeOff.CheckState = CheckState.Unchecked;

            stripDDBtnQuantize.Text = "QUANTIZE " + QuantizeModeUtil.GetString( AppManager.EditorConfig.PositionQuantize );
            switch ( AppManager.EditorConfig.PositionQuantize ) {
                case QuantizeMode.p4:
                    cMenuPianoQuantize04.CheckState = CheckState.Checked;
                    stripDDBtnQuantize04.CheckState = CheckState.Checked;
                    menuSettingPositionQuantize04.CheckState = CheckState.Checked;
                    break;
                case QuantizeMode.p8:
                    cMenuPianoQuantize08.CheckState = CheckState.Checked;
                    stripDDBtnQuantize08.CheckState = CheckState.Checked;
                    menuSettingPositionQuantize08.CheckState = CheckState.Checked;
                    break;
                case QuantizeMode.p16:
                    cMenuPianoQuantize16.CheckState = CheckState.Checked;
                    stripDDBtnQuantize16.CheckState = CheckState.Checked;
                    menuSettingPositionQuantize16.CheckState = CheckState.Checked;
                    break;
                case QuantizeMode.p32:
                    cMenuPianoQuantize32.CheckState = CheckState.Checked;
                    stripDDBtnQuantize32.CheckState = CheckState.Checked;
                    menuSettingPositionQuantize32.CheckState = CheckState.Checked;
                    break;
                case QuantizeMode.p64:
                    cMenuPianoQuantize64.CheckState = CheckState.Checked;
                    stripDDBtnQuantize64.CheckState = CheckState.Checked;
                    menuSettingPositionQuantize64.CheckState = CheckState.Checked;
                    break;
                case QuantizeMode.p128:
                    cMenuPianoQuantize128.CheckState = CheckState.Checked;
                    stripDDBtnQuantize128.CheckState = CheckState.Checked;
                    menuSettingPositionQuantize128.CheckState = CheckState.Checked;
                    break;
                case QuantizeMode.off:
                    cMenuPianoQuantizeOff.CheckState = CheckState.Checked;
                    stripDDBtnQuantizeOff.CheckState = CheckState.Checked;
                    menuSettingPositionQuantizeOff.CheckState = CheckState.Checked;
                    break;
            }
            cMenuPianoQuantizeTriplet.Checked = AppManager.EditorConfig.PositionQuantizeTriplet;
            stripDDBtnQuantizeTriplet.Checked = AppManager.EditorConfig.PositionQuantizeTriplet;
            menuSettingPositionQuantizeTriplet.Checked = AppManager.EditorConfig.PositionQuantizeTriplet;

            cMenuPianoLength04.CheckState = CheckState.Unchecked;
            cMenuPianoLength08.CheckState = CheckState.Unchecked;
            cMenuPianoLength16.CheckState = CheckState.Unchecked;
            cMenuPianoLength32.CheckState = CheckState.Unchecked;
            cMenuPianoLength64.CheckState = CheckState.Unchecked;
            cMenuPianoLength128.CheckState = CheckState.Unchecked;
            cMenuPianoLengthOff.CheckState = CheckState.Unchecked;

            stripDDBtnLength04.CheckState = CheckState.Unchecked;
            stripDDBtnLength08.CheckState = CheckState.Unchecked;
            stripDDBtnLength16.CheckState = CheckState.Unchecked;
            stripDDBtnLength32.CheckState = CheckState.Unchecked;
            stripDDBtnLength64.CheckState = CheckState.Unchecked;
            stripDDBtnLength128.CheckState = CheckState.Unchecked;
            stripDDBtnLengthOff.CheckState = CheckState.Unchecked;

            menuSettingLengthQuantize04.CheckState = CheckState.Unchecked;
            menuSettingLengthQuantize08.CheckState = CheckState.Unchecked;
            menuSettingLengthQuantize16.CheckState = CheckState.Unchecked;
            menuSettingLengthQuantize32.CheckState = CheckState.Unchecked;
            menuSettingLengthQuantize64.CheckState = CheckState.Unchecked;
            menuSettingLengthQuantize128.CheckState = CheckState.Unchecked;
            menuSettingLengthQuantizeOff.CheckState = CheckState.Unchecked;

            stripDDBtnLength.Text = "LENGTH " + QuantizeModeUtil.GetString( AppManager.EditorConfig.LengthQuantize );
            switch ( AppManager.EditorConfig.LengthQuantize ) {
                case QuantizeMode.p4:
                    cMenuPianoLength04.CheckState = CheckState.Checked;
                    stripDDBtnLength04.CheckState = CheckState.Checked;
                    menuSettingLengthQuantize04.CheckState = CheckState.Checked;
                    break;
                case QuantizeMode.p8:
                    cMenuPianoLength08.CheckState = CheckState.Checked;
                    stripDDBtnLength08.CheckState = CheckState.Checked;
                    menuSettingLengthQuantize08.CheckState = CheckState.Checked;
                    break;
                case QuantizeMode.p16:
                    cMenuPianoLength16.CheckState = CheckState.Checked;
                    stripDDBtnLength16.CheckState = CheckState.Checked;
                    menuSettingLengthQuantize16.CheckState = CheckState.Checked;
                    break;
                case QuantizeMode.p32:
                    cMenuPianoLength32.CheckState = CheckState.Checked;
                    stripDDBtnLength32.CheckState = CheckState.Checked;
                    menuSettingLengthQuantize32.CheckState = CheckState.Checked;
                    break;
                case QuantizeMode.p64:
                    cMenuPianoLength64.CheckState = CheckState.Checked;
                    stripDDBtnLength64.CheckState = CheckState.Checked;
                    menuSettingLengthQuantize64.CheckState = CheckState.Checked;
                    break;
                case QuantizeMode.p128:
                    cMenuPianoLength128.CheckState = CheckState.Checked;
                    stripDDBtnLength128.CheckState = CheckState.Checked;
                    menuSettingLengthQuantize128.CheckState = CheckState.Checked;
                    break;
                case QuantizeMode.off:
                    cMenuPianoLengthOff.CheckState = CheckState.Checked;
                    stripDDBtnLengthOff.CheckState = CheckState.Checked;
                    menuSettingLengthQuantizeOff.CheckState = CheckState.Checked;
                    break;
            }
            cMenuPianoLengthTriplet.Checked = AppManager.EditorConfig.LengthQuantizeTriplet;
            stripDDBtnLengthTriplet.Checked = AppManager.EditorConfig.LengthQuantizeTriplet;
            menuSettingLengthQuantizeTriplet.Checked = AppManager.EditorConfig.LengthQuantizeTriplet;
        }

        /// <summary>
        /// 現在選択されている編集ツールに応じて、メニューのチェック状態を更新します
        /// </summary>
        private void ApplySelectedTool() {
            switch ( m_manager.SelectedTool ) {
                case EditTool.Arrow:
                    cMenuPianoPointer.CheckState = CheckState.Checked;
                    cMenuPianoPencil.CheckState = CheckState.Unchecked;
                    cMenuPianoEraser.CheckState = CheckState.Unchecked;

                    cMenuTrackSelectorPointer.CheckState = CheckState.Checked;
                    cMenuTrackSelectorPencil.CheckState = CheckState.Unchecked;
                    cMenuTrackSelectorLine.CheckState = CheckState.Unchecked;
                    cMenuTrackSelectorEraser.CheckState = CheckState.Unchecked;

                    stripBtnPointer.Checked = true;
                    stripBtnPencil.Checked = false;
                    stripBtnLine.Checked = false;
                    stripBtnEraser.Checked = false;
                    break;
                case EditTool.Pencil:
                    cMenuPianoPointer.CheckState = CheckState.Unchecked;
                    cMenuPianoPencil.CheckState = CheckState.Checked;
                    cMenuPianoEraser.CheckState = CheckState.Unchecked;

                    cMenuTrackSelectorPointer.CheckState = CheckState.Unchecked;
                    cMenuTrackSelectorPencil.CheckState = CheckState.Checked;
                    cMenuTrackSelectorLine.CheckState = CheckState.Unchecked;
                    cMenuTrackSelectorEraser.CheckState = CheckState.Unchecked;

                    stripBtnPointer.Checked = false;
                    stripBtnPencil.Checked = true;
                    stripBtnLine.Checked = false;
                    stripBtnEraser.Checked = false;
                    break;
                case EditTool.Eraser:
                    cMenuPianoPointer.CheckState = CheckState.Unchecked;
                    cMenuPianoPencil.CheckState = CheckState.Unchecked;
                    cMenuPianoEraser.CheckState = CheckState.Checked;

                    cMenuTrackSelectorPointer.CheckState = CheckState.Unchecked;
                    cMenuTrackSelectorPencil.CheckState = CheckState.Unchecked;
                    cMenuTrackSelectorLine.CheckState = CheckState.Unchecked;
                    cMenuTrackSelectorEraser.CheckState = CheckState.Checked;

                    stripBtnPointer.Checked = false;
                    stripBtnPencil.Checked = false;
                    stripBtnLine.Checked = false;
                    stripBtnEraser.Checked = true;
                    break;
                case EditTool.Line:
                    cMenuPianoPointer.CheckState = CheckState.Unchecked;
                    cMenuPianoPencil.CheckState = CheckState.Unchecked;
                    cMenuPianoEraser.CheckState = CheckState.Unchecked;

                    cMenuTrackSelectorPointer.CheckState = CheckState.Unchecked;
                    cMenuTrackSelectorPencil.CheckState = CheckState.Unchecked;
                    cMenuTrackSelectorLine.CheckState = CheckState.Checked;
                    cMenuTrackSelectorEraser.CheckState = CheckState.Unchecked;

                    stripBtnPointer.Checked = false;
                    stripBtnPencil.Checked = false;
                    stripBtnLine.Checked = true;
                    stripBtnEraser.Checked = false;
                    break;
                default:
                    cMenuPianoPointer.CheckState = CheckState.Unchecked;
                    cMenuPianoPencil.CheckState = CheckState.Unchecked;
                    cMenuPianoEraser.CheckState = CheckState.Unchecked;

                    cMenuTrackSelectorPointer.CheckState = CheckState.Unchecked;
                    cMenuTrackSelectorPencil.CheckState = CheckState.Unchecked;
                    cMenuTrackSelectorLine.CheckState = CheckState.Unchecked;
                    cMenuTrackSelectorEraser.CheckState = CheckState.Unchecked;

                    stripBtnPointer.Checked = false;
                    stripBtnPencil.Checked = false;
                    stripBtnLine.Checked = false;
                    stripBtnEraser.Checked = false;
                    break;
            }
            cMenuPianoCurve.Checked = m_manager.IsCurveMode;
            cMenuTrackSelectorCurve.Checked = m_manager.IsCurveMode;
            stripBtnCurve.Checked = m_manager.IsCurveMode;
        }

        /// <summary>
        /// 画面上のマウス位置におけるクロック値を元に，_toolbar_measureの場所表示文字列を更新します．
        /// </summary>
        /// <param name="mouse_pos_x"></param>
        private void UpdatePositionViewFromMousePosition( int clock ) {
            int barcount = m_manager.VsqFile.GetBarCountFromClock( clock );
            int numerator, denominator;
            m_manager.VsqFile.GetTimesigAt( clock, out numerator, out denominator );
            int clock_per_beat = 480 / 4 * denominator;
            int barcount_clock = m_manager.VsqFile.GetClockFromBarCount( barcount );
            int beat = (clock - barcount_clock) / clock_per_beat;
            int odd = clock - barcount_clock - beat * clock_per_beat;
#if OBSOLUTE
            m_toolbar_measure.Measure = (barcount - AppManager.VsqFile.PreMeasure + 1) + " : " + (beat + 1) + " : " + odd.ToString( "000" );
#else
            stripLblMeasure.Text = (barcount - m_manager.VsqFile.PreMeasure + 1) + " : " + (beat + 1) + " : " + odd.ToString( "000" );
#endif
        }

        /// <summary>
        /// 描画すべきオブジェクトのリスト，m_draw_objectsを更新します
        /// </summary>
        private void UpdateDrawObjectList() {
            if ( m_manager.VsqFile == null ) {
                return;
            }
            if ( m_draw_objects != null ) {
                foreach ( List<DrawObject> list in m_draw_objects ) {
                    list.Clear();
                }
                m_draw_objects.Clear();
            }
            m_draw_objects = new List<List<DrawObject>>();

            int xoffset = 6 + AppManager._KEY_LENGTH;
            int yoffset = 127 * AppManager.EditorConfig.PxTrackHeight;
            float scalex = m_manager.ScaleX;
            using ( Font SMALL_FONT = new Font( AppManager.EditorConfig.ScreenFontName, 8 ) ) {
                for ( int track = 1; track < m_manager.VsqFile.Tracks.Count; track++ ) {
                    List<DrawObject> tmp = new List<DrawObject>();
                    foreach ( VsqEvent ev in m_manager.VsqFile.Tracks[track].Events ) {
                        int timesig = ev.Clock;
                        if ( ev.ID.LyricHandle != null ) {
                            int length = ev.ID.Length;
                            int note = ev.ID.Note;
                            int x = (int)(timesig * scalex + xoffset);
                            int y = -note * AppManager.EditorConfig.PxTrackHeight + yoffset;
                            int lyric_width = (int)(length * scalex);
                            string lyric_jp = ev.ID.LyricHandle.L0.Phrase;
                            string lyric_en = ev.ID.LyricHandle.L0.PhoneticSymbol;
                            string title = Common.TrimString( lyric_jp + " [" + lyric_en + "]", SMALL_FONT, lyric_width );
                            int accent = ev.ID.DEMaccent;
                            int vibrato_start = x + lyric_width;
                            int vibrato_end = x;
                            int vibrato_delay = lyric_width * 2;
                            if ( ev.ID.VibratoHandle != null ) {
                                double rate = (double)ev.ID.VibratoDelay / (double)length;
                                vibrato_delay = _PX_ACCENT_HEADER + (int)((lyric_width - _PX_ACCENT_HEADER) * rate);
                            }
                            tmp.Add( new DrawObject( new Rectangle( x, y, lyric_width, AppManager.EditorConfig.PxTrackHeight ),
                                                     title,
                                                     accent,
                                                     ev.InternalID,
                                                     vibrato_delay,
                                                     false,
                                                     ev.ID.LyricHandle.L0.PhoneticSymbolProtected ) );
                        }
                    }

                    // 重複部分があるかどうかを判定
                    for ( int i = 0; i < tmp.Count - 1; i++ ) {
                        bool overwrapped = false;
                        for ( int j = i + 1; j < tmp.Count; j++ ) {
                            int startx = tmp[j].pxRectangle.X;
                            int endx = tmp[j].pxRectangle.X + tmp[j].pxRectangle.Width;
                            if ( startx < tmp[i].pxRectangle.X ) {
                                if ( tmp[i].pxRectangle.X < endx ) {
                                    overwrapped = true;
                                    tmp[j].Overwrapped = true;
                                    // breakできない．2個以上の重複を検出する必要があるので．
                                }
                            } else if ( tmp[i].pxRectangle.X <= startx && startx < tmp[i].pxRectangle.X + tmp[i].pxRectangle.Width ) {
                                overwrapped = true;
                                tmp[j].Overwrapped = true;
                            }
                        }
                        if ( overwrapped ) {
                            tmp[i].Overwrapped = true;
                        }
                    }
                    m_draw_objects.Add( tmp );
                }
            }
        }

        private void DrawTo( Graphics g, Size window_size, Point mouse_position ) {
            int start_draw_x = m_manager.StartToDrawX;
            int start_draw_y = StartToDrawY;
            // [screen_x] = 67 + [clock] * ScaleX - StartToDrawX + 6
            // [screen_y] = -1 * ([note] - 127) * TRACK_HEIGHT - StartToDrawY
            //
            // [screen_x] = [clock] * _scalex + 73 - StartToDrawX
            // [screen_y] = -[note] * TRACK_HEIGHT + 127 * TRACK_HEIGHT - StartToDrawY
            int xoffset = 6 + AppManager._KEY_LENGTH - start_draw_x;
            int yoffset = 127 * AppManager.EditorConfig.PxTrackHeight - start_draw_y;
            //      ↓
            // [screen_x] = [clock] * _scalex + xoffset
            // [screen_y] = -[note] * TRACK_HEIGHT + yoffset
            int y, dy;
            float scalex = m_manager.ScaleX;
            float inv_scalex = 1f / scalex;

            if ( m_manager.SelectedEvent.Count > 0 && m_input_textbox.Enabled ) {
                VsqEvent original = m_manager.SelectedEvent.LastSelected.Original;
                int event_x = (int)(original.Clock * scalex + xoffset);
                int event_y = -original.ID.Note * AppManager.EditorConfig.PxTrackHeight + yoffset;
                m_input_textbox.Left = event_x + 4;
                m_input_textbox.Top = event_y + 2;
            }

            int width = window_size.Width;
            int height = window_size.Height;

            #region ピアノロール周りのスクロールバーなど
            // スクロール画面背景
            int calculated_height = height;
            if ( calculated_height > 0 ) {
                g.FillRectangle( s_brs_240_240_240,
                                 new Rectangle( 3, 0, width, calculated_height ) );
            }
            // ピアノロールとカーブエディタの境界
            g.DrawLine( s_pen_112_112_112,
                        new Point( 2, height - 1 ),
                        new Point( width - 1, height - 1 ) );
            // 横スクロールバー
            //hScroll.BarLength = (int)((width - 152) * inv_scalex);
            // 縦スクロールバー
            calculated_height = height - 14;//height - 93 - 14;
            if ( calculated_height > 0 ) {
                //vScroll.BarLength = (int)((float)vScroll.Maximum * vScroll.Height / (AppManager.EditorConfig.PxTrackHeight * 128f));
            }
            #endregion

            #region ピアノロール本体
            if ( m_manager.VsqFile != null ) {
                int odd = -1;
                y = 128 * AppManager.EditorConfig.PxTrackHeight - start_draw_y;
                dy = -AppManager.EditorConfig.PxTrackHeight;
                for ( int i = 0; i <= 127; i++ ) {
                    odd++;
                    if ( odd == 12 ) {
                        odd = 0;
                    }
                    int order = (i - odd) / 12 - 2;
                    y += dy;
                    if ( y > height ) {
                        continue;
                    } else if ( 0 > y + AppManager.EditorConfig.PxTrackHeight ) {
                        break;
                    }
                    bool note_is_whitekey = VsqNote.NoteIsWhiteKey( i );

                    #region ピアノロール背景
                    SolidBrush b = new SolidBrush( Color.Black );
                    SolidBrush border;
                    bool paint_required = true;
                    if ( order == -2 || order == -1 || (6 <= order && order <= 8) ) {
                        if ( note_is_whitekey ) {
                            b = s_brs_180_180_180;
                        } else {
                            b = s_brs_106_108_108;
                        }
                        border = s_brs_106_108_108;
                    } else if ( order == 5 || order == 0 ) {
                        if ( note_is_whitekey ) {
                            b = s_brs_212_212_212;
                        } else {
                            b = s_brs_180_180_180;
                        }
                        border = new SolidBrush( Color.FromArgb( 150, 152, 150 ) );
                    } else {
                        if ( note_is_whitekey ) {
                            paint_required = false;
                        } else {
                            b = s_brs_212_212_212;
                        }
                        border = new SolidBrush( Color.FromArgb( 210, 205, 172 ) );
                    }
                    if ( paint_required ) {
                        g.FillRectangle( b,
                                         new Rectangle( AppManager._KEY_LENGTH, y, width - AppManager._KEY_LENGTH, AppManager.EditorConfig.PxTrackHeight + 1 ) );
                    }
                    if ( odd == 0 || odd == 5 ) {
                        g.DrawLine( new Pen( border ),
                                    new Point( AppManager._KEY_LENGTH, y + AppManager.EditorConfig.PxTrackHeight ),
                                    new Point( width, y + AppManager.EditorConfig.PxTrackHeight ) );
                    }
                    #endregion

                    #region プリメジャー部分のピアノロール背景
                    int premeasure_start_x = xoffset;
                    if ( premeasure_start_x < AppManager._KEY_LENGTH ) {
                        premeasure_start_x = AppManager._KEY_LENGTH;
                    }
                    int premeasure_end_x = (int)(m_manager.VsqFile.PreMeasureClocks * scalex + xoffset);
                    if ( premeasure_start_x >= AppManager._KEY_LENGTH ) {
                        if ( note_is_whitekey ) {
                            g.FillRectangle( s_brs_153_153_153,
                                             new Rectangle( premeasure_start_x,
                                                            y,
                                                            premeasure_end_x - premeasure_start_x,
                                                            AppManager.EditorConfig.PxTrackHeight + 1 ) );
                        } else {
                            g.FillRectangle( s_brs_106_108_108,
                                             new Rectangle( premeasure_start_x,
                                                            y,
                                                            premeasure_end_x - premeasure_start_x,
                                                            AppManager.EditorConfig.PxTrackHeight + 1 ) );
                        }
                        if ( odd == 0 || odd == 5 ) {
                            g.DrawLine( s_pen_106_108_108,
                                        new Point( premeasure_start_x, y + AppManager.EditorConfig.PxTrackHeight ),
                                        new Point( premeasure_end_x, y + AppManager.EditorConfig.PxTrackHeight ) );
                        }
                    }
                    #endregion

                }
            }

            //ピアノロールと鍵盤部分の縦線
            g.DrawLine( s_pen_212_212_212,
                        new Point( AppManager._KEY_LENGTH, 0 ),
                        new Point( AppManager._KEY_LENGTH, height - 15 ) );
            int odd2 = -1;
            y = 128 * AppManager.EditorConfig.PxTrackHeight - start_draw_y;
            dy = -AppManager.EditorConfig.PxTrackHeight;
            for ( int i = 0; i <= 127; i++ ) {
                odd2++;
                if ( odd2 == 12 ) {
                    odd2 = 0;
                }
                y += dy;
                if ( y > height ) {
                    continue;
                } else if ( y + AppManager.EditorConfig.PxTrackHeight < 0 ) {
                    break;
                }

                #region 鍵盤部分
                g.DrawLine(
                    s_pen_212_212_212,
                    new Point( 3, y ),
                    new Point( AppManager._KEY_LENGTH, y ) );
                bool hilighted = false;
                if ( m_manager.EditMode == EditMode.AddEntry ) {
                    if ( m_adding.ID.Note == i ) {
                        hilighted = true;
                    }
                } else if ( m_manager.EditMode == EditMode.EditLeftEdge || m_manager.EditMode == EditMode.EditRightEdge ) {
                    if ( m_manager.SelectedEvent.LastSelected.Original.ID.Note == i ) {
                        hilighted = true;
                    }
                } else {
                    if ( 3 <= mouse_position.X && mouse_position.X <= width - 17 &&
                        0 <= mouse_position.Y && mouse_position.Y <= height - 1 ) {
                        if ( y <= mouse_position.Y && mouse_position.Y < y + AppManager.EditorConfig.PxTrackHeight ) {
                            hilighted = true;
                        }
                    }
                }
                if ( hilighted ) {
                    g.FillRectangle( AppManager.HilightBrush,
                                     new Rectangle( 35, y, AppManager._KEY_LENGTH - 35, AppManager.EditorConfig.PxTrackHeight ) );
                }
                if ( odd2 == 0 || hilighted ) {
                    g.DrawString( VsqNote.NoteToString( i ),
                                  s_font_small,
                                  s_brs_072_077_098,
                                  new PointF( 42, y + 2 ) );
                }
                if ( !VsqNote.NoteIsWhiteKey( i ) ) {
                    g.FillRectangle( s_brs_125_123_124, new Rectangle( 0, y, 34, AppManager.EditorConfig.PxTrackHeight ) );
                }
                #endregion
            }
            g.ResetClip();

            g.SetClip( new Rectangle( AppManager._KEY_LENGTH, 0, width - AppManager._KEY_LENGTH, height ) );
            #region 小節ごとの線
            IEnumerable<VsqBarLineType> enumerable = m_manager.VsqFile.GetEnumerator( ClockFromXCoord( width ) );
            int dashed_line_step = AppManager.GetPositionQuantizeClock();
            foreach ( VsqBarLineType blt in enumerable ) {
                int local_clock_step = 1920 / blt.LocalDenominator;
                int x = (int)(blt.Clock * scalex + xoffset);
                if ( blt.IsSeparator ) {
                    //ピアノロール上
                    g.DrawLine( s_pen_161_157_136,
                                new Point( x, 0 ),
                                new Point( x, height ) );
                } else {
                    //ピアノロール上
                    g.DrawLine( s_pen_209_204_172,
                                new Point( x, 0 ),
                                new Point( x, height ) );
                }
                if ( dashed_line_step > 1 && m_manager.GridVisible ) {
                    int numDashedLine = local_clock_step / dashed_line_step;
                    for ( int i = 1; i < numDashedLine; i++ ) {
                        int x2 = (int)((blt.Clock + i * dashed_line_step) * scalex + xoffset);
                        g.DrawLine( s_pen_dashed_209_204_172,
                                    new Point( x2, 0 ),
                                    new Point( x2, height ) );
                    }
                }
            }
            #endregion

            #region トラックのエントリを描画
            if ( m_draw_objects != null ) {
                if ( m_manager.Overlay ) {
                    // まず、選択されていないトラックの簡易表示を行う
                    for ( int i = 0; i < m_draw_objects.Count; i++ ) {
                        if ( i == m_manager.Selected - 1 ) {
                            continue;
                        }
                        int j_start = m_draw_start_index[i];
                        bool first = true;
                        for ( int j = j_start; j < m_draw_objects[i].Count; j++ ) {
                            DrawObject dobj = m_draw_objects[i][j];
                            int x = dobj.pxRectangle.X - start_draw_x;
                            y = dobj.pxRectangle.Y - start_draw_y;
                            int lyric_width = dobj.pxRectangle.Width;
                            if ( x + lyric_width < 0 ) {
                                continue;
                            } else if ( width < x ) {
                                break;
                            }
                            if ( m_manager.Playing && first ) {
                                m_draw_start_index[i] = j;
                                first = false;
                            }
                            if ( y + AppManager.EditorConfig.PxTrackHeight < 0 || y > height ) {
                                continue;
                            }
                            g.DrawLine( new Pen( AppManager.s_HILIGHT[i] ),
                                        new Point( x + 1, y + 7 ),
                                        new Point( x + lyric_width - 1, y + 7 ) );
                            g.DrawPolygon( new Pen( s_HIDDEN[i] ),
                                           new Point[] { 
                                               new Point( x, y + 7 ), 
                                               new Point( x + 1, y + 6 ),
                                               new Point( x + lyric_width - 1, y + 6 ),
                                               new Point( x + lyric_width, y + 7 ),
                                               new Point( x + lyric_width - 1, y + 8 ),
                                               new Point( x + 1, y + 8 ), 
                                               new Point( x, y + 7 ) 
                                           } );
                        }
                    }
                }

                // 選択されているトラックの表示を行う
                int selected = m_manager.Selected;
                bool show_lyrics = AppManager.EditorConfig.ShowLyric;
                bool show_exp_line = AppManager.EditorConfig.ShowExpLine;
                if ( selected >= 1 ) {
                    int j_start = m_draw_start_index[selected - 1];
                    bool first = true;
                    for ( int j = j_start; j < m_draw_objects[selected - 1].Count; j++ ) {
                        DrawObject dobj = m_draw_objects[selected - 1][j];
                        int x = dobj.pxRectangle.X - start_draw_x;
                        y = dobj.pxRectangle.Y - start_draw_y;
                        int lyric_width = dobj.pxRectangle.Width;
                        if ( x + lyric_width < 0 ) {
                            continue;
                        } else if ( width < x ) {
                            break;
                        }
                        if ( m_manager.Playing && first ) {
                            m_draw_start_index[selected - 1] = j;
                            first = false;
                        }
                        if ( y + 2 * AppManager.EditorConfig.PxTrackHeight < 0 || y > height ) {
                            continue;
                        }
                        Color id_fill;
                        if ( m_manager.SelectedEvent.Count > 0 ) {
                            bool found = m_manager.SelectedEvent.ContainsKey( dobj.InternalID );
                            if ( found ) {
                                id_fill = AppManager.HilightColor;
                            } else {
                                id_fill = Color.FromArgb( 181, 220, 86 );
                            }
                        } else {
                            id_fill = Color.FromArgb( 181, 220, 86 );
                        }
                        g.FillRectangle(
                            new SolidBrush( id_fill ),
                            new Rectangle( x, y + 1, lyric_width, AppManager.EditorConfig.PxTrackHeight - 1 ) );
                        Font lyric_font = dobj.SymbolProtected ? s_font_bold : s_font_entry;
                        if ( dobj.Overwrapped ) {
                            g.DrawRectangle( s_pen_125_123_124,
                                             new Rectangle( x, y + 1, lyric_width, AppManager.EditorConfig.PxTrackHeight - 1 ) );
                            if ( show_lyrics ) {
                                g.DrawString( dobj.Text,
                                              lyric_font,
                                              s_brs_147_147_147,
                                              new PointF( x + 1, y + 1 ) );
                            }
                        } else {
                            g.DrawRectangle( s_pen_125_123_124,
                                             new Rectangle( x, y + 1, lyric_width, AppManager.EditorConfig.PxTrackHeight - 1 ) );
                            if ( show_lyrics ) {
                                g.DrawString( dobj.Text,
                                              lyric_font,
                                              Brushes.Black,
                                              new PointF( x + 1, y + 1 ) );
                            }
                            if ( show_exp_line && lyric_width > 21 ) {
                                #region 表情線
                                DrawAccentLine( g, new Point( x, y + AppManager.EditorConfig.PxTrackHeight + 1 ), dobj.Accent );
                                int vibrato_start = x + lyric_width;
                                int vibrato_end = x;
                                if ( dobj.pxVibratoDelay <= lyric_width ) {
                                    int vibrato_delay = dobj.pxVibratoDelay;
                                    int vibrato_width = dobj.pxRectangle.Width - vibrato_delay;
                                    vibrato_start = x + vibrato_delay;
                                    vibrato_end = x + vibrato_delay + vibrato_width;
                                    if ( vibrato_start - x < 21 ) {
                                        vibrato_start = x + 21;
                                    }
                                }
                                g.DrawLine( s_pen_051_051_000,
                                            new Point( x + 21, y + AppManager.EditorConfig.PxTrackHeight + 7 ),
                                            new Point( vibrato_start, y + AppManager.EditorConfig.PxTrackHeight + 7 ) );
                                if ( dobj.pxVibratoDelay <= lyric_width ) {
                                    int next_draw = vibrato_start;
                                    if ( vibrato_start < vibrato_end ) {
                                        DrawVibratoLine( g, 
                                                         new Point( vibrato_start, y + AppManager.EditorConfig.PxTrackHeight + 1 ), 
                                                         vibrato_end - vibrato_start );
                                    }
                                #endregion
                                }
                            }
                        }

                    }
                }

                // 編集中のエントリを表示
                if ( m_manager.EditMode == EditMode.AddEntry ||
                     m_manager.EditMode == EditMode.AddFixedLengthEntry ||
                     m_manager.EditMode == EditMode.Realtime ) {
                    if ( m_adding != null ) {
                        int x = (int)(m_adding.Clock * scalex + xoffset);
                        y = -m_adding.ID.Note * AppManager.EditorConfig.PxTrackHeight + yoffset + 1;
                        if ( m_adding.ID.Length <= 0 ) {
                            g.DrawRectangle( s_pen_dashed_171_171_171,
                                             new Rectangle( x, y, 10, AppManager.EditorConfig.PxTrackHeight - 1 ) );
                        } else {
                            int length = (int)(m_adding.ID.Length * scalex);
                            g.DrawRectangle( s_pen_a136_000_000_000,
                                             new Rectangle( x, y, length, AppManager.EditorConfig.PxTrackHeight - 1 ) );
                        }
                    }
                } else if ( m_manager.EditMode == EditMode.EditVibratoDelay ) {
                    int x = (int)(m_adding.Clock * scalex + xoffset);
                    y = -m_adding.ID.Note * AppManager.EditorConfig.PxTrackHeight + yoffset + 1;
                    int length = (int)(m_adding.ID.Length * scalex);
                    g.DrawRectangle( s_pen_a136_000_000_000,
                                     new Rectangle( x, y, length, AppManager.EditorConfig.PxTrackHeight - 1 ) );
                } else if ( (m_manager.EditMode == EditMode.MoveEntry ||
                       m_manager.EditMode == EditMode.EditLeftEdge ||
                       m_manager.EditMode == EditMode.EditRightEdge) && m_manager.SelectedEvent.Count > 0 ) {
                    foreach ( SelectedEventEntry ev in m_manager.SelectedEvent.GetEnumerator() ) {
                        int x = (int)(ev.Editing.Clock * scalex + xoffset);
                        y = -ev.Editing.ID.Note * AppManager.EditorConfig.PxTrackHeight + yoffset + 1;
                        if ( ev.Editing.ID.Length == 0 ) {
                            g.DrawRectangle( s_pen_dashed_171_171_171,
                                             new Rectangle( x, y, 10, AppManager.EditorConfig.PxTrackHeight - 1 ) );
                        } else {
                            int length = (int)(ev.Editing.ID.Length * scalex);
                            g.DrawRectangle( s_pen_a136_000_000_000,
                                             new Rectangle( x, y, length, AppManager.EditorConfig.PxTrackHeight - 1 ) );
                        }
                    }
                }
            }
            #endregion

            g.ResetClip();

            #endregion

            #region 現在のマーカー
            int marker_x = (int)(m_manager.CurrentClock * scalex + xoffset);
            if ( AppManager._KEY_LENGTH <= marker_x && marker_x <= width ) {
                g.DrawLine( new Pen( Color.White, 2f ),
                            new Point( marker_x, 0 ),
                            new Point( marker_x, height ) );
            }
            #endregion

            #region 音符編集時の補助線
            if ( m_manager.EditMode == EditMode.AddEntry ) {
                #region EditMode.AddEntry
                int x = (int)(m_adding.Clock * scalex + xoffset);
                y = -m_adding.ID.Note * AppManager.EditorConfig.PxTrackHeight + yoffset + 1;
                int length;
                if ( m_adding.ID.Length == 0 ) {
                    length = 10;
                } else {
                    length = (int)(m_adding.ID.Length * scalex);
                }
                x += length;
                g.DrawLine( s_pen_LU,
                            new Point( x, 0 ),
                            new Point( x, y - 1 ) );
                g.DrawLine( s_pen_LU,
                            new Point( x, y + AppManager.EditorConfig.PxTrackHeight ),
                            new Point( x, height ) );
                g.DrawLine( s_pen_RD,
                            new Point( x + 1, 0 ),
                            new Point( x + 1, y - 1 ) );
                g.DrawLine( s_pen_RD,
                            new Point( x + 1, y + AppManager.EditorConfig.PxTrackHeight ),
                            new Point( x + 1, height ) );
                #endregion
            } else if ( m_manager.EditMode == EditMode.MoveEntry ) {
                #region EditMode.MoveEntry
                if ( m_manager.SelectedEvent.Count > 0 ) {
                    VsqEvent last = m_manager.SelectedEvent.LastSelected.Editing;
                    int x = (int)(last.Clock * scalex + xoffset);
                    y = -last.ID.Note * AppManager.EditorConfig.PxTrackHeight + yoffset + 1;
                    int length = (int)(last.ID.Length * scalex);
                    // 縦線
                    g.DrawLine( s_pen_LU,
                                new Point( x, 0 ),
                                new Point( x, y - 1 ) );
                    g.DrawLine( s_pen_LU,
                                new Point( x, y + AppManager.EditorConfig.PxTrackHeight ),
                                new Point( x, height ) );
                    // 横線
                    g.DrawLine( s_pen_LU,
                                new Point( AppManager._KEY_LENGTH, y + AppManager.EditorConfig.PxTrackHeight / 2 - 2 ),
                                new Point( x - 1, y + AppManager.EditorConfig.PxTrackHeight / 2 - 2 ) );
                    g.DrawLine( s_pen_LU,
                                new Point( x + length + 1, y + AppManager.EditorConfig.PxTrackHeight / 2 - 2 ),
                                new Point( width, y + AppManager.EditorConfig.PxTrackHeight / 2 - 2 ) );
                    // 縦線
                    g.DrawLine( s_pen_RD,
                                new Point( x + 1, 0 ),
                                new Point( x + 1, y - 1 ) );
                    g.DrawLine( s_pen_RD,
                                new Point( x + 1, y + AppManager.EditorConfig.PxTrackHeight ),
                                new Point( x + 1, height ) );
                    // 横線
                    g.DrawLine( s_pen_RD,
                                new Point( AppManager._KEY_LENGTH, y + AppManager.EditorConfig.PxTrackHeight / 2 - 1 ),
                                new Point( x - 1, y + AppManager.EditorConfig.PxTrackHeight / 2 - 1 ) );
                    g.DrawLine( s_pen_RD,
                                new Point( x + length + 1, y + AppManager.EditorConfig.PxTrackHeight / 2 - 1 ),
                                new Point( width, y + AppManager.EditorConfig.PxTrackHeight / 2 - 1 ) );
                }
                #endregion
            } else if ( m_manager.EditMode == EditMode.AddFixedLengthEntry ) {
                #region EditMode.MoveEntry
                int x = (int)(m_adding.Clock * scalex + xoffset);
                y = -m_adding.ID.Note * AppManager.EditorConfig.PxTrackHeight + yoffset + 1;
                int length = (int)(m_adding.ID.Length * scalex);
                // 縦線
                g.DrawLine( s_pen_LU,
                            new Point( x, 0 ),
                            new Point( x, y - 1 ) );
                g.DrawLine( s_pen_LU,
                            new Point( x, y + AppManager.EditorConfig.PxTrackHeight ),
                            new Point( x, height ) );
                // 横線
                g.DrawLine( s_pen_LU,
                            new Point( AppManager._KEY_LENGTH, y + AppManager.EditorConfig.PxTrackHeight / 2 - 2 ),
                            new Point( x - 1, y + AppManager.EditorConfig.PxTrackHeight / 2 - 2 ) );
                g.DrawLine( s_pen_LU,
                            new Point( x + length + 1, y + AppManager.EditorConfig.PxTrackHeight / 2 - 2 ),
                            new Point( width, y + AppManager.EditorConfig.PxTrackHeight / 2 - 2 ) );
                // 縦線
                g.DrawLine( s_pen_RD,
                            new Point( x + 1, 0 ),
                            new Point( x + 1, y - 1 ) );
                g.DrawLine( s_pen_RD,
                            new Point( x + 1, y + AppManager.EditorConfig.PxTrackHeight ),
                            new Point( x + 1, height ) );
                // 横線
                g.DrawLine( s_pen_RD,
                            new Point( AppManager._KEY_LENGTH, y + AppManager.EditorConfig.PxTrackHeight / 2 - 1 ),
                            new Point( x - 1, y + AppManager.EditorConfig.PxTrackHeight / 2 - 1 ) );
                g.DrawLine( s_pen_RD,
                            new Point( x + length + 1, y + AppManager.EditorConfig.PxTrackHeight / 2 - 1 ),
                            new Point( width, y + AppManager.EditorConfig.PxTrackHeight / 2 - 1 ) );
                #endregion
            } else if ( m_manager.EditMode == EditMode.EditLeftEdge ) {
                #region EditMode.EditLeftEdge
                VsqEvent last = m_manager.SelectedEvent.LastSelected.Editing;
                int x = (int)(last.Clock * scalex + xoffset);
                y = -last.ID.Note * AppManager.EditorConfig.PxTrackHeight + yoffset + 1;
                g.DrawLine( s_pen_LU,
                            new Point( x, 0 ),
                            new Point( x, y - 1 ) );
                g.DrawLine( s_pen_LU,
                            new Point( x, y + AppManager.EditorConfig.PxTrackHeight ),
                            new Point( x, height ) );
                g.DrawLine( s_pen_RD,
                            new Point( x + 1, 0 ),
                            new Point( x + 1, y - 1 ) );
                g.DrawLine( s_pen_RD,
                            new Point( x + 1, y + AppManager.EditorConfig.PxTrackHeight ),
                            new Point( x + 1, height ) );
                #endregion
            } else if ( m_manager.EditMode == EditMode.EditRightEdge ) {
                #region EditMode.EditRightEdge
                VsqEvent last = m_manager.SelectedEvent.LastSelected.Editing;
                int x = (int)(last.Clock * scalex + xoffset);
                y = -last.ID.Note * AppManager.EditorConfig.PxTrackHeight + yoffset + 1;
                int length = (int)(last.ID.Length * scalex);
                x += length;
                g.DrawLine( s_pen_LU,
                            new Point( x, 0 ),
                            new Point( x, y - 1 ) );
                g.DrawLine( s_pen_LU,
                            new Point( x, y + AppManager.EditorConfig.PxTrackHeight ),
                            new Point( x, height ) );
                g.DrawLine( s_pen_RD,
                            new Point( x + 1, 0 ),
                            new Point( x + 1, y - 1 ) );
                g.DrawLine( s_pen_RD,
                            new Point( x + 1, y + AppManager.EditorConfig.PxTrackHeight ),
                            new Point( x + 1, height ) );
                #endregion
            } else if ( m_manager.EditMode == EditMode.EditVibratoDelay ) {
                #region EditVibratoDelay
                int x = (int)(m_adding.Clock * scalex + xoffset);
                y = -m_adding.ID.Note * AppManager.EditorConfig.PxTrackHeight + yoffset + 1;
                g.DrawLine( s_pen_LU,
                            new Point( x, 0 ),
                            new Point( x, y - 1 ) );
                g.DrawLine( s_pen_LU,
                            new Point( x, y + AppManager.EditorConfig.PxTrackHeight ),
                            new Point( x, height ) );
                g.DrawLine( s_pen_RD,
                            new Point( x + 1, 0 ),
                            new Point( x + 1, y - 1 ) );
                g.DrawLine( s_pen_RD,
                            new Point( x + 1, y + AppManager.EditorConfig.PxTrackHeight ),
                            new Point( x + 1, height ) );
                double max_length = m_adding.ID.value - _PX_ACCENT_HEADER / scalex;
                double drate = m_adding.ID.Length / max_length;
                if ( drate > 0.99 ) {
                    drate = 1.00;
                }
                int rate = (int)(drate * 100.0);
                string percent = rate + "%";
                SizeF size = Common.MeasureString( percent, s_F9PT );
                int delay_x = (int)((m_adding.Clock + m_adding.ID.Length - m_adding.ID.value + m_adding.ID.VibratoDelay) * scalex + xoffset);
                Rectangle pxArea = new Rectangle( delay_x,
                                                  (int)(y + AppManager.EditorConfig.PxTrackHeight * 2.5),
                                                  (int)(size.Width * 1.2),
                                                  (int)(size.Height * 1.2) );
                g.FillRectangle( s_brs_192_192_192, pxArea );
                g.DrawRectangle( Pens.Black, pxArea );
                StringFormat sf = new StringFormat();
                sf.Alignment = StringAlignment.Center;
                sf.LineAlignment = StringAlignment.Center;
                g.DrawString( percent, s_F9PT, Brushes.Black, pxArea, sf );
                #endregion
            }
            #endregion

            #region 外枠
            // 左(外側)
            g.DrawLine( s_pen_160_160_160,
                        new Point( 0, 0 ),
                        new Point( 0, height ) );
            // 左(内側)
            g.DrawLine( s_pen_105_105_105,
                        new Point( 1, 0 ),
                        new Point( 1, height ) );
            #endregion
        }

        /// <summary>
        /// アクセントを表す表情線を、指定された位置を基準点として描き込みます
        /// </summary>
        /// <param name="g"></param>
        /// <param name="accent"></param>
        private void DrawAccentLine( Graphics g, Point origin, int accent ) {
            int x0 = origin.X + 1;
            int y0 = origin.Y + 10;
            int height = 4 + accent * 4 / 100;
            SmoothingMode sm = g.SmoothingMode;
            g.SmoothingMode = SmoothingMode.AntiAlias;
            Point[] ps = new Point[]{ new Point( x0, y0 ),
                                      new Point( x0 + 2, y0 ),
                                      new Point( x0 + 8, y0 - height ),
                                      new Point( x0 + 13, y0 ),
                                      new Point( x0 + 16, y0 ),
                                      new Point( x0 + 20, y0 - 4 ) };
            g.DrawLines( Pens.Black, ps );
            g.SmoothingMode = sm;
        }

        private void DrawVibratoLine( Graphics g, Point origin, int vibrato_length ) {
            int x0 = origin.X + 1;
            int y0 = origin.Y + 10;
            int clipx = origin.X + 1;
            int clip_length = vibrato_length;
            if ( clipx < AppManager._KEY_LENGTH ) {
                clipx = AppManager._KEY_LENGTH;
                clip_length = origin.X + 1 + vibrato_length - AppManager._KEY_LENGTH;
                if ( clip_length <= 0 ) {
                    return;
                }
            }
            SmoothingMode sm = g.SmoothingMode;
            g.SmoothingMode = SmoothingMode.AntiAlias;
            const int _UWID = 10;
            int count = vibrato_length / _UWID + 1;
            Point[] _BASE = new Point[]{ new Point( x0 - _UWID, y0 - 4 ),
                                         new Point( x0 + 2 - _UWID, y0 - 7 ),
                                         new Point( x0 + 4 - _UWID, y0 - 7 ),
                                         new Point( x0 + 7 - _UWID, y0 - 1 ),
                                         new Point( x0 + 9 - _UWID, y0 - 1 ),
                                         new Point( x0 + 10 - _UWID, y0 - 4 ) };
            g.Clip = new Region( new Rectangle( clipx, origin.Y + 10 - 8, clip_length, 10 ) );
            for ( int i = 0; i < count; i++ ) {
                for ( int j = 0; j < _BASE.Length; j++ ) {
                    _BASE[j].X += _UWID;
                }
                g.DrawLines( Pens.Black, _BASE );
            }
            g.SmoothingMode = sm;
            g.ResetClip();
        }

        /// <summary>
        /// クロック数から、画面に描くべきx座標の値を取得します。
        /// </summary>
        /// <param name="clocks"></param>
        /// <returns></returns>
        int XCoordFromClocks( int clocks ) {
            return (int)(AppManager._KEY_LENGTH + clocks * m_manager.ScaleX - m_manager.StartToDrawX) + 6;
        }

        /// <summary>
        /// 画面のx座標からクロック数を取得します
        /// </summary>
        /// <param name="x"></param>
        /// <returns></returns>
        int ClockFromXCoord( int x ) {
            return (int)((x + m_manager.StartToDrawX - 6 - AppManager._KEY_LENGTH) / m_manager.ScaleX);
        }
        
        /// <summary>
        /// _editor_configのRecentFilesを元に，menuFileRecentのドロップダウンアイテムを更新します
        /// </summary>
        private void UpdateRecentFileMenu() {
            int added = 0;
            menuFileRecent.DropDownItems.Clear();
            if ( AppManager.EditorConfig.RecentFiles != null ) {
                for ( int i = 0; i < AppManager.EditorConfig.RecentFiles.Count; i++ ) {
                    string item = AppManager.EditorConfig.RecentFiles[i];
                    if ( item == null ) {
                        continue;
                    }
                    if ( item != "" ) {
                        string short_name = Path.GetFileName( item );
                        bool available = File.Exists( item );
                        ToolStripItem itm = (ToolStripItem)(new ToolStripMenuItem( short_name ));
                        if ( !available ) {
                            itm.ToolTipText = _( "[file not found]" ) + " ";
                        }
                        itm.ToolTipText += item;
                        itm.Tag = item;
                        itm.Enabled = available;
                        itm.Click += new EventHandler( itm_Click );
                        menuFileRecent.DropDownItems.Add( itm );
                        added++;
                    }
                }
            } else {
                AppManager.EditorConfig.PushRecentFiles( "" );
            }
            if ( added == 0 ) {
                menuFileRecent.Enabled = false;
            } else {
                menuFileRecent.Enabled = true;
            }
        }

        /// <summary>
        /// 最後に保存したときから変更されているかどうかを取得または設定します
        /// </summary>
        private bool Edited {
            get {
                return m_edited;
            }
            set {
                m_edited = value;
                string file = m_manager.FileName;
                if ( file == "" ) {
                    file = "Untitled";
                } else {
                    file = Path.GetFileNameWithoutExtension( file );
                }
                string title = _APP_NAME + " - " + file;
                if ( m_edited ) {
                    title += " *";
                }
                if ( this.Text != title ) {
                    this.Text = title;
                }
                bool redo = m_manager.IsRedoAvailable;
                bool undo = m_manager.IsUndoAvailable;
                menuEditRedo.Enabled = redo;
                menuEditUndo.Enabled = undo;
                cMenuPianoRedo.Enabled = redo;
                cMenuPianoUndo.Enabled = undo;
                cMenuTrackSelectorRedo.Enabled = redo;
                cMenuTrackSelectorUndo.Enabled = undo;
                m_manager.setRenderRequired( m_manager.Selected, true );
                if ( m_manager.VsqFile != null ) {
                    int draft = m_manager.VsqFile.TotalClocks;
                    if ( draft > hScroll.Maximum ) {
                        SetHScrollRange( draft );
                    }
                }
                UpdateDrawObjectList();
            }
        }

        /// <summary>
        /// 入力用のテキストボックスを初期化します
        /// </summary>
        private void ShowInputTextBox( string phrase, string phonetic_symbol, Point position, bool phonetic_symbol_edit_mode ) {
#if DEBUG
            Common.DebugWriteLine( "InitializeInputTextBox" );
#endif
            HideInputTextBox();
            m_input_textbox.KeyUp += m_input_textbox_KeyUp;
            m_input_textbox.KeyDown += m_input_textbox_KeyDown;
            m_input_textbox.ImeModeChanged += m_input_textbox_ImeModeChanged;
            m_input_textbox.ImeMode = m_last_imemode;
            if ( phonetic_symbol_edit_mode ) {
                m_input_textbox.Tag = new TagLyricTextBox( phrase, true );
                m_input_textbox.Text = phonetic_symbol;
                m_input_textbox.BackColor = s_txtbox_backcolor;
            } else {
                m_input_textbox.Tag = new TagLyricTextBox( phonetic_symbol, false );
                m_input_textbox.Text = phrase;
                m_input_textbox.BackColor = Color.White;
            }
            m_input_textbox.Font = new Font( AppManager.EditorConfig.BaseFontName, 9 );
            m_input_textbox.Location = new Point( position.X + 4, position.Y + 2 );
            m_input_textbox.Parent = pictPianoRoll;
            m_input_textbox.Enabled = true;
            m_input_textbox.Visible = true;
            m_input_textbox.Focus();
            m_input_textbox.SelectAll();

        }

        private void HideInputTextBox() {
            m_input_textbox.KeyUp -= m_input_textbox_KeyUp;
            m_input_textbox.KeyDown -= m_input_textbox_KeyDown;
            m_input_textbox.ImeModeChanged -= m_input_textbox_ImeModeChanged;
            if ( m_input_textbox.Tag != null && m_input_textbox.Tag is TagLyricTextBox ) {
                TagLyricTextBox tltb = (TagLyricTextBox)m_input_textbox.Tag;
                m_last_symbol_edit_mode = tltb.PhoneticSymbolEditMode;
            }
            m_input_textbox.Visible = false;
            m_input_textbox.Parent = null;
            m_input_textbox.Enabled = false;
            pictPianoRoll.Focus();
        }

        /// <summary>
        /// 歌詞入力用テキストボックスのモード（歌詞/発音記号）を切り替えます
        /// </summary>
        private void FlipInputTextBoxMode() {
            TagLyricTextBox kvp = (TagLyricTextBox)m_input_textbox.Tag;
            string new_value = m_input_textbox.Text;
            if ( !kvp.PhoneticSymbolEditMode ) {
                m_input_textbox.BackColor = s_txtbox_backcolor;
            } else {
                m_input_textbox.BackColor = Color.White;
            }
            m_input_textbox.Text = kvp.BufferText;
            m_input_textbox.Tag = new TagLyricTextBox( new_value, !kvp.PhoneticSymbolEditMode );
        }

        /// <summary>
        /// 音の高さを表すnoteから、画面に描くべきy座標を計算します
        /// </summary>
        /// <param name="note"></param>
        /// <returns></returns>
        int YCoordFromNote( int note ) {
            return -1 * (note - 127) * AppManager.EditorConfig.PxTrackHeight - StartToDrawY;
        }

        /// <summary>
        /// ピアノロール画面のy座標から、その位置における音の高さを取得します
        /// </summary>
        /// <param name="y"></param>
        /// <returns></returns>
        int NoteFromYCoord( int y ) {
            return 127 - (int)((double)(StartToDrawY + y) / (double)AppManager.EditorConfig.PxTrackHeight);
        }

        /// <summary>
        /// アンドゥ処理を行います
        /// </summary>
        public void Undo() {
            if ( m_manager.IsUndoAvailable ) {
                m_manager.Undo();
                menuEditRedo.Enabled = m_manager.IsRedoAvailable;
                menuEditUndo.Enabled = m_manager.IsUndoAvailable;
                cMenuPianoRedo.Enabled = m_manager.IsRedoAvailable;
                cMenuPianoUndo.Enabled = m_manager.IsUndoAvailable;
                cMenuTrackSelectorRedo.Enabled = m_manager.IsRedoAvailable;
                cMenuTrackSelectorUndo.Enabled = m_manager.IsUndoAvailable;
                m_mixer_dlg.UpdateStatus();
                UpdateDrawObjectList();
            }
        }

        /// <summary>
        /// リドゥ処理を行います
        /// </summary>
        public void Redo() {
            if ( m_manager.IsRedoAvailable ) {
                m_manager.Redo();
                menuEditRedo.Enabled = m_manager.IsRedoAvailable;
                menuEditUndo.Enabled = m_manager.IsUndoAvailable;
                cMenuPianoRedo.Enabled = m_manager.IsRedoAvailable;
                cMenuPianoUndo.Enabled = m_manager.IsUndoAvailable;
                cMenuTrackSelectorRedo.Enabled = m_manager.IsRedoAvailable;
                cMenuTrackSelectorUndo.Enabled = m_manager.IsUndoAvailable;
                m_mixer_dlg.UpdateStatus();
                UpdateDrawObjectList();
            }
        }

        int StartToDrawY {
            get {
                return (int)((128 * AppManager.EditorConfig.PxTrackHeight - vScroll.Height) * (float)vScroll.Value / ((float)vScroll.Maximum));
            }
        }

        /// <summary>
        /// pがrcの中にあるかどうかを判定します
        /// </summary>
        /// <param name="p"></param>
        /// <param name="rc"></param>
        /// <returns></returns>
        static bool IsInRect( Point p, Rectangle rc ) {
            if ( rc.X <= p.X ) {
                if ( p.X <= rc.X + rc.Width ) {
                    if ( rc.Y <= p.Y ) {
                        if ( p.Y <= rc.Y + rc.Height ) {
                            return true;
                        }
                    }
                }
            }
            return false;
        }

        /// <summary>
        /// マウス位置におけるIDを返します。該当するIDが無ければnullを返します
        /// rectには、該当するIDがあればその画面上での形状を、該当するIDがなければ、
        /// 画面上で最も近かったIDの画面上での形状を返します
        /// </summary>
        /// <param name="mouse_position"></param>
        /// <returns></returns>
        VsqEvent GetItemAtClickedPosition( Point mouse_position, out Rectangle rect ) {
            rect = new Rectangle();
            if ( AppManager._KEY_LENGTH <= mouse_position.X && mouse_position.X <= pictPianoRoll.Width - 18 ) {
                if ( 0 <= mouse_position.Y && mouse_position.Y <= pictPianoRoll.Height - 18 ) {
                    int selected = m_manager.Selected;
                    if ( selected >= 1 ) {
                        for ( int j = 0; j < m_manager.VsqFile.Tracks[selected].Events.Count; j++ ) {
                            int timesig = m_manager.VsqFile.Tracks[selected].Events[j].Clock;
                            int internal_id = m_manager.VsqFile.Tracks[selected].Events[j].InternalID;
                            // イベントで指定されたIDがLyricであった場合
                            if ( m_manager.VsqFile.Tracks[selected].Events[j].ID.type == VsqIDType.Anote &&
                                m_manager.VsqFile.Tracks[selected].Events[j].ID.LyricHandle != null ) {
                                // 発音長を取得
                                int length = m_manager.VsqFile.Tracks[selected].Events[j].ID.Length;
                                int note = m_manager.VsqFile.Tracks[selected].Events[j].ID.Note;
                                int x = XCoordFromClocks( timesig );
                                int y = YCoordFromNote( note );
                                int lyric_width = (int)(length * m_manager.ScaleX);
                                if ( x + lyric_width < 0 ) {
                                    continue;
                                } else if ( pictPianoRoll.Width < x ) {
                                    break;
                                }
                                if ( x <= mouse_position.X && mouse_position.X <= x + lyric_width ) {
                                    if ( y + 1 <= mouse_position.Y && mouse_position.Y <= y + AppManager.EditorConfig.PxTrackHeight ) {
                                        rect = new Rectangle( x, y + 1, lyric_width, AppManager.EditorConfig.PxTrackHeight );
                                        return m_manager.VsqFile.Tracks[selected].Events[j];
                                    }
                                }
                            }
                        }
                    }
                }
            }
            return null;
        }

        private void OpenVsqCore( string file ) {
            m_manager.ReadVsq( file );
            if ( m_manager.VsqFile.Tracks.Count >= 2 ) {
                bool found = false;
                foreach ( MidiEvent item in m_manager.VsqFile.Tracks[0].MidiEvent ) {
                    if ( item.type == MidiEventType.tempo && !found ) {
                        AppManager.BaseTempo = item.intValue[0];
                        found = true;
                    }
                }
                SetHScrollRange( m_manager.VsqFile.TotalClocks );
            }
            AppManager.EditorConfig.PushRecentFiles( file );
            UpdateRecentFileMenu();
            Edited = false;
            m_manager.ClearCommandBuffer();
            m_mixer_dlg.UpdateStatus();
        }

        private void UpdateMenuFonts( string font_name ) {
            if ( font_name == "" ) {
                return;
            }
            Font font = new Font( font_name, menuFile.Font.Size, menuFile.Font.Unit );
            this.Font = font;
            foreach ( Control c in this.Controls ) {
                Boare.Lib.AppUtil.Misc.ApplyFontRecurse( c, font );
            }
        }

        private void picturePositionIndicatorDrawTo( Graphics g ) {
            using ( Font SMALL_FONT = new Font( AppManager.EditorConfig.ScreenFontName, 8 ) ) {
                int width = picturePositionIndicator.Width;
                int height = picturePositionIndicator.Height;

                #region 小節ごとの線
                IEnumerable<VsqBarLineType> enumerable = m_manager.VsqFile.GetEnumerator( ClockFromXCoord( width ) );
                int dashed_line_step = AppManager.GetPositionQuantizeClock();
                foreach ( VsqBarLineType blt in enumerable ) {
                    int local_clock_step = 480 * 4 / blt.LocalDenominator;
                    int x = XCoordFromClocks( blt.Clock );
                    if ( blt.IsSeparator ) {
                        int current = blt.BarCount - m_manager.VsqFile.PreMeasure + 1;
                        g.DrawLine( s_pen_105_105_105,
                                    new Point( x, 3 ),
                                    new Point( x, 46 ) );
                        // 小節の数字
                        g.SmoothingMode = System.Drawing.Drawing2D.SmoothingMode.AntiAlias;
                        g.DrawString( current.ToString(),
                                      SMALL_FONT,
                                      Brushes.Black,
                                      new PointF( x + 4, 6 ) );
                        g.SmoothingMode = System.Drawing.Drawing2D.SmoothingMode.Default;
                    } else {
                        g.DrawLine( s_pen_105_105_105,
                                    new Point( x, 11 ),
                                    new Point( x, 16 ) );
                        g.DrawLine( s_pen_105_105_105,
                                    new Point( x, 26 ),
                                    new Point( x, 31 ) );
                        g.DrawLine( s_pen_105_105_105,
                                    new Point( x, 41 ),
                                    new Point( x, 46 ) );
                    }
                    if ( dashed_line_step > 1 && m_manager.GridVisible ) {
                        int numDashedLine = local_clock_step / dashed_line_step;
                        for ( int i = 1; i < numDashedLine; i++ ) {
                            int x2 = XCoordFromClocks( blt.Clock + i * dashed_line_step );
                            g.DrawLine( s_pen_065_065_065,
                                        new Point( x2, 9 + 5 ),
                                        new Point( x2, 14 + 3 ) );
                            g.DrawLine( s_pen_065_065_065,
                                        new Point( x2, 24 + 5 ),
                                        new Point( x2, 29 + 3 ) );
                            g.DrawLine( s_pen_065_065_065,
                                        new Point( x2, 39 + 5 ),
                                        new Point( x2, 44 + 3 ) );
                        }
                    }
                }
                #endregion

                if ( m_manager.VsqFile != null ) {
                    #region 拍子の変更
                    for ( int i = 0; i < m_manager.VsqFile.TimeSigTable.Count; i++ ) {
                        int clock = m_manager.VsqFile.TimeSigTable[i].Clock;
                        int barcount = m_manager.VsqFile.TimeSigTable[i].BarCount;
                        int x = XCoordFromClocks( clock );
                        if ( width < x ) {
                            break;
                        }
                        string s = m_manager.VsqFile.TimeSigTable[i].Numerator + "/" + m_manager.VsqFile.TimeSigTable[i].Denominator;
                        if ( m_manager.SelectedTimesig.ContainsKey( barcount ) ) {
                            g.DrawString( s,
                                          SMALL_FONT,
                                          AppManager.HilightBrush,
                                          new PointF( x + 4, 36 ) );
                        } else {
                            g.DrawString( s,
                                          SMALL_FONT,
                                          Brushes.Black,
                                          new PointF( x + 4, 36 ) );
                        }

                        if ( m_timesig_dragging ) {
                            if ( m_manager.SelectedTimesig.ContainsKey( barcount ) ) {
                                int edit_clock_x = XCoordFromClocks( m_manager.VsqFile.GetClockFromBarCount( m_manager.SelectedTimesig[barcount].Editing.BarCount ) );
                                g.DrawLine( s_pen_187_187_255,
                                            new Point( edit_clock_x - 1, 32 ),
                                            new Point( edit_clock_x - 1, picturePositionIndicator.Height - 1 ) );
                                g.DrawLine( s_pen_007_007_151,
                                            new Point( edit_clock_x, 32 ),
                                            new Point( edit_clock_x, picturePositionIndicator.Height - 1 ) );
                            }
                        }
                    }
                    #endregion

                    #region テンポの変更
                    for ( int i = 0; i < m_manager.VsqFile.TempoTable.Count; i++ ) {
                        int clock = m_manager.VsqFile.TempoTable[i].Clock;
                        int x = XCoordFromClocks( clock );
                        if ( width < x ) {
                            break;
                        }
                        string s = (60e6 / (float)m_manager.VsqFile.TempoTable[i].Tempo).ToString( "#.00" );
                        if ( m_manager.SelectedTempo.ContainsKey( clock ) ) {
                            g.DrawString( s,
                                          SMALL_FONT,
                                          AppManager.HilightBrush,
                                          new PointF( x + 4, 21 ) );
                        } else {
                            g.DrawString( s,
                                          SMALL_FONT,
                                          Brushes.Black,
                                          new PointF( x + 4, 21 ) );
                        }

                        if ( m_tempo_dragging ) {
                            if ( m_manager.SelectedTempo.ContainsKey( clock ) ) {
                                int edit_clock_x = XCoordFromClocks( m_manager.SelectedTempo[clock].Editing.Clock );
                                g.DrawLine( s_pen_187_187_255,
                                            new Point( edit_clock_x - 1, 18 ),
                                            new Point( edit_clock_x - 1, 32 ) );
                                g.DrawLine( s_pen_007_007_151,
                                            new Point( edit_clock_x, 18 ),
                                            new Point( edit_clock_x, 32 ) );
                            }
                        }
                    }
                    #endregion
                }

                #region 外枠
                /* 左(外側) */
                g.DrawLine( new Pen( Color.FromArgb( 160, 160, 160 ) ),
                            new Point( 0, 0 ),
                            new Point( 0, height - 1 ) );
                /* 左(内側) */
                g.DrawLine( new Pen( Color.FromArgb( 105, 105, 105 ) ),
                            new Point( 1, 1 ),
                            new Point( 1, height - 2 ) );
                /* 中(上側) */
                g.DrawLine( new Pen( Color.FromArgb( 160, 160, 160 ) ),
                            new Point( 1, 47 ),
                            new Point( width - 2, 47 ) );
                /* 中(下側) */
                g.DrawLine( new Pen( Color.FromArgb( 105, 105, 105 ) ),
                            new Point( 2, 48 ),
                            new Point( width - 3, 48 ) );
                // 右(外側)
                g.DrawLine( new Pen( Color.White ),
                            new Point( width - 1, 0 ),
                            new Point( width - 1, height - 1 ) );
                // 右(内側)
                g.DrawLine( new Pen( Color.FromArgb( 241, 239, 226 ) ),
                            new Point( width - 2, 1 ),
                            new Point( width - 2, height - 1 ) );
                #endregion

                #region TEMPO & BEAT
                // TEMPO BEATの文字の部分。小節数が被っている可能性があるので、塗り潰す
                g.FillRectangle(
                    new SolidBrush( picturePositionIndicator.BackColor ),
                    new Rectangle( 2, 3, 65, 45 ) );
                // 横ライン上
                g.DrawLine( new Pen( Color.FromArgb( 104, 104, 104 ) ),
                            new Point( 2, 17 ),
                            new Point( width - 3, 17 ) );
                // 横ライン中央
                g.DrawLine( new Pen( Color.FromArgb( 104, 104, 104 ) ),
                            new Point( 2, 32 ),
                            new Point( width - 3, 32 ) );
                // 横ライン下
                g.DrawLine( new Pen( Color.FromArgb( 104, 104, 104 ) ),
                            new Point( 2, 47 ),
                            new Point( width - 3, 47 ) );
                // 縦ライン
                g.DrawLine( new Pen( Color.FromArgb( 104, 104, 104 ) ),
                            new Point( AppManager._KEY_LENGTH, 2 ),
                            new Point( AppManager._KEY_LENGTH, 46 ) );
                /* TEMPO&BEATとピアノロールの境界 */
                g.DrawLine( new Pen( Color.FromArgb( 104, 104, 104 ) ),
                            new Point( AppManager._KEY_LENGTH, 48 ),
                            new Point( width - 18, 48 ) );
                g.DrawString( "TEMPO",
                              SMALL_FONT,
                              Brushes.Black,
                              new PointF( 11, 20 ) );
                g.DrawString( "BEAT",
                              SMALL_FONT,
                              Brushes.Black,
                              new PointF( 11, 35 ) );
                g.DrawLine( new Pen( Color.FromArgb( 172, 168, 153 ) ),
                            new Point( 0, 0 ),
                            new Point( width, 0 ) );
                g.DrawLine( new Pen( Color.FromArgb( 113, 111, 100 ) ),
                            new Point( 1, 1 ),
                            new Point( width - 1, 1 ) );

                #endregion

                #region 現在のマーカー
                float xoffset = AppManager._KEY_LENGTH + 6 - m_manager.StartToDrawX;
                int marker_x = (int)(m_manager.CurrentClock * m_manager.ScaleX + xoffset);
                if ( AppManager._KEY_LENGTH <= marker_x && marker_x <= width ) {
                    g.DrawLine(
                        new Pen( Color.White, 2f ),
                        new Point( marker_x, 2 ),
                        new Point( marker_x, height ) );
                }
                if ( m_manager.StartMarkerEnabled ) {
                    int x = XCoordFromClocks( m_manager.StartMarker );
                    g.DrawImage(
                        Properties.Resources.start_marker, new Point( x, 3 ) );
                }
                if ( m_manager.EndMarkerEnabled ) {
                    int x = XCoordFromClocks( m_manager.EndMarker ) - 6;
                    g.DrawImage(
                        Properties.Resources.end_marker, new Point( x, 3 ) );
                }
                #endregion
            }
        }
    }

}
