﻿/*
 * VocaloRenderingRunner.cs
 * Copyright (c) 2009 kbinani
 *
 * This file is part of Boare.Cadencii.
 *
 * Boare.Cadencii is free software; you can redistribute it and/or
 * modify it under the terms of the GPLv3 License.
 *
 * Boare.Cadencii is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 */
using System;
using System.Collections.Generic;
using System.Windows.Forms;

using Boare.Lib.Vsq;
using Boare.Lib.Media;
using bocoree;

namespace Boare.Cadencii {

    using boolean = Boolean;

    public class VocaloRenderingRunner : RenderingRunner {
        public String renderer;
        public NrpnData[] nrpn;
        public TempoTableEntry[] tempo;
        public double amplify_left;
        public double amplify_right;
        public int trim_msec;
        public long total_samples;
        public String[] files;
        public double wave_read_offset_seconds;
        public boolean mode_infinite;
        public VstiRenderer driver;
        public boolean direct_play;
        public WaveWriter wave_writer;
        public Vector<WaveReader> s_reader = new Vector<WaveReader>();

        private Object m_locker;
        private int m_trim_remain;
        private boolean s_rendering;
        private long s_total_append = 0;

        public VocaloRenderingRunner( String renderer_,
                                NrpnData[] nrpn_,
                                TempoTableEntry[] tempo_,
                                double amplify_left_,
                                double amplify_right_,
                                int trim_msec_,
                                long total_samples_,
                                String[] files_,
                                double wave_read_offset_seconds_,
                                boolean mode_infinite_,
                                VstiRenderer driver_,
                                boolean direct_play_,
                                WaveWriter wave_writer_,
                                Vector<WaveReader> reader_ ) {
            m_locker = new Object();
            renderer = renderer_;
            nrpn = nrpn_;
            tempo = tempo_;
            amplify_left = amplify_left_;
            amplify_right = amplify_right_;
            trim_msec = trim_msec_;
            total_samples = total_samples_;
            files = files_;
            wave_read_offset_seconds = wave_read_offset_seconds_;
            mode_infinite = mode_infinite_;
            driver = driver_;
            direct_play = direct_play_;
            wave_writer = wave_writer_;
            s_reader = reader_;
        }

        public void abortRendering() {
            driver.DllInstance.AbortRendering();
            for ( int i = 0; i < s_reader.size(); i++ ) {
                s_reader.get( i ).Close();
                s_reader.set( i, null );
            }
            s_reader.clear();
        }

        public double getProgress() {
            return driver.DllInstance.GetProgress();
        }

        public boolean isRendering() {
            return s_rendering;
        }

        public void setRendering( boolean value ) {
            s_rendering = value;
        }

        public void run() {
            if ( driver == null ) {
                return;
            }
            if ( !driver.Loaded ) {
                return;
            }
            int tempo_count = tempo.Length;
            float first_tempo = 125.0f;
            if ( tempo.Length > 0 ) {
                first_tempo = (float)(60e6 / (double)tempo[0].Tempo);
            }
            byte[] masterEventsSrc = new byte[tempo_count * 3];
            int[] masterClocksSrc = new int[tempo_count];
            int count = -3;
            for ( int i = 0; i < tempo.Length; i++ ) {
                count += 3;
                masterClocksSrc[i] = tempo[i].Clock;
                byte b0 = (byte)(tempo[i].Tempo >> 16);
                uint u0 = (uint)(tempo[i].Tempo - (b0 << 16));
                byte b1 = (byte)(u0 >> 8);
                byte b2 = (byte)(u0 - (u0 << 8));
                masterEventsSrc[count] = b0;
                masterEventsSrc[count + 1] = b1;
                masterEventsSrc[count + 2] = b2;
            }
            driver.DllInstance.SendEvent( masterEventsSrc, masterClocksSrc, 0 );

            int numEvents = nrpn.Length;
            byte[] bodyEventsSrc = new byte[numEvents * 3];
            int[] bodyClocksSrc = new int[numEvents];
            count = -3;
            int last_clock = 0;
            for ( int i = 0; i < numEvents; i++ ) {
                count += 3;
                bodyEventsSrc[count] = 0xb0;
                bodyEventsSrc[count + 1] = nrpn[i].getParameter();
                bodyEventsSrc[count + 2] = nrpn[i].Value;
                bodyClocksSrc[i] = nrpn[i].getClock();
                last_clock = nrpn[i].getClock();
            }

            int index = tempo_count - 1;
            for ( int i = tempo_count - 1; i >= 0; i-- ) {
                if ( tempo[i].Clock < last_clock ) {
                    index = i;
                    break;
                }
            }
            int last_tempo = tempo[index].Tempo;
            int trim_remain = VSTiProxy.GetErrorSamples( first_tempo ) + (int)(trim_msec / 1000.0 * VSTiProxy._SAMPLE_RATE);
            m_trim_remain = trim_remain;

            driver.DllInstance.SendEvent( bodyEventsSrc, bodyClocksSrc, 1 );

            s_rendering = true;
            if ( wave_writer != null ) {
                if ( m_trim_remain < 0 ) {
                    double[] d = new double[-m_trim_remain];
                    for ( int i = 0; i < -m_trim_remain; i++ ) {
                        d[i] = 0.0;
                    }
                    wave_writer.Append( d, d );
                    m_trim_remain = 0;
                }
            }

            driver.DllInstance.WaveIncoming += vstidrv_WaveIncoming;
            driver.DllInstance.RenderingFinished += vstidrv_RenderingFinished;
            driver.DllInstance.StartRendering( total_samples, mode_infinite );
            while ( s_rendering ) {
                Application.DoEvents();
            }
            s_rendering = false;
            driver.DllInstance.WaveIncoming -= vstidrv_WaveIncoming;
            driver.DllInstance.RenderingFinished -= vstidrv_RenderingFinished;
            if ( direct_play ) {
                PlaySound.WaitForExit();
            }
        }

        private void vstidrv_RenderingFinished() {
            s_rendering = false;
        }

        private unsafe void vstidrv_WaveIncoming( double[] L, double[] R ) {
            if ( !s_rendering ) {
                return;
            }
            lock ( m_locker ) {
                if ( m_trim_remain > 0 ) {
                    if ( m_trim_remain >= L.Length ) {
                        m_trim_remain -= L.Length;
                        return;
                    }
                    int actual_append = L.Length - m_trim_remain;
                    double[] dL = new double[actual_append];
                    double[] dR = new double[actual_append];
                    for ( int i = 0; i < actual_append; i++ ) {
                        dL[i] = L[i + m_trim_remain] * amplify_left;
                        dR[i] = R[i + m_trim_remain] * amplify_right;
                    }
                    if ( wave_writer != null ) {
                        wave_writer.Append( dL, dR );
                    }
                    long start = s_total_append + (long)(wave_read_offset_seconds * VSTiProxy._SAMPLE_RATE);
                    for ( int i = 0; i < s_reader.size(); i++ ) {
                        double[] reader_r;
                        double[] reader_l;
                        s_reader.get( i ).Read( start, actual_append, out reader_l, out reader_r );
                        for ( int j = 0; j < actual_append; j++ ) {
                            dL[j] += reader_l[j];
                            dR[j] += reader_r[j];
                        }
                        reader_l = null;
                        reader_r = null;
                    }
                    if ( direct_play ) {
                        PlaySound.Append( dL, dR, actual_append );
                    }
                    dL = null;
                    dR = null;
                    m_trim_remain = 0;
                    s_total_append += actual_append;
                } else {
                    int length = L.Length;
                    for ( int i = 0; i < length; i++ ) {
                        L[i] = L[i] * amplify_left;
                        R[i] = R[i] * amplify_right;
                    }
                    if ( wave_writer != null ) {
                        wave_writer.Append( L, R );
                    }
                    long start = s_total_append + (long)(wave_read_offset_seconds * VSTiProxy._SAMPLE_RATE);
                    for ( int i = 0; i < s_reader.size(); i++ ) {
                        double[] reader_r;
                        double[] reader_l;
                        s_reader.get( i ).Read( start, length, out reader_l, out reader_r );
                        for ( int j = 0; j < length; j++ ) {
                            L[j] += reader_l[j];
                            R[j] += reader_r[j];
                        }
                        reader_l = null;
                        reader_r = null;
                    }
                    if ( direct_play ) {
                        PlaySound.Append( L, R, L.Length );
                    }
                    s_total_append += length;
                }
            }
        }
    }

}
