#include "VSQSequencer.h"
#include <time.h>

VSQSequencer::VSQSequencer(){
    //各ノートに対応する周波数をあらかじめ計算
    for( int i=0; i < 128; i++ ){
        mNoteFrequency[i] = A4_PITCH * pow( 2.0, (i - A4_NOTE) / 12.0 );
    }
    //無声音用
    mNoteFrequency[128] = 0.0;

    //ビブラート位相の角速度をあらかじめ計算
    for( int i = 0; i < 128; i++ ){
        mOmegaFromVibRate[i] = GetOmegaFromVibratoRate( i );
    }

    //コントロールトラックの追加
    for( int i = 0; i < CONTROL_TRACK_NUM; i++ ){
        EventManager* pTarget = new EventManager;

        //Tick=0にデフォルトの値を書きこむ
        pTarget->SetValue( 0, iControlDefaultValue[i] );

        mEventMap.insert( make_pair( sControlNames[i], pTarget ) );
        mManagerList.push_back( pTarget );
    }

    time_t timer;
    NoteManager *pTarget = new NoteManager;
    mEventMap.insert( make_pair( NOTE_TRACK_NAME, pTarget ) );
    mManagerList.push_back( pTarget );

    time( &timer );

    mFluctBegin = (double)(timer % 1000) / 10.0;

    mTempo=0.0;
}

VSQSequencer::~VSQSequencer(){
    for( list<EventManager*>::iterator i = mManagerList.begin(); i != mManagerList.end(); i++ ){
        SAFE_DELETE( (*i) );
    }
#ifdef USE_MULTI_SINGER
    for( vector<UTAUManager *>::size_type i = 0; i < mUtauManagers.size(); i++ ){
        UTAUManager *utau_manager = mUtauManagers[i];
        utau_manager->Uninitialize();
    }
#else
    mUtauManager.Uninitialize();
#endif
}

bool VSQSequencer::LoadFile( string file_name, double frame_shift ){
    FILE *ifs;
    const int BUF_LEN = 4096;
    char buffer[BUF_LEN];
    string line;
    string current_loading = "";
    MAP_TYPE<string, EventManager *>::iterator h_i;
    bool result = true;
    int current_line = 0;
    bool oto_flag = false;

    this->mFrameShift = frame_shift;

#ifdef __GNUC__
    ifs = fopen( file_name.c_str(), "rb" );
#else
    fopen_s( &ifs, file_name.c_str(), "rb" );
#endif

    if( NULL == ifs ){
        cout << "error; io error on file:" << file_name.c_str() << endl;
        return false;
    }

#ifdef USE_MULTI_SINGER
    while( NULL != c_getline( buffer, BUF_LEN, ifs ) ){
        line = buffer;

        if( line.find( "[" ) == 0 ){
            current_loading = line;

            string::size_type indx_id = line.find( "[ID#" );
            string::size_type indx_h = line.find( "[h#" );
            if( indx_id == 0 || indx_h == 0 ){
                //これから読み込むIDを設定しておく。
                h_i = mEventMap.find( NOTE_TRACK_NAME );
                if( h_i != mEventMap.end() ){
                    NoteManager *pNoteManager;
                    pNoteManager = dynamic_cast<NoteManager *>( h_i->second );

                    //書き込むべきノート管理クラスがなければエラー
                    if( NULL != pNoteManager ){
                        if( indx_id == 0 ){
                            pNoteManager->SetEditingID( atoi( line.substr( 4, line.find( "]" ) ).c_str() ) );
                        }else if( indx_h == 0 ){
                            pNoteManager->SetEditingHandle( atoi( line.substr( 3, line.find( "]" ) ).c_str() ) );
                        }
                    }else{
                        //result = false;
                    }
                }
            }
        }else{
            // 空行またはコメント
            if( line.find( "//" ) == 0 || line.size() == 0 ){
                continue;
            }
            if( current_loading.compare( "[oto.ini]" ) == 0 ){
                // oto.iniの指定
                //これから読み込むIDを設定しておく。
                h_i = mEventMap.find( NOTE_TRACK_NAME );
                if( h_i != mEventMap.end() ){
                    NoteManager *pNoteManager;
                    pNoteManager = dynamic_cast<NoteManager *>( h_i->second );

                    string::size_type indx_tab = line.find( "\t" );
                    string singer_name;
                    string path_oto_ini;
                    if( indx_tab == string::npos ){
                        // タブが無い場合この行すべてがoto.iniの指定とみなし、
                        // 歌手は無名(空文字)とする
                        singer_name = "";
                        path_oto_ini = line;
                    }else{
                        // タブの左が歌手名(ex.$07010002)、右がoto.iniのパス
                        singer_name = line.substr( 0, indx_tab );
                        path_oto_ini = line.substr( indx_tab + 1 );
                    }
                    
                    // 歌手名を登録して
                    pNoteManager->AddSinger( singer_name );
                    
                    // 原音を読み込んでリストに追加
                    UTAUManager *newone = new UTAUManager();
                    newone->Initialize( path_oto_ini );
                    mUtauManagers.push_back( newone );
                    
                    // 原音ディレクトリのリストに追加
                    string dir_oto_ini = path_oto_ini.substr( 0, path_oto_ini.find( "oto.ini" ) );
                    mFilePaths.push_back( dir_oto_ini );

                    oto_flag = true;
                }
            }else if( current_loading.compare( "[Tempo]" ) == 0 ){
                // テンポの指定
                mTempo = atof( line.c_str() );
            }else{
                if( current_loading.compare( NOTE_TRACK_NAME ) == 0 && line.find( "EOS" ) != string::npos ){
                    // イベントリストを読んでる状況で，かつEOSの文字列が発見された
                    mEndTick = atol( line.substr( 0, line.find( "=" ) ).c_str() );
                }else{
                    string act_current_loading = current_loading;
                    if( current_loading.find( "[h#" ) == 0 || 
                        current_loading.find( "[ID#" ) == 0 ){
                        act_current_loading = NOTE_TRACK_NAME;
                    }
                    h_i = mEventMap.find( act_current_loading );
                    if( h_i != mEventMap.end() ){
                        string::size_type indx_equal = line.find( "=" );
                        
                        string key  = line.substr( 0, indx_equal );
                        string value = line.substr( indx_equal + 1 );

                        if( current_loading.compare( NOTE_TRACK_NAME ) == 0 ){
                            string::size_type indx_comma = value.find( "," );
                            while( indx_comma != string::npos ){
                                string tvalue = value.substr( 0, indx_comma );
                                /*result &= */h_i->second->SetValue( key, tvalue );
                                value = value.substr( indx_comma + 1 );
                                indx_comma = value.find( "," );
                            }
                        }
                        /*result &= */h_i->second->SetValue( key, value );
                    }else{
                        cout << "error; file: " << file_name.c_str() << "; unexpected token: " << current_loading.c_str() <<endl;
                    }
                }
            }
        }

        //エラー処理
        if( !result ){
            cout << "error; line#: " << current_line << endl;
            if( NULL != ifs ){
                fclose( ifs );
            }
            return false;
        }
    }
#else
    //シーケンスファイルの解釈
    while( NULL != c_getline( buffer, BUF_LEN, ifs ) ){
        line = buffer;
        current_line++;

        if( line.size() == 0 || line.find( "//" ) == 0 ){    //読み飛ばし条件
            //特殊条件１：コメント行及び空行は読み飛ばす。
            continue;

        }else if( line.compare( "[oto.ini]" ) == 0 ){
            //特殊条件２：oto.iniの設定の時は勝手にやる。
            if( NULL == c_getline( buffer, BUF_LEN, ifs ) ){
                oto_flag = false;
                break;
            }
            line = buffer;
            current_line++;

            oto_flag = mUtauManager.Initialize( line );

            if( !oto_flag ){
                cout << "error; parser error for oto.ini; line=" << line.c_str() << endl;
            }else{
                mFilePath = line.substr( 0, line.find( "oto.ini" ) );
            }

        //特殊条件３：テンポ指定の場合も勝手にやる。
        }else if( line.compare( "[Tempo]" ) == 0 ){
            if( NULL == c_getline( buffer, BUF_LEN, ifs ) ){
                mTempo = 0.0;
                break;
            }
            current_line++;
            mTempo = atof( buffer );

        //特殊条件４：EOSが来たら終端Tickを確保する。
        }else if( line.find( "EOS" ) != string::npos ){
            mEndTick = atol( line.substr( 0, line.find( "=" ) ).c_str() );

        //特殊条件５：[ID#xxxx]の時と[h#xxxx]はEventListの読み込み扱い
        }else if( line.find( "[ID#" ) != string::npos || line.find( "[h#" ) != string::npos ){

            //これから読み込むIDを設定しておく。
            h_i = mEventMap.find( NOTE_TRACK_NAME );
            if( h_i != mEventMap.end() ){
                NoteManager *pNoteManager;
                pNoteManager = dynamic_cast<NoteManager *>( h_i->second );

                //書き込むべきノート管理クラスがなければエラー
                if( NULL != pNoteManager ){
                    if( line.find( "[ID#" ) != string::npos ){
                        pNoteManager->SetEditingID( atoi( line.substr( 4, line.find( "]" ) ).c_str() ) );
                    }else if( line.find( "[h#" ) != string::npos ){
                        pNoteManager->SetEditingHandle( atoi( line.substr( 3, line.find( "]" ) ).c_str() ) );
                    }
                }else{
                    result = false;
                }
            }

        //通常処理１：[xxxxxxxx]の場合xxxxxxxを現在読み込んでいるリスト名に設定。
        }else if( line.find( "[" ) == 0 ){
            current_loading = line;

        //通常処理２：値を設定している行の場合は現在のリストに文字列を渡す。
        }else{
            h_i = mEventMap.find( current_loading );
            if( h_i != mEventMap.end() ){
                result = h_i->second->SetValue( line.substr( 0, line.find("=") ), line.substr( line.find( "=" ) + 1 ) );
            }else{
                cout << "error; file: " << file_name.c_str() << "; unexpected token: " << current_loading.c_str() <<endl;
//                result=false;
            }
        }

        //エラー処理
        if( !result ){
            cout << "error; line#: " << current_line << endl;
            if( NULL != ifs ){
                fclose( ifs );
            }
            return false;
        }
    }
#endif

    if( NULL != ifs ){
        fclose( ifs );
    }

    //デフォルト設定を読み込んでもなおだめなとき
    if( !oto_flag ){
        return false;
    }

    //テンポ設定がinputFile内に存在しない場合
    if( mTempo == 0.0 ){
        //return false;
        mTempo = 120.0; //き・ち・く
    }

    h_i = mEventMap.find( NOTE_TRACK_NAME );
    if( h_i != mEventMap.end() ){
        NoteManager* pNoteManager;
        pNoteManager = dynamic_cast<NoteManager *>( h_i->second );

        //書き込むべきノート管理クラスがなければエラー
        if( pNoteManager != NULL ){
#ifdef USE_MULTI_SINGER
            result = pNoteManager->CheckContinuity( mTempo, &mUtauManagers, mFrameShift );
#else
            result = pNoteManager->CheckContinuity( mTempo, &mUtauManager, mFrameShift );
#endif
            if( !result ){
                cout << "error; cannot find note. synthesize will be omitted." << endl;
            }
            mEndNoteID = pNoteManager->GetEndID();
            mEndFrame = pNoteManager->GetEndFrame();
            mBeginFrame = pNoteManager->GetBeginFrame();
            mBeginTick = pNoteManager->GetBeginTick();
        }else{
            cout << "error; cannot find note manager" << endl;
            result=false;
        }
    }

    for( list<EventManager*>::iterator it = mManagerList.begin(); it != mManagerList.end(); it++ ){
        (*it)->CutDownEvent( mBeginTick );
    }
    
    return result;
}


NoteEvent *VSQSequencer::GetNoteEvent( long nFrame ){
    NoteEvent *pResult = NULL;
    NoteManager *pNoteManager = NULL;

    MAP_TYPE<string, EventManager *>::iterator h_i;

    h_i = mEventMap.find( NOTE_TRACK_NAME );

    if( h_i != mEventMap.end() ){
        pNoteManager = dynamic_cast<NoteManager*>( h_i->second );

        //書き込むべきノート管理クラスがなければエラー
        if( pNoteManager != NULL ){
            pResult=pNoteManager->GetNoteEventByFrame( nFrame );
        }
    }

    return pResult;
}

double VSQSequencer::GetF0( long nFrame, double dFrameShift ){
    double dResult=0.0;
    double dTemp;

    NoteEvent *pNoteEvent = GetNoteEvent(nFrame);

    if( pNoteEvent == NULL ){
        return dResult;
    }

    long nRelativeFrame = nFrame - (long)((double)(pNoteEvent->mTick) / TICK_PER_SEC / mTempo * 1000.0 / dFrameShift);

    dResult = GetPitchFromNote( pNoteEvent, nFrame, dFrameShift );

    /*==========================DynamicF0Model===========================*/
    double dBendLength = 400.0;
    double dBendDepth = -3.0 * (double)(pNoteEvent->mBendDepth) / 100.0;
    double dPosition = (double)nRelativeFrame * dFrameShift / dBendLength;
    
    /*----------------------------ApplyOverShoot*/
    if( pNoteEvent->mIsContinuousFront ){
        NoteEvent *pPrev = pNoteEvent->mPreviousNote;

        if( pPrev != NULL ){
            dBendLength = (double)(pPrev->mBendLength) / 100.0 * (double)(pPrev->mEndFrame - pPrev->mBeginFrame) * dFrameShift;
            dBendDepth = (double)(pNoteEvent->mNote - pPrev->mNote) * (double)(pPrev->mBendDepth) / 100.0;

            if( 50.0 > dBendLength ){
                dBendLength = 50.0;
            }
            dPosition = (double)(nFrame - pPrev->mEndFrame) * dFrameShift / (dBendLength - (double)(pNoteEvent->mBeginFrame - pPrev->mEndFrame) * dFrameShift);
        }
    }else{
        dPosition = dPosition / 2.0 + 0.5;
    }

    if( dPosition * 4.0 < 1.0 ){
        dResult *= pow( 2.0, dBendDepth * sin( ST_PI * dPosition * 4.0 ) / 2.0 / 12.0 );
    }
    /*----------------------------------------End*/

    /*-------------------ApplyF0Morph&Preparation*/
    dBendLength = 80.0;
    dBendDepth = 0.0;

    if( pNoteEvent->mIsContinuousBack ){
        if( pNoteEvent->mNextNote != NULL ){
            dBendLength = (double)(pNoteEvent->mBendLength) / 100.0 * (double)(pNoteEvent->mEndFrame - pNoteEvent->mBeginFrame) * dFrameShift;
            dBendDepth = (double)(pNoteEvent->mNote - pNoteEvent->mNextNote->mNote) * (double)(pNoteEvent->mBendDepth) / 100.0;

            if( 50.0 > dBendLength ){
                dBendLength = 100.0;
            }
        }
    }

    double dBendBeginFrame = (double)(pNoteEvent->mEndFrame) - dBendLength / dFrameShift;

    if( dBendBeginFrame < nFrame ){
        double dLengthRate = ((double)(pNoteEvent->mEndFrame) - (double)nFrame) / ((double)(pNoteEvent->mEndFrame) - dBendBeginFrame);
        dTemp = dResult; // ignore :*pow(2.0,-4.0*dBendDepth/12.0);
        if( pNoteEvent->mIsContinuousBack ){
            if( pNoteEvent->mNextNote != NULL ){
                dTemp = pow( 2.0, -dLengthRate * dBendDepth / 12.0 ) * GetPitchFromNote( pNoteEvent->mNextNote, nFrame, dFrameShift );
            }
        }
        //Morph
        dResult = pow( dResult, dLengthRate ) * pow( dTemp, 1.0 - dLengthRate );
        //Preparation
        dResult *= pow( 2.0, 2.0 * dBendDepth * sin( ST_PI * dLengthRate ) / 12.0 );
    }
    /*----------------------------------------End*/

    /*=======================================================================*/
    dResult *= GetPitchBendRate( nFrame, dFrameShift );
    dResult *= GetPitchFluctuation( nFrame, dFrameShift );

    return dResult;
}

double VSQSequencer::GetPitchFluctuation( long nFrame, double dFrameShift ){
    double dSec = mFluctBegin + (double)nFrame * dFrameShift / 1000.0;
    double dResult = 1.0 + (sin( 12.7 * ST_PI * dSec ) + sin( 7.1 * ST_PI * dSec ) + sin( 4.7 * ST_PI * dSec ) / 3.0) / 300.0;

    return dResult;
}

double    VSQSequencer::GetDynamics( long nFrame, double dFrameShift ){
    double dResult = 0.0;
    double dRate;
    double dTemp;

    NoteEvent *pNoteEvent = GetNoteEvent( nFrame );

    if( pNoteEvent == NULL ){
        return dResult;
    }

    /*==========================DynamicVolumeControl===========================*/
    long nRelativeFrame = nFrame - (long)((double)(pNoteEvent->mTick) / TICK_PER_SEC / mTempo * 1000.0 * dFrameShift);

    dResult = GetDynamicsFromNote( pNoteEvent, nFrame, dFrameShift );

    if( pNoteEvent->mIsContinuousBack ){
        if( pNoteEvent->mNextNote->mBeginFrame < nFrame ){
            double dBendBeginFrame = (double)(pNoteEvent->mNextNote->mBeginFrame);

            dTemp = GetDynamicsFromNote( pNoteEvent->mNextNote, nFrame, dFrameShift );

            dResult = dResult * (1.0 - ((double)(pNoteEvent->mEndFrame) - (double)nFrame) / ((double)(pNoteEvent->mEndFrame) - dBendBeginFrame))
                    + dTemp * ((double)(pNoteEvent->mEndFrame) - (double)nFrame) / ((double)(pNoteEvent->mEndFrame) - dBendBeginFrame);
        }
    }

    dRate = GetDynamicsRate( nFrame, dFrameShift );
    dRate /= GetBrethiness( nFrame, dFrameShift );
    double dBrightness = ((double)(GetControlValue( BRIGHTNESS, nFrame, dFrameShift )) - 64.0) / 64.0;
    dRate *= pow( 2.0, dBrightness );
    dResult *= dRate;

    return dResult;
}

double VSQSequencer::GetPitchFromNote( NoteEvent *pNoteEvent, long nFrame, double dFrameShift ){
    if( pNoteEvent == NULL ){
        return 0.0;
    }

    double dRate;
    double dTemp = mNoteFrequency[pNoteEvent->mNote];

    dRate = pow( 2.0, 1.0 / 12.0 * GetVibratoRate( pNoteEvent, nFrame, dFrameShift ) );
    dTemp *= dRate;

    return dTemp;
}

double VSQSequencer::GetPitchBendRate( long nFrame, double dFrameShift ){
    double dRate = 1.0;
    int iPitchBend;
    int iPitchBendSensitivity;

    iPitchBend = GetControlValue( PITCH_BEND, nFrame, dFrameShift );
    iPitchBendSensitivity = GetControlValue( PITCH_SENS, nFrame, dFrameShift );

    dRate = pow( 2.0, (double)iPitchBendSensitivity * (double)iPitchBend / 98304.0 );

    return dRate;
}

double VSQSequencer::GetVibratoRate( NoteEvent *pNoteEvent, long nFrame, double dFrameShift ){
    // 接頭辞 t : 単位がTickな変数
    //        s : 単位が秒な変数
    //        f : 単位がframeな変数
    long fFrame = nFrame;

    long tNoteStart = pNoteEvent->mTick;
    long tStart = tNoteStart + pNoteEvent->mVibratoDelay;
    long tEnd = (long)(fFrame * dFrameShift * mTempo * TICK_PER_SEC / 1000.0);
    if( tEnd < tStart ){
        // まだビブラートが始まっていないので0を返す
#ifdef _DEBUG
        FILE* fp = fopen( "vibrato.txt", "a" );
        fprintf( fp, "%d\t%f\n", nFrame, 0.0 );
        fclose( fp );
#endif
        return 0.0;
    }

    // 第tStart tickから第tEnd tickまでの，ビブラート位相を積算する
    double phase = 0.0; // 初期位相(ラヂアン)
    double sLast = tStart / TICK_PER_SEC / mTempo; // 直前のtickでの，曲頭からの秒数
    int last_vibrato_rate = pNoteEvent->mVibratoRate.GetValue( tStart - tNoteStart ); // 直前のtickでの，VibRate
    double sThis;
    double omega = mOmegaFromVibRate[last_vibrato_rate];
    double last_omega = omega;
    for( long tTick = tStart + 1; tTick <= tEnd; tTick++ ){
        int vibrato_rate = pNoteEvent->mVibratoRate.GetValue( tTick - tNoteStart );
        if( vibrato_rate == last_vibrato_rate ){
            // VibRateが同じ場合は計算無駄なので次回以降に繰越
            continue;
        }
        omega = mOmegaFromVibRate[vibrato_rate];
        
        sThis = tTick / TICK_PER_SEC / mTempo; // 第tTick tickにおける，曲頭からの秒数
        double sDt = sThis - sLast; // デルタ秒
        phase += omega * sDt; // デルタ秒分だけ，位相が増える
        sLast = sThis;
        last_vibrato_rate = vibrato_rate;
        last_omega = omega;
    }
    sThis = fFrame * dFrameShift / 1000.0;
    if( sThis > sLast ){
        phase += last_omega * (sThis - sLast);
    }

    // 要求されたフレーム（nFrame）での，ビブラートの振幅
    int vibrato_depth = pNoteEvent->mVibratoDepth.GetValue( tEnd - tNoteStart );
    double amplitude = vibrato_depth * 2.5 / 127.0 / 2.0; // ビブラートの振幅。

    // 位相・振幅を加味して，要求されたフレームでの振動カーブ値を計算．
    // ビブラートの最初で，ピッチ変化を下降形にする場合は，phaseの初期値double phase = 0.0の変わりにdouble phase = ST_PIとすればおｋ
    double result = amplitude * sin( phase );
    return result;
}

double VSQSequencer::GetDynamicsRate( long nFrame, double dFrameShift ){
    double dRate;
    int iDynamics;

    iDynamics = GetControlValue( DYNAMICS, nFrame, dFrameShift );

    dRate = (double)iDynamics / 64.0;

    return dRate;
}

double VSQSequencer::GetDynamicsFromNote( NoteEvent *pNoteEvent, long nFrame, double dFrameShift ){
    double dRate = 1.0;

    double dAccentRate = pow( 2.0, ((double)(pNoteEvent->mAccent) - 50.0) / 50.0 );
    double dDecayRate = 1.0 - (double)(pNoteEvent->mDecay) / 100.0;
    double dNextAccentRate = 0.0;

    long nNoteLength = pNoteEvent->mEndFrame - pNoteEvent->mBeginFrame;
    long nAttackEnd = nNoteLength / 3;
    long nReleaseBegin = 2 * nNoteLength / 3;
    long nRelativeFrame = nFrame - pNoteEvent->mBeginFrame;

    if( nAttackEnd > 100.0 / dFrameShift ){
        nReleaseBegin = (long)( nNoteLength - 100.0 / dFrameShift );
    }

    if( pNoteEvent->mIsContinuousBack ){
        dNextAccentRate = pow( 2.0, ((double)(pNoteEvent->mNextNote->mAccent) - 50.0) / 50.0 );
    }

    if( pNoteEvent->mEndFrame - pNoteEvent->mBeginFrame < nAttackEnd ){
        nAttackEnd = pNoteEvent->mEndFrame - pNoteEvent->mBeginFrame;
    }

    /*--Attack--*/
    if( nRelativeFrame < nAttackEnd ){
        dRate *= pow( dAccentRate, (1.0 - (double)nRelativeFrame / (double)nAttackEnd) );
    }

    /*--Decay--*/
    //Nothing to do.

    /*--Release--*/
    if( nRelativeFrame > nReleaseBegin ){
        dRate *= (1.0 - (double)(nRelativeFrame - nReleaseBegin) / (double)(nNoteLength - nReleaseBegin)) 
              + dNextAccentRate * (double)(nRelativeFrame - nReleaseBegin) / (double)(nNoteLength - nReleaseBegin);
    }

    return dRate;
}

int VSQSequencer::GetControlValue( int iControlNum, long nFrame, double dFrameShift ){
    if( iControlNum < 0 || iControlNum >= CONTROL_TRACK_NUM ){
        return -1;
    }

    MAP_TYPE<string, EventManager *>::iterator h_i;
    int iResult = -1;
    EventManager *pEventManager;
    long nTick = (long)((double)nFrame * dFrameShift * mTempo * TICK_PER_SEC / 1000.0);

    h_i = mEventMap.find( sControlNames[iControlNum] );

    if( h_i != mEventMap.end() ){
        pEventManager = h_i->second;
        if( pEventManager ){
            if( (iResult = pEventManager->GetValue( nTick )) == -1 ){
                iResult = iControlDefaultValue[iControlNum];
            }
        }
    }

    return iResult;
}

double VSQSequencer::GetBrethiness( long nFrame, double dFrameShift ){
    double dRate;
    int iBrethiness;

    iBrethiness = GetControlValue( BRETHINESS, nFrame, dFrameShift );

    dRate = 1.0 - (double)iBrethiness / 144.0;

    return dRate;
}

double VSQSequencer::GetOmegaFromVibratoRate( int vibrato_rate ){
    double period = exp( 5.24 - 1.07e-2 * vibrato_rate ) * 2.0 / 1000.0;
    return 2.0 * ST_PI / period;
}
