﻿/*
 * VsqHandle.h
 * Copyright (c) 2009 kbinani
 *
 * This file is part of Boare.Lib.Vsq.
 *
 * Boare.Lib.Vsq is free software; you can redistribute it and/or
 * modify it under the terms of the BSD License.
 *
 * Boare.Lib.Vsq is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 */

namespace Boare{ namespace Lib{ namespace Vsq{

    /// <summary>
    /// ハンドルを取り扱います。ハンドルにはLyricHandle、VibratoHandleおよびIconHandleがある
    /// </summary>
    class VsqHandle {
    private:
        VsqHandleType m_type;
        int m_index;
    protected:
        string m_icon_id;
        string m_ids;
        Lyric m_lyric;
        int m_original;
        string m_caption;
        int m_length;
        int m_start_depth;
        VibratoBPList m_depth_bp;
        int m_start_rate;
        VibratoBPList m_rate_bp;
        int m_language;
        int m_program;
    public:
        LyricHandle *ConvertToLyricHandle() {
            LyricHandle *ret = new LyricHandle();
            ret->m_lyric = m_lyric;
            ret->m_type = m_type;
            ret->m_index = m_index;
            return ret;
        }

        VibratoHandle *ConvertToVibratoHandle() {
            VibratoHandle *ret = new VibratoHandle();
            ret->m_type = m_type;
            ret->m_index = m_index;
            ret->Caption = m_caption;
            ret->m_depth_bp = m_depth_bp.Clone();
            //ret.DepthBPX = m_depth_bp_x;
            //ret.DepthBPY = m_depth_bp_y;
            ret->IconID = m_icon_id;
            ret->IDS = m_ids;
            ret->Index = m_index;
            ret->Length = m_length;
            ret->Original = m_original;
            ret->m_rate_bp = m_rate_bp.Clone();
            //ret.RateBPNum = m_rate_bp_num;
            //ret.RateBPX = m_rate_bp_x;
            //ret.RateBPY = m_rate_bp_y;
            ret->StartDepth = m_start_depth;
            ret->StartRate = m_start_rate;
            return ret;
        }

        public IconHandle ConvertToIconHandle() {
            IconHandle ret = new IconHandle();
            ret.m_type = m_type;
            ret.m_index = m_index;
            ret.Caption = m_caption;
            ret.IconID = m_icon_id;
            ret.IDS = m_ids;
            ret.Index = m_index;
            ret.Language = m_language;
            ret.Length = m_length;
            ret.Original = m_original;
            ret.Program = m_program;
            return ret;
        }

        public VsqHandleType Type {
            get {
                return m_type;
            }
            set {
                m_type = value;
            }
        }

        public int Index {
            get {
                return m_index;
            }
            set {
                m_index = value;
            }
        }

        protected VsqHandle() {
        }
        
        /// <summary>
        /// インスタンスをストリームに書き込みます。
        /// encode=trueの場合、2バイト文字をエンコードして出力します。
        /// </summary>
        /// <param name="sw">書き込み対象</param>
        /// <param name="encode">2バイト文字をエンコードするか否かを指定するフラグ</param>
        public void write( TextMemoryStream sw, bool encode ) {
            sw.WriteLine( this.ToString( encode ) );
        }

        /// <summary>
        /// FileStreamから読み込みながらコンストラクト
        /// </summary>
        /// <param name="sr">読み込み対象</param>
        public VsqHandle( TextMemoryStream sr, int value, ref string last_line ) {
            this.Index = value;
            string[] spl;
            string[] spl2;

            // default値で梅
            this.Type = VsqHandleType.Vibrato;
            m_icon_id = "";
            m_ids = "normal";
            m_lyric = new Lyric( "" );
            m_original = 0;
            m_caption = "";
            m_length = 0;
            m_start_depth = 0;
            m_depth_bp = null;
            //m_depth_bp_num = 0;
            int depth_bp_num = 0;
            //m_depth_bp_x = null;
            //m_depth_bp_y = null;
            m_start_rate = 0;
            m_rate_bp = null;
            //m_rate_bp_num = 0;
            int rate_bp_num = 0;
            //m_rate_bp_x = null;
            //m_rate_bp_y = null;
            m_language = 0;
            m_program = 0;

            string tmpDepthBPX = "";
            string tmpDepthBPY = "";
            string tmpRateBPX = "";
            string tmpRateBPY = "";

            // "["にぶち当たるまで読込む
            last_line = sr.ReadLine();
            while ( !last_line.StartsWith( "[" ) ) {
                spl = last_line.Split( new char[] { '=' } );
                switch ( spl[0] ) {
                    case "Language":
                        m_language = int.Parse( spl[1] );
                        break;
                    case "Program":
                        m_program = int.Parse( spl[1] );
                        break;
                    case "IconID":
                        m_icon_id = spl[1];
                        break;
                    case "IDS":
                        m_ids = spl[1];
                        break;
                    case "Original":
                       m_original = int.Parse( spl[1] );
                        break;
                    case "Caption":
                        m_caption = spl[1];
                        for ( int i = 2; i < spl.Length; i++ ) {
                            m_caption += "=" + spl[i];
                        }
                        break;
                    case "Length":
                        m_length = int.Parse( spl[1] );
                        break;
                    case "StartDepth":
                        m_start_depth = int.Parse( spl[1] );
                        break;
                    case "DepthBPNum":
                        depth_bp_num = int.Parse( spl[1] );
                        break;
                    case "DepthBPX":
                        tmpDepthBPX = spl[1];
                        break;
                    case "DepthBPY":
                        tmpDepthBPY = spl[1];
                        break;
                    case "StartRate":
                        m_start_rate = int.Parse( spl[1] );
                        break;
                    case "RateBPNum":
                        rate_bp_num = int.Parse( spl[1] );
                        break;
                    case "RateBPX":
                        tmpRateBPX = spl[1];
                        break;
                    case "RateBPY":
                        tmpRateBPY = spl[1];
                        break;
                    case "L0":
                        m_type = VsqHandleType.Lyric;
                        m_lyric = new Lyric( spl[1] );
                        break;
                }
                if ( sr.Peek() < 0 ) {
                    break;
                }
                last_line = sr.ReadLine();
            }
            if ( m_ids != "normal" ) {
                Type = VsqHandleType.Singer;
            } else if ( m_icon_id != "" ) {
                Type = VsqHandleType.Vibrato;
            } else {
                Type = VsqHandleType.Lyric;
            }

            // RateBPX, RateBPYの設定
            if ( this.Type == VsqHandleType.Vibrato ) {
                if ( rate_bp_num > 0 ) {
                    float[] rate_bp_x = new float[rate_bp_num];
                    spl2 = tmpRateBPX.Split( new char[] { ',' } );
                    for ( int i = 0; i < rate_bp_num; i++ ) {
                        rate_bp_x[i] = float.Parse( spl2[i] );
                    }

                    int[] rate_bp_y = new int[rate_bp_num];
                    spl2 = tmpRateBPY.Split( new char[] { ',' } );
                    for ( int i = 0; i < rate_bp_num; i++ ) {
                        rate_bp_y[i] = int.Parse( spl2[i] );
                    }
                    m_rate_bp = new VibratoBPList( rate_bp_x, rate_bp_y );
                } else {
                    //m_rate_bp_x = null;
                    //m_rate_bp_y = null;
                    m_rate_bp = new VibratoBPList();
                }

                // DepthBPX, DepthBPYの設定
                if ( depth_bp_num > 0 ) {
                    float[] depth_bp_x = new float[depth_bp_num];
                    spl2 = tmpDepthBPX.Split( new char[] { ',' } );
                    for ( int i = 0; i < depth_bp_num; i++ ) {
                        depth_bp_x[i] = float.Parse( spl2[i] );
                    }

                    int[] depth_bp_y = new int[depth_bp_num];
                    spl2 = tmpDepthBPY.Split( new char[] { ',' } );
                    for ( int i = 0; i < depth_bp_num; i++ ) {
                        depth_bp_y[i] = int.Parse( spl2[i] );
                    }
                    m_depth_bp = new VibratoBPList( depth_bp_x, depth_bp_y );
                } else {
                    m_depth_bp = new VibratoBPList();
                    //m_depth_bp_x = null;
                    //m_depth_bp_y = null;
                }
            } else {
                m_depth_bp = new VibratoBPList();
                m_rate_bp = new VibratoBPList();
            }
        }

        /// <summary>
        /// ハンドル指定子（例えば"h#0123"という文字列）からハンドル番号を取得します
        /// </summary>
        /// <param name="_string">ハンドル指定子</param>
        /// <returns>ハンドル番号</returns>
        public static int HandleIndexFromString( string _string ) {
            string[] spl = _string.Split( new char[] { '#' } );
            return int.Parse( spl[1] );
        }


        /// <summary>
        /// インスタンスをテキストファイルに出力します
        /// </summary>
        /// <param name="sw">出力先</param>
        public void Print( StreamWriter sw ) {
            string result = this.ToString();
            sw.WriteLine( result );
        }


        /// <summary>
        /// インスタンスをコンソール画面に出力します
        /// </summary>
        private void Print() {
            string result = this.ToString();
            Console.WriteLine( result );
        }


        /// <summary>
        /// インスタンスを文字列に変換します
        /// </summary>
        /// <param name="encode">2バイト文字をエンコードするか否かを指定するフラグ</param>
        /// <returns>インスタンスを変換した文字列</returns>
        public string ToString( bool encode ) {
            string result = "";
            result += "[h#" + Index.ToString( "0000" ) + "]";
            switch ( Type ) {
                case VsqHandleType.Lyric:
                    result += Environment.NewLine + "L0=" + m_lyric.ToString( encode );
                    break;
                case VsqHandleType.Vibrato:
                    result += Environment.NewLine + "IconID=" + m_icon_id + Environment.NewLine;
                    result += "IDS=" + m_ids + Environment.NewLine;
                    result += "Original=" + m_original + Environment.NewLine;
                    result += "Caption=" + m_caption + Environment.NewLine;
                    result += "Length=" + m_length + Environment.NewLine;
                    result += "StartDepth=" + m_start_depth + Environment.NewLine;
                    result += "DepthBPNum=" + m_depth_bp.Num + Environment.NewLine;
                    if ( m_depth_bp.Num > 0 ) {
                        result += "DepthBPX=" + m_depth_bp[0].X.ToString( "0.000000" );
                        for ( int i = 1; i < m_depth_bp.Num; i++ ) {
                            result += "," + m_depth_bp[i].X.ToString( "0.000000" );
                        }
                        result += Environment.NewLine + "DepthBPY=" + m_depth_bp[0].Y;
                        for ( int i = 1; i < m_depth_bp.Num; i++ ) {
                            result += "," + m_depth_bp[i].Y;
                        }
                        result += Environment.NewLine;
                    }
                    result += "StartRate=" + m_start_rate + Environment.NewLine;
                    result += "RateBPNum=" + m_rate_bp.Num;
                    if ( m_rate_bp.Num > 0 ) {
                        result += Environment.NewLine + "RateBPX=" + m_rate_bp[0].X.ToString( "0.000000" );
                        for ( int i = 1; i < m_rate_bp.Num; i++ ) {
                            result += "," + m_rate_bp[i].X.ToString( "0.000000" );
                        }
                        result += Environment.NewLine + "RateBPY=" + m_rate_bp[0].Y;
                        for ( int i = 1; i < m_rate_bp.Num; i++ ) {
                            result += "," + m_rate_bp[i].Y;
                        }
                    }
                    break;
                case VsqHandleType.Singer:
                    result += Environment.NewLine + "IconID=" + m_icon_id + Environment.NewLine;
                    result += "IDS=" + m_ids + Environment.NewLine;
                    result += "Original=" + m_original + Environment.NewLine;
                    result += "Caption=" + m_caption + Environment.NewLine;
                    result += "Length=" + m_length + Environment.NewLine;
                    result += "Language=" + m_language + Environment.NewLine;
                    result += "Program=" + m_program;
                    break;
                default:
                    break;
            }
            return result;

        }

    }

    class IconHandle : VsqHandle, ICloneable {
        public object Clone() {
            IconHandle ret = new IconHandle();
            ret.Caption = m_caption;
            ret.IconID = m_icon_id;
            ret.IDS = m_ids;
            ret.Index = Index;
            ret.Language = Language;
            ret.Length = Length;
            ret.Original = Original;
            ret.Program = Program;
            ret.Type = Type;
            return ret;
        }


        public int Program {
            get {
                return m_program;
            }
            set {
                m_program = value;
            }
        }


        public int Language {
            get {
                return m_language;
            }
            set {
                m_language = value;
            }
        }


        public int Length {
            get {
                return m_length;
            }
            set {
                m_length = value;
            }
        }


        public string Caption {
            get {
                return m_caption;
            }
            set {
                m_caption = value;
            }
        }


        public string IconID {
            get {
                return m_icon_id;
            }
            set {
                m_icon_id = value;
            }
        }


        public string IDS {
            get {
                return m_ids;
            }
            set {
                m_ids = value;
            }
        }


        public int Original {
            get {
                return m_original;
            }
            set {
                m_original = value;
            }
        }
    }


    [Serializable]
    public class LyricHandle : VsqHandle, ICloneable {
        internal LyricHandle() {
        }


        /// <summary>
        /// type = Lyric用のhandleのコンストラクタ
        /// </summary>
        /// <param name="phrase">歌詞</param>
        /// <param name="phonetic_symbol">発音記号</param>
        public LyricHandle( string phrase, string phonetic_symbol ) {
            Type = VsqHandleType.Lyric;
            m_lyric = new Lyric( phrase, phonetic_symbol );
        }

        
        public Lyric L0 {
            get {
                return m_lyric;
            }
            set {
                m_lyric = value;
            }
        }


        public object Clone() {
            LyricHandle ret = new LyricHandle();
            ret.Type = Type;
            ret.Index = Index;
            ret.m_lyric = (Lyric)m_lyric.Clone();
            return ret;
        }
    }


    [Serializable]
    public class VibratoHandle : VsqHandle, ICloneable {
        public VibratoHandle() : base() {
            m_rate_bp = new VibratoBPList();
            m_depth_bp = new VibratoBPList();
        }

        public object Clone() {
            VibratoHandle result = new VibratoHandle();
            result.Type = Type;
            result.Index = Index;
            result.IconID = IconID;
            result.IDS = this.IDS;
            result.Original = this.Original;
            result.Caption = this.Caption;
            result.Length = this.Length;
            result.StartDepth = this.StartDepth;
            /*result.DepthBPNum = this.DepthBPNum;
            if ( this.DepthBPX != null ) {
                result.DepthBPX = (float[])this.DepthBPX.Clone();
            }
            if ( this.DepthBPY != null ) {
                result.DepthBPY = (int[])this.DepthBPY.Clone();
            }*/
            result.m_depth_bp = (VibratoBPList)m_depth_bp.Clone();
            result.StartRate = this.StartRate;
            /*result.RateBPNum = this.RateBPNum;
            if ( this.RateBPX != null ) {
                result.RateBPX = (float[])this.RateBPX.Clone();
            }
            if ( this.RateBPY != null ) {
                result.RateBPY = (int[])this.RateBPY.Clone();
            }*/
            result.m_rate_bp = (VibratoBPList)m_rate_bp.Clone();
            return result;
        }


        public int Original {
            get {
                return m_original;
            }
            set {
                m_original = value;
            }
        }


        public int Length {
            get {
                return m_length;
            }
            set {
                m_length = value;
            }
        }


        public string Caption {
            get {
                return m_caption;
            }
            set {
                m_caption = value;
            }
        }


        public string IDS {
            get {
                return m_ids;
            }
            set {
                m_ids = value;
            }
        }


        public string IconID {
            get {
                return m_icon_id;
            }
            set {
                m_icon_id = value;
            }
        }


        public int StartDepth {
            get {
                return m_start_depth;
            }
            set {
                m_start_depth = value;
            }
        }


        public int StartRate {
            get {
                return m_start_rate;
            }
            set {
                m_start_rate = value;
            }
        }

        public VibratoBPList DepthBP {
            get {
                return m_depth_bp;
            }
            set {
                m_depth_bp = value;
            }
        }

        public VibratoBPList RateBP {
            get {
                return m_rate_bp;
            }
            set {
                m_rate_bp = value;
            }
        }

        /*public int DepthBPNum {
            get {
                return m_depth_bp_num;
            }
            set {
                m_depth_bp_num = value;
            }
        }


        public int RateBPNum {
            get {
                return m_rate_bp_num;
            }
            set {
                m_rate_bp_num = value;
            }
        }


        public float[] DepthBPX {
            get {
                return m_depth_bp_x;
            }
            set {
                m_depth_bp_x = value;
            }
        }


        public int[] DepthBPY {
            get {
                return m_depth_bp_y;
            }
            set {
                m_depth_bp_y = value;
            }
        }


        public float[] RateBPX {
            get {
                return m_rate_bp_x;
            }
            set {
                m_rate_bp_x = value;
            }
        }


        public int[] RateBPY {
            get {
                return m_rate_bp_y;
            }
            set {
                m_rate_bp_y = value;
            }
        }*/
    }


}
