﻿/*
 * VSTiProxy.cs
 * Copyright (c) 2008-2009 kbinani
 *
 * This file is part of Boare.Cadencii.
 *
 * Boare.Cadencii is free software; you can redistribute it and/or
 * modify it under the terms of the BSD License.
 *
 * Boare.Cadencii is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 */
#define TEST
//#define UNMANAGED_VSTIDRV
using System;
using System.IO;
using System.Runtime.InteropServices;
using System.Collections.Generic;
using System.Threading;
using System.ComponentModel;
using System.Security;
using System.Windows.Forms;

using Boare.Lib.Vsq;
using Boare.Lib.Media;

namespace Boare.Cadencii {

    public enum VSTiProxyTrackType {
        TempoTrack,
        MainTrack,
    }

    [StructLayout( LayoutKind.Sequential, Pack = 1 )]
    struct VstEvent : IComparable<VstEvent> {
        public Int32 type;
        public Int32 byteSize;
        public Int32 deltaFrames;
        public Int32 flags;
        byte data0;
        byte data1;
        byte data2;
        byte data3;
        byte data4;
        byte data5;
        byte data6;
        byte data7;
        byte data8;
        byte data9;
        byte data10;
        byte data11;
        byte data12;
        byte data13;
        byte data14;
        byte data15;

        public int CompareTo( VstEvent item ) {
            return deltaFrames - item.deltaFrames;
        }

        public byte this[int index] {
            get {
                switch ( index ) {
                    case 0:
                        return data0;
                    case 1:
                        return data1;
                    case 2:
                        return data2;
                    case 3:
                        return data3;
                    case 4:
                        return data4;
                    case 5:
                        return data5;
                    case 6:
                        return data6;
                    case 7:
                        return data7;
                    case 8:
                        return data8;
                    case 9:
                        return data9;
                    case 10:
                        return data10;
                    case 11:
                        return data11;
                    case 12:
                        return data12;
                    case 13:
                        return data13;
                    case 14:
                        return data14;
                    case 15:
                        return data15;
                    default:
                        throw new ArgumentOutOfRangeException();
                }
            }
            set {
                switch ( index ) {
                    case 0:
                        data0 = value;
                        break;
                    case 1:
                        data1 = value;
                        break;
                    case 2:
                        data2 = value;
                        break;
                    case 3:
                        data3 = value;
                        break;
                    case 4:
                        data4 = value;
                        break;
                    case 5:
                        data5 = value;
                        break;
                    case 6:
                        data6 = value;
                        break;
                    case 7:
                        data7 = value;
                        break;
                    case 8:
                        data8 = value;
                        break;
                    case 9:
                        data9 = value;
                        break;
                    case 10:
                        data10 = value;
                        break;
                    case 11:
                        data11 = value;
                        break;
                    case 12:
                        data12 = value;
                        break;
                    case 13:
                        data13 = value;
                        break;
                    case 14:
                        data14 = value;
                        break;
                    case 15:
                        data15 = value;
                        break;
                    default:
                        throw new ArgumentOutOfRangeException();
                }
            }
        }
    }

    unsafe struct MIDI_EVENT {
        public uint clock;
        public MIDI_EVENT* pNext;
        public uint dwDataSize;
        public byte dwOffset;
        public byte* pMidiEvent;
    }

    public static class VSTiProxy {
        private const int _SAMPLE_RATE = 44100;
        private const int _BLOCK_SIZE = 44100;

        private static string s_dll_path = "";
        public static string CurrentUser = "";
        private static bool s_loaded = false;
        private static WaveWriter s_wave;
        private static bool s_rendering = false;
        private static int s_trim_remain = 0;
        private static object s_locker;
        private static double s_amplify_left = 1.0;
        private static double s_amplify_right = 1.0;
#if !UNMANAGED_VSTIDRV
        private static vstildr vstidrv;
#endif
        private static FirstBufferWrittenEventHandler s_first_buffer_written_event_handler;

        private class StartRenderArg {
            public NrpnData[] nrpn;
            public TempoTableEntry[] tempo;
            public double amplify_left;
            public double amplify_right;
            public int trim_msec;
            public long total_samples;
            public string[] files;
            public double wave_read_offset_seconds;
            public bool mode_infinite;
        }

        static VSTiProxy() {
#if DEBUG
            Common.DebugWriteLine( "VSTiProxy..cctor" );
#endif
            s_locker = new object();
            s_dll_path = VsqUtil.VstiDllPath;
            if ( Environment.OSVersion.Platform == PlatformID.Unix ) {
                return;
            }
#if !UNMANAGED_VSTIDRV
            try {
                string driver = Path.Combine( Application.StartupPath, "vstidrv.dll" );
                System.Reflection.Assembly asm = System.Reflection.Assembly.LoadFile( driver );
                Type[] types = asm.GetTypes();
                string driver_fullname = "";
                foreach ( Type t in types ) {
                    if ( t.IsClass && t.IsPublic && !t.IsAbstract ) {
                        try {
                            Type intfc = t.GetInterface( typeof( vstildr ).FullName );
                            if ( intfc != null ) {
                                driver_fullname = t.FullName;
                                break;
                            }
                        } catch {
                        }
                    }
                }
#if DEBUG
                Common.DebugWriteLine( "    driver_fullname=" + driver_fullname );
                bocoree.debug.push_log( "driver_fullname=" + driver_fullname );
#endif
                if ( driver_fullname != "" ) {
                    vstidrv = (vstildr)asm.CreateInstance( driver_fullname );
                } else {
                    vstidrv = null;
                    return;
                }
            } catch ( Exception ex ){
#if DEBUG
                Common.DebugWriteLine( "    ex=" + ex );
                bocoree.debug.push_log( "    ex=" + ex );
#endif
            }
#endif // !UNMANAGED_VSTIDRV

            try {
                char[] str = s_dll_path.ToCharArray();
                if ( s_dll_path != "" ) {
                    s_loaded = vstidrv.Init( str, _BLOCK_SIZE, _SAMPLE_RATE );
                    s_first_buffer_written_event_handler = new FirstBufferWrittenEventHandler( HandleFirstBufferWrittenEvent );
                    vstidrv.SetFirstBufferWrittenCallback( s_first_buffer_written_event_handler );
                } else {
                    s_loaded = false;
                }
#if DEBUG
#if TEST
                bocoree.debug.push_log( "VSTiProxy..cctor()" );
                bocoree.debug.push_log( "    s_dll_path=" + s_dll_path );
                bocoree.debug.push_log( "    s_loaded=" + s_loaded );
#endif
#endif
            } catch ( Exception ex ) {
#if TEST
                bocoree.debug.push_log( "    ex=" + ex );
#endif
            }
        }

        public static int JoyGetNumJoyDev() {
            int ret = 0;
#if UNMANAGED_VSTIDRV
            try {
                ret = vstidrv.JoyGetNumJoyDev();
            } catch {
            }
#else
            if ( vstidrv != null ) {
                try {
                    ret = vstidrv.JoyGetNumJoyDev();
                } catch {
                    ret = 0;
                }
            }
#endif
            return ret;
        }

        public static bool JoyGetStatus( int index, out byte[] buttons, out int pov ) {
            bool ret = false;
            buttons = new byte[0];
            pov = -1;
#if UNMANAGED_VSTIDRV
            try {
                ret = vstidrv.JoyGetStatus( index, out buttons, out pov );
            } catch {
                ret = false;
            }
#else
            if ( vstidrv != null ) {
                try {
                    ret = vstidrv.JoyGetStatus( index, out buttons, out pov );
                } catch {
                    ret = false;
                }
            }
#endif
            return ret;
        }

        public static event FirstBufferWrittenEventHandler FirstBufferWritten;

        private static void HandleFirstBufferWrittenEvent() {
#if DEBUG
            Common.DebugWriteLine( "VSTiProxy.HandleFirstBufferWrittenEvent" );
#endif
            if ( FirstBufferWritten != null ) {
                FirstBufferWritten();
            }
        }

        public static int JoyInit() {
            int ret = 0;
#if UNMANAGED_VSTIDRV
            try {
                ret = vstidrv.JoyInit();
            } catch {
                ret = 0;
            }
#else
            if ( vstidrv != null ) {
                try {
                    ret = vstidrv.JoyInit();
                } catch {
                    ret = 0;
                }
            }
#endif
            return ret;
        }

        public static int SampleRate {
            get {
                return _SAMPLE_RATE;
            }
        }

        public static void Terminate() {
            if ( s_loaded ) {
#if UNMANAGED_VSTIDRV
                vstidrv.Terminate();
#else
                if ( vstidrv != null ) {
                    vstidrv.Terminate();
                }
#endif
            }
        }

        /// <summary>
        /// 
        /// </summary>
        /// <param name="vsq">レンダリング元</param>
        /// <param name="track">レンダリング対象のトラック番号</param>
        /// <param name="file">レンダリング結果を出力するファイル名。空文字ならファイルには出力されない</param>
        /// <param name="start_sec">レンダリングの開始位置</param>
        /// <param name="end_sec">レンダリングの終了位置</param>
        /// <param name="amplify_left">左チャンネルの増幅率</param>
        /// <param name="amplify_right">右チャンネルの増幅率</param>
        /// <param name="ms_presend">プリセンドするミリセカンド</param>
        /// <param name="direct_play">レンダリングと同時に再生するかどうか</param>
        /// <param name="files">レンダリング結果と同時に再生するWAVEファイルのリスト</param>
        /// <param name="wave_read_offset_seconds">filesに指定したファイルの先頭から読み飛ばす秒数</param>
        public static void Render(
            VsqFile vsq,
            int track,
            string file,
            double start_sec,
            double end_sec,
            double amplify_left,
            double amplify_right,
            int ms_presend,
            bool direct_play,
            string[] files,
            double wave_read_offset_seconds,
            bool mode_infinite
        ) {
#if TEST
            Common.DebugWriteLine( "VSTiProxy+Render" );
            Common.DebugWriteLine( "    start_sec,end_sec=" + start_sec + "," + end_sec );
#endif
            VsqFile split = (VsqFile)vsq.Clone();
            split.UpdateTotalClocks();
            int clock_start = (int)vsq.ClockFromSec( start_sec );
            int clock_end = (int)vsq.ClockFromSec( end_sec );

            if ( clock_end < vsq.TotalClocks ) {
                VsqFile.RemovePart( split, clock_end, split.TotalClocks + 480 );
            }

            int extra_note_clock = (int)vsq.ClockFromSec( (float)end_sec + 10.0f );
            int extra_note_clock_end = (int)vsq.ClockFromSec( (float)end_sec + 10.0f + 3.1f ); //ブロックサイズが1秒分で、バッファの個数が3だから +3.1f。0.1fは安全のため。
            VsqEvent extra_note = new VsqEvent( extra_note_clock, new VsqID( 0 ) );
            extra_note.ID.type = VsqIDType.Anote;
            extra_note.ID.Note = 60;
            extra_note.ID.Length = extra_note_clock_end - extra_note_clock;
            extra_note.ID.VibratoHandle = null;
            extra_note.ID.LyricHandle = new LyricHandle( "a", "a" );
            split.Tracks[track].Events.Add( extra_note );

            double trim_sec = 0.0; // レンダリング結果から省かなければならない秒数。
            if ( clock_start < split.PreMeasureClocks ) {
                trim_sec = split.SecFromClock( clock_start );
            } else {
                VsqFile.RemovePart( split, vsq.PreMeasureClocks, clock_start );
                trim_sec = split.SecFromClock( split.PreMeasureClocks );
            }
            split.UpdateTotalClocks();
#if DEBUG
            split.Write( "split.vsq" );
#endif
            VsqNrpn[] nrpn = VsqFile.GenerateNRPN( split, track, ms_presend );
            NrpnData[] nrpn_data = VsqNrpn.Convert( nrpn );
#if DEBUG
            Common.DebugWriteLine( "trim_sec=" + trim_sec );
            using ( StreamWriter sw = new StreamWriter( "nrpn.txt" ) ) {
                byte msb = 0, lsb = 0, dmsb = 0, dlsb = 0;
                sw.WriteLine( "clock\t\tparam.\t\tvalue" );
                for ( int i = 0; i < nrpn_data.Length; i++ ) {
                    switch ( nrpn_data[i].Parameter ) {
                        case 0x63:
                            msb = nrpn_data[i].Value;
                            break;
                        case 0x62:
                            lsb = nrpn_data[i].Value;
                            break;
                        case 0x06:
                            dmsb = nrpn_data[i].Value;
                            break;
                        case 0x26:
                            dlsb = nrpn_data[i].Value;
                            break;
                    }
                    if ( nrpn_data[i].Parameter == 0x62 ) {
                        NRPN foo = NRPN.CC_BS_DELAY;
                        ushort val = (ushort)(msb << 8 | lsb);
                        foreach ( NRPN n in Enum.GetValues( typeof( NRPN ) ) ) {
                            if ( val == (ushort)n ) {
                                foo = n;
                                break;
                            }
                        }
                        sw.WriteLine( nrpn_data[i].Clock + "\t\t0x" + string.Format( "{0:X02}", nrpn_data[i].Parameter ) + "\t\t0x" + string.Format( "{0:X02}", nrpn_data[i].Value ) + "\t" + foo.ToString() );
                    } else {
                        sw.WriteLine( nrpn_data[i].Clock + "\t\t0x" + string.Format( "{0:X02}", nrpn_data[i].Parameter ) + "\t\t0x" + string.Format( "{0:X02}", nrpn_data[i].Value ) );
                    }
                }
            }
#endif
            long total_samples = (long)((end_sec - start_sec) * _SAMPLE_RATE);
            int trim_msec = (int)(trim_sec * 1000.0);
            if ( direct_play ) {
                StartRenderArg sra = new StartRenderArg();
                sra.nrpn = nrpn_data;
                sra.tempo = split.TempoTable.ToArray();
                sra.amplify_left = amplify_left;
                sra.amplify_right = amplify_right;
                sra.trim_msec = trim_msec;
                sra.total_samples = total_samples;
                sra.files = files;
                sra.wave_read_offset_seconds = wave_read_offset_seconds;
                sra.mode_infinite = mode_infinite;

                Thread thread = new Thread( new ParameterizedThreadStart( RenderWithDirectPlay ) );
                thread.Priority = ThreadPriority.BelowNormal;
                thread.Start( sra );
            } else {
                RenderCor( nrpn_data,
                           split.TempoTable.ToArray(),
                           file,
                           amplify_left,
                           amplify_right,
                           direct_play,
                           trim_msec,
                           total_samples,
                           files,
                           wave_read_offset_seconds,
                           mode_infinite );
            }
        }

        private static void RenderWithDirectPlay( object argument ) {
            StartRenderArg sra = (StartRenderArg)argument;
#if DEBUG
            RenderCor( sra.nrpn, 
                       sra.tempo,
                       "last.wav",
                       sra.amplify_left,
                       sra.amplify_right, 
                       true, 
                       sra.trim_msec,
                       sra.total_samples,
                       sra.files,
                       sra.wave_read_offset_seconds,
                       sra.mode_infinite );
#else
            RenderCor( sra.nrpn,
                       sra.tempo,
                       "",
                       sra.amplify_left,
                       sra.amplify_right,
                       true,
                       sra.trim_msec,
                       sra.total_samples,
                       sra.files,
                       sra.wave_read_offset_seconds,
                       sra.mode_infinite );
#endif
        }

        public static void WaveOutReset() {
#if UNMANAGED_VSTIDRV
            vstidrv.WaveOutReset();
#else
            if ( vstidrv != null ) {
                vstidrv.WaveOutReset();
            }
#endif
        }

        private static unsafe void RenderCor(
            NrpnData[] nrpn,
            TempoTableEntry[] tempo,
            string file,
            double amplify_left,
            double amplify_right,
            bool direct_play,
            int presend_msec,
            long total_samples,
            string[] files,
            double wave_read_offset_seconds,
            bool mode_infinite
        ) {
#if TEST
            Console.WriteLine( "VSTiProxy.RenderToWave( NRPN[], TempoTableEntry[], int, string )" );
            bocoree.debug.push_log( "VSTiPRoxy+RenderToWave(NrpnData[], TempoTableEntry[], int, string)" );
#endif
            if ( !s_loaded ) {
                return;
            }
#if !UNMANAGED_VSTIDRV
            if ( vstidrv == null ) {
                return;
            }
#endif
            int tempo_count = tempo.Length;
            float first_tempo = 125.0f;
            if ( tempo.Length > 0 ) {
                first_tempo = (float)(60e6 / (double)tempo[0].Tempo);
            }
            byte[] masterEventsSrc = new byte[tempo_count * 3];
            int[] masterClocksSrc = new int[tempo_count];
            int count = -3;
            for ( int i = 0; i < tempo.Length; i++ ) {
                count += 3;
                masterClocksSrc[i] = tempo[i].Clock;
                byte b0 = (byte)(tempo[i].Tempo >> 16);
                uint u0 = (uint)(tempo[i].Tempo - (b0 << 16));
                byte b1 = (byte)(u0 >> 8);
                byte b2 = (byte)(u0 - (u0 << 8));
#if DEBUG
                Console.WriteLine( "    b0-b2=0x" + Convert.ToString( b0, 16 ) + ", 0x" + Convert.ToString( b1, 16 ) + ", 0x" + Convert.ToString( b2, 16 ) );
#endif
                masterEventsSrc[count] = b0;
                masterEventsSrc[count + 1] = b1;
                masterEventsSrc[count + 2] = b2;
            }
            fixed ( byte* masterEvents = &masterEventsSrc[0] )
            fixed ( int* masterClocks = &masterClocksSrc[0] ) {
                vstidrv.SendEvent( masterEvents, masterClocks, tempo_count, (int)VSTiProxyTrackType.TempoTrack );
            }

            int numEvents = nrpn.Length;
            byte[] bodyEventsSrc = new byte[numEvents * 3];
            int[] bodyClocksSrc = new int[numEvents];
            count = -3;
            int last_clock = 0;
            for ( int i = 0; i < numEvents; i++ ) {
                count += 3;
                bodyEventsSrc[count] = 0xb0;
                bodyEventsSrc[count + 1] = nrpn[i].Parameter;
                bodyEventsSrc[count + 2] = nrpn[i].Value;
                bodyClocksSrc[i] = nrpn[i].Clock;
                last_clock = nrpn[i].Clock;
            }

            int index = tempo_count - 1;
            for ( int i = tempo_count - 1; i >= 0; i-- ) {
                if ( tempo[i].Clock < last_clock ) {
                    index = i;
                    break;
                }
            }
            int last_tempo = tempo[index].Tempo;
            int trim_remain = GetErrorSamples( first_tempo ) + (int)(presend_msec / 1000.0 * _SAMPLE_RATE);
            s_trim_remain = trim_remain;

#if DEBUG
            Common.DebugWriteLine( "    s_trim_remain=" + s_trim_remain );
#endif
            fixed ( byte* bodyEvents = &bodyEventsSrc[0] )
            fixed ( int* bodyClocks = &bodyClocksSrc[0] ) {
                vstidrv.SendEvent( bodyEvents, bodyClocks, numEvents, (int)VSTiProxyTrackType.MainTrack );
            }

            s_rendering = true;
            if ( file != "" ) {
                s_wave = new WaveWriter( file, WaveChannel.Stereo, 16, _SAMPLE_RATE );
                if ( s_trim_remain < 0 ) {
                    double[] d = new double[-s_trim_remain];
                    for ( int i = 0; i < -s_trim_remain; i++ ) {
                        d[i] = 0.0;
                    }
                    s_wave.Append( d, d );
                    s_trim_remain = 0;
                }
            }
#if DEBUG
            bocoree.debug.push_log( "    first_tempo=" + first_tempo );
            bocoree.debug.push_log( "    s_trim_remain=" + s_trim_remain );
#endif

            s_amplify_left = amplify_left;
            s_amplify_right = amplify_right;
            if ( file != "" ) {
                vstidrv.WaveIncoming += vstidrv_WaveIncoming;
            }
            vstidrv.RenderingFinished += vstidrv_RenderingFinished;
            vstidrv.StartRendering( total_samples, 1.0, 1.0, trim_remain, (file != ""), direct_play, files, wave_read_offset_seconds, mode_infinite );
            while ( s_rendering ) {
                Application.DoEvents();
            }
            s_rendering = false;
            if ( file != "" ) {
                vstidrv.WaveIncoming -= vstidrv_WaveIncoming;
                s_wave.Close();
            }
            vstidrv.RenderingFinished -= vstidrv_RenderingFinished;
            s_wave = null;
        }

        static void vstidrv_RenderingFinished() {
            s_rendering = false;
        }

        static void vstidrv_WaveIncoming( double[] L, double[] R ) {
            lock ( s_locker ) {
                if ( s_trim_remain > 0 ) {
                    if ( s_trim_remain >= L.Length ) {
                        s_trim_remain -= L.Length;
                        return;
                    }
                    int actual_append = L.Length - s_trim_remain;
                    double[] dL = new double[actual_append];
                    double[] dR = new double[actual_append];
                    for ( int i = 0; i < actual_append; i++ ) {
                        dL[i] = L[i + s_trim_remain] * s_amplify_left;
                        dR[i] = R[i + s_trim_remain] * s_amplify_right;
                    }
                    s_wave.Append( dL, dR );
                    dL = null;
                    dR = null;
                    s_trim_remain = 0;
                } else {
                    for ( int i = 0; i < L.Length; i++ ) {
                        L[i] = L[i] * s_amplify_left;
                    }
                    for ( int i = 0; i < R.Length; i++ ) {
                        R[i] = R[i] * s_amplify_right;
                    }
                    s_wave.Append( L, R );
                }
            }
        }

        public static double GetProgress() {
            if ( s_loaded ) {
#if UNMANAGED_VSTIDRV
                return vstidrv.GetProgress();
#else
                if ( vstidrv != null ) {
                    return vstidrv.GetProgress();
                }
#endif
            }
            return 0.0;
        }

        public static void AbortRendering() {
            if ( s_rendering ) {
                s_rendering = false;
            }
#if UNMANAGED_VSTIDRV
            vstidrv.AbortRendering();
#else
            if ( vstidrv != null ) {
                vstidrv.AbortRendering();
            }
#endif
        }

        public static int GetErrorSamples( float tempo ) {
            const float a0 = -17317.563f;
            const float a1 = 86.7312112f;
            const float a2 = -0.237323499f;
            if ( tempo <= 240 ) {
                return 4666;
            } else {
                float x = tempo - 240;
                return (int)((a2 * x + a1) * x + a0);
            }
        }

        public static float GetPlayTime() {
#if UNMANAGED_VSTIDRV
            return vstidrv.GetPlayTime();
#else
            if ( vstidrv != null ) {
                return vstidrv.GetPlayTime();
            } else {
                return 0.0f;
            }
#endif
        }

#if UNMANAGED_VSTIDRV
        public delegate void WaveIncomingEventHandler( double[] L, double[] R );
        public delegate void FirstBufferWrittenEventHandler();
        public delegate void RenderingFinishedEventHandler();

        public static unsafe class vstidrv {
            private delegate void __WaveIncomingCallback( double* L, double* R, int length );
            private delegate void __FirstBufferWrittenCallback();
            private delegate void __RenderingFinishedCallback();

            private volatile static __WaveIncomingCallback s_wave_incoming_callback;
            private volatile static __FirstBufferWrittenCallback s_first_buffer_written_callback;
            private volatile static __RenderingFinishedCallback s_rendering_finished_callback;

            private volatile static FirstBufferWrittenEventHandler s_first_buffer_written_callback_body;

            public static event WaveIncomingEventHandler WaveIncoming;
            //public static event FirstBufferWrittenEventHandler FirstBufferWritten;
            public static event RenderingFinishedEventHandler RenderingFinished;

            static vstidrv() {
                s_wave_incoming_callback = new __WaveIncomingCallback( HandleWaveIncomingCallback );
                s_first_buffer_written_callback = new __FirstBufferWrittenCallback( HandleFirstBufferWrittenCallback );
                s_rendering_finished_callback = new __RenderingFinishedCallback( HandleRenderingFinishedCallback );
                try {
                    vstidrv_setFirstBufferWrittenCallback( s_first_buffer_written_callback );
                    vstidrv_setWaveIncomingCallback( s_wave_incoming_callback );
                    vstidrv_setRenderingFinishedCallback( s_rendering_finished_callback );
                } catch ( Exception ex ) {
#if DEBUG
                Common.DebugWriteLine( "vstidrv.cctor()" );
                Common.DebugWriteLine( "    ex=" + ex );
#endif
                }
            }

            public static void SetFirstBufferWrittenCallback( FirstBufferWrittenEventHandler handler ) {
                s_first_buffer_written_callback_body = handler;
            }

            public static void Terminate() {
                try {
                    vstidrv_Terminate();
                } catch {
                }
            }

            private static void HandleWaveIncomingCallback( double* L, double* R, int length ) {
                if ( WaveIncoming != null ) {
                    double[] ret_l = new double[length];
                    double[] ret_r = new double[length];
                    for ( int i = 0; i < length; i++ ) {
                        ret_l[i] = L[i];
                        ret_r[i] = R[i];
                    }
                    WaveIncoming( ret_l, ret_r );
                }
            }

            private static void HandleFirstBufferWrittenCallback() {
#if DEBUG
            Common.DebugWriteLine( "vstidrv+HandleFirstBufferWrittenCallback" );
#endif
                if ( s_first_buffer_written_callback_body != null ) {
                    s_first_buffer_written_callback_body();
                }
            }

            private static void HandleRenderingFinishedCallback() {
                if ( RenderingFinished != null ) {
                    RenderingFinished();
                }
            }

            public static bool Init( char[] dll_path, int block_size, int sample_rate ) {
                bool ret = false;
                try {
                    byte[] b_dll_path = new byte[dll_path.Length + 1];
                    for ( int i = 0; i < dll_path.Length; i++ ) {
                        b_dll_path[i] = (byte)dll_path[i];
                    }
                    b_dll_path[dll_path.Length] = 0x0;
                    fixed ( byte* ptr = &b_dll_path[0] ) {
                        ret = vstidrv_Init( ptr, block_size, sample_rate );
                    }
                } catch {
                    ret = false;
                }
                return ret;
            }

            public static int SendEvent( byte* src, int* deltaFrames, int numEvents, int targetTrack ) {
                int ret = 0;
                try {
                    ret = vstidrv_SendEvent( src, deltaFrames, numEvents, targetTrack );
                } catch {
                    ret = -1;
                }
                return ret;
            }

            public static int StartRendering(
                long total_samples,
                double amplify_left,
                double amplify_right,
                int error_samples,
                bool event_enabled,
                bool direct_play_enabled,
                string[] files,
                double wave_read_offset_seconds
            ) {
                int ret = 0;
                try {
                    IntPtr intptr_files = Marshal.AllocHGlobal( sizeof( char* ) * files.Length );
                    char** ptr_files = (char**)intptr_files.ToPointer();
                    IntPtr[] cont_intptr_files = new IntPtr[files.Length];
                    for ( int i = 0; i < files.Length; i++ ) {
                        cont_intptr_files[i] = Marshal.AllocHGlobal( sizeof( char ) * (files.Length + 1) );
                        ptr_files[i] = (char*)cont_intptr_files[i].ToPointer();
                        for ( int j = 0; j < files[i].Length; j++ ) {
                            ptr_files[i][j] = files[i][j];
                        }
                        ptr_files[i][files[i].Length] = '\0';
                    }
                    ret = vstidrv_StartRendering( total_samples,
                                                  amplify_left,
                                                  amplify_right,
                                                  error_samples,
                                                  event_enabled,
                                                  direct_play_enabled,
                                                  ptr_files,
                                                  files.Length,
                                                  wave_read_offset_seconds );
                    for ( int i = 0; i < files.Length; i++ ) {
                        Marshal.FreeHGlobal( cont_intptr_files[i] );
                    }
                    Marshal.FreeHGlobal( intptr_files );
                } catch {
                    ret = -1;
                }
                return ret;
            }

            public static void AbortRendering() {
                try {
                    vstidrv_AbortRendering();
                } catch {
                }
            }

            public static double GetProgress() {
                double ret = 0.0;
                try {
                    ret = vstidrv_GetProgress();
                } catch {
                }
                return ret;
            }

            public static float GetPlayTime() {
                float ret = -1.0f;
#if DEBUG
            Common.DebugWriteLine( "vstidrv+GetPlayTime" );
#endif
                try {
                    ret = vstidrv_GetPlayTime();
                } catch ( Exception ex ) {
#if DEBUG
                Common.DebugWriteLine( "    ex=" + ex );
#endif
                    ret = -1.0f;
                }
#if DEBUG
            Common.DebugWriteLine( "    ret=" + ret );
#endif
                return ret;
            }

            public static void WaveOutReset() {
                try {
                    vstidrv_WaveOutReset();
                } catch {
                }
            }

            public static int JoyInit() {
                int ret = 0;
                try {
                    ret = vstidrv_JoyInit();
                } catch {
                    ret = 0;
                }
                return ret;
            }

            public static bool JoyGetStatus( int index, out byte[] buttons, out int pov ) {
                bool ret = false;
                try {
                    int len = vstidrv_JoyGetNumButtons( index );
                    buttons = new byte[len];
                    int src_pov;
                    fixed ( byte* ptr_buttons = &buttons[0] ) {
                        ret = vstidrv_JoyGetStatus( index, ptr_buttons, &src_pov );
                    }
                    pov = src_pov;
                } catch {
                    buttons = new byte[0];
                    pov = -1;
                    ret = false;
                }
                return ret;
            }

            public static int JoyGetNumJoyDev() {
                int ret = 0;
                try {
                    ret = vstidrv_JoyGetNumJoyDev();
                } catch {
                    ret = 0;
                }
                return ret;
            }

            //void vstidrv_setFirstBufferWrittenCallback( FirstBufferWrittenCallback proc );
            [DllImport( "vstidrv.dll" )]
            private static extern void vstidrv_setFirstBufferWrittenCallback( __FirstBufferWrittenCallback proc );

            //void vstidrv_setWaveIncomingCallback( WaveIncomingCallback proc );
            [DllImport( "vstidrv.dll" )]
            private static extern void vstidrv_setWaveIncomingCallback( __WaveIncomingCallback proc );

            //void vstidrv_setRenderingFinishedCallback( RenderingFinishedCallback proc );
            [DllImport( "vstidrv.dll" )]
            private static extern void vstidrv_setRenderingFinishedCallback( __RenderingFinishedCallback proc );

            //bool vstidrv_Init( string dll_path, int block_size, int sample_rate );
            [DllImport( "vstidrv.dll" )]
            private static extern bool vstidrv_Init( byte* dll_path, int block_size, int sample_rate );

            //int  vstidrv_SendEvent( unsigned char *src, int *deltaFrames, int numEvents, int targetTrack );
            [DllImport( "vstidrv.dll" )]
            private static extern int vstidrv_SendEvent( byte* src, int* deltaFrames, int numEvents, int targetTrack );

            //int vstidrv_StartRendering( __int64 total_samples, double amplify_left, double amplify_right, int error_samples, bool event_enabled, bool direct_play_enabled, wchar_t** files, int num_files, double wave_read_offset_seconds );
            [DllImport( "vstidrv.dll" )]
            private static extern int vstidrv_StartRendering( long total_samples,
                double amplify_left,
                double amplify_right,
                int error_samples,
                bool event_enabled,
                bool direct_play_enabled,
                char** files,
                int num_files,
                double wave_read_offset_seconds );


            //void vstidrv_AbortRendering();
            [DllImport( "vstidrv.dll" )]
            private static extern void vstidrv_AbortRendering();

            //double vstidrv_GetProgress();        
            [DllImport( "vstidrv.dll" )]
            private static extern double vstidrv_GetProgress();

            //float vstidrv_GetPlayTime();
            [DllImport( "vstidrv.dll" )]
            private static extern float vstidrv_GetPlayTime();

            //void vstidrv_WaveOutReset();
            [DllImport( "vstidrv.dll" )]
            private static extern void vstidrv_WaveOutReset();

            //void vstidrv_Terminate();
            [DllImport( "vstidrv.dll" )]
            private static extern void vstidrv_Terminate();

            //int vstidrv_JoyInit();
            [DllImport( "vstidrv.dll" )]
            private static extern int vstidrv_JoyInit();

            //bool vstidrv_JoyIsJoyAttatched( int index );
            [DllImport( "vstidrv.dll" )]
            private static extern bool vstidrv_JoyIsJoyAttatched( int index );

            //bool vstidrv_JoyGetStatus( int index, unsigned char *buttons, int *pov );
            [DllImport( "vstidrv.dll" )]
            private static extern bool vstidrv_JoyGetStatus( int index, byte* buttons, int* pov );

            //int vstidrv_JoyGetNumButtons( int index );
            [DllImport( "vstidrv.dll" )]
            private static extern int vstidrv_JoyGetNumButtons( int index );

            //void vstidrv_JoyReset();
            [DllImport( "vstidrv.dll" )]
            private static extern void vstidrv_JoyReset();

            //int vstidrv_JoyGetNumJoyDev();
            [DllImport( "vstidrv.dll" )]
            private static extern int vstidrv_JoyGetNumJoyDev();

        }
#endif
    }

}
