﻿/*
 * SymbolTable.cs
 * Copyright (C) 2008-2010 kbinani
 *
 * This file is part of Boare.Lib.Vsq.
 *
 * Boare.Lib.Vsq is free software; you can redistribute it and/or
 * modify it under the terms of the BSD License.
 *
 * Boare.Lib.Vsq is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 */
#if JAVA
package org.kbinani.vsq;

import java.util.*;
import java.io.*;
import org.kbinani.*;
#else
using System;
using System.Text;
using System.Windows.Forms;
using bocoree;
using bocoree.util;
using bocoree.io;

namespace Boare.Lib.Vsq {
    using boolean = System.Boolean;
    using Integer = System.Int32;
#endif

    /// <summary>
    /// 歌詞から発音記号列を引き当てるための辞書を表現するクラス
    /// </summary>
#if JAVA
    public class SymbolTable implements Cloneable {
#else
    public class SymbolTable : ICloneable {
#endif
        /// <summary>
        /// 辞書本体
        /// </summary>
        private TreeMap<String, SymbolTableEntry> m_dict;
        /// <summary>
        /// 辞書の名前
        /// </summary>
        private String m_name;
        /// <summary>
        /// 辞書を有効とするかどうか
        /// </summary>
        private boolean m_enabled;
        /// <summary>
        /// 英単語の分節分割などにより，この辞書を使うことによって最大いくつの発音記号列に分割されるか
        /// </summary>
        private int m_max_divisions = 1;

        #region static field
        /// <summary>
        /// 辞書のリスト（TreeMapのkeyは、辞書の優先順位）
        /// </summary>
        private static TreeMap<Integer, SymbolTable> s_table = new TreeMap<Integer, SymbolTable>();
        /// <summary>
        /// VOCALOID2のシステム辞書を読み込んだかどうか
        /// </summary>
        private static boolean s_initialized = false;
        #endregion

        #region Static Method and Property
        /// <summary>
        /// 英単語の分節分割などにより，登録されている辞書を使うことによって最大いくつの発音記号列に分割されるか
        /// </summary>
        /// <returns></returns>
        public static int getMaxDivisions() {
            int max = 1;
            for ( Iterator<Integer> itr = s_table.keySet().iterator(); itr.hasNext(); ) {
                int key = itr.next();
                SymbolTable table = s_table.get( key );
#if DEBUG
                PortUtil.println( "SymbolTable#getMaxDivisions; table.m_name=" + table.m_name + "; max=" + max + "; table.m_max_divisions=" + table.m_max_divisions );
#endif
                max = Math.Max( max, table.m_max_divisions );
            }
            return max;
        }

        /// <summary>
        /// 指定した優先順位の辞書本体を取得します
        /// </summary>
        /// <param name="index"></param>
        /// <returns></returns>
        public static SymbolTable getSymbolTable( int index ) {
            if ( !s_initialized ) {
                loadSystemDictionaries();
            }
            if ( 0 <= index && index < s_table.size() ) {
                return s_table.get( index );
            } else {
                return null;
            }
        }

        /// <summary>
        /// 指定した辞書ファイルを読み込みます。
        /// </summary>
        /// <param name="dictionary_file"></param>
        /// <param name="name"></param>
        public static void loadDictionary( String dictionary_file, String name ) {
            SymbolTable table = new SymbolTable( dictionary_file, false, true );
            table.m_name = name;
            int count = s_table.size();
            s_table.put( count, table );
        }

        /// <summary>
        /// VOCALOID2システムが使用する辞書を読み込みます。
        /// </summary>
        public static void loadSystemDictionaries() {
            if ( s_initialized ) {
                return;
            }
            int count = s_table.size();
            // 辞書フォルダからの読込み
            String editor_path = VocaloSysUtil.getEditorPath( SynthesizerType.VOCALOID2 );
            if ( editor_path != "" ) {
                String path = PortUtil.combinePath( PortUtil.getDirectoryName( editor_path ), "UDIC" );
                if ( !PortUtil.isDirectoryExists( path ) ) {
                    return;
                }
                String[] files = PortUtil.listFiles( path, "*.udc" );
                for ( int i = 0; i < files.Length; i++ ) {
                    files[i] = PortUtil.getFileName( files[i] );
#if DEBUG
                    PortUtil.println( "    files[i]=" + files[i] );
#endif
                    String dict = PortUtil.combinePath( path, files[i] );
                    s_table.put( count, new SymbolTable( dict, true, false ) );
                    count++;
                }
            }
            s_initialized = true;
        }

        /// <summary>
        /// 指定したディレクトリにある拡張辞書ファイル(拡張子*.eudc)を全て読み込みます
        /// </summary>
        /// <param name="directory"></param>
        public static void loadAllDictionaries( String directory ) {
            // 起動ディレクトリ
            int count = s_table.size();
            if ( PortUtil.isDirectoryExists( directory ) ) {
                String[] files2 = PortUtil.listFiles( directory, "*.eudc" );
                for ( int i = 0; i < files2.Length; i++ ) {
                    files2[i] = PortUtil.getFileName( files2[i] );
                    String dict = PortUtil.combinePath( directory, files2[i] );
                    s_table.put( count, new SymbolTable( dict, false, false ) );
                    count++;
                }
            }
        }

        /// <summary>
        /// 指定した歌詞から、発音記号列を引き当てます
        /// </summary>
        /// <param name="phrase"></param>
        /// <returns></returns>
        public static SymbolTableEntry attatch( String phrase ) {
#if DEBUG
            PortUtil.println( "SymbolTable.Attatch" );
            PortUtil.println( "    phrase=" + phrase );
#endif
            for ( Iterator<Integer> itr = s_table.keySet().iterator(); itr.hasNext(); ) {
                int key = itr.next();
                SymbolTable table = s_table.get( key );
                if ( table.isEnabled() ) {
                    SymbolTableEntry ret = table.attatchImp( phrase );
                    if ( ret != null ) {
                        return ret;
                    }
                }
            }
            return null;
        }

        /// <summary>
        /// 登録されている辞書の個数を取得します
        /// </summary>
        /// <returns></returns>
        public static int getCount() {
            if ( !s_initialized ) {
                loadSystemDictionaries();
            }
            return s_table.size();
        }

        /// <summary>
        /// 辞書の優先順位と有効・無効を一括設定します
        /// </summary>
        /// <param name="list">辞書の名前・有効かどうかを表したValuePairを、辞書の優先順位の順番に格納したリスト</param>
        public static void changeOrder( Vector<ValuePair<String, Boolean>> list ) {
#if DEBUG
            PortUtil.println( "SymbolTable#changeOrder" );
#endif
            TreeMap<Integer, SymbolTable> buff = new TreeMap<Integer, SymbolTable>();
            for ( Iterator<Integer> itr = s_table.keySet().iterator(); itr.hasNext(); ) {
                int key = itr.next();
                buff.put( key, (SymbolTable)s_table.get( key ).clone() );
            }
            s_table.clear();
            int count = list.size();
            for ( int i = 0; i < count; i++ ) {
                ValuePair<String, Boolean> itemi = list.get( i );
#if DEBUG
                PortUtil.println( "SymbolTable#changeOrder; list[" + i + "]=" + itemi.getKey() + "," + itemi.getValue() );
#endif
                for ( Iterator<Integer> itr = buff.keySet().iterator(); itr.hasNext(); ) {
                    int key = itr.next();
                    SymbolTable table = buff.get( key );
                    if ( table.getName().Equals( itemi.getKey() ) ) {
                        table.setEnabled( itemi.getValue() );
                        s_table.put( i, table );
                        break;
                    }
                }
            }
        }
        #endregion

        /// <summary>
        /// この辞書のディープ・コピーを取得します
        /// </summary>
        /// <returns></returns>
#if !JAVA
        public object Clone() {
            return clone();
        }
#endif

        /// <summary>
        /// この辞書のディープ・コピーを取得します
        /// </summary>
        /// <returns></returns>
        public Object clone() {
            SymbolTable ret = new SymbolTable();
            ret.m_dict = new TreeMap<String, SymbolTableEntry>();
            for ( Iterator<String> itr = m_dict.keySet().iterator(); itr.hasNext(); ) {
                String key = itr.next();
                ret.m_dict.put( key, (SymbolTableEntry)m_dict.get( key ).clone() );
            }
            ret.m_name = m_name;
            ret.m_enabled = m_enabled;
            ret.m_max_divisions = m_max_divisions;
            return ret;
        }

        /// <summary>
        /// 使ってはいけないコンストラクタ
        /// </summary>
        private SymbolTable() {
        }

        /// <summary>
        /// 辞書の名前を取得します
        /// </summary>
        /// <returns></returns>
        public String getName() {
            return m_name;
        }

        /// <summary>
        /// 辞書が有効かどうかを取得します
        /// </summary>
        /// <returns></returns>
        public boolean isEnabled() {
            return m_enabled;
        }

        /// <summary>
        /// 辞書が有効かどうかを設定します
        /// </summary>
        /// <param name="value"></param>
        public void setEnabled( boolean value ) {
            m_enabled = value;
        }

        /// <summary>
        /// コンストラクタ
        /// </summary>
        /// <param name="path">読み込む辞書ファイルのパス</param>
        /// <param name="is_udc_mode">VOCALOID2仕様の辞書ファイルかどうか</param>
        /// <param name="enabled">辞書ファイルを有効とするかどうか</param>
        public SymbolTable( String path, boolean is_udc_mode, boolean enabled ) {
            m_dict = new TreeMap<String, SymbolTableEntry>();
            m_enabled = enabled;
            if ( !PortUtil.isFileExists( path ) ) {
                return;
            }
            m_name = PortUtil.getFileName( path );
            BufferedReader sr = null;
            try {
                if ( is_udc_mode ) {
                    sr = new BufferedReader( new InputStreamReader( new FileInputStream( path ), "Shift_JIS" ) );
                    if ( sr == null ) {
                        return;
                    }
                } else {
                    sr = new BufferedReader( new InputStreamReader( new FileInputStream( path ), "UTF-8" ) );
                    if ( sr == null ) {
                        return;
                    }
                }
                String line;
                while ( sr.ready() ) {
                    line = sr.readLine();
                    if ( line.StartsWith( "//" ) ) {
                        continue;
                    }
                    String key = "";
                    String word = "";
                    String symbol = "";
                    if ( is_udc_mode ) {
                        String[] spl = PortUtil.splitString( line, new String[] { "\t" }, 2, true );
                        if ( spl.Length >= 2 ) {
                            key = spl[0].ToLower();
                            word = key;
                            symbol = spl[1];
                        }
                    } else {
                        String[] spl = PortUtil.splitString( line, new String[] { "\t\t" }, 2, true );
                        if ( spl.Length >= 2 ) {
                            String[] spl_word = PortUtil.splitString( spl[0], '\t' );
                            m_max_divisions = Math.Max( spl_word.Length, m_max_divisions );
                            key = spl[0].Replace( "-\t", "" );
                            word = spl[0];
                            symbol = spl[1];
                        }
                    }
                    if ( !key.Equals( "" ) ) {
                        if ( !m_dict.containsKey( key ) ) {
                            m_dict.put( key, new SymbolTableEntry( word, symbol ) );
                        }
                    }
                }
            } catch ( Exception ex ) {
                Console.Error.WriteLine( "SymbolTable#.ctor; ex=" + ex );
            } finally {
                if ( sr != null ) {
                    try {
                        sr.close();
                    } catch ( Exception ex2 ) {
                        Console.Error.WriteLine( "SymbolTable#.ctor; ex=" + ex2 );
                    }
                }
            }
#if DEBUG
            PortUtil.println( "SymbolTable#.ctor; m_max_divisions=" + m_max_divisions );
#endif
        }

        /// <summary>
        /// 指定した文字列から、発音記号列を引き当てます
        /// </summary>
        /// <param name="phrase"></param>
        /// <returns></returns>
        private SymbolTableEntry attatchImp( String phrase ) {
            String s = phrase.ToLower();
            if ( m_dict.containsKey( s ) ) {
                return m_dict.get( s );
                /*result.value = m_dict.get( s ).Symbol.Replace( '\t', ' ' );
                return true;*/
            } else {
                return null;
            }
        }
    }

#if !JAVA
}

namespace Boare.Lib.Vsq.Old {
    using boolean = System.Boolean;
    using Integer = System.Int32;
#endif

#if JAVA
    public class SymbolTable implements Cloneable {
#else
    public class SymbolTable : ICloneable {
#endif
        private TreeMap<String, String> m_dict;
        private String m_name;
        private boolean m_enabled;

        #region Static Field
        private static TreeMap<Integer, SymbolTable> s_table = new TreeMap<Integer, SymbolTable>();
        private static SymbolTable s_default_jp = null;
        private static SymbolTable s_default_en = null;
        private static boolean s_initialized = false;
        #region KEY_EN
#if JAVA
        public static final String[][] _KEY_EN = {
#else
        public static readonly String[,] _KEY_EN = {
#endif
            {"a", "@"},
            {"a.", "eI"},
            {"ab", "{ b"},
            {"abbs", "{ b z"},
            {"abe", "eI b"},
            {"abt", "{ b t"},
            {"ace", "eI s"},
            {"aced", "eI s t"},
            {"ache", "eI k"},
            {"aches", "eI k s"},
            {"ack", "{ k"},
            {"act", "{ k t"},
            {"acts", "{ k t s"},
            {"act's", "{ k t s"},
            {"ad", "{ d"},
            {"add", "{ d"},
            {"addle", "{ d ADDLEMANl l"},
            {"addled", "{ d ADDLEMANl l d"},
            {"adds", "{ d z"},
            {"ade", "eI d"},
            {"ads", "{ d z"},
            {"ad's", "{ d z"},
            {"adz", "{ d z"},
            {"ae", "eI"},
            {"aer", "e@"},
            {"aft", "{ f t"},
            {"ag", "{ g"},
            {"age", "eI dZ"},
            {"aged", "eI dZ d"},
            {"ah", "Q"},
            {"ahh", "Q"},
            {"ahl", "Q l"},
            {"ahs", "Q z"},
            {"ai", "aI"},
            {"aid", "eI d"},
            {"aide", "eI d"},
            {"aides", "eI d z"},
            {"aide's", "eI d z"},
            {"aids", "eI d z"},
            {"ail", "eI l"},
            {"ails", "eI l z"},
            {"aim", "eI m"},
            {"aime", "eI m"},
            {"aimed", "eI m d"},
            {"aims", "eI m z"},
            {"ain't", "eI n t"},
            {"air", "e@"},
            {"aired", "e@ d"},
            {"airs", "e@ z"},
            {"air's", "e@ z"},
            {"ais", "aI z"},
            {"aisle", "aI l"},
            {"aisles", "aI l z"},
            {"akc", "{ k"},
            {"ake", "eI k"},
            {"akst", "{ k s t"},
            {"al", "{ l"},
            {"al.", "{ l"},
            {"ale", "eI l"},
            {"ales", "eI l z"},
            {"alf", "{ l f"},
            {"all", "O: l"},
            {"alles", "eI l z"},
            {"alls", "O: l z"},
            {"alm", "Q m"},
            {"alms", "Q l m z"},
            {"alps", "{ l p s"},
            {"als", "{ l z"},
            {"al's", "{ l z"},
            {"alt", "Q l t"},
            {"am", "{ m"},
            {"amc", "{ m k"},
            {"ame", "eI m"},
            {"ames", "eI m z"},
            {"amp", "{ m p"},
            {"ams", "{ m z"},
            {"am's", "{ m z"},
            {"an", "{ n"},
            {"and", "{ n d"},
            {"ands", "{ n d z"},
            {"ang", "{ N"},
            {"ange", "eI n dZ"},
            {"angst", "Q N k s t"},
            {"ann", "{ n"},
            {"anne", "{ n"},
            {"anne's", "{ n z"},
            {"ann's", "{ n z"},
            {"anse", "{ n s"},
            {"ant", "{ n t"},
            {"ants", "{ n t s"},
            {"ape", "eI p"},
            {"apes", "eI p s"},
            {"app", "{ p"},
            {"apps", "{ p s"},
            {"apt", "{ p t"},
            {"ar", "Q@"},
            {"arb", "Q@ b"},
            {"arbs", "Q@ b z"},
            {"arc", "Q@ k"},
            {"arce", "Q@ s"},
            {"arch", "Q@ tS"},
            {"arched", "Q@ tS t"},
            {"arcs", "Q@ k s"},
            {"ard", "Q@ d"},
            {"are", "Q@"},
            {"ares", "Q@ z"},
            {"ark", "Q@ k"},
            {"arm", "Q@ m"},
            {"armed", "Q@ m d"},
            {"armes", "Q@ m z"},
            {"arms", "Q@ m z"},
            {"arm's", "Q@ m z"},
            {"arn", "Q@ n"},
            {"arndt", "Q@ n t"},
            {"arne", "Q@ n"},
            {"arnst", "Q@ n s t"},
            {"arp", "Q@ p"},
            {"art", "Q@ t"},
            {"arts", "Q@ t s"},
            {"art's", "Q@ t s"},
            {"arx", "Q@ k s"},
            {"as", "{ z"},
            {"a's", "eI z"},
            {"asch", "{ S"},
            {"ash", "{ S"},
            {"ashe", "{ S"},
            {"ask", "{ s k"},
            {"asked", "{ s k t"},
            {"asks", "{ s k s"},
            {"asp", "{ s p"},
            {"ass", "{ s"},
            {"assed", "{ s t"},
            {"ast", "{ s t"},
            {"at", "{ t"},
            {"ate", "eI t"},
            {"ates", "eI t s"},
            {"au", "@U"},
            {"aube", "O: b"},
            {"aud", "O: d"},
            {"auge", "O: dZ"},
            {"aul", "O: l"},
            {"auld", "O: l d"},
            {"ault", "O: l t"},
            {"aune", "O: n"},
            {"aunt", "{ n t"},
            {"aunts", "{ n t s"},
            {"aunt's", "{ n t s"},
            {"aus", "aU z"},
            {"aust", "O: s t"},
            {"auth", "O: T"},
            {"aux", "@U"},
            {"aw", "O:"},
            {"awb", "Q w b"},
            {"awe", "Q"},
            {"awed", "O: d"},
            {"ax", "{ k s"},
            {"axe", "{ k s"},
            {"axed", "{ k s t"},
            {"ay", "eI"},
            {"aye", "aI"},
            {"ayer's", "e@ z"},
            {"ayes", "aI z"},
            {"ayn", "eI n"},
            {"ayr", "eI r"},
            {"ayre", "e@"},
            {"ayres", "e@ z"},
            {"b", "bh i:"},
            {"b.", "bh i:"},
            {"baar", "bh Q@"},
            {"baas", "bh Q z"},
            {"bab", "bh { b"},
            {"babb", "bh { b"},
            {"babe", "bh eI b"},
            {"babes", "bh eI b z"},
            {"babs", "bh { b z"},
            {"bach", "bh Q k"},
            {"bache", "bh { tS"},
            {"back", "bh { k"},
            {"backed", "bh { k t"},
            {"backs", "bh { k s"},
            {"bad", "bh { d"},
            {"bade", "bh eI d"},
            {"badge", "bh { dZ"},
            {"bae", "bh aI"},
            {"baer", "bh e@"},
            {"bag", "bh { g"},
            {"bagge", "bh { g"},
            {"bagged", "bh { g d"},
            {"baggs", "bh { g z"},
            {"bags", "bh { g z"},
            {"bah", "bh Q"},
            {"bahr", "bh e@"},
            {"baht", "bh Q t"},
            {"bail", "bh eI l"},
            {"baile", "bh eI l"},
            {"bailed", "bh eI l d"},
            {"bails", "bh eI l z"},
            {"bain", "bh eI n"},
            {"baines", "bh eI n z"},
            {"bains", "bh eI n z"},
            {"bain's", "bh eI n z"},
            {"baird", "bh e@ d"},
            {"baird's", "bh e@ d z"},
            {"bait", "bh eI t"},
            {"baits", "bh eI t s"},
            {"baize", "bh eI z"},
            {"bak", "bh { k"},
            {"bake", "bh eI k"},
            {"baked", "bh eI k t"},
            {"bakes", "bh eI k s"},
            {"bakke", "bh { k"},
            {"bakst", "bh { k s t"},
            {"bal", "bh { l"},
            {"balch", "bh { l tS"},
            {"bald", "bh O: l d"},
            {"bale", "bh eI l"},
            {"bales", "bh eI l z"},
            {"balk", "bh O: k"},
            {"balked", "bh O: k t"},
            {"balks", "bh O: k s"},
            {"ball", "bh O: l"},
            {"balled", "bh O: l d"},
            {"balls", "bh O: l z"},
            {"ball's", "bh O: l z"},
            {"balm", "bh Q m"},
            {"balms", "bh Q m z"},
            {"balt's", "bh O: l t s"},
            {"bam", "bh { m"},
            {"ban", "bh { n"},
            {"banc", "bh { N k"},
            {"band", "bh { n d"},
            {"bands", "bh { n d z"},
            {"band's", "bh { n d z"},
            {"bane", "bh eI n"},
            {"banes", "bh eI n z"},
            {"banff", "bh { n f"},
            {"bang", "bh { N"},
            {"bange", "bh { n dZ"},
            {"banged", "bh { N d"},
            {"bangs", "bh { N z"},
            {"bank", "bh { N k"},
            {"banked", "bh { N k t"},
            {"banks", "bh { N k s"},
            {"bank's", "bh { N k s"},
            {"banks'", "bh { N k s"},
            {"bann", "bh { n"},
            {"banned", "bh { n d"},
            {"banque", "bh { N k"},
            {"bans", "bh { n z"},
            {"bao", "bh aU"},
            {"bar", "bh Q@"},
            {"barb", "bh Q@ b"},
            {"barbe", "bh Q@ b"},
            {"barbed", "bh Q@ b d"},
            {"barbs", "bh Q@ b z"},
            {"barb's", "bh Q@ b z"},
            {"barch", "bh Q@ k"},
            {"bard", "bh Q@ d"},
            {"bard's", "bh Q@ d z"},
            {"bare", "bh e@"},
            {"bared", "bh e@ d"},
            {"bares", "bh e@ z"},
            {"barge", "bh Q@ dZ"},
            {"barged", "bh Q@ dZ d"},
            {"bark", "bh Q@ k"},
            {"barked", "bh Q@ k t"},
            {"barks", "bh Q@ k s"},
            {"barn", "bh Q@ n"},
            {"barnes", "bh Q@ n z"},
            {"barns", "bh Q@ n z"},
            {"barr", "bh Q@"},
            {"barred", "bh Q@ d"},
            {"barres", "bh Q@ z"},
            {"barr's", "bh Q@ z"},
            {"bars", "bh Q@ z"},
            {"bar's", "bh Q@ z"},
            {"bart", "bh Q@ t"},
            {"barth", "bh Q@ T"},
            {"barthes", "bh Q@ T s"},
            {"base", "bh eI s"},
            {"based", "bh eI s t"},
            {"bash", "bh { S"},
            {"bashed", "bh { S t"},
            {"bask", "bh { s k"},
            {"basked", "bh { s k t"},
            {"basks", "bh { s k s"},
            {"basque", "bh { s k"},
            {"basques", "bh { s k s"},
            {"bass", "bh { s"},
            {"basse", "bh { s"},
            {"bast", "bh { s t"},
            {"bat", "bh { t"},
            {"batch", "bh { tS"},
            {"bate", "bh eI t"},
            {"bates", "bh eI t s"},
            {"bath", "bh { T"},
            {"bathe", "bh eI D"},
            {"bathed", "bh eI D d"},
            {"bathes", "bh eI D z"},
            {"baths", "bh { T s"},
            {"bats", "bh { t s"},
            {"batt", "bh { t"},
            {"batts", "bh { t s"},
            {"batz", "bh { t s"},
            {"bauch", "bh O: tS"},
            {"baud", "bh O: d"},
            {"baugh", "bh O:"},
            {"baum", "bh O: m"},
            {"baun", "bh O: n"},
            {"baur", "bh O@"},
            {"bausch", "bh aU S"},
            {"bawd", "bh O: d"},
            {"bawl", "bh O: l"},
            {"bawled", "bh O: l d"},
            {"bax", "bh { k s"},
            {"bay", "bh eI"},
            {"baye", "bh eI"},
            {"bayed", "bh eI d"},
            {"bayes", "bh eI z"},
            {"bayh", "bh eI"},
            {"bayle", "bh eI l"},
            {"bayles", "bh eI l z"},
            {"bays", "bh eI z"},
            {"bay's", "bh eI z"},
            {"baze", "bh eI z"},
            {"be", "bh i:"},
            {"bea", "bh i:"},
            {"beach", "bh i: tS"},
            {"beached", "bh i: tS t"},
            {"bead", "bh i: d"},
            {"beads", "bh i: d z"},
            {"beak", "bh i: k"},
            {"beal", "bh i: l"},
            {"beale", "bh i: l"},
            {"beall", "bh i: l"},
            {"beals", "bh i: l z"},
            {"beam", "bh i: m"},
            {"beamed", "bh i: m d"},
            {"beams", "bh i: m z"},
            {"beam's", "bh i: m z"},
            {"bean", "bh i: n"},
            {"beane", "bh i: n"},
            {"beans", "bh i: n z"},
            {"bean's", "bh i: n z"},
            {"bear", "bh e@"},
            {"bearce", "bh @r s"},
            {"beard", "bh I@ d"},
            {"beards", "bh I@ d z"},
            {"beare", "bh I@"},
            {"bears", "bh e@ z"},
            {"bear's", "bh e@ z"},
            {"beast", "bh i: s t"},
            {"beasts", "bh i: s t s"},
            {"beat", "bh i: t"},
            {"beats", "bh i: t s"},
            {"beau", "bh @U"},
            {"beaux", "bh @U"},
            {"bec", "bh e k"},
            {"becht", "bh e k t"},
            {"beck", "bh e k"},
            {"becks", "bh e k s"},
            {"beck's", "bh e k s"},
            {"bed", "bh e d"},
            {"bede", "bh i: d"},
            {"beds", "bh e d z"},
            {"bee", "bh i:"},
            {"beech", "bh i: tS"},
            {"beeck", "bh i: k"},
            {"beef", "bh i: f"},
            {"beefed", "bh i: f t"},
            {"beefs", "bh i: f s"},
            {"beek", "bh i: k"},
            {"been", "bh I n"},
            {"beens", "bh I n z"},
            {"beep", "bh i: p"},
            {"beeps", "bh i: p s"},
            {"beer", "bh I@"},
            {"beers", "bh I@ z"},
            {"beer's", "bh I@ z"},
            {"bees", "bh i: z"},
            {"bee's", "bh i: z"},
            {"beet", "bh i: t"},
            {"beets", "bh i: t s"},
            {"beg", "bh e g"},
            {"begged", "bh e g d"},
            {"beggs", "bh e g z"},
            {"begs", "bh e g z"},
            {"behl", "bh e l"},
            {"behlke", "bh e l k"},
            {"behm", "bh e m"},
            {"behn", "bh e n"},
            {"beige", "bh eI Z"},
            {"bein", "bh i: n"},
            {"beirne", "bh I@ n"},
            {"beitz", "bh i: t s"},
            {"bel", "bh e l"},
            {"belch", "bh e l tS"},
            {"belched", "bh e l tS t"},
            {"belk", "bh e l k"},
            {"bell", "bh e l"},
            {"belle", "bh e l"},
            {"belles", "bh e l z"},
            {"bells", "bh e l z"},
            {"bell's", "bh e l z"},
            {"bels", "bh e l z"},
            {"belt", "bh e l t"},
            {"belts", "bh e l t s"},
            {"belt's", "bh e l t s"},
            {"bem", "bh e m"},
            {"ben", "bh e n"},
            {"bench", "bh e n tS"},
            {"bend", "bh e n d"},
            {"bends", "bh e n d z"},
            {"benge", "bh e n dZ"},
            {"bengt", "bh e N k t"},
            {"benn", "bh e n"},
            {"benne", "bh e n"},
            {"bens", "bh e n z"},
            {"ben's", "bh e n z"},
            {"bent", "bh e n t"},
            {"benz", "bh e n z"},
            {"bere", "bh I@"},
            {"beres", "bh i: r z"},
            {"berg", "bh @r g"},
            {"bergh", "bh @r g"},
            {"bergs", "bh @r g z"},
            {"berg's", "bh @r g z"},
            {"berk", "bh @r k"},
            {"berke", "bh @r k"},
            {"berle", "bh @r l"},
            {"berm", "bh @r m"},
            {"bern", "bh @r n"},
            {"berne", "bh @r n"},
            {"berns", "bh @r n z"},
            {"bernt", "bh @r n t"},
            {"bert", "bh @r t"},
            {"berte", "bh @r t"},
            {"berth", "bh @r T"},
            {"berths", "bh @r T s"},
            {"bertsche", "bh @r tS"},
            {"bes", "bh i: z"},
            {"bess", "bh e s"},
            {"besse", "bh e s"},
            {"best", "bh e s t"},
            {"best's", "bh e s t s"},
            {"bet", "bh e t"},
            {"beth", "bh e T"},
            {"bets", "bh e t s"},
            {"bet's", "bh e t s"},
            {"bett", "bh e t"},
            {"betz", "bh e t s"},
            {"bev", "bh e v"},
            {"bey", "bh eI"},
            {"bhatt", "bh { t"},
            {"bi", "bh aI"},
            {"bib", "bh I b"},
            {"bibb", "bh I b"},
            {"bibbs", "bh I b z"},
            {"bibs", "bh I b z"},
            {"bic", "bh I k"},
            {"bice", "bh aI s"},
            {"bick", "bh I k"},
            {"bid", "bh I d"},
            {"bide", "bh aI d"},
            {"bids", "bh I d z"},
            {"bid's", "bh I d z"},
            {"biel", "bh i: l"},
            {"bien", "bh i: n"},
            {"bier", "bh i: r"},
            {"bierce", "bh I@ s"},
            {"biers", "bh i: r z"},
            {"biff", "bh I f"},
            {"big", "bh I g"},
            {"biggs", "bh I g z"},
            {"bigs", "bh I g z"},
            {"bike", "bh aI k"},
            {"biked", "bh aI k t"},
            {"bikes", "bh aI k s"},
            {"bil", "bh I l"},
            {"bile", "bh aI l"},
            {"biles", "bh aI l z"},
            {"bilk", "bh I l k"},
            {"bilked", "bh I l k t"},
            {"bill", "bh I l"},
            {"bille", "bh aI l"},
            {"billed", "bh I l d"},
            {"bills", "bh I l z"},
            {"bill's", "bh I l z"},
            {"bin", "bh I n"},
            {"bind", "bh aI n d"},
            {"binds", "bh aI n d z"},
            {"bines", "bh aI n z"},
            {"bing", "bh I N"},
            {"binge", "bh I n dZ"},
            {"bink", "bh I N k"},
            {"bins", "bh I n z"},
            {"birch", "bh @r tS"},
            {"bird", "bh @r d"},
            {"birds", "bh @r d z"},
            {"bird's", "bh @r d z"},
            {"birk", "bh @r k"},
            {"birks", "bh @r k s"},
            {"birle", "bh @r l"},
            {"birr", "bh @r"},
            {"birt", "bh @r t"},
            {"birth", "bh @r T"},
            {"births", "bh @r T s"},
            {"bis", "bh I s"},
            {"bise", "bh aI z"},
            {"bish", "bh I S"},
            {"bisque", "bh I s k"},
            {"bit", "bh I t"},
            {"bitch", "bh I tS"},
            {"bite", "bh aI t"},
            {"bites", "bh aI t s"},
            {"bits", "bh I t s"},
            {"bix", "bh I k s"},
            {"biz", "bh I z"},
            {"bjork", "bh j O@ k"},
            {"bjorn", "bh j O@ n"},
            {"black", "bh l { k"},
            {"blacked", "bh l { k t"},
            {"blacks", "bh l { k s"},
            {"black's", "bh l { k s"},
            {"blade", "bh l eI d"},
            {"blades", "bh l eI d z"},
            {"blagg", "bh l { g"},
            {"blah", "bh l Q"},
            {"blain", "bh l eI n"},
            {"blaine", "bh l eI n"},
            {"blair", "bh l e@"},
            {"blair's", "bh l e@ z"},
            {"blaise", "bh l eI z"},
            {"blaize", "bh l eI z"},
            {"blake", "bh l eI k"},
            {"blake's", "bh l eI k s"},
            {"blame", "bh l eI m"},
            {"blamed", "bh l eI m d"},
            {"blames", "bh l eI m z"},
            {"blan", "bh l { n"},
            {"blanc", "bh l { N k"},
            {"blanch", "bh l { n tS"},
            {"blanche", "bh l { n tS"},
            {"bland", "bh l { n d"},
            {"blane", "bh l eI n"},
            {"blank", "bh l { N k"},
            {"blanked", "bh l { N k t"},
            {"blanks", "bh l { N k s"},
            {"blare", "bh l e@"},
            {"blared", "bh l e@ d"},
            {"blares", "bh l e@ z"},
            {"blas", "bh l Q s"},
            {"blase", "bh l eI z"},
            {"blast", "bh l { s t"},
            {"blasts", "bh l { s t s"},
            {"blatt", "bh l { t"},
            {"blatz", "bh l { t s"},
            {"blau", "bh l aU"},
            {"blay", "bh l eI"},
            {"blayne", "bh l eI n"},
            {"blayze", "bh l eI z"},
            {"blaze", "bh l eI z"},
            {"blazed", "bh l eI z d"},
            {"blea", "bh l i:"},
            {"bleach", "bh l i: tS"},
            {"bleached", "bh l i: tS t"},
            {"bleak", "bh l i: k"},
            {"blech", "bh l e k"},
            {"bleck", "bh l e k"},
            {"bled", "bh l e d"},
            {"bleed", "bh l i: d"},
            {"bleeds", "bh l i: d z"},
            {"bleep", "bh l i: p"},
            {"blend", "bh l e n d"},
            {"blends", "bh l e n d z"},
            {"bless", "bh l e s"},
            {"blessed", "bh l e s t"},
            {"blest", "bh l e s t"},
            {"bleu", "bh l u:"},
            {"blew", "bh l u:"},
            {"bley", "bh l eI"},
            {"blick", "bh l I k"},
            {"bligh", "bh l aI"},
            {"blight", "bh l aI t"},
            {"blimp", "bh l I m p"},
            {"blimps", "bh l I m p s"},
            {"blind", "bh l aI n d"},
            {"blinds", "bh l aI n d z"},
            {"blink", "bh l I N k"},
            {"blinked", "bh l I N k t"},
            {"blinks", "bh l I N k s"},
            {"blinn", "bh l I n"},
            {"blip", "bh l I p"},
            {"blips", "bh l I p s"},
            {"bliss", "bh l I s"},
            {"blithe", "bh l aI D"},
            {"blitz", "bh l I t s"},
            {"blitzed", "bh l I t s t"},
            {"bloat", "bh l @U t"},
            {"blob", "bh l Q b"},
            {"blobs", "bh l Q b z"},
            {"bloc", "bh l Q k"},
            {"bloch", "bh l Q k"},
            {"bloch's", "bh l Q k s"},
            {"block", "bh l Q k"},
            {"blocked", "bh l Q k t"},
            {"blocks", "bh l Q k s"},
            {"block's", "bh l Q k s"},
            {"blocs", "bh l Q k s"},
            {"bloc's", "bh l Q k s"},
            {"bloem", "bh l @U m"},
            {"blok", "bh l Q k"},
            {"bloke", "bh l @U k"},
            {"blokes", "bh l @U k s"},
            {"blond", "bh l Q n d"},
            {"blonde", "bh l Q n d"},
            {"blondes", "bh l Q n d z"},
            {"blonde's", "bh l Q n d z"},
            {"blonds", "bh l Q n d z"},
            {"blood", "bh l V d"},
            {"bloods", "bh l V d z"},
            {"blood's", "bh l V d z"},
            {"bloom", "bh l u: m"},
            {"bloomed", "bh l u: m d"},
            {"blooms", "bh l u: m z"},
            {"bloom's", "bh l u: m z"},
            {"blot", "bh l Q t"},
            {"blots", "bh l Q t s"},
            {"blount", "bh l aU n t"},
            {"blouse", "bh l aU s"},
            {"blow", "bh l @U"},
            {"blowed", "bh l @U d"},
            {"blown", "bh l @U n"},
            {"blows", "bh l @U z"},
            {"blue", "bh l u:"},
            {"blues", "bh l u: z"},
            {"blue's", "bh l u: z"},
            {"bluff", "bh l V f"},
            {"bluffed", "bh l V f t"},
            {"bluffs", "bh l V f s"},
            {"bluhm", "bh l V m"},
            {"blum", "bh l u: m"},
            {"blume", "bh l u: m"},
            {"blunk", "bh l V N k"},
            {"blunt", "bh l V n t"},
            {"blunts", "bh l V n t s"},
            {"blur", "bh l @r"},
            {"blurb", "bh l @r b"},
            {"blurbs", "bh l @r b z"},
            {"blurred", "bh l @r d"},
            {"blurs", "bh l @r z"},
            {"blurt", "bh l @r t"},
            {"blurts", "bh l @r t s"},
            {"blush", "bh l V S"},
            {"blushed", "bh l V S t"},
            {"bly", "bh l aI"},
            {"blyth", "bh l I T"},
            {"blythe", "bh l aI D"},
            {"bo", "bh @U"},
            {"boak", "bh @U k"},
            {"boar", "bh O@"},
            {"board", "bh O@ d"},
            {"boards", "bh O@ d z"},
            {"board's", "bh O@ d z"},
            {"boart", "bh O@ t"},
            {"boast", "bh @U s t"},
            {"boasts", "bh @U s t s"},
            {"boat", "bh @U t"},
            {"boats", "bh @U t s"},
            {"boat's", "bh @U t s"},
            {"bob", "bh Q b"},
            {"bob's", "bh Q b z"},
            {"boche", "bh Q tS"},
            {"bock", "bh Q k"},
            {"bode", "bh @U d"},
            {"bodes", "bh @U d z"},
            {"boe", "bh @U"},
            {"boehm", "bh @U m"},
            {"boehme", "bh @U m"},
            {"boeke", "bh @U k"},
            {"boer", "bh O@"},
            {"boers", "bh O@ z"},
            {"boesch", "bh @U S"},
            {"boff", "bh O: f"},
            {"bog", "bh Q g"},
            {"bogged", "bh Q g d"},
            {"boggs", "bh O: g z"},
            {"bogs", "bh Q g z"},
            {"bogue", "bh @U g"},
            {"bohm", "bh Q m"},
            {"bohr", "bh O@"},
            {"boice", "bh OI s"},
            {"boies", "bh OI z"},
            {"boil", "bh OI l"},
            {"boiled", "bh OI l d"},
            {"boils", "bh OI l z"},
            {"bois", "bh w Q"},
            {"bok", "bh Q k"},
            {"bol", "bh Q l"},
            {"bold", "bh @U l d"},
            {"bolds", "bh @U l d z"},
            {"bole", "bh @U l"},
            {"boles", "bh @U l z"},
            {"boll", "bh @U l"},
            {"bolls", "bh @U l z"},
            {"bolt", "bh @U l t"},
            {"bolte", "bh @U l t"},
            {"bolts", "bh @U l t s"},
            {"bom", "bh Q m"},
            {"bomb", "bh Q m"},
            {"bombed", "bh Q m d"},
            {"bombs", "bh Q m z"},
            {"bon", "bh Q n"},
            {"bond", "bh Q n d"},
            {"bonds", "bh Q n d z"},
            {"bond's", "bh Q n d z"},
            {"bone", "bh @U n"},
            {"boned", "bh @U n d"},
            {"bones", "bh @U n z"},
            {"bong", "bh Q N"},
            {"bonk", "bh Q N k"},
            {"bonn", "bh Q n"},
            {"bonne", "bh Q n"},
            {"bonnes", "bh O: n z"},
            {"bonn's", "bh Q n z"},
            {"boo", "bh u:"},
            {"boob", "bh u: b"},
            {"boobs", "bh u: b z"},
            {"booed", "bh u: d"},
            {"book", "bh U k"},
            {"booked", "bh U k t"},
            {"books", "bh U k s"},
            {"book's", "bh U k s"},
            {"boole", "bh u: l"},
            {"boom", "bh u: m"},
            {"boomed", "bh u: m d"},
            {"booms", "bh u: m z"},
            {"boon", "bh u: n"},
            {"boone", "bh u: n"},
            {"boor", "bh U@"},
            {"boord", "bh U@ d"},
            {"boors", "bh U@ z"},
            {"boos", "bh u: z"},
            {"boose", "bh u: s"},
            {"boost", "bh u: s t"},
            {"boosts", "bh u: s t s"},
            {"boot", "bh u: t"},
            {"boote", "bh u: t"},
            {"booth", "bh u: T"},
            {"boothe", "bh u: D"},
            {"booths", "bh u: T s"},
            {"boots", "bh u: t s"},
            {"booz", "bh u: z"},
            {"booze", "bh u: z"},
            {"bop", "bh Q p"},
            {"bopp", "bh Q p"},
            {"bordes", "bh O@ d z"},
            {"bore", "bh O@"},
            {"bored", "bh O@ d"},
            {"bores", "bh O@ z"},
            {"borg", "bh O@ g"},
            {"bork", "bh O@ k"},
            {"bork's", "bh O@ k s"},
            {"born", "bh O@ n"},
            {"borne", "bh O@ n"},
            {"bors", "bh O@ z"},
            {"borsch", "bh O@ S"},
            {"borscht", "bh O@ S t"},
            {"bort", "bh O@ t"},
            {"bortz", "bh O@ t s"},
            {"bos", "bh Q s"},
            {"bosch", "bh O: S"},
            {"bose", "bh @U z"},
            {"bosh", "bh Q S"},
            {"boss", "bh O: s"},
            {"bossed", "bh Q s t"},
            {"botch", "bh Q tS"},
            {"botched", "bh Q tS t"},
            {"both", "bh @U T"},
            {"bothe", "bh @U D"},
            {"bott", "bh Q t"},
            {"botts", "bh Q t s"},
            {"bouch", "bh aU tS"},
            {"bouche", "bh aU tS"},
            {"bough", "bh aU"},
            {"boughs", "bh aU z"},
            {"bought", "bh Q t"},
            {"boule", "bh u: l"},
            {"bounce", "bh aU n s"},
            {"bounced", "bh aU n s t"},
            {"bound", "bh aU n d"},
            {"bounds", "bh aU n d z"},
            {"bour", "bh aU r"},
            {"bourg", "bh O@ g"},
            {"bourke", "bh @r k"},
            {"bourn", "bh O@ n"},
            {"bourne", "bh O@ n"},
            {"bourque", "bh U@ k"},
            {"bourse", "bh O@ s"},
            {"bouse", "bh aU s"},
            {"bout", "bh aU t"},
            {"bouts", "bh aU t s"},
            {"boutte", "bh u: t"},
            {"bove", "bh @U v"},
            {"bow", "bh aU"},
            {"bowe", "bh @U"},
            {"bowed", "bh aU d"},
            {"bowes", "bh @U z"},
            {"bowl", "bh @U l"},
            {"bowlds", "bh @U l d z"},
            {"bowled", "bh @U l d"},
            {"bowles", "bh @U l z"},
            {"bowls", "bh @U l z"},
            {"bown", "bh @U n"},
            {"bowne", "bh @U n"},
            {"bows", "bh aU z"},
            {"bowse", "bh @U s"},
            {"bowsed", "bh aU z d"},
            {"box", "bh Q k s"},
            {"boxed", "bh Q k s t"},
            {"boy", "bh OI"},
            {"boyce", "bh OI s"},
            {"boyd", "bh OI d"},
            {"boyd's", "bh OI d z"},
            {"boyes", "bh OI z"},
            {"boyle", "bh OI l"},
            {"boyne", "bh OI n"},
            {"boys", "bh OI z"},
            {"boy's", "bh OI z"},
            {"boys'", "bh OI z"},
            {"boyt", "bh OI t"},
            {"bra", "bh r Q"},
            {"braasch", "bh r Q S"},
            {"brace", "bh r eI s"},
            {"braced", "bh r eI s t"},
            {"brach", "bh r { tS"},
            {"brack", "bh r { k"},
            {"brad", "bh r { d"},
            {"brad's", "bh r { d z"},
            {"brae", "bh r eI"},
            {"brag", "bh r { g"},
            {"bragg", "bh r { g"},
            {"bragged", "bh r { g d"},
            {"braggs", "bh r { g z"},
            {"brags", "bh r { g z"},
            {"brahm", "bh r Q m"},
            {"brahms", "bh r Q m z"},
            {"braid", "bh r eI d"},
            {"braids", "bh r eI d z"},
            {"braille", "bh r eI l"},
            {"brailles", "bh r eI l z"},
            {"brain", "bh r eI n"},
            {"brained", "bh r eI n d"},
            {"brains", "bh r eI n z"},
            {"brain's", "bh r eI n z"},
            {"braise", "bh r eI z"},
            {"braised", "bh r eI z d"},
            {"brake", "bh r eI k"},
            {"braked", "bh r eI k t"},
            {"brakes", "bh r eI k s"},
            {"bram", "bh r { m"},
            {"brame", "bh r eI m"},
            {"bran", "bh r { n"},
            {"branch", "bh r { n tS"},
            {"branched", "bh r { n tS t"},
            {"brand", "bh r { n d"},
            {"brande", "bh r { n d"},
            {"brandes", "bh r { n d z"},
            {"brands", "bh r { n d z"},
            {"brand's", "bh r { n d z"},
            {"brandt", "bh r { n t"},
            {"brank", "bh r { N k"},
            {"brant", "bh r { n t"},
            {"braque", "bh r { k"},
            {"bras", "bh r { s"},
            {"brash", "bh r { S"},
            {"brass", "bh r { s"},
            {"brat", "bh r { t"},
            {"brats", "bh r { t s"},
            {"brau", "bh r aU"},
            {"braun", "bh r O: n"},
            {"braun's", "bh r O: n z"},
            {"brause", "bh r O: z"},
            {"brave", "bh r eI v"},
            {"braved", "bh r eI v d"},
            {"braves", "bh r eI v z"},
            {"brawl", "bh r O: l"},
            {"brawls", "bh r O: l z"},
            {"brawn", "bh r O: n"},
            {"bray", "bh r eI"},
            {"braz", "bh r { z"},
            {"brea", "bh r i:"},
            {"breach", "bh r i: tS"},
            {"breached", "bh r i: tS t"},
            {"bread", "bh r e d"},
            {"breads", "bh r e d z"},
            {"breadth", "bh r e d T"},
            {"break", "bh r eI k"},
            {"breaks", "bh r eI k s"},
            {"bream", "bh r i: m"},
            {"breast", "bh r e s t"},
            {"breasts", "bh r e s t s"},
            {"breath", "bh r e T"},
            {"breathe", "bh r i: D"},
            {"breathed", "bh r i: D d"},
            {"breathes", "bh r i: D z"},
            {"breaths", "bh r e T s"},
            {"breaux", "bh r @U"},
            {"brecht", "bh r e k t"},
            {"breck", "bh r e k"},
            {"bred", "bh r e d"},
            {"brede", "bh r i: d"},
            {"bree", "bh r i:"},
            {"breech", "bh r i: tS"},
            {"breed", "bh r i: d"},
            {"breeds", "bh r i: d z"},
            {"breed's", "bh r i: d z"},
            {"breen", "bh r i: n"},
            {"brees", "bh r i: z"},
            {"breese", "bh r i: z"},
            {"breeze", "bh r i: z"},
            {"breezed", "bh r i: z d"},
            {"bren", "bh r e n"},
            {"brenn", "bh r e n"},
            {"brent", "bh r e n t"},
            {"brents", "bh r e n t s"},
            {"bress", "bh r e s"},
            {"brest", "bh r e s t"},
            {"bret", "bh r e t"},
            {"breth", "bh r e T"},
            {"brett", "bh r e t"},
            {"bretz", "bh r e t s"},
            {"brew", "bh r u:"},
            {"brewed", "bh r u: d"},
            {"brews", "bh r u: z"},
            {"brey", "bh r eI"},
            {"bribe", "bh r aI b"},
            {"bribed", "bh r aI b d"},
            {"bribes", "bh r aI b z"},
            {"brice", "bh r aI s"},
            {"brick", "bh r I k"},
            {"bricks", "bh r I k s"},
            {"bride", "bh r aI d"},
            {"brides", "bh r aI d z"},
            {"bride's", "bh r aI d z"},
            {"bridge", "bh r I dZ"},
            {"bridged", "bh r I dZ d"},
            {"brie", "bh r i:"},
            {"brief", "bh r i: f"},
            {"briefed", "bh r i: f t"},
            {"briefs", "bh r i: f s"},
            {"brig", "bh r I g"},
            {"briggs", "bh r I g z"},
            {"bright", "bh r aI t"},
            {"bright's", "bh r aI t s"},
            {"brill", "bh r I l"},
            {"brim", "bh r I m"},
            {"brimmed", "bh r I m d"},
            {"brin", "bh r I n"},
            {"brine", "bh r aI n"},
            {"brines", "bh r aI n z"},
            {"bring", "bh r I N"},
            {"brings", "bh r I N z"},
            {"brink", "bh r I N k"},
            {"brinks", "bh r I N k s"},
            {"brink's", "bh r I N k s"},
            {"brinn", "bh r I n"},
            {"brisk", "bh r I s k"},
            {"brit", "bh r I t"},
            {"brite", "bh r aI t"},
            {"b'rith", "bh r I T"},
            {"brits", "bh r I t s"},
            {"britt", "bh r I t"},
            {"britts", "bh r I t s"},
            {"brix", "bh r I k s"},
            {"bro", "bh r @U"},
            {"broach", "bh r @U tS"},
            {"broached", "bh r @U tS t"},
            {"broad", "bh r O: d"},
            {"broad's", "bh r O: d z"},
            {"brock", "bh r Q k"},
            {"brock's", "bh r Q k s"},
            {"brod", "bh r Q d"},
            {"broil", "bh r OI l"},
            {"broiled", "bh r OI l d"},
            {"brok", "bh r Q k"},
            {"broke", "bh r @U k"},
            {"brom", "bh r Q m"},
            {"bronc", "bh r Q N k"},
            {"bronk", "bh r Q N k"},
            {"bronx", "bh r Q N k s"},
            {"bronze", "bh r Q n z"},
            {"bronzed", "bh r Q n z d"},
            {"brooch", "bh r @U tS"},
            {"brood", "bh r u: d"},
            {"brook", "bh r U k"},
            {"brooke", "bh r U k"},
            {"brookes", "bh r U k s"},
            {"brooke's", "bh r U k s"},
            {"brooks", "bh r U k s"},
            {"brook's", "bh r U k s"},
            {"broom", "bh r u: m"},
            {"broome", "bh r u: m"},
            {"brooms", "bh r u: m z"},
            {"bro's", "bh r @U z"},
            {"brose", "bh r @U z"},
            {"brost", "bh r Q s t"},
            {"broth", "bh r O: T"},
            {"broths", "bh r O: T s"},
            {"brott", "bh r Q t"},
            {"brough", "bh r aU"},
            {"brought", "bh r O: t"},
            {"broun", "bh r u: n"},
            {"brow", "bh r aU"},
            {"browed", "bh r aU d"},
            {"brown", "bh r aU n"},
            {"browne", "bh r aU n"},
            {"browned", "bh r aU n d"},
            {"browns", "bh r aU n z"},
            {"brown's", "bh r aU n z"},
            {"brows", "bh r aU z"},
            {"browse", "bh r aU z"},
            {"browsed", "bh r aU z d"},
            {"broz", "bh r Q z"},
            {"bruce", "bh r u: s"},
            {"bruch", "bh r V tS"},
            {"brugh", "bh r V"},
            {"bruhn", "bh r V n"},
            {"bruise", "bh r u: z"},
            {"bruised", "bh r u: z d"},
            {"brule", "bh r u: l"},
            {"brum", "bh r V m"},
            {"brunch", "bh r V n tS"},
            {"brune", "bh r u: n"},
            {"brunk", "bh r V N k"},
            {"brunn", "bh r V n"},
            {"bruns", "bh r V n z"},
            {"brunt", "bh r V n t"},
            {"brush", "bh r V S"},
            {"brushed", "bh r V S t"},
            {"brusque", "bh r V s k"},
            {"brust", "bh r V s t"},
            {"brut", "bh r u: t"},
            {"brute", "bh r u: t"},
            {"bryce", "bh r aI s"},
            {"bryn", "bh r I n"},
            {"b's", "bh i: z"},
            {"bub", "bh V b"},
            {"bubb", "bh V b"},
            {"buch", "bh V tS"},
            {"buck", "bh V k"},
            {"bucked", "bh V k t"},
            {"bucks", "bh V k s"},
            {"buck's", "bh V k s"},
            {"bud", "bh V d"},
            {"budd", "bh V d"},
            {"budde", "bh V d"},
            {"budge", "bh V dZ"},
            {"budged", "bh V dZ d"},
            {"buds", "bh V d z"},
            {"bud's", "bh V d z"},
            {"bueche", "bh u: tS"},
            {"buff", "bh V f"},
            {"buffs", "bh V f s"},
            {"buff's", "bh V f s"},
            {"bug", "bh V g"},
            {"bugged", "bh V g d"},
            {"bugs", "bh V g z"},
            {"buhl", "bh j u: l"},
            {"buhr", "bh j U@"},
            {"bui", "bh I"},
            {"build", "bh I l d"},
            {"builds", "bh I l d z"},
            {"built", "bh I l t"},
            {"bulb", "bh V l b"},
            {"bulbs", "bh V l b z"},
            {"bulge", "bh V l dZ"},
            {"bulged", "bh V l dZ d"},
            {"bulk", "bh V l k"},
            {"bull", "bh U l"},
            {"bulls", "bh U l z"},
            {"bull's", "bh U l z"},
            {"bult", "bh V l t"},
            {"bum", "bh V m"},
            {"bummed", "bh V m d"},
            {"bump", "bh V m p"},
            {"bumped", "bh V m p t"},
            {"bumps", "bh V m p s"},
            {"bums", "bh V m z"},
            {"bun", "bh V n"},
            {"bunce", "bh V n s"},
            {"bunch", "bh V n tS"},
            {"bunche", "bh V n tS"},
            {"bunched", "bh V n tS t"},
            {"bund", "bh V n d"},
            {"bunde", "bh V n d"},
            {"bunds", "bh V n d z"},
            {"bung", "bh V N"},
            {"bunk", "bh V N k"},
            {"bunks", "bh V N k s"},
            {"bunn", "bh V n"},
            {"buns", "bh V n z"},
            {"bunt", "bh V n"},
            {"bur", "bh @r"},
            {"burch", "bh @r tS"},
            {"burck", "bh @r k"},
            {"burd", "bh @r d"},
            {"burg", "bh @r g"},
            {"burge", "bh @r g"},
            {"burk", "bh @r k"},
            {"burke", "bh @r k"},
            {"burkes", "bh @r k s"},
            {"burke's", "bh @r k s"},
            {"burks", "bh @r k s"},
            {"burl", "bh @r l"},
            {"burn", "bh @r n"},
            {"burne", "bh @r n"},
            {"burned", "bh @r n d"},
            {"burnes", "bh @r n z"},
            {"burns", "bh @r n z"},
            {"burnsed", "bh @r n z d"},
            {"burnt", "bh @r n t"},
            {"burp", "bh @r p"},
            {"burr", "bh @r"},
            {"bursch", "bh @r S"},
            {"burse", "bh @r s"},
            {"burst", "bh @r s t"},
            {"bursts", "bh @r s t s"},
            {"burt", "bh @r t"},
            {"burtt", "bh @r t"},
            {"bus", "bh V s"},
            {"busch", "bh U S"},
            {"bused", "bh V s t"},
            {"bush", "bh U S"},
            {"busk", "bh V s k"},
            {"buss", "bh V s"},
            {"bussed", "bh V s t"},
            {"bust", "bh V s t"},
            {"busts", "bh V s t s"},
            {"but", "bh V t"},
            {"butch", "bh U tS"},
            {"bute", "bh j u: t"},
            {"buts", "bh V t s"},
            {"butt", "bh V t"},
            {"butte", "bh j u: t"},
            {"butts", "bh V t s"},
            {"buy", "bh aI"},
            {"buys", "bh aI z"},
            {"buzz", "bh V z"},
            {"buzzed", "bh V z d"},
            {"by", "bh aI"},
            {"byard", "bh j Q@ d"},
            {"bye", "bh aI"},
            {"byrd", "bh @r d"},
            {"byrd's", "bh @r d z"},
            {"byrn", "bh I@ n"},
            {"byrne", "bh @r n"},
            {"byrnes", "bh @r n z"},
            {"byrne's", "bh @r n z"},
            {"byte", "bh aI t"},
            {"bytes", "bh aI t s"},
            {"c", "s i:"},
            {"c.", "s i:"},
            {"ca", "kh V"},
            {"cab", "kh { b"},
            {"cabe", "kh eI b"},
            {"cabs", "kh { b z"},
            {"cab's", "kh { b z"},
            {"cac", "kh { k"},
            {"cache", "kh { S"},
            {"cad", "kh { d"},
            {"cade", "kh eI d"},
            {"cads", "kh { d s"},
            {"caen", "kh { n"},
            {"caen's", "kh { n z"},
            {"cage", "kh eI dZ"},
            {"caged", "kh eI dZ d"},
            {"cahn", "kh { n"},
            {"cai", "kh aI"},
            {"cain", "kh eI n"},
            {"caine", "kh eI n"},
            {"caines", "kh eI n z"},
            {"cain's", "kh eI n z"},
            {"cairns", "kh e@ n z"},
            {"caisse", "kh eI s"},
            {"cake", "kh eI k"},
            {"caked", "kh eI k t"},
            {"cakes", "kh eI k s"},
            {"cake's", "kh eI k s"},
            {"cal", "kh { l"},
            {"cale", "kh eI l"},
            {"calf", "kh { f"},
            {"calf's", "kh { f s"},
            {"calk", "kh O: k"},
            {"call", "kh O: l"},
            {"calle", "kh eI l"},
            {"called", "kh O: l d"},
            {"calles", "kh eI l z"},
            {"calls", "kh O: l z"},
            {"call's", "kh Q l z"},
            {"calm", "kh Q m"},
            {"calmed", "kh Q m d"},
            {"calms", "kh Q m z"},
            {"calves", "kh { v z"},
            {"cam", "kh { m"},
            {"came", "kh eI m"},
            {"camm", "kh { m"},
            {"camp", "kh { m p"},
            {"camped", "kh { m p t"},
            {"camps", "kh { m p s"},
            {"camp's", "kh { m p s"},
            {"cam's", "kh { m z"},
            {"can", "kh { n"},
            {"cane", "kh eI n"},
            {"caned", "kh eI n d"},
            {"canes", "kh eI n z"},
            {"cann", "kh { n"},
            {"canned", "kh { n d"},
            {"cannes", "kh { n z"},
            {"cans", "kh { n z"},
            {"can's", "kh { n z"},
            {"cant", "kh { n t"},
            {"can't", "kh { n t"},
            {"cap", "kh { p"},
            {"cape", "kh eI p"},
            {"caped", "kh eI p t"},
            {"capes", "kh eI p s"},
            {"capp", "kh { p"},
            {"capped", "kh { p t"},
            {"capps", "kh { p s"},
            {"caps", "kh { p s"},
            {"cap's", "kh { p s"},
            {"car", "kh Q@"},
            {"card", "kh Q@ d"},
            {"cards", "kh Q@ d z"},
            {"card's", "kh Q@ d z"},
            {"care", "kh e@"},
            {"cared", "kh e@ d"},
            {"cares", "kh e@ z"},
            {"care's", "kh e@ z"},
            {"carl", "kh Q@ l"},
            {"carles", "kh Q@ l z"},
            {"carls", "kh Q@ l z"},
            {"carl's", "kh Q@ l z"},
            {"carn", "kh Q@ n"},
            {"carne", "kh Q@ n"},
            {"carnes", "kh Q@ n z"},
            {"carns", "kh Q@ n z"},
            {"carp", "kh Q@ p"},
            {"carr", "kh Q@"},
            {"carr's", "kh Q@ z"},
            {"cars", "kh Q@ z"},
            {"car's", "kh Q@ z"},
            {"carse", "kh Q@ s"},
            {"cart", "kh Q@ t"},
            {"carte", "kh Q@ t"},
            {"carts", "kh Q@ t s"},
            {"carve", "kh Q@ v"},
            {"carved", "kh Q@ v d"},
            {"carves", "kh Q@ v z"},
            {"cas", "kh { s"},
            {"case", "kh eI s"},
            {"cash", "kh { S"},
            {"cashed", "kh { S t"},
            {"cask", "kh { s k"},
            {"casks", "kh { s k s"},
            {"cass", "kh { s"},
            {"cast", "kh { s t"},
            {"caste", "kh { s t"},
            {"castes", "kh { s t s"},
            {"casts", "kh { s t s"},
            {"cat", "kh { t"},
            {"catch", "kh { tS"},
            {"cate", "kh eI t"},
            {"cates", "kh eI t s"},
            {"cats", "kh { t s"},
            {"cat's", "kh { t s"},
            {"catt", "kh { t"},
            {"caught", "kh Q t"},
            {"caul", "kh Q l"},
            {"caulk", "kh Q k"},
            {"cause", "kh Q z"},
            {"caused", "kh Q z d"},
            {"cave", "kh eI v"},
            {"caved", "kh eI v d"},
            {"caves", "kh eI v z"},
            {"cave's", "kh eI v z"},
            {"caw", "kh O:"},
            {"cay", "kh eI"},
            {"cayce", "kh eI s"},
            {"cayes", "kh eI z"},
            {"cease", "s i: s"},
            {"ceased", "s i: s t"},
            {"cece", "s i: s"},
            {"cede", "s i: d"},
            {"cedes", "s i: d z"},
            {"cees", "s i: z"},
            {"cel", "s e l"},
            {"cele", "s i: l"},
            {"cell", "s e l"},
            {"celled", "s e l d"},
            {"cells", "s e l z"},
            {"cell's", "s e l z"},
            {"cels", "s e l z"},
            {"celt", "s e l t"},
            {"celts", "s e l t s"},
            {"cent", "s e n t"},
            {"cents", "s e n t s"},
            {"cep", "s e p"},
            {"cerf", "s @r f"},
            {"cern", "s @r n"},
            {"c'est", "s e s t"},
            {"cha", "tS Q"},
            {"chace", "tS eI s"},
            {"chad", "tS { d"},
            {"chadd", "tS { d"},
            {"chad's", "tS { d z"},
            {"chae", "tS aI"},
            {"chafe", "tS eI f"},
            {"chafed", "tS eI f t"},
            {"chafes", "tS eI f s"},
            {"chaff", "tS { f"},
            {"chai", "tS aI"},
            {"chain", "tS eI n"},
            {"chained", "tS eI n d"},
            {"chains", "tS eI n z"},
            {"chain's", "tS eI n z"},
            {"chair", "tS e@"},
            {"chaired", "tS e@ d"},
            {"chairs", "tS e@ z"},
            {"chaise", "S eI z"},
            {"chait", "tS eI t"},
            {"chalk", "tS Q k"},
            {"chalked", "tS O: k t"},
            {"chalks", "tS O: k s"},
            {"cham", "tS { m"},
            {"champ", "tS { m p"},
            {"champs", "tS { m p s"},
            {"chan", "tS { n"},
            {"chance", "tS { n s"},
            {"chang", "tS { N"},
            {"change", "tS eI n dZ"},
            {"changed", "tS eI n dZ d"},
            {"chan's", "tS { n z"},
            {"chant", "tS { n t"},
            {"chants", "tS { n t s"},
            {"chao", "tS aU"},
            {"chap", "tS { p"},
            {"chaps", "tS { p s"},
            {"char", "tS Q@"},
            {"chard", "tS Q@ d"},
            {"charge", "tS Q@ dZ"},
            {"charged", "tS Q@ dZ d"},
            {"charles", "tS Q@ l z"},
            {"charles'", "tS Q@ l z"},
            {"charm", "tS Q@ m"},
            {"charmed", "tS Q@ m d"},
            {"charms", "tS Q@ m z"},
            {"charred", "tS Q@ d"},
            {"chart", "tS Q@ t"},
            {"charts", "tS Q@ t s"},
            {"chas", "tS Q@ l z"},
            {"chase", "tS eI s"},
            {"chased", "tS eI s t"},
            {"chasse", "tS { s"},
            {"chaste", "tS eI s t"},
            {"chat", "tS { t"},
            {"chats", "tS { t s"},
            {"chaunce", "tS O: n s"},
            {"chaus", "tS aU s"},
            {"chausse", "tS aU s"},
            {"chaves", "tS eI v z"},
            {"chaw", "tS O:"},
            {"che", "tS eI"},
            {"cheap", "tS i: p"},
            {"cheat", "tS i: t"},
            {"cheats", "tS i: t s"},
            {"check", "tS e k"},
            {"checked", "tS e k t"},
            {"checks", "tS e k s"},
            {"chee", "tS i:"},
            {"cheek", "tS i: k"},
            {"cheeked", "tS i: k t"},
            {"cheeks", "tS i: k s"},
            {"cheer", "tS I@"},
            {"cheered", "tS I@ d"},
            {"cheers", "tS I@ z"},
            {"cheese", "tS i: z"},
            {"chef", "S e f"},
            {"chefs", "S e f s"},
            {"chef's", "S e f s"},
            {"chem", "kh e m"},
            {"chen", "tS e n"},
            {"cheng", "tS e N"},
            {"chen's", "tS e n z"},
            {"cher", "S e@"},
            {"chert", "tS @r t"},
            {"cherts", "tS @r t s"},
            {"ches", "tS e s"},
            {"chess", "tS e s"},
            {"chest", "tS e s t"},
            {"chests", "tS e s t s"},
            {"chet", "tS e t"},
            {"cheung", "tS j u: N"},
            {"cheves", "tS i: v z"},
            {"chew", "tS u:"},
            {"chewed", "tS u: d"},
            {"chews", "tS u: z"},
            {"cheyne", "tS eI n"},
            {"chez", "tS e z"},
            {"chi", "kh aI"},
            {"chiang", "tS { N"},
            {"chiang's", "tS { N z"},
            {"chic", "S i: k"},
            {"chick", "tS I k"},
            {"chicks", "tS I k s"},
            {"chide", "tS aI d"},
            {"chides", "tS aI d z"},
            {"chief", "tS i: f"},
            {"chiefs", "tS i: f s"},
            {"chief's", "tS i: f s"},
            {"chien", "tS e n"},
            {"child", "tS aI l d"},
            {"childe", "tS I l d"},
            {"childs", "tS aI l d z"},
            {"child's", "tS aI l d z"},
            {"chiles", "tS aI l z"},
            {"chill", "tS I l"},
            {"chilled", "tS I l d"},
            {"chills", "tS I l z"},
            {"chime", "tS aI m"},
            {"chimed", "tS aI m d"},
            {"chimes", "tS aI m z"},
            {"chimp", "tS I m p"},
            {"chimps", "tS I m p s"},
            {"chin", "tS I n"},
            {"ching", "tS I N"},
            {"chink", "tS I N k"},
            {"chinks", "tS I N k s"},
            {"chip", "tS I p"},
            {"chipped", "tS I p t"},
            {"chips", "tS I p s"},
            {"chip's", "tS I p s"},
            {"chirp", "tS @r p"},
            {"chirps", "tS @r p s"},
            {"chi's", "kh aI z"},
            {"chit", "tS I t"},
            {"chits", "tS I t s"},
            {"chives", "tS aI v z"},
            {"cho", "tS @U"},
            {"choate", "tS @U t"},
            {"chock", "tS Q k"},
            {"choi", "tS OI"},
            {"choice", "tS OI s"},
            {"choirs", "kh w aI r z"},
            {"chok", "tS Q k"},
            {"choke", "tS @U k"},
            {"choked", "tS @U k t"},
            {"chokes", "tS @U k s"},
            {"chomp", "tS Q m p"},
            {"chon", "tS Q n"},
            {"chong", "tS O: N"},
            {"choo", "tS u:"},
            {"choong", "tS u: N"},
            {"choose", "tS u: z"},
            {"chop", "tS Q p"},
            {"chopped", "tS Q p t"},
            {"chops", "tS Q p s"},
            {"chord", "kh O@ d"},
            {"chords", "kh O@ d z"},
            {"chore", "tS O@"},
            {"chores", "tS O@ z"},
            {"chose", "tS @U z"},
            {"chou", "tS u:"},
            {"chow", "tS aU"},
            {"chows", "tS aU z"},
            {"choy", "tS OI"},
            {"chris", "kh r I s"},
            {"christ", "kh r aI s t"},
            {"christ's", "kh r aI s t s"},
            {"chrome", "kh r @U m"},
            {"chu", "tS u:"},
            {"chubb", "tS V b"},
            {"chubb's", "tS V b z"},
            {"chuck", "tS V k"},
            {"chucked", "tS V k t"},
            {"chuck's", "tS V k s"},
            {"chug", "tS V g"},
            {"chugged", "tS V g d"},
            {"chum", "tS V m"},
            {"chump", "tS V m p"},
            {"chums", "tS V m z"},
            {"chun", "tS V n"},
            {"chung", "tS V N"},
            {"chung's", "tS V N z"},
            {"chunk", "tS V N k"},
            {"chunks", "tS V N k s"},
            {"chun's", "tS V n z"},
            {"church", "tS @r tS"},
            {"churn", "tS @r n"},
            {"churned", "tS @r n d"},
            {"churns", "tS @r n z"},
            {"chuse", "tS j u: z"},
            {"chute", "S u: t"},
            {"chutes", "S u: t s"},
            {"cid", "s I d"},
            {"cie", "s i:"},
            {"ciel", "s i: l"},
            {"cinch", "s I n tS"},
            {"cinched", "s I n tS t"},
            {"cinque", "s I N k"},
            {"cirque", "s @r k"},
            {"cist", "s I s t"},
            {"cite", "s aI t"},
            {"cites", "s aI t s"},
            {"clack", "kh l { k"},
            {"clad", "kh l { d"},
            {"claes", "kh l eI z"},
            {"claim", "kh l eI m"},
            {"claimed", "kh l eI m d"},
            {"claims", "kh l eI m z"},
            {"clair", "kh l e@"},
            {"claire", "kh l e@"},
            {"claire's", "kh l e@ z"},
            {"clam", "kh l { m"},
            {"clamp", "kh l { m p"},
            {"clamped", "kh l { m p t"},
            {"clamps", "kh l { m p s"},
            {"clams", "kh l { m z"},
            {"clan", "kh l { n"},
            {"clang", "kh l { N"},
            {"clank", "kh l { N k"},
            {"clans", "kh l { n z"},
            {"clap", "kh l { p"},
            {"clapp", "kh l { p"},
            {"clapped", "kh l { p t"},
            {"claps", "kh l { p s"},
            {"clar", "kh l Q@"},
            {"clare", "kh l e@"},
            {"clark", "kh l Q@ k"},
            {"clarke", "kh l Q@ k"},
            {"clarke's", "kh l Q@ k s"},
            {"clarks", "kh l Q@ k s"},
            {"clark's", "kh l Q@ k s"},
            {"clash", "kh l { S"},
            {"clashed", "kh l { S t"},
            {"clasp", "kh l { s p"},
            {"clasped", "kh l { s p t"},
            {"class", "kh l { s"},
            {"classed", "kh l { s t"},
            {"claud", "kh l O: d"},
            {"claude", "kh l O: d"},
            {"claunch", "kh l O: n tS"},
            {"claus", "kh l O: z"},
            {"clause", "kh l O: z"},
            {"clave", "kh l eI v"},
            {"claw", "kh l O:"},
            {"clawed", "kh l O: d"},
            {"claws", "kh l O: z"},
            {"clay", "kh l eI"},
            {"clays", "kh l eI z"},
            {"clean", "kh l i: n"},
            {"cleaned", "kh l i: n d"},
            {"cleans", "kh l i: n z"},
            {"cleanse", "kh l e n z"},
            {"cleansed", "kh l e n z d"},
            {"clear", "kh l I@"},
            {"cleared", "kh l I@ d"},
            {"clears", "kh l I@ z"},
            {"cleat", "kh l i: t"},
            {"cleats", "kh l i: t s"},
            {"cleave", "kh l i: v"},
            {"cleaves", "kh l i: v z"},
            {"cleek", "kh l i: k"},
            {"cleese", "kh l i: s"},
            {"clef", "kh l e f"},
            {"cleft", "kh l e f t"},
            {"clegg", "kh l e g"},
            {"clem", "kh l e m"},
            {"clench", "kh l e n tS"},
            {"clenched", "kh l e n tS t"},
            {"clerc", "kh l @r k"},
            {"clercq", "kh l @r k"},
            {"clerk", "kh l @r k"},
            {"clerks", "kh l @r k s"},
            {"clerk's", "kh l @r k s"},
            {"cleve", "kh l i: v"},
            {"clews", "kh l u: z"},
            {"click", "kh l I k"},
            {"clicked", "kh l I k t"},
            {"clicks", "kh l I k s"},
            {"cliff", "kh l I f"},
            {"cliffs", "kh l I f s"},
            {"cliff's", "kh l I f s"},
            {"clift", "kh l I f t"},
            {"clim", "kh l I m"},
            {"climb", "kh l aI m"},
            {"climbed", "kh l aI m d"},
            {"climbs", "kh l aI m z"},
            {"climes", "kh l aI m z"},
            {"clinch", "kh l I n tS"},
            {"clinched", "kh l I n tS t"},
            {"cline", "kh l aI n"},
            {"clines", "kh l aI n z"},
            {"cling", "kh l I N"},
            {"clings", "kh l I N z"},
            {"clink", "kh l I N k"},
            {"clint", "kh l I n t"},
            {"clip", "kh l I p"},
            {"clipped", "kh l I p t"},
            {"clips", "kh l I p s"},
            {"clique", "kh l i: k"},
            {"cliques", "kh l I k s"},
            {"clites", "kh l aI t s"},
            {"clive", "kh l aI v"},
            {"cloak", "kh l @U k"},
            {"cloaked", "kh l @U k t"},
            {"clock", "kh l Q k"},
            {"clocked", "kh l Q k t"},
            {"clocks", "kh l Q k s"},
            {"clock's", "kh l Q k s"},
            {"cloe", "kh l @U"},
            {"clog", "kh l Q g"},
            {"clogged", "kh l O: g d"},
            {"clogs", "kh l Q g z"},
            {"cloke", "kh l @U k"},
            {"clone", "kh l @U n"},
            {"cloned", "kh l @U n d"},
            {"clones", "kh l @U n z"},
            {"clos", "kh l Q s"},
            {"close", "kh l @U s"},
            {"closed", "kh l @U z d"},
            {"clot", "kh l Q t"},
            {"cloth", "kh l O: T"},
            {"clothe", "kh l @U D"},
            {"clothed", "kh l @U D d"},
            {"clothes", "kh l @U D z"},
            {"cloths", "kh l O: T s"},
            {"clots", "kh l Q t s"},
            {"cloud", "kh l aU d"},
            {"clouds", "kh l aU d z"},
            {"clough", "kh l aU"},
            {"clout", "kh l aU t"},
            {"cloves", "kh l @U v z"},
            {"clow", "kh l @U"},
            {"clown", "kh l aU n"},
            {"clowns", "kh l aU n z"},
            {"cloy", "kh l OI"},
            {"club", "kh l V b"},
            {"clubb", "kh l V b"},
            {"clubbed", "kh l V b d"},
            {"clubs", "kh l V b z"},
            {"club's", "kh l V b z"},
            {"cluck", "kh l V k"},
            {"clucks", "kh l V k s"},
            {"clue", "kh l u:"},
            {"clued", "kh l u: d"},
            {"clues", "kh l u: z"},
            {"cluff", "kh l V f"},
            {"cluj", "kh l u: dZ"},
            {"clum", "kh l V m"},
            {"clump", "kh l V m p"},
            {"clumps", "kh l V m p s"},
            {"clune", "kh l u: n"},
            {"clung", "kh l V N"},
            {"clunk", "kh l V N k"},
            {"clunks", "kh l V N k s"},
            {"clutch", "kh l V tS"},
            {"clutched", "kh l V tS t"},
            {"clute", "kh l u: t"},
            {"clyde", "kh l aI d"},
            {"clyte", "kh l aI t"},
            {"clyve", "kh l aI v"},
            {"co", "kh @U"},
            {"co.", "kh @U"},
            {"coach", "kh @U tS"},
            {"coached", "kh @U tS t"},
            {"coad", "kh @U d"},
            {"coal", "kh @U l"},
            {"coals", "kh @U l z"},
            {"coal's", "kh @U l z"},
            {"coan", "kh @U n"},
            {"coarse", "kh O@ s"},
            {"coast", "kh @U s t"},
            {"coasts", "kh @U s t s"},
            {"coast's", "kh @U s t s"},
            {"coat", "kh @U t"},
            {"coats", "kh @U t s"},
            {"coat's", "kh @U t s"},
            {"coax", "kh @U k s"},
            {"coaxed", "kh @U k s t"},
            {"cob", "kh Q b"},
            {"cobb", "kh Q b"},
            {"cobbs", "kh Q b z"},
            {"cobe", "kh @U b"},
            {"cobs", "kh Q b z"},
            {"cock", "kh Q k"},
            {"cocke", "kh @U k"},
            {"cocked", "kh Q k t"},
            {"cocks", "kh Q k s"},
            {"cod", "kh Q d"},
            {"codd", "kh Q d"},
            {"code", "kh @U d"},
            {"codes", "kh @U d z"},
            {"code's", "kh @U d z"},
            {"coe", "kh @U"},
            {"coeur", "kh u: r"},
            {"cog", "kh O: g"},
            {"cohn", "kh @U n"},
            {"coiffe", "kh OI f"},
            {"coiffed", "kh OI f t"},
            {"coil", "kh OI l"},
            {"coiled", "kh OI l d"},
            {"coils", "kh OI l z"},
            {"coin", "kh OI n"},
            {"coined", "kh OI n d"},
            {"coins", "kh OI n z"},
            {"coin's", "kh OI n z"},
            {"coit", "kh OI t"},
            {"coke", "kh @U k"},
            {"cokes", "kh @U k s"},
            {"coke's", "kh @U k s"},
            {"cold", "kh @U l d"},
            {"colds", "kh @U l d z"},
            {"cole", "kh @U l"},
            {"coles", "kh @U l z"},
            {"cole's", "kh @U l z"},
            {"coll", "kh Q l"},
            {"colt", "kh @U l t"},
            {"colts", "kh @U l t s"},
            {"colt's", "kh @U l t s"},
            {"com", "kh Q m"},
            {"comb", "kh @U m"},
            {"combe", "kh @U m"},
            {"combed", "kh @U m d"},
            {"combes", "kh @U m z"},
            {"combs", "kh @U m z"},
            {"come", "kh V m"},
            {"comes", "kh V m z"},
            {"comp", "kh Q m p"},
            {"coms", "kh Q m z"},
            {"com's", "kh Q m z"},
            {"con", "kh Q n"},
            {"conch", "kh Q n tS"},
            {"conde", "kh Q n d"},
            {"cone", "kh @U n"},
            {"cones", "kh @U n z"},
            {"cone's", "kh @U n z"},
            {"cong", "kh O: N"},
            {"conk", "kh Q N k"},
            {"conn", "kh Q n"},
            {"conn.", "kh Q n"},
            {"conned", "kh Q n d"},
            {"cons", "kh Q n z"},
            {"conte", "kh O: n t"},
            {"coo", "kh u:"},
            {"cook", "kh U k"},
            {"cooke", "kh U k"},
            {"cooked", "kh U k t"},
            {"cooks", "kh U k s"},
            {"cook's", "kh U k s"},
            {"cool", "kh u: l"},
            {"cooled", "kh u: l d"},
            {"cools", "kh u: l z"},
            {"coombe", "kh u: m b"},
            {"coombes", "kh u: m b z"},
            {"coombs", "kh u: m z"},
            {"coon", "kh u: n"},
            {"coons", "kh u: n z"},
            {"coop", "kh u: p"},
            {"cooped", "kh u: p t"},
            {"coors", "kh u: r z"},
            {"coos", "kh u: s"},
            {"coot", "kh u: t"},
            {"coots", "kh u: t s"},
            {"cop", "kh Q p"},
            {"cope", "kh @U p"},
            {"coped", "kh @U p t"},
            {"copes", "kh @U p s"},
            {"copp", "kh Q p"},
            {"copped", "kh Q p t"},
            {"copps", "kh Q p s"},
            {"cops", "kh Q p s"},
            {"cor", "kh O@"},
            {"cord", "kh O@ d"},
            {"cords", "kh O@ d z"},
            {"core", "kh O@"},
            {"cores", "kh O@ z"},
            {"cork", "kh O@ k"},
            {"corks", "kh O@ k s"},
            {"corn", "kh O@ n"},
            {"corns", "kh O@ n z"},
            {"corp", "kh O@ p"},
            {"corp.", "kh O@ p"},
            {"corps", "kh O@"},
            {"corpse", "kh O@ p s"},
            {"corr", "kh O@"},
            {"corse", "kh O@ s"},
            {"cort", "kh O@ t"},
            {"cortes", "kh O@ t s"},
            {"cos", "kh O: s"},
            {"cose", "kh @U z"},
            {"cosme", "kh @U z m"},
            {"coss", "kh O: s"},
            {"cost", "kh Q s t"},
            {"coste", "kh @U s t"},
            {"costs", "kh Q s t s"},
            {"cot", "kh Q t"},
            {"cote", "kh @U t"},
            {"cotes", "kh @U t s"},
            {"cots", "kh Q t s"},
            {"cott", "kh Q t"},
            {"couch", "kh aU tS"},
            {"couched", "kh aU tS t"},
            {"cough", "kh Q f"},
            {"coughed", "kh Q f t"},
            {"coughs", "kh O: f s"},
            {"could", "kh U d"},
            {"counce", "kh aU n s"},
            {"count", "kh aU n t"},
            {"counts", "kh aU n t s"},
            {"coup", "kh u:"},
            {"coupe", "kh u: p"},
            {"coupes", "kh u: p s"},
            {"coups", "kh u: z"},
            {"course", "kh O@ s"},
            {"court", "kh O@ t"},
            {"courts", "kh O@ t s"},
            {"court's", "kh O@ t s"},
            {"cove", "kh @U v"},
            {"coves", "kh @U v z"},
            {"cow", "kh aU"},
            {"cowed", "kh aU d"},
            {"cowl", "kh aU l"},
            {"cows", "kh aU z"},
            {"cow's", "kh aU z"},
            {"cox", "kh Q k s"},
            {"coy", "kh OI"},
            {"coyle", "kh OI l"},
            {"coz", "kh Q z"},
            {"crab", "kh r { b"},
            {"crabb", "kh r { b"},
            {"crabbe", "kh r { b"},
            {"crabbed", "kh r { b d"},
            {"crabs", "kh r { b z"},
            {"crack", "kh r { k"},
            {"cracked", "kh r { k t"},
            {"cracks", "kh r { k s"},
            {"craft", "kh r { f t"},
            {"crafts", "kh r { f t s"},
            {"craft's", "kh r { f t s"},
            {"craig", "kh r eI g"},
            {"craig's", "kh r eI g z"},
            {"crain", "kh r eI n"},
            {"crain's", "kh r eI n z"},
            {"cram", "kh r { m"},
            {"crames", "kh r eI m z"},
            {"crammed", "kh r { m d"},
            {"cramp", "kh r { m p"},
            {"cramped", "kh r { m p t"},
            {"cramps", "kh r { m p s"},
            {"crams", "kh r { m z"},
            {"crance", "kh r { n s"},
            {"crane", "kh r eI n"},
            {"craned", "kh r eI n d"},
            {"cranes", "kh r eI n z"},
            {"crane's", "kh r eI n z"},
            {"crank", "kh r { N k"},
            {"cranked", "kh r { N k t"},
            {"cranks", "kh r { N k s"},
            {"crans", "kh r { n z"},
            {"crap", "kh r { p"},
            {"craps", "kh r { p s"},
            {"crash", "kh r { S"},
            {"crashed", "kh r { S t"},
            {"crass", "kh r { s"},
            {"crate", "kh r eI t"},
            {"crates", "kh r eI t s"},
            {"crave", "kh r eI v"},
            {"craved", "kh r eI v d"},
            {"craves", "kh r eI v z"},
            {"craw", "kh r O:"},
            {"crawl", "kh r O: l"},
            {"crawled", "kh r O: l d"},
            {"crawls", "kh r O: l z"},
            {"cray", "kh r eI"},
            {"crayne", "kh r eI n"},
            {"crays", "kh r eI z"},
            {"cray's", "kh r eI z"},
            {"craze", "kh r eI z"},
            {"crazed", "kh r eI z d"},
            {"crea", "kh r i:"},
            {"creach", "kh r i: tS"},
            {"creagh", "kh r i: g"},
            {"creak", "kh r i: k"},
            {"creaked", "kh r i: k t"},
            {"cream", "kh r i: m"},
            {"creamed", "kh r i: m d"},
            {"creams", "kh r i: m z"},
            {"crean", "kh r i: n"},
            {"crease", "kh r i: s"},
            {"creath", "kh r e T"},
            {"cree", "kh r i:"},
            {"creed", "kh r i: d"},
            {"creeds", "kh r i: d z"},
            {"creek", "kh r i: k"},
            {"creeks", "kh r i: k s"},
            {"creek's", "kh r i: k s"},
            {"creel", "kh r i: l"},
            {"creep", "kh r i: p"},
            {"creeps", "kh r i: p s"},
            {"crees", "kh r i: z"},
            {"creme", "kh r i: m"},
            {"crepe", "kh r eI p"},
            {"crepes", "kh r eI p s"},
            {"crept", "kh r e p t"},
            {"cress", "kh r e s"},
            {"crest", "kh r e s t"},
            {"crests", "kh r e s t s"},
            {"crete", "kh r i: t"},
            {"crew", "kh r u:"},
            {"crewe", "kh r u:"},
            {"crewes", "kh r u: z"},
            {"crews", "kh r u: z"},
            {"crew's", "kh r u: z"},
            {"crib", "kh r I b"},
            {"cribs", "kh r I b z"},
            {"crick", "kh r I k"},
            {"cried", "kh r aI d"},
            {"cries", "kh r aI z"},
            {"crim", "kh r I m"},
            {"crime", "kh r aI m"},
            {"crimes", "kh r aI m z"},
            {"crime's", "kh r aI m z"},
            {"crimp", "kh r I m p"},
            {"crimped", "kh r I m p t"},
            {"crimps", "kh r I m p s"},
            {"cringe", "kh r I n dZ"},
            {"cringed", "kh r I n dZ d"},
            {"crip", "kh r I p"},
            {"cripe", "kh r aI p"},
            {"cripps", "kh r I p s"},
            {"crips", "kh r I p s"},
            {"cris", "kh r I s"},
            {"crise", "kh r aI z"},
            {"crisp", "kh r I s p"},
            {"criss", "kh r I s"},
            {"crist", "kh r I s t"},
            {"critz", "kh r I t s"},
            {"cro", "kh r @U"},
            {"croak", "kh r @U k"},
            {"croc", "kh r Q k"},
            {"crock", "kh r Q k"},
            {"croft", "kh r O: f t"},
            {"crofts", "kh r O: f t s"},
            {"croix", "kh r OI"},
            {"croix's", "kh r OI z"},
            {"crom", "kh r Q m"},
            {"crone", "kh r @U n"},
            {"cronk", "kh r Q N k"},
            {"crook", "kh r U k"},
            {"crooks", "kh r U k s"},
            {"croom", "kh r u: m"},
            {"croon", "kh r u: n"},
            {"croons", "kh r u: n z"},
            {"crop", "kh r Q p"},
            {"cropped", "kh r Q p t"},
            {"crops", "kh r Q p s"},
            {"crop's", "kh r Q p s"},
            {"crose", "kh r @U z"},
            {"cross", "kh r O: s"},
            {"crosse", "kh r Q s"},
            {"crossed", "kh r O: s t"},
            {"crotch", "kh r Q tS"},
            {"crouch", "kh r aU tS"},
            {"crouched", "kh r aU tS t"},
            {"croup", "kh r u: p"},
            {"crouse", "kh r aU s"},
            {"crout", "kh r aU t"},
            {"crow", "kh r @U"},
            {"crowd", "kh r aU d"},
            {"crowds", "kh r aU d z"},
            {"crowd's", "kh r aU d z"},
            {"crowe", "kh r @U"},
            {"crowed", "kh r @U d"},
            {"crowl", "kh r aU l"},
            {"crown", "kh r aU n"},
            {"crowned", "kh r aU n d"},
            {"crowns", "kh r aU n z"},
            {"crown's", "kh r aU n z"},
            {"crows", "kh r @U z"},
            {"crow's", "kh r @U z"},
            {"croy", "kh r OI"},
            {"cruce", "kh r u: s"},
            {"crud", "kh r V d"},
            {"crude", "kh r u: d"},
            {"crudes", "kh r u: d z"},
            {"cruise", "kh r u: z"},
            {"cruised", "kh r u: z d"},
            {"crull", "kh r V l"},
            {"crum", "kh r V m"},
            {"crumb", "kh r V m"},
            {"crumbs", "kh r V m z"},
            {"crump", "kh r V m p"},
            {"crunch", "kh r V n tS"},
            {"crunched", "kh r V n tS t"},
            {"crunk", "kh r V N k"},
            {"cruse", "kh r u: z"},
            {"crush", "kh r V S"},
            {"crushed", "kh r V S t"},
            {"crust", "kh r V s t"},
            {"crusts", "kh r V s t s"},
            {"crutch", "kh r V tS"},
            {"crux", "kh r V k s"},
            {"cruz", "kh r u: z"},
            {"cry", "kh r aI"},
            {"crypt", "kh r I p t"},
            {"crypts", "kh r I p t s"},
            {"c's", "s i: z"},
            {"csar", "z Q@"},
            {"ct", "kh O@ t"},
            {"cub", "kh V b"},
            {"cube", "kh j u: b"},
            {"cubed", "kh j u: b d"},
            {"cubes", "kh j u: b z"},
            {"cubs", "kh V b z"},
            {"cue", "kh j u:"},
            {"cued", "kh j u: d"},
            {"cues", "kh j u: z"},
            {"cuff", "kh V f"},
            {"cuffed", "kh V f t"},
            {"cuffs", "kh V f s"},
            {"cul", "kh V l"},
            {"cull", "kh V l"},
            {"culled", "kh V l d"},
            {"culp", "kh V l p"},
            {"cult", "kh V l t"},
            {"cults", "kh V l t s"},
            {"cult's", "kh V l t s"},
            {"cum", "kh V m"},
            {"cup", "kh V p"},
            {"cups", "kh V p s"},
            {"cur", "kh @r"},
            {"curb", "kh @r b"},
            {"curbed", "kh @r b d"},
            {"curbs", "kh @r b z"},
            {"curd", "kh @r d"},
            {"cure", "kh j U@"},
            {"cured", "kh j U@ d"},
            {"cures", "kh j U@ z"},
            {"curl", "kh @r l"},
            {"curled", "kh @r l d"},
            {"curls", "kh @r l z"},
            {"curse", "kh @r s"},
            {"cursed", "kh @r s t"},
            {"curt", "kh @r t"},
            {"curt's", "kh @r t s"},
            {"curve", "kh @r v"},
            {"curved", "kh @r v d"},
            {"curves", "kh @r v z"},
            {"cush", "kh V S"},
            {"cusk", "kh V s k"},
            {"cusp", "kh V s p"},
            {"cuss", "kh V s"},
            {"cussed", "kh V s t"},
            {"cut", "kh V t"},
            {"cute", "kh j u: t"},
            {"cuts", "kh V t s"},
            {"cy", "s aI"},
            {"cyb", "s aI b"},
            {"cyd", "s I d"},
            {"cynth", "s I n T"},
            {"cyst", "s I s t"},
            {"cysts", "s I s t s"},
            {"czar", "z Q@"},
            {"czars", "z Q@ z"},
            {"czar's", "z Q@ z"},
            {"czech", "tS e k"},
            {"czechs", "tS e k s"},
            {"d", "dh i:"},
            {"d.", "dh i:"},
            {"da", "dh Q"},
            {"dab", "dh { b"},
            {"dabbs", "dh { b z"},
            {"dac", "dh { k"},
            {"dace", "dh eI s"},
            {"dacs", "dh { k s"},
            {"dad", "dh { d"},
            {"dade", "dh eI d"},
            {"dads", "dh { d z"},
            {"dad's", "dh { d z"},
            {"dae", "dh eI"},
            {"daft", "dh { f t"},
            {"dag", "dh { g"},
            {"dah", "dh V"},
            {"dahl", "dh Q l"},
            {"dahl's", "dh Q l z"},
            {"dahms", "dh Q m z"},
            {"dai", "dh aI"},
            {"dail", "dh eI l"},
            {"daile", "dh eI l"},
            {"dain", "dh eI n"},
            {"daines", "dh eI n z"},
            {"dais", "dh eI z"},
            {"dak", "dh { k"},
            {"dal", "dh { l"},
            {"dale", "dh eI l"},
            {"dales", "dh eI l z"},
            {"dale's", "dh eI l z"},
            {"dall", "dh O: l"},
            {"dalpe", "dh eI l p"},
            {"dam", "dh { m"},
            {"dame", "dh eI m"},
            {"dames", "dh eI m z"},
            {"dame's", "dh eI m z"},
            {"damme", "dh { m"},
            {"dammed", "dh { m d"},
            {"damn", "dh { m"},
            {"damned", "dh { m d"},
            {"damp", "dh { m p"},
            {"damped", "dh { m p t"},
            {"damps", "dh { m p s"},
            {"dams", "dh { m z"},
            {"dam's", "dh { m z"},
            {"dan", "dh { n"},
            {"dance", "dh { n s"},
            {"danced", "dh { n s t"},
            {"dane", "dh eI n"},
            {"danes", "dh eI n z"},
            {"dane's", "dh eI n z"},
            {"dang", "dh { N"},
            {"dank", "dh { N k"},
            {"danks", "dh { N k s"},
            {"dann", "dh { n"},
            {"dan's", "dh { n z"},
            {"dant", "dh { n t"},
            {"dao", "dh aU"},
            {"dar", "dh Q@"},
            {"dare", "dh e@"},
            {"dared", "dh e@ d"},
            {"dares", "dh e@ z"},
            {"dark", "dh Q@ k"},
            {"darn", "dh Q@ n"},
            {"darned", "dh Q@ n d"},
            {"darns", "dh Q@ n z"},
            {"darr", "dh e@"},
            {"darst", "dh Q@ s t"},
            {"dart", "dh Q@ t"},
            {"darth", "dh Q@ T"},
            {"darts", "dh Q@ t s"},
            {"dart's", "dh Q@ t s"},
            {"das", "dh { s"},
            {"dase", "dh eI z"},
            {"dash", "dh { S"},
            {"dashed", "dh { S t"},
            {"dat", "dh { t"},
            {"date", "dh eI t"},
            {"dates", "dh eI t s"},
            {"dau", "dh @U"},
            {"daub", "dh O: b"},
            {"daube", "dh O: b"},
            {"daubed", "dh O: b d"},
            {"dault", "dh O: l t"},
            {"daun", "dh O: n"},
            {"daunt", "dh O: n t"},
            {"dave", "dh eI v"},
            {"dave's", "dh eI v z"},
            {"daw", "dh O:"},
            {"dawe", "dh O:"},
            {"dawes", "dh O: z"},
            {"dawn", "dh O: n"},
            {"dawned", "dh O: n d"},
            {"dawns", "dh O: n z"},
            {"dawn's", "dh O: n z"},
            {"daws", "dh O: z"},
            {"dax", "dh { k s"},
            {"day", "dh eI"},
            {"daye", "dh eI"},
            {"dayle", "dh eI l"},
            {"days", "dh eI z"},
            {"day's", "dh eI z"},
            {"days'", "dh eI z"},
            {"daze", "dh eI z"},
            {"dazed", "dh eI z d"},
            {"de", "dh i:"},
            {"dea", "dh i:"},
            {"dead", "dh e d"},
            {"deaf", "dh e f"},
            {"deal", "dh i: l"},
            {"deale", "dh i: l"},
            {"deals", "dh i: l z"},
            {"dealt", "dh e l t"},
            {"dean", "dh i: n"},
            {"deane", "dh i: n"},
            {"deans", "dh i: n z"},
            {"dean's", "dh i: n z"},
            {"dear", "dh I@"},
            {"dearth", "dh @r T"},
            {"deas", "dh i: z"},
            {"death", "dh e T"},
            {"deaths", "dh e T s"},
            {"death's", "dh e T s"},
            {"deb", "dh e b"},
            {"debes", "dh i: b z"},
            {"debs", "dh e b z"},
            {"debt", "dh e t"},
            {"debts", "dh e t s"},
            {"debt's", "dh e t s"},
            {"dec", "dh e k"},
            {"deck", "dh e k"},
            {"decked", "dh e k t"},
            {"decks", "dh e k s"},
            {"dec's", "dh e k s"},
            {"dede", "dh i: d"},
            {"dee", "dh i:"},
            {"deed", "dh i: d"},
            {"deeds", "dh i: d z"},
            {"deem", "dh i: m"},
            {"deemed", "dh i: m d"},
            {"deems", "dh i: m z"},
            {"deen", "dh i: n"},
            {"deep", "dh i: p"},
            {"deer", "dh I@"},
            {"deere", "dh I@"},
            {"deere's", "dh I@ z"},
            {"dees", "dh i: z"},
            {"dee's", "dh i: z"},
            {"deets", "dh i: t s"},
            {"deft", "dh e f t"},
            {"deis", "dh i: z"},
            {"del", "dh e l"},
            {"delfs", "dh e l f s"},
            {"delft", "dh e l f t"},
            {"dell", "dh e l"},
            {"delle", "dh e l"},
            {"dell's", "dh e l z"},
            {"delp", "dh e l p"},
            {"delph", "dh e l f"},
            {"delve", "dh e l v"},
            {"delved", "dh e l v d"},
            {"delves", "dh e l v z"},
            {"den", "dh e n"},
            {"denes", "dh i: n z"},
            {"deng", "dh e N"},
            {"deng's", "dh e N z"},
            {"dengue", "dh e n g"},
            {"denn", "dh e n"},
            {"dens", "dh e n z"},
            {"dense", "dh e n s"},
            {"dent", "dh e n t"},
            {"dente", "dh e n t"},
            {"dents", "dh e n t s"},
            {"depp", "dh e p"},
            {"depth", "dh e p T"},
            {"depths", "dh e p T s"},
            {"der", "dh @r"},
            {"derk", "dh @r k"},
            {"dern", "dh @r n"},
            {"derr", "dh e@"},
            {"des", "dh e"},
            {"desk", "dh e s k"},
            {"desks", "dh e s k s"},
            {"dest", "dh e s t"},
            {"deuce", "dh u: s"},
            {"deutsch", "dh OI tS"},
            {"deutsche", "dh OI tS"},
            {"deux", "dh u:"},
            {"dev", "dh e v"},
            {"dew", "dh u:"},
            {"dews", "dh u: z"},
            {"dex", "dh e k s"},
            {"dey", "dh eI"},
            {"dhole", "dh @U l"},
            {"dhows", "dh aU z"},
            {"di", "dh i:"},
            {"dibb", "dh I b"},
            {"dibs", "dh I b z"},
            {"dic", "dh I k"},
            {"dice", "dh aI s"},
            {"diced", "dh aI s t"},
            {"dick", "dh I k"},
            {"dicks", "dh I k s"},
            {"dick's", "dh I k s"},
            {"did", "dh I d"},
            {"die", "dh aI"},
            {"died", "dh aI d"},
            {"diehl", "dh i: l"},
            {"diel", "dh i: l"},
            {"diem", "dh i: m"},
            {"dierks", "dh i: r k s"},
            {"dies", "dh aI z"},
            {"dietsche", "dh i: tS"},
            {"dietz", "dh i: t s"},
            {"diff", "dh I f"},
            {"dig", "dh I g"},
            {"diggs", "dh I g z"},
            {"digs", "dh I g z"},
            {"dike", "dh aI k"},
            {"dikes", "dh aI k s"},
            {"dilks", "dh I l k s"},
            {"dill", "dh I l"},
            {"dille", "dh I l"},
            {"dills", "dh I l z"},
            {"dim", "dh I m"},
            {"dime", "dh aI m"},
            {"dimes", "dh aI m z"},
            {"dime's", "dh aI m z"},
            {"dimmed", "dh I m d"},
            {"dims", "dh I m z"},
            {"din", "dh I n"},
            {"dine", "dh aI n"},
            {"dined", "dh aI n d"},
            {"dines", "dh aI n z"},
            {"ding", "dh I N"},
            {"dings", "dh I N z"},
            {"dinh", "dh I n"},
            {"dink", "dh I N k"},
            {"dinse", "dh I n s"},
            {"dint", "dh I n t"},
            {"dip", "dh I p"},
            {"dipped", "dh I p t"},
            {"dips", "dh I p s"},
            {"dire", "dh aI r"},
            {"dirge", "dh @r dZ"},
            {"dirk", "dh @r k"},
            {"dirkes", "dh @r k s"},
            {"dirks", "dh @r k s"},
            {"dirt", "dh @r t"},
            {"dirt's", "dh @r t s"},
            {"dis", "dh I s"},
            {"di's", "dh aI z"},
            {"disc", "dh I s k"},
            {"discs", "dh I s k s"},
            {"dish", "dh I S"},
            {"dished", "dh I S t"},
            {"disk", "dh I s k"},
            {"disks", "dh I s k s"},
            {"diss", "dh I s"},
            {"ditch", "dh I tS"},
            {"ditched", "dh I tS t"},
            {"dive", "dh aI v"},
            {"dived", "dh aI v d"},
            {"dives", "dh aI v z"},
            {"dix", "dh I k s"},
            {"do", "dh u:"},
            {"doak", "dh @U k"},
            {"doane", "dh @U n"},
            {"dob", "dh Q b"},
            {"dobb", "dh Q b"},
            {"dobbs", "dh Q b z"},
            {"doc", "dh Q k"},
            {"dock", "dh Q k"},
            {"docked", "dh Q k t"},
            {"docks", "dh Q k s"},
            {"dock's", "dh Q k s"},
            {"dodd", "dh Q d"},
            {"dodds", "dh Q d z"},
            {"dodd's", "dh Q d z"},
            {"dodge", "dh Q dZ"},
            {"dodged", "dh Q dZ d"},
            {"doe", "dh @U"},
            {"doer", "dh u: r"},
            {"does", "dh V z"},
            {"doe's", "dh @U z"},
            {"d'oeuvre", "dh @r v"},
            {"doff", "dh O: f"},
            {"doffs", "dh O: f s"},
            {"dog", "dh Q g"},
            {"doge", "dh @U dZ"},
            {"dogged", "dh O: g d"},
            {"dogs", "dh Q g z"},
            {"dog's", "dh O: g z"},
            {"doh", "dh @U"},
            {"doi", "dh OI"},
            {"doig", "dh OI g"},
            {"doke", "dh @U k"},
            {"dol", "dh Q l"},
            {"dole", "dh @U l"},
            {"doled", "dh @U l d"},
            {"doles", "dh @U l z"},
            {"dole's", "dh @U l z"},
            {"dolf", "dh @U l f"},
            {"doll", "dh Q l"},
            {"dolled", "dh Q l d"},
            {"dolls", "dh Q l z"},
            {"doll's", "dh Q l z"},
            {"dolph", "dh @U l f"},
            {"dols", "dh Q l z"},
            {"dom", "dh Q m"},
            {"dome", "dh @U m"},
            {"domed", "dh @U m d"},
            {"domes", "dh @U m z"},
            {"dome's", "dh @U m z"},
            {"don", "dh Q n"},
            {"done", "dh V n"},
            {"dong", "dh O: N"},
            {"donn", "dh Q n"},
            {"donne", "dh V n"},
            {"donned", "dh Q n d"},
            {"dons", "dh Q n z"},
            {"don's", "dh Q n z"},
            {"don't", "dh @U n t"},
            {"don'ts", "dh @U n t s"},
            {"doo", "dh u:"},
            {"doom", "dh u: m"},
            {"doomed", "dh u: m d"},
            {"dooms", "dh u: m z"},
            {"door", "dh O@"},
            {"doorn", "dh O@ n"},
            {"doors", "dh O@ z"},
            {"door's", "dh O@ z"},
            {"dope", "dh @U p"},
            {"doped", "dh @U p t"},
            {"dopp", "dh Q p"},
            {"d'or", "dh O@"},
            {"dore", "dh O@"},
            {"dorm", "dh O@ m"},
            {"dorms", "dh O@ m z"},
            {"dorn", "dh O@ n"},
            {"dorr", "dh O@"},
            {"dort", "dh O@ t"},
            {"dos", "dh O: s"},
            {"do's", "dh u: z"},
            {"dose", "dh @U s"},
            {"dosh", "dh Q S"},
            {"doss", "dh O: s"},
            {"dost", "dh Q s t"},
            {"dot", "dh Q t"},
            {"dote", "dh @U t"},
            {"doth", "dh O: T"},
            {"dots", "dh Q t s"},
            {"dott", "dh Q t"},
            {"doub", "dh aU b"},
            {"doubt", "dh aU t"},
            {"doubts", "dh aU t s"},
            {"douds", "dh aU d z"},
            {"doug", "dh V g"},
            {"dough", "dh @U"},
            {"doug's", "dh V g z"},
            {"douse", "dh aU s"},
            {"doused", "dh aU s t"},
            {"dov", "dh Q v"},
            {"dove", "dh V v"},
            {"doves", "dh V v z"},
            {"dow", "dh aU"},
            {"dowd", "dh aU d"},
            {"dowds", "dh aU d z"},
            {"dower", "dh aU r"},
            {"dowers", "dh aU r z"},
            {"down", "dh aU n"},
            {"downe", "dh aU n"},
            {"downed", "dh aU n d"},
            {"downes", "dh aU n z"},
            {"downs", "dh aU n z"},
            {"down's", "dh aU n z"},
            {"dow's", "dh aU z"},
            {"dowse", "dh aU s"},
            {"doyle", "dh OI l"},
            {"doyle's", "dh OI l z"},
            {"doze", "dh @U z"},
            {"dozed", "dh @U z d"},
            {"dr", "dh r aI v"},
            {"dr.", "dh r aI v"},
            {"drab", "dh r { b"},
            {"drabbed", "dh r { b d"},
            {"drabs", "dh r { b z"},
            {"draft", "dh r { f t"},
            {"drafts", "dh r { f t s"},
            {"draft's", "dh r { f t s"},
            {"drag", "dh r { g"},
            {"dragged", "dh r { g d"},
            {"drags", "dh r { g z"},
            {"drain", "dh r eI n"},
            {"draine", "dh r eI n"},
            {"drained", "dh r eI n d"},
            {"drains", "dh r eI n z"},
            {"drake", "dh r eI k"},
            {"drakes", "dh r eI k s"},
            {"dram", "dh r { m"},
            {"drams", "dh r { m z"},
            {"drang", "dh r { N"},
            {"drank", "dh r { N k"},
            {"drape", "dh r eI p"},
            {"draped", "dh r eI p t"},
            {"drapes", "dh r eI p s"},
            {"draught", "dh r { f t"},
            {"draughts", "dh r { f t s"},
            {"draw", "dh r O:"},
            {"drawer", "dh r O@"},
            {"drawers", "dh r O@ z"},
            {"drawl", "dh r O: l"},
            {"drawled", "dh r O: l d"},
            {"drawls", "dh r O: l z"},
            {"drawn", "dh r O: n"},
            {"draws", "dh r O: z"},
            {"dray", "dh r eI"},
            {"dread", "dh r e d"},
            {"dreads", "dh r e d z"},
            {"dream", "dh r i: m"},
            {"dreamed", "dh r i: m d"},
            {"dreams", "dh r i: m z"},
            {"dreamt", "dh r e m t"},
            {"dred", "dh r e d"},
            {"dredge", "dh r e dZ"},
            {"dredged", "dh r e dZ d"},
            {"drees", "dh r i: z"},
            {"dregs", "dh r e g z"},
            {"dreher", "dh r e@"},
            {"drench", "dh r e n tS"},
            {"drenched", "dh r e n tS t"},
            {"dress", "dh r e s"},
            {"dressed", "dh r e s t"},
            {"drew", "dh r u:"},
            {"drews", "dh r u: z"},
            {"drey", "dh r eI"},
            {"dribbed", "dh r I b d"},
            {"dribs", "dh r I b z"},
            {"dried", "dh r aI d"},
            {"dries", "dh r aI z"},
            {"drift", "dh r I f t"},
            {"drifts", "dh r I f t s"},
            {"driggs", "dh r I g z"},
            {"drill", "dh r I l"},
            {"drilled", "dh r I l d"},
            {"drills", "dh r I l z"},
            {"drink", "dh r I N k"},
            {"drinks", "dh r I N k s"},
            {"drip", "dh r I p"},
            {"dripped", "dh r I p t"},
            {"dripps", "dh r I p s"},
            {"drips", "dh r I p s"},
            {"drive", "dh r aI v"},
            {"drives", "dh r aI v z"},
            {"drogue", "dh r @U g"},
            {"droll", "dh r @U l"},
            {"drone", "dh r @U n"},
            {"droned", "dh r @U n d"},
            {"drones", "dh r @U n z"},
            {"drool", "dh r u: l"},
            {"droop", "dh r u: p"},
            {"drooped", "dh r u: p t"},
            {"drop", "dh r Q p"},
            {"dropped", "dh r Q p t"},
            {"drops", "dh r Q p s"},
            {"dross", "dh r O: s"},
            {"drought", "dh r aU t"},
            {"droughts", "dh r aU t s"},
            {"drought's", "dh r aU t s"},
            {"drove", "dh r @U v"},
            {"droves", "dh r @U v z"},
            {"drown", "dh r aU n"},
            {"drowned", "dh r aU n d"},
            {"drowns", "dh r aU n z"},
            {"dru", "dh r u:"},
            {"drub", "dh r V b"},
            {"drubbed", "dh r V b d"},
            {"druce", "dh r u: s"},
            {"drudge", "dh r V dZ"},
            {"drug", "dh r V g"},
            {"drugged", "dh r V g d"},
            {"drugs", "dh r V g z"},
            {"drug's", "dh r V g z"},
            {"drum", "dh r V m"},
            {"drummed", "dh r V m d"},
            {"drums", "dh r V m z"},
            {"drunk", "dh r V N k"},
            {"drunks", "dh r V N k s"},
            {"drupe", "dh r u: p"},
            {"drupes", "dh r u: p s"},
            {"druse", "dh r u: z"},
            {"druze", "dh r u: z"},
            {"dry", "dh r aI"},
            {"drye", "dh r aI"},
            {"d's", "dh i: z"},
            {"du", "dh u:"},
            {"duan", "dh w { n"},
            {"duane", "dh w eI n"},
            {"dub", "dh V b"},
            {"dubbed", "dh V b d"},
            {"dubs", "dh V b z"},
            {"duc", "dh V k"},
            {"duce", "dh u: s"},
            {"duck", "dh V k"},
            {"ducked", "dh V k t"},
            {"ducks", "dh V k s"},
            {"duck's", "dh V k s"},
            {"duct", "dh V k t"},
            {"ducts", "dh V k t s"},
            {"dud", "dh V d"},
            {"dude", "dh u: d"},
            {"dudes", "dh j u: d z"},
            {"duds", "dh V d z"},
            {"due", "dh u:"},
            {"dues", "dh u: z"},
            {"duff", "dh V f"},
            {"dug", "dh V g"},
            {"duh", "dh V"},
            {"duhr", "dh @r"},
            {"duke", "dh u: k"},
            {"dukes", "dh u: k s"},
            {"duke's", "dh u: k s"},
            {"dulce", "dh V l s"},
            {"dule", "dh u: l"},
            {"dull", "dh V l"},
            {"dulled", "dh V l d"},
            {"dum", "dh V m"},
            {"dumb", "dh V m"},
            {"dumm", "dh V m"},
            {"dump", "dh V m p"},
            {"dumped", "dh V m p t"},
            {"dumps", "dh V m p s"},
            {"dun", "dh V n"},
            {"dunc", "dh V N k"},
            {"dune", "dh u: n"},
            {"dunes", "dh u: n z"},
            {"dung", "dh V N"},
            {"dunk", "dh V N k"},
            {"dunked", "dh V N k t"},
            {"dunks", "dh V N k s"},
            {"dunn", "dh V n"},
            {"dunne", "dh V n"},
            {"dunned", "dh V n d"},
            {"duns", "dh V n z"},
            {"dun's", "dh V n z"},
            {"dunst", "dh V n s t"},
            {"dupe", "dh u: p"},
            {"duped", "dh u: p t"},
            {"dupes", "dh u: p s"},
            {"duque", "dh u: k"},
            {"dur", "dh @r"},
            {"durn", "dh @r n"},
            {"durr", "dh @r"},
            {"durst", "dh @r s t"},
            {"dush", "dh V S"},
            {"dusk", "dh V s k"},
            {"dust", "dh V s t"},
            {"dusts", "dh V s t s"},
            {"dutch", "dh V tS"},
            {"duwe", "dh u: w"},
            {"dux", "dh V k s"},
            {"dwan", "dh w Q n"},
            {"dwarf", "dh w O@ f"},
            {"dwarfed", "dh w O@ f t"},
            {"dwarfs", "dh w O@ f s"},
            {"dwarves", "dh w O@ v z"},
            {"dwayne", "dh w eI n"},
            {"dweck", "dh w e k"},
            {"dwell", "dh w e l"},
            {"dwelled", "dh w e l d"},
            {"dwells", "dh w e l z"},
            {"dwelt", "dh w e l t"},
            {"dwight", "dh w aI t"},
            {"dyce", "dh aI s"},
            {"dyche", "dh aI tS"},
            {"dyck", "dh aI k"},
            {"dye", "dh aI"},
            {"dyed", "dh aI d"},
            {"dyes", "dh aI z"},
            {"dyke", "dh aI k"},
            {"dykes", "dh aI k s"},
            {"dynes", "dh aI n z"},
            {"e", "i:"},
            {"e.", "i:"},
            {"each", "i: tS"},
            {"eade", "i: d"},
            {"eads", "i: d z"},
            {"eales", "i: l z"},
            {"eanes", "i: n z"},
            {"ear", "I@"},
            {"eared", "I@ d"},
            {"earl", "@r l"},
            {"earle", "@r l"},
            {"earls", "@r l z"},
            {"earn", "@r n"},
            {"earned", "@r n d"},
            {"earns", "@r n z"},
            {"earp", "@r p"},
            {"ears", "I@ z"},
            {"earth", "@r T"},
            {"earths", "@r T s"},
            {"earth's", "@r T s"},
            {"ease", "i: z"},
            {"eased", "i: z d"},
            {"east", "i: s t"},
            {"east's", "i: s t s"},
            {"eat", "i: t"},
            {"eats", "i: t s"},
            {"eau", "@U"},
            {"eaux", "@U z"},
            {"eave", "i: v"},
            {"eaves", "i: v z"},
            {"eb", "e b"},
            {"ebb", "e b"},
            {"ebbed", "e b d"},
            {"ebbs", "e b z"},
            {"eck", "e k"},
            {"ed", "e d"},
            {"ede", "i: d"},
            {"edes", "i: d z"},
            {"edge", "e dZ"},
            {"edged", "e dZ d"},
            {"ed's", "e d z"},
            {"ee", "i:"},
            {"eeg", "i: g"},
            {"eel", "i: l"},
            {"eels", "i: l z"},
            {"ege", "i: dZ"},
            {"egg", "e g"},
            {"egged", "e g d"},
            {"eggs", "e g z"},
            {"eh", "e"},
            {"ehlke", "e l k"},
            {"eide", "aI d"},
            {"eight", "eI t"},
            {"eighth", "eI t T"},
            {"eighths", "eI t T s"},
            {"eights", "eI t s"},
            {"eight's", "eI t s"},
            {"ein", "aI n"},
            {"eir", "aI r"},
            {"eis", "aI z"},
            {"eke", "i: k"},
            {"eked", "i: k t"},
            {"el", "e l"},
            {"elbe", "e l b"},
            {"elf", "e l f"},
            {"elf's", "e l f s"},
            {"elk", "e l k"},
            {"elks", "e l k s"},
            {"ell", "e l"},
            {"elle", "e l"},
            {"ells", "e l z"},
            {"elm", "e l m"},
            {"elms", "e l m z"},
            {"els", "e l z"},
            {"else", "e l s"},
            {"elves", "e l v z"},
            {"em", "e m"},
            {"empt", "e m p t"},
            {"en", "e n"},
            {"end", "e n d"},
            {"ende", "e n d"},
            {"ends", "e n d z"},
            {"eng", "e N"},
            {"ent", "e n t"},
            {"epes", "i: p s"},
            {"epp", "e p"},
            {"eppes", "e p s"},
            {"epps", "e p s"},
            {"er", "@r"},
            {"erb", "@r b"},
            {"erbe", "@r b"},
            {"erbes", "@r b z"},
            {"erk", "@r k"},
            {"erl", "@r l"},
            {"erne", "@r n"},
            {"ernst", "@r n s t"},
            {"err", "e@"},
            {"erred", "e@ d"},
            {"errs", "e@ z"},
            {"ers", "@r z"},
            {"es", "e s"},
            {"e's", "i: z"},
            {"esch", "e S"},
            {"esh", "e S"},
            {"esme", "e z m"},
            {"ess", "e s"},
            {"esse", "e s"},
            {"este", "e s t"},
            {"et", "e t"},
            {"etch", "e tS"},
            {"etched", "e tS t"},
            {"eure", "@r"},
            {"ev", "e v"},
            {"eve", "i: v"},
            {"eves", "i: v z"},
            {"ewe", "j u:"},
            {"ewes", "j u: z"},
            {"ex", "e k s"},
            {"eyde", "eI d"},
            {"eye", "aI"},
            {"eyed", "aI d"},
            {"eyes", "aI z"},
            {"eye's", "aI z"},
            {"eyre", "e@"},
            {"f", "e f"},
            {"f.", "e f"},
            {"fab", "f { b"},
            {"fac", "f { k"},
            {"face", "f eI s"},
            {"faced", "f eI s t"},
            {"fact", "f { k t"},
            {"facts", "f { k t s"},
            {"fad", "f { d"},
            {"fade", "f eI d"},
            {"fades", "f eI d z"},
            {"fads", "f { d z"},
            {"fae", "f aI"},
            {"fag", "f { g"},
            {"fags", "f { g z"},
            {"fahd", "f Q d"},
            {"fahr", "f Q@"},
            {"fahs", "f { s"},
            {"fail", "f eI l"},
            {"failed", "f eI l d"},
            {"fails", "f eI l z"},
            {"fain", "f eI n"},
            {"faint", "f eI n t"},
            {"fair", "f e@"},
            {"faire", "f e@"},
            {"faires", "f e@ z"},
            {"fairs", "f e@ z"},
            {"fair's", "f e@ z"},
            {"fait", "f eI t"},
            {"faith", "f eI T"},
            {"faiths", "f eI T s"},
            {"fake", "f eI k"},
            {"faked", "f eI k t"},
            {"fakes", "f eI k s"},
            {"falk", "f O: k"},
            {"fall", "f Q l"},
            {"falls", "f O: l z"},
            {"fall's", "f Q l z"},
            {"false", "f O: l s"},
            {"fame", "f eI m"},
            {"famed", "f eI m d"},
            {"fan", "f { n"},
            {"fane", "f eI n"},
            {"fang", "f { N"},
            {"fangs", "f { N z"},
            {"fanned", "f { n d"},
            {"fans", "f { n z"},
            {"fan's", "f { n z"},
            {"fant", "f { n t"},
            {"fao", "f aU"},
            {"far", "f Q@"},
            {"farce", "f Q@ s"},
            {"fare", "f e@"},
            {"fared", "f e@ d"},
            {"fares", "f e@ z"},
            {"farge", "f Q@ dZ"},
            {"farm", "f Q@ m"},
            {"farmed", "f Q@ m d"},
            {"farms", "f Q@ m z"},
            {"farm's", "f Q@ m z"},
            {"farr", "f Q@"},
            {"far's", "f Q@ z"},
            {"fass", "f { s"},
            {"fast", "f { s t"},
            {"fasts", "f { s t s"},
            {"fat", "f { t"},
            {"fate", "f eI t"},
            {"fates", "f eI t s"},
            {"fath", "f { T"},
            {"fats", "f { t s"},
            {"faulds", "f O: l d z"},
            {"fault", "f O: l t"},
            {"faults", "f O: l t s"},
            {"faure", "f O@"},
            {"faus", "f O: z"},
            {"faust", "f aU s t"},
            {"faux", "f O: k s"},
            {"faw", "f O:"},
            {"fawkes", "f O: k s"},
            {"fawn", "f O: n"},
            {"fax", "f { k s"},
            {"faxed", "f { k s t"},
            {"fay", "f eI"},
            {"faye", "f eI"},
            {"fayed", "f eI d"},
            {"fayme", "f eI m"},
            {"fayne", "f eI n"},
            {"fay's", "f eI z"},
            {"faze", "f eI z"},
            {"fazed", "f eI z d"},
            {"fe", "f eI"},
            {"fear", "f I@"},
            {"feared", "f I@ d"},
            {"fears", "f I@ z"},
            {"feast", "f i: s t"},
            {"feasts", "f i: s t s"},
            {"feat", "f i: t"},
            {"feats", "f i: t s"},
            {"fecht", "f e k t"},
            {"feck", "f e k"},
            {"fed", "f e d"},
            {"feds", "f e d z"},
            {"fed's", "f e d z"},
            {"fee", "f i:"},
            {"feed", "f i: d"},
            {"feeds", "f i: d z"},
            {"feel", "f i: l"},
            {"feels", "f i: l z"},
            {"fees", "f i: z"},
            {"feese", "f i: z"},
            {"feet", "f i: t"},
            {"feign", "f eI n"},
            {"feigned", "f eI n d"},
            {"feil", "f i: l"},
            {"fein", "f aI n"},
            {"fein's", "f aI n z"},
            {"feint", "f eI n t"},
            {"feis", "f aI s"},
            {"feist", "f aI s t"},
            {"felch", "f e l tS"},
            {"feld", "f e l d"},
            {"feldt", "f e l t"},
            {"fell", "f e l"},
            {"felled", "f e l d"},
            {"fells", "f e l z"},
            {"fels", "f e l z"},
            {"felske", "f e l s k"},
            {"felt", "f e l t"},
            {"felts", "f e l t s"},
            {"femme", "f e m"},
            {"fence", "f e n s"},
            {"fenced", "f e n s t"},
            {"fend", "f e n d"},
            {"fends", "f e n d z"},
            {"fenn", "f e n"},
            {"fent", "f e n t"},
            {"fer", "f @r"},
            {"ferd", "f @r d"},
            {"ferm", "f @r m"},
            {"fern", "f @r n"},
            {"ferns", "f @r n z"},
            {"fe's", "f eI z"},
            {"fess", "f e s"},
            {"fessed", "f e s t"},
            {"fest", "f e s t"},
            {"fests", "f e s t s"},
            {"fetch", "f e tS"},
            {"fetched", "f e tS t"},
            {"fete", "f eI t"},
            {"fett", "f e t"},
            {"fettes", "f e t s"},
            {"feucht", "f OI k t"},
            {"feud", "f j u: d"},
            {"feuds", "f j u: d z"},
            {"few", "f j u:"},
            {"fey", "f eI"},
            {"fez", "f e z"},
            {"fi", "f aI"},
            {"fide", "f aI d"},
            {"fides", "f aI d z"},
            {"fief", "f i: f"},
            {"fiel", "f i: l"},
            {"field", "f i: l d"},
            {"fields", "f i: l d z"},
            {"field's", "f i: l d z"},
            {"fiend", "f i: n d"},
            {"fiends", "f i: n d z"},
            {"fier", "f I@"},
            {"fierce", "f I@ s"},
            {"fife", "f aI f"},
            {"fifth", "f I f T"},
            {"fifths", "f I f T s"},
            {"fig", "f I g"},
            {"figge", "f I g"},
            {"fight", "f aI t"},
            {"fights", "f aI t s"},
            {"figs", "f I g z"},
            {"fike", "f aI k"},
            {"fikes", "f aI k s"},
            {"fil", "f I l"},
            {"filch", "f I l tS"},
            {"filched", "f I l tS t"},
            {"file", "f aI l"},
            {"filed", "f aI l d"},
            {"files", "f aI l z"},
            {"fill", "f I l"},
            {"filled", "f I l d"},
            {"fills", "f I l z"},
            {"film", "f I l m"},
            {"filmed", "f I l m d"},
            {"films", "f I l m z"},
            {"film's", "f I l m z"},
            {"filth", "f I l T"},
            {"fin", "f I n"},
            {"finch", "f I n tS"},
            {"find", "f aI n d"},
            {"finds", "f aI n d z"},
            {"fine", "f aI n"},
            {"fined", "f aI n d"},
            {"fines", "f aI n z"},
            {"fink", "f I N k"},
            {"finks", "f I N k s"},
            {"fink's", "f I N k s"},
            {"finn", "f I n"},
            {"finns", "f I n z"},
            {"finn's", "f I n z"},
            {"fins", "f I n z"},
            {"fir", "f @r"},
            {"firm", "f @r m"},
            {"firmed", "f @r m d"},
            {"firms", "f @r m z"},
            {"firm's", "f @r m z"},
            {"firs", "f @r z"},
            {"first", "f @r s t"},
            {"firsts", "f @r s t s"},
            {"firth", "f @r T"},
            {"fisc", "f I s k"},
            {"fisch", "f I S"},
            {"fish", "f I S"},
            {"fished", "f I S t"},
            {"fisk", "f I s k"},
            {"fiske", "f I s k"},
            {"fiske's", "f I s k s"},
            {"fisk's", "f I s k s"},
            {"fist", "f I s t"},
            {"fists", "f I s t s"},
            {"fit", "f I t"},
            {"fitch", "f I tS"},
            {"fits", "f I t s"},
            {"fitt", "f I t"},
            {"fitts", "f I t s"},
            {"fitz", "f I t s"},
            {"five", "f aI v"},
            {"fives", "f aI v z"},
            {"five's", "f aI v z"},
            {"fix", "f I k s"},
            {"fixed", "f I k s t"},
            {"fizz", "f I z"},
            {"fjeld", "f j e l d"},
            {"fjord", "f j O@ d"},
            {"fjords", "f j O@ d z"},
            {"flab", "f l { b"},
            {"flack", "f l { k"},
            {"flag", "f l { g"},
            {"flagg", "f l { g"},
            {"flagged", "f l { g d"},
            {"flags", "f l { g z"},
            {"flail", "f l eI l"},
            {"flair", "f l e@"},
            {"flak", "f l { k"},
            {"flake", "f l eI k"},
            {"flakes", "f l eI k s"},
            {"flam", "f l { m"},
            {"flame", "f l eI m"},
            {"flamed", "f l eI m d"},
            {"flames", "f l eI m z"},
            {"flamm", "f l { m"},
            {"flan", "f l { n"},
            {"flange", "f l { n dZ"},
            {"flank", "f l { N k"},
            {"flanked", "f l { N k t"},
            {"flanks", "f l { N k s"},
            {"flann", "f l { n"},
            {"flap", "f l { p"},
            {"flapped", "f l { p t"},
            {"flaps", "f l { p s"},
            {"flare", "f l e@"},
            {"flared", "f l e@ d"},
            {"flares", "f l e@ z"},
            {"flash", "f l { S"},
            {"flashed", "f l { S t"},
            {"flask", "f l { s k"},
            {"flasks", "f l { s k s"},
            {"flat", "f l { t"},
            {"flats", "f l { t s"},
            {"flatt", "f l { t"},
            {"flaunt", "f l O: n t"},
            {"flaunts", "f l O: n t s"},
            {"flaw", "f l O:"},
            {"flawed", "f l O: d"},
            {"flawn", "f l O: n"},
            {"flaws", "f l O: z"},
            {"flax", "f l { k s"},
            {"flay", "f l eI"},
            {"flayed", "f l eI d"},
            {"flea", "f l i:"},
            {"fleas", "f l i: z"},
            {"fleck", "f l e k"},
            {"flecks", "f l e k s"},
            {"fled", "f l e d"},
            {"fledge", "f l e dZ"},
            {"fledged", "f l e dZ d"},
            {"flee", "f l i:"},
            {"fleece", "f l i: s"},
            {"fleeced", "f l i: s t"},
            {"fleer", "f l I@"},
            {"flees", "f l i: z"},
            {"fleet", "f l i: t"},
            {"fleets", "f l i: t s"},
            {"fleet's", "f l i: t s"},
            {"flesh", "f l e S"},
            {"fleshed", "f l e S t"},
            {"fleur", "f l @r"},
            {"flew", "f l u:"},
            {"flex", "f l e k s"},
            {"flexed", "f l e k s t"},
            {"flick", "f l I k"},
            {"flicks", "f l I k s"},
            {"flied", "f l aI d"},
            {"flies", "f l aI z"},
            {"flight", "f l aI t"},
            {"flights", "f l aI t s"},
            {"flight's", "f l aI t s"},
            {"flinch", "f l I n tS"},
            {"flinched", "f l I n tS t"},
            {"fling", "f l I N"},
            {"flings", "f l I N z"},
            {"flinn", "f l I n"},
            {"flint", "f l I n t"},
            {"flints", "f l I n t s"},
            {"flint's", "f l I n t s"},
            {"flip", "f l I p"},
            {"flipped", "f l I p t"},
            {"flips", "f l I p s"},
            {"flirt", "f l @r t"},
            {"flirts", "f l @r t s"},
            {"flit", "f l I t"},
            {"flo", "f l @U"},
            {"float", "f l @U t"},
            {"floats", "f l @U t s"},
            {"floc", "f l Q k"},
            {"flock", "f l Q k"},
            {"flocked", "f l Q k t"},
            {"flocks", "f l Q k s"},
            {"floe", "f l @U"},
            {"flog", "f l Q g"},
            {"flom", "f l Q m"},
            {"flood", "f l V d"},
            {"floods", "f l V d z"},
            {"flook", "f l U k"},
            {"floor", "f l O@"},
            {"floored", "f l O@ d"},
            {"floors", "f l O@ z"},
            {"flop", "f l Q p"},
            {"flopped", "f l Q p t"},
            {"flops", "f l Q p s"},
            {"flor", "f l O@"},
            {"flore", "f l O@"},
            {"floss", "f l Q s"},
            {"flounce", "f l aU n s"},
            {"flout", "f l aU t"},
            {"flow", "f l @U"},
            {"flowe", "f l @U"},
            {"flowed", "f l @U d"},
            {"flown", "f l @U n"},
            {"flows", "f l @U z"},
            {"floy", "f l OI"},
            {"floyd", "f l OI d"},
            {"flu", "f l u:"},
            {"flubs", "f l V b z"},
            {"flue", "f l u:"},
            {"fluff", "f l V f"},
            {"fluffed", "f l V f t"},
            {"fluffs", "f l V f s"},
            {"fluke", "f l u: k"},
            {"flukes", "f l u: k s"},
            {"flume", "f l u: m"},
            {"flumes", "f l u: m z"},
            {"flung", "f l V N"},
            {"flunk", "f l V N k"},
            {"flunked", "f l V N k t"},
            {"flunks", "f l V N k s"},
            {"flus", "f l u: z"},
            {"flush", "f l V S"},
            {"flushed", "f l V S t"},
            {"flute", "f l u: t"},
            {"flutes", "f l u: t s"},
            {"flux", "f l V k s"},
            {"fly", "f l aI"},
            {"flynn", "f l I n"},
            {"foal", "f @U l"},
            {"foam", "f @U m"},
            {"foams", "f @U m z"},
            {"fob", "f O: b"},
            {"foe", "f @U"},
            {"foes", "f @U z"},
            {"fog", "f Q g"},
            {"fogg", "f Q g"},
            {"foie", "f OI"},
            {"foil", "f OI l"},
            {"foiled", "f OI l d"},
            {"foils", "f OI l z"},
            {"foist", "f OI s t"},
            {"fold", "f @U l d"},
            {"folds", "f @U l d z"},
            {"folk", "f @U k"},
            {"folks", "f @U k s"},
            {"fond", "f Q n d"},
            {"fone", "f @U n"},
            {"fong", "f O: N"},
            {"fons", "f Q n z"},
            {"font", "f Q n t"},
            {"fonts", "f Q n t s"},
            {"foo", "f u:"},
            {"food", "f u: d"},
            {"foods", "f u: d z"},
            {"food's", "f u: d z"},
            {"fool", "f u: l"},
            {"fooled", "f u: l d"},
            {"fools", "f u: l z"},
            {"fool's", "f u: l z"},
            {"foos", "f u: z"},
            {"foot", "f U t"},
            {"foote", "f U t"},
            {"foote's", "f U t s"},
            {"fop", "f O: p"},
            {"for", "f O@"},
            {"forbes", "f O@ b z"},
            {"force", "f O@ s"},
            {"forced", "f O@ s t"},
            {"ford", "f O@ d"},
            {"fords", "f O@ d z"},
            {"ford's", "f O@ d z"},
            {"fore", "f O@"},
            {"foret", "f O@ t"},
            {"forge", "f O@ dZ"},
            {"forged", "f O@ dZ d"},
            {"fork", "f O@ k"},
            {"forked", "f O@ k t"},
            {"forks", "f O@ k s"},
            {"form", "f O@ m"},
            {"formed", "f O@ m d"},
            {"forms", "f O@ m z"},
            {"forst", "f O@ s t"},
            {"fort", "f O@ t"},
            {"forth", "f O@ T"},
            {"forts", "f O@ t s"},
            {"foss", "f Q s"},
            {"fosse", "f Q s"},
            {"fought", "f O: t"},
            {"foul", "f aU l"},
            {"fouled", "f aU l d"},
            {"foulk", "f aU l k"},
            {"fouls", "f aU l z"},
            {"found", "f aU n d"},
            {"four", "f O@"},
            {"fours", "f O@ z"},
            {"four's", "f O@ z"},
            {"fourth", "f O@ T"},
            {"fourths", "f O@ T s"},
            {"fowkes", "f aU k s"},
            {"fowl", "f aU l"},
            {"fowlkes", "f aU l k s"},
            {"fox", "f Q k s"},
            {"foy", "f OI"},
            {"foyle", "f OI l"},
            {"frail", "f r eI l"},
            {"fraim", "f r eI m"},
            {"frakes", "f r eI k s"},
            {"fram", "f r { m"},
            {"frame", "f r eI m"},
            {"framed", "f r eI m d"},
            {"frames", "f r eI m z"},
            {"fran", "f r { n"},
            {"franc", "f r { N k"},
            {"france", "f r { n s"},
            {"franck", "f r { N k"},
            {"francs", "f r { N k s"},
            {"franc's", "f r { N k s"},
            {"frank", "f r { N k"},
            {"franke", "f r { N k"},
            {"franked", "f r { N k t"},
            {"franks", "f r { N k s"},
            {"frank's", "f r { N k s"},
            {"frans", "f r { n z"},
            {"frantz", "f r { n t s"},
            {"franz", "f r { n z"},
            {"frap", "f r { p"},
            {"frasch", "f r { S"},
            {"frase", "f r eI z"},
            {"frate", "f r eI t"},
            {"frau", "f r aU"},
            {"fraud", "f r O: d"},
            {"frauds", "f r O: d z"},
            {"fraught", "f r O: t"},
            {"fray", "f r eI"},
            {"frayed", "f r eI d"},
            {"frayn", "f r eI n"},
            {"frayne", "f r eI n"},
            {"fraze", "f r eI z"},
            {"freak", "f r i: k"},
            {"freaked", "f r i: k t"},
            {"freaks", "f r i: k s"},
            {"frear", "f r I@"},
            {"frech", "f r e k"},
            {"freck", "f r e k"},
            {"fred", "f r e d"},
            {"freda", "f r e d"},
            {"fred's", "f r e d z"},
            {"free", "f r i:"},
            {"freed", "f r i: d"},
            {"freen", "f r i: n"},
            {"frees", "f r i: z"},
            {"freet", "f r i: t"},
            {"freeze", "f r i: z"},
            {"frei", "f r aI"},
            {"freight", "f r eI t"},
            {"fremd", "f r e m d"},
            {"french", "f r e n tS"},
            {"frentz", "f r e n t s"},
            {"frere", "f r I@"},
            {"fresh", "f r e S"},
            {"fret", "f r e t"},
            {"frets", "f r e t s"},
            {"frett", "f r e t"},
            {"freud", "f r OI d"},
            {"freud's", "f r OI d z"},
            {"frey", "f r eI"},
            {"freyre", "f r e@"},
            {"frick", "f r I k"},
            {"fricke", "f r I k"},
            {"fridge", "f r I dZ"},
            {"fried", "f r aI d"},
            {"friede", "f r i: d"},
            {"friend", "f r e n d"},
            {"friends", "f r e n d z"},
            {"friend's", "f r e n d z"},
            {"fries", "f r aI z"},
            {"friese", "f r i: z"},
            {"friesz", "f r i: S"},
            {"frieze", "f r i: z"},
            {"fright", "f r aI t"},
            {"frill", "f r I l"},
            {"frills", "f r I l z"},
            {"fringe", "f r I n dZ"},
            {"frink", "f r I N k"},
            {"frisch", "f r I S"},
            {"frisk", "f r I s k"},
            {"frisked", "f r I s k t"},
            {"frist", "f r I s t"},
            {"fritch", "f r I tS"},
            {"frith", "f r I T"},
            {"frits", "f r I t s"},
            {"fritts", "f r I t s"},
            {"fritz", "f r I t s"},
            {"fritze", "f r I t z"},
            {"fritzsche", "f r I t S"},
            {"fro", "f r @U"},
            {"frock", "f r Q k"},
            {"frocks", "f r Q k s"},
            {"frog", "f r Q g"},
            {"frogs", "f r Q g z"},
            {"frog's", "f r Q g z"},
            {"froh", "f r @U"},
            {"from", "f r Q m"},
            {"frome", "f r @U m"},
            {"fromm", "f r Q m"},
            {"frond", "f r Q n d"},
            {"fronde", "f r Q n d"},
            {"fronds", "f r Q n d z"},
            {"front", "f r V n t"},
            {"fronts", "f r V n t s"},
            {"front's", "f r V n t s"},
            {"frost", "f r O: s t"},
            {"frosts", "f r O: s t s"},
            {"froth", "f r O: T"},
            {"froths", "f r O: T s"},
            {"frown", "f r aU n"},
            {"frowned", "f r aU n d"},
            {"frowns", "f r aU n z"},
            {"froze", "f r @U z"},
            {"fruit", "f r u: t"},
            {"fruits", "f r u: t s"},
            {"frum", "f r V m"},
            {"frump", "f r V m p"},
            {"frunze", "f r V n z"},
            {"frush", "f r V S"},
            {"fry", "f r aI"},
            {"frye", "f r aI"},
            {"fry's", "f r aI z"},
            {"f's", "e f s"},
            {"fu", "f u:"},
            {"fuchs", "f j u: k s"},
            {"fuck", "f V k"},
            {"fucked", "f V k t"},
            {"fucks", "f V k s"},
            {"fudge", "f V dZ"},
            {"fudged", "f V dZ d"},
            {"fugue", "f j u: g"},
            {"fugues", "f j u: g z"},
            {"fuld", "f U l d"},
            {"fulk", "f V l k"},
            {"fulks", "f V l k s"},
            {"full", "f U l"},
            {"fults", "f U l t s"},
            {"fultz", "f U l t s"},
            {"fume", "f j u: m"},
            {"fumed", "f j u: m d"},
            {"fumes", "f j u: m z"},
            {"fun", "f V n"},
            {"fund", "f V n d"},
            {"funds", "f V n d z"},
            {"fund's", "f V n d z"},
            {"fung", "f V N"},
            {"funk", "f V N k"},
            {"funks", "f V N k s"},
            {"funs", "f V n z"},
            {"fur", "f @r"},
            {"furr", "f @r"},
            {"furr's", "f @r z"},
            {"furs", "f @r z"},
            {"furth", "f @r T"},
            {"fuse", "f j u: z"},
            {"fused", "f j u: z d"},
            {"fuss", "f V s"},
            {"fussed", "f V s t"},
            {"fust", "f V s t"},
            {"fuzz", "f V z"},
            {"fyffe", "f aI f"},
            {"fyke", "f aI k"},
            {"g", "dZ i:"},
            {"ga", "gh Q"},
            {"gaal", "gh Q l"},
            {"gab", "gh { b"},
            {"gabe", "gh eI b"},
            {"gad", "gh { d"},
            {"gade", "gh eI d"},
            {"gads", "gh { d z"},
            {"gae", "gh aI"},
            {"gael", "gh eI l"},
            {"gaff", "gh { f"},
            {"gaffe", "gh { f"},
            {"gaffes", "gh { f s"},
            {"gag", "gh { g"},
            {"gage", "gh eI dZ"},
            {"gagged", "gh { g d"},
            {"gags", "gh { g z"},
            {"gaige", "gh eI dZ"},
            {"gail", "gh eI l"},
            {"gaile", "gh eI l"},
            {"gail's", "gh eI l z"},
            {"gain", "gh eI n"},
            {"gained", "gh eI n d"},
            {"gaines", "gh eI n z"},
            {"gains", "gh eI n z"},
            {"gair", "gh e@"},
            {"gait", "gh eI t"},
            {"gal", "gh { l"},
            {"gale", "gh eI l"},
            {"gales", "gh eI l z"},
            {"gale's", "gh eI l z"},
            {"gall", "gh O: l"},
            {"galle", "gh eI l"},
            {"galls", "gh O: l z"},
            {"gals", "gh { l z"},
            {"galt", "gh O: l t"},
            {"game", "gh eI m"},
            {"games", "gh eI m z"},
            {"game's", "gh eI m z"},
            {"gamp", "gh { m p"},
            {"gan", "gh { n"},
            {"gang", "gh { N"},
            {"gange", "gh { n dZ"},
            {"gangs", "gh { N z"},
            {"gang's", "gh { N z"},
            {"gans", "gh { n z"},
            {"gant", "gh { n t"},
            {"gantt", "gh { n t"},
            {"gap", "gh { p"},
            {"gape", "gh eI p"},
            {"gaps", "gh { p s"},
            {"gap's", "gh { p s"},
            {"gar", "gh Q@"},
            {"garb", "gh Q@ b"},
            {"garbe", "gh Q@ b"},
            {"garbed", "gh Q@ b d"},
            {"gard", "gh Q@ d"},
            {"garde", "gh Q@ d"},
            {"gare", "gh e@"},
            {"garg", "gh Q@ g"},
            {"garn", "gh Q@ n"},
            {"garnes", "gh Q@ n z"},
            {"gars", "gh Q@ z"},
            {"garth", "gh Q@ T"},
            {"gas", "gh { s"},
            {"gash", "gh { S"},
            {"gashed", "gh { S t"},
            {"gasp", "gh { s p"},
            {"gasped", "gh { s p t"},
            {"gasps", "gh { s p s"},
            {"gass", "gh { s"},
            {"gassed", "gh { s t"},
            {"gast", "gh { s t"},
            {"gat", "gh { t"},
            {"gatch", "gh { tS"},
            {"gate", "gh eI t"},
            {"gates", "gh eI t s"},
            {"gath", "gh { T"},
            {"gatt", "gh { t"},
            {"gatzke", "gh { t s k"},
            {"gau", "gh @U"},
            {"gaub", "gh O: b"},
            {"gauche", "gh @U S"},
            {"gauge", "gh eI dZ"},
            {"gauged", "gh eI dZ d"},
            {"gaul", "gh O: l"},
            {"gaulle", "gh O: l"},
            {"gauls", "gh O: l z"},
            {"gault", "gh O: l t"},
            {"gaunt", "gh O: n t"},
            {"gaus", "gh O: z"},
            {"gause", "gh O: z"},
            {"gauss", "gh aU s"},
            {"gaut", "gh O: t"},
            {"gauze", "gh O: z"},
            {"gave", "gh eI v"},
            {"gaw", "gh O:"},
            {"gawk", "gh O: k"},
            {"gay", "gh eI"},
            {"gaye", "gh eI"},
            {"gayle", "gh eI l"},
            {"gays", "gh eI z"},
            {"gaz", "gh Q z"},
            {"gaze", "gh eI z"},
            {"gazed", "gh eI z d"},
            {"gean", "dZ i: n"},
            {"gear", "gh I@"},
            {"geared", "gh I@ d"},
            {"gears", "gh I@ z"},
            {"gear's", "gh I@ z"},
            {"geck", "dZ e k"},
            {"ged", "gh e d"},
            {"gee", "dZ i:"},
            {"geek", "gh i: k"},
            {"geeks", "gh i: k s"},
            {"geer", "gh I@"},
            {"geerts", "gh I@ t s"},
            {"gees", "dZ i: s"},
            {"gee's", "dZ i: z"},
            {"geese", "gh i: s"},
            {"geez", "dZ i: z"},
            {"gegg", "dZ e g"},
            {"geis", "gh aI z"},
            {"geiss", "gh aI s"},
            {"geist", "gh aI s t"},
            {"gel", "dZ e l"},
            {"gelb", "dZ e l b"},
            {"gell", "dZ e l"},
            {"gels", "dZ e l z"},
            {"gem", "dZ e m"},
            {"gems", "dZ e m z"},
            {"gem's", "dZ e m z"},
            {"gene", "dZ i: n"},
            {"genes", "dZ i: n z"},
            {"gene's", "dZ i: n z"},
            {"genk", "dZ e N k"},
            {"gens", "dZ e n z"},
            {"gent", "dZ e n t"},
            {"geoff", "dZ e f"},
            {"george", "dZ O@ dZ"},
            {"gerd", "gh @r d"},
            {"gere", "dZ I@"},
            {"germ", "dZ @r m"},
            {"germs", "dZ @r m z"},
            {"gert", "gh @r t"},
            {"gess", "dZ e s"},
            {"gest", "dZ e s t"},
            {"get", "gh e t"},
            {"gets", "gh e t s"},
            {"getz", "gh e t s"},
            {"ghee", "gh i:"},
            {"gheen", "gh i: n"},
            {"ghent", "gh e n t"},
            {"ghost", "gh @U s t"},
            {"ghosts", "gh @U s t s"},
            {"gib", "gh I b"},
            {"gibb", "dZ I b"},
            {"gibbs", "gh I b z"},
            {"gibb's", "gh I b z"},
            {"gibe", "dZ aI b"},
            {"gies", "gh i: z"},
            {"giff", "gh I f"},
            {"gift", "gh I f t"},
            {"gifts", "gh I f t s"},
            {"gig", "gh I g"},
            {"gigs", "gh I g z"},
            {"gil", "gh I l"},
            {"gild", "gh I l d"},
            {"gile", "gh aI l"},
            {"giles", "dZ aI l z"},
            {"gill", "gh I l"},
            {"gilles", "Z i: l"},
            {"gills", "gh I l z"},
            {"gill's", "gh I l z"},
            {"gilt", "gh I l t"},
            {"gilts", "gh I l t s"},
            {"gin", "dZ I n"},
            {"ging", "dZ I N"},
            {"ginn", "dZ I n"},
            {"ginned", "dZ I n d"},
            {"gintz", "gh I n t s"},
            {"gird", "gh @r d"},
            {"girds", "gh @r d z"},
            {"gire", "gh aI r"},
            {"girl", "gh @r l"},
            {"girls", "gh @r l z"},
            {"girl's", "gh @r l z"},
            {"girt", "gh @r t"},
            {"girth", "gh @r T"},
            {"gish", "dZ I S"},
            {"gist", "dZ I s t"},
            {"give", "gh I v"},
            {"gives", "gh I v z"},
            {"glaab", "gh l Q b"},
            {"glace", "gh l eI s"},
            {"glad", "gh l { d"},
            {"glade", "gh l eI d"},
            {"glades", "gh l eI d z"},
            {"glance", "gh l { n s"},
            {"glanced", "gh l { n s t"},
            {"gland", "gh l { n d"},
            {"glands", "gh l { n d z"},
            {"glantz", "gh l { n t s"},
            {"glare", "gh l e@"},
            {"glared", "gh l e@ d"},
            {"glares", "gh l e@ z"},
            {"glass", "gh l { s"},
            {"glassed", "gh l { s t"},
            {"glaze", "gh l eI z"},
            {"glazed", "gh l eI z d"},
            {"gleam", "gh l i: m"},
            {"gleamed", "gh l i: m d"},
            {"gleams", "gh l i: m z"},
            {"glean", "gh l i: n"},
            {"gleaned", "gh l i: n d"},
            {"gleave", "gh l i: v"},
            {"gleaves", "gh l i: v z"},
            {"glee", "gh l i:"},
            {"gleich", "gh l aI k"},
            {"glen", "gh l e n"},
            {"glenn", "gh l e n"},
            {"glenn's", "gh l e n z"},
            {"glens", "gh l e n z"},
            {"glew", "gh l u:"},
            {"glib", "gh l I b"},
            {"glick", "gh l I k"},
            {"glide", "gh l aI d"},
            {"glides", "gh l aI d z"},
            {"glimpse", "gh l I m p s"},
            {"glimpsed", "gh l I m p s t"},
            {"glint", "gh l I n t"},
            {"gliss", "gh l I s"},
            {"glitch", "gh l I tS"},
            {"glitz", "gh l I t s"},
            {"gloat", "gh l @U t"},
            {"gloats", "gh l @U t s"},
            {"glob", "gh l Q b"},
            {"globe", "gh l @U b"},
            {"globes", "gh l @U b z"},
            {"globe's", "gh l @U b z"},
            {"globs", "gh l Q b z"},
            {"glock", "gh l Q k"},
            {"glod", "gh l Q d"},
            {"gloom", "gh l u: m"},
            {"glop", "gh l Q p"},
            {"glor", "gh l O@"},
            {"glore", "gh l O@"},
            {"gloss", "gh l O: s"},
            {"glossed", "gh l O: s t"},
            {"glove", "gh l V v"},
            {"gloved", "gh l V v d"},
            {"gloves", "gh l V v z"},
            {"glow", "gh l @U"},
            {"glowed", "gh l @U d"},
            {"glows", "gh l @U z"},
            {"gluck", "gh l V k"},
            {"gluck's", "gh l V k s"},
            {"glue", "gh l u:"},
            {"glued", "gh l u: d"},
            {"glues", "gh l u: z"},
            {"glum", "gh l V m"},
            {"glut", "gh l V t"},
            {"gluts", "gh l V t s"},
            {"glyn", "gh l I n"},
            {"glynn", "gh l I n"},
            {"gmbh", "gh @ m"},
            {"gnarl", "n Q@ l"},
            {"gnarled", "n Q@ l d"},
            {"gnash", "n { S"},
            {"gnat", "n { t"},
            {"gnats", "n { t s"},
            {"gnaw", "n O:"},
            {"gnawed", "n O: d"},
            {"gnaws", "n O: z"},
            {"gneiss", "n aI s"},
            {"gnomes", "n @U m z"},
            {"gnu", "n u:"},
            {"go", "gh @U"},
            {"goad", "gh @U d"},
            {"goal", "gh @U l"},
            {"goals", "gh @U l z"},
            {"goan", "gh @U n"},
            {"goans", "gh @U n z"},
            {"goar", "gh O@"},
            {"goat", "gh @U t"},
            {"goats", "gh @U t s"},
            {"goat's", "gh @U t s"},
            {"gob", "gh Q b"},
            {"gobs", "gh Q b z"},
            {"god", "gh Q d"},
            {"gods", "gh Q d z"},
            {"god's", "gh Q d z"},
            {"goe", "gh @U"},
            {"goen", "gh @U n"},
            {"goerke", "gh O@ k"},
            {"goes", "gh @U z"},
            {"goethe", "gh @U T"},
            {"goetz", "gh e t s"},
            {"goff", "gh O: f"},
            {"gogh", "gh @U"},
            {"goines", "gh OI n z"},
            {"gold", "gh @U l d"},
            {"golds", "gh @U l d z"},
            {"gold's", "gh @U l d z"},
            {"golf", "gh Q l f"},
            {"golfed", "gh Q l f t"},
            {"golfs", "gh Q l f s"},
            {"golf's", "gh Q l f s"},
            {"goll", "gh Q l"},
            {"goltz", "gh @U l t s"},
            {"gond", "gh Q n d"},
            {"gone", "gh Q n"},
            {"gong", "gh O: N"},
            {"gongs", "gh O: N z"},
            {"goo", "gh u:"},
            {"gooch", "gh u: tS"},
            {"good", "gh U d"},
            {"goode", "gh U d"},
            {"goods", "gh U d z"},
            {"good's", "gh U d z"},
            {"goof", "gh u: f"},
            {"goofed", "gh u: f t"},
            {"goofs", "gh u: f s"},
            {"gook", "gh U k"},
            {"goold", "gh u: l d"},
            {"goon", "gh u: n"},
            {"goons", "gh u: n z"},
            {"goop", "gh u: p"},
            {"goos", "gh u: z"},
            {"goose", "gh u: s"},
            {"gore", "gh O@"},
            {"gored", "gh O@ d"},
            {"gores", "gh O@ z"},
            {"gore's", "gh O@ z"},
            {"gorge", "gh O@ dZ"},
            {"gosh", "gh Q S"},
            {"goss", "gh O: s"},
            {"gosse", "gh Q s"},
            {"got", "gh Q t"},
            {"gotch", "gh Q tS"},
            {"goth", "gh Q T"},
            {"gott", "gh Q t"},
            {"gotz", "gh Q t s"},
            {"gouge", "gh aU dZ"},
            {"gouged", "gh aU dZ d"},
            {"gough", "gh O: f"},
            {"gould", "gh u: l d"},
            {"gould's", "gh u: l d z"},
            {"gourd", "gh O@ d"},
            {"gourds", "gh O@ d z"},
            {"gout", "gh aU t"},
            {"gov", "gh V v"},
            {"gove", "gh @U v"},
            {"govs", "gh Q v z"},
            {"gow", "gh aU"},
            {"gown", "gh aU n"},
            {"gowns", "gh aU n z"},
            {"goy", "gh OI"},
            {"grab", "gh r { b"},
            {"grabbed", "gh r { b d"},
            {"grabs", "gh r { b z"},
            {"grace", "gh r eI s"},
            {"graced", "gh r eI s t"},
            {"grad", "gh r { d"},
            {"grade", "gh r eI d"},
            {"grades", "gh r eI d z"},
            {"grads", "gh r { d z"},
            {"graeme", "gh r eI m"},
            {"graf", "gh r { f"},
            {"graff", "gh r { f"},
            {"graft", "gh r { f t"},
            {"grafts", "gh r { f t s"},
            {"grahn", "gh r { n"},
            {"grail", "gh r eI l"},
            {"grain", "gh r eI n"},
            {"grained", "gh r eI n d"},
            {"grains", "gh r eI n z"},
            {"gram", "gh r { m"},
            {"grames", "gh r eI m z"},
            {"grams", "gh r { m z"},
            {"gran", "gh r { n"},
            {"grand", "gh r { n d"},
            {"grande", "gh r { n d"},
            {"grands", "gh r { n d z"},
            {"grand's", "gh r { n d z"},
            {"grange", "gh r eI n dZ"},
            {"grant", "gh r { n t"},
            {"grants", "gh r { n t s"},
            {"grant's", "gh r { n t s"},
            {"grape", "gh r eI p"},
            {"grapes", "gh r eI p s"},
            {"graph", "gh r { f"},
            {"graphs", "gh r { f s"},
            {"gras", "gh r { s"},
            {"grasp", "gh r { s p"},
            {"grasped", "gh r { s p t"},
            {"grasps", "gh r { s p s"},
            {"grass", "gh r { s"},
            {"grasse", "gh r { s"},
            {"grassed", "gh r { s t"},
            {"grate", "gh r eI t"},
            {"grates", "gh r eI t s"},
            {"gratz", "gh r { t s"},
            {"grave", "gh r eI v"},
            {"graves", "gh r eI v z"},
            {"gray", "gh r eI"},
            {"grays", "gh r eI z"},
            {"gray's", "gh r eI z"},
            {"graze", "gh r eI z"},
            {"grazed", "gh r eI z d"},
            {"grease", "gh r i: s"},
            {"greased", "gh r i: s t"},
            {"great", "gh r eI t"},
            {"greats", "gh r eI t s"},
            {"great's", "gh r eI t s"},
            {"greave", "gh r i: v"},
            {"greaves", "gh r i: v z"},
            {"greb", "gh r e b"},
            {"grebe", "gh r i: b"},
            {"greece", "gh r i: s"},
            {"greed", "gh r i: d"},
            {"greek", "gh r i: k"},
            {"greeks", "gh r i: k s"},
            {"green", "gh r i: n"},
            {"greene", "gh r i: n"},
            {"greened", "gh r i: n d"},
            {"greene's", "gh r i: n z"},
            {"greens", "gh r i: n z"},
            {"green's", "gh r i: n z"},
            {"greer", "gh r I@"},
            {"greet", "gh r i: t"},
            {"greets", "gh r i: t s"},
            {"grefe", "gh r i: f"},
            {"greff", "gh r e f"},
            {"greg", "gh r e g"},
            {"gregg", "gh r e g"},
            {"greggs", "gh r e g z"},
            {"greg's", "gh r e g z"},
            {"greig", "gh r i: g"},
            {"grein", "gh r eI n"},
            {"gren", "gh r e n"},
            {"grete", "gh r i: t"},
            {"greve", "gh r i: v"},
            {"grew", "gh r u:"},
            {"grey", "gh r eI"},
            {"grey's", "gh r eI z"},
            {"grice", "gh r aI s"},
            {"grid", "gh r I d"},
            {"grids", "gh r I d z"},
            {"grid's", "gh r I d z"},
            {"grief", "gh r i: f"},
            {"grieve", "gh r i: v"},
            {"grieved", "gh r i: v d"},
            {"grieves", "gh r i: v z"},
            {"griff", "gh r I f"},
            {"griggs", "gh r I g z"},
            {"grill", "gh r I l"},
            {"grille", "gh r I l"},
            {"grilled", "gh r I l d"},
            {"grills", "gh r I l z"},
            {"grim", "gh r I m"},
            {"grime", "gh r aI m"},
            {"grimes", "gh r aI m z"},
            {"grimm", "gh r I m"},
            {"grimme", "gh r I m"},
            {"grimm's", "gh r I m z"},
            {"grin", "gh r I n"},
            {"grinch", "gh r I n tS"},
            {"grind", "gh r aI n d"},
            {"grinds", "gh r aI n d z"},
            {"gring", "gh r I N"},
            {"grinned", "gh r I n d"},
            {"grins", "gh r I n z"},
            {"grip", "gh r I p"},
            {"gripe", "gh r aI p"},
            {"griped", "gh r aI p t"},
            {"gripes", "gh r aI p s"},
            {"gripped", "gh r I p t"},
            {"grips", "gh r I p s"},
            {"grise", "gh r aI z"},
            {"grist", "gh r I s t"},
            {"grit", "gh r I t"},
            {"grits", "gh r I t s"},
            {"griz", "gh r I z"},
            {"gro", "gh r @U"},
            {"groan", "gh r @U n"},
            {"groaned", "gh r @U n d"},
            {"groans", "gh r @U n z"},
            {"groat", "gh r @U t"},
            {"grobe", "gh r @U b"},
            {"groff", "gh r O: f"},
            {"groh", "gh r @U"},
            {"groin", "gh r OI n"},
            {"groined", "gh r OI n d"},
            {"groins", "gh r OI n z"},
            {"grone", "gh r @U n"},
            {"groom", "gh r u: m"},
            {"groome", "gh r u: m"},
            {"groomed", "gh r u: m d"},
            {"grooms", "gh r u: m z"},
            {"groos", "gh r u: z"},
            {"groot", "gh r u: t"},
            {"groove", "gh r u: v"},
            {"grooves", "gh r u: v z"},
            {"grope", "gh r @U p"},
            {"groped", "gh r @U p t"},
            {"gros", "gh r @U s"},
            {"grose", "gh r @U z"},
            {"gross", "gh r @U s"},
            {"grosse", "gh r Q s"},
            {"grossed", "gh r @U s t"},
            {"grosz", "gh r @U s"},
            {"grote", "gh r @U t"},
            {"grouch", "gh r aU tS"},
            {"ground", "gh r aU n d"},
            {"grounds", "gh r aU n d z"},
            {"group", "gh r u: p"},
            {"grouped", "gh r u: p t"},
            {"groups", "gh r u: p s"},
            {"group's", "gh r u: p s"},
            {"grouse", "gh r aU s"},
            {"groused", "gh r aU s t"},
            {"grout", "gh r aU t"},
            {"grove", "gh r @U v"},
            {"groves", "gh r @U v z"},
            {"grove's", "gh r @U v z"},
            {"grow", "gh r @U"},
            {"growl", "gh r aU l"},
            {"growled", "gh r aU l d"},
            {"growls", "gh r aU l z"},
            {"grown", "gh r @U n"},
            {"grows", "gh r @U z"},
            {"growth", "gh r @U T"},
            {"growths", "gh r @U T s"},
            {"grub", "gh r V b"},
            {"grubb", "gh r V b"},
            {"grubbs", "gh r V b z"},
            {"grube", "gh r u: b"},
            {"grubs", "gh r V b z"},
            {"grudge", "gh r V dZ"},
            {"gruff", "gh r V f"},
            {"grum", "gh r V m"},
            {"grump", "gh r V m p"},
            {"grun", "gh r V n"},
            {"grunge", "gh r V n dZ"},
            {"grunt", "gh r V n t"},
            {"grunts", "gh r V n t s"},
            {"grush", "gh r V S"},
            {"gruss", "gh r V s"},
            {"g's", "dZ i: z"},
            {"gu", "gh u:"},
            {"guam", "gh w Q m"},
            {"guard", "gh Q@ d"},
            {"guards", "gh Q@ d z"},
            {"guard's", "gh Q@ d z"},
            {"guck", "gh V k"},
            {"gude", "gh j u: d"},
            {"gue", "gh j u:"},
            {"guerre", "gh e@"},
            {"guess", "gh e s"},
            {"guessed", "gh e s t"},
            {"guest", "gh e s t"},
            {"guests", "gh e s t s"},
            {"guest's", "gh e s t s"},
            {"guff", "gh V f"},
            {"guide", "gh aI d"},
            {"guides", "gh aI d z"},
            {"guide's", "gh aI d z"},
            {"guild", "gh I l d"},
            {"guilds", "gh I l d z"},
            {"guild's", "gh I l d z"},
            {"guile", "gh aI l"},
            {"guiles", "gh aI l z"},
            {"guilt", "gh I l t"},
            {"guin", "gh I n"},
            {"guinn", "gh I n"},
            {"guise", "gh aI z"},
            {"gul", "gh V l"},
            {"gulch", "gh V l tS"},
            {"gulf", "gh V l f"},
            {"gulf's", "gh V l f s"},
            {"gull", "gh V l"},
            {"gulls", "gh V l z"},
            {"gull's", "gh V l z"},
            {"gulp", "gh V l p"},
            {"gulped", "gh V l p t"},
            {"gulps", "gh V l p s"},
            {"gum", "gh V m"},
            {"gummed", "gh V m d"},
            {"gump", "gh V m p"},
            {"gump's", "gh V m p s"},
            {"gums", "gh V m z"},
            {"gun", "gh V n"},
            {"gung", "gh V N"},
            {"gunn", "gh V n"},
            {"gunned", "gh V n d"},
            {"gunn's", "gh V n z"},
            {"guns", "gh V n z"},
            {"gun's", "gh V n z"},
            {"gurr", "gh @r"},
            {"gus", "gh V s"},
            {"gush", "gh V S"},
            {"gushed", "gh V S t"},
            {"guss", "gh V s"},
            {"gust", "gh V s t"},
            {"gusts", "gh V s t s"},
            {"gut", "gh V t"},
            {"guts", "gh V t s"},
            {"gutt", "gh V t"},
            {"guy", "gh aI"},
            {"guys", "gh aI z"},
            {"guy's", "gh aI z"},
            {"gwen", "gh w e n"},
            {"gwin", "gh w I n"},
            {"gwinn", "gh w I n"},
            {"gwyn", "gh w I n"},
            {"gwynn", "gh w I n"},
            {"gwynne", "gh w I n"},
            {"gym", "dZ I m"},
            {"gyms", "dZ I m z"},
            {"gym's", "dZ I m z"},
            {"gyn", "gh I n"},
            {"gyp", "dZ I p"},
            {"gypped", "dZ I p t"},
            {"h", "eI tS"},
            {"h.", "eI tS"},
            {"ha", "h Q"},
            {"haab", "h Q b"},
            {"haaf", "h Q f"},
            {"haag", "h Q g"},
            {"haak", "h Q k"},
            {"haar", "h Q@"},
            {"haas", "h Q s"},
            {"haase", "h Q s"},
            {"hach", "h { tS"},
            {"hack", "h { k"},
            {"hacked", "h { k t"},
            {"hacks", "h { k s"},
            {"had", "h { d"},
            {"hade", "h eI d"},
            {"haen", "h i: n"},
            {"haff", "h { f"},
            {"haft", "h { f t"},
            {"hafts", "h { f t s"},
            {"haft's", "h { f t s"},
            {"hag", "h { g"},
            {"hagg", "h { g"},
            {"hague", "h eI g"},
            {"hah", "h Q"},
            {"hahn", "h Q n"},
            {"hahs", "h Q s"},
            {"haig", "h eI g"},
            {"haig's", "h eI g z"},
            {"haik", "h eI k"},
            {"hail", "h eI l"},
            {"haile", "h eI l"},
            {"hailed", "h eI l d"},
            {"hailes", "h eI l z"},
            {"hails", "h eI l z"},
            {"haim", "h aI m"},
            {"haimes", "h eI m z"},
            {"hain", "h eI n"},
            {"haines", "h eI n z"},
            {"hains", "h eI n z"},
            {"hair", "h e@"},
            {"haire", "h e@"},
            {"haired", "h e@ d"},
            {"hairs", "h e@ z"},
            {"hair's", "h e@ z"},
            {"hait", "h eI t"},
            {"haith", "h eI T"},
            {"hajj", "h { dZ"},
            {"hake", "h eI k"},
            {"hakes", "h eI k s"},
            {"hal", "h { l"},
            {"hale", "h eI l"},
            {"hales", "h eI l z"},
            {"hale's", "h eI l z"},
            {"half", "h { f"},
            {"half's", "h { f s"},
            {"hall", "h O: l"},
            {"halle", "h { l"},
            {"halls", "h O: l z"},
            {"hall's", "h O: l z"},
            {"halm", "h Q m"},
            {"hals", "h { l s"},
            {"hal's", "h { l z"},
            {"halt", "h O: l t"},
            {"halts", "h O: l t s"},
            {"halve", "h { v"},
            {"halved", "h { v d"},
            {"halves", "h { v z"},
            {"ham", "h { m"},
            {"hamed", "h { m d"},
            {"hames", "h eI m z"},
            {"hamm", "h { m"},
            {"hams", "h { m z"},
            {"han", "h Q n"},
            {"hance", "h { n s"},
            {"hand", "h { n d"},
            {"hands", "h { n d z"},
            {"hang", "h { N"},
            {"hanged", "h { N d"},
            {"hangs", "h { N z"},
            {"hank", "h { N k"},
            {"hanks", "h { N k s"},
            {"hank's", "h { n k s"},
            {"hans", "h Q n s"},
            {"han's", "h Q n z"},
            {"hao", "h aU"},
            {"hap", "h { p"},
            {"happ", "h { p"},
            {"haq", "h { k"},
            {"harb", "h Q@ b"},
            {"hard", "h Q@ d"},
            {"hards", "h Q@ d z"},
            {"hare", "h e@"},
            {"hares", "h e@ z"},
            {"hare's", "h e@ z"},
            {"hark", "h Q@ k"},
            {"harks", "h Q@ k s"},
            {"harl", "h Q@ l"},
            {"harm", "h Q@ m"},
            {"harmed", "h Q@ m d"},
            {"harms", "h Q@ m z"},
            {"harm's", "h Q@ m z"},
            {"harn", "h Q@ n"},
            {"harned", "h Q@ n d"},
            {"harp", "h Q@ p"},
            {"harped", "h Q@ p t"},
            {"harps", "h Q@ p s"},
            {"harr", "h { r"},
            {"harre", "h { r"},
            {"harsh", "h Q@ S"},
            {"hart", "h Q@ t"},
            {"harte", "h Q@ t"},
            {"harts", "h Q@ t s"},
            {"hart's", "h Q@ t s"},
            {"harv", "h Q@ v"},
            {"harve", "h Q@ v"},
            {"has", "h { z"},
            {"hase", "h eI z"},
            {"hash", "h { S"},
            {"hashed", "h { S t"},
            {"hasse", "h Q s"},
            {"hast", "h { s t"},
            {"haste", "h eI s t"},
            {"hat", "h { t"},
            {"hatch", "h { tS"},
            {"hatched", "h { tS t"},
            {"hate", "h eI t"},
            {"hates", "h eI t s"},
            {"hath", "h { T"},
            {"hats", "h { t s"},
            {"hat's", "h { t s"},
            {"hatt", "h { t"},
            {"hau", "h aU"},
            {"hauck", "h O: k"},
            {"hauge", "h O: dZ"},
            {"haugh", "h O:"},
            {"haught", "h O: t"},
            {"haul", "h O: l"},
            {"hauled", "h O: l d"},
            {"hauls", "h O: l z"},
            {"haunt", "h O: n t"},
            {"haunts", "h O: n t s"},
            {"hause", "h aU s"},
            {"haut", "h O: t"},
            {"haute", "h @U t"},
            {"have", "h { v"},
            {"haves", "h { v z"},
            {"haw", "h O:"},
            {"hawes", "h O: z"},
            {"hawk", "h O: k"},
            {"hawke", "h O: k"},
            {"hawked", "h O: k t"},
            {"hawke's", "h O: k s"},
            {"hawks", "h O: k s"},
            {"haws", "h O: z"},
            {"hay", "h eI"},
            {"haye", "h eI"},
            {"hayes", "h eI z"},
            {"hayles", "h eI l z"},
            {"haymes", "h eI m z"},
            {"hayne", "h eI n"},
            {"haynes", "h eI n z"},
            {"hays", "h eI z"},
            {"hayse", "h eI z"},
            {"haze", "h eI z"},
            {"he", "h i:"},
            {"head", "h e d"},
            {"heads", "h e d z"},
            {"head's", "h e d z"},
            {"heal", "h i: l"},
            {"heald", "h i: l d"},
            {"healed", "h i: l d"},
            {"heals", "h i: l z"},
            {"health", "h e l T"},
            {"health's", "h e l T s"},
            {"heap", "h i: p"},
            {"heaped", "h i: p t"},
            {"heaps", "h i: p s"},
            {"hear", "h I@"},
            {"heard", "h @r d"},
            {"hearn", "h @r n"},
            {"hearne", "h @r n"},
            {"hears", "h I@ z"},
            {"hearse", "h @r s"},
            {"hearst", "h @r s t"},
            {"hearst's", "h @r s t s"},
            {"heart", "h Q@ t"},
            {"hearth", "h Q@ T"},
            {"hearths", "h Q@ T s"},
            {"hearts", "h Q@ t s"},
            {"heart's", "h Q@ t s"},
            {"heat", "h i: t"},
            {"heath", "h i: T"},
            {"heath's", "h i: T s"},
            {"heats", "h i: t s"},
            {"heat's", "h i: t s"},
            {"heave", "h i: v"},
            {"heaved", "h i: v d"},
            {"heaves", "h i: v z"},
            {"hebe", "h i: b"},
            {"hecht", "h e k t"},
            {"hecht's", "h e k t s"},
            {"heck", "h e k"},
            {"heck's", "h e k s"},
            {"he'd", "h i: d"},
            {"hedge", "h e dZ"},
            {"hedged", "h e dZ d"},
            {"hee", "h i:"},
            {"heed", "h i: d"},
            {"heeds", "h i: d z"},
            {"heel", "h i: l"},
            {"heeled", "h i: l d"},
            {"heels", "h i: l z"},
            {"heft", "h e f t"},
            {"hege", "h i: dZ"},
            {"heh", "h e"},
            {"heid", "h aI d"},
            {"heide", "h aI d"},
            {"heidt", "h aI d t"},
            {"height", "h aI t"},
            {"heighth", "h aI T"},
            {"heights", "h aI t s"},
            {"heil", "h aI l"},
            {"heim", "h aI m"},
            {"hein", "h aI n"},
            {"heine", "h aI n"},
            {"heins", "h aI n z"},
            {"heintz", "h aI n t s"},
            {"heinz", "h aI n z"},
            {"heinze", "h aI n z"},
            {"heir", "e@"},
            {"heirs", "e@ z"},
            {"heise", "h aI s"},
            {"heist", "h aI s t"},
            {"held", "h e l d"},
            {"hell", "h e l"},
            {"he'll", "h i: l"},
            {"helle", "h e l"},
            {"hell's", "h e l z"},
            {"helm", "h e l m"},
            {"helms", "h e l m z"},
            {"help", "h e l p"},
            {"helped", "h e l p t"},
            {"helps", "h e l p s"},
            {"hem", "h e m"},
            {"hemmed", "h e m d"},
            {"hemp", "h e m p"},
            {"hems", "h e m z"},
            {"hen", "h e n"},
            {"hence", "h e n s"},
            {"hench", "h e n tS"},
            {"heng", "h e N"},
            {"henke", "h e N k"},
            {"henn", "h e n"},
            {"hennes", "h e n z"},
            {"hens", "h e n z"},
            {"hen's", "h e n z"},
            {"henze", "h e n z"},
            {"hep", "h e p"},
            {"her", "h @r"},
            {"herb", "@r b"},
            {"herbs", "@r b z"},
            {"herb's", "@r b z"},
            {"herbst", "h @r b s t"},
            {"herd", "h @r d"},
            {"herds", "h @r d z"},
            {"here", "h I@"},
            {"here's", "h I@ z"},
            {"herl", "h @r l"},
            {"herm", "h @r m"},
            {"herms", "h @r m z"},
            {"hern", "h @r n"},
            {"herne", "h @r n"},
            {"herr", "h e@"},
            {"hers", "h @r z"},
            {"hersch", "h @r S"},
            {"hersh", "h @r S"},
            {"hert", "h @r t"},
            {"hertz", "h e@ t s"},
            {"herve", "h @r v"},
            {"he's", "h i: z"},
            {"hess", "h e s"},
            {"hesse", "h e s"},
            {"heth", "h e T"},
            {"hett", "h e t"},
            {"hew", "h j u:"},
            {"hewe", "h j u:"},
            {"hewes", "h j u: z"},
            {"hewn", "h j u: n"},
            {"hews", "h j u: z"},
            {"hext", "h e k s t"},
            {"hey", "h eI"},
            {"heyde", "h eI d"},
            {"heys", "h eI z"},
            {"heyse", "h eI s"},
            {"hi", "h aI"},
            {"hibbs", "h I b z"},
            {"hick", "h I k"},
            {"hicks", "h I k s"},
            {"hid", "h I d"},
            {"hide", "h aI d"},
            {"hides", "h aI d z"},
            {"hiems", "h i: m z"},
            {"hiett", "h aI t"},
            {"higgs", "h I g z"},
            {"high", "h aI"},
            {"highs", "h aI z"},
            {"hight", "h aI t"},
            {"hike", "h aI k"},
            {"hiked", "h aI k t"},
            {"hikes", "h aI k s"},
            {"hild", "h I l d"},
            {"hilde", "h I l d"},
            {"hile", "h aI l"},
            {"hill", "h I l"},
            {"hills", "h I l z"},
            {"hill's", "h I l z"},
            {"hilt", "h I l t"},
            {"hilts", "h I l t s"},
            {"him", "h I m"},
            {"hime", "h aI m"},
            {"hinch", "h I n tS"},
            {"hind", "h aI n d"},
            {"hinds", "h aI n d z"},
            {"hine", "h aI n"},
            {"hines", "h aI n z"},
            {"hing", "h I N"},
            {"hinge", "h I n dZ"},
            {"hinged", "h I n dZ d"},
            {"hint", "h I n t"},
            {"hints", "h I n t s"},
            {"hintze", "h I n t z"},
            {"hinze", "h I n z"},
            {"hip", "h I p"},
            {"hips", "h I p s"},
            {"hirsch", "h @r S"},
            {"hirsh", "h @r S"},
            {"hirst", "h @r s t"},
            {"his", "h I z"},
            {"hiss", "h I s"},
            {"hissed", "h I s t"},
            {"hit", "h I t"},
            {"hitch", "h I tS"},
            {"hitched", "h I tS t"},
            {"hite", "h aI t"},
            {"hits", "h I t s"},
            {"hitt", "h I t"},
            {"hive", "h aI v"},
            {"hives", "h aI v z"},
            {"hmong", "m O: N"},
            {"ho", "h @U"},
            {"hoag", "h @U g"},
            {"hoang", "h @U N"},
            {"hoar", "h O@"},
            {"hoard", "h O@ d"},
            {"hoards", "h O@ d z"},
            {"hoare", "h O@"},
            {"hoarse", "h O@ s"},
            {"hoax", "h @U k s"},
            {"hob", "h Q b"},
            {"hobbs", "h Q b z"},
            {"hobs", "h Q b z"},
            {"hoc", "h Q k"},
            {"hoch", "h Q k"},
            {"hock", "h Q k"},
            {"hodge", "h Q dZ"},
            {"hoe", "h @U"},
            {"hoeg", "h @U g"},
            {"hoehne", "h @U n"},
            {"hoek", "h @U k"},
            {"hoes", "h @U z"},
            {"hoff", "h O: f"},
            {"hog", "h Q g"},
            {"hogg", "h Q g"},
            {"hogs", "h Q g z"},
            {"hogue", "h @U g"},
            {"hoh", "h @U"},
            {"hoi", "h OI"},
            {"hoist", "h OI s t"},
            {"hoists", "h OI s t s"},
            {"hoit", "h OI t"},
            {"hoke", "h @U k"},
            {"hold", "h @U l d"},
            {"holds", "h @U l d z"},
            {"hole", "h @U l"},
            {"holed", "h @U l d"},
            {"holes", "h @U l z"},
            {"holl", "h Q l"},
            {"holle", "h Q l"},
            {"holm", "h @U m"},
            {"holme", "h @U l m"},
            {"holmes", "h @U m z"},
            {"holst", "h @U l s t"},
            {"holt", "h @U l t"},
            {"holts", "h @U l t s"},
            {"holt's", "h @U l t s"},
            {"holtz", "h @U l t s"},
            {"hom", "h Q m"},
            {"home", "h @U m"},
            {"homed", "h @U m d"},
            {"homes", "h @U m z"},
            {"home's", "h @U m z"},
            {"homme", "h Q m"},
            {"hommes", "h Q m z"},
            {"homs", "h Q m z"},
            {"hon", "h Q n"},
            {"hone", "h @U n"},
            {"honed", "h @U n d"},
            {"hong", "h O: N"},
            {"honk", "h Q N k"},
            {"hons", "h Q n z"},
            {"hoo", "h u:"},
            {"hooch", "h u: tS"},
            {"hood", "h U d"},
            {"hoods", "h U d z"},
            {"hoof", "h u: f"},
            {"hoofed", "h U f t"},
            {"hoofs", "h U f s"},
            {"hooge", "h u: dZ"},
            {"hook", "h U k"},
            {"hooke", "h U k"},
            {"hooked", "h U k t"},
            {"hooks", "h U k s"},
            {"hook's", "h U k s"},
            {"hoon", "h u: n"},
            {"hoop", "h u: p"},
            {"hoopes", "h u: p s"},
            {"hoops", "h u: p s"},
            {"hoos", "h u: z"},
            {"hoose", "h u: s"},
            {"hoot", "h u: t"},
            {"hoots", "h u: t s"},
            {"hooves", "h U v z"},
            {"hop", "h Q p"},
            {"hope", "h @U p"},
            {"hoped", "h @U p t"},
            {"hopes", "h @U p s"},
            {"hope's", "h @U p s"},
            {"hoppe", "h Q p"},
            {"hopped", "h Q p t"},
            {"hops", "h Q p s"},
            {"hord", "h O@ d"},
            {"horde", "h O@ d"},
            {"hordes", "h O@ d z"},
            {"horn", "h O@ n"},
            {"horne", "h O@ n"},
            {"horned", "h O@ n d"},
            {"horns", "h O@ n z"},
            {"horn's", "h O@ n z"},
            {"hors", "h O@ z"},
            {"horse", "h O@ s"},
            {"horst", "h O@ s t"},
            {"ho's", "h @U z"},
            {"hose", "h @U z"},
            {"hosed", "h @U z d"},
            {"hoss", "h Q s"},
            {"host", "h @U s t"},
            {"hosts", "h @U s t s"},
            {"host's", "h @U s t s"},
            {"hot", "h Q t"},
            {"hoth", "h Q T"},
            {"hots", "h Q t s"},
            {"hotze", "h @U t z"},
            {"houck", "h aU k"},
            {"houff", "h @U f"},
            {"hough", "h V f"},
            {"houghs", "h aU z"},
            {"hound", "h aU n d"},
            {"hounds", "h aU n d z"},
            {"house", "h aU s"},
            {"housed", "h aU z d"},
            {"hout", "h aU t"},
            {"houts", "h aU t s"},
            {"hove", "h @U v"},
            {"how", "h aU"},
            {"howe", "h aU"},
            {"howes", "h aU z"},
            {"howe's", "h aU z"},
            {"howk", "h aU k"},
            {"howl", "h aU l"},
            {"howled", "h aU l d"},
            {"howls", "h aU l z"},
            {"how's", "h aU z"},
            {"hoy", "h OI"},
            {"hoye", "h OI"},
            {"hoyle", "h OI l"},
            {"hoyt", "h OI t"},
            {"hsia", "S Q"},
            {"hsu", "S u:"},
            {"hu", "h u:"},
            {"huan", "h w Q n"},
            {"huang", "h w { N"},
            {"hub", "h V b"},
            {"hube", "h j u: b"},
            {"hubs", "h V b z"},
            {"huck", "h V k"},
            {"hucks", "h V k s"},
            {"hud", "h V d"},
            {"hud's", "h V d z"},
            {"hue", "h j u:"},
            {"hued", "h j u: d"},
            {"hues", "h j u: z"},
            {"huff", "h V f"},
            {"huffed", "h V f t"},
            {"huffs", "h V f s"},
            {"huff's", "h V f s"},
            {"hug", "h V g"},
            {"huge", "h j u: dZ"},
            {"hugged", "h V g d"},
            {"hugh", "h j u:"},
            {"hughes", "h j u: z"},
            {"hugs", "h V g z"},
            {"huh", "h V"},
            {"hulk", "h V l k"},
            {"hulks", "h V l k s"},
            {"hull", "h V l"},
            {"hulled", "h V l d"},
            {"hulls", "h V l z"},
            {"hull's", "h V l z"},
            {"hulme", "h V l m"},
            {"hum", "h V m"},
            {"hume", "h j u: m"},
            {"hump", "h V m p"},
            {"humped", "h V m p t"},
            {"humph", "h V m f"},
            {"hums", "h V m z"},
            {"hun", "h V n"},
            {"hunch", "h V n tS"},
            {"hunched", "h V n tS t"},
            {"hund", "h V n d"},
            {"hung", "h V N"},
            {"hunk", "h V N k"},
            {"hunks", "h V N k s"},
            {"hunt", "h V n t"},
            {"hunts", "h V n t s"},
            {"hunt's", "h V n t s"},
            {"hurd", "h @r d"},
            {"hurl", "h @r l"},
            {"hurled", "h @r l d"},
            {"hurst", "h @r s t"},
            {"hurt", "h @r t"},
            {"hurts", "h @r t s"},
            {"huse", "h j u: z"},
            {"hush", "h V S"},
            {"hushed", "h V S t"},
            {"husk", "h V s k"},
            {"husks", "h V s k s"},
            {"huss", "h V s"},
            {"hust", "h V s t"},
            {"hut", "h V t"},
            {"hutch", "h V tS"},
            {"huts", "h V t s"},
            {"hut's", "h V t s"},
            {"hutt", "h V t"},
            {"hux", "h V k s"},
            {"hwan", "h w Q n"},
            {"hwang", "h w { N"},
            {"hwan's", "h w Q n z"},
            {"hy", "h aI"},
            {"hyde", "h aI d"},
            {"hyde's", "h aI d z"},
            {"hye", "h aI"},
            {"hykes", "h aI k s"},
            {"hyle", "h aI l"},
            {"hymn", "h I m"},
            {"hymns", "h I m z"},
            {"hynd", "h I n d"},
            {"hynes", "h aI n z"},
            {"hype", "h aI p"},
            {"hyped", "h aI p t"},
            {"hypes", "h aI p s"},
            {"i", "aI"},
            {"i.", "aI"},
            {"ib", "I b"},
            {"ice", "aI s"},
            {"iced", "aI s t"},
            {"ich", "I tS"},
            {"ickes", "I k s"},
            {"id", "I d"},
            {"i'd", "aI d"},
            {"ide", "aI d"},
            {"ierne", "I@ n"},
            {"if", "I f"},
            {"ifs", "I f s"},
            {"ige", "aI dZ"},
            {"ike", "aI k"},
            {"ike's", "aI k s"},
            {"il", "I l"},
            {"iles", "aI l z"},
            {"ilk", "I l k"},
            {"ill", "I l"},
            {"i'll", "aI l"},
            {"ills", "I l z"},
            {"ilse", "I l s"},
            {"im", "I m"},
            {"i'm", "aI m"},
            {"imp", "I m p"},
            {"in", "I n"},
            {"in.", "I n"},
            {"inc", "I N k"},
            {"inc.", "I N k"},
            {"inch", "I n tS"},
            {"inched", "I n tS t"},
            {"ing", "I N"},
            {"inge", "I n dZ"},
            {"ink", "I N k"},
            {"inks", "I N k s"},
            {"inn", "I n"},
            {"inns", "I n z"},
            {"inn's", "I n z"},
            {"ins", "I n z"},
            {"ip", "I p"},
            {"ire", "aI r"},
            {"irk", "@r k"},
            {"irked", "@r k t"},
            {"irks", "@r k s"},
            {"irv", "@r v"},
            {"is", "I z"},
            {"i's", "aI z"},
            {"isch", "I S"},
            {"ise", "aI z"},
            {"ish", "I S"},
            {"isle", "aI l"},
            {"isles", "aI l z"},
            {"it", "I t"},
            {"itch", "I tS"},
            {"its", "I t s"},
            {"it's", "I t s"},
            {"i've", "aI v"},
            {"ives", "aI v z"},
            {"j", "dZ eI"},
            {"j.", "dZ eI"},
            {"ja", "j Q"},
            {"jaan", "j Q n"},
            {"jab", "dZ { b"},
            {"jabbed", "dZ { b d"},
            {"jabs", "dZ { b z"},
            {"jac", "dZ { k"},
            {"jack", "dZ { k"},
            {"jacked", "dZ { k t"},
            {"jacks", "dZ { k s"},
            {"jack's", "dZ { k s"},
            {"jacque", "Z eI k"},
            {"jacques", "Z Q k"},
            {"jade", "dZ eI d"},
            {"jades", "dZ eI d z"},
            {"jae", "dZ eI"},
            {"jae's", "dZ eI z"},
            {"jagged", "dZ { g d"},
            {"jags", "dZ { g z"},
            {"jahn", "dZ { n"},
            {"jai", "dZ aI"},
            {"jail", "dZ eI l"},
            {"jailed", "dZ eI l d"},
            {"jails", "dZ eI l z"},
            {"jail's", "dZ eI l z"},
            {"jain", "dZ aI n"},
            {"jake", "dZ eI k"},
            {"jakes", "dZ eI k s"},
            {"jake's", "dZ eI k s"},
            {"jal", "dZ Q l"},
            {"jam", "dZ { m"},
            {"jamb", "dZ { m"},
            {"james", "dZ eI m z"},
            {"jammed", "dZ { m d"},
            {"jams", "dZ { m z"},
            {"jam's", "dZ { m z"},
            {"jan", "dZ { n"},
            {"jan.", "dZ { n"},
            {"jane", "dZ eI n"},
            {"janes", "dZ eI n z"},
            {"jane's", "dZ eI n z"},
            {"jank", "dZ { N k"},
            {"jann", "dZ { n"},
            {"jan's", "dZ { n z"},
            {"jap", "dZ { p"},
            {"japs", "dZ { p s"},
            {"jaques", "dZ { k s"},
            {"jar", "dZ Q@"},
            {"jarred", "dZ Q@ d"},
            {"jars", "dZ Q@ z"},
            {"jas", "dZ eI m z"},
            {"jass", "dZ { s"},
            {"jauch", "dZ O: tS"},
            {"jaunt", "dZ O: n t"},
            {"jaunts", "dZ O: n t s"},
            {"jaw", "dZ O:"},
            {"jawed", "dZ O: d"},
            {"jaws", "dZ O: z"},
            {"jay", "dZ eI"},
            {"jaye", "dZ eI"},
            {"jayne", "dZ eI n"},
            {"jaynes", "dZ eI n z"},
            {"jays", "dZ eI z"},
            {"jay's", "dZ eI z"},
            {"jazz", "dZ { z"},
            {"jazzed", "dZ { z d"},
            {"je", "dZ i:"},
            {"jean", "dZ i: n"},
            {"jeane", "dZ i: n"},
            {"jeanne", "dZ i: n"},
            {"jeans", "dZ i: n z"},
            {"jean's", "dZ i: n z"},
            {"jeb", "dZ e b"},
            {"jed", "dZ e d"},
            {"jee", "dZ i:"},
            {"jeep", "dZ i: p"},
            {"jeeps", "dZ i: p s"},
            {"jeep's", "dZ i: p s"},
            {"jeer", "dZ I@"},
            {"jeered", "dZ I@ d"},
            {"jeers", "dZ i: r z"},
            {"jeez", "dZ i: z"},
            {"jeff", "dZ e f"},
            {"jeffs", "dZ e f s"},
            {"jeff's", "dZ e f s"},
            {"jelks", "dZ e l k s"},
            {"jell", "dZ e l"},
            {"jem", "dZ e m"},
            {"jem's", "dZ e m z"},
            {"jen", "dZ e n"},
            {"jenks", "dZ e N k s"},
            {"jenn", "dZ e n"},
            {"jenne", "dZ e n"},
            {"jenn's", "dZ e n z"},
            {"jens", "dZ e n z"},
            {"jerk", "dZ @r k"},
            {"jerked", "dZ @r k t"},
            {"jerks", "dZ @r k s"},
            {"jess", "dZ e s"},
            {"jest", "dZ e s t"},
            {"jests", "dZ e s t s"},
            {"jet", "dZ e t"},
            {"jets", "dZ e t s"},
            {"jet's", "dZ e t s"},
            {"jew", "dZ u:"},
            {"jews", "dZ u: z"},
            {"jews'", "dZ u: z"},
            {"jfet", "dZ f e t"},
            {"ji", "dZ i:"},
            {"jib", "dZ I b"},
            {"jibe", "dZ aI b"},
            {"jibes", "dZ aI b z"},
            {"jig", "dZ I g"},
            {"jigs", "dZ I g z"},
            {"jill", "dZ I l"},
            {"jilt", "dZ I l t"},
            {"jim", "dZ I m"},
            {"jim's", "dZ I m z"},
            {"jin", "dZ I n"},
            {"jing", "dZ I N"},
            {"jinks", "dZ I N k s"},
            {"jinx", "dZ I N k s"},
            {"jive", "dZ aI v"},
            {"jo", "dZ @U"},
            {"joan", "dZ @U n"},
            {"job", "dZ Q b"},
            {"jobe", "dZ @U b"},
            {"jobes", "dZ @U b z"},
            {"jobs", "dZ Q b z"},
            {"job's", "dZ Q b z"},
            {"jock", "dZ Q k"},
            {"jocks", "dZ Q k s"},
            {"joe", "dZ @U"},
            {"joerg", "dZ O@ g"},
            {"joes", "dZ @U z"},
            {"joe's", "dZ @U z"},
            {"jog", "dZ Q g"},
            {"jogged", "dZ Q g d"},
            {"jogs", "dZ Q g z"},
            {"joh", "dZ @U"},
            {"john", "dZ Q n"},
            {"johns", "dZ Q n z"},
            {"john's", "dZ Q n z"},
            {"joice", "dZ OI s"},
            {"joie", "Z w Q"},
            {"join", "dZ OI n"},
            {"joined", "dZ OI n d"},
            {"joins", "dZ OI n z"},
            {"joint", "dZ OI n t"},
            {"joints", "dZ OI n t s"},
            {"joist", "dZ OI s t"},
            {"joists", "dZ OI s t s"},
            {"joke", "dZ @U k"},
            {"joked", "dZ @U k t"},
            {"jokes", "dZ @U k s"},
            {"joles", "dZ @U l z"},
            {"jolt", "dZ @U l t"},
            {"jolts", "dZ @U l t s"},
            {"jon", "dZ Q n"},
            {"jone", "dZ @U n"},
            {"jones", "dZ @U n z"},
            {"jones'", "dZ @U n z"},
            {"jon's", "dZ Q n z"},
            {"joo", "dZ u:"},
            {"joon", "dZ u: n"},
            {"joost", "dZ u: s t"},
            {"jos", "dZ @U z"},
            {"josh", "dZ Q S"},
            {"joss", "dZ O: s"},
            {"jot", "dZ Q t"},
            {"joule", "dZ u: l"},
            {"joules", "dZ u: l z"},
            {"jour", "dZ aU r"},
            {"joust", "dZ aU s t"},
            {"jowl", "dZ aU l"},
            {"jowls", "dZ aU l z"},
            {"joy", "dZ OI"},
            {"joyce", "dZ OI s"},
            {"joye", "dZ OI"},
            {"joys", "dZ OI z"},
            {"joy's", "dZ OI z"},
            {"j's", "dZ eI z"},
            {"ju", "dZ u:"},
            {"juan", "w Q n"},
            {"judd", "dZ V d"},
            {"jude", "dZ u: d"},
            {"jude's", "dZ u: d z"},
            {"judge", "dZ V dZ"},
            {"judged", "dZ V dZ d"},
            {"jug", "dZ V g"},
            {"jugs", "dZ V g z"},
            {"juice", "dZ u: s"},
            {"juke", "dZ u: k"},
            {"jukes", "dZ u: k s"},
            {"jule", "dZ u: l"},
            {"jules", "dZ u: l z"},
            {"jump", "dZ V m p"},
            {"jumped", "dZ V m p t"},
            {"jumps", "dZ V m p s"},
            {"jun", "dZ V n"},
            {"june", "dZ u: n"},
            {"junes", "dZ u: n z"},
            {"june's", "dZ u: n z"},
            {"jung", "j U N"},
            {"jung's", "j U N z"},
            {"junk", "dZ V N k"},
            {"junked", "dZ V N k t"},
            {"jure", "dZ U@"},
            {"just", "dZ V s t"},
            {"jut", "dZ V t"},
            {"jute", "dZ u: t"},
            {"juts", "dZ V t s"},
            {"juve", "dZ u: v"},
            {"jynx", "dZ I N k s"},
            {"k", "kh eI"},
            {"k.", "kh eI"},
            {"ka", "kh Q"},
            {"kaas", "kh Q z"},
            {"kade", "kh eI d"},
            {"kahl", "kh Q l"},
            {"kahle", "kh Q l"},
            {"kahn", "kh Q n"},
            {"kahn's", "kh Q n z"},
            {"kai", "kh aI"},
            {"kail", "kh eI l"},
            {"kaim", "kh eI m"},
            {"kain", "kh eI n"},
            {"kaine", "kh eI n"},
            {"kal", "kh { l"},
            {"kalb", "kh { l b"},
            {"kale", "kh eI l"},
            {"kalk", "kh O: k"},
            {"kall", "kh O: l"},
            {"kam", "kh { m"},
            {"kamp", "kh { m p"},
            {"kampf", "kh { m p f"},
            {"kamp's", "kh { m p s"},
            {"kan", "kh { n"},
            {"kane", "kh eI n"},
            {"kane's", "kh eI n z"},
            {"kang", "kh { N"},
            {"kanpur", "kh Q@ n p r r"},
            {"kant", "kh { n t"},
            {"kao", "kh aU"},
            {"kapp", "kh { p"},
            {"karch", "kh Q@ k"},
            {"karl", "kh Q@ l"},
            {"karl's", "kh Q@ l z"},
            {"karn", "kh Q@ n"},
            {"karnes", "kh Q@ n z"},
            {"karns", "kh Q@ n z"},
            {"karp", "kh Q@ p"},
            {"karr", "kh Q@"},
            {"karst", "kh Q@ s t"},
            {"kase", "kh eI z"},
            {"kass", "kh { s"},
            {"kast", "kh { s t"},
            {"kat", "kh { t"},
            {"kate", "kh eI t"},
            {"kate's", "kh eI t s"},
            {"kath", "kh { T"},
            {"kats", "kh { t s"},
            {"katt", "kh { t"},
            {"katz", "kh { t s"},
            {"kay", "kh eI"},
            {"kaye", "kh eI"},
            {"kayes", "kh eI z"},
            {"kayne", "kh eI n"},
            {"kays", "kh eI z"},
            {"kay's", "kh eI z"},
            {"kea", "kh i:"},
            {"keach", "kh i: tS"},
            {"kean", "kh i: n"},
            {"keane", "kh i: n"},
            {"kean's", "kh i: n z"},
            {"kearn", "kh @r n"},
            {"kearns", "kh @r n z"},
            {"keas", "kh i: z"},
            {"keats", "kh i: t s"},
            {"keck", "kh e k"},
            {"keds", "kh e d z"},
            {"kee", "kh i:"},
            {"keech", "kh i: tS"},
            {"keef", "kh i: f"},
            {"keefe", "kh i: f"},
            {"keel", "kh i: l"},
            {"keele", "kh i: l"},
            {"keels", "kh i: l z"},
            {"keen", "kh i: n"},
            {"keene", "kh i: n"},
            {"keep", "kh i: p"},
            {"keeps", "kh i: p s"},
            {"kee's", "kh i: z"},
            {"keese", "kh i: z"},
            {"keg", "kh e g"},
            {"kegs", "kh e g z"},
            {"kehr", "kh e@"},
            {"keil", "kh i: l"},
            {"keir", "kh i: r"},
            {"keith", "kh i: T"},
            {"keith's", "kh i: T s"},
            {"kell", "kh e l"},
            {"kells", "kh e l z"},
            {"kelp", "kh e l p"},
            {"kelps", "kh e l p s"},
            {"kemp", "kh e m p"},
            {"kempe", "kh e m p"},
            {"kemp's", "kh e m p s"},
            {"ken", "kh e n"},
            {"kenn", "kh e n"},
            {"ken's", "kh e n z"},
            {"kent", "kh e n t"},
            {"kent's", "kh e n t s"},
            {"kept", "kh e p t"},
            {"ker", "kh @r"},
            {"kerb", "kh @r b"},
            {"kerbs", "kh @r b z"},
            {"kerl", "kh @r l"},
            {"kern", "kh @r n"},
            {"kernes", "kh @r n z"},
            {"kerns", "kh @r n z"},
            {"kern's", "kh @r n z"},
            {"kerr", "kh @r"},
            {"kerst", "kh @r s t"},
            {"ketch", "kh e tS"},
            {"kew", "kh j u:"},
            {"key", "kh i:"},
            {"keyed", "kh i: d"},
            {"keyes", "kh i: z"},
            {"keynes", "kh eI n z"},
            {"keys", "kh i: z"},
            {"key's", "kh i: z"},
            {"khan", "kh Q n"},
            {"khan's", "kh Q n z"},
            {"khat", "kh Q t"},
            {"khem", "kh e m"},
            {"khmer", "kh m e@"},
            {"ki", "kh i:"},
            {"kibbe", "kh I b"},
            {"kick", "kh I k"},
            {"kicked", "kh I k t"},
            {"kicks", "kh I k s"},
            {"kid", "kh I d"},
            {"kidd", "kh I d"},
            {"kidde", "kh I d"},
            {"kids", "kh I d z"},
            {"kid's", "kh I d z"},
            {"kids'", "kh I d z"},
            {"kief", "kh i: f"},
            {"kiehl", "kh i: l"},
            {"kiehn", "kh i: n"},
            {"kiel", "kh i: l"},
            {"kier", "kh I@"},
            {"kies", "kh aI s"},
            {"kight", "kh aI t"},
            {"kile", "kh aI l"},
            {"kill", "kh I l"},
            {"killed", "kh I l d"},
            {"kills", "kh I l z"},
            {"kiln", "kh I l n"},
            {"kilns", "kh I l n z"},
            {"kilt", "kh I l t"},
            {"kilts", "kh I l t s"},
            {"kim", "kh I m"},
            {"kim's", "kh I m z"},
            {"kin", "kh I n"},
            {"kinch", "kh I n tS"},
            {"kind", "kh aI n d"},
            {"kinds", "kh aI n d z"},
            {"kines", "kh aI n z"},
            {"king", "kh I N"},
            {"kings", "kh I N z"},
            {"king's", "kh I N z"},
            {"kings'", "kh I N z"},
            {"kinks", "kh I N k s"},
            {"kin's", "kh I n z"},
            {"kip", "kh I p"},
            {"kipp", "kh I p"},
            {"kirch", "kh @r k"},
            {"kirk", "kh @r k"},
            {"kirks", "kh @r k s"},
            {"kirk's", "kh @r k s"},
            {"kirn", "kh @r n"},
            {"kirsch", "kh @r S"},
            {"kirst", "kh @r s t"},
            {"kirt", "kh @r t"},
            {"kish", "kh I S"},
            {"kiss", "kh I s"},
            {"kissed", "kh I s t"},
            {"kist", "kh I s t"},
            {"kit", "kh I t"},
            {"kite", "kh aI t"},
            {"kites", "kh aI t s"},
            {"kits", "kh I t s"},
            {"kitsch", "kh I tS"},
            {"kitt", "kh I t"},
            {"kitts", "kh I t s"},
            {"klan", "kh l { n"},
            {"klang", "kh l { N"},
            {"klans", "kh l { n z"},
            {"klan's", "kh l { n z"},
            {"klapp", "kh l { p"},
            {"klatt", "kh l { t"},
            {"klaus", "kh l aU s"},
            {"klee", "kh l i:"},
            {"klee's", "kh l i: z"},
            {"klein", "kh l aI n"},
            {"klein's", "kh l aI n z"},
            {"kleist", "kh l aI s t"},
            {"klemm", "kh l e m"},
            {"klemme", "kh l e m"},
            {"klenk", "kh l e N k"},
            {"kleve", "kh l i: v"},
            {"klick", "kh l I k"},
            {"klieg", "kh l i: g"},
            {"kline", "kh l aI n"},
            {"kling", "kh l I N"},
            {"kloc", "kh l Q k"},
            {"klom", "kh l O: m"},
            {"klos", "kh l Q s"},
            {"klotz", "kh l Q t s"},
            {"kluck", "kh l V k"},
            {"klug", "kh l V g"},
            {"kluge", "kh l u: dZ"},
            {"klump", "kh l V m p"},
            {"klunk", "kh l V N k"},
            {"klute", "kh l u: t"},
            {"klutz", "kh l V t s"},
            {"klux", "kh l V k s"},
            {"knab", "n { b"},
            {"knack", "n { k"},
            {"knacks", "n { k s"},
            {"knape", "n eI p"},
            {"knapp", "n { p"},
            {"knapp's", "n { p s"},
            {"knave", "n eI v"},
            {"knaves", "n eI v z"},
            {"knead", "n i: d"},
            {"knee", "n i:"},
            {"kneed", "n i: d"},
            {"kneel", "n i: l"},
            {"knees", "n i: z"},
            {"knell", "n e l"},
            {"knelt", "n e l t"},
            {"knew", "n u:"},
            {"knick", "n I k"},
            {"knicks", "n I k s"},
            {"knies", "n aI z"},
            {"knife", "n aI f"},
            {"knifed", "n aI f t"},
            {"knight", "n aI t"},
            {"knights", "n aI t s"},
            {"knight's", "n aI t s"},
            {"knipe", "n aI p"},
            {"knit", "n I t"},
            {"knits", "n I t s"},
            {"knives", "n aI v z"},
            {"knob", "n Q b"},
            {"knobs", "n Q b z"},
            {"knock", "n Q k"},
            {"knocked", "n Q k t"},
            {"knocks", "n Q k s"},
            {"knoke", "n @U k"},
            {"knoll", "n @U l"},
            {"knoll's", "n @U l z"},
            {"knoop", "n u: p"},
            {"knop", "n Q p"},
            {"knopf", "n Q p f"},
            {"knorr", "n O@"},
            {"knot", "n Q t"},
            {"knots", "n Q t s"},
            {"knott", "n Q t"},
            {"knott's", "n Q t s"},
            {"know", "n @U"},
            {"knowles", "n @U l z"},
            {"known", "n @U n"},
            {"knowns", "n @U n z"},
            {"knows", "n @U z"},
            {"knox", "n Q k s"},
            {"knut", "n V t"},
            {"knute", "n u: t"},
            {"knuth", "n u: T"},
            {"knuts", "n V t s"},
            {"ko", "kh @U"},
            {"kobs", "kh Q b z"},
            {"koch", "kh O: tS"},
            {"koff", "kh O: f"},
            {"koh", "kh @U"},
            {"kohl", "kh @U l"},
            {"kohls", "kh @U l z"},
            {"kohl's", "kh @U l z"},
            {"kohn", "kh Q n"},
            {"kol", "kh @U l"},
            {"kolb", "kh @U l b"},
            {"kolbe", "kh @U l b"},
            {"kolk", "kh @U k"},
            {"kon", "kh Q n"},
            {"kong", "kh O: N"},
            {"kong's", "kh O: N z"},
            {"koo", "kh u:"},
            {"kook", "kh u: k"},
            {"kooks", "kh u: k s"},
            {"koontz", "kh u: n t s"},
            {"kopp", "kh Q p"},
            {"kops", "kh Q p s"},
            {"korff", "kh O@ f"},
            {"korns", "kh O: n z"},
            {"kort", "kh O@ t"},
            {"kos", "kh Q s"},
            {"koss", "kh O: s"},
            {"kotz", "kh Q t s"},
            {"krafft", "kh r { f t"},
            {"kraft", "kh r { f t"},
            {"kraft's", "kh r { f t s"},
            {"kragh", "kh r { g"},
            {"kral", "kh r { l"},
            {"krall", "kh r O: l"},
            {"kram", "kh r { m"},
            {"krans", "kh r { n z"},
            {"krantz", "kh r { n t s"},
            {"kranz", "kh r { n z"},
            {"kras", "kh r { s"},
            {"kraus", "kh r aU s"},
            {"krause", "kh r O: s"},
            {"krauss", "kh r aU s"},
            {"kraut", "kh r aU t"},
            {"krebs", "kh r e b z"},
            {"kreg", "kh r e g"},
            {"krein", "kh r eI n"},
            {"kreis", "kh r i: z"},
            {"krell", "kh r e l"},
            {"krenn", "kh r e n"},
            {"kress", "kh r e s"},
            {"krieg", "kh r i: g"},
            {"krill", "kh r I l"},
            {"krim", "kh r I m"},
            {"kris", "kh r I s"},
            {"kriss", "kh r I s"},
            {"kroc", "kh r Q k"},
            {"krock", "kh r Q k"},
            {"kroc's", "kh r Q k s"},
            {"krogh", "kh r @U"},
            {"kroll", "kh r O: l"},
            {"krome", "kh r @U m"},
            {"kroon", "kh r u: n"},
            {"krug", "kh r V g"},
            {"krum", "kh r V m"},
            {"krupp", "kh r V p"},
            {"krupp's", "kh r V p s"},
            {"kruse", "kh r u: z"},
            {"krutz", "kh r V t s"},
            {"k's", "kh eI z"},
            {"ku", "kh u:"},
            {"kuan", "kh w Q n"},
            {"kuch", "kh V tS"},
            {"kuehn", "kh u: n"},
            {"kuhn", "kh u: n"},
            {"kun", "kh V n"},
            {"kung", "kh V N"},
            {"kunz", "kh V n z"},
            {"kurd", "kh @r d"},
            {"kurds", "kh @r d z"},
            {"kurt", "kh @r t"},
            {"kurth", "kh @r T"},
            {"kurtz", "kh @r t s"},
            {"kus", "kh V s"},
            {"kusch", "kh V S"},
            {"kush", "kh U S"},
            {"kutch", "kh V tS"},
            {"kwai", "kh w aI"},
            {"kwan", "kh w Q n"},
            {"kwang", "kh w Q N"},
            {"kwok", "kh w Q k"},
            {"kwon", "kh w Q n"},
            {"kyd", "kh I d"},
            {"kyd's", "kh I d z"},
            {"kyes", "kh aI z"},
            {"kyl", "kh aI l"},
            {"kyle", "kh aI l"},
            {"kyles", "kh aI l z"},
            {"kyle's", "kh aI l z"},
            {"kym", "kh I m"},
            {"kyne", "kh aI n"},
            {"kyte", "kh aI t"},
            {"kyu", "kh j u:"},
            {"l", "e l"},
            {"l.", "e l"},
            {"la", "l0 Q"},
            {"laas", "l0 Q z"},
            {"lab", "l0 { b"},
            {"labe", "l0 eI b"},
            {"labs", "l0 { b z"},
            {"lab's", "l0 { b z"},
            {"lac", "l0 { k"},
            {"lace", "l0 eI s"},
            {"laced", "l0 eI s t"},
            {"lach", "l0 { tS"},
            {"lache", "l0 { tS"},
            {"lack", "l0 { k"},
            {"lacked", "l0 { k t"},
            {"lacks", "l0 { k s"},
            {"lac's", "l0 { k s"},
            {"lad", "l0 { d"},
            {"ladd", "l0 { d"},
            {"ladd's", "l0 { d z"},
            {"lade", "l0 eI d"},
            {"lads", "l0 { d z"},
            {"laff", "l0 { f"},
            {"lag", "l0 { g"},
            {"lage", "l0 eI dZ"},
            {"lagged", "l0 { g d"},
            {"lags", "l0 { g z"},
            {"lai", "l0 aI"},
            {"laid", "l0 eI d"},
            {"lail", "l0 eI l"},
            {"lain", "l0 eI n"},
            {"laine", "l0 eI n"},
            {"lair", "l0 e@"},
            {"laird", "l0 e@ d"},
            {"lais", "l0 eI z"},
            {"lait", "l0 eI"},
            {"lak", "l0 { k"},
            {"lake", "l0 eI k"},
            {"lakes", "l0 eI k s"},
            {"lake's", "l0 eI k s"},
            {"lal", "l0 { l"},
            {"lall", "l0 O: l"},
            {"lam", "l0 { m"},
            {"lamb", "l0 { m"},
            {"lambs", "l0 { m z"},
            {"lamb's", "l0 { m z"},
            {"lame", "l0 eI m"},
            {"lamm", "l0 { m"},
            {"lamp", "l0 { m p"},
            {"lampe", "l0 { m p"},
            {"lampf", "l0 { m p f"},
            {"lamps", "l0 { m p s"},
            {"lan", "l0 { n"},
            {"lance", "l0 { n s"},
            {"land", "l0 { n d"},
            {"lande", "l0 { n d"},
            {"landes", "l0 { n d z"},
            {"lands", "l0 { n d z"},
            {"land's", "l0 { n d z"},
            {"lane", "l0 eI n"},
            {"lanes", "l0 eI n z"},
            {"lane's", "l0 eI n z"},
            {"lang", "l0 { N"},
            {"lange", "l0 { N"},
            {"lange's", "l0 { N z"},
            {"lank", "l0 { N k"},
            {"lant", "l0 { n t"},
            {"lantz", "l0 { n t s"},
            {"lao", "l0 aU"},
            {"lap", "l0 { p"},
            {"lapp", "l0 { p"},
            {"lapped", "l0 { p t"},
            {"lapps", "l0 { p s"},
            {"laps", "l0 { p s"},
            {"lapse", "l0 { p s"},
            {"lapsed", "l0 { p s t"},
            {"lar", "l0 Q@"},
            {"larch", "l0 Q@ tS"},
            {"lard", "l0 Q@ d"},
            {"lare", "l0 e@"},
            {"large", "l0 Q@ dZ"},
            {"lark", "l0 Q@ k"},
            {"larks", "l0 Q@ k s"},
            {"larned", "l0 Q@ n d"},
            {"lars", "l0 Q@ z"},
            {"las", "l0 Q s"},
            {"lase", "l0 eI z"},
            {"lash", "l0 { S"},
            {"lashed", "l0 { S t"},
            {"lask", "l0 { s k"},
            {"lass", "l0 { s"},
            {"last", "l0 { s t"},
            {"lasts", "l0 { s t s"},
            {"lat", "l0 Q t"},
            {"latch", "l0 { tS"},
            {"latched", "l0 { tS t"},
            {"late", "l0 eI t"},
            {"lath", "l0 { T"},
            {"lathe", "l0 eI D"},
            {"lathes", "l0 eI D z"},
            {"lats", "l0 Q t s"},
            {"lau", "l0 aU"},
            {"lauch", "l0 Q tS"},
            {"laud", "l0 O: d"},
            {"laude", "l0 O: d"},
            {"lauds", "l0 O: d z"},
            {"lauf", "l0 O: f"},
            {"laugh", "l0 { f"},
            {"laughed", "l0 { f t"},
            {"laughs", "l0 { f s"},
            {"laun", "l0 O: n"},
            {"launch", "l0 O: n tS"},
            {"launched", "l0 O: n tS t"},
            {"laur", "l0 O@"},
            {"laure", "l0 O@"},
            {"laux", "l0 O: k s"},
            {"lave", "l0 eI v"},
            {"law", "l0 Q"},
            {"lawes", "l0 O: z"},
            {"lawn", "l0 O: n"},
            {"lawns", "l0 O: n z"},
            {"laws", "l0 O: z"},
            {"law's", "l0 O: z"},
            {"lax", "l0 { k s"},
            {"lay", "l0 eI"},
            {"layne", "l0 eI n"},
            {"lays", "l0 eI z"},
            {"lay's", "l0 eI z"},
            {"le", "l0 @"},
            {"lea", "l0 i:"},
            {"leach", "l0 i: tS"},
            {"leached", "l0 i: tS t"},
            {"lead", "l0 i: d"},
            {"leads", "l0 i: d z"},
            {"leaf", "l0 i: f"},
            {"leafed", "l0 i: f t"},
            {"leafs", "l0 i: f s"},
            {"league", "l0 i: g"},
            {"leagues", "l0 i: g z"},
            {"league's", "l0 i: g z"},
            {"leak", "l0 i: k"},
            {"leake", "l0 i: k"},
            {"leaked", "l0 i: k t"},
            {"leaks", "l0 i: k s"},
            {"leal", "l0 i: l"},
            {"lean", "l0 i: n"},
            {"leaned", "l0 i: n d"},
            {"leans", "l0 i: n z"},
            {"leap", "l0 i: p"},
            {"leaped", "l0 e p t"},
            {"leaps", "l0 i: p s"},
            {"leapt", "l0 e p t"},
            {"lear", "l0 I@"},
            {"learn", "l0 @r n"},
            {"learned", "l0 @r n d"},
            {"learns", "l0 @r n z"},
            {"learnt", "l0 @r n t"},
            {"lear's", "l0 i: r z"},
            {"leas", "l0 i: z"},
            {"lease", "l0 i: s"},
            {"leased", "l0 i: s t"},
            {"leash", "l0 i: S"},
            {"leashed", "l0 i: S t"},
            {"least", "l0 i: s t"},
            {"leath", "l0 i: T"},
            {"leave", "l0 i: v"},
            {"leaves", "l0 i: v z"},
            {"lech", "l0 e k"},
            {"leck", "l0 e k"},
            {"led", "l0 e d"},
            {"ledge", "l0 e dZ"},
            {"lee", "l0 i:"},
            {"leech", "l0 i: tS"},
            {"leed", "l0 i: d"},
            {"leeds", "l0 i: d z"},
            {"leek", "l0 i: k"},
            {"leeks", "l0 i: k s"},
            {"leep", "l0 i: p"},
            {"lees", "l0 i: z"},
            {"lee's", "l0 i: z"},
            {"leese", "l0 i: s"},
            {"leet", "l0 i: t"},
            {"leeth", "l0 i: T"},
            {"leff", "l0 e f"},
            {"leff's", "l0 e f s"},
            {"left", "l0 e f t"},
            {"leg", "l0 e g"},
            {"lege", "l0 i: dZ"},
            {"legge", "l0 e g"},
            {"legs", "l0 e g z"},
            {"leg's", "l0 e g z"},
            {"lehr", "l0 e@"},
            {"lei", "l0 eI"},
            {"leif", "l0 i: f"},
            {"leigh", "l0 i:"},
            {"leis", "l0 eI z"},
            {"leiss", "l0 aI s"},
            {"leith", "l0 i: T"},
            {"lek", "l0 e k"},
            {"lem", "l0 e m"},
            {"lemcke", "l0 e m k"},
            {"len", "l0 e n"},
            {"lend", "l0 e n d"},
            {"lends", "l0 e n d z"},
            {"lene", "l0 i: n"},
            {"leng", "l0 e N"},
            {"length", "l0 e N k T"},
            {"lengths", "l0 e N k T s"},
            {"lens", "l0 e n z"},
            {"lent", "l0 e n t"},
            {"lents", "l0 e n t s"},
            {"lenz", "l0 e n z"},
            {"leong", "l0 e N"},
            {"lep", "l0 e p"},
            {"lepp", "l0 e p"},
            {"les", "l0 e s"},
            {"lesh", "l0 e S"},
            {"less", "l0 e s"},
            {"lest", "l0 e s t"},
            {"let", "l0 e t"},
            {"lets", "l0 e t s"},
            {"let's", "l0 e t s"},
            {"lett", "l0 e t"},
            {"letts", "l0 e t s"},
            {"leu", "l0 u:"},
            {"leu's", "l0 u: z"},
            {"lev", "l0 e v"},
            {"leve", "l0 i: v"},
            {"lew", "l0 u:"},
            {"lewd", "l0 u: d"},
            {"lex", "l0 e k s"},
            {"ley", "l0 eI"},
            {"leys", "l0 eI z"},
            {"li", "l0 i:"},
            {"lian", "l0 j Q n"},
            {"liang", "l0 j { N"},
            {"lib", "l0 I b"},
            {"libbed", "l0 I b d"},
            {"lice", "l0 aI s"},
            {"lich", "l0 I tS"},
            {"licht", "l0 I k t"},
            {"lick", "l0 I k"},
            {"licked", "l0 I k t"},
            {"licks", "l0 I k s"},
            {"lid", "l0 I d"},
            {"lide", "l0 aI d"},
            {"lids", "l0 I d z"},
            {"lie", "l0 aI"},
            {"lied", "l0 aI d"},
            {"lief", "l0 i: f"},
            {"liege", "l0 i: dZ"},
            {"lien", "l0 i: n"},
            {"liens", "l0 i: n z"},
            {"lies", "l0 aI z"},
            {"lieu", "l0 u:"},
            {"life", "l0 aI f"},
            {"life's", "l0 aI f s"},
            {"lift", "l0 I f t"},
            {"lifts", "l0 I f t s"},
            {"light", "l0 aI t"},
            {"lights", "l0 aI t s"},
            {"light's", "l0 aI t s"},
            {"like", "l0 aI k"},
            {"liked", "l0 aI k t"},
            {"likes", "l0 aI k s"},
            {"lil", "l0 I l"},
            {"lile", "l0 aI l"},
            {"liles", "l0 aI l z"},
            {"lill", "l0 I l"},
            {"lille", "l0 I l"},
            {"lilt", "l0 I l t"},
            {"lim", "l0 I m"},
            {"limb", "l0 I m"},
            {"limbed", "l0 I m d"},
            {"limbs", "l0 I m z"},
            {"lime", "l0 aI m"},
            {"limes", "l0 aI m z"},
            {"limp", "l0 I m p"},
            {"limped", "l0 I m p t"},
            {"limps", "l0 I m p s"},
            {"lin", "l0 I n"},
            {"linc", "l0 I N k"},
            {"linch", "l0 I n tS"},
            {"lind", "l0 I n d"},
            {"line", "l0 aI n"},
            {"lined", "l0 aI n d"},
            {"lines", "l0 aI n z"},
            {"line's", "l0 aI n z"},
            {"ling", "l0 I N"},
            {"link", "l0 I N k"},
            {"linked", "l0 I N k t"},
            {"links", "l0 I N k s"},
            {"link's", "l0 I N k s"},
            {"linn", "l0 I n"},
            {"linne", "l0 I n"},
            {"lins", "l0 I n z"},
            {"lint", "l0 I n t"},
            {"linz", "l0 I n z"},
            {"lip", "l0 I p"},
            {"lipe", "l0 aI p"},
            {"lipp", "l0 I p"},
            {"lippe", "l0 I p"},
            {"lipped", "l0 I p t"},
            {"lipps", "l0 I p s"},
            {"lips", "l0 I p s"},
            {"lis", "l0 I s"},
            {"li's", "l0 i: z"},
            {"lise", "l0 aI z"},
            {"lish", "l0 I S"},
            {"lisk", "l0 I s k"},
            {"lisp", "l0 I s p"},
            {"liss", "l0 I s"},
            {"list", "l0 I s t"},
            {"lists", "l0 I s t s"},
            {"list's", "l0 I s t s"},
            {"liszt", "l0 I s t"},
            {"lit", "l0 I t"},
            {"lite", "l0 aI t"},
            {"lites", "l0 aI t s"},
            {"lithe", "l0 aI D"},
            {"lits", "l0 I t s"},
            {"litt", "l0 I t"},
            {"litz", "l0 I t s"},
            {"liu", "l0 j u:"},
            {"liv", "l0 I v"},
            {"live", "l0 aI v"},
            {"lived", "l0 aI v d"},
            {"lives", "l0 I v z"},
            {"liz", "l0 I z"},
            {"lloyd", "l0 OI d"},
            {"lloyd's", "l0 OI d z"},
            {"ln", "l0 eI n"},
            {"lo", "l0 @U"},
            {"load", "l0 @U d"},
            {"loads", "l0 @U d z"},
            {"loaf", "l0 @U f"},
            {"loafs", "l0 @U f s"},
            {"loam", "l0 @U m"},
            {"loan", "l0 @U n"},
            {"loaned", "l0 @U n d"},
            {"loans", "l0 @U n z"},
            {"loan's", "l0 @U n z"},
            {"loar", "l0 O@"},
            {"loath", "l0 @U T"},
            {"loathe", "l0 @U D"},
            {"loathed", "l0 @U D d"},
            {"loaves", "l0 @U v z"},
            {"lob", "l0 Q b"},
            {"lobbed", "l0 Q b d"},
            {"lobe", "l0 @U b"},
            {"lobed", "l0 @U b d"},
            {"lobes", "l0 @U b z"},
            {"lobs", "l0 Q b z"},
            {"loch", "l0 Q k"},
            {"lock", "l0 Q k"},
            {"locke", "l0 Q k"},
            {"locked", "l0 Q k t"},
            {"locks", "l0 Q k s"},
            {"lode", "l0 @U d"},
            {"lodes", "l0 @U d z"},
            {"lodge", "l0 Q dZ"},
            {"lodged", "l0 Q dZ d"},
            {"loe", "l0 @U"},
            {"loeb", "l0 @U b"},
            {"loews", "l0 @U z"},
            {"loft", "l0 O: f t"},
            {"lofts", "l0 O: f t s"},
            {"loft's", "l0 O: f t s"},
            {"log", "l0 O: g"},
            {"loge", "l0 @U dZ"},
            {"logged", "l0 O: g d"},
            {"logs", "l0 O: g z"},
            {"lohn", "l0 Q n"},
            {"lohse", "l0 @U s"},
            {"loin", "l0 OI n"},
            {"loire", "l0 OI r"},
            {"lok", "l0 Q k"},
            {"loll", "l0 Q l"},
            {"lomb", "l0 Q m"},
            {"lon", "l0 Q n"},
            {"lone", "l0 @U n"},
            {"long", "l0 O: N"},
            {"longe", "l0 Q n dZ"},
            {"longed", "l0 O: N d"},
            {"longs", "l0 O: N z"},
            {"long's", "l0 O: N z"},
            {"loo", "l0 u:"},
            {"look", "l0 U k"},
            {"looked", "l0 U k t"},
            {"looks", "l0 U k s"},
            {"loom", "l0 u: m"},
            {"loomed", "l0 u: m d"},
            {"looms", "l0 u: m z"},
            {"loon", "l0 u: n"},
            {"loons", "l0 u: n z"},
            {"loop", "l0 u: p"},
            {"looped", "l0 u: p t"},
            {"loops", "l0 u: p s"},
            {"loos", "l0 u: z"},
            {"loose", "l0 u: s"},
            {"loosed", "l0 u: s t"},
            {"loot", "l0 u: t"},
            {"lop", "l0 Q p"},
            {"lope", "l0 @U p"},
            {"lopes", "l0 @U p s"},
            {"lopped", "l0 Q p t"},
            {"lor", "l0 O@"},
            {"lord", "l0 O@ d"},
            {"lords", "l0 O@ d z"},
            {"lord's", "l0 O@ d z"},
            {"lore", "l0 O@"},
            {"lorne", "l0 O@ n"},
            {"los", "l0 @U s"},
            {"lo's", "l0 @U z"},
            {"lose", "l0 u: z"},
            {"losh", "l0 Q S"},
            {"loss", "l0 O: s"},
            {"lost", "l0 O: s t"},
            {"lot", "l0 Q t"},
            {"loth", "l0 Q T"},
            {"lots", "l0 Q t s"},
            {"lot's", "l0 Q t s"},
            {"lott", "l0 Q t"},
            {"lotte", "l0 Q t"},
            {"lottes", "l0 Q t s"},
            {"lott's", "l0 Q t z"},
            {"lotz", "l0 Q t s"},
            {"lotze", "l0 @U t z"},
            {"lou", "l0 u:"},
            {"loud", "l0 aU d"},
            {"lough", "l0 aU"},
            {"louk", "l0 aU k"},
            {"lounge", "l0 aU n dZ"},
            {"loup", "l0 u: p"},
            {"loupe", "l0 u: p"},
            {"lour", "l0 aU r"},
            {"lourdes", "l0 O@ d z"},
            {"lou's", "l0 u: z"},
            {"louse", "l0 aU s"},
            {"louth", "l0 aU T"},
            {"loux", "l0 u:"},
            {"love", "l0 V v"},
            {"loved", "l0 V v d"},
            {"loves", "l0 V v z"},
            {"love's", "l0 V v z"},
            {"low", "l0 @U"},
            {"lowe", "l0 @U"},
            {"lowes", "l0 @U z"},
            {"lowe's", "l0 @U z"},
            {"lown", "l0 aU n"},
            {"lows", "l0 @U z"},
            {"lox", "l0 Q k s"},
            {"loy", "l0 OI"},
            {"loyce", "l0 OI s"},
            {"loyd", "l0 OI d"},
            {"l's", "e l z"},
            {"lu", "l0 u:"},
            {"lube", "l0 u: b"},
            {"lube's", "l0 u: b z"},
            {"lubke", "l0 V b k"},
            {"luce", "l0 u: s"},
            {"luck", "l0 V k"},
            {"lucks", "l0 V k s"},
            {"lue", "l0 u:"},
            {"luebke", "l0 u: b k"},
            {"luelle", "l0 u: l"},
            {"luff", "l0 V f"},
            {"luffed", "l0 V f t"},
            {"lug", "l0 V g"},
            {"luge", "l0 u: dZ"},
            {"lugs", "l0 V g z"},
            {"luk", "l0 V k"},
            {"luke", "l0 u: k"},
            {"luke's", "l0 u: k s"},
            {"luks", "l0 V k s"},
            {"lull", "l0 V l"},
            {"lulled", "l0 V l d"},
            {"lulls", "l0 V l z"},
            {"lum", "l0 V m"},
            {"lump", "l0 V m p"},
            {"lumped", "l0 V m p t"},
            {"lumps", "l0 V m p s"},
            {"lun", "l0 V n"},
            {"lunch", "l0 V n tS"},
            {"lunched", "l0 V n tS t"},
            {"lund", "l0 V n d"},
            {"lundt", "l0 V n t"},
            {"lung", "l0 V N"},
            {"lunge", "l0 V n dZ"},
            {"lunged", "l0 V n dZ d"},
            {"lungs", "l0 V N z"},
            {"lunn", "l0 V n"},
            {"lunt", "l0 V n t"},
            {"luo", "l0 w @U"},
            {"lupe", "l0 u: p"},
            {"lurch", "l0 @r tS"},
            {"lurched", "l0 @r tS t"},
            {"lure", "l0 U@"},
            {"lured", "l0 U@ d"},
            {"lures", "l0 U@ z"},
            {"lurk", "l0 @r k"},
            {"lurked", "l0 @r k t"},
            {"lurks", "l0 @r k s"},
            {"lush", "l0 V S"},
            {"lusk", "l0 V s k"},
            {"lust", "l0 V s t"},
            {"lute", "l0 u: t"},
            {"lutes", "l0 u: t s"},
            {"luth", "l0 u: T"},
            {"lutz", "l0 V t s"},
            {"luvs", "l0 V v z"},
            {"lux", "l0 V k s"},
            {"luz", "l0 V z"},
            {"lxi", "l0 k s i:"},
            {"ly", "l0 aI"},
            {"lye", "l0 aI"},
            {"lykes", "l0 aI k s"},
            {"lyle", "l0 aI l"},
            {"lyles", "l0 aI l z"},
            {"lyle's", "l0 aI l z"},
            {"lyme", "l0 aI m"},
            {"lymph", "l0 I m f"},
            {"lyn", "l0 I n"},
            {"lynch", "l0 I n tS"},
            {"lynched", "l0 I n tS t"},
            {"lynd", "l0 I n d"},
            {"lynde", "l0 I n d"},
            {"lyne", "l0 aI n"},
            {"lynes", "l0 aI n z"},
            {"lynn", "l0 I n"},
            {"lynne", "l0 I n"},
            {"lynn's", "l0 I n z"},
            {"lynx", "l0 I N k s"},
            {"lyre", "l0 aI r"},
            {"m", "e m"},
            {"m.", "e m"},
            {"ma", "m Q"},
            {"maag", "m Q g"},
            {"maam", "m V m"},
            {"ma'am", "m { m"},
            {"maas", "m Q z"},
            {"mab", "m { b"},
            {"mabe", "m eI b"},
            {"mac", "m { k"},
            {"mace", "m eI s"},
            {"mach", "m Q k"},
            {"mache", "m { tS"},
            {"mack", "m { k"},
            {"macks", "m { k s"},
            {"mack's", "m { k s"},
            {"macs", "m { k s"},
            {"mac's", "m { k s"},
            {"mad", "m { d"},
            {"madd", "m { d"},
            {"made", "m eI d"},
            {"madge", "m { dZ"},
            {"mae", "m eI"},
            {"maes", "m eI z"},
            {"mae's", "m eI z"},
            {"mag", "m { g"},
            {"maggs", "m { g z"},
            {"mags", "m { g z"},
            {"mah", "m Q"},
            {"maher", "m Q@"},
            {"mai", "m aI"},
            {"maid", "m eI d"},
            {"maids", "m eI d z"},
            {"maid's", "m eI d z"},
            {"mail", "m eI l"},
            {"maile", "m eI l"},
            {"mailed", "m eI l d"},
            {"maille", "m eI l"},
            {"mails", "m eI l z"},
            {"mail's", "m eI l z"},
            {"maim", "m eI m"},
            {"maimed", "m eI m d"},
            {"maims", "m eI m z"},
            {"main", "m eI n"},
            {"maine", "m eI n"},
            {"maine's", "m eI n z"},
            {"mains", "m eI n z"},
            {"mainz", "m eI n z"},
            {"mair", "m e@"},
            {"maire", "m e@"},
            {"mairs", "m e@ z"},
            {"maize", "m eI z"},
            {"maj", "m { dZ"},
            {"mak", "m { k"},
            {"make", "m eI k"},
            {"makes", "m eI k s"},
            {"mal", "m { l"},
            {"male", "m eI l"},
            {"males", "m eI l z"},
            {"male's", "m eI l z"},
            {"mall", "m O: l"},
            {"malle", "m { l"},
            {"malls", "m O: l z"},
            {"mall's", "m O: l z"},
            {"malm", "m Q m"},
            {"mal's", "m { l z"},
            {"malt", "m O: l t"},
            {"malts", "m O: l t s"},
            {"maltz", "m { l t s"},
            {"mam", "m Q m"},
            {"mame", "m eI m"},
            {"man", "m { n"},
            {"mane", "m eI n"},
            {"maned", "m eI n d"},
            {"manes", "m eI n z"},
            {"mang", "m { N"},
            {"mange", "m eI n dZ"},
            {"mank", "m { N k"},
            {"mann", "m { n"},
            {"manned", "m { n d"},
            {"mannes", "m { n z"},
            {"mann's", "m { n z"},
            {"mans", "m { n z"},
            {"man's", "m { n z"},
            {"manx", "m { N k s"},
            {"mao", "m aU"},
            {"mao's", "m aU z"},
            {"map", "m { p"},
            {"mapes", "m eI p s"},
            {"mapped", "m { p t"},
            {"maps", "m { p s"},
            {"mar", "m Q@"},
            {"marc", "m Q@ k"},
            {"march", "m Q@ tS"},
            {"marche", "m Q@ S"},
            {"marched", "m Q@ tS t"},
            {"mare", "m e@"},
            {"mares", "m e@ z"},
            {"mare's", "m e@ z"},
            {"marge", "m Q@ dZ"},
            {"marje", "m Q@ dZ"},
            {"mark", "m Q@ k"},
            {"marked", "m Q@ k t"},
            {"marks", "m Q@ k s"},
            {"mark's", "m Q@ k s"},
            {"marque", "m Q@ k"},
            {"marques", "m Q@ k s"},
            {"marr", "m Q@"},
            {"marred", "m Q@ d"},
            {"mars", "m Q@ z"},
            {"marse", "m Q@ s"},
            {"marsh", "m Q@ S"},
            {"mart", "m Q@ t"},
            {"marte", "m Q@ t"},
            {"marth", "m Q@ T"},
            {"marthe", "m Q@ D"},
            {"marts", "m Q@ t s"},
            {"mart's", "m Q@ t s"},
            {"martz", "m Q@ t s"},
            {"marv", "m Q@ v"},
            {"marx", "m Q@ k s"},
            {"mas", "m Q z"},
            {"mase", "m eI z"},
            {"mash", "m { S"},
            {"mashed", "m { S t"},
            {"mask", "m { s k"},
            {"masked", "m { s k t"},
            {"masks", "m { s k s"},
            {"mass", "m { s"},
            {"mass.", "m { s"},
            {"masse", "m { s"},
            {"massed", "m { s t"},
            {"mast", "m { s t"},
            {"masts", "m { s t s"},
            {"mat", "m { t"},
            {"match", "m { tS"},
            {"matched", "m { tS t"},
            {"mate", "m eI t"},
            {"mates", "m eI t s"},
            {"math", "m { T"},
            {"mathe", "m eI D"},
            {"mats", "m { t s"},
            {"matt", "m { t"},
            {"matte", "m { t"},
            {"mattes", "m { t s"},
            {"matts", "m { t s"},
            {"matt's", "m { t z"},
            {"mau", "m @U"},
            {"maud", "m O: d"},
            {"maude", "m O: d"},
            {"mauk", "m O: k"},
            {"maul", "m O: l"},
            {"mauled", "m O: l d"},
            {"maund", "m O: n d"},
            {"mauve", "m O: v"},
            {"maw", "m O:"},
            {"max", "m { k s"},
            {"maxed", "m { k s t"},
            {"may", "m eI"},
            {"maye", "m eI"},
            {"mayes", "m eI z"},
            {"mayme", "m eI m"},
            {"mayne", "m eI n"},
            {"mays", "m eI z"},
            {"may's", "m eI z"},
            {"maze", "m eI z"},
            {"mc", "m I k"},
            {"me", "m i:"},
            {"mea", "m i:"},
            {"mead", "m i: d"},
            {"meade", "m i: d"},
            {"meads", "m i: d z"},
            {"mead's", "m i: d z"},
            {"meagher", "m Q@"},
            {"meal", "m i: l"},
            {"meals", "m i: l z"},
            {"meal's", "m i: l z"},
            {"mean", "m i: n"},
            {"means", "m i: n z"},
            {"meant", "m e n t"},
            {"mear", "m I@"},
            {"meares", "m i: r z"},
            {"mears", "m I@ z"},
            {"mease", "m i: z"},
            {"meat", "m i: t"},
            {"meath", "m i: T"},
            {"meats", "m i: t s"},
            {"meave", "m i: v"},
            {"mech", "m e k"},
            {"meck", "m e k"},
            {"med", "m e d"},
            {"medes", "m i: d z"},
            {"mee", "m i:"},
            {"meece", "m i: s"},
            {"meech", "m i: tS"},
            {"meek", "m i: k"},
            {"meeks", "m i: k s"},
            {"meese", "m i: s"},
            {"meet", "m i: t"},
            {"meets", "m i: t s"},
            {"meg", "m e g"},
            {"meggs", "m e g z"},
            {"mei", "m aI"},
            {"mein", "m i: n"},
            {"meir", "m I@"},
            {"mel", "m e l"},
            {"meld", "m e l d"},
            {"mele", "m i: l"},
            {"mell", "m e l"},
            {"mel's", "m e l z"},
            {"melt", "m e l t"},
            {"melts", "m e l t s"},
            {"men", "m e n"},
            {"mend", "m e n d"},
            {"mende", "m e n d"},
            {"meng", "m e N"},
            {"menge", "m e n dZ"},
            {"mens", "m e n z"},
            {"men's", "m e n z"},
            {"mensch", "m e n S"},
            {"mer", "m e@"},
            {"merc", "m @r k"},
            {"merce", "m @r s"},
            {"merck", "m @r k"},
            {"merck's", "m @r k s"},
            {"merc's", "m @r k s"},
            {"mere", "m I@"},
            {"merge", "m @r dZ"},
            {"merged", "m @r dZ d"},
            {"merk", "m @r k"},
            {"merl", "m @r l"},
            {"merle", "m @r l"},
            {"mers", "m @r z"},
            {"merv", "m @r v"},
            {"mesh", "m e S"},
            {"meshed", "m e S t"},
            {"mess", "m e s"},
            {"messed", "m e s t"},
            {"mest", "m e s t"},
            {"met", "m e t"},
            {"mete", "m i: t"},
            {"meth", "m e T"},
            {"mets", "m e t s"},
            {"metts", "m e t s"},
            {"metz", "m e t s"},
            {"metze", "m e t z"},
            {"meuse", "m j u: z"},
            {"mew", "m j u:"},
            {"mex", "m e k s"},
            {"mi", "m i:"},
            {"mib", "m I b"},
            {"mic", "m I k"},
            {"mice", "m aI s"},
            {"mich", "m I tS"},
            {"mick", "m I k"},
            {"mid", "m I d"},
            {"midge", "m I dZ"},
            {"midst", "m I d s t"},
            {"mien", "m i: n"},
            {"mies", "m aI z"},
            {"miffed", "m I f t"},
            {"mig", "m I g"},
            {"might", "m aI t"},
            {"migs", "m I g z"},
            {"mike", "m aI k"},
            {"mikes", "m aI k s"},
            {"mike's", "m aI k s"},
            {"mil", "m I l"},
            {"milch", "m I l tS"},
            {"mild", "m aI l d"},
            {"mile", "m aI l"},
            {"miles", "m aI l z"},
            {"milk", "m I l k"},
            {"milked", "m I l k t"},
            {"milks", "m I l k s"},
            {"mill", "m I l"},
            {"mille", "m I l"},
            {"milled", "m I l d"},
            {"mills", "m I l z"},
            {"mill's", "m I l z"},
            {"milne", "m I l n"},
            {"milnes", "m I l n z"},
            {"mils", "m I l z"},
            {"milt", "m I l t"},
            {"mim", "m I m"},
            {"mime", "m aI m"},
            {"mims", "m I m z"},
            {"min", "m I n"},
            {"mince", "m I n s"},
            {"minced", "m I n s t"},
            {"minch", "m I n tS"},
            {"mind", "m aI n d"},
            {"minds", "m aI n d z"},
            {"mind's", "m aI n d z"},
            {"mine", "m aI n"},
            {"mined", "m aI n d"},
            {"mines", "m aI n z"},
            {"mine's", "m aI n z"},
            {"ming", "m I N"},
            {"minge", "m I n dZ"},
            {"mings", "m I N z"},
            {"minh", "m I n"},
            {"mink", "m I N k"},
            {"minke", "m I N k"},
            {"minks", "m I N k s"},
            {"minsk", "m I n s k"},
            {"mint", "m I n t"},
            {"mints", "m I n t s"},
            {"mint's", "m I n t s"},
            {"mintz", "m I n t s"},
            {"mips", "m I p s"},
            {"mir", "m I@"},
            {"mire", "m aI r"},
            {"mired", "m aI r d"},
            {"mires", "m aI r z"},
            {"mirth", "m @r T"},
            {"mirv", "m @r v"},
            {"mis", "m I s"},
            {"misch", "m I S"},
            {"miss", "m I s"},
            {"missed", "m I s t"},
            {"mist", "m I s t"},
            {"mists", "m I s t s"},
            {"mitch", "m I tS"},
            {"mite", "m aI t"},
            {"mites", "m aI t s"},
            {"mitt", "m I t"},
            {"mitts", "m I t s"},
            {"mix", "m I k s"},
            {"mixed", "m I k s t"},
            {"mize", "m aI z"},
            {"mk", "m Q@ k"},
            {"mo", "m @U"},
            {"moan", "m @U n"},
            {"moaned", "m @U n d"},
            {"moans", "m @U n z"},
            {"moat", "m @U t"},
            {"moats", "m @U t s"},
            {"mob", "m Q b"},
            {"mobbed", "m Q b d"},
            {"mobs", "m Q b z"},
            {"mob's", "m Q b z"},
            {"moch", "m Q k"},
            {"mock", "m Q k"},
            {"mocked", "m Q k t"},
            {"mocks", "m Q k s"},
            {"mod", "m O: d"},
            {"mode", "m @U d"},
            {"modes", "m @U d z"},
            {"moe", "m @U"},
            {"moen", "m @U n"},
            {"moes", "m @U z"},
            {"mohl", "m @U l"},
            {"mohn", "m Q n"},
            {"mohr", "m O@"},
            {"mohs", "m Q s"},
            {"moi", "m w Q"},
            {"moines", "m OI n z"},
            {"moir", "m OI r"},
            {"moise", "m OI z"},
            {"moist", "m OI s t"},
            {"mok", "m Q k"},
            {"mokes", "m @U k s"},
            {"mol", "m O: l"},
            {"mold", "m @U l d"},
            {"molds", "m @U l d z"},
            {"mole", "m @U l"},
            {"moles", "m @U l z"},
            {"moll", "m Q l"},
            {"molle", "m Q l"},
            {"molt", "m @U l t"},
            {"molto", "m Q l t"},
            {"mom", "m Q m"},
            {"moms", "m Q m z"},
            {"mom's", "m Q m z"},
            {"mon", "m @U n"},
            {"monde", "m Q n d"},
            {"mone", "m @U n"},
            {"mong", "m O: N"},
            {"monge", "m Q n dZ"},
            {"monk", "m V N k"},
            {"monks", "m V N k s"},
            {"monk's", "m V N k s"},
            {"mont", "m Q n t"},
            {"month", "m V n T"},
            {"months", "m V n T s"},
            {"month's", "m V n T s"},
            {"monts", "m Q n t s"},
            {"mood", "m u: d"},
            {"moods", "m u: d z"},
            {"moog", "m u: g"},
            {"mook", "m U k"},
            {"moon", "m u: n"},
            {"moons", "m u: n z"},
            {"moon's", "m u: n z"},
            {"moor", "m U@"},
            {"moore", "m U@"},
            {"moored", "m U@ d"},
            {"moore's", "m U@ z"},
            {"moors", "m U@ z"},
            {"moos", "m u: z"},
            {"moose", "m u: s"},
            {"moot", "m u: t"},
            {"moots", "m u: t s"},
            {"mop", "m Q p"},
            {"mope", "m @U p"},
            {"mopped", "m Q p t"},
            {"mops", "m Q p s"},
            {"mor", "m O@"},
            {"mord", "m O@ d"},
            {"more", "m O@"},
            {"morgue", "m O@ g"},
            {"morgues", "m O@ g z"},
            {"morph", "m O@ f"},
            {"morphs", "m O@ f s"},
            {"mor's", "m O@ z"},
            {"morse", "m O@ s"},
            {"mort", "m O@ t"},
            {"mose", "m @U z"},
            {"mosh", "m O: S"},
            {"mosk", "m O: s k"},
            {"mosque", "m Q s k"},
            {"mosques", "m Q s k s"},
            {"moss", "m O: s"},
            {"most", "m @U s t"},
            {"mote", "m @U t"},
            {"motes", "m @U t s"},
            {"moth", "m O: T"},
            {"moths", "m O: T s"},
            {"mots", "m Q t s"},
            {"mott", "m Q t"},
            {"motte", "m Q t"},
            {"motts", "m Q t s"},
            {"mott's", "m Q t s"},
            {"moul", "m aU l"},
            {"mould", "m @U l d"},
            {"moulds", "m @U l d z"},
            {"mound", "m aU n d"},
            {"mounds", "m aU n d z"},
            {"mount", "m aU n t"},
            {"mounts", "m aU n t s"},
            {"mourn", "m O@ n"},
            {"mourned", "m O@ n d"},
            {"mourns", "m O@ n z"},
            {"mouse", "m aU s"},
            {"mousse", "m u: s"},
            {"mouth", "m aU T"},
            {"mouthed", "m aU D d"},
            {"mouths", "m aU D z"},
            {"move", "m u: v"},
            {"moved", "m u: v d"},
            {"moves", "m u: v z"},
            {"mow", "m @U"},
            {"mowed", "m @U d"},
            {"mows", "m aU z"},
            {"moy", "m OI"},
            {"moyl", "m OI l"},
            {"moyle", "m OI l"},
            {"mraz", "m r { z"},
            {"mroz", "m r Q z"},
            {"ms", "m I z"},
            {"m's", "e m z"},
            {"ms.", "m I z"},
            {"mt", "m aU n t"},
            {"mu", "m u:"},
            {"much", "m V tS"},
            {"muck", "m V k"},
            {"mud", "m V d"},
            {"mudd", "m V d"},
            {"mudge", "m V dZ"},
            {"muds", "m V d z"},
            {"muff", "m V f"},
            {"muffs", "m V f s"},
            {"mug", "m V g"},
            {"mugged", "m V g d"},
            {"mugs", "m V g z"},
            {"muir", "m j U@"},
            {"mulch", "m V l tS"},
            {"mulched", "m V l tS t"},
            {"mule", "m j u: l"},
            {"mules", "m j u: l z"},
            {"mull", "m V l"},
            {"mulled", "m V l d"},
            {"mulls", "m V l z"},
            {"mum", "m V m"},
            {"mumm", "m V m"},
            {"mumps", "m V m p s"},
            {"mums", "m V m z"},
            {"mun", "m V n"},
            {"munch", "m V n tS"},
            {"munched", "m V n tS t"},
            {"mund", "m V n d"},
            {"mundt", "m V n t"},
            {"munn", "m V n"},
            {"muns", "m V n z"},
            {"munt", "m V n t"},
            {"muntz", "m V n t s"},
            {"murk", "m @r k"},
            {"murr", "m @r"},
            {"muse", "m j u: z"},
            {"mused", "m j u: z d"},
            {"mush", "m V S"},
            {"musk", "m V s k"},
            {"must", "m V s t"},
            {"mutch", "m V tS"},
            {"mute", "m j u: t"},
            {"muth", "m u: T"},
            {"mutt", "m V t"},
            {"mutts", "m V t s"},
            {"mutz", "m V t s"},
            {"my", "m aI"},
            {"myles", "m aI l z"},
            {"myrrh", "m @r"},
            {"myth", "m I T"},
            {"myths", "m I T s"},
            {"n", "e n"},
            {"n.", "e n"},
            {"na", "n Q"},
            {"naan", "n { n n"},
            {"naas", "n Q z"},
            {"nab", "n { b"},
            {"nabb", "n { b"},
            {"nabbed", "n { b d"},
            {"nace", "n eI s"},
            {"nad", "n { d"},
            {"naeve", "n i: v"},
            {"naff", "n { f"},
            {"nag", "n { g"},
            {"nagged", "n { g d"},
            {"nagpur", "n @r g p l l"},
            {"nags", "n { g z"},
            {"nah", "n Q"},
            {"naik", "n eI k"},
            {"nail", "n eI l"},
            {"nailed", "n eI l d"},
            {"nails", "n eI l z"},
            {"nair", "n e@"},
            {"nairn", "n e@ n"},
            {"nale", "n eI l"},
            {"nam", "n { m"},
            {"name", "n eI m"},
            {"named", "n eI m d"},
            {"names", "n eI m z"},
            {"name's", "n eI m z"},
            {"nan", "n { n"},
            {"nance", "n { n s"},
            {"nang", "n { N"},
            {"nan's", "n { n z"},
            {"nantes", "n { n t s"},
            {"nantz", "n { n t s"},
            {"nap", "n { p"},
            {"napped", "n { p t"},
            {"naps", "n { p s"},
            {"narc", "n Q@ k"},
            {"nard", "n Q@ d"},
            {"nares", "n { r z"},
            {"nash", "n { S"},
            {"nast", "n { s t"},
            {"nast's", "n { s t s"},
            {"nat", "n { t"},
            {"nate", "n eI t"},
            {"nath", "n { T"},
            {"nathe", "n eI D"},
            {"nats", "n { t s"},
            {"nau", "n @U"},
            {"naught", "n O: t"},
            {"nav", "n { v"},
            {"nave", "n eI v"},
            {"naw", "n Q"},
            {"nay", "n eI"},
            {"nays", "n eI z"},
            {"ne", "n i:"},
            {"neal", "n i: l"},
            {"neale", "n i: l"},
            {"neall", "n i: l"},
            {"neal's", "n i: l z"},
            {"near", "n I@"},
            {"neared", "n I@ d"},
            {"nears", "n I@ z"},
            {"neat", "n i: t"},
            {"nec", "n e k"},
            {"neck", "n e k"},
            {"necked", "n e k t"},
            {"necks", "n e k s"},
            {"ned", "n e d"},
            {"ned's", "n e d z"},
            {"nee", "n i:"},
            {"need", "n i: d"},
            {"needs", "n i: d z"},
            {"neel", "n i: l"},
            {"neeld", "n i: l d"},
            {"neem", "n i: m m"},
            {"neer", "n I@"},
            {"ne'er", "n e@"},
            {"neese", "n i: z"},
            {"neet", "n i: t"},
            {"neff", "n e f"},
            {"neg", "n e g"},
            {"negs", "n e g z"},
            {"neil", "n i: l"},
            {"neile", "n i: l"},
            {"neill", "n i: l"},
            {"neils", "n i: l z"},
            {"neil's", "n i: l z"},
            {"nein", "n i: n"},
            {"nel", "n e l"},
            {"nell", "n e l"},
            {"nelle", "n e l"},
            {"nels", "n e l z"},
            {"nene", "n i: n"},
            {"nerd", "n @r d"},
            {"nerds", "n @r d z"},
            {"nerve", "n @r v"},
            {"nerves", "n @r v z"},
            {"nes", "n e s"},
            {"ness", "n e s"},
            {"nest", "n e s t"},
            {"nests", "n e s t s"},
            {"net", "n e t"},
            {"neth", "n e T"},
            {"nets", "n e t s"},
            {"nett", "n e t"},
            {"neu", "n OI"},
            {"neve", "n i: v"},
            {"neves", "n i: v z"},
            {"new", "n u:"},
            {"news", "n u: z"},
            {"newt", "n u: t"},
            {"newts", "n u: t s"},
            {"newt's", "n u: t s"},
            {"next", "n e k s t"},
            {"ney", "n eI"},
            {"nez", "n e z"},
            {"ng", "e N"},
            {"ngai", "gh aI"},
            {"ngo", "gh @U"},
            {"ngo's", "gh @U z"},
            {"niall", "n aI l"},
            {"nib", "n I b"},
            {"nibs", "n I b z"},
            {"nice", "n aI s"},
            {"niche", "n I tS"},
            {"nick", "n I k"},
            {"nicked", "n I k t"},
            {"nicks", "n I k s"},
            {"nick's", "n I k s"},
            {"nie", "n i:"},
            {"niece", "n i: s"},
            {"nied", "n i: d"},
            {"niel", "n i: l"},
            {"niels", "n i: l z"},
            {"nies", "n aI z"},
            {"nigh", "n aI"},
            {"night", "n aI t"},
            {"nights", "n aI t s"},
            {"night's", "n aI t s"},
            {"nights'", "n aI t s"},
            {"nil", "n I l"},
            {"nile", "n aI l"},
            {"niles", "n aI l z"},
            {"nill", "n I l"},
            {"nils", "n I l z"},
            {"nims", "n I m z"},
            {"nine", "n aI n"},
            {"nines", "n aI n z"},
            {"nine's", "n aI n z"},
            {"ninth", "n aI n T"},
            {"ninths", "n aI n T s"},
            {"nip", "n I p"},
            {"nipped", "n I p t"},
            {"nir", "n I@"},
            {"nist", "n I s t"},
            {"nit", "n I t"},
            {"nite", "n aI t"},
            {"nitz", "n I t s"},
            {"nix", "n I k s"},
            {"nixed", "n I k s t"},
            {"no", "n @U"},
            {"noakes", "n @U k s"},
            {"noam", "n @U m"},
            {"nock", "n Q k"},
            {"nod", "n Q d"},
            {"node", "n @U d"},
            {"nodes", "n @U d z"},
            {"nods", "n Q d z"},
            {"noe", "n @U"},
            {"noell", "n @U l"},
            {"noes", "n @U z"},
            {"noh", "n @U"},
            {"noir", "n OI r"},
            {"noirs", "n OI r z"},
            {"noise", "n OI z"},
            {"nold", "n @U l d"},
            {"nolde", "n @U l d"},
            {"nole", "n @U l"},
            {"noles", "n @U l z"},
            {"noll", "n @U l"},
            {"nolls", "n @U l z"},
            {"nolt", "n @U l t"},
            {"nolte", "n @U l t"},
            {"nom", "n Q m"},
            {"nome", "n @U m"},
            {"nome's", "n @U m z"},
            {"non", "n Q n"},
            {"none", "n V n"},
            {"nook", "n U k"},
            {"nooks", "n U k s"},
            {"noon", "n u: n"},
            {"noone", "n u: n"},
            {"noons", "n u: n z"},
            {"noose", "n u: s"},
            {"nope", "n @U p"},
            {"nor", "n O@"},
            {"nord", "n O@ d"},
            {"norge", "n O@ dZ"},
            {"nork", "n O@ k"},
            {"norm", "n O@ m"},
            {"norms", "n O@ m z"},
            {"norse", "n O@ s"},
            {"norsk", "n O@ s k"},
            {"north", "n O@ T"},
            {"north's", "n O@ T s"},
            {"no's", "n @U z"},
            {"nose", "n @U z"},
            {"nosed", "n @U z d"},
            {"not", "n Q t"},
            {"notch", "n Q tS"},
            {"notched", "n Q tS t"},
            {"note", "n @U t"},
            {"notes", "n @U t s"},
            {"note's", "n @U t s"},
            {"nots", "n Q t s"},
            {"nott", "n Q t"},
            {"notte", "n Q t"},
            {"noun", "n aU n"},
            {"now", "n aU"},
            {"noyce", "n OI s"},
            {"noyes", "n OI z"},
            {"n's", "e n z"},
            {"nth", "e n T"},
            {"nu", "n u:"},
            {"nub", "n V b"},
            {"nudd", "n V d"},
            {"nude", "n u: d"},
            {"nudes", "n u: d z"},
            {"nudge", "n V dZ"},
            {"nudged", "n V dZ d"},
            {"nuke", "n u: k"},
            {"nukes", "n u: k s"},
            {"null", "n V l"},
            {"numb", "n V m"},
            {"numbed", "n V m d"},
            {"nun", "n V n"},
            {"nunes", "n u: n z"},
            {"nunn", "n V n"},
            {"nunn's", "n V n z"},
            {"nuns", "n V n z"},
            {"nun's", "n V n z"},
            {"nurse", "n @r s"},
            {"nursed", "n @r s t"},
            {"nut", "n V t"},
            {"nuts", "n V t s"},
            {"nut's", "n V t s"},
            {"nye", "n aI"},
            {"nyet", "n j e t"},
            {"nymph", "n I m f"},
            {"nymphs", "n I m f s"},
            {"nyx", "n I k s"},
            {"o", "@U"},
            {"o'", "@U"},
            {"o.", "@U"},
            {"oak", "@U k"},
            {"oakes", "@U k s"},
            {"oaks", "@U k s"},
            {"oak's", "@U k s"},
            {"oar", "O@"},
            {"oared", "O@ d"},
            {"oars", "O@ z"},
            {"oat", "@U t"},
            {"oates", "@U t s"},
            {"oath", "@U T"},
            {"oaths", "@U D z"},
            {"oats", "@U t s"},
            {"obst", "Q b s t"},
            {"och", "Q k"},
            {"ochs", "Q k s"},
            {"oct.", "O: k t"},
            {"odd", "Q d"},
            {"odds", "Q d z"},
            {"ode", "@U d"},
            {"odes", "@U d z"},
            {"of", "Q v"},
            {"off", "Q f"},
            {"offs", "O: f s"},
            {"oft", "O: f t"},
            {"og", "Q g"},
            {"ogg", "Q g"},
            {"oh", "@U"},
            {"ohl", "@U l"},
            {"ohm", "@U m"},
            {"ohms", "@U m z"},
            {"ohm's", "@U m z"},
            {"ohs", "@U z"},
            {"oh's", "@U z"},
            {"oi", "OI"},
            {"oie", "OI"},
            {"oil", "OI l"},
            {"oiled", "OI l d"},
            {"oils", "OI l z"},
            {"oil's", "OI l z"},
            {"old", "@U l d"},
            {"olde", "@U l d"},
            {"olds", "@U l d z"},
            {"old's", "@U l d z"},
            {"ole", "@U l"},
            {"oles", "@U l z"},
            {"on", "Q n"},
            {"once", "w V n s"},
            {"one", "w V n"},
            {"ones", "w V n z"},
            {"one's", "w V n z"},
            {"ong", "O: N"},
            {"ons", "Q n z"},
            {"oohs", "u: z"},
            {"oomph", "u: m f"},
            {"oops", "u: p s"},
            {"ooze", "u: z"},
            {"oozed", "u: z d"},
            {"op", "Q p"},
            {"opp", "Q p"},
            {"ops", "Q p s"},
            {"op's", "Q p s"},
            {"opt", "Q p t"},
            {"opts", "Q p t s"},
            {"or", "O@"},
            {"orb", "O@ b"},
            {"ord", "O@ d"},
            {"ore", "O@"},
            {"ores", "O@ z"},
            {"orf", "O@ f"},
            {"org", "O@ g"},
            {"orme", "O@ m"},
            {"orne", "O@ n"},
            {"orr", "O@"},
            {"ort", "O@ t"},
            {"orth", "O@ T"},
            {"os", "Q s"},
            {"o's", "@U z"},
            {"ose", "@U z"},
            {"oss", "O: s"},
            {"ost", "@U s t"},
            {"ot", "O: t"},
            {"ott", "Q t"},
            {"otte", "Q t"},
            {"ouch", "aU tS"},
            {"ought", "O: t"},
            {"oui", "w i:"},
            {"ounce", "aU n s"},
            {"oust", "aU s t"},
            {"out", "aU t"},
            {"outs", "aU t s"},
            {"ow", "@U"},
            {"owe", "@U"},
            {"owed", "@U d"},
            {"owes", "@U z"},
            {"owl", "aU l"},
            {"owls", "aU l z"},
            {"own", "@U n"},
            {"owned", "@U n d"},
            {"owns", "@U n z"},
            {"ox", "Q k s"},
            {"oy", "OI"},
            {"oye", "OI"},
            {"oz", "Q z"},
            {"p", "ph i:"},
            {"p.", "ph i:"},
            {"pa", "ph Q"},
            {"paar", "ph Q@"},
            {"pabst", "ph { b s t"},
            {"pac", "ph { k"},
            {"pace", "ph eI s"},
            {"paced", "ph eI s t"},
            {"pack", "ph { k"},
            {"packed", "ph { k t"},
            {"packs", "ph { k s"},
            {"pacs", "ph { k s"},
            {"pact", "ph { k t"},
            {"pacts", "ph { k t s"},
            {"pact's", "ph { k t s"},
            {"pad", "ph { d"},
            {"pads", "ph { d z"},
            {"paff", "ph { f"},
            {"page", "ph eI dZ"},
            {"paged", "ph eI dZ d"},
            {"pah", "ph {"},
            {"pahl", "ph Q l"},
            {"paid", "ph eI d"},
            {"paige", "ph eI dZ"},
            {"paik", "ph eI k"},
            {"pail", "ph eI l"},
            {"pails", "ph eI l z"},
            {"pain", "ph eI n"},
            {"paine", "ph eI n"},
            {"pained", "ph eI n d"},
            {"pains", "ph eI n z"},
            {"paint", "ph eI n t"},
            {"paints", "ph eI n t s"},
            {"pair", "ph e@"},
            {"paired", "ph e@ d"},
            {"pairs", "ph e@ z"},
            {"pais", "ph eI z"},
            {"pak", "ph { k"},
            {"pal", "ph { l"},
            {"pale", "ph eI l"},
            {"paled", "ph eI l d"},
            {"pales", "ph eI l z"},
            {"pall", "ph Q l"},
            {"palm", "ph Q m"},
            {"palms", "ph Q m z"},
            {"pals", "ph { l z"},
            {"pal's", "ph { l z"},
            {"paltz", "ph O: l t s"},
            {"pam", "ph { m"},
            {"pam's", "ph { m z"},
            {"pan", "ph { n"},
            {"pane", "ph eI n"},
            {"panes", "ph eI n z"},
            {"pang", "ph { N"},
            {"pangs", "ph { N z"},
            {"panned", "ph { n d"},
            {"pans", "ph { n z"},
            {"pan's", "ph { n z"},
            {"pant", "ph { n t"},
            {"pants", "ph { n t s"},
            {"pao", "ph aU"},
            {"pap", "ph { p"},
            {"pape", "ph eI p"},
            {"papke", "ph eI p k"},
            {"papp", "ph { p"},
            {"paque", "ph { k"},
            {"par", "ph Q@"},
            {"parc", "ph Q@ k"},
            {"parch", "ph Q@ tS"},
            {"parched", "ph Q@ tS t"},
            {"parde", "ph Q@ d"},
            {"pare", "ph e@"},
            {"pared", "ph e@ d"},
            {"pares", "ph e@ z"},
            {"park", "ph Q@ k"},
            {"parke", "ph Q@ k"},
            {"parked", "ph Q@ k t"},
            {"parks", "ph Q@ k s"},
            {"park's", "ph Q@ k s"},
            {"parr", "ph Q@"},
            {"pars", "ph Q@ z"},
            {"par's", "ph Q@ z"},
            {"parse", "ph Q@ s"},
            {"part", "ph Q@ t"},
            {"parte", "ph Q@ t"},
            {"parts", "ph Q@ t s"},
            {"part's", "ph Q@ t s"},
            {"pas", "ph Q z"},
            {"pasch", "ph { s k"},
            {"pash", "ph { S"},
            {"pask", "ph { s k"},
            {"paske", "ph eI s k"},
            {"pasque", "ph { s k"},
            {"pass", "ph { s"},
            {"passed", "ph { s t"},
            {"past", "ph { s t"},
            {"paste", "ph eI s t"},
            {"pastes", "ph eI s t s"},
            {"pasts", "ph { s t s"},
            {"pat", "ph { t"},
            {"patch", "ph { tS"},
            {"patched", "ph { tS t"},
            {"pate", "ph eI t"},
            {"pates", "ph eI t s"},
            {"path", "ph { T"},
            {"pathe", "ph { T"},
            {"paths", "ph { D z"},
            {"pats", "ph { t s"},
            {"pat's", "ph { t s"},
            {"patt", "ph { t"},
            {"paugh", "ph O:"},
            {"paul", "ph O: l"},
            {"paule", "ph O: l"},
            {"paull", "ph O: l"},
            {"pauls", "ph O: l z"},
            {"paul's", "ph O: l z"},
            {"paup", "ph O: p"},
            {"pause", "ph O: z"},
            {"paused", "ph O: z d"},
            {"pave", "ph eI v"},
            {"paved", "ph eI v d"},
            {"paves", "ph eI v z"},
            {"paw", "ph O:"},
            {"pawn", "ph O: n"},
            {"pawned", "ph O: n d"},
            {"pawns", "ph O: n z"},
            {"paws", "ph O: z"},
            {"pax", "ph { k s"},
            {"pay", "ph eI"},
            {"paye", "ph eI"},
            {"payne", "ph eI n"},
            {"pays", "ph eI z"},
            {"paz", "ph Q z"},
            {"pea", "ph i:"},
            {"peace", "ph i: s"},
            {"peach", "ph i: tS"},
            {"peak", "ph i: k"},
            {"peake", "ph i: k"},
            {"peaked", "ph i: k t"},
            {"peaks", "ph i: k s"},
            {"peal", "ph i: l"},
            {"peale", "ph i: l"},
            {"pear", "ph e@"},
            {"pearce", "ph I@ s"},
            {"pearl", "ph @r l"},
            {"pearle", "ph @r l"},
            {"pearls", "ph @r l z"},
            {"pears", "ph e@ z"},
            {"pearse", "ph @r s"},
            {"peart", "ph @r t"},
            {"peas", "ph i: z"},
            {"pease", "ph i: z"},
            {"peat", "ph i: t"},
            {"peat's", "ph i: t s"},
            {"pech", "ph e k"},
            {"pecht", "ph e k t"},
            {"peck", "ph e k"},
            {"peck's", "ph e k s"},
            {"pee", "ph i:"},
            {"peed", "ph i: d"},
            {"peek", "ph i: k"},
            {"peeked", "ph i: k t"},
            {"peeks", "ph i: k s"},
            {"peel", "ph i: l"},
            {"peele", "ph i: l"},
            {"peeled", "ph i: l d"},
            {"peels", "ph i: l z"},
            {"peens", "ph i: n z"},
            {"peep", "ph i: p"},
            {"peeps", "ph i: p s"},
            {"peer", "ph I@"},
            {"peered", "ph I@ d"},
            {"peers", "ph I@ z"},
            {"peetz", "ph i: t s"},
            {"peeve", "ph i: v"},
            {"peeved", "ph i: v d"},
            {"peeves", "ph i: v z"},
            {"peg", "ph e g"},
            {"pegg", "ph e g"},
            {"pegged", "ph e g d"},
            {"peggs", "ph e g z"},
            {"pegs", "ph e g z"},
            {"pei", "ph eI"},
            {"peine", "ph i: n"},
            {"peirce", "ph I@ s"},
            {"pell", "ph e l"},
            {"pell's", "ph e l z"},
            {"pelt", "ph e l t"},
            {"pelts", "ph e l t s"},
            {"peltz", "ph e l t s"},
            {"pen", "ph e n"},
            {"pence", "ph e n s"},
            {"peng", "ph e N"},
            {"penh", "ph e n"},
            {"penh's", "ph e n z"},
            {"penn", "ph e n"},
            {"penned", "ph e n d"},
            {"penns", "ph e n z"},
            {"penn's", "ph e n z"},
            {"pens", "ph e n z"},
            {"pen's", "ph e n z"},
            {"pense", "ph e n s"},
            {"pent", "ph e n t"},
            {"pep", "ph e p"},
            {"per", "ph @r"},
            {"perc", "ph @r k"},
            {"perce", "ph @r s"},
            {"perch", "ph @r tS"},
            {"perched", "ph @r tS t"},
            {"pere", "ph @r"},
            {"perk", "ph @r k"},
            {"perked", "ph @r k t"},
            {"perks", "ph @r k s"},
            {"perl", "ph @r l"},
            {"perle", "ph @r l"},
            {"perls", "ph @r l z"},
            {"pers", "ph @r s"},
            {"perse", "ph @r s"},
            {"pert", "ph @r t"},
            {"perth", "ph @r T"},
            {"pest", "ph e s t"},
            {"pests", "ph e s t s"},
            {"pet", "ph e t"},
            {"pete", "ph i: t"},
            {"pete's", "ph i: t s"},
            {"pets", "ph e t s"},
            {"pet's", "ph e t s"},
            {"pew", "ph j u:"},
            {"pews", "ph j u: z"},
            {"pfaff", "f { f"},
            {"pfund", "f V n d"},
            {"phar", "f Q@"},
            {"phares", "f e@ z"},
            {"pharr", "f Q@"},
            {"phase", "f eI z"},
            {"phased", "f eI z d"},
            {"phelps", "f e l p s"},
            {"phew", "f j u:"},
            {"phi", "f aI"},
            {"phil", "f I l"},
            {"phil's", "f I l z"},
            {"phipps", "f I p s"},
            {"phnom", "f n Q m"},
            {"phone", "f @U n"},
            {"phoned", "f @U n d"},
            {"phones", "f @U n z"},
            {"phone's", "f @U n z"},
            {"phrase", "f r eI z"},
            {"phrased", "f r eI z d"},
            {"phu", "f u:"},
            {"phut", "f V t"},
            {"phyle", "f aI l"},
            {"pi", "ph aI"},
            {"pic", "ph I k"},
            {"pick", "ph I k"},
            {"picked", "ph I k t"},
            {"picks", "ph I k s"},
            {"picts", "ph I k t s"},
            {"pie", "ph aI"},
            {"piece", "ph i: s"},
            {"pieced", "ph i: s t"},
            {"pied", "ph aI d"},
            {"pier", "ph I@"},
            {"pierce", "ph I@ s"},
            {"pierced", "ph I@ s t"},
            {"piers", "ph I@ z"},
            {"pies", "ph aI z"},
            {"pig", "ph I g"},
            {"pigg", "ph I g"},
            {"pigs", "ph I g z"},
            {"pig's", "ph I g z"},
            {"pik", "ph I k"},
            {"pike", "ph aI k"},
            {"pikes", "ph aI k s"},
            {"pil", "ph I l"},
            {"pilch", "ph I l tS"},
            {"pile", "ph aI l"},
            {"piled", "ph aI l d"},
            {"piles", "ph aI l z"},
            {"pill", "ph I l"},
            {"pills", "ph I l z"},
            {"pill's", "ph I l z"},
            {"pimp", "ph I m p"},
            {"pimps", "ph I m p s"},
            {"pin", "ph I n"},
            {"pinch", "ph I n tS"},
            {"pinched", "ph I n tS t"},
            {"pine", "ph aI n"},
            {"pines", "ph aI n z"},
            {"ping", "ph I N"},
            {"pink", "ph I N k"},
            {"pinks", "ph I N k s"},
            {"pinned", "ph I n d"},
            {"pins", "ph I n z"},
            {"pint", "ph aI n t"},
            {"pints", "ph aI n t s"},
            {"pip", "ph I p"},
            {"pipe", "ph aI p"},
            {"piped", "ph aI p t"},
            {"pipes", "ph aI p s"},
            {"pips", "ph I p s"},
            {"pique", "ph i: k"},
            {"piqued", "ph i: k t"},
            {"piss", "ph I s"},
            {"pissed", "ph I s t"},
            {"pit", "ph I t"},
            {"pitch", "ph I tS"},
            {"pitched", "ph I tS t"},
            {"pith", "ph I T"},
            {"pits", "ph I t s"},
            {"pitt", "ph I t"},
            {"pitts", "ph I t s"},
            {"pitt's", "ph I t s"},
            {"pix", "ph I k s"},
            {"pla", "ph l Q"},
            {"place", "ph l eI s"},
            {"placed", "ph l eI s t"},
            {"plack", "ph l { k"},
            {"plack's", "ph l { k s"},
            {"plague", "ph l eI g"},
            {"plagued", "ph l eI g d"},
            {"plagues", "ph l eI g z"},
            {"plaid", "ph l { d"},
            {"plain", "ph l eI n"},
            {"plaines", "ph l eI n z"},
            {"plains", "ph l eI n z"},
            {"plan", "ph l { n"},
            {"planck", "ph l { N k"},
            {"plane", "ph l eI n"},
            {"planed", "ph l eI n d"},
            {"planes", "ph l eI n z"},
            {"plane's", "ph l eI n z"},
            {"plank", "ph l { N k"},
            {"planks", "ph l { N k s"},
            {"planned", "ph l { n d"},
            {"plans", "ph l { n z"},
            {"plan's", "ph l { n z"},
            {"plant", "ph l { n t"},
            {"plante", "ph l { n t"},
            {"plants", "ph l { n t s"},
            {"plant's", "ph l { n t s"},
            {"plaque", "ph l { k"},
            {"plaques", "ph l { k s"},
            {"plass", "ph l { s"},
            {"plate", "ph l eI t"},
            {"plates", "ph l eI t s"},
            {"plath", "ph l { T"},
            {"platt", "ph l { t"},
            {"platte", "ph l { t"},
            {"play", "ph l eI"},
            {"played", "ph l eI d"},
            {"plays", "ph l eI z"},
            {"plea", "ph l i:"},
            {"plead", "ph l i: d"},
            {"pleads", "ph l i: d z"},
            {"pleas", "ph l i: z"},
            {"please", "ph l i: z"},
            {"pleased", "ph l i: z d"},
            {"pleat", "ph l i: t"},
            {"pleats", "ph l i: t s"},
            {"pled", "ph l e d"},
            {"pledge", "ph l e dZ"},
            {"pledged", "ph l e dZ d"},
            {"plein", "ph l i: n"},
            {"plew", "ph l u:"},
            {"plex", "ph l e k s"},
            {"plied", "ph l aI d"},
            {"plies", "ph l aI z"},
            {"plight", "ph l aI t"},
            {"plink", "ph l I N k"},
            {"plinth", "ph l I n T"},
            {"ploch", "ph l Q k"},
            {"plock", "ph l Q k"},
            {"plod", "ph l Q d"},
            {"plop", "ph l Q p"},
            {"plopped", "ph l Q p t"},
            {"ploss", "ph l O: s"},
            {"plot", "ph l Q t"},
            {"plots", "ph l Q t s"},
            {"plott", "ph l Q t"},
            {"plotz", "ph l Q t s"},
            {"plough", "ph l aU"},
            {"ploughed", "ph l aU d"},
            {"plow", "ph l aU"},
            {"plowed", "ph l aU d"},
            {"plows", "ph l aU z"},
            {"ploy", "ph l OI"},
            {"ploys", "ph l OI z"},
            {"pluck", "ph l V k"},
            {"plucked", "ph l V k t"},
            {"plucks", "ph l V k s"},
            {"pluff", "ph l V f"},
            {"plug", "ph l V g"},
            {"plugged", "ph l V g d"},
            {"plugs", "ph l V g z"},
            {"plum", "ph l V m"},
            {"plumb", "ph l V m"},
            {"plumbed", "ph l V m d"},
            {"plume", "ph l u: m"},
            {"plumes", "ph l u: m z"},
            {"plump", "ph l V m p"},
            {"plumped", "ph l V m p t"},
            {"plums", "ph l V m z"},
            {"plum's", "ph l V m z"},
            {"plunge", "ph l V n dZ"},
            {"plunged", "ph l V n dZ d"},
            {"plunk", "ph l V N k"},
            {"plunked", "ph l V N k t"},
            {"plus", "ph l V s"},
            {"plush", "ph l V S"},
            {"ply", "ph l aI"},
            {"po", "ph @U"},
            {"poach", "ph @U tS"},
            {"poached", "ph @U tS t"},
            {"poche", "ph Q tS"},
            {"pock", "ph Q k"},
            {"pocked", "ph Q k t"},
            {"pocks", "ph Q k s"},
            {"pod", "ph Q d"},
            {"podge", "ph Q dZ"},
            {"pods", "ph Q d z"},
            {"pod's", "ph Q d z"},
            {"poe", "ph @U"},
            {"poe's", "ph @U z"},
            {"pogue", "ph @U g"},
            {"poh", "ph @U"},
            {"point", "ph OI n t"},
            {"pointe", "ph OI n t"},
            {"points", "ph OI n t s"},
            {"point's", "ph OI n t s"},
            {"poise", "ph OI z"},
            {"poised", "ph OI z d"},
            {"poke", "ph @U k"},
            {"poked", "ph @U k t"},
            {"pokes", "ph @U k s"},
            {"pol", "ph O: l"},
            {"pole", "ph @U l"},
            {"poles", "ph @U l z"},
            {"polk", "ph @U k"},
            {"polk's", "ph @U k s"},
            {"poll", "ph @U l"},
            {"polled", "ph @U l d"},
            {"polls", "ph @U l z"},
            {"poll's", "ph @U l z"},
            {"pols", "ph @U l z"},
            {"pom", "ph O: m"},
            {"pomp", "ph Q m p"},
            {"pon", "ph Q n"},
            {"pond", "ph Q n d"},
            {"ponds", "ph Q n d z"},
            {"pond's", "ph Q n d z"},
            {"pong", "ph O: N"},
            {"pons", "ph Q n z"},
            {"pont", "ph Q n t"},
            {"ponte", "ph Q n t"},
            {"ponts", "ph Q n t s"},
            {"pont's", "ph Q n t s"},
            {"poo", "ph u:"},
            {"pooch", "ph u: tS"},
            {"poof", "ph u: f"},
            {"pooh", "ph u:"},
            {"poohed", "ph u: d"},
            {"pool", "ph u: l"},
            {"poole", "ph u: l"},
            {"pooled", "ph u: l d"},
            {"pools", "ph u: l z"},
            {"pool's", "ph u: l z"},
            {"poon", "ph u: n"},
            {"poop", "ph u: p"},
            {"pooped", "ph u: p t"},
            {"poops", "ph u: p s"},
            {"poor", "ph U@"},
            {"poore", "ph U@"},
            {"pop", "ph Q p"},
            {"pope", "ph @U p"},
            {"popes", "ph @U p z"},
            {"pope's", "ph @U p s"},
            {"popp", "ph Q p"},
            {"popped", "ph Q p t"},
            {"pops", "ph Q p s"},
            {"pop's", "ph Q p s"},
            {"por", "ph O@"},
            {"porch", "ph O@ tS"},
            {"pore", "ph O@"},
            {"pored", "ph O@ d"},
            {"pores", "ph O@ z"},
            {"pork", "ph O@ k"},
            {"porn", "ph O@ n"},
            {"porr", "ph O@"},
            {"port", "ph O@ t"},
            {"porte", "ph O@ t"},
            {"ports", "ph O@ t s"},
            {"port's", "ph O@ t s"},
            {"pos", "ph Q s"},
            {"pose", "ph @U z"},
            {"posed", "ph @U z d"},
            {"posh", "ph Q S"},
            {"poss", "ph O: s"},
            {"post", "ph @U s t"},
            {"posts", "ph @U s t s"},
            {"post's", "ph @U s t s"},
            {"pot", "ph Q t"},
            {"pote", "ph @U t"},
            {"poth", "ph Q T"},
            {"pots", "ph Q t s"},
            {"pott", "ph Q t"},
            {"potts", "ph Q t s"},
            {"pou", "ph u:"},
            {"pouch", "ph aU tS"},
            {"poul", "ph u: l"},
            {"pounce", "ph aU n s"},
            {"pounced", "ph aU n s t"},
            {"pound", "ph aU n d"},
            {"pounds", "ph aU n d z"},
            {"pound's", "ph aU n d z"},
            {"pour", "ph O@"},
            {"poured", "ph O@ d"},
            {"pours", "ph O@ z"},
            {"pout", "ph aU t"},
            {"pouts", "ph aU t s"},
            {"pow", "ph aU"},
            {"powe", "ph aU"},
            {"pows", "ph @U z"},
            {"pox", "ph Q k s"},
            {"prague", "ph r Q g"},
            {"prague's", "ph r Q g z"},
            {"praise", "ph r eI z"},
            {"praised", "ph r eI z d"},
            {"prance", "ph r { n s"},
            {"prank", "ph r { N k"},
            {"pranks", "ph r { N k s"},
            {"prats", "ph r { t s"},
            {"pratt", "ph r { t"},
            {"pratte", "ph r { t"},
            {"pratt's", "ph r { t s"},
            {"prawn", "ph r O: n"},
            {"prawns", "ph r O: n z"},
            {"pray", "ph r eI"},
            {"prayed", "ph r eI d"},
            {"prayer", "ph r e@"},
            {"prayers", "ph r e@ z"},
            {"prays", "ph r eI z"},
            {"pre", "ph r i:"},
            {"preach", "ph r i: tS"},
            {"preached", "ph r i: tS t"},
            {"pree", "ph r i:"},
            {"preen", "ph r i: n"},
            {"preened", "ph r i: n d"},
            {"prem", "ph r e m"},
            {"prep", "ph r e p"},
            {"prepped", "ph r e p t"},
            {"pres", "ph r e z"},
            {"press", "ph r e s"},
            {"pressed", "ph r e s t"},
            {"prest", "ph r e s t"},
            {"pret", "ph r e t"},
            {"preuss", "ph r u: s"},
            {"prew", "ph r u:"},
            {"prey", "ph r eI"},
            {"preyed", "ph r eI d"},
            {"preys", "ph r eI z"},
            {"pri", "ph r aI"},
            {"price", "ph r aI s"},
            {"priced", "ph r aI s t"},
            {"prick", "ph r I k"},
            {"pricked", "ph r I k t"},
            {"pricks", "ph r I k s"},
            {"pride", "ph r aI d"},
            {"prides", "ph r aI d z"},
            {"priebe", "ph r i: b"},
            {"pried", "ph r aI d"},
            {"pries", "ph r aI z"},
            {"priest", "ph r i: s t"},
            {"priests", "ph r i: s t s"},
            {"priest's", "ph r i: s t s"},
            {"prig", "ph r I g"},
            {"prill", "ph r I l"},
            {"prim", "ph r I m"},
            {"prime", "ph r aI m"},
            {"primed", "ph r aI m d"},
            {"primes", "ph r aI m z"},
            {"prime's", "ph r aI m z"},
            {"primp", "ph r I m p"},
            {"prince", "ph r I n s"},
            {"prine", "ph r aI n"},
            {"print", "ph r I n t"},
            {"prints", "ph r I n t s"},
            {"prinz", "ph r I n z"},
            {"pris", "ph r I s"},
            {"prix", "ph r i:"},
            {"prize", "ph r aI z"},
            {"prized", "ph r aI z d"},
            {"pro", "ph r @U"},
            {"prob", "ph r Q b"},
            {"probe", "ph r @U b"},
            {"probed", "ph r @U b d"},
            {"probes", "ph r @U b z"},
            {"prod", "ph r Q d"},
            {"prods", "ph r Q d z"},
            {"prof.", "ph r O: f"},
            {"profs", "ph r Q f s"},
            {"prom", "ph r Q m"},
            {"prompt", "ph r Q m p t"},
            {"prompts", "ph r Q m p t s"},
            {"proms", "ph r Q m z"},
            {"prone", "ph r @U n"},
            {"prong", "ph r O: N"},
            {"pronged", "ph r O: N d"},
            {"prongs", "ph r O: N z"},
            {"proof", "ph r u: f"},
            {"proofed", "ph r u: f t"},
            {"proofs", "ph r u: f s"},
            {"prop", "ph r Q p"},
            {"propped", "ph r Q p t"},
            {"props", "ph r Q p s"},
            {"pros", "ph r @U z"},
            {"pro's", "ph r @U z"},
            {"prose", "ph r @U z"},
            {"pross", "ph r O: s"},
            {"prost", "ph r Q s t"},
            {"proud", "ph r aU d"},
            {"proulx", "ph r aU l k s"},
            {"proust", "ph r aU s t"},
            {"prout", "ph r aU t"},
            {"prove", "ph r u: v"},
            {"proved", "ph r u: v d"},
            {"proves", "ph r u: v z"},
            {"prow", "ph r aU"},
            {"prowl", "ph r aU l"},
            {"prows", "ph r aU z"},
            {"pru", "ph r u:"},
            {"prude", "ph r u: d"},
            {"prue", "ph r u:"},
            {"pruett", "ph r u: t"},
            {"prune", "ph r u: n"},
            {"pruned", "ph r u: n d"},
            {"prunes", "ph r u: n z"},
            {"prus", "ph r V s"},
            {"pru's", "ph r u: z"},
            {"pruss", "ph r V s"},
            {"pry", "ph r aI"},
            {"pryce", "ph r aI s"},
            {"prynne", "ph r I n"},
            {"p's", "ph i: z"},
            {"psalm", "s Q l m"},
            {"psalms", "s Q l m z"},
            {"psi", "s aI"},
            {"psych", "s aI k"},
            {"psyched", "s aI k t"},
            {"pty", "th aI"},
            {"pu", "ph u:"},
            {"pub", "ph V b"},
            {"pubs", "ph V b z"},
            {"puck", "ph V k"},
            {"puente", "ph u: n t"},
            {"puett", "ph u: t"},
            {"puff", "ph V f"},
            {"puffed", "ph V f t"},
            {"puffs", "ph V f s"},
            {"pugh", "ph j u:"},
            {"pughs", "ph j u: z"},
            {"puke", "ph j u: k"},
            {"pukes", "ph j u: k s"},
            {"pull", "ph U l"},
            {"pulled", "ph U l d"},
            {"pulls", "ph U l z"},
            {"pulp", "ph V l p"},
            {"puls", "ph U l z"},
            {"pulse", "ph V l s"},
            {"pump", "ph V m p"},
            {"pumped", "ph V m p t"},
            {"pumps", "ph V m p s"},
            {"pun", "ph V n"},
            {"punch", "ph V n tS"},
            {"punched", "ph V n tS t"},
            {"pung", "ph V N"},
            {"punk", "ph V N k"},
            {"punks", "ph V N k s"},
            {"puns", "ph V n z"},
            {"punt", "ph V n t"},
            {"punts", "ph V n t s"},
            {"pup", "ph V p"},
            {"pups", "ph V p s"},
            {"pur", "ph @r"},
            {"pure", "ph j U@"},
            {"purge", "ph @r dZ"},
            {"purged", "ph @r dZ d"},
            {"purr", "ph @r"},
            {"purrs", "ph @r z"},
            {"purse", "ph @r s"},
            {"pursed", "ph @r s t"},
            {"pus", "ph V s"},
            {"push", "ph U S"},
            {"pushed", "ph U S t"},
            {"puss", "ph U s"},
            {"put", "ph U t"},
            {"puts", "ph U t s"},
            {"putsch", "ph U tS"},
            {"putt", "ph V t"},
            {"putts", "ph V t s"},
            {"putz", "ph V t s"},
            {"pye", "ph aI"},
            {"pyke", "ph aI k"},
            {"pyle", "ph aI l"},
            {"pyne", "ph aI n"},
            {"pynes", "ph aI n z"},
            {"q", "kh j u:"},
            {"q.", "kh j u:"},
            {"qi", "kh i:"},
            {"qom", "kh Q m"},
            {"q's", "kh j u: z"},
            {"qu", "kh u:"},
            {"qua", "kh w Q"},
            {"quack", "kh w { k"},
            {"quacks", "kh w { k s"},
            {"quad", "kh w Q d"},
            {"quads", "kh w Q d z"},
            {"quai", "kh i:"},
            {"quail", "kh w eI l"},
            {"quails", "kh w eI l z"},
            {"quaint", "kh w eI n t"},
            {"quake", "kh w eI k"},
            {"quakes", "kh w eI k s"},
            {"qual", "kh w Q l"},
            {"quale", "kh w eI l"},
            {"qualms", "kh w Q m z"},
            {"quam", "kh w Q m"},
            {"quan", "kh w O: n"},
            {"quant", "kh w { n t"},
            {"quark", "kh w Q@ k"},
            {"quarks", "kh w Q@ k s"},
            {"quarles", "kh w O@ l z"},
            {"quart", "kh w O@ t"},
            {"quarts", "kh w O@ t s"},
            {"quartz", "kh w O@ t s"},
            {"quash", "kh w O: S"},
            {"quashed", "kh w O: S t"},
            {"quave", "kh w eI v"},
            {"quay", "kh i:"},
            {"quayle", "kh w eI l"},
            {"quayle's", "kh w eI l z"},
            {"que", "kh j u:"},
            {"queen", "kh w i: n"},
            {"queens", "kh w i: n z"},
            {"queen's", "kh w i: n z"},
            {"queer", "kh w I@"},
            {"queers", "kh w I@ z"},
            {"quell", "kh w e l"},
            {"quelled", "kh w e l d"},
            {"quench", "kh w e n tS"},
            {"querns", "kh w @r n z"},
            {"ques", "kh j u: z"},
            {"quest", "kh w e s t"},
            {"quests", "kh w e s t s"},
            {"queue", "kh j u:"},
            {"queues", "kh j u: z"},
            {"qui", "kh w i:"},
            {"quiche", "kh i: S"},
            {"quick", "kh w I k"},
            {"quid", "kh w I d"},
            {"quill", "kh w I l"},
            {"quilmes", "kh w I l m z"},
            {"quilt", "kh w I l t"},
            {"quilts", "kh w I l t s"},
            {"quin", "kh w I n"},
            {"quince", "kh w I n s"},
            {"quine", "kh w aI n"},
            {"quinn", "kh w I n"},
            {"quinn's", "kh w I n z"},
            {"quint", "kh w I n t"},
            {"quints", "kh w I n t s"},
            {"quip", "kh w I p"},
            {"quipped", "kh w I p t"},
            {"quips", "kh w I p s"},
            {"quire", "kh w aI r"},
            {"quirk", "kh w @r k"},
            {"quirks", "kh w @r k s"},
            {"quist", "kh w I s t"},
            {"quist's", "kh w I s t s"},
            {"quit", "kh w I t"},
            {"quite", "kh w aI t"},
            {"quits", "kh w I t s"},
            {"quiz", "kh w I z"},
            {"quizzed", "kh w I z d"},
            {"quo", "kh w @U"},
            {"quote", "kh w @U t"},
            {"quotes", "kh w @U t s"},
            {"r", "Q@"},
            {"r.", "Q@"},
            {"ra", "r Q"},
            {"raab", "r Q b"},
            {"raab's", "r Q b z"},
            {"raad", "r Q d"},
            {"raasch", "r Q S"},
            {"rab", "r { b"},
            {"rabe", "r eI b"},
            {"race", "r eI s"},
            {"raced", "r eI s t"},
            {"rach", "r { tS"},
            {"rack", "r { k"},
            {"racked", "r { k t"},
            {"racks", "r { k s"},
            {"rad", "r { d"},
            {"rae", "r eI"},
            {"raf", "r { f"},
            {"rafe", "r eI f"},
            {"raff", "r { f"},
            {"raft", "r { f t"},
            {"rafts", "r { f t s"},
            {"rag", "r { g"},
            {"rage", "r eI dZ"},
            {"raged", "r eI dZ d"},
            {"rags", "r { g z"},
            {"rah", "r Q"},
            {"rahm", "r { m"},
            {"raid", "r eI d"},
            {"raids", "r eI d z"},
            {"rail", "r eI l"},
            {"raile", "r eI l"},
            {"railed", "r eI l d"},
            {"rails", "r eI l z"},
            {"rail's", "r eI l z"},
            {"rain", "r eI n"},
            {"raine", "r eI n"},
            {"rained", "r eI n d"},
            {"raines", "r eI n z"},
            {"rains", "r eI n z"},
            {"raise", "r eI z"},
            {"raised", "r eI z d"},
            {"rait", "r eI t"},
            {"raj", "r Q Z"},
            {"rake", "r eI k"},
            {"raked", "r eI k t"},
            {"rakes", "r eI k s"},
            {"rales", "r eI l z"},
            {"ralf", "r Q l f"},
            {"rall", "r O: l"},
            {"ralls", "r O: l z"},
            {"ralph", "r { l f"},
            {"ralphs", "r { l f s"},
            {"ralph's", "r { l f s"},
            {"ram", "r { m"},
            {"rammed", "r { m d"},
            {"ramp", "r { m p"},
            {"ramps", "r { m p s"},
            {"rams", "r { m z"},
            {"ran", "r { n"},
            {"rance", "r { n s"},
            {"ranch", "r { n tS"},
            {"rand", "r { n d"},
            {"rands", "r { n d z"},
            {"rand's", "r { n d z"},
            {"rang", "r { N"},
            {"range", "r eI n dZ"},
            {"ranged", "r eI n dZ d"},
            {"ranh", "r { n"},
            {"rank", "r { N k"},
            {"ranke", "r { N k"},
            {"ranked", "r { N k t"},
            {"ranks", "r { N k s"},
            {"rann", "r { n"},
            {"rant", "r { n t"},
            {"rao", "r aU"},
            {"rao's", "r aU z"},
            {"rap", "r { p"},
            {"rape", "r eI p"},
            {"raped", "r eI p t"},
            {"rapes", "r eI p s"},
            {"rapp", "r { p"},
            {"rappe", "r { p"},
            {"rapped", "r { p t"},
            {"raps", "r { p s"},
            {"rapt", "r { p t"},
            {"rare", "r e@"},
            {"ras", "r { s"},
            {"rase", "r eI z"},
            {"rash", "r { S"},
            {"rask", "r { s k"},
            {"rasp", "r { s p"},
            {"rasped", "r { s p t"},
            {"rasps", "r { s p s"},
            {"rast", "r { s t"},
            {"rat", "r { t"},
            {"rate", "r eI t"},
            {"rates", "r eI t s"},
            {"rate's", "r eI t s"},
            {"rath", "r { T"},
            {"rathe", "r eI D"},
            {"rats", "r { t s"},
            {"rauch", "r O: tS"},
            {"raul", "r O: l"},
            {"raun", "r O: n"},
            {"rausch", "r aU S"},
            {"rave", "r eI v"},
            {"raved", "r eI v d"},
            {"raves", "r eI v z"},
            {"raw", "r Q"},
            {"rawl", "r O: l"},
            {"rax", "r { k s"},
            {"ray", "r eI"},
            {"raye", "r eI"},
            {"rayed", "r eI d"},
            {"rayle", "r eI l"},
            {"rayne", "r eI n"},
            {"raynes", "r eI n z"},
            {"rays", "r eI z"},
            {"ray's", "r eI z"},
            {"raze", "r eI z"},
            {"razed", "r eI z d"},
            {"re", "r eI"},
            {"rea", "r i:"},
            {"reach", "r i: tS"},
            {"reached", "r i: tS t"},
            {"read", "r e d"},
            {"reade", "r e d"},
            {"reads", "r i: d z"},
            {"read's", "r i: d z"},
            {"real", "r i: l"},
            {"realm", "r e l m"},
            {"realms", "r e l m z"},
            {"ream", "r i: m"},
            {"reames", "r i: m z"},
            {"reams", "r i: m z"},
            {"reap", "r i: p"},
            {"reaped", "r i: p t"},
            {"reaps", "r i: p s"},
            {"rear", "r I@"},
            {"reared", "r I@ d"},
            {"rears", "r I@ z"},
            {"reave", "r i: v"},
            {"reaves", "r i: v z"},
            {"rec", "r e k"},
            {"rech", "r e k"},
            {"reck", "r e k"},
            {"red", "r e d"},
            {"redd", "r e d"},
            {"reds", "r e d z"},
            {"ree", "r i:"},
            {"reece", "r i: s"},
            {"reed", "r i: d"},
            {"reeds", "r i: d z"},
            {"reed's", "r i: d z"},
            {"reef", "r i: f"},
            {"reefs", "r i: f s"},
            {"reek", "r i: k"},
            {"reeks", "r i: k s"},
            {"reel", "r i: l"},
            {"reeled", "r i: l d"},
            {"reels", "r i: l z"},
            {"reen", "r i: n"},
            {"rees", "r i: z"},
            {"reese", "r i: s"},
            {"reeve", "r i: v"},
            {"reeves", "r i: v z"},
            {"ref", "r e f"},
            {"reg", "r e g"},
            {"regs", "r e g z"},
            {"reh", "r e"},
            {"rehm", "r e m"},
            {"reich", "r aI k"},
            {"reiche", "r aI k"},
            {"reich's", "r aI k s"},
            {"reid", "r i: d"},
            {"reid's", "r i: d z"},
            {"reif", "r i: f"},
            {"reign", "r eI n"},
            {"reigned", "r eI n d"},
            {"reigns", "r eI n z"},
            {"reil", "r eI l"},
            {"reim", "r i: m"},
            {"rein", "r eI n"},
            {"reine", "r eI n"},
            {"reined", "r eI n d"},
            {"reinke", "r eI N k"},
            {"reins", "r eI n z"},
            {"reis", "r i: z"},
            {"reisch", "r aI S"},
            {"reiss", "r aI s"},
            {"reit", "r aI t"},
            {"reith", "r i: T"},
            {"rem", "r e m"},
            {"rempe", "r e m p"},
            {"ren", "r e n"},
            {"renk", "r e N k"},
            {"renn", "r e n"},
            {"renne", "r e n"},
            {"rens", "r e n z"},
            {"rent", "r e n t"},
            {"rents", "r e n t s"},
            {"rentsch", "r e n tS"},
            {"rentz", "r e n t s"},
            {"rep", "r e p"},
            {"rep.", "r e p"},
            {"repp", "r e p"},
            {"reps", "r e p s"},
            {"rep's", "r e p s"},
            {"res", "r eI z"},
            {"re's", "r eI z"},
            {"resh", "r e S"},
            {"ress", "r e s"},
            {"rest", "r e s t"},
            {"rests", "r e s t s"},
            {"ret", "r e t"},
            {"rev", "r e v"},
            {"revs", "r e v z"},
            {"revved", "r e v d"},
            {"rew", "r u:"},
            {"rex", "r e k s"},
            {"rey", "r eI"},
            {"rha", "r Q"},
            {"rhame", "r eI m"},
            {"rhee", "r i:"},
            {"rhein", "r aI n"},
            {"rhett", "r e t"},
            {"rhine", "r aI n"},
            {"rhines", "r aI n z"},
            {"rho", "r @U"},
            {"rhoades", "r @U d z"},
            {"rhoads", "r @U d z"},
            {"rhode", "r @U d"},
            {"rhodes", "r @U d z"},
            {"rhone", "r @U n"},
            {"rhyme", "r aI m"},
            {"rhymed", "r aI m d"},
            {"rhymes", "r aI m z"},
            {"rhyne", "r aI n"},
            {"rhynes", "r aI n z"},
            {"rials", "r aI l z"},
            {"rib", "r I b"},
            {"ribbed", "r I b d"},
            {"ribs", "r I b z"},
            {"ric", "r I k"},
            {"rice", "r aI s"},
            {"rich", "r I tS"},
            {"riche", "r I tS"},
            {"richts", "r I k t s"},
            {"rick", "r I k"},
            {"ricks", "r I k s"},
            {"rick's", "r I k s"},
            {"rid", "r I d"},
            {"ride", "r aI d"},
            {"rides", "r aI d z"},
            {"ridge", "r I dZ"},
            {"ridged", "r I dZ d"},
            {"rids", "r I d z"},
            {"riehl", "r i: l"},
            {"riel", "r i: l"},
            {"ries", "r aI z"},
            {"riess", "r i: s"},
            {"rieves", "r i: v z"},
            {"rife", "r aI f"},
            {"riff", "r I f"},
            {"riffs", "r I f s"},
            {"rift", "r I f t"},
            {"rifts", "r I f t s"},
            {"rig", "r I g"},
            {"rigg", "r I g"},
            {"rigged", "r I g d"},
            {"riggs", "r I g z"},
            {"right", "r aI t"},
            {"rights", "r aI t s"},
            {"rigs", "r I g z"},
            {"rile", "r aI l"},
            {"riled", "r aI l d"},
            {"riles", "r aI l z"},
            {"rill", "r I l"},
            {"rill's", "r I l z"},
            {"rim", "r I m"},
            {"rimes", "r aI m z"},
            {"rimmed", "r I m d"},
            {"rims", "r I m z"},
            {"rind", "r aI n d"},
            {"rinds", "r aI n d z"},
            {"rine", "r aI n"},
            {"rines", "r aI n z"},
            {"ring", "r I N"},
            {"ringe", "r I N"},
            {"ringed", "r I N d"},
            {"rings", "r I N z"},
            {"ring's", "r I N z"},
            {"rink", "r I N k"},
            {"rinks", "r I N k s"},
            {"rinse", "r I n s"},
            {"rip", "r I p"},
            {"ripe", "r aI p"},
            {"ripp", "r I p"},
            {"ripped", "r I p t"},
            {"rips", "r I p s"},
            {"risc", "r I s k"},
            {"rise", "r aI z"},
            {"risk", "r I s k"},
            {"risked", "r I s k t"},
            {"risks", "r I s k s"},
            {"riss", "r I s"},
            {"rist", "r I s t"},
            {"ritch", "r I tS"},
            {"rite", "r aI t"},
            {"rites", "r aI t s"},
            {"rite's", "r aI t s"},
            {"ritt", "r I t"},
            {"ritts", "r I t s"},
            {"ritz", "r I t s"},
            {"rives", "r aI v z"},
            {"rix", "r I k s"},
            {"riz", "r I z"},
            {"ro", "r @U"},
            {"roach", "r @U tS"},
            {"road", "r @U d"},
            {"roads", "r @U d z"},
            {"road's", "r @U d z"},
            {"roald", "r @U l d"},
            {"roam", "r @U m"},
            {"roamed", "r @U m d"},
            {"roams", "r @U m z"},
            {"roan", "r @U n"},
            {"roane", "r @U n"},
            {"roar", "r O@"},
            {"roared", "r O@ d"},
            {"roark", "r O@ k"},
            {"roarke", "r O@ k"},
            {"roars", "r O@ z"},
            {"roast", "r @U s t"},
            {"roasts", "r @U s t s"},
            {"roath", "r @U T"},
            {"rob", "r Q b"},
            {"robb", "r Q b"},
            {"robbed", "r Q b d"},
            {"robb's", "r Q b z"},
            {"robe", "r @U b"},
            {"robed", "r @U b d"},
            {"robes", "r @U b z"},
            {"robs", "r Q b z"},
            {"rob's", "r Q b z"},
            {"roc", "r Q k"},
            {"roch", "r Q k"},
            {"roche", "r @U tS"},
            {"rock", "r Q k"},
            {"rocked", "r Q k t"},
            {"rocks", "r Q k s"},
            {"rock's", "r Q k s"},
            {"rod", "r Q d"},
            {"rodd", "r Q d"},
            {"rode", "r @U d"},
            {"rodes", "r @U d z"},
            {"rodge", "r Q dZ"},
            {"rods", "r Q d z"},
            {"roe", "r @U"},
            {"roed", "r @U d"},
            {"roehm", "r @U m"},
            {"roes", "r @U z"},
            {"roff", "r O: f"},
            {"rog", "r Q g"},
            {"rogue", "r @U g"},
            {"rogues", "r @U g z"},
            {"roh", "r @U"},
            {"rohe", "r @U"},
            {"rohn", "r Q n"},
            {"roi", "r OI"},
            {"roil", "r OI l"},
            {"roiled", "r OI l d"},
            {"role", "r @U l"},
            {"roles", "r @U l z"},
            {"rolf", "r @U l f"},
            {"rolfe", "r @U l f"},
            {"rolfs", "r @U l f s"},
            {"roll", "r @U l"},
            {"rolled", "r @U l d"},
            {"rolls", "r @U l z"},
            {"rolph", "r @U l f"},
            {"rom", "r Q m"},
            {"rome", "r @U m"},
            {"rome's", "r @U m z"},
            {"romp", "r Q m p"},
            {"roms", "r Q m z"},
            {"rom's", "r Q m z"},
            {"ron", "r Q n"},
            {"rone", "r @U n"},
            {"rong", "r O: N"},
            {"ronne", "r Q n"},
            {"ron's", "r Q n z"},
            {"rood", "r u: d"},
            {"roof", "r u: f"},
            {"roofed", "r u: f t"},
            {"roofs", "r u: f s"},
            {"rook", "r U k"},
            {"rooke", "r U k"},
            {"rooks", "r U k s"},
            {"room", "r u: m"},
            {"rooms", "r u: m z"},
            {"room's", "r u: m z"},
            {"roop", "r u: p"},
            {"roos", "r u: z"},
            {"roose", "r u: z"},
            {"roost", "r u: s t"},
            {"root", "r u: t"},
            {"roots", "r u: t s"},
            {"rope", "r @U p"},
            {"roped", "r @U p t"},
            {"ropes", "r @U p s"},
            {"ropp", "r Q p"},
            {"roque", "r @U k"},
            {"rorke", "r O@ k"},
            {"ros", "r O: s"},
            {"rose", "r @U z"},
            {"rosh", "r O: S"},
            {"ross", "r Q s"},
            {"rosse", "r Q s"},
            {"rost", "r Q s t"},
            {"rot", "r Q t"},
            {"rote", "r @U t"},
            {"roth", "r O: T"},
            {"roth's", "r Q T s"},
            {"rots", "r Q t s"},
            {"rouge", "r u: Z"},
            {"rough", "r V f"},
            {"roughed", "r V f t"},
            {"rought", "r O: t"},
            {"round", "r aU n d"},
            {"rounds", "r aU n d z"},
            {"row", "r @U"},
            {"rowe", "r @U"},
            {"rowed", "r @U d"},
            {"rowe's", "r @U z"},
            {"rows", "r @U z"},
            {"row's", "r @U z"},
            {"rox", "r Q k s"},
            {"roy", "r OI"},
            {"royce", "r OI s"},
            {"royd", "r OI d"},
            {"roye", "r OI"},
            {"roy's", "r OI z"},
            {"roz", "r Q z"},
            {"r's", "Q@ z"},
            {"ru", "r u:"},
            {"rub", "r V b"},
            {"rubbed", "r V b d"},
            {"rube", "r u: b"},
            {"rubs", "r V b z"},
            {"ruck", "r V k"},
            {"rucks", "r V k s"},
            {"rud", "r V d"},
            {"rudd", "r V d"},
            {"rude", "r u: d"},
            {"rudge", "r V dZ"},
            {"rue", "r u:"},
            {"ruel", "r u: l"},
            {"rufe", "r u: f"},
            {"ruff", "r V f"},
            {"rug", "r V g"},
            {"rugg", "r V g"},
            {"rugs", "r V g z"},
            {"ruh", "r V"},
            {"ruhl", "r V l"},
            {"ruhnke", "r V N k"},
            {"ruhr", "r U@"},
            {"rule", "r u: l"},
            {"ruled", "r u: l d"},
            {"rules", "r u: l z"},
            {"rule's", "r u: l z"},
            {"rum", "r V m"},
            {"rump", "r V m p"},
            {"rumpf", "r V m p f"},
            {"rums", "r V m z"},
            {"run", "r V n"},
            {"runck", "r V N k"},
            {"rund", "r V n d"},
            {"rung", "r V N"},
            {"runge", "r V n dZ"},
            {"rungs", "r V N z"},
            {"runs", "r V n z"},
            {"rus", "r V s"},
            {"ruse", "r u: z"},
            {"rush", "r V S"},
            {"rushed", "r V S t"},
            {"rusk", "r V s k"},
            {"rusk's", "r V s k s"},
            {"russ", "r V s"},
            {"russe", "r V s"},
            {"rust", "r V s t"},
            {"rusts", "r V s t s"},
            {"rust's", "r V s t s"},
            {"rut", "r V t"},
            {"ruth", "r u: T"},
            {"ruth's", "r u: T s"},
            {"ruts", "r V t s"},
            {"rux", "r V k s"},
            {"ruyle", "r aI l"},
            {"rye", "r aI"},
            {"ryle", "r aI l"},
            {"s", "e s"},
            {"s.", "e s"},
            {"sa", "s Q"},
            {"saab", "s Q b"},
            {"saab's", "s Q b z"},
            {"saad", "s Q d"},
            {"saam", "s Q m"},
            {"saar", "s Q@"},
            {"sab's", "s { b z"},
            {"sac", "s { k"},
            {"sachs", "s { k s"},
            {"sack", "s { k"},
            {"sacked", "s { k t"},
            {"sacks", "s { k s"},
            {"sacs", "s { k s"},
            {"sad", "s { d"},
            {"safe", "s eI f"},
            {"safes", "s eI f s"},
            {"sag", "s { g"},
            {"sage", "s eI dZ"},
            {"sagged", "s { g d"},
            {"sags", "s { g z"},
            {"sai", "s aI"},
            {"said", "s e d"},
            {"sail", "s eI l"},
            {"sailed", "s eI l d"},
            {"sails", "s eI l z"},
            {"sain", "s eI n"},
            {"saine", "s eI n"},
            {"saint", "s eI n t"},
            {"saints", "s eI n t s"},
            {"saint's", "s eI n t s"},
            {"saints'", "s eI n t s"},
            {"sais", "s eI z"},
            {"sak", "s { k"},
            {"sake", "s eI k"},
            {"sakes", "s eI k s"},
            {"saks", "s { k s"},
            {"sal", "s { l"},
            {"sale", "s eI l"},
            {"sales", "s eI l z"},
            {"sale's", "s eI l z"},
            {"salk", "s O: k"},
            {"salk's", "s O: k s"},
            {"sall", "s O: l"},
            {"salle", "s eI l"},
            {"salm", "s Q m"},
            {"sal's", "s { l z"},
            {"salt", "s O: l t"},
            {"salts", "s O: l t s"},
            {"salve", "s Q v"},
            {"salves", "s Q v z"},
            {"salz", "s O: l z"},
            {"sam", "s { m"},
            {"same", "s eI m"},
            {"sames", "s eI m z"},
            {"samp", "s { m p"},
            {"sams", "s { m z"},
            {"sam's", "s { m z"},
            {"san", "s { n"},
            {"sand", "s { n d"},
            {"sande", "s { n d"},
            {"sands", "s { n d z"},
            {"sand's", "s { n d z"},
            {"sane", "s eI n"},
            {"sang", "s { N"},
            {"sank", "s { N k"},
            {"sans", "s { n z"},
            {"sant", "s { n t"},
            {"sao", "s aU"},
            {"sap", "s { p"},
            {"sapped", "s { p t"},
            {"saps", "s { p s"},
            {"sar", "s Q@"},
            {"sare", "s e@"},
            {"sarge", "s Q@ dZ"},
            {"sark", "s Q@ k"},
            {"sarles", "s Q@ l z"},
            {"sars", "s Q@ z"},
            {"sas", "s { s"},
            {"sash", "s { S"},
            {"sass", "s { s"},
            {"sasse", "s { s"},
            {"sat", "s { t"},
            {"sate", "s eI t"},
            {"sates", "s eI t s"},
            {"sauce", "s O: s"},
            {"saud", "s O: d"},
            {"sauk", "s O: k"},
            {"saul", "s O: l"},
            {"sauls", "s O: l z"},
            {"sault", "s O: l t"},
            {"saum", "s O: m"},
            {"saur", "s O@"},
            {"sauve", "s O: v"},
            {"save", "s eI v"},
            {"saved", "s eI v d"},
            {"saves", "s eI v z"},
            {"save's", "s eI v z"},
            {"saw", "s O:"},
            {"sawed", "s O: d"},
            {"saws", "s O: z"},
            {"sax", "s { k s"},
            {"saxe", "s { k s"},
            {"say", "s eI"},
            {"saye", "s eI"},
            {"sayles", "s eI l z"},
            {"sayre", "s e@"},
            {"sayres", "s e@ z"},
            {"says", "s e z"},
            {"scab", "s k { b"},
            {"scabs", "s k { b z"},
            {"scad", "s k { d"},
            {"scads", "s k { d z"},
            {"scaff", "s k { f"},
            {"scags", "s k { g z"},
            {"scaife", "s k eI f"},
            {"scald", "s k O: l d"},
            {"scalds", "s k O: l d z"},
            {"scale", "s k eI l"},
            {"scaled", "s k eI l d"},
            {"scales", "s k eI l z"},
            {"scalf", "s k { l f"},
            {"scalp", "s k { l p"},
            {"scalped", "s k { l p t"},
            {"scalps", "s k { l p s"},
            {"scam", "s k { m"},
            {"scammed", "s k { m d"},
            {"scams", "s k { m z"},
            {"scan", "s k { n"},
            {"scanned", "s k { n d"},
            {"scans", "s k { n z"},
            {"scant", "s k { n t"},
            {"scape", "s k eI p"},
            {"scar", "s k Q@"},
            {"scarce", "s k e@ s"},
            {"scare", "s k e@"},
            {"scared", "s k e@ d"},
            {"scares", "s k e@ z"},
            {"scarf", "s k Q@ f"},
            {"scarfs", "s k Q@ f s"},
            {"scarp", "s k Q@ p"},
            {"scarred", "s k Q@ d"},
            {"scars", "s k Q@ z"},
            {"scarves", "s k Q@ v z"},
            {"scat", "s k { t"},
            {"scene", "s i: n"},
            {"scenes", "s i: n z"},
            {"scent", "s e n t"},
            {"scents", "s e n t s"},
            {"schaab", "S Q b"},
            {"schacht", "S { k t"},
            {"schantz", "S { n t s"},
            {"schanz", "S { n s"},
            {"scharf", "S Q@ f"},
            {"scharff", "S Q@ f"},
            {"schatz", "S { t s"},
            {"schear", "S I@"},
            {"scheck", "S e k"},
            {"scheel", "S i: l"},
            {"scheele", "S i: l"},
            {"scheer", "S I@"},
            {"scheidt", "S aI t"},
            {"schein", "S aI n"},
            {"schell", "s k e l"},
            {"scheme", "s k i: m"},
            {"schemed", "s k i: m d"},
            {"schemes", "s k i: m z"},
            {"schenck", "S e N k"},
            {"schenk", "S e N k"},
            {"schertz", "S @r t s"},
            {"schick", "S I k"},
            {"schiele", "S i: l"},
            {"schiff", "S I f"},
            {"schild", "S aI l d"},
            {"schilt", "S I l t"},
            {"schist", "S I s t"},
            {"schists", "S I s t s"},
            {"schlang", "S l { N"},
            {"schlepp", "S l e p"},
            {"schley", "S l eI"},
            {"schlick", "S l I k"},
            {"schlitz", "S l I t s"},
            {"schlock", "S l Q k"},
            {"schloss", "S l O: s"},
            {"schmaltz", "S m Q l t s"},
            {"schmalz", "S m O: l z"},
            {"schmeck", "S m e k"},
            {"schmelz", "S m e l z"},
            {"schmidt", "S m I t"},
            {"schmidt's", "S m I t s"},
            {"schmitt", "S m I t"},
            {"schmitz", "S m I t s"},
            {"schmooze", "S m u: z"},
            {"schmuck", "S m V k"},
            {"schnapp", "S n { p"},
            {"schnapps", "S n { p s"},
            {"schnell", "S n e l"},
            {"schnitz", "S n I t s"},
            {"schnook", "S n U k"},
            {"schnooks", "S n U k s"},
            {"schnorr", "S n O@"},
            {"schnur", "S n @r"},
            {"schnurr", "S n @r"},
            {"scholz", "S @U l z"},
            {"schone", "S @U n"},
            {"schoof", "S U f"},
            {"school", "s k u: l"},
            {"schooled", "s k u: l d"},
            {"schools", "s k u: l z"},
            {"school's", "s k u: l z"},
            {"schoon", "s k u: n"},
            {"schott", "S Q t"},
            {"schott's", "S Q t s"},
            {"schou", "S u:"},
            {"schow", "S aU"},
            {"schram", "S r { m"},
            {"schrank", "S r { N k"},
            {"schreck", "S r e k"},
            {"schrick", "S r I k"},
            {"schroth", "S r O: T"},
            {"schtick", "S t I k"},
            {"schug", "S V g"},
            {"schuh", "S u:"},
            {"schul", "S U l"},
            {"schultz", "S U l t s"},
            {"schultze", "S V l t z"},
            {"schulz", "S U l t s"},
            {"schulze", "S U l t s"},
            {"schwab", "S w Q b"},
            {"schwab's", "S w Q b z"},
            {"schwartz", "S w O@ t s"},
            {"schwarz", "S w O@ t s"},
            {"schwenk", "S w e N k"},
            {"schweppes", "S w e p s"},
            {"schwing", "S w I N"},
            {"schwinn", "S w I n"},
            {"sci", "s i:"},
            {"scire", "s aI r"},
            {"scoff", "s k O: f"},
            {"scoffed", "s k O: f t"},
            {"scoffs", "s k O: f s"},
            {"scold", "s k @U l d"},
            {"scolds", "s k @U l d z"},
            {"scoles", "s k @U l z"},
            {"sconce", "s k Q n s"},
            {"scone", "s k @U n"},
            {"scones", "s k @U n z"},
            {"scoop", "s k u: p"},
            {"scooped", "s k u: p t"},
            {"scoops", "s k u: p s"},
            {"scoot", "s k u: t"},
            {"scoots", "s k u: t s"},
            {"scope", "s k @U p"},
            {"scopes", "s k @U p s"},
            {"scorch", "s k O@ tS"},
            {"scorched", "s k O@ tS t"},
            {"score", "s k O@"},
            {"scored", "s k O@ d"},
            {"scores", "s k O@ z"},
            {"scorn", "s k O@ n"},
            {"scorned", "s k O@ n d"},
            {"scorns", "s k O@ n z"},
            {"scot", "s k Q t"},
            {"scotch", "s k Q tS"},
            {"scotched", "s k Q tS t"},
            {"scots", "s k Q t s"},
            {"scott", "s k Q t"},
            {"scotts", "s k Q t s"},
            {"scott's", "s k Q t s"},
            {"scourge", "s k @r dZ"},
            {"scout", "s k aU t"},
            {"scouts", "s k aU t s"},
            {"scout's", "s k aU t s"},
            {"scow", "s k aU"},
            {"scowl", "s k aU l"},
            {"scowled", "s k aU l d"},
            {"scrap", "s k r { p"},
            {"scrape", "s k r eI p"},
            {"scraped", "s k r eI p t"},
            {"scrapes", "s k r eI p s"},
            {"scrapped", "s k r { p t"},
            {"scraps", "s k r { p s"},
            {"scratch", "s k r { tS"},
            {"scratched", "s k r { tS t"},
            {"scrawl", "s k r O: l"},
            {"scrawled", "s k r O: l d"},
            {"scream", "s k r i: m"},
            {"screamed", "s k r i: m d"},
            {"screams", "s k r i: m z"},
            {"screech", "s k r i: tS"},
            {"screeched", "s k r i: tS t"},
            {"screed", "s k r i: d"},
            {"screen", "s k r i: n"},
            {"screened", "s k r i: n d"},
            {"screens", "s k r i: n z"},
            {"screw", "s k r u:"},
            {"screwed", "s k r u: d"},
            {"screws", "s k r u: z"},
            {"scribe", "s k r aI b"},
            {"scribes", "s k r aI b z"},
            {"scrimp", "s k r I m p"},
            {"scrip", "s k r I p"},
            {"scripps", "s k r I p s"},
            {"script", "s k r I p t"},
            {"scripts", "s k r I p t s"},
            {"scrod", "s k r Q d"},
            {"scroll", "s k r @U l"},
            {"scrolls", "s k r @U l z"},
            {"scrooge", "s k r u: dZ"},
            {"scrounge", "s k r aU n dZ"},
            {"scrounged", "s k r aU n dZ d"},
            {"scrub", "s k r V b"},
            {"scrubbed", "s k r V b d"},
            {"scruggs", "s k r V g z"},
            {"scrunch", "s k r V n tS"},
            {"scrunched", "s k r V n tS t"},
            {"scud", "s k V d"},
            {"scuds", "s k V d z"},
            {"scuff", "s k V f"},
            {"scuffed", "s k V f t"},
            {"scull", "s k V l"},
            {"sculpt", "s k V l p t"},
            {"sculpts", "s k V l p t s"},
            {"scum", "s k V m"},
            {"scutt", "s k V t"},
            {"sea", "s i:"},
            {"seal", "s i: l"},
            {"seale", "s i: l"},
            {"sealed", "s i: l d"},
            {"seals", "s i: l z"},
            {"seal's", "s i: l z"},
            {"seam", "s i: m"},
            {"seams", "s i: m z"},
            {"sean", "S O: n"},
            {"sean's", "S O: n z"},
            {"sear", "s I@"},
            {"search", "s @r tS"},
            {"searched", "s @r tS t"},
            {"seared", "s I@ d"},
            {"searle", "s @r l"},
            {"searles", "s @r l z"},
            {"searle's", "s @r l z"},
            {"sears", "s I@ z"},
            {"seas", "s i: z"},
            {"sea's", "s i: z"},
            {"sease", "s i: z"},
            {"seat", "s i: t"},
            {"seats", "s i: t s"},
            {"seat's", "s i: t s"},
            {"sec", "s e k"},
            {"seck", "s e k"},
            {"sect", "s e k t"},
            {"sects", "s e k t s"},
            {"sect's", "s e k t s"},
            {"sedge", "s e dZ"},
            {"see", "s i:"},
            {"seed", "s i: d"},
            {"seeds", "s i: d z"},
            {"seek", "s i: k"},
            {"seeks", "s i: k s"},
            {"seel", "s i: l"},
            {"seem", "s i: m"},
            {"seemed", "s i: m d"},
            {"seems", "s i: m z"},
            {"seen", "s i: n"},
            {"seep", "s i: p"},
            {"seeped", "s i: p t"},
            {"seeps", "s i: p s"},
            {"seer", "s i: r"},
            {"seers", "s i: r z"},
            {"sees", "s i: z"},
            {"seethe", "s i: D"},
            {"segue", "s e g"},
            {"seid", "s aI d"},
            {"seif", "s i: f"},
            {"seige", "s i: dZ"},
            {"seil", "s aI l"},
            {"seils", "s aI l z"},
            {"seis", "s i: s"},
            {"seitz", "s aI t s"},
            {"seize", "s i: z"},
            {"seized", "s i: z d"},
            {"sejm", "s eI m"},
            {"self", "s e l f"},
            {"selfs", "s e l f s"},
            {"self's", "s e l f s"},
            {"sell", "s e l"},
            {"selle", "s e l"},
            {"sells", "s e l z"},
            {"selves", "s e l v z"},
            {"selz", "s e l z"},
            {"semmes", "s e m z"},
            {"sen", "s e n"},
            {"send", "s e n d"},
            {"sends", "s e n d z"},
            {"senn", "s e n"},
            {"sens", "s e n z"},
            {"sense", "s e n s"},
            {"sensed", "s e n s t"},
            {"sent", "s e n t"},
            {"seoul", "s @U l"},
            {"seoul's", "s @U l z"},
            {"sep", "s e p"},
            {"sept", "s e p t"},
            {"serb", "s @r b"},
            {"serbs", "s @r b z"},
            {"serb's", "s @r b z"},
            {"sere", "s I@"},
            {"seres", "s i: r z"},
            {"serf", "s @r f"},
            {"serfs", "s @r f s"},
            {"serge", "s @r dZ"},
            {"serr", "s e@"},
            {"serv", "s @r v"},
            {"serve", "s @r v"},
            {"served", "s @r v d"},
            {"serves", "s @r v z"},
            {"set", "s e t"},
            {"seth", "s e T"},
            {"seth's", "s e T s"},
            {"sets", "s e t s"},
            {"seuss", "s u: s"},
            {"sew", "s @U"},
            {"sewed", "s @U d"},
            {"sewn", "s @U n"},
            {"sex", "s e k s"},
            {"sexed", "s e k s t"},
            {"sha", "S Q"},
            {"shack", "S { k"},
            {"shacks", "S { k s"},
            {"shad", "S { d"},
            {"shade", "S eI d"},
            {"shades", "S eI d z"},
            {"shad's", "S { d z"},
            {"shaff", "S { f"},
            {"shaft", "S { f t"},
            {"shafts", "S { f t s"},
            {"shag", "S { g"},
            {"shah", "S Q"},
            {"shah's", "S Q z"},
            {"shaikh", "S eI k"},
            {"shaine", "S eI n"},
            {"shake", "S eI k"},
            {"shakes", "S eI k s"},
            {"shale", "S eI l"},
            {"shales", "S eI l z"},
            {"shall", "S { l"},
            {"shalt", "S { l t"},
            {"sham", "S { m"},
            {"shame", "S eI m"},
            {"shamed", "S eI m d"},
            {"shames", "S eI m z"},
            {"shan", "S { n"},
            {"shand", "S { n d"},
            {"shands", "S { n d z"},
            {"shane", "S eI n"},
            {"shank", "S { N k"},
            {"shanks", "S { N k s"},
            {"shap", "S { p"},
            {"shape", "S eI p"},
            {"shaped", "S eI p t"},
            {"shapes", "S eI p s"},
            {"shard", "S Q@ d"},
            {"shards", "S Q@ d z"},
            {"share", "S e@"},
            {"shared", "S e@ d"},
            {"shares", "S e@ z"},
            {"share's", "S e@ z"},
            {"shark", "S Q@ k"},
            {"sharks", "S Q@ k s"},
            {"sharp", "S Q@ p"},
            {"sharpe", "S Q@ p"},
            {"sharps", "S Q@ p s"},
            {"sharp's", "S Q@ p s"},
            {"shaul", "S O: l"},
            {"shaun", "S O: n"},
            {"shave", "S eI v"},
            {"shaved", "S eI v d"},
            {"shaves", "S eI v z"},
            {"shaw", "S O:"},
            {"shawl", "S O: l"},
            {"shawls", "S O: l z"},
            {"shawn", "S O: n"},
            {"shawn's", "S O: n z"},
            {"shaw's", "S O: z"},
            {"shay", "S eI"},
            {"shayne", "S eI n"},
            {"shays", "S eI z"},
            {"she", "S i:"},
            {"shea", "S eI"},
            {"sheaf", "S i: f"},
            {"shear", "S I@"},
            {"sheard", "S I@ d"},
            {"sheared", "S I@ d"},
            {"shearn", "S I@ n"},
            {"shears", "S i: r z"},
            {"shear's", "S i: r z"},
            {"shea's", "S eI z"},
            {"sheath", "S i: T"},
            {"sheathe", "S i: D"},
            {"sheathed", "S i: D d"},
            {"sheaves", "S i: v z"},
            {"shed", "S e d"},
            {"she'd", "S i: d"},
            {"shedd", "S e d"},
            {"sheds", "S e d z"},
            {"sheen", "S i: n"},
            {"sheep", "S i: p"},
            {"sheep's", "S i: p s"},
            {"sheer", "S I@"},
            {"sheesh", "S i: S"},
            {"sheet", "S i: t"},
            {"sheets", "S i: t s"},
            {"sheff", "S e f"},
            {"sheik", "S i: k"},
            {"sheikh", "S i: k"},
            {"sheiks", "S i: k s"},
            {"sheik's", "S i: k s"},
            {"shelf", "S e l f"},
            {"shell", "S e l"},
            {"she'll", "S i: l"},
            {"shelled", "S e l d"},
            {"shells", "S e l z"},
            {"shell's", "S e l z"},
            {"shelve", "S e l v"},
            {"shelved", "S e l v d"},
            {"shelves", "S e l v z"},
            {"shen", "S e n"},
            {"sheng", "S e N"},
            {"shep", "S e p"},
            {"shepp", "S e p"},
            {"sher", "S @r"},
            {"shere", "S I@"},
            {"sherr", "S e@"},
            {"sherr's", "S e@ z"},
            {"she's", "S i: z"},
            {"sheth", "S e T"},
            {"shew", "S u:"},
            {"shi", "S i:"},
            {"shick", "S I k"},
            {"shied", "S aI d"},
            {"shieh", "S i:"},
            {"shiel", "S i: l"},
            {"shield", "S i: l d"},
            {"shields", "S i: l d z"},
            {"shield's", "S i: l d z"},
            {"shiels", "S i: l z"},
            {"shies", "S aI z"},
            {"shiff", "S I f"},
            {"shift", "S I f t"},
            {"shifts", "S I f t s"},
            {"shih", "S i:"},
            {"shill", "S I l"},
            {"shim", "S I m"},
            {"shin", "S I n"},
            {"shine", "S aI n"},
            {"shined", "S aI n d"},
            {"shines", "S aI n z"},
            {"shing", "S I N"},
            {"ship", "S I p"},
            {"shipp", "S I p"},
            {"shipped", "S I p t"},
            {"shipp's", "S I p s"},
            {"ships", "S I p s"},
            {"ship's", "S I p s"},
            {"shir", "S @r"},
            {"shire", "S aI r"},
            {"shires", "S aI r z"},
            {"shirk", "S @r k"},
            {"shirked", "S @r k t"},
            {"shirl", "S @r l"},
            {"shirr", "S @r"},
            {"shirt", "S @r t"},
            {"shirts", "S @r t s"},
            {"shit", "S i: t"},
            {"shive", "S aI v"},
            {"shives", "S aI v z"},
            {"shoals", "S @U l z"},
            {"shock", "S Q k"},
            {"shocked", "S Q k t"},
            {"shocks", "S Q k s"},
            {"shod", "S Q d"},
            {"shoe", "S u:"},
            {"shoes", "S u: z"},
            {"shoe's", "S u: z"},
            {"sholes", "S @U l z"},
            {"shone", "S Q n"},
            {"shoo", "S u:"},
            {"shooed", "S u: d"},
            {"shook", "S U k"},
            {"shoop", "S u: p"},
            {"shoot", "S u: t"},
            {"shoots", "S u: t s"},
            {"shop", "S Q p"},
            {"shope", "S @U p"},
            {"shoppe", "S Q p"},
            {"shopped", "S Q p t"},
            {"shoppes", "S Q p s"},
            {"shops", "S Q p s"},
            {"shop's", "S Q p s"},
            {"shore", "S O@"},
            {"shored", "S O@ d"},
            {"shores", "S O@ z"},
            {"shore's", "S O@ z"},
            {"shorn", "S O@ n"},
            {"short", "S O@ t"},
            {"shorts", "S O@ t s"},
            {"shot", "S Q t"},
            {"shots", "S Q t s"},
            {"shott", "S Q t"},
            {"shotts", "S Q t s"},
            {"shough", "S aU"},
            {"should", "S U d"},
            {"shoup", "S u: p"},
            {"shouse", "s h aU s"},
            {"shout", "S aU t"},
            {"shouts", "S aU t s"},
            {"shove", "S V v"},
            {"shoved", "S V v d"},
            {"shoves", "S V v z"},
            {"show", "S @U"},
            {"showed", "S @U d"},
            {"shown", "S @U n"},
            {"shows", "S @U z"},
            {"shrank", "S r { N k"},
            {"shred", "S r e d"},
            {"shreds", "S r e d z"},
            {"shreeve", "S r i: v"},
            {"shreve", "S r i: v"},
            {"shrewd", "S r u: d"},
            {"shri", "S r i:"},
            {"shriek", "S r i: k"},
            {"shrieked", "S r i: k t"},
            {"shrieks", "S r i: k s"},
            {"shrift", "S r I f t"},
            {"shrill", "S r I l"},
            {"shrimp", "S r I m p"},
            {"shrine", "S r aI n"},
            {"shrines", "S r aI n z"},
            {"shrink", "S r I N k"},
            {"shrinks", "S r I N k s"},
            {"shroff", "S r O: f"},
            {"shroud", "S r aU d"},
            {"shrouds", "S r aU d z"},
            {"shrub", "S r V b"},
            {"shrubs", "S r V b z"},
            {"shrug", "S r V g"},
            {"shrugged", "S r V g d"},
            {"shrugs", "S r V g z"},
            {"shrunk", "S r V N k"},
            {"shtick", "S t I k"},
            {"shu", "S u:"},
            {"shuck", "S V k"},
            {"shucked", "S V k t"},
            {"shucks", "S V k s"},
            {"shue", "S u:"},
            {"shuff", "S V f"},
            {"shull", "S V l"},
            {"shultz", "S U l t s"},
            {"shum", "S V m"},
            {"shun", "S V n"},
            {"shunk", "S V N k"},
            {"shunned", "S V n d"},
            {"shuns", "S V n z"},
            {"shunt", "S V n t"},
            {"shure", "S U@"},
            {"shut", "S V t"},
            {"shute", "S u: t"},
            {"shutes", "S u: t s"},
            {"shuts", "S V t s"},
            {"shy", "S aI"},
            {"si", "s i:"},
            {"siang", "s j { N"},
            {"sias", "Z O: z"},
            {"sib", "s I b"},
            {"sic", "s I k"},
            {"sick", "s I k"},
            {"sid", "s I d"},
            {"side", "s aI d"},
            {"sides", "s aI d z"},
            {"side's", "s aI d z"},
            {"sids", "s I d z"},
            {"sid's", "s I d z"},
            {"sie", "s i:"},
            {"sieg", "s i: g"},
            {"siege", "s i: dZ"},
            {"sieve", "s I v"},
            {"sieves", "s I v z"},
            {"sift", "s I f t"},
            {"sifts", "s I f t s"},
            {"sig", "s I g"},
            {"sigh", "s aI"},
            {"sighed", "s aI d"},
            {"sighs", "s aI z"},
            {"sight", "s aI t"},
            {"sights", "s aI t s"},
            {"sign", "s aI n"},
            {"signed", "s aI n d"},
            {"signs", "s aI n z"},
            {"sign's", "s aI n z"},
            {"sikes", "s aI k s"},
            {"sikh", "s i: k"},
            {"sikhs", "s i: k s"},
            {"sil", "s I l"},
            {"silk", "s I l k"},
            {"silks", "s I l k s"},
            {"sill", "s I l"},
            {"sills", "s I l z"},
            {"silt", "s I l t"},
            {"sim", "s I m"},
            {"sime", "s aI m"},
            {"simm", "s I m"},
            {"simms", "s I m z"},
            {"sims", "s I m z"},
            {"sin", "s I n"},
            {"since", "s I n s"},
            {"sind", "s I n d"},
            {"sine", "s aI n"},
            {"sines", "s aI n z"},
            {"sing", "s I N"},
            {"singe", "s I n dZ"},
            {"singed", "s I n dZ d"},
            {"singh", "s I N"},
            {"singh's", "s I N z"},
            {"sings", "s I N z"},
            {"sink", "s I N k"},
            {"sinks", "s I N k s"},
            {"sinn", "s I n"},
            {"sinned", "s I n d"},
            {"sins", "s I n z"},
            {"sioux", "s u:"},
            {"sip", "s I p"},
            {"sipe", "s aI p"},
            {"sipes", "s aI p s"},
            {"sipp", "s I p"},
            {"sipped", "s I p t"},
            {"sips", "s I p s"},
            {"sir", "s @r"},
            {"sires", "s aI r z"},
            {"sis", "s I s"},
            {"sit", "s I t"},
            {"site", "s aI t"},
            {"sites", "s aI t s"},
            {"site's", "s aI t s"},
            {"sithe", "s aI D"},
            {"sits", "s I t s"},
            {"sitz", "s I t s"},
            {"six", "s I k s"},
            {"sixth", "s I k s T"},
            {"sixths", "s I k s T s"},
            {"size", "s aI z"},
            {"sized", "s aI z d"},
            {"skaff", "s k { f"},
            {"skate", "s k eI t"},
            {"skates", "s k eI t s"},
            {"skeat", "s k i: t"},
            {"skeel", "s k i: l"},
            {"skeen", "s k i: n"},
            {"skeens", "s k i: n z"},
            {"skees", "s k i: z"},
            {"skeet", "s k i: t"},
            {"skeets", "s k i: t s"},
            {"skene", "s k i: n"},
            {"sketch", "s k e tS"},
            {"sketched", "s k e tS t"},
            {"skew", "s k j u:"},
            {"skewed", "s k j u: d"},
            {"skews", "s k j u: z"},
            {"ski", "s k i:"},
            {"skid", "s k I d"},
            {"skids", "s k I d z"},
            {"skied", "s k i: d"},
            {"skies", "s k aI z"},
            {"skiff", "s k I f"},
            {"skiffs", "s k I f s"},
            {"skill", "s k I l"},
            {"skilled", "s k I l d"},
            {"skills", "s k I l z"},
            {"skim", "s k I m"},
            {"skimmed", "s k I m d"},
            {"skimp", "s k I m p"},
            {"skin", "s k I n"},
            {"skinks", "s k I N k s"},
            {"skinned", "s k I n d"},
            {"skins", "s k I n z"},
            {"skin's", "s k I n z"},
            {"skip", "s k I p"},
            {"skipp", "s k I p"},
            {"skipped", "s k I p t"},
            {"skips", "s k I p s"},
            {"skirt", "s k @r t"},
            {"skirts", "s k @r t s"},
            {"skis", "s k i: z"},
            {"skit", "s k I t"},
            {"skits", "s k I t s"},
            {"sklar", "s k l Q@"},
            {"skoal", "s k @U l"},
            {"skoff", "s k O: f"},
            {"skulk", "s k V l k"},
            {"skull", "s k V l"},
            {"skulls", "s k V l z"},
            {"skunk", "s k V N k"},
            {"skunks", "s k V N k s"},
            {"sky", "s k aI"},
            {"sky's", "s k aI z"},
            {"slab", "s l { b"},
            {"slabs", "s l { b z"},
            {"slack", "s l { k"},
            {"slacked", "s l { k t"},
            {"slacks", "s l { k s"},
            {"slade", "s l eI d"},
            {"slag", "s l { g"},
            {"slain", "s l eI n"},
            {"slam", "s l { m"},
            {"slammed", "s l { m d"},
            {"slams", "s l { m z"},
            {"slane", "s l eI n"},
            {"slang", "s l { N"},
            {"slant", "s l { n t"},
            {"slap", "s l { p"},
            {"slape", "s l eI p"},
            {"slapped", "s l { p t"},
            {"slaps", "s l { p s"},
            {"slash", "s l { S"},
            {"slashed", "s l { S t"},
            {"slat", "s l { t"},
            {"slate", "s l eI t"},
            {"slates", "s l eI t s"},
            {"slats", "s l { t s"},
            {"slav", "s l Q v"},
            {"slave", "s l eI v"},
            {"slaves", "s l eI v z"},
            {"slavs", "s l Q v z"},
            {"slaw", "s l O:"},
            {"slay", "s l eI"},
            {"sleaze", "s l i: z"},
            {"sled", "s l e d"},
            {"sledge", "s l e dZ"},
            {"sleds", "s l e d z"},
            {"slee", "s l i:"},
            {"sleek", "s l i: k"},
            {"sleep", "s l i: p"},
            {"sleeps", "s l i: p s"},
            {"sleet", "s l i: t"},
            {"sleeve", "s l i: v"},
            {"sleeved", "s l i: v d"},
            {"sleeves", "s l i: v z"},
            {"sleigh", "s l eI"},
            {"sleight", "s l aI t"},
            {"slemp", "s l e m p"},
            {"slept", "s l e p t"},
            {"sleuth", "s l u: T"},
            {"sleuths", "s l u: T s"},
            {"slew", "s l u:"},
            {"slice", "s l aI s"},
            {"sliced", "s l aI s t"},
            {"slick", "s l I k"},
            {"slicked", "s l I k t"},
            {"slicks", "s l I k s"},
            {"slid", "s l I d"},
            {"slide", "s l aI d"},
            {"slides", "s l aI d z"},
            {"slight", "s l aI t"},
            {"slights", "s l aI t s"},
            {"slim", "s l I m"},
            {"slime", "s l aI m"},
            {"slimmed", "s l I m d"},
            {"slims", "s l I m z"},
            {"sling", "s l I N"},
            {"slings", "s l I N z"},
            {"slip", "s l I p"},
            {"slipped", "s l I p t"},
            {"slips", "s l I p s"},
            {"slit", "s l I t"},
            {"slits", "s l I t s"},
            {"sloan", "s l @U n"},
            {"sloane", "s l @U n"},
            {"sloan's", "s l @U n z"},
            {"sloat", "s l @U t"},
            {"slob", "s l Q b"},
            {"slobs", "s l Q b z"},
            {"sloe", "s l @U"},
            {"slog", "s l Q g"},
            {"slogged", "s l Q g d"},
            {"slone", "s l @U n"},
            {"sloop", "s l u: p"},
            {"slop", "s l Q p"},
            {"slope", "s l @U p"},
            {"slopes", "s l @U p s"},
            {"slosh", "s l Q S"},
            {"slot", "s l Q t"},
            {"sloth", "s l @U T"},
            {"slots", "s l Q t s"},
            {"slouch", "s l aU tS"},
            {"slough", "s l V f"},
            {"sloughs", "s l V f s"},
            {"slow", "s l @U"},
            {"slowed", "s l @U d"},
            {"slows", "s l @U z"},
            {"sludge", "s l V dZ"},
            {"slug", "s l V g"},
            {"slugged", "s l V g d"},
            {"slugs", "s l V g z"},
            {"sluice", "s l u: s"},
            {"slum", "s l V m"},
            {"slump", "s l V m p"},
            {"slumped", "s l V m p t"},
            {"slumps", "s l V m p s"},
            {"slums", "s l V m z"},
            {"slung", "s l V N"},
            {"slur", "s l @r"},
            {"slurp", "s l @r p"},
            {"slurred", "s l @r d"},
            {"slurs", "s l @r z"},
            {"slush", "s l V S"},
            {"slut", "s l V t"},
            {"sly", "s l aI"},
            {"smack", "s m { k"},
            {"smacked", "s m { k t"},
            {"smacks", "s m { k s"},
            {"smail", "s m eI l"},
            {"small", "s m O: l"},
            {"smalls", "s m O: l z"},
            {"smaltz", "s m { l t s"},
            {"smarr", "s m { r"},
            {"smart", "s m Q@ t"},
            {"smarts", "s m Q@ t s"},
            {"smartt", "s m Q@ t"},
            {"smash", "s m { S"},
            {"smashed", "s m { S t"},
            {"smear", "s m I@"},
            {"smeared", "s m I@ d"},
            {"smears", "s m I@ z"},
            {"smee", "s m i:"},
            {"smell", "s m e l"},
            {"smelled", "s m e l d"},
            {"smells", "s m e l z"},
            {"smelt", "s m e l t"},
            {"smile", "s m aI l"},
            {"smiled", "s m aI l d"},
            {"smiles", "s m aI l z"},
            {"smirk", "s m @r k"},
            {"smirks", "s m @r k s"},
            {"smit", "s m I t"},
            {"smith", "s m I T"},
            {"smiths", "s m I T s"},
            {"smith's", "s m I T s"},
            {"smits", "s m I t s"},
            {"smock", "s m Q k"},
            {"smog", "s m Q g"},
            {"smoke", "s m @U k"},
            {"smoked", "s m @U k t"},
            {"smokes", "s m @U k s"},
            {"smoot", "s m u: t"},
            {"smooth", "s m u: D"},
            {"smoothed", "s m u: D d"},
            {"smooths", "s m u: D z"},
            {"smoots", "s m u: t s"},
            {"smouse", "s m aU s"},
            {"smudge", "s m V dZ"},
            {"smudged", "s m V dZ d"},
            {"smug", "s m V g"},
            {"smurf", "s m @r f"},
            {"smurfs", "s m @r f s"},
            {"smut", "s m V t"},
            {"smyth", "s m aI T"},
            {"smythe", "s m aI D"},
            {"snack", "s n { k"},
            {"snacks", "s n { k s"},
            {"snag", "s n { g"},
            {"snagged", "s n { g d"},
            {"snags", "s n { g z"},
            {"snail", "s n eI l"},
            {"snails", "s n eI l z"},
            {"snail's", "s n eI l z"},
            {"snake", "s n eI k"},
            {"snakes", "s n eI k s"},
            {"snake's", "s n eI k s"},
            {"snap", "s n { p"},
            {"snape", "s n eI p"},
            {"snapp", "s n { p"},
            {"snapped", "s n { p t"},
            {"snaps", "s n { p s"},
            {"snare", "s n e@"},
            {"snared", "s n e@ d"},
            {"snares", "s n e@ z"},
            {"snarl", "s n Q@ l"},
            {"snarls", "s n Q@ l z"},
            {"snatch", "s n { tS"},
            {"snatched", "s n { tS t"},
            {"snead", "s n i: d"},
            {"sneak", "s n i: k"},
            {"sneaked", "s n i: k t"},
            {"sneaks", "s n i: k s"},
            {"sneath", "s n e T"},
            {"snee", "s n i:"},
            {"sneed", "s n i: d"},
            {"sneer", "s n I@"},
            {"sneered", "s n I@ d"},
            {"sneers", "s n I@ z"},
            {"sneeze", "s n i: z"},
            {"sneezed", "s n i: z d"},
            {"snell", "s n e l"},
            {"snide", "s n aI d"},
            {"sniff", "s n I f"},
            {"sniffed", "s n I f t"},
            {"sniffs", "s n I f s"},
            {"snip", "s n I p"},
            {"snipe", "s n aI p"},
            {"sniped", "s n aI p t"},
            {"snipes", "s n aI p s"},
            {"snips", "s n I p s"},
            {"snitch", "s n I tS"},
            {"snob", "s n Q b"},
            {"snobs", "s n Q b z"},
            {"snoke", "s n @U k"},
            {"snook", "s n U k"},
            {"snooks", "s n U k s"},
            {"snoop", "s n u: p"},
            {"snooze", "s n u: z"},
            {"snore", "s n O@"},
            {"snort", "s n O@ t"},
            {"snorts", "s n O@ t s"},
            {"snot", "s n O: t"},
            {"snout", "s n aU t"},
            {"snow", "s n @U"},
            {"snowed", "s n @U d"},
            {"snows", "s n @U z"},
            {"snub", "s n V b"},
            {"snubbed", "s n V b d"},
            {"snuck", "s n V k"},
            {"snuff", "s n V f"},
            {"snuffed", "s n V f t"},
            {"snuffs", "s n V f s"},
            {"snug", "s n V g"},
            {"so", "s @U"},
            {"soak", "s @U k"},
            {"soaked", "s @U k t"},
            {"soaks", "s @U k s"},
            {"soap", "s @U p"},
            {"soaps", "s @U p s"},
            {"soar", "s O@"},
            {"soared", "s O@ d"},
            {"soars", "s O@ z"},
            {"soave", "s @U v"},
            {"sob", "s Q b"},
            {"sobbed", "s Q b d"},
            {"sobs", "s Q b z"},
            {"sock", "s Q k"},
            {"socked", "s Q k t"},
            {"socks", "s Q k s"},
            {"sod", "s Q d"},
            {"soft", "s Q f t"},
            {"sohn", "s Q n"},
            {"soil", "s OI l"},
            {"soiled", "s OI l d"},
            {"soils", "s OI l z"},
            {"soil's", "s OI l z"},
            {"soir", "s OI r"},
            {"sok", "s Q k"},
            {"sol", "s Q l"},
            {"sold", "s @U l d"},
            {"sole", "s @U l"},
            {"soles", "s @U l z"},
            {"soll", "s Q l"},
            {"solve", "s Q l v"},
            {"solved", "s Q l v d"},
            {"solves", "s Q l v z"},
            {"som", "s Q m"},
            {"some", "s V m"},
            {"son", "s V n"},
            {"sones", "s @U n z"},
            {"song", "s O: N"},
            {"songs", "s O: N z"},
            {"song's", "s O: N z"},
            {"sonne", "s Q n"},
            {"sons", "s V n z"},
            {"son's", "s V n z"},
            {"soo", "s u:"},
            {"soon", "s u: n"},
            {"soong", "s u: N"},
            {"soot", "s U t"},
            {"soothe", "s u: D"},
            {"soothed", "s u: D d"},
            {"soothes", "s u: D z"},
            {"soots", "s U t s"},
            {"sop", "s Q p"},
            {"sope", "s @U p"},
            {"sops", "s Q p s"},
            {"sor", "s O@"},
            {"sorce", "s O@ s"},
            {"sore", "s O@"},
            {"sores", "s O@ z"},
            {"sorge", "s O@ dZ"},
            {"sort", "s O@ t"},
            {"sorts", "s O@ t s"},
            {"so's", "s @U z"},
            {"soss", "s O: s"},
            {"sought", "s O: t"},
            {"souk", "s u: k"},
            {"soul", "s @U l"},
            {"soule", "s aU l"},
            {"souls", "s @U l z"},
            {"sound", "s aU n d"},
            {"sounds", "s aU n d z"},
            {"sound's", "s aU n d z"},
            {"soup", "s u: p"},
            {"souped", "s u: p t"},
            {"soups", "s u: p s"},
            {"soup's", "s u: p s"},
            {"source", "s O@ s"},
            {"sous", "s u: z"},
            {"south", "s aU T"},
            {"souths", "s aU T s"},
            {"south's", "s aU T s"},
            {"sow", "s aU"},
            {"sowed", "s aU d"},
            {"sowle", "s @U l"},
            {"sowles", "s @U l z"},
            {"sown", "s @U n"},
            {"sows", "s @U z"},
            {"sox", "s Q k s"},
            {"soy", "s OI"},
            {"spa", "s p Q"},
            {"space", "s p eI s"},
            {"spaced", "s p eI s t"},
            {"spack", "s p { k"},
            {"spade", "s p eI d"},
            {"spades", "s p eI d z"},
            {"spahn", "s p Q n"},
            {"spaid", "s p eI d"},
            {"spain", "s p eI n"},
            {"spain's", "s p eI n z"},
            {"spak", "s p { k"},
            {"spake", "s p eI k"},
            {"spall", "s p O: l"},
            {"spam", "s p { m"},
            {"span", "s p { n"},
            {"spang", "s p { N"},
            {"spank", "s p { N k"},
            {"spanked", "s p { N k t"},
            {"spann", "s p { n"},
            {"spanned", "s p { n d"},
            {"spans", "s p { n z"},
            {"span's", "s p { n z"},
            {"spar", "s p Q@"},
            {"sparc", "s p Q@ k"},
            {"spare", "s p e@"},
            {"spared", "s p e@ d"},
            {"spares", "s p e@ z"},
            {"spark", "s p Q@ k"},
            {"sparked", "s p Q@ k t"},
            {"sparkes", "s p Q@ k s"},
            {"sparks", "s p Q@ k s"},
            {"sparr", "s p { r"},
            {"sparred", "s p Q@ d"},
            {"sparse", "s p Q@ s"},
            {"spas", "s p Q z"},
            {"spat", "s p { t"},
            {"spate", "s p eI t"},
            {"spates", "s p eI t s"},
            {"spath", "s p { T"},
            {"spats", "s p { t s"},
            {"spatz", "s p { t s"},
            {"spaw", "s p O:"},
            {"spawn", "s p Q n"},
            {"spawned", "s p O: n d"},
            {"spawns", "s p Q n z"},
            {"spayd", "s p eI d"},
            {"speak", "s p i: k"},
            {"speaks", "s p i: k s"},
            {"spear", "s p I@"},
            {"spears", "s p I@ z"},
            {"spec", "s p e k"},
            {"specht", "s p e k t"},
            {"speck", "s p e k"},
            {"specks", "s p e k s"},
            {"specs", "s p e k s"},
            {"sped", "s p e d"},
            {"speece", "s p i: s"},
            {"speech", "s p i: tS"},
            {"speed", "s p i: d"},
            {"speeds", "s p i: d z"},
            {"speer", "s p I@"},
            {"speight", "s p eI t"},
            {"speir", "s p I@"},
            {"speirs", "s p I@ z"},
            {"spell", "s p e l"},
            {"spelled", "s p e l d"},
            {"spells", "s p e l z"},
            {"speltz", "s p e l t s"},
            {"spence", "s p e n s"},
            {"spend", "s p e n d"},
            {"spends", "s p e n d z"},
            {"spent", "s p e n t"},
            {"sperm", "s p @r m"},
            {"sperms", "s p @r m z"},
            {"spew", "s p j u:"},
            {"spewed", "s p j u: d"},
            {"spews", "s p j u: z"},
            {"spey", "s p eI"},
            {"sphere", "s f I@"},
            {"spheres", "s f I@ z"},
            {"sphinx", "s f I N k s"},
            {"spic", "s p I k"},
            {"spice", "s p aI s"},
            {"spiced", "s p aI s t"},
            {"spie", "s p i:"},
            {"spied", "s p aI d"},
            {"spiel", "s p i: l"},
            {"spies", "s p aI z"},
            {"spight", "s p aI t"},
            {"spike", "s p aI k"},
            {"spiked", "s p aI k t"},
            {"spikes", "s p aI k s"},
            {"spill", "s p I l"},
            {"spilled", "s p I l d"},
            {"spills", "s p I l z"},
            {"spilt", "s p I l t"},
            {"spin", "s p I n"},
            {"spine", "s p aI n"},
            {"spines", "s p aI n z"},
            {"spink", "s p I N k"},
            {"spinks", "s p I N k s"},
            {"spins", "s p I n z"},
            {"spire", "s p aI r"},
            {"spires", "s p aI r z"},
            {"spit", "s p I t"},
            {"spite", "s p aI t"},
            {"spits", "s p I t s"},
            {"spitz", "s p I t s"},
            {"splash", "s p l { S"},
            {"splashed", "s p l { S t"},
            {"splat", "s p l { t"},
            {"spleen", "s p l i: n"},
            {"splice", "s p l aI s"},
            {"spliced", "s p l aI s t"},
            {"splint", "s p l I n t"},
            {"splints", "s p l I n t s"},
            {"split", "s p l I t"},
            {"splits", "s p l I t s"},
            {"splurge", "s p l @r dZ"},
            {"splurged", "s p l @r dZ d"},
            {"spock", "s p Q k"},
            {"spohr", "s p Q@"},
            {"spoil", "s p OI l"},
            {"spoiled", "s p OI l d"},
            {"spoils", "s p OI l z"},
            {"spoke", "s p @U k"},
            {"spokes", "s p @U k s"},
            {"spong", "s p O: N"},
            {"sponge", "s p V n dZ"},
            {"sponged", "s p V n dZ d"},
            {"spoof", "s p u: f"},
            {"spoofed", "s p u: f t"},
            {"spoofs", "s p u: f s"},
            {"spook", "s p u: k"},
            {"spooked", "s p u: k t"},
            {"spooks", "s p u: k s"},
            {"spool", "s p u: l"},
            {"spoon", "s p u: n"},
            {"spoons", "s p u: n z"},
            {"spoor", "s p U@"},
            {"spoor's", "s p U@ z"},
            {"spore", "s p O@"},
            {"spores", "s p O@ z"},
            {"sport", "s p O@ t"},
            {"sports", "s p O@ t s"},
            {"sport's", "s p O@ t s"},
            {"spot", "s p Q t"},
            {"spots", "s p Q t s"},
            {"spouse", "s p aU s"},
            {"spout", "s p aU t"},
            {"sprague", "s p r eI g"},
            {"sprague's", "s p r eI g z"},
            {"sprain", "s p r eI n"},
            {"sprained", "s p r eI n d"},
            {"sprains", "s p r eI n z"},
            {"sprang", "s p r { N"},
            {"spratt", "s p r { t"},
            {"sprawl", "s p r O: l"},
            {"sprawled", "s p r O: l d"},
            {"sprawls", "s p r O: l z"},
            {"spray", "s p r eI"},
            {"sprayed", "s p r eI d"},
            {"sprays", "s p r eI z"},
            {"spread", "s p r e d"},
            {"spreads", "s p r e d z"},
            {"spree", "s p r i:"},
            {"sprees", "s p r i: z"},
            {"spreng", "s p r e N"},
            {"sprigg", "s p r I g"},
            {"sprigged", "s p r I g d"},
            {"spring", "s p r I N"},
            {"springs", "s p r I N z"},
            {"spring's", "s p r I N z"},
            {"sprint", "s p r I n t"},
            {"sprints", "s p r I n t s"},
            {"sprint's", "s p r I n t s"},
            {"sprite", "s p r aI t"},
            {"sproat", "s p r @U t"},
            {"sprong", "s p r O: N"},
            {"sprott", "s p r Q t"},
            {"sproul", "s p r aU l"},
            {"sprout", "s p r aU t"},
            {"sprouts", "s p r aU t s"},
            {"spruce", "s p r u: s"},
            {"spruced", "s p r u: s t"},
            {"sprung", "s p r V N"},
            {"spry", "s p r aI"},
            {"spud", "s p V d"},
            {"spuds", "s p V d z"},
            {"spud's", "s p V d z"},
            {"spun", "s p V n"},
            {"spunk", "s p V N k"},
            {"spur", "s p @r"},
            {"spurge", "s p @r dZ"},
            {"spurn", "s p @r n"},
            {"spurned", "s p @r n d"},
            {"spurns", "s p @r n z"},
            {"spurred", "s p @r d"},
            {"spurs", "s p @r z"},
            {"spurt", "s p @r t"},
            {"spurts", "s p @r t s"},
            {"spy", "s p aI"},
            {"squad", "s k w Q d"},
            {"squads", "s k w Q d z"},
            {"squad's", "s k w Q d z"},
            {"squall", "s k w O: l"},
            {"squalls", "s k w O: l z"},
            {"square", "s k w e@"},
            {"squared", "s k w e@ d"},
            {"squares", "s k w e@ z"},
            {"squash", "s k w Q S"},
            {"squashed", "s k w Q S t"},
            {"squat", "s k w Q t"},
            {"squats", "s k w Q t s"},
            {"squawk", "s k w O: k"},
            {"squawks", "s k w O: k s"},
            {"squeak", "s k w i: k"},
            {"squeaked", "s k w i: k t"},
            {"squeaks", "s k w i: k s"},
            {"squeal", "s k w i: l"},
            {"squeals", "s k w i: l z"},
            {"squeeze", "s k w i: z"},
            {"squeezed", "s k w i: z d"},
            {"squelch", "s k w e l tS"},
            {"squelched", "s k w e l tS t"},
            {"squibb", "s k w I b"},
            {"squibb's", "s k w I b z"},
            {"squid", "s k w I d"},
            {"squint", "s k w I n t"},
            {"squire", "s k w aI r"},
            {"squirm", "s k w @r m"},
            {"squirms", "s k w @r m z"},
            {"squirt", "s k w @r t"},
            {"squirts", "s k w @r t s"},
            {"sram", "S r { m"},
            {"sri", "S r i:"},
            {"st", "s t r i: t"},
            {"sta", "s t V"},
            {"staab", "s t Q b"},
            {"staal", "s t Q l"},
            {"stab", "s t { b"},
            {"stabbed", "s t { b d"},
            {"stabs", "s t { b z"},
            {"stack", "s t { k"},
            {"stacked", "s t { k t"},
            {"stacks", "s t { k s"},
            {"stade", "s t eI d"},
            {"stadt", "s t { d t"},
            {"staff", "s t { f"},
            {"staffed", "s t { f t"},
            {"staffs", "s t { f s"},
            {"staff's", "s t { f s"},
            {"stag", "s t { g"},
            {"stage", "s t eI dZ"},
            {"staged", "s t eI dZ d"},
            {"stagg", "s t { g"},
            {"stahl", "s t Q l"},
            {"staid", "s t eI d"},
            {"stain", "s t eI n"},
            {"stained", "s t eI n d"},
            {"staines", "s t eI n z"},
            {"stains", "s t eI n z"},
            {"stair", "s t e@"},
            {"stairs", "s t e@ z"},
            {"stake", "s t eI k"},
            {"staked", "s t eI k t"},
            {"stakes", "s t eI k s"},
            {"stake's", "s t eI k s"},
            {"stale", "s t eI l"},
            {"stalk", "s t O: k"},
            {"stalked", "s t O: k t"},
            {"stalks", "s t O: k s"},
            {"stall", "s t O: l"},
            {"stalled", "s t O: l d"},
            {"stalls", "s t O: l z"},
            {"stam", "s t { m"},
            {"stamp", "s t { m p"},
            {"stamped", "s t { m p t"},
            {"stamps", "s t { m p s"},
            {"stan", "s t { n"},
            {"stance", "s t { n s"},
            {"stanch", "s t { n tS"},
            {"stanched", "s t { n tS t"},
            {"stand", "s t { n d"},
            {"stands", "s t { n d z"},
            {"stang", "s t { N"},
            {"stank", "s t { N k"},
            {"stan's", "s t { n z"},
            {"star", "s t Q@"},
            {"starch", "s t Q@ tS"},
            {"starched", "s t Q@ tS t"},
            {"stare", "s t e@"},
            {"stared", "s t e@ d"},
            {"stares", "s t e@ z"},
            {"stark", "s t Q@ k"},
            {"starke", "s t Q@ k"},
            {"starks", "s t Q@ k s"},
            {"stark's", "s t Q@ k s"},
            {"starn", "s t Q@ n"},
            {"starns", "s t Q@ n z"},
            {"starr", "s t Q@"},
            {"starred", "s t Q@ d"},
            {"starrs", "s t Q@ z"},
            {"starr's", "s t Q@ z"},
            {"stars", "s t Q@ z"},
            {"star's", "s t Q@ z"},
            {"start", "s t Q@ t"},
            {"starts", "s t Q@ t s"},
            {"starve", "s t Q@ v"},
            {"starved", "s t Q@ v d"},
            {"stash", "s t { S"},
            {"stashed", "s t { S t"},
            {"stat", "s t { t"},
            {"state", "s t eI t"},
            {"states", "s t eI t s"},
            {"state's", "s t eI t s"},
            {"states'", "s t eI t s"},
            {"stats", "s t { t s"},
            {"statz", "s t { t s"},
            {"staunch", "s t O: n tS"},
            {"staup", "s t O: p"},
            {"stave", "s t eI v"},
            {"staved", "s t eI v d"},
            {"staves", "s t eI v z"},
            {"stay", "s t eI"},
            {"stayed", "s t eI d"},
            {"stays", "s t eI z"},
            {"stead", "s t e d"},
            {"steak", "s t eI k"},
            {"steaks", "s t eI k s"},
            {"steal", "s t i: l"},
            {"steals", "s t i: l z"},
            {"stealth", "s t e l T"},
            {"steam", "s t i: m"},
            {"steamed", "s t i: m d"},
            {"steams", "s t i: m z"},
            {"stear", "s t I@"},
            {"stearn", "s t @r n"},
            {"stearne", "s t @r n"},
            {"stearns", "s t @r n z"},
            {"stears", "s t i: r z"},
            {"stech", "s t e k"},
            {"steck", "s t e k"},
            {"steed", "s t i: d"},
            {"steed's", "s t i: d z"},
            {"steel", "s t i: l"},
            {"steele", "s t i: l"},
            {"steels", "s t i: l z"},
            {"steel's", "s t i: l z"},
            {"steen", "s t i: n"},
            {"steep", "s t i: p"},
            {"steeped", "s t i: p t"},
            {"steer", "s t I@"},
            {"steere", "s t I@"},
            {"steered", "s t I@ d"},
            {"steers", "s t I@ z"},
            {"steeves", "s t i: v z"},
            {"steil", "s t i: l"},
            {"stein", "s t aI n"},
            {"steinke", "s t aI N k"},
            {"stein's", "s t aI n z"},
            {"stell", "s t e l"},
            {"stelle", "s t e l"},
            {"stem", "s t e m"},
            {"stemmed", "s t e m d"},
            {"stems", "s t e m z"},
            {"sten", "s t e n"},
            {"stench", "s t e n tS"},
            {"stent", "s t e n t"},
            {"step", "s t e p"},
            {"steppe", "s t e p"},
            {"stepped", "s t e p t"},
            {"steppes", "s t e p s"},
            {"steps", "s t e p s"},
            {"sterk", "s t @r k"},
            {"stern", "s t @r n"},
            {"sterne", "s t @r n"},
            {"sterns", "s t @r n z"},
            {"stern's", "s t @r n z"},
            {"stet", "s t e t"},
            {"steve", "s t i: v"},
            {"steve's", "s t i: v z"},
            {"stew", "s t u:"},
            {"stewed", "s t u: d"},
            {"stews", "s t u: z"},
            {"stew's", "s t u: z"},
            {"stich", "s t I tS"},
            {"stick", "s t I k"},
            {"sticks", "s t I k s"},
            {"stieve", "s t i: v"},
            {"stiff", "s t I f"},
            {"stiffed", "s t I f t"},
            {"stiffs", "s t I f s"},
            {"stig", "s t I g"},
            {"stile", "s t aI l"},
            {"stiles", "s t aI l z"},
            {"still", "s t I l"},
            {"stilled", "s t I l d"},
            {"stills", "s t I l z"},
            {"stilt", "s t I l t"},
            {"stilts", "s t I l t s"},
            {"stine", "s t aI n"},
            {"sting", "s t I N"},
            {"stings", "s t I N z"},
            {"stink", "s t I N k"},
            {"stinks", "s t I N k s"},
            {"stint", "s t I n t"},
            {"stints", "s t I n t s"},
            {"stipe", "s t aI p"},
            {"stipes", "s t aI p s"},
            {"stir", "s t @r"},
            {"stires", "s t aI r z"},
            {"stirred", "s t @r d"},
            {"stirs", "s t @r z"},
            {"stitch", "s t I tS"},
            {"stitched", "s t I tS t"},
            {"stites", "s t aI t s"},
            {"stith", "s t I T"},
            {"stock", "s t Q k"},
            {"stocked", "s t Q k t"},
            {"stocks", "s t Q k s"},
            {"stock's", "s t Q k s"},
            {"stoff", "s t O: f"},
            {"stoke", "s t @U k"},
            {"stoked", "s t @U k t"},
            {"stokes", "s t @U k s"},
            {"stol", "s t @U l"},
            {"stole", "s t @U l"},
            {"stoll", "s t @U l"},
            {"stomp", "s t Q m p"},
            {"stomped", "s t Q m p t"},
            {"stone", "s t @U n"},
            {"stoned", "s t @U n d"},
            {"stones", "s t @U n z"},
            {"stone's", "s t @U n z"},
            {"stong", "s t O: N"},
            {"stood", "s t U d"},
            {"stooge", "s t u: dZ"},
            {"stool", "s t u: l"},
            {"stools", "s t u: l z"},
            {"stoop", "s t u: p"},
            {"stooped", "s t u: p t"},
            {"stoops", "s t u: p s"},
            {"stop", "s t Q p"},
            {"stopped", "s t Q p t"},
            {"stops", "s t Q p s"},
            {"store", "s t O@"},
            {"stored", "s t O@ d"},
            {"stores", "s t O@ z"},
            {"store's", "s t O@ z"},
            {"stork", "s t O@ k"},
            {"storm", "s t O@ m"},
            {"stormed", "s t O@ m d"},
            {"storms", "s t O@ m z"},
            {"storm's", "s t O@ m z"},
            {"storr", "s t O@"},
            {"storrs", "s t O@ z"},
            {"stortz", "s t O@ t s"},
            {"storz", "s t O@ z"},
            {"stott", "s t Q t"},
            {"stoup", "s t u: p"},
            {"stout", "s t aU t"},
            {"stove", "s t @U v"},
            {"stoves", "s t @U v z"},
            {"stow", "s t @U"},
            {"stowe", "s t @U"},
            {"stowed", "s t @U d"},
            {"stoy", "s t OI"},
            {"strack", "s t r { k"},
            {"strafe", "s t r eI f"},
            {"straight", "s t r eI t"},
            {"straights", "s t r eI t s"},
            {"strain", "s t r eI n"},
            {"strained", "s t r eI n d"},
            {"strains", "s t r eI n z"},
            {"strait", "s t r eI t"},
            {"straits", "s t r eI t s"},
            {"stram", "s t r { m"},
            {"strand", "s t r { n d"},
            {"strands", "s t r { n d z"},
            {"strang", "s t r { N"},
            {"strange", "s t r eI n dZ"},
            {"strap", "s t r { p"},
            {"strapped", "s t r { p t"},
            {"straps", "s t r { p s"},
            {"strate", "s t r eI t"},
            {"straub", "s t r aU b"},
            {"straughn", "s t r O: n"},
            {"straus", "s t r aU s"},
            {"strauss", "s t r aU s"},
            {"straw", "s t r O:"},
            {"strawn", "s t r O: n"},
            {"straws", "s t r O: z"},
            {"stray", "s t r eI"},
            {"strayed", "s t r eI d"},
            {"strays", "s t r eI z"},
            {"streak", "s t r i: k"},
            {"streaked", "s t r i: k t"},
            {"streaks", "s t r i: k s"},
            {"stream", "s t r i: m"},
            {"streamed", "s t r i: m d"},
            {"streams", "s t r i: m z"},
            {"streck", "s t r e k"},
            {"streep", "s t r i: p"},
            {"street", "s t r i: t"},
            {"streets", "s t r i: t s"},
            {"street's", "s t r i: t s"},
            {"streight", "s t r eI t"},
            {"streit", "s t r aI t"},
            {"streng", "s t r e N"},
            {"strength", "s t r e N k T"},
            {"strengths", "s t r e N k T s"},
            {"strep", "s t r e p"},
            {"streps", "s t r e p s"},
            {"stress", "s t r e s"},
            {"stressed", "s t r e s t"},
            {"stretch", "s t r e tS"},
            {"stretched", "s t r e tS t"},
            {"strew", "s t r u:"},
            {"strewn", "s t r u: n"},
            {"strey", "s t r eI"},
            {"strick", "s t r I k"},
            {"strict", "s t r I k t"},
            {"stride", "s t r aI d"},
            {"strides", "s t r aI d z"},
            {"strife", "s t r aI f"},
            {"strifes", "s t r aI f s"},
            {"strike", "s t r aI k"},
            {"strikes", "s t r aI k s"},
            {"strine", "s t r aI n"},
            {"string", "s t r I N"},
            {"stringed", "s t r I N d"},
            {"strings", "s t r I N z"},
            {"strip", "s t r I p"},
            {"stripe", "s t r aI p"},
            {"striped", "s t r aI p t"},
            {"stripes", "s t r aI p s"},
            {"stripped", "s t r I p t"},
            {"strips", "s t r I p s"},
            {"strip's", "s t r I p s"},
            {"strive", "s t r aI v"},
            {"strives", "s t r aI v z"},
            {"strobe", "s t r @U b"},
            {"strode", "s t r @U d"},
            {"stroh", "s t r @U"},
            {"strohl", "s t r @U l"},
            {"stroke", "s t r @U k"},
            {"stroked", "s t r @U k t"},
            {"strokes", "s t r @U k s"},
            {"stroll", "s t r @U l"},
            {"strolled", "s t r @U l d"},
            {"strolls", "s t r @U l z"},
            {"strom", "s t r Q m"},
            {"strome", "s t r @U m"},
            {"strong", "s t r O: N"},
            {"stroot", "s t r u: t"},
            {"stroud", "s t r aU d"},
            {"stroup", "s t r u: p"},
            {"strout", "s t r aU t"},
            {"strove", "s t r @U v"},
            {"strow", "s t r @U"},
            {"strub", "s t r V b"},
            {"struck", "s t r V k"},
            {"strum", "s t r V m"},
            {"strums", "s t r V m z"},
            {"strung", "s t r V N"},
            {"strunk", "s t r V N k"},
            {"strunk's", "s t r V N k s"},
            {"strut", "s t r V t"},
            {"struts", "s t r V t s"},
            {"struve", "s t r u: v"},
            {"stu", "s t u:"},
            {"stub", "s t V b"},
            {"stubbed", "s t V b d"},
            {"stubbs", "s t V b z"},
            {"stubs", "s t V b z"},
            {"stuck", "s t V k"},
            {"stud", "s t V d"},
            {"stude", "s t u: d"},
            {"studs", "s t V d z"},
            {"stuff", "s t V f"},
            {"stuffed", "s t V f t"},
            {"stuffs", "s t V f s"},
            {"stull", "s t V l"},
            {"stultz", "s t V l t s"},
            {"stum", "s t V m"},
            {"stumm", "s t V m"},
            {"stump", "s t V m p"},
            {"stumped", "s t V m p t"},
            {"stumps", "s t V m p s"},
            {"stun", "s t V n"},
            {"stung", "s t V N"},
            {"stunk", "s t V N k"},
            {"stunned", "s t V n d"},
            {"stunt", "s t V n t"},
            {"stunts", "s t V n t s"},
            {"stupp", "s t V p"},
            {"sturm", "s t @r m"},
            {"sty", "s t aI"},
            {"style", "s t aI l"},
            {"styled", "s t aI l d"},
            {"styles", "s t aI l z"},
            {"styx", "s t I k s"},
            {"su", "s u:"},
            {"suave", "s w Q v"},
            {"sub", "s V b"},
            {"subbed", "s V b d"},
            {"subs", "s V b z"},
            {"sub's", "s V b z"},
            {"such", "s V tS"},
            {"suck", "s V k"},
            {"sucked", "s V k t"},
            {"sucks", "s V k s"},
            {"sudd", "s V d"},
            {"suds", "s V d z"},
            {"sue", "s u:"},
            {"sued", "s u: d"},
            {"suede", "s w eI d"},
            {"suedes", "s w eI d z"},
            {"sues", "s u: z"},
            {"sugg", "s V g"},
            {"suh", "s V"},
            {"suisse", "s w I s"},
            {"suit", "s u: t"},
            {"suite", "s w i: t"},
            {"suites", "s w i: t s"},
            {"suits", "s u: t s"},
            {"suit's", "s u: t s"},
            {"suk", "s V k"},
            {"sul", "s V l"},
            {"sulk", "s V l k"},
            {"sulked", "s V l k t"},
            {"sum", "s V m"},
            {"summed", "s V m d"},
            {"sump", "s V m p"},
            {"sums", "s V m z"},
            {"sun", "s V n"},
            {"sung", "s V N"},
            {"sung's", "s V N z"},
            {"sunk", "s V N k"},
            {"suns", "s V n z"},
            {"sun's", "s V n z"},
            {"sup", "s V p"},
            {"sur", "s @r"},
            {"sure", "S U@"},
            {"sures", "S U@ z"},
            {"surf", "s @r f"},
            {"surfed", "s @r f t"},
            {"surf's", "s @r f s"},
            {"surge", "s @r dZ"},
            {"surged", "s @r dZ d"},
            {"sus", "s V s"},
            {"suss", "s V s"},
            {"svelte", "s v e l t"},
            {"sven", "s v e n"},
            {"swab", "s w Q b"},
            {"swabs", "s w Q b z"},
            {"swails", "s w eI l z"},
            {"swain", "s w eI n"},
            {"swaine", "s w eI n"},
            {"swales", "s w eI l z"},
            {"swam", "s w { m"},
            {"swamp", "s w Q m p"},
            {"swamped", "s w Q m p t"},
            {"swamps", "s w Q m p s"},
            {"swan", "s w Q n"},
            {"swank", "s w { N k"},
            {"swann", "s w Q n"},
            {"swann's", "s w Q n z"},
            {"swans", "s w Q n z"},
            {"swap", "s w Q p"},
            {"swape", "s w eI p"},
            {"swapped", "s w Q p t"},
            {"swaps", "s w Q p s"},
            {"sward", "s w Q@ d"},
            {"swarm", "s w O@ m"},
            {"swarmed", "s w O@ m d"},
            {"swarms", "s w O@ m z"},
            {"swart", "s w O@ t"},
            {"swarts", "s w O@ t s"},
            {"swartz", "s w O@ t s"},
            {"swat", "s w Q t"},
            {"swatch", "s w Q tS"},
            {"swath", "s w Q T"},
            {"swathe", "s w Q D"},
            {"swathed", "s w Q D d"},
            {"sway", "s w eI"},
            {"swayed", "s w eI d"},
            {"swayne", "s w eI n"},
            {"sways", "s w eI z"},
            {"swear", "s w e@"},
            {"swears", "s w e@ z"},
            {"sweat", "s w e t"},
            {"sweats", "s w e t s"},
            {"sweatt", "s w i: t"},
            {"swed", "s w e d"},
            {"swede", "s w i: d"},
            {"swedes", "s w i: d z"},
            {"sweep", "s w i: p"},
            {"sweeps", "s w i: p s"},
            {"sweet", "s w i: t"},
            {"sweets", "s w i: t s"},
            {"swell", "s w e l"},
            {"swelled", "s w e l d"},
            {"swells", "s w e l z"},
            {"swept", "s w e p t"},
            {"swerve", "s w @r v"},
            {"swerved", "s w @r v d"},
            {"swick", "s w I k"},
            {"swift", "s w I f t"},
            {"swifts", "s w I f t s"},
            {"swift's", "s w I f t s"},
            {"swig", "s w I g"},
            {"swill", "s w I l"},
            {"swim", "s w I m"},
            {"swims", "s w I m z"},
            {"swine", "s w aI n"},
            {"swing", "s w I N"},
            {"swings", "s w I N z"},
            {"swink", "s w I N k"},
            {"swipe", "s w aI p"},
            {"swiped", "s w aI p t"},
            {"swipes", "s w aI p s"},
            {"swire", "s w aI r"},
            {"swirl", "s w @r l"},
            {"swirled", "s w @r l d"},
            {"swirls", "s w @r l z"},
            {"swish", "s w I S"},
            {"swished", "s w I S t"},
            {"swiss", "s w I s"},
            {"switch", "s w I tS"},
            {"switched", "s w I tS t"},
            {"swoon", "s w u: n"},
            {"swooned", "s w u: n d"},
            {"swoop", "s w u: p"},
            {"swoope", "s w u: p"},
            {"swooped", "s w u: p t"},
            {"swoops", "s w u: p s"},
            {"swoosh", "s w u: S"},
            {"swooshed", "s w u: S t"},
            {"swope", "s w @U p"},
            {"swor", "s w @r"},
            {"sword", "s O@ d"},
            {"swords", "s O@ d z"},
            {"swore", "s w O@"},
            {"sworn", "s w O@ n"},
            {"swum", "s w V m"},
            {"swung", "s w V N"},
            {"sy", "s aI"},
            {"syd", "s I d"},
            {"syed", "s aI d"},
            {"sykes", "s aI k s"},
            {"sym", "s I m"},
            {"syn", "s I n"},
            {"sync", "s I N k"},
            {"synch", "s I n tS"},
            {"t", "th i:"},
            {"t.", "th i:"},
            {"ta", "th Q"},
            {"tab", "th { b"},
            {"tabb", "th { b"},
            {"tabs", "th { b z"},
            {"tac", "th { k"},
            {"tack", "th { k"},
            {"tacked", "th { k t"},
            {"tacks", "th { k s"},
            {"tact", "th { k t"},
            {"tad", "th { d"},
            {"tadd", "th { d"},
            {"tae", "th eI"},
            {"taft", "th { f t"},
            {"taft's", "th { f t s"},
            {"tag", "th { g"},
            {"tagged", "th { g d"},
            {"tags", "th { g z"},
            {"tai", "th aI"},
            {"tail", "th eI l"},
            {"tailed", "th eI l d"},
            {"tails", "th eI l z"},
            {"taint", "th eI n t"},
            {"taints", "th eI n t s"},
            {"tait", "th eI t"},
            {"taite", "th eI t"},
            {"taj", "th Q Z"},
            {"tak", "th { k"},
            {"take", "th eI k"},
            {"takes", "th eI k s"},
            {"tal", "th Q l"},
            {"talc", "th { l k"},
            {"talcs", "th { l k s"},
            {"tale", "th eI l"},
            {"tales", "th eI l z"},
            {"talk", "th O: k"},
            {"talked", "th O: k t"},
            {"talks", "th O: k s"},
            {"tall", "th O: l"},
            {"tam", "th { m"},
            {"tame", "th eI m"},
            {"tamed", "th eI m d"},
            {"tames", "th eI m z"},
            {"tamp", "th { m p"},
            {"tamps", "th { m p s"},
            {"tan", "th { n"},
            {"tang", "th { N"},
            {"t'ang", "th { N"},
            {"tangs", "th { N z"},
            {"tank", "th { N k"},
            {"tanked", "th { N k t"},
            {"tanks", "th { N k s"},
            {"tank's", "th { N k s"},
            {"tann", "th { n"},
            {"tanned", "th { n d"},
            {"tans", "th { n z"},
            {"tant", "th { n t"},
            {"tao", "th aU"},
            {"taos", "th aU z"},
            {"tap", "th { p"},
            {"tape", "th eI p"},
            {"taped", "th eI p t"},
            {"tapes", "th eI p s"},
            {"tape's", "th eI p s"},
            {"tapped", "th { p t"},
            {"taps", "th { p s"},
            {"tar", "th Q@"},
            {"tarp", "th Q@ p"},
            {"tarps", "th Q@ p s"},
            {"tarr", "th Q@"},
            {"tarred", "th Q@ d"},
            {"tart", "th Q@ t"},
            {"tarte", "th Q@ t"},
            {"tarts", "th Q@ t s"},
            {"tash", "th { S"},
            {"task", "th { s k"},
            {"tasked", "th { s k t"},
            {"tasks", "th { s k s"},
            {"tass", "th { s"},
            {"taste", "th eI s t"},
            {"tastes", "th eI s t s"},
            {"tat", "th { t"},
            {"tate", "th eI t"},
            {"tate's", "th eI t s"},
            {"taub", "th aU b"},
            {"taube", "th O: b"},
            {"taubes", "th aU b z"},
            {"taught", "th O: t"},
            {"taunt", "th O: n t"},
            {"taunts", "th O: n t s"},
            {"taut", "th O: t"},
            {"tawes", "th O: z"},
            {"tax", "th { k s"},
            {"taxed", "th { k s t"},
            {"tay", "th eI"},
            {"tays", "th eI z"},
            {"te", "th i:"},
            {"tea", "th i:"},
            {"teach", "th i: tS"},
            {"teague", "th i: g"},
            {"teak", "th i: k"},
            {"teal", "th i: l"},
            {"team", "th i: m"},
            {"teamed", "th i: m d"},
            {"teams", "th i: m z"},
            {"team's", "th i: m z"},
            {"tear", "th e@"},
            {"tears", "th e@ z"},
            {"teas", "th i: z"},
            {"tease", "th i: z"},
            {"teased", "th i: z d"},
            {"teat", "th i: t"},
            {"teats", "th i: t s"},
            {"tec", "th e k"},
            {"tech", "th e k"},
            {"techs", "th e k s"},
            {"tech's", "th e k s"},
            {"teck", "th e k"},
            {"ted", "th e d"},
            {"ted's", "th e d z"},
            {"tee", "th i:"},
            {"teed", "th i: d"},
            {"teel", "th i: l"},
            {"teem", "th i: m"},
            {"teems", "th i: m z"},
            {"teen", "th i: n"},
            {"teens", "th i: n z"},
            {"teer", "th I@"},
            {"tees", "th i: z"},
            {"teet", "th i: t"},
            {"teeth", "th i: T"},
            {"teethe", "th i: D"},
            {"tefft", "th e f t"},
            {"tel", "th e l"},
            {"tell", "th e l"},
            {"tells", "th e l z"},
            {"temp", "th e m p"},
            {"temps", "th e m p s"},
            {"tempt", "th e m p t"},
            {"tempts", "th e m p t s"},
            {"ten", "th e n"},
            {"tench", "th e n tS"},
            {"tend", "th e n d"},
            {"tends", "th e n d z"},
            {"teng", "th e N"},
            {"tens", "th e n z"},
            {"ten's", "th e n z"},
            {"tense", "th e n s"},
            {"tent", "th e n t"},
            {"tenth", "th e n T"},
            {"tenths", "th e n T s"},
            {"tents", "th e n t s"},
            {"tepe", "th i: p"},
            {"ter", "th @r"},
            {"term", "th @r m"},
            {"termed", "th @r m d"},
            {"terms", "th @r m z"},
            {"term's", "th @r m z"},
            {"ternes", "th @r n z"},
            {"terns", "th @r n z"},
            {"terse", "th @r s"},
            {"tess", "th e s"},
            {"test", "th e s t"},
            {"tests", "th e s t s"},
            {"test's", "th e s t s"},
            {"tet", "th e t"},
            {"tete", "th e t"},
            {"tew", "tS u:"},
            {"tews", "tS u: z"},
            {"tex", "th e k s"},
            {"text", "th e k s t"},
            {"texts", "th e k s t s"},
            {"thach", "T { tS"},
            {"thad", "T { d"},
            {"thai", "th aI"},
            {"thain", "T eI n"},
            {"thaine", "T eI n"},
            {"thais", "th aI z"},
            {"tham", "T { m"},
            {"thames", "th e m z"},
            {"than", "D { n"},
            {"thane", "T eI n"},
            {"thanh", "T { n"},
            {"thank", "T { N k"},
            {"thanked", "T { N k t"},
            {"thanks", "T { N k s"},
            {"thar", "T Q@"},
            {"tharp", "T Q@ p"},
            {"that", "D { t"},
            {"thatch", "T { tS"},
            {"thatched", "T { tS t"},
            {"that's", "D { t s"},
            {"thaw", "T O:"},
            {"thawed", "T O: d"},
            {"thaws", "T O: z"},
            {"thayne", "T eI n"},
            {"the", "D @"},
            {"theall", "T i: l"},
            {"thebes", "T i: b z"},
            {"thee", "D i:"},
            {"theft", "T e f t"},
            {"thefts", "T e f t s"},
            {"thein", "T aI n"},
            {"their", "D e@"},
            {"theirs", "D e@ z"},
            {"theis", "D aI z"},
            {"theiss", "T aI s"},
            {"them", "D e m"},
            {"theme", "T i: m"},
            {"themed", "T i: m d"},
            {"themes", "T i: m z"},
            {"then", "D e n"},
            {"thence", "D e n s"},
            {"there", "D e@"},
            {"there'd", "D e@ d"},
            {"there's", "D e@ z"},
            {"therm", "T @r m"},
            {"these", "D i: z"},
            {"thew", "T u:"},
            {"they", "D eI"},
            {"they'd", "D eI d"},
            {"they'll", "D eI l"},
            {"they're", "D e@"},
            {"they've", "D eI v"},
            {"thi", "T i:"},
            {"thick", "T I k"},
            {"thief", "T i: f"},
            {"thief's", "T i: f s"},
            {"thieves", "T i: v z"},
            {"thieves'", "T i: v z"},
            {"thigh", "T aI"},
            {"thighs", "T aI z"},
            {"thill", "T I l"},
            {"thin", "T I n"},
            {"thine", "D aI n"},
            {"thing", "T I N"},
            {"things", "T I N z"},
            {"thing's", "T I N z"},
            {"think", "T I N k"},
            {"thinks", "T I N k s"},
            {"thinned", "T I n d"},
            {"thins", "T I n z"},
            {"third", "T @r d"},
            {"thirds", "T @r d z"},
            {"thirst", "T @r s t"},
            {"this", "D I s"},
            {"tho", "D @U"},
            {"thole", "T @U l"},
            {"thom", "th Q m"},
            {"thome", "T @U m"},
            {"thon", "T Q n"},
            {"thone", "T @U n"},
            {"thong", "T O: N"},
            {"thor", "T O@"},
            {"thorn", "T O@ n"},
            {"thorne", "T O@ n"},
            {"thorns", "T O@ n z"},
            {"thorn's", "T O@ n z"},
            {"thorp", "T O@ p"},
            {"thorpe", "T O@ p"},
            {"those", "D @U z"},
            {"thou", "D aU"},
            {"though", "D @U"},
            {"thought", "T O: t"},
            {"thoughts", "T O: t s"},
            {"thous", "D aU z"},
            {"thrall", "T r O: l"},
            {"thrash", "T r { S"},
            {"thrashed", "T r { S t"},
            {"thread", "T r e d"},
            {"threads", "T r e d z"},
            {"threat", "T r e t"},
            {"threats", "T r e t s"},
            {"three", "T r i:"},
            {"threes", "T r i: z"},
            {"three's", "T r i: z"},
            {"thresh", "T r e S"},
            {"threw", "T r u:"},
            {"thrice", "T r aI s"},
            {"thrift", "T r I f t"},
            {"thrifts", "T r I f t s"},
            {"thrift's", "T r I f t s"},
            {"thrill", "T r I l"},
            {"thrilled", "T r I l d"},
            {"thrills", "T r I l z"},
            {"thrips", "T r I p s"},
            {"thrive", "T r aI v"},
            {"thrived", "T r aI v d"},
            {"thrives", "T r aI v z"},
            {"throat", "T r @U t"},
            {"throats", "T r @U t s"},
            {"throb", "T r Q b"},
            {"throes", "T r @U z"},
            {"throne", "T r @U n"},
            {"throng", "T r O: N"},
            {"thronged", "T r O: N d"},
            {"throngs", "T r O: N z"},
            {"throop", "T r u: p"},
            {"through", "T r u:"},
            {"throve", "T r @U v"},
            {"throw", "T r @U"},
            {"thrown", "T r @U n"},
            {"throws", "T r @U z"},
            {"thru", "T r u:"},
            {"thrush", "T r V S"},
            {"thrust", "T r V s t"},
            {"thrusts", "T r V s t s"},
            {"thud", "T V d"},
            {"thug", "T V g"},
            {"thugs", "T V g z"},
            {"thumb", "T V m"},
            {"thumbed", "T V m d"},
            {"thumbs", "T V m z"},
            {"thump", "T V m p"},
            {"thumped", "T V m p t"},
            {"thumps", "T V m p s"},
            {"thun", "T V n"},
            {"thunk", "T V n k"},
            {"thur", "D @r"},
            {"thurm", "T @r m"},
            {"thurn", "T @r n"},
            {"thus", "D V s"},
            {"thwart", "T w O@ t"},
            {"thwarts", "T w O@ t s"},
            {"thwing", "T w I N"},
            {"thy", "D aI"},
            {"thyme", "T aI m"},
            {"ti", "th i:"},
            {"tian", "th j Q n"},
            {"tibbs", "th I b z"},
            {"tic", "th I k"},
            {"tice", "th aI s"},
            {"tick", "th I k"},
            {"ticked", "th I k t"},
            {"ticks", "th I k s"},
            {"tics", "th I k s"},
            {"tide", "th aI d"},
            {"tides", "th aI d z"},
            {"tie", "th aI"},
            {"tied", "th aI d"},
            {"tien", "th j e n"},
            {"tier", "th i: r"},
            {"tierce", "th i: r s"},
            {"tiered", "th i: r d"},
            {"tiers", "th i: r z"},
            {"ties", "th aI z"},
            {"tiff", "th I f"},
            {"tift", "th I f t"},
            {"tig", "th I g"},
            {"tigges", "th I g z"},
            {"tight", "th aI t"},
            {"tights", "th aI t s"},
            {"tike", "th aI k"},
            {"tikes", "th aI k s"},
            {"til", "th I l"},
            {"tile", "th aI l"},
            {"tiled", "th aI l d"},
            {"tiles", "th aI l z"},
            {"till", "th I l"},
            {"tilt", "th I l t"},
            {"tilth", "th I l T"},
            {"tilts", "th I l t s"},
            {"tim", "th I m"},
            {"time", "th aI m"},
            {"timed", "th aI m d"},
            {"times", "th aI m z"},
            {"time's", "th aI m z"},
            {"timms", "th I m z"},
            {"tims", "th I m z"},
            {"tim's", "th I m z"},
            {"tin", "th I n"},
            {"tine", "th aI n"},
            {"tines", "th aI n z"},
            {"ting", "th I N"},
            {"tinge", "th I n dZ"},
            {"tinged", "th I N d"},
            {"tins", "th I n z"},
            {"tint", "th I n t"},
            {"tints", "th I n t s"},
            {"tip", "th I p"},
            {"tipp", "th I p"},
            {"tipped", "th I p t"},
            {"tips", "th I p s"},
            {"tis", "th I z"},
            {"tish", "th I S"},
            {"tit", "th I t"},
            {"tithe", "th aI D"},
            {"to", "th u:"},
            {"toad", "th @U d"},
            {"toads", "th @U d z"},
            {"toast", "th @U s t"},
            {"toasts", "th @U s t s"},
            {"tobe", "th @U b"},
            {"tock", "th Q k"},
            {"tod", "th Q d"},
            {"todd", "th Q d"},
            {"todd's", "th Q d z"},
            {"todt", "th Q d t"},
            {"toe", "th @U"},
            {"toed", "th @U d"},
            {"toes", "th @U z"},
            {"toft", "th Q f t"},
            {"tofte", "th @U f t"},
            {"togs", "th Q g z"},
            {"toil", "th OI l"},
            {"toiled", "th OI l d"},
            {"toils", "th OI l z"},
            {"told", "th @U l d"},
            {"tole", "th @U l"},
            {"toles", "th @U l z"},
            {"toll", "th @U l"},
            {"tolled", "th @U l d"},
            {"tolls", "th @U l z"},
            {"tom", "th Q m"},
            {"tomb", "th u: m"},
            {"tombs", "th u: m z"},
            {"tome", "th @U m"},
            {"tomes", "th @U m z"},
            {"toms", "th Q m z"},
            {"tom's", "th Q m z"},
            {"ton", "th V n"},
            {"tone", "th @U n"},
            {"toned", "th @U n d"},
            {"tones", "th @U n z"},
            {"tong", "th O: N"},
            {"tongs", "th Q N z"},
            {"tongue", "th V N"},
            {"tongued", "th V N d"},
            {"tongues", "th V N z"},
            {"tonk", "th O: N k"},
            {"tonks", "th Q N k s"},
            {"tonn", "th Q n"},
            {"tonne", "th V n"},
            {"tonnes", "th V n z"},
            {"tons", "th V n z"},
            {"too", "th u:"},
            {"took", "th U k"},
            {"tooke", "th U k"},
            {"tool", "th u: l"},
            {"toole", "th u: l"},
            {"tooled", "th u: l d"},
            {"tools", "th u: l z"},
            {"toombs", "th u: m z"},
            {"toon", "th u: n"},
            {"toone", "th u: n"},
            {"toot", "th u: t"},
            {"tooth", "th u: T"},
            {"toothed", "th u: T t"},
            {"toots", "th u: t s"},
            {"top", "th Q p"},
            {"tope", "th @U p"},
            {"topped", "th Q p t"},
            {"tops", "th Q p s"},
            {"tor", "th O@"},
            {"torch", "th O@ tS"},
            {"torched", "th O@ tS t"},
            {"tore", "th O@"},
            {"torn", "th O@ n"},
            {"torp", "th O@ p"},
            {"torque", "th O@ k"},
            {"torr", "th O@"},
            {"torre", "th O@"},
            {"tort", "th O@ t"},
            {"torte", "th O@ t"},
            {"torts", "th O@ t s"},
            {"tosch", "th O: S"},
            {"tosh", "th Q S"},
            {"toss", "th O: s"},
            {"tossed", "th O: s t"},
            {"tot", "th Q t"},
            {"tote", "th @U t"},
            {"totes", "th @U t s"},
            {"toth", "th Q T"},
            {"tots", "th Q t s"},
            {"touch", "th V tS"},
            {"touche", "th u: S"},
            {"touched", "th V tS t"},
            {"tough", "th V f"},
            {"toughed", "th V f t"},
            {"toughs", "th V f s"},
            {"tour", "th U@"},
            {"toured", "th U@ d"},
            {"tours", "th U@ z"},
            {"tour's", "th U@ z"},
            {"tout", "th aU t"},
            {"touts", "th aU t s"},
            {"tow", "th @U"},
            {"towed", "th @U d"},
            {"town", "th aU n"},
            {"towne", "th aU n"},
            {"townes", "th aU n z"},
            {"towns", "th aU n z"},
            {"town's", "th aU n z"},
            {"tows", "th @U z"},
            {"toy", "th OI"},
            {"toye", "th OI"},
            {"toyed", "th OI d"},
            {"toys", "th OI z"},
            {"toy's", "th OI z"},
            {"trac", "th r { k"},
            {"trace", "th r eI s"},
            {"traced", "th r eI s t"},
            {"track", "th r { k"},
            {"tracked", "th r { k t"},
            {"tracks", "th r { k s"},
            {"track's", "th r { k s"},
            {"tract", "th r { k t"},
            {"tracts", "th r { k t s"},
            {"trade", "th r eI d"},
            {"trades", "th r eI d z"},
            {"trade's", "th r eI d z"},
            {"traer", "th r e@"},
            {"trail", "th r eI l"},
            {"trailed", "th r eI l d"},
            {"trails", "th r eI l z"},
            {"train", "th r eI n"},
            {"trained", "th r eI n d"},
            {"trains", "th r eI n z"},
            {"traipse", "th r eI p s"},
            {"trait", "th r eI t"},
            {"traits", "th r eI t s"},
            {"tram", "th r { m"},
            {"tramp", "th r { m p"},
            {"tramps", "th r { m p z"},
            {"trams", "th r { m z"},
            {"tran", "th r { n"},
            {"trance", "th r { n s"},
            {"tranche", "th r { n tS"},
            {"trang", "th r { N"},
            {"trans", "th r { n z"},
            {"trant", "th r { n t"},
            {"trap", "th r { p"},
            {"trappe", "th r { p"},
            {"trapped", "th r { p t"},
            {"traps", "th r { p s"},
            {"trash", "th r { S"},
            {"trashed", "th r { S t"},
            {"trask", "th r { s k"},
            {"tray", "th r eI"},
            {"trays", "th r eI z"},
            {"tread", "th r e d"},
            {"treads", "th r e d z"},
            {"treat", "th r i: t"},
            {"treats", "th r i: t s"},
            {"tree", "th r i:"},
            {"treece", "th r i: s"},
            {"treen", "th r i: n"},
            {"trees", "th r i: z"},
            {"trek", "th r e k"},
            {"trekked", "th r e k t"},
            {"treks", "th r e k s"},
            {"trench", "th r e n tS"},
            {"trend", "th r e n d"},
            {"trends", "th r e n d z"},
            {"trent", "th r e n t"},
            {"trent's", "th r e n t s"},
            {"tress", "th r e s"},
            {"trest", "th r e s t"},
            {"trew", "th r u:"},
            {"trey", "th r eI"},
            {"tri", "th r aI"},
            {"trib", "th r I b"},
            {"tribe", "th r aI b"},
            {"tribes", "th r aI b z"},
            {"tribe's", "th r aI b z"},
            {"trice", "th r aI s"},
            {"trick", "th r I k"},
            {"tricked", "th r I k t"},
            {"tricks", "th r I k s"},
            {"tried", "th r aI d"},
            {"tries", "th r aI z"},
            {"trieste", "th r i: s t"},
            {"trig", "th r I g"},
            {"trill", "th r I l"},
            {"trim", "th r I m"},
            {"trimmed", "th r I m d"},
            {"trims", "th r I m z"},
            {"trine", "th r aI n"},
            {"trip", "th r I p"},
            {"tripe", "th r aI p"},
            {"tripp", "th r I p"},
            {"trippe", "th r I p"},
            {"tripped", "th r I p t"},
            {"trips", "th r I p s"},
            {"trip's", "th r I p s"},
            {"trish", "th r I S"},
            {"trite", "th r aI t"},
            {"trites", "th r aI t s"},
            {"troche", "th r Q tS"},
            {"trod", "th r Q d"},
            {"trois", "th w Q"},
            {"troll", "th r @U l"},
            {"tromp", "th r Q m p"},
            {"tron", "th r Q n"},
            {"trone", "th r @U n"},
            {"troon", "th r u: n"},
            {"troop", "th r u: p"},
            {"trooped", "th r u: p t"},
            {"troops", "th r u: p s"},
            {"troop's", "th r u: p s"},
            {"trot", "th r Q t"},
            {"troth", "th r @U T"},
            {"trots", "th r Q t s"},
            {"trough", "th r O: f"},
            {"troughs", "th r O: f s"},
            {"trounce", "th r aU n s"},
            {"trounced", "th r aU n s t"},
            {"troup", "th r u: p"},
            {"troupe", "th r u: p"},
            {"troupes", "th r u: p s"},
            {"troupe's", "th r u: p s"},
            {"trout", "th r aU t"},
            {"trout's", "th r aU t s"},
            {"trove", "th r @U v"},
            {"trow", "th r @U"},
            {"troy", "th r OI"},
            {"truce", "th r u: s"},
            {"truck", "th r V k"},
            {"trucked", "th r V k t"},
            {"trucks", "th r V k s"},
            {"truck's", "th r V k s"},
            {"trude", "th r u: d"},
            {"trudge", "th r V dZ"},
            {"trudged", "th r V dZ d"},
            {"true", "th r u:"},
            {"truitt", "th r u: t"},
            {"trull", "th r V l"},
            {"trump", "th r V m p"},
            {"trumped", "th r V m p t"},
            {"trumps", "th r V m p s"},
            {"trump's", "th r V m p s"},
            {"trunk", "th r V N k"},
            {"trunks", "th r V N k s"},
            {"truong", "th r O: N"},
            {"truss", "th r V s"},
            {"trust", "th r V s t"},
            {"trusts", "th r V s t s"},
            {"trust's", "th r V s t s"},
            {"truth", "th r u: T"},
            {"truths", "th r u: T s"},
            {"try", "th r aI"},
            {"tryst", "th r aI s t"},
            {"t's", "th i: z"},
            {"tsai", "th s aI"},
            {"tsai's", "th s aI z"},
            {"tsar", "z Q@"},
            {"tse", "th s i:"},
            {"tseng", "th s e N"},
            {"tso", "th s @U"},
            {"tu", "th u:"},
            {"tub", "th V b"},
            {"tubb", "th V b"},
            {"tube", "th u: b"},
            {"tubes", "th u: b z"},
            {"tubs", "th V b z"},
            {"tuck", "th V k"},
            {"tucked", "th V k t"},
            {"tucks", "th V k s"},
            {"tue", "th u:"},
            {"tuel", "th u: l"},
            {"tuff", "th V f"},
            {"tuft", "th V f t"},
            {"tufts", "th V f t s"},
            {"tug", "th V g"},
            {"tugged", "th V g d"},
            {"tugs", "th V g z"},
            {"tull", "th V l"},
            {"tums", "th V m z"},
            {"tune", "th u: n"},
            {"tuned", "th u: n d"},
            {"tunes", "th u: n z"},
            {"tung", "th V N"},
            {"tung's", "th V N z"},
            {"ture", "th U@"},
            {"turf", "th @r f"},
            {"turk", "th @r k"},
            {"turks", "th @r k s"},
            {"turn", "th @r n"},
            {"turned", "th @r n d"},
            {"turns", "th @r n z"},
            {"tush", "th U S"},
            {"tusk", "th V s k"},
            {"tusks", "th V s k s"},
            {"tut", "th V t"},
            {"tutt", "th V t"},
            {"tux", "th V k s"},
            {"twain", "th w eI n"},
            {"twain's", "th w eI n z"},
            {"twang", "th w Q N"},
            {"twas", "th w V z"},
            {"tweak", "th w i: k"},
            {"tweaked", "th w i: k t"},
            {"tweaks", "th w i: k s"},
            {"tweed", "th w i: d"},
            {"tweet", "th w i: t"},
            {"twelfth", "th w e l f T"},
            {"twelve", "th w e l v"},
            {"twice", "th w aI s"},
            {"twig", "th w I g"},
            {"twigged", "th w I g d"},
            {"twigs", "th w I g z"},
            {"twill", "th w I l"},
            {"twin", "th w I n"},
            {"twine", "th w aI n"},
            {"twined", "th w aI n d"},
            {"twinge", "th w I n dZ"},
            {"twins", "th w I n z"},
            {"twirl", "th w @r l"},
            {"twirled", "th w @r l d"},
            {"twirls", "th w @r l z"},
            {"twist", "th w I s t"},
            {"twists", "th w I s t s"},
            {"twitch", "th w I tS"},
            {"twitched", "th w I tS t"},
            {"twite", "th w aI t"},
            {"two", "th u:"},
            {"twos", "th u: z"},
            {"two's", "th u: z"},
            {"ty", "th aI"},
            {"tye", "th aI"},
            {"tyke", "th aI k"},
            {"tyne", "th aI n"},
            {"tynes", "th aI n z"},
            {"type", "th aI p"},
            {"typed", "th aI p t"},
            {"types", "th aI p s"},
            {"tyre", "th aI r"},
            {"u", "j u:"},
            {"u.", "I k"},
            {"ude", "u: d"},
            {"ugh", "V g"},
            {"uh", "V"},
            {"uhde", "V d"},
            {"ul", "V l"},
            {"ulm", "V l m"},
            {"um", "V m"},
            {"umm", "V m"},
            {"un", "V n"},
            {"und", "V n d"},
            {"ung", "V N"},
            {"up", "V p"},
            {"upped", "V p t"},
            {"ups", "V p s"},
            {"ur", "@r"},
            {"ure", "j u: r"},
            {"urge", "@r dZ"},
            {"urged", "@r dZ d"},
            {"urn", "@r n"},
            {"us", "V s"},
            {"u's", "j u: z"},
            {"use", "j u: s"},
            {"used", "j u: z d"},
            {"v", "v i:"},
            {"v.", "v i:"},
            {"vaal", "v Q l"},
            {"vague", "v eI g"},
            {"vail", "v eI l"},
            {"vain", "v eI n"},
            {"val", "v { l"},
            {"vale", "v eI l"},
            {"vales", "v eI l z"},
            {"valle", "v eI l"},
            {"valve", "v { l v"},
            {"valves", "v { l v z"},
            {"vamp", "v { m p"},
            {"van", "v { n"},
            {"vance", "v { n s"},
            {"vane", "v eI n"},
            {"vanes", "v eI n z"},
            {"vang", "v { N"},
            {"vans", "v { n z"},
            {"vars", "v Q@ z"},
            {"varves", "v Q@ v z"},
            {"vase", "v eI s"},
            {"vass", "v { s"},
            {"vast", "v { s t"},
            {"vat", "v { t"},
            {"vats", "v { t s"},
            {"vaughan", "v O: n"},
            {"vaughan's", "v O: n z"},
            {"vaughn", "v O: n"},
            {"vaughn's", "v O: n z"},
            {"vault", "v O: l t"},
            {"vaults", "v O: l t s"},
            {"vax", "v { k s"},
            {"ve", "v i:"},
            {"veal", "v i: l"},
            {"veale", "v i: l"},
            {"veals", "v i: l z"},
            {"vee", "v i:"},
            {"veen", "v i: n"},
            {"veep", "v i: p"},
            {"veer", "v I@"},
            {"veered", "v I@ d"},
            {"veers", "v I@ z"},
            {"vees", "v i: z"},
            {"veil", "v eI l"},
            {"veiled", "v eI l d"},
            {"veils", "v eI l z"},
            {"vein", "v eI n"},
            {"veins", "v eI n z"},
            {"veit", "v i: t"},
            {"velte", "v e l t"},
            {"venn", "v e n"},
            {"vent", "v e n t"},
            {"vents", "v e n t s"},
            {"verb", "v @r b"},
            {"verbs", "v @r b z"},
            {"vere", "v I@"},
            {"verge", "v @r dZ"},
            {"vern", "v @r n"},
            {"verne", "v @r n"},
            {"verse", "v @r s"},
            {"versed", "v @r s t"},
            {"vert", "v @r t"},
            {"verve", "v @r v"},
            {"vest", "v e s t"},
            {"vests", "v e s t s"},
            {"vet", "v e t"},
            {"vets", "v e t s"},
            {"veuve", "v j u: v"},
            {"vex", "v e k s"},
            {"vexed", "v e k s t"},
            {"vey", "v eI"},
            {"vi", "v aI"},
            {"vibe", "v aI b"},
            {"vibes", "v aI b z"},
            {"vic", "v I k"},
            {"vice", "v aI s"},
            {"vick", "v I k"},
            {"vic's", "v I k s"},
            {"vide", "v aI d"},
            {"vie", "v aI"},
            {"vied", "v aI d"},
            {"viens", "v i: n z"},
            {"viers", "v I@ z"},
            {"vies", "v aI z"},
            {"view", "v j u:"},
            {"viewed", "v j u: d"},
            {"views", "v j u: z"},
            {"vig", "v I g"},
            {"vile", "v aI l"},
            {"ville", "v I l"},
            {"vim", "v I m"},
            {"vin", "v I n"},
            {"vince", "v I n s"},
            {"vine", "v aI n"},
            {"vines", "v aI n z"},
            {"vint", "v I n t"},
            {"vires", "v aI r z"},
            {"virge", "v @r dZ"},
            {"vis", "v I s"},
            {"vise", "v aI s"},
            {"viv", "v I v"},
            {"vive", "v i: v"},
            {"vives", "v aI v z"},
            {"vlach", "v l { tS"},
            {"vlad", "v l { d"},
            {"vlad's", "v l { d z"},
            {"vo", "v @U"},
            {"vogt", "v @U t"},
            {"vogue", "v @U g"},
            {"voice", "v OI s"},
            {"voiced", "v OI s t"},
            {"void", "v OI d"},
            {"voids", "v OI d z"},
            {"voiles", "v OI l z"},
            {"voir", "v w Q@"},
            {"voles", "v @U l z"},
            {"volk", "v @U l k"},
            {"volpe", "v @U l p"},
            {"volt", "v @U l t"},
            {"volts", "v @U l t s"},
            {"voltz", "v @U l t s"},
            {"von", "v O: n"},
            {"vos", "v Q s"},
            {"voss", "v O: s"},
            {"vote", "v @U t"},
            {"votes", "v @U t s"},
            {"vote's", "v @U t s"},
            {"vouch", "v aU tS"},
            {"vouched", "v aU tS t"},
            {"vought", "v O: t"},
            {"vow", "v aU"},
            {"vowed", "v aU d"},
            {"vows", "v aU z"},
            {"vox", "v Q k s"},
            {"vries", "v r i: z"},
            {"vroom", "v r u: m"},
            {"v's", "v i: z"},
            {"vu", "v u:"},
            {"vue", "v j u:"},
            {"wa", "w Q"},
            {"waag", "w Q g"},
            {"waal", "w Q l"},
            {"wace", "w eI s"},
            {"wack", "w { k"},
            {"wacks", "w { k s"},
            {"wad", "w Q d"},
            {"wade", "w eI d"},
            {"wades", "w eI d z"},
            {"wade's", "w eI d z"},
            {"wads", "w Q d z"},
            {"waft", "w Q f t"},
            {"wag", "w { g"},
            {"wage", "w eI dZ"},
            {"waged", "w eI dZ d"},
            {"wagged", "w { g d"},
            {"wags", "w { g z"},
            {"wah", "w Q"},
            {"wahl", "w Q l"},
            {"waid", "w eI d"},
            {"waide", "w eI d"},
            {"waif", "w eI f"},
            {"wail", "w eI l"},
            {"wailed", "w eI l d"},
            {"wails", "w eI l z"},
            {"wain", "w eI n"},
            {"wais", "w eI z"},
            {"waist", "w eI s t"},
            {"waists", "w eI s t s"},
            {"wait", "w eI t"},
            {"waite", "w eI t"},
            {"waites", "w eI t s"},
            {"waite's", "w eI t s"},
            {"waits", "w eI t s"},
            {"waive", "w eI v"},
            {"waived", "w eI v d"},
            {"waives", "w eI v z"},
            {"wake", "w eI k"},
            {"wakes", "w eI k s"},
            {"wal", "w O: l"},
            {"wald", "w O: l d"},
            {"wale", "w eI l"},
            {"wales", "w eI l z"},
            {"walk", "w Q k"},
            {"walke", "w O: k"},
            {"walked", "w O: k t"},
            {"walks", "w O: k s"},
            {"wall", "w O: l"},
            {"walled", "w O: l d"},
            {"walls", "w O: l z"},
            {"wall's", "w O: l z"},
            {"walsh", "w O: l S"},
            {"walt", "w O: l t"},
            {"walth", "w O: l T"},
            {"waltz", "w O: l t s"},
            {"wan", "w Q n"},
            {"wand", "w Q n d"},
            {"wands", "w Q n d z"},
            {"wane", "w eI n"},
            {"waned", "w eI n d"},
            {"wanes", "w eI n z"},
            {"wang", "w { N"},
            {"wang's", "w { N z"},
            {"wank", "w Q N k"},
            {"wann", "w Q n"},
            {"want", "w Q n t"},
            {"wants", "w Q n t s"},
            {"war", "w O@"},
            {"ward", "w O@ d"},
            {"warde", "w O@ d"},
            {"wards", "w O@ d z"},
            {"ward's", "w O@ d z"},
            {"ware", "w e@"},
            {"wares", "w e@ z"},
            {"warf", "w O@ f"},
            {"wark", "w O@ k"},
            {"warm", "w O@ m"},
            {"warmed", "w O@ m d"},
            {"warms", "w O@ m z"},
            {"warmth", "w O@ m T"},
            {"warn", "w O@ n"},
            {"warne", "w O@ n"},
            {"warned", "w O@ n d"},
            {"warns", "w O@ n z"},
            {"warp", "w O@ p"},
            {"warped", "w O@ p t"},
            {"warps", "w O@ p s"},
            {"warr", "w O@"},
            {"wars", "w O@ z"},
            {"war's", "w O@ z"},
            {"wart", "w O@ t"},
            {"warth", "w O@ T"},
            {"warts", "w O@ t s"},
            {"was", "w Q z"},
            {"wash", "w Q S"},
            {"washed", "w Q S t"},
            {"wasp", "w Q s p"},
            {"wasps", "w Q s p s"},
            {"wasp's", "w Q s p s"},
            {"waste", "w eI s t"},
            {"wastes", "w eI s t s"},
            {"wat", "w O: t"},
            {"watch", "w Q tS"},
            {"watched", "w Q tS t"},
            {"watt", "w Q t"},
            {"watts", "w Q t s"},
            {"watt's", "w Q t s"},
            {"waugh", "w O:"},
            {"waugh's", "w O: z"},
            {"wave", "w eI v"},
            {"waved", "w eI v d"},
            {"waves", "w eI v z"},
            {"wax", "w { k s"},
            {"waxed", "w { k s t"},
            {"way", "w eI"},
            {"wayne", "w eI n"},
            {"wayne's", "w eI n z"},
            {"ways", "w eI z"},
            {"way's", "w eI z"},
            {"we", "w i:"},
            {"weak", "w i: k"},
            {"weaks", "w i: k s"},
            {"wealth", "w e l T"},
            {"wealths", "w e l T s"},
            {"wean", "w i: n"},
            {"weaned", "w i: n d"},
            {"wear", "w e@"},
            {"wears", "w e@ z"},
            {"weave", "w i: v"},
            {"weaved", "w i: v d"},
            {"weaves", "w i: v z"},
            {"web", "w e b"},
            {"webb", "w e b"},
            {"webbed", "w e b d"},
            {"webbs", "w e b z"},
            {"webb's", "w e b z"},
            {"webs", "w e b z"},
            {"wecht", "w e k t"},
            {"weck", "w e k"},
            {"wed", "w e d"},
            {"we'd", "w i: d"},
            {"wedge", "w e dZ"},
            {"wedged", "w e dZ d"},
            {"wee", "w i:"},
            {"weed", "w i: d"},
            {"weeds", "w i: d z"},
            {"week", "w i: k"},
            {"weekes", "w i: k s"},
            {"weeks", "w i: k s"},
            {"week's", "w i: k s"},
            {"weeks'", "w i: k s"},
            {"weems", "w i: m z"},
            {"weep", "w i: p"},
            {"weeps", "w i: p s"},
            {"wehr", "w e@"},
            {"wei", "w eI"},
            {"weigh", "w eI"},
            {"weighed", "w eI d"},
            {"weighs", "w eI z"},
            {"weight", "w eI t"},
            {"weights", "w eI t s"},
            {"weil", "w aI l"},
            {"weill", "w aI l"},
            {"weill's", "w aI l z"},
            {"weil's", "w aI l z"},
            {"wein", "w i: n"},
            {"weir", "w i: r"},
            {"weird", "w I@ d"},
            {"weir's", "w i: r z"},
            {"weise", "w i: s"},
            {"weiss", "w aI s"},
            {"welch", "w e l tS"},
            {"weld", "w e l d"},
            {"welds", "w e l d z"},
            {"welk", "w e l k"},
            {"welke", "w e l k"},
            {"well", "w e l"},
            {"we'll", "w i: l"},
            {"welles", "w e l z"},
            {"wells", "w e l z"},
            {"welsh", "w e l tS"},
            {"welt", "w e l t"},
            {"wen", "w e n"},
            {"wend", "w e n d"},
            {"wende", "w e n d"},
            {"wends", "w e n d z"},
            {"wendt", "w e n t"},
            {"wenge", "w e n dZ"},
            {"went", "w e n t"},
            {"wenz", "w e n z"},
            {"wept", "w e p t"},
            {"were", "w @r"},
            {"we're", "w i: r"},
            {"wert", "w @r t"},
            {"wertz", "w @r t s"},
            {"wes", "w e s"},
            {"west", "w e s t"},
            {"west's", "w e s t s"},
            {"wet", "w e t"},
            {"we've", "w i: v"},
            {"wey", "w eI"},
            {"weyl", "w eI l"},
            {"whack", "w { k"},
            {"whacked", "w { k t"},
            {"whacks", "w { k s"},
            {"whale", "w eI l"},
            {"whales", "w eI l z"},
            {"whale's", "w eI l z"},
            {"wham", "w { m"},
            {"whan", "w { n"},
            {"whang", "w { N"},
            {"wharf", "w O@ f"},
            {"wharf's", "w O@ f s"},
            {"what", "w Q t"},
            {"whats", "w @ t s"},
            {"what's", "w V t s"},
            {"wheat", "w i: t"},
            {"wheel", "w i: l"},
            {"wheeled", "w i: l d"},
            {"wheels", "w i: l z"},
            {"wheeze", "w i: z"},
            {"when", "w e n"},
            {"whence", "w e n s"},
            {"when's", "w e n z"},
            {"where", "w e@"},
            {"where'd", "w e@ d"},
            {"where's", "w e@ z"},
            {"whet", "w e t"},
            {"whew", "w u:"},
            {"whey", "w eI"},
            {"which", "w I tS"},
            {"whiff", "w I f"},
            {"whig", "w I g"},
            {"whigs", "w I g z"},
            {"while", "w aI l"},
            {"whiles", "w aI l z"},
            {"whilst", "w aI l s t"},
            {"whim", "w I m"},
            {"whims", "w I m z"},
            {"whine", "w aI n"},
            {"whines", "w aI n z"},
            {"whip", "w I p"},
            {"whipped", "w I p t"},
            {"whips", "w I p s"},
            {"whirl", "w @r l"},
            {"whirled", "w @r l d"},
            {"whisk", "w I s k"},
            {"whisked", "w I s k t"},
            {"whisks", "w I s k s"},
            {"whit", "w I t"},
            {"white", "w aI t"},
            {"whites", "w aI t s"},
            {"white's", "w aI t s"},
            {"whitt", "w I t"},
            {"whiz", "w I z"},
            {"whizzed", "w I z d"},
            {"who", "h u:"},
            {"whoa", "w @U"},
            {"who'd", "h u: d"},
            {"whole", "h @U l"},
            {"wholes", "h @U l z"},
            {"who'll", "h u: l"},
            {"whom", "h u: m"},
            {"whoop", "w u: p"},
            {"whooped", "w u: p t"},
            {"whoops", "w u: p s"},
            {"whoosh", "w u: S"},
            {"whore", "h O@"},
            {"whores", "h O@ z"},
            {"whorl", "w @r l"},
            {"whorls", "w @r l z"},
            {"who's", "h u: z"},
            {"whose", "h u: z"},
            {"who've", "h u: v"},
            {"why", "w aI"},
            {"why'd", "w aI d"},
            {"whys", "w aI z"},
            {"why's", "w aI z"},
            {"whyte", "w aI t"},
            {"wice", "w aI s"},
            {"wich", "w I tS"},
            {"wick", "w I k"},
            {"wickes", "w I k s"},
            {"wicks", "w I k s"},
            {"wide", "w aI d"},
            {"width", "w I d T"},
            {"widths", "w I d T s"},
            {"wied", "w i: d"},
            {"wield", "w i: l d"},
            {"wields", "w i: l d z"},
            {"wien", "w i: n"},
            {"wier", "w I@"},
            {"wiese", "w i: z"},
            {"wife", "w aI f"},
            {"wife's", "w aI f s"},
            {"wig", "w I g"},
            {"wight", "w aI t"},
            {"wigs", "w I g z"},
            {"wil", "w I l"},
            {"wild", "w aI l d"},
            {"wilde", "w aI l d"},
            {"wilds", "w aI l d z"},
            {"wile", "w aI l"},
            {"wiles", "w aI l z"},
            {"wilk", "w I l k"},
            {"wilkes", "w I l k s"},
            {"will", "w I l"},
            {"willed", "w I l d"},
            {"willes", "w aI l z"},
            {"willms", "w I l m z"},
            {"wills", "w I l z"},
            {"will's", "w I l z"},
            {"wil's", "w I l z"},
            {"wilt", "w I l t"},
            {"wilts", "w I l t s"},
            {"wiltz", "w I l t s"},
            {"wimp", "w I m p"},
            {"wimps", "w I m p s"},
            {"win", "w I n"},
            {"wince", "w I n s"},
            {"winced", "w I n s t"},
            {"winch", "w I n tS"},
            {"wind", "w I n d"},
            {"winds", "w I n d z"},
            {"wind's", "w I n d z"},
            {"wine", "w aI n"},
            {"wined", "w aI n d"},
            {"wines", "w aI n z"},
            {"wine's", "w aI n z"},
            {"wing", "w I N"},
            {"winge", "w I n dZ"},
            {"winged", "w I N d"},
            {"wings", "w I N z"},
            {"wing's", "w I N z"},
            {"wink", "w I N k"},
            {"winked", "w I N k t"},
            {"winks", "w I N k s"},
            {"winn", "w I n"},
            {"winne", "w I n"},
            {"wins", "w I n z"},
            {"win's", "w I n z"},
            {"wint", "w I n t"},
            {"wipe", "w aI p"},
            {"wiped", "w aI p t"},
            {"wipes", "w aI p s"},
            {"wirth", "w @r T"},
            {"wirtz", "w @r t s"},
            {"wis", "w I s"},
            {"wise", "w aI z"},
            {"wised", "w aI z d"},
            {"wish", "w I S"},
            {"wished", "w I S t"},
            {"wisp", "w I s p"},
            {"wiss", "w I s"},
            {"wist", "w I s t"},
            {"wit", "w I t"},
            {"witch", "w I tS"},
            {"with", "w I D"},
            {"wits", "w I t s"},
            {"wit's", "w I t s"},
            {"witt", "w I t"},
            {"witte", "w I t"},
            {"wives", "w aI v z"},
            {"wives'", "w aI v z"},
            {"wiz", "w I z"},
            {"wo", "w @U"},
            {"woe", "w @U"},
            {"woes", "w @U z"},
            {"wok", "w Q k"},
            {"woke", "w @U k"},
            {"wold", "w @U l d"},
            {"wolf", "w U l f"},
            {"wolfe", "w U l f"},
            {"wolfe's", "w U l f s"},
            {"wolff", "w U l f"},
            {"wolf's", "w U l f s"},
            {"wolk", "w @U k"},
            {"woll", "w Q l"},
            {"wolves", "w U l v z"},
            {"womb", "w u: m"},
            {"won", "w V n"},
            {"wong", "w O: N"},
            {"wong's", "w O: N z"},
            {"wonk", "w Q n k"},
            {"wonks", "w Q n k s"},
            {"wont", "w @U n t"},
            {"won't", "w @U n t"},
            {"woo", "w u:"},
            {"wood", "w U d"},
            {"woods", "w U d z"},
            {"wood's", "w U d z"},
            {"wooed", "w u: d"},
            {"woof", "w u: f"},
            {"wool", "w U l"},
            {"woolf", "w U l f"},
            {"wools", "w U l z"},
            {"woos", "w u: z"},
            {"woo's", "w u: z"},
            {"wop", "w Q p"},
            {"wops", "w Q p s"},
            {"word", "w @r d"},
            {"words", "w @r d z"},
            {"word's", "w @r d z"},
            {"wore", "w O@"},
            {"work", "w @r k"},
            {"worked", "w @r k t"},
            {"works", "w @r k s"},
            {"work's", "w @r k s"},
            {"world", "w @r l d"},
            {"worlds", "w @r l d z"},
            {"world's", "w @r l d z"},
            {"worm", "w @r m"},
            {"worms", "w @r m z"},
            {"worm's", "w @r m z"},
            {"worn", "w O@ n"},
            {"worse", "w @r s"},
            {"worst", "w @r s t"},
            {"worth", "w @r T"},
            {"worths", "w @r T s"},
            {"worth's", "w @r T s"},
            {"worts", "w @r t s"},
            {"wouk", "w u: k"},
            {"would", "w U d"},
            {"woulfe", "w aU l f"},
            {"wound", "w aU n d"},
            {"wounds", "w u: n d z"},
            {"wove", "w @U v"},
            {"wow", "w aU"},
            {"wowed", "w aU d"},
            {"wows", "w aU z"},
            {"woy", "w OI"},
            {"wrack", "r { k"},
            {"wracked", "r { k t"},
            {"wrap", "r { p"},
            {"wrapped", "r { p t"},
            {"wraps", "r { p s"},
            {"wrath", "r { T"},
            {"wray", "r eI"},
            {"wreak", "r i: k"},
            {"wreaked", "r i: k t"},
            {"wreaks", "r i: k s"},
            {"wreath", "r i: T"},
            {"wreaths", "r i: T s"},
            {"wreck", "r e k"},
            {"wrecked", "r e k t"},
            {"wrecks", "r e k s"},
            {"wren", "r e n"},
            {"wrench", "r e n tS"},
            {"wrenched", "r e n tS t"},
            {"wrest", "r e s t"},
            {"wright", "r aI t"},
            {"wrights", "r aI t s"},
            {"wright's", "r aI t s"},
            {"wring", "r I N"},
            {"wrist", "r I s t"},
            {"wrists", "r I s t s"},
            {"writ", "r I t"},
            {"write", "r aI t"},
            {"writes", "r aI t s"},
            {"writhe", "r I T"},
            {"writs", "r I t s"},
            {"wrong", "r O: N"},
            {"wronged", "r O: N d"},
            {"wrongs", "r O: N z"},
            {"wrote", "r @U t"},
            {"wrought", "r O: t"},
            {"wrung", "r V N"},
            {"wry", "r aI"},
            {"wu", "w u:"},
            {"wulf", "w U l f"},
            {"wurst", "w @r s t"},
            {"wurtz", "w @r t s"},
            {"wu's", "w u: z"},
            {"wuss", "w V s"},
            {"wy", "w eI"},
            {"wye", "w aI"},
            {"wyke", "w aI k"},
            {"wylde", "w aI l d"},
            {"wyle", "w aI l"},
            {"wyles", "w aI l z"},
            {"wyne", "w aI n"},
            {"wynn", "w I n"},
            {"wynne", "w I n"},
            {"wynns", "w I n z"},
            {"wynn's", "w I n z"},
            {"wyss", "w I s"},
            {"x", "e k s"},
            {"xanthe", "z { n D"},
            {"xi", "S i:"},
            {"xian", "z j Q n"},
            {"xie", "z i:"},
            {"xing", "S I N"},
            {"xu", "z u:"},
            {"y", "w aI"},
            {"y.", "w aI"},
            {"ya", "j Q"},
            {"yacht", "j Q t"},
            {"yachts", "j Q t s"},
            {"yack", "j { k"},
            {"yah", "j Q"},
            {"yak", "j { k"},
            {"yale", "j eI l"},
            {"yale's", "j eI l z"},
            {"y'all", "j O: l"},
            {"yam", "j { m"},
            {"yams", "j { m z"},
            {"yan", "j { n"},
            {"yang", "j { N"},
            {"yank", "j { N k"},
            {"yanked", "j { N k t"},
            {"yanks", "j { N k s"},
            {"yao", "j aU"},
            {"yap", "j { p"},
            {"yapp", "j { p"},
            {"yaps", "j { p s"},
            {"yard", "j Q@ d"},
            {"yards", "j Q@ d z"},
            {"yard's", "j Q@ d z"},
            {"yarn", "j Q@ n"},
            {"yarns", "j Q@ n z"},
            {"yates", "j eI t s"},
            {"yaw", "j O:"},
            {"yawn", "j O: n"},
            {"yawned", "j O: n d"},
            {"yawns", "j O: n z"},
            {"yay", "j eI"},
            {"ye", "j i:"},
            {"yea", "j eI"},
            {"yeah", "j {"},
            {"year", "j I@"},
            {"yearn", "j @r n"},
            {"yearned", "j @r n d"},
            {"yearns", "j @r n z"},
            {"years", "j I@ z"},
            {"year's", "j I@ z"},
            {"years'", "j I@ z"},
            {"yeas", "j eI z"},
            {"yeast", "j i: s t"},
            {"yeasts", "j i: s t s"},
            {"yeats", "j eI t s"},
            {"yee", "j i:"},
            {"yeh", "j e"},
            {"yell", "j e l"},
            {"yelled", "j e l d"},
            {"yells", "j e l z"},
            {"yen", "j e n"},
            {"yen's", "j e n z"},
            {"yep", "j e p"},
            {"yes", "j e s"},
            {"yet", "j e t"},
            {"yett", "j e t"},
            {"yeung", "j u: N"},
            {"yew", "j u:"},
            {"yew's", "j u: z"},
            {"yi", "j i:"},
            {"yield", "j i: l d"},
            {"yields", "j i: l d z"},
            {"yim", "j I m"},
            {"yin", "j I n"},
            {"yip", "j I p"},
            {"ynez", "j n e z"},
            {"yo", "j @U"},
            {"yoe", "j @U"},
            {"yoke", "j @U k"},
            {"yoked", "j @U k t"},
            {"yokes", "j @U k s"},
            {"yolk", "j @U k"},
            {"yolks", "j @U k s"},
            {"yom", "j Q m"},
            {"yon", "j Q n"},
            {"yong", "j O: N"},
            {"yoo", "j u:"},
            {"yore", "j O@"},
            {"york", "j O@ k"},
            {"yorke", "j O@ k"},
            {"yorks", "j O@ k s"},
            {"york's", "j O@ k s"},
            {"yost", "j @U s t"},
            {"you", "j u:"},
            {"you'd", "j u: d"},
            {"you'll", "j u: l"},
            {"young", "j V N"},
            {"youngs", "j V N z"},
            {"young's", "j V N z"},
            {"your", "j O@"},
            {"you're", "j U@"},
            {"yours", "j U@ z"},
            {"youse", "j aU s"},
            {"youth", "j u: T"},
            {"youths", "j u: D z"},
            {"youth's", "j u: T s"},
            {"you've", "j u: v"},
            {"yow", "j aU"},
            {"yoy", "j OI"},
            {"y's", "w aI z"},
            {"yu", "j u:"},
            {"yuck", "j V k"},
            {"yue", "j u:"},
            {"yuille", "j u: l"},
            {"yuk", "j V k"},
            {"yuks", "j V k s"},
            {"yul", "j V l"},
            {"yule", "j u: l"},
            {"yum", "j V m"},
            {"yun", "j V n"},
            {"yung", "j V N"},
            {"yup", "j V p"},
            {"yurt", "j U@ t"},
            {"yu's", "j u: z"},
            {"yves", "i: v"},
            {"z", "z i:"},
            {"zach", "z { k"},
            {"zack", "z { k"},
            {"zacks", "z { k s"},
            {"zack's", "z { k s"},
            {"zag", "z { g"},
            {"zagged", "z { g d"},
            {"zags", "z { g z"},
            {"zaid", "z eI d"},
            {"zak", "z { k"},
            {"zandt", "z { n t"},
            {"zane", "z eI n"},
            {"zant", "z { n t"},
            {"zap", "z { p"},
            {"zapped", "z { p t"},
            {"zaps", "z { p s"},
            {"zared", "z e@ d"},
            {"zea", "z i:"},
            {"zeal", "z i: l"},
            {"zech", "z e k"},
            {"zed", "z e d"},
            {"zee", "z i:"},
            {"zeiss", "z aI s"},
            {"zeke", "z i: k"},
            {"zel", "z e l"},
            {"zell", "z e l"},
            {"zelle", "z e l"},
            {"zen", "z e n"},
            {"zens", "z e n z"},
            {"zest", "z e s t"},
            {"zeus", "z u: s"},
            {"zhang", "Z { N"},
            {"ziff", "z I f"},
            {"ziff's", "z I f s"},
            {"zig", "z I g"},
            {"zilch", "z I l tS"},
            {"zill", "z I l"},
            {"zim", "z I m"},
            {"zinc", "z I N k"},
            {"zinck", "z I N k"},
            {"zine", "z aI n"},
            {"zines", "z aI n z"},
            {"zing", "z I N"},
            {"zingg", "z I N g"},
            {"zink", "z I N k"},
            {"zinke", "z I N k"},
            {"zinn", "z I n"},
            {"zins", "z I n z"},
            {"zip", "z I p"},
            {"zipped", "z I p t"},
            {"zips", "z I p s"},
            {"ziv", "z I v"},
            {"zoll", "z Q l"},
            {"zone", "z @U n"},
            {"zoned", "z @U n d"},
            {"zones", "z @U n z"},
            {"zoo", "z u:"},
            {"zoom", "z u: m"},
            {"zoomed", "z u: m d"},
            {"zooms", "z u: m z"},
            {"zoos", "z u: z"},
            {"zoo's", "z u: z"},
            {"zorn", "z O@ n"},
            {"z's", "z i: z"},
            {"zsa", "Z Q"},
            {"zug", "z V g"},
            {"zug's", "z V g z"},
            {"zurn", "z @r n"},
            {"zwart", "z w O@ t"},
            {"zweig", "z w aI g"},
            {"zwick", "z w I k"},
        };
        #endregion
        #region KEY_JP
#if JAVA
        public static final String[][] _KEY_JP = {
#else
        public static readonly String[,] _KEY_JP = {
#endif
            {"あ", "a"},
            {"い", "i"},
            {"う", "M"},
            {"え", "e"},
            {"お", "o"},
            {"か", "k a"},
            {"き", "k' i"},
            {"く", "k M"},
            {"け", "k e"},
            {"こ", "k o"},
            {"さ", "s a"},
            {"し", "S i"},
            {"す", "s M"},
            {"せ", "s e"},
            {"そ", "s o"},
            {"た", "t a"},
            {"ち", "tS i"},
            {"つ", "ts M"},
            {"て", "t e"},
            {"と", "t o"},
            {"な", "n a"},
            {"に", "J i"},
            {"ぬ", "n M"},
            {"ね", "n e"},
            {"の", "n o"},
            {"は", "h a"},
            {"ひ", "C i"},
            {"ふ", "p\\ M"},
            {"へ", "h e"},
            {"ほ", "h o"},
            {"ま", "m a"},
            {"み", "m' i"},
            {"む", "m M"},
            {"め", "m e"},
            {"も", "m o"},
            {"や", "j a"},
            {"ゆ", "j M"},
            {"よ", "j o"},
            {"ら", "4 a"},
            {"り", "4' i"},
            {"る", "4 M"},
            {"れ", "4 e "},
            {"ろ", "4 o"},
            {"わ", "w a"},
            {"ゐ", "w i"},
            {"ゑ", "w e"},
            {"を", "o"},
            {"ぁ", "a"},
            {"ぃ", "i"},
            {"ぅ", "M"},
            {"ぇ", "e"},
            {"ぉ", "o"},
            {"が", "g a"},
            {"ぎ", "g' i"},
            {"ぐ", "g M"},
            {"げ", "g e"},
            {"ご", "g o"},
            {"ざ", "dz a"},
            {"じ", "dZ i"},
            {"ず", "dz M"},
            {"ぜ", "dz e"},
            {"ぞ", "dz o"},
            {"だ", "d a"},
            {"ぢ", "dZ i"},
            {"づ", "dz M"},
            {"で", "d e"},
            {"ど", "d o"},
            {"ば", "b a"},
            {"び", "b' i"},
            {"ぶ", "b M"},
            {"べ", "b e"},
            {"ぼ", "b o"},
            {"ぱ", "p a"},
            {"ぴ", "p' i"},
            {"ぷ", "p M"},
            {"ぺ", "p e"},
            {"ぽ", "p o"},
            {"いぇ", "j e"},
            {"うぃ", "w i"},
            {"うぇ", "w e"},
            {"うぉ", "w o"},
            {"きゃ", "k' a"},
            {"きゅ", "k' M"},
            {"きぇ", "k' e"},
            {"きょ", "k' o"},
            {"しゃ", "S a"},
            {"しゅ", "S M"},
            {"しぇ", "S e"},
            {"しょ", "S o"},
            {"ちゃ", "tS a"},
            {"ちゅ", "tS M"},
            {"ちぇ", "tS e"},
            {"ちょ", "tS o"},
            {"にゃ", "J a"},
            {"にゅ", "J M"},
            {"にぇ", "J e"},
            {"にょ", "J o"},
            {"ひゃ", "C a"},
            {"ひゅ", "C M"},
            {"ひぇ", "C e"},
            {"ひょ", "C o"},
            {"ふゃ", "p\\' a"},
            {"ふぃ", "p\\' i"},
            {"ふゅ", "p\\' M"},
            {"ふぇ", "p\\ e"},
            {"みゃ", "m' a"},
            {"みゅ", "m' M"},
            {"みぇ", "m' e"},
            {"みょ", "m' o"},
            {"りゃ", "4' a"},
            {"りゅ", "4' M"},
            {"りぇ", "4' e"},
            {"りょ", "4' o"},
            {"ぎゃ", "g' a"},
            {"ぎゅ", "g' M"},
            {"ぎぇ", "g' e"},
            {"ぎょ", "g' o"},
            {"じゃ", "dZ a"},
            {"じゅ", "dZ M"},
            {"じぇ", "dZ e"},
            {"じょ", "dZ o"},
            {"びゃ", "b' a"},
            {"びゅ", "b' M"},
            {"びぇ", "b' e"},
            {"びょ", "b' o"},
            {"ぴゃ", "p' a"},
            {"ぴゅ", "p' M"},
            {"ぴぇ", "p' e"},
            {"ぴょ", "p' o"},
            {"ふぁ", "p\\ a"},
            {"ふぉ", "p\\ o"},
            {"てゃ", "t' a"},
            {"てぃ", "t' i"},
            {"てゅ", "t' M"},
            {"てぇ", "t' e"},
            {"てょ", "t' o"},
            {"でゃ", "d' a"},
            {"でぃ", "d' i"},
            {"でゅ", "d' M"},
            {"でぇ", "d' e"},
            {"でょ", "d' o"},
            {"すぃ", "s i"},
            {"ずぃ", "dz i"},
            {"とぅ", "t M"},
            {"どぅ", "d M"},
            {"ゃ", "j a"},
            {"ゅ", "j M"},
            {"ょ", "j o"},
            {"ん", "n"},
            {"ア", "a"},
            {"イ", "i"},
            {"ウ", "M"},
            {"エ", "e"},
            {"オ", "o"},
            {"カ", "k a"},
            {"キ", "k' i"},
            {"ク", "k M"},
            {"ケ", "k e"},
            {"コ", "k o"},
            {"サ", "s a"},
            {"シ", "S i"},
            {"ス", "s M"},
            {"セ", "s e"},
            {"ソ", "s o"},
            {"タ", "t a"},
            {"チ", "tS i"},
            {"ツ", "ts M"},
            {"テ", "t e"},
            {"ト", "t o"},
            {"ナ", "n a"},
            {"ニ", "J i"},
            {"ヌ", "n M"},
            {"ネ", "n e"},
            {"ノ", "n o"},
            {"ハ", "h a"},
            {"ヒ", "C i"},
            {"フ", "p\\ M"},
            {"ヘ", "h e"},
            {"ホ", "h o"},
            {"マ", "m a"},
            {"ミ", "m' i"},
            {"ム", "m M"},
            {"メ", "m e"},
            {"モ", "m o"},
            {"ヤ", "j a"},
            {"ユ", "j M"},
            {"ヨ", "j o"},
            {"ラ", "4 a"},
            {"リ", "4' i"},
            {"ル", "4 M"},
            {"レ", "4 e "},
            {"ロ", "4 o"},
            {"ワ", "w a"},
            {"ヰ", "w i"},
            {"ヱ", "w e"},
            {"ヲ", "o"},
            {"ァ", "a"},
            {"ィ", "i"},
            {"ゥ", "M"},
            {"ェ", "e"},
            {"ォ", "o"},
            {"ガ", "g a"},
            {"ギ", "g' i"},
            {"グ", "g M"},
            {"ゲ", "g e"},
            {"ゴ", "g o"},
            {"ザ", "dz a"},
            {"ジ", "dZ i"},
            {"ズ", "dz M"},
            {"ゼ", "dz e"},
            {"ゾ", "dz o"},
            {"ダ", "d a"},
            {"ヂ", "dZ i"},
            {"ヅ", "dz M"},
            {"デ", "d e"},
            {"ド", "d o"},
            {"バ", "b a"},
            {"ビ", "b' i"},
            {"ブ", "b M"},
            {"ベ", "b e"},
            {"ボ", "b o"},
            {"パ", "p a"},
            {"ピ", "p' i"},
            {"プ", "p M"},
            {"ペ", "p e"},
            {"ポ", "p o"},
            {"イェ", "j e"},
            {"ウィ", "w i"},
            {"ウェ", "w e"},
            {"ウォ", "w o"},
            {"キャ", "k' a"},
            {"キュ", "k' M"},
            {"キェ", "k' e"},
            {"キョ", "k' o"},
            {"シャ", "S a"},
            {"シュ", "S M"},
            {"シェ", "S e"},
            {"ショ", "S o"},
            {"チャ", "tS a"},
            {"チュ", "tS M"},
            {"チェ", "tS e"},
            {"チョ", "tS o"},
            {"ニャ", "J a"},
            {"ニュ", "J M"},
            {"ニェ", "J e"},
            {"ニョ", "J o"},
            {"ヒャ", "C a"},
            {"ヒュ", "C M"},
            {"ヒェ", "C e"},
            {"ヒョ", "C o"},
            {"フャ", "p\\' a"},
            {"フィ", "p\\' i"},
            {"フュ", "p\\' M"},
            {"フェ", "p\\ e"},
            {"ミャ", "m' a"},
            {"ミュ", "m' M"},
            {"ミェ", "m' e"},
            {"ミョ", "m' o"},
            {"リャ", "4' a"},
            {"リュ", "4' M"},
            {"リェ", "4' e"},
            {"リョ", "4' o"},
            {"ギャ", "g' a"},
            {"ギュ", "g' M"},
            {"ギェ", "g' e"},
            {"ギョ", "g' o"},
            {"ジャ", "dZ a"},
            {"ジュ", "dZ M"},
            {"ジェ", "dZ e"},
            {"ジョ", "dZ o"},
            {"ビャ", "b' a"},
            {"ビュ", "b' M"},
            {"ビェ", "b' e"},
            {"ビョ", "b' o"},
            {"ピャ", "p' a"},
            {"ピュ", "p' M"},
            {"ピェ", "p' e"},
            {"ピョ", "p' o"},
            {"ファ", "p\\ a"},
            {"フォ", "p\\ o"},
            {"テャ", "t' a"},
            {"ティ", "t' i"},
            {"テュ", "t' M"},
            {"テェ", "t' e"},
            {"テョ", "t' o"},
            {"デャ", "d' a"},
            {"ディ", "d' i"},
            {"デュ", "d' M"},
            {"デェ", "d' e"},
            {"デョ", "d' o"},
            {"スィ", "s i"},
            {"ズィ", "dz i"},
            {"トゥ", "t M"},
            {"ドゥ", "d M"},
            {"ャ", "j a"},
            {"ュ", "j M"},
            {"ョ", "j o"},
            {"ン", "n"},
            {"ヴ", "b M"},
            {"a", "a"},
            {"e", "e"},
            {"i", "i"},
            {"m", "n"},
            {"n", "n"},
            {"o", "o"},
            {"u", "M"},
            {"A", "a"},
            {"E", "e"},
            {"I", "i"},
            {"M", "n"},
            {"N", "n"},
            {"O", "o"},
            {"U", "M"},
            {"ka", "k a"},
            {"ki", "k' i"},
            {"ku", "k M"},
            {"ke", "k e"},
            {"ko", "k o"},
            {"kya", "k' a"},
            {"kyu", "k' M"},
            {"kyo", "k' o"},
            {"sa", "s a"},
            {"si", "s i"},
            {"su", "s M"},
            {"se", "s e"},
            {"so", "s o"},
            {"ta", "t a"},
            {"ti", "t' i"},
            {"tu", "t M"},
            {"te", "t e"},
            {"to", "t o"},
            {"tya", "t' a"},
            {"tyu", "t' M"},
            {"tyo", "t' o"},
            {"na", "n a"},
            {"ni", "J i"},
            {"nu", "n M"},
            {"ne", "n e"},
            {"no", "n o"},
            {"nya", "J a"},
            {"nyu", "J M"},
            {"nyo", "J o"},
            {"ha", "h a"},
            {"hi", "C i"},
            {"he", "h e"},
            {"ho", "h o"},
            {"hya", "C a"},
            {"hyu", "C M"},
            {"hyo", "C o"},
            {"ma", "m a"},
            {"mi", "m' i"},
            {"mu", "m M"},
            {"me", "m e"},
            {"mo", "m o"},
            {"mya", "m' a"},
            {"myu", "m' M"},
            {"myo", "m' o"},
            {"ya", "j a"},
            {"yu", "j M"},
            {"ye", "j e"},
            {"yo", "j o"},
            {"ra", "4 a"},
            {"ri", "4' i"},
            {"ru", "4 M"},
            {"re", "4 e"},
            {"ro", "4 o"},
            {"rya", "4' a"},
            {"ryu", "4' M"},
            {"ryo", "4' o"},
            {"wa", "w a"},
            {"wi", "w i"},
            {"we", "w e"},
            {"wo", "w o"},
            {"ga", "g a"},
            {"gi", "g' i"},
            {"gu", "g M"},
            {"ge", "g e"},
            {"go", "g o"},
            {"gya", "g' a"},
            {"gyu", "g' M"},
            {"gyo", "g' o"},
            {"za", "dz a"},
            {"zi", "dz i"},
            {"zu", "dz M"},
            {"ze", "dz e"},
            {"zo", "dz o"},
            {"da", "d a"},
            {"di", "d' i"},
            {"du", "d M"},
            {"de", "d e"},
            {"do", "d o"},
            {"dya", "d' a"},
            {"dyu", "d' M"},
            {"dyo", "d' o"},
            {"ba", "b a"},
            {"bi", "b' i"},
            {"bu", "b M"},
            {"be", "b e"},
            {"bo", "b o"},
            {"bya", "b' a"},
            {"byu", "b' M"},
            {"byo", "b' o"},
            {"pa", "p a"},
            {"pi", "p' i"},
            {"pu", "p M"},
            {"pe", "p e"},
            {"po", "p o"},
            {"pya", "p' a"},
            {"pyu", "p' M"},
            {"pyo", "p' o"},
            {"sha", "S a"},
            {"shi", "S i"},
            {"shu", "S M"},
            {"sho", "S o"},
            {"tsu", "ts M"},
            {"cha", "tS a"},
            {"chi", "tS i"},
            {"chu", "tS M"},
            {"cho", "tS o"},
            {"fu", "p\\ M"},
            {"ja", "dZ a"},
            {"ji", "dZ i"},
            {"ju", "dZ M"},
            {"jo", "dZ o"},
        };
        #endregion
        #endregion

        #region Static Method and Property
        public static SymbolTable getSymbolTable( int index ) {
            if ( !s_initialized ) {
                loadDictionary();
            }
            if ( 0 <= index && index < s_table.size() ) {
                return s_table.get( index );
            } else {
                return null;
            }
        }

        public static void loadDictionary() {
#if DEBUG
            PortUtil.println( "SymbolTable.LoadDictionary()" );
#endif
            s_default_jp = new SymbolTable( "DEFAULT_JP", _KEY_JP, true );
            s_default_en = new SymbolTable( "DEFAULT_EN", _KEY_EN, true );
            s_table.clear();
            int count = 0;
            s_table.put( count, s_default_en );
            count++;
            s_table.put( count, s_default_jp );
            count++;

            // 辞書フォルダからの読込み
            String editor_path = VocaloSysUtil.getEditorPath( SynthesizerType.VOCALOID2 );
            if ( editor_path != "" ) {
                String path = PortUtil.combinePath( PortUtil.getDirectoryName( editor_path ), "UDIC" );
                if ( !PortUtil.isDirectoryExists( path ) ) {
                    return;
                }
                String[] files = PortUtil.listFiles( path, "*.udc" );
                for ( int i = 0; i < files.Length; i++ ) {
                    files[i] = PortUtil.getFileName( files[i] );
#if DEBUG
                    PortUtil.println( "    files[i]=" + files[i] );
#endif
                    String dict = PortUtil.combinePath( path, files[i] );
                    s_table.put( count, new SymbolTable( dict, true, false ) );
                    count++;
                }
            }

            // 起動ディレクトリ
            String path2 = PortUtil.combinePath( PortUtil.getApplicationStartupPath(), "udic" );
            if ( PortUtil.isDirectoryExists( path2 ) ) {
                String[] files2 = PortUtil.listFiles( path2, "*.eudc" );
                for ( int i = 0; i < files2.Length; i++ ) {
                    files2[i] = PortUtil.getFileName( files2[i] );
#if DEBUG
                    PortUtil.println( "    files2[i]=" + files2[i] );
#endif
                    String dict = PortUtil.combinePath( path2, files2[i] );
                    s_table.put( count, new SymbolTable( dict, false, false ) );
                    count++;
                }
            }
            s_initialized = true;
        }


        public static boolean attatch( String phrase, ByRef<String> result ) {
#if DEBUG
            PortUtil.println( "SymbolTable.Attatch" );
            PortUtil.println( "    phrase=" + phrase );
#endif
            for ( Iterator<Integer> itr = s_table.keySet().iterator(); itr.hasNext(); ) {
                int key = itr.next();
                SymbolTable table = s_table.get( key );
                if ( table.isEnabled() ) {
                    if ( table.attatchImp( phrase, result ) ) {
                        return true;
                    }
                }
            }
            result.value = "a";
            return false;
        }

        public static int getCount() {
            if ( !s_initialized ) {
                loadDictionary();
            }
            return s_table.size();
        }

        public static void changeOrder( Vector<ValuePair<String, Boolean>> list ) {
#if DEBUG
            PortUtil.println( "SymbolTable#changeOrder" );
#endif
            TreeMap<Integer, SymbolTable> buff = new TreeMap<Integer, SymbolTable>();
            for ( Iterator<Integer> itr = s_table.keySet().iterator(); itr.hasNext(); ) {
                int key = itr.next();
                buff.put( key, (SymbolTable)s_table.get( key ).clone() );
            }
            s_table.clear();
            int count = list.size();
            for ( int i = 0; i < count; i++ ) {
                ValuePair<String, Boolean> itemi = list.get( i );
#if DEBUG
                PortUtil.println( "SymbolTable#changeOrder; list[" + i + "]=" + itemi.getKey() + "," + itemi.getValue() );
#endif
                for ( Iterator<Integer> itr = buff.keySet().iterator(); itr.hasNext(); ) {
                    int key = itr.next();
                    SymbolTable table = buff.get( key );
                    if ( table.getName().Equals( itemi.getKey() ) ) {
                        table.setEnabled( itemi.getValue() );
                        s_table.put( i, table );
                        break;
                    }
                }
            }
        }
        #endregion

#if !JAVA
        public object Clone() {
            return clone();
        }
#endif

        public Object clone() {
            SymbolTable ret = new SymbolTable();
            ret.m_dict = new TreeMap<String, String>();
            for ( Iterator<String> itr = m_dict.keySet().iterator(); itr.hasNext(); ) {
                String key = itr.next();
                ret.m_dict.put( key, m_dict.get( key ) );
            }
            ret.m_name = m_name;
            ret.m_enabled = m_enabled;
            return ret;
        }

        private SymbolTable() {
        }

        public String getName() {
            return m_name;
        }

        public boolean isEnabled() {
            return m_enabled;
        }

        public void setEnabled( boolean value ) {
            m_enabled = value;
        }

        public SymbolTable( String path, boolean is_udc_mode, boolean enabled ) {
            m_dict = new TreeMap<String, String>();
            m_enabled = enabled;
            if ( !PortUtil.isFileExists( path ) ) {
                return;
            }
            m_name = PortUtil.getFileName( path );
            BufferedReader sr = null;
            try {
                if ( is_udc_mode ) {
                    sr = new BufferedReader( new InputStreamReader( new FileInputStream( path ), "Shift_JIS" ) );
                    if ( sr == null ) {
                        return;
                    }
                } else {
                    sr = new BufferedReader( new InputStreamReader( new FileInputStream( path ), "UTF8" ) );
                    if ( sr == null ) {
                        return;
                    }
                }
                String line;
                while ( sr.ready() ) {
                    line = sr.readLine();
                    if ( !line.StartsWith( "//" ) ) {
                        String[] spl = PortUtil.splitString( line, new String[] { "\t" }, 2, true );
                        if ( spl.Length >= 2 ) {
                            if ( m_dict.containsKey( spl[0] ) ) {
                                PortUtil.println( "SymbolTable..ctor" );
                                PortUtil.println( "    dictionary already contains key: " + spl[0] );
                            } else {
                                m_dict.put( spl[0], spl[1] );
                            }
                        }
                    }
                }
            } catch ( Exception ex ) {
                PortUtil.println( "SymbolTable..ctor" );
                PortUtil.println( "    " + ex );
            } finally {
                if ( sr != null ) {
                    try {
                        sr.close();
                    } catch ( Exception ex2 ) {
                    }
                }
            }
        }

        private boolean attatchImp( String phrase, ByRef<String> result ) {
            String s = phrase.ToLower();
            if ( m_dict.containsKey( s ) ) {
                result.value = m_dict.get( s );
                return true;
            } else {
                result.value = "a";
                return false;
            }
        }

#if JAVA
        private SymbolTable( String name, String[][] key, boolean enabled ){
#else
        private SymbolTable( String name, String[,] key, boolean enabled ) {
#endif
            m_enabled = enabled;
            m_name = name;
            m_dict = new TreeMap<String, String>();
#if JAVA
            for( int i = 0; i < key.length; i++ ){
                if( m_dict.containsKey( key[i][0] ) ){
                }else{
                    m_dict.put( key[i][0], key[i][1] );
                }
            }
#else
            for ( int i = 0; i < key.GetLength( 0 ); i++ ) {
                if ( m_dict.containsKey( key[i, 0] ) ) {
                } else {
                    m_dict.put( key[i, 0], key[i, 1] );
                }
            }
#endif
        }
    }

#if !JAVA
}
#endif
