/*
 *
 *    vConnect.cpp
 *                        (c) HAL 2010-
 *
 *  This files is a part of v.Connect.
 * vConnect class is a main class that connects UTAU and WORLD.
 * It is consisted of spectral, pitch and dynamics calculation.
 *
 * These files are distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *
 */
#include "vConnect.h"
#include <time.h>

#define TRANS_MAX 4096
double temporary1[TRANS_MAX];
double temporary2[TRANS_MAX];
double temporary3[TRANS_MAX];

double vConnect::noteFrequency[NOTE_NUM];
double vConnect::vibrato[VIB_NUM];

#ifdef WIN32
#include <windows.h>
#include <process.h>
unsigned __stdcall calculateSpecgram(void *arg);
#else
void    calculateSpecgram(void *arg);
#endif

struct vConnectArg{
    long beginFrame;
    long frameLength;
    long offset;
    int  fft_len;
    int  ap_len;
    vsqFileEx *vsq;
    standSpecgram *specgram;
    vector<double> *dynamics;
    vector<vector<standBP> > *controlCurves;
    runtimeOptions *options;
    vector<corpusManager*> *managers;
};


void    applyTexture( double* target, standTexture* texture, int length, double rate )
{
    double *trans, *filter, f0, step, index, tmp;
    int len;
    texture->getTexturePointer( &trans, &filter, &f0, &len );

    // 暫定処置
    step = (double)len / 2 / (double)length;

    for( int i = 0; i < length; i++ ){
        index = (double)i * step;
        temporary1[i] = step * ( interpolateArray( index, trans ) - index ) + i;
        temporary2[i] = rate * interpolateArray( index, filter );
    }

    // y = x の直線からの距離で適用率を変化させる．
    double prev_y = 0, prev_x = 0, next_y, next_x;
    int j = 0, k = 0;

    for( ; ; j++ ){
        //cout << prev_x << " , " << prev_y << " , j = " << j << " , k = " << k << endl;
        next_x = 0.5 * ( ( 1 + rate ) * temporary1[j+1] + ( 1 - rate ) * (double)j );
        next_y = 0.5 * ( ( 1 - rate ) * temporary1[j+1] + ( 1 + rate ) * (double)j );
        for( ; prev_y <= k && k <= next_y; k++ ){
            // ゼロ除算チェック．
            if( next_y - prev_y ){
                tmp = ( (double)k - prev_y ) / ( next_y - prev_y );
            }else{
                tmp = 0;
            }
            temporary3[k] = tmp * next_x + ( 1 - tmp ) * prev_x;
        }
        prev_x = next_x;
        prev_y = next_y;
        if( j >= length - 1 ){
            for( ; k < length; k++ ){
                temporary3[k] = k;
            }
            break;
        }
    }

    // フィルタを適用
    for( int i = 0; i < length; i++ )
        target[i] *= exp( temporary2[i] );
    // 軸の変換
    applyStretching( temporary3, target, length );
}

void    transform_F0( standTexture* texture, double* target, double dstF0, double srcF0, int length, double absRate ){
    length /= 2;
    double *trans, *filter, f0Base;
    int tex_len;
    texture->getTexturePointer( &trans, &filter, &f0Base, &tex_len );
    double rate1 = log( dstF0 / f0Base ) * absRate;
    double rate2 = log( srcF0 / f0Base ) * absRate;
    memcpy( temporary3, filter, sizeof(double)*length );
    for( int i = 0; i < length; i++ ){
        temporary3[i] = temporary3[i*2];
    }
    if( f0Base ){
        /* 伸縮関数１： srcF0 -> f0Base への写像を表す */
        for( int i = 0; i < length && i < tex_len; i++ )
            temporary1[i] = trans[i*2] * rate2 / 2.0;
        /* 伸縮関数２： f0Base -> dstF0 への写像を表す */
        for( int i = 0; i < length && i < tex_len; i++ )
            temporary2[i] = - trans[i*2] * rate1 / 2.0;

        /*== まず伸縮関数１を srcF0 -> f0Base -> dstF0 を表す合成写像へ変換 ==*/
        applyStretching( temporary2, temporary1, length );

        /*== 次にその変換関数を使用して、スペクトルを srcF0 の世界から dstF0 の世界へ ==*/
        applyStretching( temporary1, target, length );

        /*== SpectrumFilter を f0Base の世界から dstF0 の世界へ ==*/
        applyStretching( temporary2, temporary3, length );

        /*== Filter を ⊿logF0 で定まる係数で適用. マジックナンバーで割り算. ==*/
        for( int i = 0; i < length; i++ )
            target[i] *= exp( ( rate1 - rate2 ) * temporary3[i] / 5.0 );
    }
}

double    getPitchFluctuation( double second )
{
    double result = 1.0 + ( sin( 12.7 * ST_PI * second ) + sin ( 7.1 * ST_PI * second ) + sin( 4.7 * ST_PI * second ) / 3.0 ) / 300.0;

    return result;
}

bool vConnect::createWspFile( string_t v_path, string_t output, string_t alias, runtimeOptions options )
{
    utauVoiceDataBase utauDB;
    utauParameters parameters;
    standData *data;
    standFrame frame;

    // 強制的に platinum 使用とする．
    options.fast = true;

    // 初期化
    if( utauDB.readUtauVoiceDataBase( v_path, options.encodingOtoIni.c_str() ) != 1 )
        return false;

    utauDB.getUtauParameters( parameters, alias );

    // 読み込み
    manager.setVoiceDB( &utauDB, options );
    data = manager.getStandData( alias, options );

    if( data == NULL ) // 失敗
        return false;

    // 成功
    double *spectrum = new double[2048];
    double *aperiodicity = new double[4];
    double f0;
    int index, begin, end, c = 0;

    memset( spectrum, 0, sizeof(double) * 2048 );
    memset( aperiodicity, 0, sizeof(double) * 4 );
    // 固定長以下 100 フレームが代表点．
    begin = (int)( parameters.msFixedLength / framePeriod );
    end = begin + 100;
    if( end >= data->specgram->getTimeLength() )
        end = data->specgram->getTimeLength() - 1;

    double w = (double)( end - begin );
    f0 = 0.0;

    for( index = begin; index < end; index++ ){
        data->specgram->getFramePointer( index, frame );
        if( !( *frame.f0 ) )
            continue;
        c++;
        for( int i = 0; i < 1024; i++ )
            spectrum[i] += frame.spectrum[i];
        for( int i = 0; i < 4; i ++ )
            aperiodicity[i] += frame.aperiodicity[i];
        f0 += *( frame.f0 ) / w;
    }
    for( int i = 0; i < 1024; i++ )
        spectrum[i] /= (double)c;
    for( int i = 0; i < 4; i++ )
        aperiodicity[i] /= (double)c;
    for( int i = 1024; i < 2048; i++ )
        spectrum[i] = frame.spectrum[i];
    // spectrum, aperiodicity, f0 の順に格納．
    char buf[2048];
    sprintf( buf, "%s", output.c_str() );  // うーん....
    FILE *fp = fopen( buf, "w" );          // うーん．．．．
    if( fp ){
        for( int i = 0; i < 2048; i++ )
            fprintf( fp, "%.10e\n", spectrum[i] );
        for( int i = 0; i < 4; i++ )
            fprintf( fp, "%.10e\n", aperiodicity[i] );
        fprintf( fp, "%lf\n", f0 );
    } else {
        SAFE_DELETE_ARRAY( spectrum );
        SAFE_DELETE_ARRAY( aperiodicity );
        return false;
    }

    fclose( fp );

    SAFE_DELETE_ARRAY( spectrum );
    SAFE_DELETE_ARRAY( aperiodicity );

    return true;
}

vConnect::vConnect()
{
    for( int i = 0; i < NOTE_NUM; i++ )
        noteFrequency[i] = A4_PITCH * pow( 2.0, (double)( i - A4_NOTE ) / 12.0 );
    vibrato[0] = 0.0;
    for( int i = 1; i < VIB_NUM; i++ ){
        double period = exp( 5.24 - 1.07e-2 * i ) * 2.0 / 1000.0;
        vibrato[i] = 2.0 * ST_PI / period;
    }

    time_t timer;
    time( &timer );
    srand( (unsigned int)timer );
    fluctTheta = 2.0 * (double)rand() / (double)RAND_MAX * ST_PI;
}

vConnect::~vConnect()
{
    for( unsigned int i = 0; i < managers.size(); i++ )
        SAFE_DELETE( managers[i] );
}

void    emptyPath( double secOffset, string_t output )
{
    waveFileEx wave;
    wave.setOffset( secOffset );
    string toutput;
    mb_conv( output, toutput );
    wave.writeWaveFile( toutput );
    return;
}

bool vConnect::synthesize( string_t input, string_t output, runtimeOptions options ){
#ifdef _DEBUG
    cout << "vConnect::synthesize; calling vsq.readVsqFile...";
#endif
    // 読み込みこけたら帰る
    if( vsq.readVsqFile( input, options ) != 1 ){
#ifdef _DEBUG
        cout << " done, failed" << endl;
#endif
        return false;
    }
#ifdef _DEBUG
        cout << " done, successed" << endl;
#endif

    // 空のときは空の wave を出力して終了
    if( vsq.events.eventList.empty() ){
        emptyPath( vsq.getEndSec(), output );
        return true;
    }

    long beginFrame, frameLength, waveLength;
    int  fft_len, ap_len;
    standSpecgram specgram;
    double *wave;

    vector<utauVoiceDataBase*> *pDBs = vsq.getVoiceDBs();
    for( unsigned int i = 0; i < pDBs->size(); i++ ){
        corpusManager *p = new corpusManager;
        p->setVoiceDB( (*pDBs)[i], options );
        managers.push_back( p );
    }

    // 準備１．先行発音などパラメータの適用，及びコントロールカーブをフレーム時刻へ変換
    this->calculateVsqInfo();

    // 準備２．合成に必要なローカル変数の初期化
    beginFrame = vsq.events.eventList[0]->beginFrame;
    frameLength = endFrame - beginFrame;

    specgram.setFrameLength( frameLength, options.fast );
    fft_len = specgram.getFFTLength();
    ap_len = specgram.getAperiodicityLength();
    vector<double> dynamics( frameLength, 0.0 );

    // 準備３．振幅・基本周波数・時刻 t を計算する．
    this->calculateF0( specgram, dynamics );

    vConnectArg arg1, arg2;

    arg1.ap_len = ap_len;
    arg1.beginFrame = beginFrame;
    arg1.dynamics = &dynamics;
    arg1.fft_len = fft_len;
    arg1.managers = &managers;
    arg1.offset = 0;
    arg1.frameLength = frameLength;
    arg1.options = &options;
    arg1.vsq = &vsq;
    arg1.specgram = &specgram;
    arg1.controlCurves = &controlCurves;
#ifdef WIN32
    HANDLE hThread[2];
    hMutex = CreateMutex(NULL, FALSE, NULL);
    hFFTWMutex = CreateMutex(NULL, FALSE, NULL);
    memcpy(&arg2, &arg1, sizeof(vConnectArg));
    arg1.frameLength /= 2;
    arg2.offset = arg1.frameLength;

    hThread[0] = (HANDLE)_beginthreadex(NULL, 0, calculateSpecgram, &arg1, 0, NULL);
    hThread[1] = (HANDLE)_beginthreadex(NULL, 0, calculateSpecgram, &arg2, 0, NULL);

    WaitForMultipleObjects(2, hThread, TRUE, INFINITE);

    CloseHandle(hThread[0]);
    CloseHandle(hThread[1]);
    CloseHandle(hMutex);
    CloseHandle(hFFTWMutex);
    hMutex = hFFTWMutex = 0;

#else
    calculateSpecgram(&arg1);
#endif

    // 二段階に分けた．
    wave = specgram.synthesizeWave( &waveLength );
    /* 波形自体の編集をここに書く． */
    calculateDynamics( dynamics, wave, waveLength, options.volumeNormalization );

    specgram.writeWaveFile( output, beginFrame, &dynamics );

    return true;
}

#ifdef WIN32
unsigned __stdcall calculateSpecgram(void *arg)
#else
void    calculateSpecgram(void *arg)
#endif
{
    vConnectArg* p = (vConnectArg*)arg;
    long beginFrame, frameLength, index, position;
    int  fft_len, ap_len;
    vector<double> *dynamics;
    vector<vector<standBP> > *controlCurves;
    vector<corpusManager*> *managers;
    runtimeOptions options;

    options.encodingOtoIni = p->options->encodingOtoIni;
    options.encodingVoiceTexture = p->options->encodingVoiceTexture;
    options.encodingVowelTable = p->options->encodingVowelTable;
    options.encodingVsqText = p->options->encodingVsqText;
    options.f0Transform = p->options->f0Transform;
    options.fast = p->options->fast;
    options.volumeNormalization = p->options->volumeNormalization;
    options.wspMode = p->options->wspMode;

    fft_len = p->fft_len;
    ap_len  = p->ap_len;
    beginFrame = p->beginFrame;
    vsqFileEx *vsq = p->vsq;
    standSpecgram *specgram = p->specgram;
    index = p->offset;
    dynamics = p->dynamics;
    controlCurves = p->controlCurves;
    frameLength = p->frameLength;
    managers = p->managers;

    double *aperiodicity = new double[ap_len];
    double *spectrum = new double[fft_len];
    double *trans = new double[fft_len];
    double *temporary = new double[fft_len];

    standData *present, *next = NULL;
    standFrame target, pres_src, next_src;

    int bri, bre, cle, gen, ope;
    double pres_vel, next_vel, c_end, n_c_end, tmp;
    double bre_rate, gen_rate, bri_rate, cle_rate;

    bri = bre = cle = gen = ope = 0;

    standTexture* texture = new standTexture, *temp_texture;
    texture->setPlainTexture( fft_len );

    // 合成開始．長くなってしまうのはどうにもならんのか
    int num_events = vsq->events.eventList.size();
    vsqEventEx *item_next = NULL;

    item_next = vsq->events.eventList[0];
    int begin_num;
    for( int i = 0; i < num_events; i++ ){
        if( index < vsq->events.eventList[i]->endFrame - beginFrame ){
            item_next = vsq->events.eventList[i];
            begin_num = i;
            break;
        }
    }
    for( int i = begin_num; i < num_events && index < frameLength; i++ ){
        vsqEventEx *item_this = item_next;
        if( i + 1 < num_events ){
            item_next = vsq->events.eventList[i + 1];
        }

        // 合成範囲外は合成範囲になるまで無音を与える
        if( item_this->beginFrame > beginFrame + index ){
            for( ; index < item_this->beginFrame - beginFrame; index++ ){
                specgram->getFramePointer( index, target );
                memset( target.aperiodicity, 0, sizeof( double ) * ap_len );
                memset( target.spectrum, 0, sizeof( double ) * fft_len );
                (*dynamics)[index] = 0.0;
                *(target.f0) = 0.0;
            }
        }
        // 合成可能範囲内なので wav ファイルを検索，適宜 WORLD で分析．
        if( item_this->singerIndex != -1 ){
            present = (*managers)[item_this->singerIndex]->getStandData( item_this->lyricHandle.getLyric(), options );
        }
        if( item_this->isContinuousBack ){
            if( item_this->singerIndex != -1 ){
                next = (*managers)[item_next->singerIndex]->getStandData( item_next->lyricHandle.getLyric(), options );
            }
            next_vel = pow( 2.0, (double)( 64 - item_next->velocity ) / 64.0 );
            n_c_end = item_next->utauSetting.msFixedLength * next_vel / framePeriod;
            //if( next && item_next->singerIndex >= 0 && item_next->singerIndex < managers.size() ){
            //    cle_texture_next = (*managers)[item_next->singerIndex]->getClearness( item_next->lyricHandle.getLyric(), options );
            //}
        }

        //---現在の音符を処理せずにゼロクリアするのは以下のいずれかのとき
        //  1.) 音源データベース中に該当するエイリアスが存在しない．
        //  2.) 音源データベース中に該当するエイリアスに対応するファイルが存在しない．
        //  3.) 音源データベース管理クラスが生成されていない．
        if( !present || !present->specgram || item_this->singerIndex < 0 || item_this->singerIndex >= managers->size() ) {

            // 現在音符が明示的休符が後続音を持つ場合のみ例外処理．
            if( !item_this->isContinuousBack || !item_this->isRest ){
                int begin_j = max(p->offset + beginFrame, item_this->beginFrame);
                for( int j = begin_j; j < item_this->endFrame && j - beginFrame < frameLength; j++ ){ 
                    specgram->getFramePointer( j - beginFrame, target );
                    memset( target.aperiodicity, 0, sizeof( double ) * ap_len );
                    memset( target.spectrum, 0, sizeof( double ) * fft_len );
                    (*dynamics)[j - beginFrame] = 0.0;
                    *(target.f0) = 0.0;
                    index = j - beginFrame;
                }
            }
            continue;
        }

        // 準備
        pres_vel = pow( 2.0, (double)(64 - item_this->velocity) / 64.0);
        c_end = item_this->utauSetting.msFixedLength * pres_vel / framePeriod;

        //cle_texture = (*managers)[item_this->singerIndex]->getClearness( item_this->lyricHandle.getLyric(), options );

        for( ; index < item_this->endFrame - beginFrame && index < frameLength; index++ ){
            specgram->getFramePointer( index, target );

            position = index - item_this->beginFrame + beginFrame;

            if( position < c_end ){
                position = (long)((double)position / pres_vel);
            }else{
                position = (long)(item_this->utauSetting.msFixedLength / framePeriod);
            }

            present->specgram->getFramePointer( position, pres_src );

            memcpy( spectrum, pres_src.spectrum, sizeof( double ) * (fft_len / 2 + 1) );
            memcpy( aperiodicity, pres_src.aperiodicity, sizeof( double ) * ap_len );
            temp_texture = present->texture;

            // 現在注目しているフレームが無声音の場合それを優先する．
            if( *(pres_src.f0) == 0.0 ){
                *(target.f0) = 0.0;

                //--- 音符末尾におけるスペクトルの線形補間を行うのは以下の全てを満たすとき
                //  1.) 後続音が楽譜上・データーベース上で存在する．（明示的休符は自動的にこちらを満たせない．）
                //  2.) 後続音の開始時刻が現在時刻よりも早いとき．
            }else if( item_this->isContinuousBack && next && next->specgram && item_next->beginFrame < index + beginFrame ){
                position = index - item_next->beginFrame + beginFrame;

                if( position < n_c_end ){
                    position = (long)((double)position / next_vel);
                }else{
                    position = (long)(item_next->utauSetting.msFixedLength / framePeriod);
                }
                next->specgram->getFramePointer( position, next_src );

                position = item_next->beginFrame - beginFrame;
                tmp = (double)(index - position) / (double)(item_this->endFrame - beginFrame - position);
                tmp = 0.5 - 0.5 * cos( ST_PI * tmp );

                if( *(next_src.f0) == 0.0 ){
                    tmp = 1.0;
                    *(target.f0) = 0.0;
                    memcpy( spectrum, next_src.spectrum, sizeof( double ) * fft_len );
                    memcpy( aperiodicity, next_src.aperiodicity, sizeof( double ) * ap_len );
                }else{
                    for( int k = 0; k <= fft_len / 2; k++ ){
                        spectrum[k] = pow( spectrum[k], 1.0 - tmp ) * pow( next_src.spectrum[k], tmp );
                    }
                    for( int k = 0; k < ap_len; k++ ){
                        aperiodicity[k] = aperiodicity[k] * (1.0 - tmp) + next_src.aperiodicity[k] * tmp;
                    }
                    if( present->texture != next->texture && item_this->endFrame - beginFrame - 50 < index ){
                        long len = 50;
                        if( item_this->endFrame - beginFrame - 50 < 0 ){
                            len = item_this->endFrame - beginFrame;
                        }
                        double rate = (double)(index - item_this->endFrame + beginFrame + len) / (double)len;
                        temp_texture = texture;
                    }
                }
            }

            /* ここに周波数変換 */
            if( *(target.f0) && options.f0Transform ){                // 無声音の場合とりあえず何もしない

                // 表情系のコントロールはここにまとめておこう．
                while( index + beginFrame > (*controlCurves)[BRIGHTNESS][bri].frameTime ) bri++;
                while( index + beginFrame > (*controlCurves)[BRETHINESS][bre].frameTime ) bre++;
                while( index + beginFrame > (*controlCurves)[CLEARNESS][cle].frameTime ) cle++;
                while( index + beginFrame > (*controlCurves)[GENDER][gen].frameTime ) gen++;

                // いまはまだはやい
                // transform_F0( temp_texture, spectrum, *(target.f0), *(pres_src.f0), fft_len, 1.0 );

                /* GEN 適用 */
                gen_rate = pow( 2.0, (double)((*controlCurves)[GENDER][gen].value - 64) / 64.0 );

                for( int k = 0; k <= fft_len / 2; k++ ){
                    tmp = gen_rate * (double)k;
                    if( tmp > fft_len / 2 ){
                        temporary[k] = spectrum[fft_len / 2 - 1];
                    }else{
                        temporary[k] = interpolateArray( tmp, spectrum );
                    }
                }
                memcpy( spectrum, temporary, sizeof(double)*( fft_len / 2 + 1 ) );

                /* BRE / BRI / CLE */
                bre_rate = (double)(*controlCurves)[BRETHINESS][bre].value / 128.0;
                bri_rate = (double)( (*controlCurves)[BRIGHTNESS][bri].value - 64 ) / 64.0;
                cle_rate = (double)(*controlCurves)[CLEARNESS][cle].value / 128.0;
                if( options.fast ){
                    // WORLD 0.0.1 における非周期性パラメタはシグモイド関数のパラメタであり，
                    // それぞれ以下のパラメタを決定する．
                    // ap[0] = u :: 非周期性指標のダイナミックレンジ
                    // ap[1] = b :: 非周期性指標の最大値 max = 1.0 - b で決定する．
                    // ap[2] = fc:: 非周期性指標の値が(u + b) / 2になる f0 の値
                    // ap[3] = w :: シグモイド関数の傾き
                    // 値は再考の余地大有り．
                    aperiodicity[0] = 0.1 + 0.4 * bre_rate;
                    aperiodicity[1] = 0.9 - 0.7 * bre_rate;
                    aperiodicity[2] *= pow(2.0, bri_rate); // うーん...
                    aperiodicity[3] *= pow(4.0, 0.5 - bre_rate);
                    //
                    // なお非周期性指標は周波数軸上で
                    // noiseSpectrum[i] = spectrum[i] * aperiodicityRatio[i]
                    // の形でスペクトル中に含まれるノイズの量を決定する
                }else {
                    for( int k = 0; k < ap_len; k++ ){
                        aperiodicity[k] += (1.0 - aperiodicity[k]) * bre_rate;
                    }
                }

                for(int k = 0; k <= fft_len / 2; k++){
                    double freq = (double)k / (double)fft_len * (double)fs;
                    if(freq < *target.f0 * 2){
                        spectrum[k] /= pow(4.0, bri_rate);
                    }else if(freq > *target.f0 * 4){
                        spectrum[k] *= pow(4.0, bri_rate);
                    }else{
                        spectrum[k] /= pow(4.0, bri_rate * cos((freq - *target.f0 * 2.0) / *target.f0 * ST_PI));
                    }
                }
                // 不必要な部分はゼロ詰め．
                for(int k = fft_len / 2 + 1; k < fft_len; k++){
                    spectrum[k] = 0.0;
                }
                //applyTexture( spectrum, &cle_actual, 1024, cle_rate );
            }

            memcpy( target.spectrum, spectrum, sizeof( double ) * fft_len );
            memcpy( target.aperiodicity, aperiodicity, sizeof( double ) * ap_len );
        }
    }

    SAFE_DELETE_ARRAY( temporary );
    SAFE_DELETE_ARRAY( spectrum );
    SAFE_DELETE_ARRAY( trans );
    SAFE_DELETE( texture );

#ifdef WIN32
    _endthreadex(0);
#endif
    return 0;
}

void vConnect::calculateVsqInfo( void )
{
    // 書きづらいので
    vector<vsqEventEx*>* events = &( vsq.events.eventList );
    string_t temp;
    vector<utauVoiceDataBase*> *pDBs = vsq.getVoiceDBs();
    utauVoiceDataBase* voiceDB;

    float msPreUtterance, msVoiceOverlap;
    int singerIndex = 0;

    endFrame = 0;

    /////////
    // 前から後ろをチェック
    for( unsigned int i = 0; i < events->size(); i++ ){
        vsqEventEx* itemi = vsq.events.eventList[i];
        
        // タイプ判定
        if( itemi->type == _T( "Singer" ) ){
            // 歌手なら歌手番号拾ってきて
            singerIndex = vsq.getSingerIndex( itemi->iconHandle.getIDS() );

            // 自分を消して
            vector<vsqEventEx*>::iterator it = events->begin();
            int j = 0;
            while( it != events->end() ){
                if( itemi == (*it) ) break;
                j++;
                it++;
            }
            if( it != events->end() ){
                events->erase( it );
                SAFE_DELETE( itemi );
            }

            // （ i 番目今消しちゃったから次に進んでるのと一緒だから ）
            if( i >= events->size() )
                break;
            // 次の音符へ
            itemi = vsq.events.eventList[i];
        }
        if( singerIndex < 0 || singerIndex >= pDBs->size() )
            continue;
        voiceDB = (*pDBs)[singerIndex];
        // 原音設定の反映
        temp = itemi->lyricHandle.getLyric();
        msPreUtterance = itemi->utauSetting.msPreUtterance;
        msVoiceOverlap = itemi->utauSetting.msVoiceOverlap;
        voiceDB->getUtauParameters( itemi->utauSetting, temp );
        itemi->utauSetting.msPreUtterance = msPreUtterance;
        itemi->utauSetting.msVoiceOverlap = msVoiceOverlap;

        // 空白文字が存在したときはVCV音素片
        itemi->isVCV = ( temp.find( _T(" ") ) != string_t::npos );

        // 休符の文字はとりあえず 'R', 'r' を対象にしてUTAUパラメタを初期化しておこう．
        itemi->isRest = ( temp.compare( _T("R") ) == 0 || temp.compare( _T("r") ) == 0);
        if(itemi->isRest){
            itemi->utauSetting.msPreUtterance = itemi->utauSetting.msVoiceOverlap = 0.0;
        }

        // 開始位置の計算
        itemi->beginFrame = (long)( (
                        vsq.vsqTempoBp.tickToSecond( itemi->tick ) * 1000.0 - itemi->utauSetting.msPreUtterance
                        * pow( 2.0, ( 64.0 - itemi->velocity ) / 64.0 ) ) / framePeriod );
        // ポルタメントが０％の場合適当な値を入れておく
        if( itemi->portamentoLength < 2 )
            itemi->portamentoLength = 2;
        itemi->singerIndex = singerIndex;
    }

    /////////
    // 後ろから前をチェック
    for( unsigned int i = 0; i < events->size(); i++ ){
        // まず Tick 時刻から終了時刻を計算
        (*events)[i]->endFrame = (long)(
            ( vsq.vsqTempoBp.tickToSecond( vsq.events.eventList[i]->tick + vsq.events.eventList[i]->length ) * 1000.0 ) / framePeriod );

        // 一個前の音符がある場合，連続性のチェック
        if( i ){
            // i 番目の音符が，i - 1 番目の音符が終わる前に始まる場合連続とみなす
            (*events)[i-1]->isContinuousBack = ( (*events)[i]->beginFrame <= (*events)[i-1]->endFrame );

            ////* 連続時のオーバーラップの設定 */
            
            if( (*events)[i-1]->isContinuousBack ){

                // i 番目が CV 音素片の場合
                if( !(*events)[i]->isVCV ){
                    // まず i 番目の先行発音を i - 1 番目に適用する
                    (*events)[i-1]->endFrame -= (long)( (*events)[i]->utauSetting.msPreUtterance
                                                        * pow( 2.0, ( 64.0 - (*events)[i]->velocity ) / 64.0 ) / framePeriod );
                    // さらにオーバーラップも適用する
                    (*events)[i-1]->endFrame += (long)( (*events)[i]->utauSetting.msVoiceOverlap / framePeriod );
                } else
                    if( (*events)[i-1]->endFrame - (*events)[i-1]->beginFrame > 20 )
                        (*events)[i-1]->endFrame -= 20;
                    else
                        (*events)[i-1]->endFrame = (*events)[i-1]->beginFrame;

                // 最後にセーフガードとして短くなりすぎないようにチェック
                if( (*events)[i-1]->endFrame < (*events)[i-1]->beginFrame )
                        (*events)[i-1]->endFrame = (*events)[i-1]->beginFrame;
            }
        }

    }

    for( unsigned int i = 0; i < events->size(); i++ )
        if( endFrame < (*events)[i]->endFrame )
            endFrame = (*events)[i]->endFrame;

    // コントロールカーブは vsq 管理クラスにやってもらう
    controlCurves.resize( vsq.controlCurves.size() );
    for( unsigned int i = 0; i < controlCurves.size(); i++ )
        vsq.controlCurves[i].getList( controlCurves[i] );
}

void    vConnect::calculateF0( standSpecgram& dst, vector<double>& dynamics )
{
    standFrame frame;
    double *f0, *t, pitch_change, tmp, vibratoTheta = 0.0, vibratoRate, vibratoDepth;
    long beginFrame = vsq.events.eventList[0]->beginFrame;
    long frameLength = endFrame - beginFrame;
    long index = 0;
    long por_begin, por_length;
	long prev_end = LONG_MIN, vib_begin = 0, note_begin;
    int  pit = 0, pbs = 0, dyn = 0; // これはインデックス

    // 仕様依存．f0 と t は連続した配列でなければならない．
    dst.getFramePointer( 0, frame );
    f0 = frame.f0;
    t = frame.t;

    for( unsigned int i = 0; i < vsq.events.eventList.size(); i++ ){
        vsqEventEx *itemi = vsq.events.eventList[i];

        // デフォルト値で埋める
        for( ; index < itemi->beginFrame - beginFrame; index++ ){
            f0[index] = 0.0;
            t[index] = (double)index * framePeriod / 1000.0;
            dynamics[index] = 0.0;
        }

        // 後続のノートがあるかどうか
        if( !itemi->isContinuousBack ){
            // ないなら，ポルタメントの開始位置を設定
            por_begin = itemi->endFrame - 50;
            if( por_begin < itemi->beginFrame )
                por_begin = itemi->beginFrame;
            por_begin -= beginFrame;
        }else{
            // ありえない開始位置にしておくよ！！
            por_begin = LONG_MAX;
        }

		// vibrato 開始位置は元々の音符開始位置近辺から計算する．そこまで厳密じゃなくていいよね．
		if( prev_end > itemi->beginFrame - beginFrame ){
			vib_begin = prev_end;
			vib_begin += (long)( 1000.0 * vsq.vsqTempoBp.tickToSecond( itemi->vibratoDelay ) /framePeriod );
		} else {
			vib_begin = itemi->beginFrame - beginFrame;
			vib_begin += (long)( 1000.0 * vsq.vsqTempoBp.tickToSecond( itemi->vibratoDelay ) /framePeriod );
		}

        // ノート・ビブラート・微細振動を書く
        for( ; index < itemi->endFrame - beginFrame; index++ ){
            // ピッチetcカーブに格納されている値の内どれを使うか？
            while( index + beginFrame > controlCurves[PITCH_BEND][pit].frameTime ){
                pit++;
            }
            while( index + beginFrame > controlCurves[PITCH_BEND_SENS][pbs].frameTime ){
                pbs++;
            }
            while( index + beginFrame > controlCurves[DYNAMICS][dyn].frameTime ){
                dyn++;
            }
            pitch_change = pow( 2, (double)controlCurves[PITCH_BEND][pit].value / 8192.0 * (double)controlCurves[PITCH_BEND_SENS][pbs].value / 12.0 );
            f0[index] = noteFrequency[itemi->note] * pitch_change * getPitchFluctuation( (double)index * framePeriod / 1000.0 );
            t[index] = (double)index * framePeriod / 1000.0;
            dynamics[index] = (double)controlCurves[DYNAMICS][dyn].value / 64.0;
            if( index > por_begin ){
                dynamics[index] *= 1.0 - (double)( index - por_begin ) / 50.0;
            }
            
            /* Vibrato */
            if( index > vib_begin ){
				double pos = (double)(index - vib_begin ) / (double)( itemi->endFrame - beginFrame - vib_begin );
                vibratoRate = vibrato[itemi->vibratoHandle.getVibratoRate( pos )];
                vibratoDepth = (double)itemi->vibratoHandle.getVibratoDepth( pos ) * 2.5 / 127.0 / 2.0;
                vibratoTheta += vibratoRate * framePeriod / 1000.0;
                f0[index] *= pow( 2.0, 1.0 / 12.0 * vibratoDepth * sin( vibratoTheta ) );
                if( vibratoTheta > 2.0 * ST_PI ){
                    vibratoTheta -= 2.0 * ST_PI;
                }
            }else{
                vibratoTheta = 0.0;
            }
		}
		prev_end = itemi->endFrame - beginFrame;
    }

    prev_end = LONG_MIN;

    // ポルタメントを描きます．（ビブラートとは実は順番依存）
    for( unsigned int i = 0; i < vsq.events.eventList.size(); i++ ){
        vsqEventEx *itemi = vsq.events.eventList[i];

        // 後続がいる場合は必要なパラメータを計算してポルタメントを書く
        if( itemi->isContinuousBack ){
            if( prev_end > itemi->beginFrame )
                note_begin = prev_end;
            else
                if( itemi->isVCV )
                    note_begin = (long)( vsq.vsqTempoBp.tickToSecond( itemi->tick ) * 1000.0 / framePeriod );
                else
                    note_begin = itemi->beginFrame;
            por_begin = note_begin
                      + (long)((double)(itemi->endFrame - note_begin)
                      * (1.0 - (double)(itemi->portamentoLength) / 100.0));
            tmp = noteFrequency[vsq.events.eventList[i + 1]->note] / noteFrequency[itemi->note];
        }else{
            continue;
        }

        por_length = itemi->endFrame - por_begin;
        double inv_por_length = 1.0 / (double)por_length;
        long frame_offset = por_begin - beginFrame;
        for( long j = 0; j < por_length; j++ ){
            double x = (double)j * inv_por_length;
            f0[j + frame_offset] *= pow( tmp, 0.5 * (1.0 - cos( ST_PI * x ))
                                    - (double)itemi->portamentoDepth / 100.0 * (sin( ST_PI * 4.0 / 3.0 * x ) * (1.5 - x) / 1.5) );
        }
        for( long j = por_length; j < por_length * 3 / 2; j++ ){
            double x = (double)j * inv_por_length;
            f0[j + frame_offset] *= pow( tmp, - (double)itemi->portamentoDepth / 100.0 * sin( ST_PI * 4.0 / 3.0 * x ) * (1.5 - x) / 1.5 );
        }
        prev_end = itemi->endFrame;
    }
}

void vConnect::calculateDynamics( vector<double> &dynamics, double *wave, long wave_len, bool volumeNormalization )
{
    vector<long>   decay_frame( vsq.events.eventList.size() );
    vector<long>   attack_frame( vsq.events.eventList.size() );
    double pres_vel;
    long note_begin;
    long prev_end = LONG_MIN;
    long beginFrame = vsq.events.eventList[0]->beginFrame;
    long c_end, preutterance;

    /*== 適用する位置を計算する ==*/
    for( unsigned int i = 0; i < vsq.events.eventList.size(); i++ ){
        vsqEventEx* itemi = vsq.events.eventList[i];

        /*============= 固定長終了時を attack, 真ん中か，可能であれば， attack + 100[ms] の位置を decay とする．============*/

        // 準備
        pres_vel = pow( 2.0, (double)(64 - itemi->velocity ) / 64.0);
        c_end = (long)(itemi->utauSetting.msFixedLength * pres_vel / framePeriod);
        preutterance = (long)(itemi->utauSetting.msPreUtterance * pres_vel / framePeriod);

        // 音符の開始位置は固定長終了位置か, 直前の音符が終了した位置.
        if( prev_end > itemi->beginFrame + c_end - beginFrame ){
            note_begin = prev_end;
        }else{
            note_begin = itemi->beginFrame + c_end - beginFrame;
        }

        // attack 位置の計算
        if( itemi->isVCV ){
            attack_frame[i] = itemi->beginFrame + preutterance - beginFrame;
        }else{
            attack_frame[i] = itemi->beginFrame + c_end - beginFrame;
        }

        // decay 位置の計算
        // attack + 100ms はあまりよろしくない位置のとき．
        if( itemi->endFrame - beginFrame < attack_frame[i] + 100 / framePeriod )
            decay_frame[i] = ( note_begin + itemi->endFrame - beginFrame ) / 2;
        else
            decay_frame[i] = attack_frame[i] + 100 / framePeriod;

        prev_end = itemi->endFrame - beginFrame;

    }

    /*== 抽出した音量と音符情報から音量遷移曲線を描画 ==*/
    prev_end = LONG_MIN;
    // 取り扱っている音符が直前の音符と連続しているかどうか
    // (直前の音符のisContinuousBackフラグが起ってるかどうか)
    bool   isContinuousFront = false;
    int    prev_frame, tmp_len;
    double prev_value, tmp_value;
    for( unsigned int i = 0; i < vsq.events.eventList.size(); i++ ){
        vsqEventEx* itemi = vsq.events.eventList[i];

        // 明示的休符は無視する．
        if( itemi->isRest ){
            continue;
        }

        double attack_rate = pow( 2.0, (double)( itemi->attack - 50 ) / 50.0 );
        double decay_rate  = pow( 2.0, (double)( itemi->decay  - 50 ) / 50.0 );

        if( isContinuousFront ){
            note_begin = prev_frame;
        } else {
            note_begin = itemi->beginFrame - beginFrame;
            prev_value = attack_rate;
        }

        // 前回終了位置から attack の開始（固定長終了）位置までを余弦関数で接続．
        int frame_offset = note_begin;
        tmp_len = attack_frame[i] - frame_offset;
        if( tmp_len + frame_offset >= (int)dynamics.size() ){
            // dynamicsのout of range例外を回避
            tmp_len = dynamics.size() - frame_offset;
        }
        tmp_value = attack_rate - prev_value;
        for( int index = 0; index < tmp_len; index++ ){
            dynamics[index + frame_offset] *= ( prev_value + tmp_value * ( 0.5 - 0.5 * cos( ST_PI * (double)index / (double)tmp_len ) ) );
        }

        // attack の開始（固定長終了）位置から decay 位置まで．
        frame_offset = attack_frame[i];
        tmp_len = decay_frame[i] - frame_offset;
        if( tmp_len + frame_offset >= (int)dynamics.size() ){
            // dynamicsのout of range例外を回避
            tmp_len = dynamics.size() - frame_offset;
        }
        tmp_value = decay_rate - attack_rate;
        for( int index = 0; index < tmp_len; index++ ){
            dynamics[index + frame_offset] *= ( attack_rate + tmp_value * ( 0.5 - 0.5 * cos( ST_PI * (double)index / (double)tmp_len ) ) );
        }

        if( !itemi->isContinuousBack || vsq.events.eventList[i+1]->isRest ){
            /* 後続音は存在しないか明示的休符なので最低限の正規化 */
            for( int index = decay_frame[i]; index < itemi->endFrame - beginFrame; index++ ){
                dynamics[index] *= decay_rate;
            }
            isContinuousFront = false;
        } else {
            /* 次の音符用に準備 */
            prev_frame = decay_frame[i];
            prev_value = decay_rate;
            isContinuousFront = true;
        }
    }
}
