#ifndef __VSQNoteManager_h__
#define __VSQNoteManager_h__

/////includes
//
#include <list>
#include <string>
#include <iostream>
#include <sstream>
#include <fstream>
#include <map>
using namespace std;

#include <math.h>

#include "UTAUManager.h"
#include "VSQEventManager.h"
#include "VSQHandleParseContext.h"

//Tick per Second 960/60　or 480/60
#define TICK_PER_SEC 8.0

class NoteManager : public EventManager{
public:

    /**
     * @brief コンストラクタ。管理用変数の初期化をする。
     */
#ifdef USE_MULTI_SINGER
    NoteManager(){
        mEditTarget = NULL;
        mCurrentIdOrHandle = -1;
        mSingerIndex = 0;
    };
#else
    NoteManager(){
        mEditTarget = NULL;
        mCurrentIdOrHandle = -1;
    }
#endif

    /**
     * @brief デストラクタ。確保したリソースを開放する。
     */
    ~NoteManager();

    /**
     * @brief 指定されたIDを編集用に指定する。
     * @param iID ID番号
     */
    void    SetEditingID( int id ){
        mHandle = false;
        mCurrentIdOrHandle = id;
        mEditTarget = GetNoteEvent( id );
    };

    /**
     * @brief 指定されたHandleを編集用に指定する。
     * @param iID Handle番号
     */
    void    SetEditingHandle( int handle ){
        mHandle = true;
        mCurrentIdOrHandle = handle;
    };

    /**
     * @brief 指定されたIDに対応するノートイベントへのポインタを探す。
     * @param iID ID番号。
     * @return 成功したらポインタ、失敗したらNULLを返す。
     */
    NoteEvent*    GetNoteEvent(int iID);

    /**
     * @brief 指定されたTick時刻のノートイベントへのポインタを探す。
     * @param nTick Tick時刻。
     * @return 成功したらポインタ、失敗したらNULLを返す。
     */
    NoteEvent*    GetNoteEvent(long nTick);

    /**
     * @brief 指定されたフレーム時刻のノートイベントへのポインタを探す。
     * @param nFrame フレーム時刻。
     * @return 失敗時にはNULL。
     */
    NoteEvent*    GetNoteEventByFrame(long nFrame);

    /**
     * @brief ノート間の連続性をチェックする。
     * @param dTempo 曲のテンポ
     */
#ifdef USE_MULTI_SINGER
    bool    CheckContinuity(double dTempo,vector<UTAUManager*> *pUtauManagers,double dFrameShift);
#else
    bool    CheckContinuity(double dTempo,UTAUManager* pUtauManager,double dFrameShift);
#endif

    /**
     * @brief 末尾の音符のEventIDを返す。
     */
    long    GetEndID(void);

    /**
     * @brief シーケンスデータの開始フレーム時刻を返す。
     */
    long    GetBeginFrame(void);

    /**
     * @brief シーケンスデータの終了フレーム時刻を返す。
     */
    long    GetEndFrame(void);

    long    GetBeginTick(void);

#ifdef USE_MULTI_SINGER
    void    AddSinger( string singer_name ){
        mSingerMap.insert( make_pair( singer_name, mSingerIndex ) );
        mSingerIndex++;
    }
#endif

protected:

    /**
     * @brief 指定された文字列から値を登録する。
     * @param key 設定値のキー文字列 
     * @param value 値
     */
    bool    SetValue( string key, string value );

private:

    /**
     * @brief 指定されたTick時刻にノートイベントを登録する。
     * @param iID ID番号 nTick Tick時刻。
     */
    void    RegisterNote(long nTick,int iID);

    /**
     * @brief 指定されたIDのノートイベントを削除する。
     * @param iTargetID 削除するID番号。
     */
    void    UnregisterNoteByID(int iTargetID);

    /**
     * @brief 指定されたIDに対応するノートイベントの実体を登録する。
     * @param iID ID番号 pNoteEvent 実体へのポインタ。
     */
    void    SetNoteEvent(int iID,NoteEvent* pNoteEvent);

    /**
     * @brief IDに、指定したキーに値を設定する
     * @param key 値を設定するキー
     * @param value 設定する値
     */
    bool    SetValueID( string key, string value );

    /**
     * @brief ハンドルに、指定したキーに値を設定する
     * @param key 値を設定するキー
     * @param value 設定する値
     */
    bool    SetValueHandle( string key, string value );

#ifdef _DEBUG
    /**
     * NoteListの現在の内容を出力します 
     */
    void    PrintNote();
#endif

    vector<NoteEvent *>        mNoteList;         //!< @brief ノート情報のリスト
    MAP_TYPE<int, NoteEvent *> mNoteMap;          //!< @brief IDとノート情報との紐付けを格納したマップ
    MAP_TYPE<int, NoteEvent *> mHandleToNoteMap;  //!< @brief Handleとノート情報との紐付けを格納したマップ
    MAP_TYPE<string, int>      mSingerMap;        //!< @brief 歌手名と歌手リストのインデクスとの紐付けを格納したマップ。

    int           mCurrentIdOrHandle;  //!< @brief 現在編集中のIDまたはHandle番号
    NoteEvent     *mEditTarget;        //!< @brief 現在編集中のノート情報の本体へのポインタ

    bool                  mHandle;              //!< @brief Handleを編集中か否か
    VSQHandleParseContext mHandleParseContext;  //!< @brief ビブラートの解釈を行うためのパースコンテキスト  

#ifdef USE_MULTI_SINGER
    int                 mSingerIndex;
#endif
};

#endif // __VSQNoteManager_h__
