﻿using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Drawing;
using System.Drawing.Extended;
using System.Threading;
using System.Windows.Forms;
using ChaKi.Common;
using ChaKi.Entity.Corpora;
using ChaKi.Entity.Kwic;
using ChaKi.Entity.Search;
using ChaKi.Entity.Corpora.Annotations;

namespace ChaKi.Views.KwicView
{
    /// <summary>
    /// 単一のKwicListに結び付けられた、Kwic/Text Viewerパネル
    /// </summary>
    public partial class KwicViewPanel : UserControl
    {
        #region properties
        public bool WordWrap { get; set; }
        public bool KwicMode { get; set; }
        public bool TwoLineMode { get; set; }

        public int EffectiveWidth
        {
            get { return this.Width - this.vScrollBar1.Width; }
        }

        /// <summary>
        /// 現在表示されている最初の行番号
        /// </summary>
        public int DrawLinesFrom { get; set; }

        /// <summary>
        /// 現在表示されている最後の行番号
        /// </summary>
        public int DrawLinesTo { get; set; }

        public bool SuspendUpdateView { get; set; }

        public static KwicViewSettings Settings;
        #endregion

        #region Private members
        private SearchHistory m_Model;
        private bool m_bNotify;
        private int m_recordsUpdated;

        private List<LineElement> m_LineElements;
        private Font[] m_Fonts;
        private Brush m_NormalBrush;
        private Brush m_HilightBrush;
        private Dictionary<string, Pen> m_SegmentPen;
        public Dictionary<string, Pen> LinkPen;
        private Dictionary<string, Brush> m_GroupBrush;

        public Dictionary<Segment, Rectangle> SegPosCache;  // Segment表示位置のキャッシュ

        private Selection m_CurSelLines;  //選択行番号(m_LineElementsへのインデックス）
        private CharRange m_CurSelCharRange;  // 文字列選択範囲

        private Caret m_Caret;

        private bool m_Dragging;

        private Columns m_Columns;

        private System.Threading.Timer m_Timer;      // 描画更新タイマ
        #endregion

        #region KwicView Events
        public event EventHandler OnPaintDone;
        /// <summary>
        /// Current行が変化したときの親(MainForm)への通知
        /// （子Viewからの中継）
        /// </summary>
        public event CurrentChangedDelegate OnCurrentChanged;
        /// <summary>
        /// Context表示指示時の親(MainForm)への通知 
        /// （子Viewからの中継）
        /// </summary>
        public event RequestContextDelegate OnContextRequested;
        /// <summary>
        /// Guide Panelの表示更新通知 
        /// （子Viewからの中継）
        /// </summary>
        public event UpdateGuidePanelDelegate OnUpdateGuidePanel;
        #endregion

        public KwicViewPanel()
        {
            InitializeComponent();

            m_Fonts = new Font[WordElementStyle.Count];
            FurnishFonts();
            m_NormalBrush = new SolidBrush(Color.Black);
            m_HilightBrush = new SolidBrush(Color.Red);
            m_GroupBrush = new Dictionary<string, Brush>();
            m_GroupBrush.Add("TestGroup", new SolidBrush(Color.FromArgb(150, Color.Pink)));

            m_LineElements = new List<LineElement>();
            this.SegPosCache = new Dictionary<Segment, Rectangle>();
            this.SuspendUpdateView = false;

            this.DrawLinesFrom = 0;
            this.DrawLinesTo = 0;

            m_recordsUpdated = 0;
            m_CurSelLines = new Selection();
            m_CurSelCharRange = new CharRange();
            m_Model = null;

            m_Columns = Columns.Default;

            m_Caret = new Caret(this.pictureBox1);

            // 描画更新タイマ
            TimerCallback timerDelegate = new TimerCallback(this.OnTimer);
            m_Timer = new System.Threading.Timer(timerDelegate, null, 1000, 1000);

            GUISetting.Instance.FontChanged += new EventHandler(OnFontChanged);
        }

        public void SetModel(SearchHistory model)
        {
            DeleteAll();
            m_Model = model;
            if (m_Model != null && m_Model.KwicList != null)
            {
                m_Model.KwicList.OnModelChanged += new UpdateKwicListEventHandler(this.ModelChangedHandler);
                UpdateKwicList();
            }
        }

        public void DeleteAll()
        {
            ResetUpdateStatus();
            m_Model = null;
        }

        /// <summary>
        /// Modelの内容を強制的にInvalidateする。
        /// </summary>
        public void ResetUpdateStatus()
        {
            m_LineElements.Clear();
            m_recordsUpdated = 0;
            m_CurSelCharRange.Clear();
            m_CurSelLines.Clear();
            LineElement.LastIndex = 0;
            WordElement.LastIndex = 0;
        }

        internal void OnWidthChanged(Columns cols)
        {
            if (cols != null)
            {
                m_Columns = cols;
                if (!this.SuspendUpdateView)
                {
                    RecalcLayout();
                    Invalidate(true);
                }
            }
        }

        /// <summary>
        /// ChaKiOptionのフォントが変更されたときのEvent Handler。
        /// 再描画を行う。
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        void OnFontChanged(object sender, EventArgs e)
        {
            FurnishFonts();
            this.pictureBox1.Invalidate(true);
        }

        private void FurnishFonts()
        {
            m_Fonts[WordElementStyle.Normal] = GUISetting.Instance.GetBaseTextFont();
            m_Fonts[WordElementStyle.Small] = new Font(m_Fonts[WordElementStyle.Normal].Name, m_Fonts[WordElementStyle.Normal].Size - 4F);
            m_Fonts[WordElementStyle.Header] = GUISetting.Instance.GetBaseAnsiFont();
            m_Fonts[WordElementStyle.Tiny] = new Font(m_Fonts[WordElementStyle.Header].Name, m_Fonts[WordElementStyle.Header].Size - 3F);

            WordElement.FurnishFonts(m_Fonts[WordElementStyle.Normal], m_Fonts[WordElementStyle.Small]);
            LineElement.AnsiFont = m_Fonts[WordElementStyle.Header];
        }

        public void CalculateLineHeight()
        {
            int lineHeight = m_Fonts[WordElementStyle.Normal].Height + Settings.LineMargin;
            if (this.TwoLineMode) {
                lineHeight += (m_Fonts[WordElementStyle.Small].Height + Settings.LineMargin);
            }
            LineElement.LineHeight = lineHeight;
            WordElement.LineHeight = LineElement.LineHeight - (Settings.LineMargin);
        }

        delegate void UpdateKwicListDelegate();

        static bool TimerGuard = false;

        void OnTimer(object obj)
        {
            if (TimerGuard)
            {
                return;
            }
            if (m_Model == null)
            {
                return;
            }
            TimerGuard = true;
            if (m_bNotify)
            {
                try
                {
                    Invoke(new UpdateKwicListDelegate(this.UpdateKwicList), null);
                }
                catch (Exception ex)
                {
                    Debug.WriteLine(ex);
                }
                m_bNotify = false;
            }
            TimerGuard = false;
        }

        void ModelChangedHandler(object sender, UpdateKwicListEventArgs e)
        {
            if (e.Sorted)
            {
                SynchronizeKwicListOrder();
                RecalcLayout();
                return;
            }
            m_bNotify = true;
            if (e.RedrawNeeded)
            {
                Invalidate(true);
            }
        }

        // m_Model.KwicList.Recordsの順序が変更されたので、
        // lineElementの順序を合わせる（ソートする）
        private void SynchronizeKwicListOrder()
        {
            LineElementComparer comp = new LineElementComparer(m_Model.KwicList.Records);
            m_LineElements.Sort(comp);
            for (int i = 0; i < m_LineElements.Count; i++)
            {
                m_LineElements[i].Index = i;
            }
        }

        public void UpdateKwicList()
        {
            if (m_Model == null || m_Model.KwicList == null)
            {
                return;
            }
            m_Model.AnnotationList.Clear();

            lock (this)
            {
                // Add new LineElements from KwicList starting at 'm_recordsUpdated'
                using (Graphics g = this.pictureBox1.CreateGraphics())
                {
                    // WordBlockのサイズ等を計算する
                    int updateTo = m_Model.KwicList.Records.Count;
                    for (int i = m_recordsUpdated; i < updateTo; i++)
                    {
                        AddKwicItem(m_Model.KwicList.Records[i], g);
                    }
                    m_recordsUpdated = updateTo;
                }
                RecalcLayout();
            }
        }

        private void AddKwicItem(KwicItem ki, Graphics g)
        {
            LineElement lineElement = new LineElement();
            lineElement.KwicItem = ki;

            int charPos = ki.StartCharPos;
            lineElement.AddKwicPortion(ki.Left, g, KwicPortionType.Left, this.TwoLineMode, ref charPos);
            lineElement.AddKwicPortion(ki.Center, g, KwicPortionType.Center, this.TwoLineMode, ref charPos);
            lineElement.AddKwicPortion(ki.Right, g, KwicPortionType.Right, this.TwoLineMode, ref charPos);

            m_LineElements.Add(lineElement);
        }

        public void UpdateAnnotations()
        {
            if (m_Model == null || m_Model.AnnotationList == null)
            {
                return;
            }
            // 単純なSegment->CharRange算出→要改良
            if (m_Model.KwicList.Records.Count == 0) {
                return;
            }
            foreach (LineElement le in m_LineElements)
            {
                le.SegElements.Clear();
                le.LinkElements.Clear();
            }

            Dictionary<Segment,SegElement> segElementTable = new Dictionary<Segment,SegElement>();

            Corpus corpus = m_Model.KwicList.Records[0].Crps;
            foreach (Segment seg in m_Model.AnnotationList.Segments)
            {
                if (seg.StartChar >= seg.EndChar)
                {
                    continue;
                }
                foreach (LineElement le in m_LineElements)
                {
                    if (le.ContainsRange(seg.StartChar, seg.EndChar))
                    {
                        SegElement se = le.AddSegment(seg);
                        if (se != null && !segElementTable.ContainsKey(seg))
                        {
                            segElementTable.Add(seg, se);
                        }
                    }
                }
            }
            foreach (Link lnk in m_Model.AnnotationList.Links)
            {
                LinkElement lke = new LinkElement();
                lke.Link = lnk;
                SegElement se;
                if (segElementTable.TryGetValue(lnk.From, out se))
                {
                    lke.SegFrom = se;
                }
                if (segElementTable.TryGetValue(lnk.To, out se))
                {
                    lke.SegTo = se;
                }

                foreach (LineElement le in m_LineElements)
                {
                    if (le.ContainsRange(lnk.From.StartChar, lnk.From.EndChar)
                     || le.ContainsRange(lnk.To.StartChar, lnk.To.EndChar))
                    {
                        le.AddLinkElement(lke);
                    }
                }
            }

            // SegmentのLevelをアサインする
            AssignSegmentLevels();

            Settings.ShowSegments = true;
            Settings.ShowLinks = true;

            Invalidate(true);
        }


        private void AssignSegmentLevels()
        {
#if false
            // Segmentsは開始点でソートされていること
            // 先頭から重ならないようにLevelをアサインしていく。
            bool bRemain = true;
            short level = 0;
            while (bRemain) {
                bRemain = false;
                int chpos = -1;
                foreach (Segment seg in m_Model.AnnotationList.Segments)
                {
                    if (seg.Level >= 0)
                    {   // Already assigned
                        continue;
                    }
                    if (seg.Range.Start.CharID > chpos)
                    {   // このsegは重ならない
                        seg.Level = level;
                        chpos = seg.Range.End.CharID;
                    }
                    else
                    {   // このlevelではアサインできないSegmentである
                        bRemain = true;
                    }
                }
                level++;
                if (level > 5)
                {
                    throw new Exception("Too many overlapping segments!");
                }
            }
#endif
        }

        public void RecalcLayout()
        {
            // 描画パラメータをLineElement, WordElementにセットしておく
            WordElement.FurnishFonts(m_Fonts[WordElementStyle.Normal], m_Fonts[WordElementStyle.Small]);
            WordElement.RenderTwoLine = this.TwoLineMode;
			LineElement.AnsiFont = m_Fonts[WordElementStyle.Header];
            LineElement.HeaderHeight = m_Fonts[WordElementStyle.Header].Height;
            LineElement.Cols = m_Columns;
            SegElement.L1Height = m_Fonts[WordElementStyle.Normal].Height;
            SegElement.LineHeight = LineElement.LineHeight + Settings.LineMargin;

            // Calculate WordBlock Locations
            Rectangle clientR = this.ClientRectangle;
            int curX = 0;
            int curY = 0;
            int maxWidth = m_Columns.GetMaxSentenceWidth();
            foreach (LineElement le in m_LineElements)
            {
                Rectangle bounds = new Rectangle(0, curY, clientR.Width, curY);

                for (int i = 0; i < le.WordElements.Count; i++)
                {
                    WordElement we = le.WordElements[i];
                    if (this.WordWrap)
                    {
                        if (curX + we.Bounds.Width > maxWidth)
                        {
                            curY += (LineElement.LineHeight + Settings.LineMargin);
                            curX = 0;
                        }
                    }
                    if (le.CenterIndexOffset == 0 && we.IsCenter)
                    {
                        le.CenterIndexOffset = i;
                    }
                    we.Bounds.Location = new Point(curX - bounds.X, curY - bounds.Y);
                    curX += (we.Bounds.Width + Settings.WordMargin);
                }
                // end of sentence
                curY += (LineElement.LineHeight + Settings.LineMargin);
                curX = 0;
                // KWIC Mode --> 行全体を中心語を軸にオフセット
                if (this.KwicMode)
                {
                    Rectangle clientr = pictureBox1.ClientRectangle;
                    int xcenter = m_Columns.GetLeftWidth(); // Centerカラムの開始位置
                    if (le.WordElements.Count > 0)
                    {
                        WordElement wbcen = le.WordElements[le.CenterIndexOffset];
                        // Center wordまでの座標オフセット値
                        int xoffset = wbcen.Bounds.Left - xcenter;
                        foreach (WordElement wb in le.WordElements)
                        {
                            wb.Bounds.Offset(-xoffset, 0);
                        }
                    }
                }
                else
                {
                    foreach (WordElement wb in le.WordElements)
                    {
                        wb.IsCenter = false;
                    }
                }
                bounds.Height = curY - bounds.Top;
                le.Bounds = bounds;
            }

            this.vScrollBar1.Maximum = curY;
            this.vScrollBar1.SmallChange = LineElement.LineHeight;
            this.vScrollBar1.LargeChange = (clientR.Height / LineElement.LineHeight) * LineElement.LineHeight;
            Invalidate(true);
        }

        public void DoKwicViewSettings(Point location)
        {
            KwicViewSettingDialog dlg = new KwicViewSettingDialog();
            KwicViewSettings oldSetting = Settings;
            dlg.View = this;
            dlg.Location = location;
            if (dlg.ShowDialog() != DialogResult.OK)
            { // Revert
                Settings = oldSetting;
            }
            CalculateLineHeight();
            RecalcLayout();
            Invalidate(true);
        }

        /// <summary>
        /// index行目のLineElementを選択状態にする
        /// </summary>
        /// <param name="index"></param>
        public void AlterCheckState(int index)
        {
            m_Model.KwicList.AlterCheckState(index);
        }

        private void pictureBox1_SizeChanged(object sender, EventArgs e)
        {

        }

#if DEBUG
        private static int count = 0;
        private static Stopwatch swatch = new Stopwatch();
#endif

        private void pictureBox1_Paint(object sender, PaintEventArgs e)
        {
#if DEBUG
            swatch.Reset();
            swatch.Start();
#endif
            Graphics g = e.Graphics;
            ExtendedGraphics eg = new ExtendedGraphics(g);
            g.TextRenderingHint = System.Drawing.Text.TextRenderingHint.AntiAlias;
            g.SmoothingMode = System.Drawing.Drawing2D.SmoothingMode.AntiAlias;
            Rectangle clientR = this.ClientRectangle;

            int xoffset = 0;
            int yoffset = 0;

            // カラム境界線を描画する
            for (int i = 0; i < m_Columns.Widths.Length-1; i++)
            {
                xoffset += m_Columns.Widths[i];
                if (i != 5 && i != 6)
                {
                    g.DrawLine(Pens.LightGray, xoffset - 1, 0, xoffset - 1, clientR.Height);
                }
            }

            if (m_Model == null || m_LineElements.Count == 0)
            {
                return;
            }

            SegPosCache.Clear();

            // WordBlockの描画範囲を決定する
            lock (this)
            {
                xoffset = 0;
                yoffset = GetCurrentYOffset();
                this.DrawLinesFrom = 0;
                this.DrawLinesTo = m_LineElements.Count-1;
                for (int i = 0; i < m_LineElements.Count; i++)
                {
                    if (m_LineElements[i].Bounds.Bottom > yoffset)
                    {
                        this.DrawLinesFrom = i;
                        break;
                    }
                }
                for (int i = this.DrawLinesFrom; i < m_LineElements.Count; i++)
                {
                    Rectangle r = m_LineElements[i].Bounds;
                    r.Offset(xoffset, -yoffset);
                    if (r.Top > clientR.Height)
                    {
                        this.DrawLinesTo = i;
                        break;
                    }
                }
            }

            CharRange sel = m_CurSelCharRange.ToAscendingRange();
            bool continueSelectionFlag = false;
#if false // 表示範囲を超えて続く選択範囲をどうするか?
            if (sel.Valid && sel.Start.WordID < this.WordElements[0].Index)
            {
                continueSelectionFlag = true;
            }
#endif
//            Debug.WriteLine(string.Format("Start={0}, End={1}", this.DrawLinesFrom, this.DrawLinesTo));
//            Debug.WriteLine(string.Format("SelStart={0};{1}, SelEnd={2};{3}", 
//                                sel.Start.WordID, sel.Start.CharInWord, sel.End.WordID, sel.End.CharInWord));
            for (int i = this.DrawLinesFrom; i <= this.DrawLinesTo; i++)
            {
                m_LineElements[i].Render(eg, xoffset, yoffset, m_CurSelLines, sel, clientR, ref continueSelectionFlag, Settings);
            }

            #region Group描画
#if false
            if (this.ShowGroups)
            {
                foreach (Group grp in m_Model.Groups)
                {
                    foreach (Segment seg in grp.Members)
                    {
                        Rectangle r;
                        if (!this.SegPosCache.TryGetValue(seg, out r))
                        {
                            continue;
                        }
                        r.Inflate(0, -5);
                        Brush br;
                        if (!m_GroupBrush.TryGetValue(grp.Tag, out br))
                        {
                            continue;
                        }
                        g.FillRectangle(br, r);
                        g.DrawString(string.Format("G{0}", grp.ID), m_TinyFont, Brushes.Black, (float)r.Left, (float)r.Top - 12);
                    }
                }
            }
#endif
            #endregion

            g.ResetClip();


            if (OnPaintDone != null)
            {
                OnPaintDone(this, null);
            }
#if DEBUG
            swatch.Stop();
            Debug.WriteLine(string.Format("pictureBox1_Paint {0} time={1}", count++, swatch.ElapsedMilliseconds));
#endif

            m_Caret.Visible = true;
        }

        private void vScrollBar1_Scroll(object sender, ScrollEventArgs e)
        {
            VScrollBar vs = this.vScrollBar1;
            if (e.NewValue > vs.Maximum || e.NewValue < vs.Minimum)
            {
                return;
            }
            if (vs.Value == e.NewValue)
            {
                return;
            }
            vs.Value = e.NewValue;
            pictureBox1.Refresh();  // PictureBoxはInvalidateだけでなくRefreshしなければ全体が再描画されない。
            pictureBox1.Invalidate();
        }

        private int GetCurrentYOffset()
        {
            return this.vScrollBar1.Value;
        }

        private LineElement HitTestLineElement(Point p, out HitType ht)
        {
            ht = HitType.None;

            for (int n = this.DrawLinesFrom; n <= this.DrawLinesTo; n++)
            {
                if (n >= m_LineElements.Count)
                {
                    return null;
                }
                LineElement le = m_LineElements[n];
                Rectangle r = le.Bounds;
                if (r.Contains(p))
                {
                    int xoffset = p.X - r.Left;
                    if (m_Columns.IsInSentence(xoffset))
                    {
                        ht = HitType.AtSentence;
                    }
                    else if (m_Columns.IsInCheckBox(xoffset))
                    {
                        ht = HitType.AtCheckBox;
                    }
                    else
                    {
                        ht = HitType.AtLine;
                    }
                    return le;
                }
            }
            return null;
        }

        public void ClearSelection()
        {
            ClearLineSelection();
            ClearCharSelection();
        }

        public void ClearLineSelection()
        {
            m_CurSelLines.Clear();
        }

        public void ClearCharSelection()
        {
            m_CurSelCharRange.Clear();
        }

        public void Sort(int colno, SortOrder order)
        {
//            ResetUpdateStatus();
            if (m_Model == null || m_Model.KwicList == null)
            {
                return;
            }
            m_Model.KwicList.Sort(colno, (order == SortOrder.Ascending));
        }

        private void pictureBox1_MouseDown(object sender, MouseEventArgs e)
        {
            this.vScrollBar1.Focus(); // このPanelのトップレベルコントロールであるvscrollがキーイベントを受け取るようにする

            m_Caret.Create(m_Fonts[0].Height);

            if (e.Button == MouseButtons.Left)
            {
                ClearCharSelection();
                // 選択開始Boxを検索する
                Point p = e.Location;
                p.Offset(0, GetCurrentYOffset());
                HitType ht;
                LineElement le = HitTestLineElement(p, out ht);
                switch (ht)
                {
                    case HitType.AtCheckBox:
                        ClearLineSelection();
                        m_CurSelLines.AddSelection(le.Index);
                        AlterCheckState(le.Index);
                        break;
                    case HitType.AtLine:
                        // カレント行を変更する
                        ClearLineSelection();
                        m_CurSelLines.AddSelection(le.Index);
                        break;
                    case HitType.AtSentence:
                        if (le != null)
                        {
                            Size leBase = new Size(le.Bounds.Location);
                            p -= leBase;
                            WordElement we = le.HitTestWordElement(p);
                            if (we != null)
                            {
                                // 選択開始文字を特定して、ドラッグモードに入る
                                Size weBase = new Size(m_Columns.GetSentenceOffset() + we.Bounds.X, we.Bounds.Y);
                                p -= weBase;
                                Point at;
                                m_CurSelCharRange.Start = we.HitTestChar(we, p, out at);
                                at += (leBase + weBase);
                                m_Caret.Location = at;
                                m_CurSelCharRange.End = m_CurSelCharRange.Start;
                                if (m_CurSelCharRange.Start != null)
                                {
                                    m_Dragging = true;
                                }
                            }
                        }
                        break;
                }
            }
            else if (e.Button == MouseButtons.Right)
            {
                // 設定変更ダイアログを出す
                DoKwicViewSettings(this.pictureBox1.PointToScreen(e.Location));
            }
            Invalidate(true);
        }

        private void pictureBox1_MouseDoubleClick(object sender, MouseEventArgs e)
        {
            if (e.Button == MouseButtons.Left)
            {
                Point p = e.Location;
                p.Offset(0, GetCurrentYOffset());
                HitType ht;
                LineElement le = HitTestLineElement(p, out ht);
                switch (ht)
                {
                    case HitType.AtLine:
                        // 選択行を変更する
                        ClearLineSelection();
                        m_CurSelLines.AddSelection(le.Index);
                        if (OnContextRequested != null)
                        {
                            OnContextRequested(this.m_Model.KwicList, le.Index);
                        }
                        if (OnCurrentChanged != null)
                        {
                            KwicItem ki = m_Model.KwicList.Records[le.Index];
                            OnCurrentChanged(ki.Crps, ki.SenPos);
                        }
                        break;
                    case HitType.AtSentence:
                        if (le != null)
                        {
                            p.Offset(-le.Bounds.X, -le.Bounds.Y);
                            WordElement we = le.HitTestWordElement(p);
                            if (we != null)
                            {
                                // １語全体を選択状態にする
                                ClearCharSelection();
                                m_CurSelCharRange.Start = new CharPos(-1, we.Index, 0);
                                m_CurSelCharRange.End = new CharPos(-1, we.Index, we.Length);
                                m_Dragging = false;
                            }
                        }
                        break;
                }
            }

            Invalidate(true);
        }

        private void pictureBox1_MouseUp(object sender, MouseEventArgs e)
        {
            if (e.Button == MouseButtons.Left)
            {
                if (m_Dragging)
                {
                    if (m_CurSelCharRange.Valid)
                    {
                        if (m_CurSelCharRange.IsEmpty)
                        {
                            ClearCharSelection();
                        }
                    }
                    else
                    {
                        ClearCharSelection();
                    }
                    m_Dragging = false;
                    pictureBox1.Invalidate();
                }
            }
        }

        private void pictureBox1_MouseMove(object sender, MouseEventArgs e)
        {
            Point p = e.Location;
            p.Offset(0, GetCurrentYOffset());
            WordElement we = null;
            HitType ht;
            LineElement le = HitTestLineElement(p, out ht);
            if (le != null)
            {
                p.Offset(-le.Bounds.X,-le.Bounds.Y);
                we = le.HitTestWordElement(p);
            }
            if (we == null)
            {
                this.pictureBox1.Cursor = Cursors.Arrow;
                return;
            }
            this.pictureBox1.Cursor = Cursors.IBeam;
            // Guide Panelへのイベント通知
            if (this.OnUpdateGuidePanel != null && we.KwicWord.Lex != null)
            {
                this.OnUpdateGuidePanel(we.KwicWord.Lex);
            }
            if (m_Dragging)
            {
                // 選択終了位置を更新する
                p.Offset(-m_Columns.GetSentenceOffset() - we.Bounds.X, -we.Bounds.Y);
                Point at;
                CharPos cs = we.HitTestChar(we, p, out at);
                if (cs != null)
                {
                    m_CurSelCharRange.End = cs;
                }

                pictureBox1.Invalidate();
            }
        }
    }
}
