﻿using System.Collections.Generic;
using System.Drawing;
using ChaKi.Entity.Corpora;
using ChaKi.Entity.Kwic;
using System.Diagnostics;
using System;
using System.Windows.Forms;

namespace ChaKi.Views.KwicView
{
    internal class WordElement
    {
        public int StartChar;      // Corpus内の文字位置
        public int EndChar;        // Corpus内の文字位置
        public Rectangle Bounds;
        public Point L1Location;    // rに対する1行目（表層）の描画開始位置(LeftTop)
        public Point L2Location;    // rに対する2行目（POS）の描画開始位置(LeftTop)
        public KwicWord KwicWord;
        public bool IsCenter;
        public KwicItem KwicItem;   // source of this object
        public int Index;   // index of word in sentence
        public List<RectangleF> CharPos;

        public static int LastIndex { get; set; }
        public static int LineHeight { get; set; }
        public static bool RenderTwoLine { get; set; }

        private static Brush NormalBrush { get; set; }
        private static Brush HilightBrush { get; set; }
        private static Font L1FontNormal { get; set; }
        private static Font L2FontNormal { get; set; }
        private static Font L1FontBold { get; set; }
        private static Font L2FontBold { get; set; }
        private static Brush SelectionBrush;
        
       
        static WordElement()
        {
        	sformat = new StringFormat() { Alignment = StringAlignment.Near, LineAlignment = StringAlignment.Near };
            NormalBrush = Brushes.Black;
            HilightBrush = Brushes.Red;
            SelectionBrush = new SolidBrush(Color.FromArgb(50, Color.Blue));
        }

        public WordElement()
        {
            StartChar = 0;
            EndChar = 0;
            CharPos = null;
            IsCenter = false;
            this.Index = LastIndex++;
        }

        public static void FurnishFonts(Font l1Font, Font l2Font)
        {
            if (L1FontBold != null) L1FontBold.Dispose();
            if (L2FontBold != null) L2FontBold.Dispose();

            L1FontNormal = l1Font;
            L2FontNormal = l2Font;
            L1FontBold = new Font(l1Font, FontStyle.Bold);
            L2FontBold = new Font(l2Font, FontStyle.Bold);
        }

        public void MeasureStrings(Graphics g, out SizeF sz1, out SizeF sz2)
        {
            sz1 = g.MeasureString(this.L1String, L1FontNormal);
            sz2 = g.MeasureString(this.L2String, L2FontNormal);
        }

        public void Render(Graphics g, int xoffset, int yoffset, CharRange sel, ref bool continueSelectionFlag)
        {
            if (this.L1String == null)
            {
                return;
            }
            xoffset += this.Bounds.X;
            yoffset += this.Bounds.Y;
            Font l1Font = L1FontNormal;
            Font l2Font = L2FontNormal;
            Brush brush = NormalBrush;
            if ((this.KwicWord.ExtAttr & KwicWord.KWA_HILIGHT) != 0)
            {
                l1Font = L1FontBold;
                l2Font = L2FontBold;
            }
            if ((this.KwicWord.ExtAttr & KwicWord.KWA_PIVOT) != 0)
            {
                l1Font = L1FontBold;
                l2Font = L2FontBold;
                brush = HilightBrush;
            }
            g.DrawString(this.L1String, l1Font, brush, this.L1Location.X + xoffset, this.L1Location.Y + yoffset);
            if (RenderTwoLine)
            {
                g.DrawString(this.L2String, l2Font, brush, this.L2Location.X + xoffset, this.L2Location.Y + yoffset);
            }

            if (sel.Valid && this.Length > 0)
            {
                #region Selection描画
                if (continueSelectionFlag)
                {
                    if (sel.End.WordID > this.Index)
                    {
                        // fullly select this WordElement
                        Rectangle rs = new Rectangle(xoffset, yoffset, this.Bounds.Width, WordElement.L1FontNormal.Height);
                        g.FillRectangle(SelectionBrush, rs);
                    }
                    else if (sel.End.WordID == this.Index)
                    {
                        // selection ends in this WordElement
                        Rectangle rs = new Rectangle(xoffset, yoffset, this.IndexToOffset(sel.End.CharInWord), WordElement.L1FontNormal.Height);
                        g.FillRectangle(SelectionBrush, rs);
                        continueSelectionFlag = false;
                    }
                }
                else
                {
                    if (sel.Start.WordID == this.Index)
                    {
                        if (sel.End.WordID == this.Index)
                        {
                            // selection starts and ends in this WordElement
                            Rectangle rs = GetRangeRectangle(sel);
                            rs.Height = WordElement.L1FontNormal.Height;
                            rs.Offset(xoffset, yoffset);
                            g.FillRectangle(SelectionBrush, rs);
                        }
                        else
                        {
                            // selection starts in this WordElement and continues
                            int x = IndexToOffset(sel.Start.CharInWord);
                            Rectangle rs = new Rectangle(x + xoffset, yoffset, this.Bounds.Width - x, WordElement.L1FontNormal.Height);
                            g.FillRectangle(SelectionBrush, rs);
                            continueSelectionFlag = true;
                        }
                    }
                }
                #endregion
            }
        }

        public CharPos HitTestChar(WordElement wb, Point p, out Point at)
        {
            for (int i = 0; i < this.CharPos.Count; i++)
            {
                RectangleF rect = this.CharPos[i];
                if (rect.Contains(p))
                {
                    if (p.X < rect.Width / 2)
                    {
                        // ヒットした文字の左
                        at = new Point((int)rect.Left, (int)rect.Top);
                        return new CharPos(-1, this.Index, i);
                    }
                    // ヒットした文字の右
                    if (i+1 >= this.Length)
                    {
                        at = new Point((int)rect.Right, (int)rect.Top);
                        return new CharPos(-1, this.Index + 1, 0);
                    }
                    at = new Point((int)rect.Left, (int)rect.Top);
                    return new CharPos(-1, this.Index, i + 1);
                }
            }
            at = Point.Empty;
            return null;
        }


        public void Invalidate()
        {
            CharPos = null;
        }

        public int Length
        {
            get
            {
                return this.L1String.Length;
            }
        }

        public string L1String
        {
            get
            {
                if (this.KwicWord.Lex != null)
                {
                    return this.KwicWord.Lex.Surface;
                }
                return this.KwicWord.Text;
            }
        }

        public string L2String
        {
            get
            {
                if (this.KwicWord.Lex != null)
                {
                    return this.KwicWord.Lex.PartOfSpeech.Name1;
                }
                return this.KwicWord.Text;
            }
        }

        public void CalculateCharPos(Graphics g)
        {
            if (this.CharPos != null)
            {
                return;
            }
            this.CharPos = new List<RectangleF>();
            string s = this.L1String;
            float x = 0F;
            for (int i = 0; i < s.Length; i++) {
                Size sz = TextRenderer.MeasureText(
                    s.Substring(i,1),
                    GUISetting.Instance.GetBaseTextFont(),
                    new Size(int.MaxValue, int.MaxValue),
                    TextFormatFlags.Default );
                this.CharPos.Add(new RectangleF(x,0F,(float)sz.Width,(float)sz.Height));
                x += (float)sz.Width;
            }
        }

        static private Dictionary<int, CharacterRange[]> characterRangesMap = new Dictionary<int, CharacterRange[]>();
        private static StringFormat sformat;

        public void CalculateCharPos2(Graphics g)
        {
            if (this.CharPos != null || this.L1String == null)
            {
                return;
            }
            this.CharPos = new List<RectangleF>();
            string s = this.L1String;
            Rectangle rect = this.Bounds;
            rect.Offset(-Bounds.Left, -Bounds.Top);
            rect.Width += 100;   // 文字が途切れないように
            CharacterRange[] characterRanges;
            if (!characterRangesMap.TryGetValue(s.Length, out characterRanges))
            {
                characterRanges = new CharacterRange[s.Length];
                characterRangesMap[s.Length] = characterRanges;
                for (int i = 0; i < s.Length; i++)
                {
                    characterRanges[i] = new CharacterRange(i, 1);
                }
            }
            try
            {
                sformat.SetMeasurableCharacterRanges(characterRanges);
                Region[] rgns = g.MeasureCharacterRanges(s, GUISetting.Instance.GetBaseTextFont(), rect, sformat);
                for (int i = 0; i < rgns.Length; i++)
                {
                    RectangleF r = rgns[i].GetBounds(g);
                    r.Height = WordElement.LineHeight;
                    this.CharPos.Add(r);
                    rgns[i].Dispose();
                }
            }
            catch (Exception ex)
            {
                Debug.WriteLine(ex);
                for (int i = 0; i < s.Length; i++)
                {
                    this.CharPos.Add(new RectangleF());
                }
            }
        }

        public int IndexToOffset(int index)
        {
            if (index == this.CharPos.Count)
            {
                return (int)this.CharPos[index-1].Right;
            }
            return (int)this.CharPos[index].Left;
        }

        public Rectangle GetRangeRectangle(CharRange range)
        {
            int index1 = range.Start.CharInWord;
            int index2 = range.End.CharInWord;

            if (index2 == this.CharPos.Count)
            {
                return new Rectangle(
                    (int)this.CharPos[index1].Left,
                    0,
                    (int)this.CharPos[index2 - 1].Right - (int)this.CharPos[index1].Left,
                    20);
            }
            return new Rectangle(
                (int)this.CharPos[index1].Left,
                0,
                (int)this.CharPos[index2].Left - (int)this.CharPos[index1].Left,
                20);
        }
    }

}
