using System;
using System.Collections.Generic;
using System.Text;

namespace ChaKi.Entity.Corpora
{
    public enum LP
    {
        Surface = 0,
        Reading = 1,
        Pronunciation = 2,
        BaseLexeme = 3,
        PartOfSpeech = 4,
        CType = 5,
        CForm = 6,
    }

    public class Lexeme : IComparable
    {
        private Dictionary<LP, object> properties;
        private int frequency;

        /// <summary>
        /// MIXRA̎ZoŎgpꎞIȃ}[LOtO
        /// ꎞIuWFNgƂĂLexemeł̂ݎgpAR[pXƌтĂLexemeł͏false.
        /// </summary>
        public bool Marked { get; set; }

        public virtual int ID { get; set; }

        /// <summary>
        /// ꑮ̖́iLexemeNX̃vpeBƈv邱Ɓj
        /// </summary>
        public static readonly Dictionary<LP,string> PropertyName =
            new Dictionary<LP, string> {
                {LP.Surface, "Surface" },
                {LP.Reading, "Reading" },
                {LP.Pronunciation, "Pronunciation" },
                {LP.BaseLexeme, "BaseLexeme" },
                {LP.PartOfSpeech, "PartOfSpeech" },
                {LP.CType, "CType" },
                {LP.CForm, "CForm" }
            };

        public static int CompareKey = (int)LP.Surface;
        public static bool CompareAscending = true;

        public Lexeme() 
        {
            this.properties = new Dictionary<LP, object>();
            this.properties.Add(LP.Surface, string.Empty);
            this.properties.Add(LP.Reading, string.Empty);
            this.properties.Add(LP.Pronunciation, string.Empty);
            this.properties.Add(LP.BaseLexeme, null);
            this.properties.Add(LP.PartOfSpeech, null);
            this.properties.Add(LP.CType, null);
            this.properties.Add(LP.CForm, null);

            this.Marked = false;
        }

        public Lexeme(Lexeme org)
            : this()
        {
            if (org == null)
            {
                return;
            }
            this.properties = new Dictionary<LP, object>();
            foreach (KeyValuePair<LP, object> pair in org.properties)
            {
                this.properties.Add(pair.Key, pair.Value);
            }
            this.frequency = org.frequency;
            this.ID = org.ID;
            this.Marked = org.Marked;
        }

        public int CharLength
        {
            get { return this.Surface.Length; }
        }

        public virtual string Surface
        {
            get { return properties[LP.Surface] as string; }
            set { properties[LP.Surface] = value; }
        }
        public virtual string Reading
        {
            get { return properties[LP.Reading] as string; }
            set { properties[LP.Reading] = value; }
        }
        public virtual string Pronunciation
        {
            get { return properties[LP.Pronunciation] as string; }
            set { properties[LP.Pronunciation] = value; }
        }
        public virtual Lexeme BaseLexeme
        {
            get { return properties[LP.BaseLexeme] as Lexeme; }
            set { properties[LP.BaseLexeme] = value; }
        }
        public virtual PartOfSpeech PartOfSpeech
        {
            get { return properties[LP.PartOfSpeech] as PartOfSpeech; }
            set { properties[LP.PartOfSpeech] = value; }
        }
        public virtual CType CType
        {
            get { return properties[LP.CType] as CType; }
            set { properties[LP.CType] = value; }
        }
        public virtual CForm CForm
        {
            get { return properties[LP.CForm] as CForm; }
            set { properties[LP.CForm] = value; }
        }
        public virtual int Frequency
        {
            get { return this.frequency; }
            set { this.frequency = value; }
        }

        public virtual object GetProperty( LP tag )
        {
            return properties[tag];
        }

        public virtual string GetStringProperty(LP tag)
        {
            string s = string.Empty;
            switch (tag)
            {
                case LP.Surface:
                    s = this.Surface;
                    break;
                case LP.Reading:
                    s = this.Reading;
                    break;
                case LP.BaseLexeme:
                    if (this.BaseLexeme != null) s = this.BaseLexeme.Surface;
                    break;
                case LP.Pronunciation:
                    s = this.Pronunciation;
                    break;
                case LP.PartOfSpeech:
                    if (this.PartOfSpeech != null) s = this.PartOfSpeech.Name;
                    break;
                case LP.CType:
                    if (this.CType != null) s = this.CType.Name;
                    break;
                case LP.CForm:
                    if (this.CForm != null) s = this.CForm.Name;
                    break;
            }
            if (this.Marked)
            {
                return ("<<" + s + ">>");
            }
            return s;
        }

        public virtual string GetStringPropertyShort(LP tag)
        {
            string s = string.Empty;
            switch (tag)
            {
                case LP.Surface:
                    s = this.Surface;
                    break;
                case LP.Reading:
                    s = this.Reading;
                    break;
                case LP.BaseLexeme:
                    if (this.BaseLexeme != null) s = this.BaseLexeme.Surface;
                    break;
                case LP.Pronunciation:
                    s =  this.Pronunciation;
                    break;
                case LP.PartOfSpeech:
                    if (this.PartOfSpeech != null) s = this.PartOfSpeech.Name1;   //Zk`
                    break;
                case LP.CType:
                    if (this.CType != null) s = this.CType.Name;
                    break;
                case LP.CForm:
                    if (this.CForm != null) s = this.CForm.Name;
                    break;
            }
            if (this.Marked)
            {
                return ("<<" + s + ">>");
            }
            return s;
        }

        public override string ToString()
        {
            StringBuilder sb = new StringBuilder();
            sb.AppendFormat("{0},", GetStringProperty(LP.Surface));
            sb.AppendFormat("{0},", GetStringProperty(LP.Reading));
            sb.AppendFormat("{0},", GetStringProperty(LP.Pronunciation));
            sb.AppendFormat("{0},", GetStringProperty(LP.PartOfSpeech));
            sb.AppendFormat("{0},", GetStringProperty(LP.BaseLexeme));
            sb.AppendFormat("{0},", GetStringProperty(LP.CType));
            sb.AppendFormat("{0}", GetStringProperty(LP.CForm));
            return sb.ToString();
        }

        // Reading, Pronunciation\
        // {`̈vr݂̂Ɏgp
        public string ToString2()
        {
            StringBuilder sb = new StringBuilder();
            sb.AppendFormat("{0},,,", GetStringProperty(LP.Surface));
            sb.AppendFormat("{0},", GetStringProperty(LP.PartOfSpeech));
            sb.AppendFormat("{0},", GetStringProperty(LP.BaseLexeme));
            sb.AppendFormat("{0},", GetStringProperty(LP.CType));
            sb.AppendFormat("{0}", GetStringProperty(LP.CForm));
            return sb.ToString();
        }

        public int CompareTo(object obj)
        {
            Lexeme compareTo = (Lexeme)obj;
            if (CompareKey == 7)    // Compare by Frequency
            {
                if (CompareAscending)
                {
                    return this.Frequency - compareTo.Frequency;
                }
                else
                {
                    return compareTo.Frequency - this.Frequency;
                }
            }
            string s1 = GetStringProperty((LP)CompareKey);
            string s2 = compareTo.GetStringProperty((LP)CompareKey);
            if (CompareAscending)
            {
                return s1.CompareTo(s2);
            }
            return s2.CompareTo(s1);
        }

        /// <summary>
        /// LexemëꕔAComparen/aƂȂ悤ɋɂB
        /// ꎞILexemeɂ̂ݎgpB
        /// </summary>
        /// <param name="filter"></param>
        public void ApplyFilter(LexemeFilter filter)
        {
            if (filter.IsFiltered(LP.Surface))
            {
                this.Surface = null;
            }
            if (filter.IsFiltered(LP.Reading))
            {
                this.Reading = null;
            }
            if (filter.IsFiltered(LP.Pronunciation))
            {
                this.Pronunciation = null;
            }
            if (filter.IsFiltered(LP.BaseLexeme))
            {
                this.BaseLexeme = null;
            }
            if (filter.IsFiltered(LP.PartOfSpeech))
            {
                this.PartOfSpeech = null;
            }
            if (filter.IsFiltered(LP.CType))
            {
                this.CType = null;
            }
            if (filter.IsFiltered(LP.CForm))
            {
                this.CForm = null;
            }
        }
    }
}
