﻿using System;
using System.Collections.Generic;
using System.Text;
using System.Diagnostics;

namespace ChaKi.Entity.Search
{
    public class DepSearchCondition : ISearchCondition, ICloneable
    {
        public List<TagSearchCondition> BunsetsuConds { get; set; }
        public List<LinkCondition> LinkConds { get; set; }

        public event EventHandler OnModelChanged;

        public DepSearchCondition()
        {
            this.BunsetsuConds = new List<TagSearchCondition>();
            this.LinkConds = new List<LinkCondition>();
        }

        public DepSearchCondition(DepSearchCondition src)
        {
            this.BunsetsuConds = new List<TagSearchCondition>();
            foreach (TagSearchCondition bc in src.BunsetsuConds)
            {
                this.BunsetsuConds.Add(new TagSearchCondition(bc));
            }
            this.LinkConds = new List<LinkCondition>();
            foreach (LinkCondition lc in src.LinkConds)
            {
                this.LinkConds.Add(new LinkCondition(lc));
            }
            OnModelChanged = src.OnModelChanged;
        }

        public object Clone()
        {
            return new DepSearchCondition(this);
        }

        public void Reset()
        {
            this.BunsetsuConds.Clear();
            TagSearchCondition tcond = new TagSearchCondition();
            this.BunsetsuConds.Add(tcond);

            this.LinkConds.Clear();

            if (OnModelChanged != null) OnModelChanged(this, null);
        }

        public void AddLinkConstraint(LinkCondition lc)
        {
            this.LinkConds.Add(lc);
        }

        public void RemoveLinkConstraint(LinkCondition lc)
        {
            this.LinkConds.Remove(lc);
        }

        public TagSearchCondition InsertAt(int pos)
        {
            TagSearchCondition item = null;
            try
            {
                item = new TagSearchCondition();
                BunsetsuConds.Insert(pos, item);
            }
            catch (ArgumentOutOfRangeException e)
            {
                Debug.WriteLine(e);
                return null;
            }
            if (OnModelChanged != null) OnModelChanged(this, null);
            return item;
        }

        public void RemoveAt(int i)
        {
            BunsetsuConds.RemoveAt(i);
            if (OnModelChanged != null) OnModelChanged(this, null);
        }

        public void SetConnection(int bunsetsu_pos, char conn_char)
        {
            TagSearchCondition tcond;
            // bunsetsu_posの左側
            if (bunsetsu_pos >= 0 && bunsetsu_pos < BunsetsuConds.Count)
            {
                tcond = BunsetsuConds[bunsetsu_pos];
                tcond.LeftConnection = conn_char;
            }
            // bunsetsu_pos-1の右側
            if (bunsetsu_pos-1 >= 0 && bunsetsu_pos-1 < BunsetsuConds.Count)
            {
                tcond = BunsetsuConds[bunsetsu_pos - 1];
                tcond.RightConnection = conn_char;
            }
            // View updateは行わない
        }


        public void InsertLexemeConditionAt(int bunsetsu_pos, int property_pos)
        {
            try
            {
                TagSearchCondition item = BunsetsuConds[bunsetsu_pos];
                item.InsertAt(property_pos);    // itemにcallbackを設定していないため、ここではModel Updateは起きない。
            }
            catch (ArgumentOutOfRangeException e)
            {
                Debug.WriteLine(e);
                return;
            }
            if (OnModelChanged != null) OnModelChanged(this, null);
        }

        /// <summary>
        /// このDepSearch条件に含まれるPivotを返す。
        /// </summary>
        /// <returns>Pivotがあれば[0..LexemeConds.Count-1]のいずれかの値。なければ-1</returns>
        public LexemeCondition GetPivot()
        {
            foreach (TagSearchCondition item in this.BunsetsuConds)
            {
                LexemeCondition c = item.GetPivot();
                if (c != null)
                {
                    return c;
                }
            }
            return null;
        }

        /// <summary>
        /// このDepSearch条件に含まれるPivotの位置を返す。
        /// </summary>
        /// <returns>Pivotがあれば[0..LexemeConds.Count-1]のいずれかの値。なければ-1</returns>
        public int GetPivotPos()
        {
            int i = 0;
            foreach (TagSearchCondition tcond in this.BunsetsuConds)
            {
                for (int j = 0; j < tcond.LexemeConds.Count; j++)
                {
                    LexemeCondition lcond = tcond.LexemeConds[j];
                    if (lcond.IsPivot)
                    {
                        return i;
                    }
                }
                i++;
            }
            return -1;
        }

        /// <summary>
        /// 文節を無視して、前からこの条件に含まれるすべてのLexeme条件をリストする
        /// </summary>
        /// <returns></returns>
        public List<LexemeCondition> GetLexemeCondList()
        {
            List<LexemeCondition> res = new List<LexemeCondition>();
            foreach (TagSearchCondition tcond in this.BunsetsuConds)
            {
                res.AddRange(tcond.LexemeConds);
            }
            return res;
        }
    }
}
