﻿using System;
using System.Collections.Generic;
using System.Text;
using System.Data.Common;
using ChaKi.Service.Properties;
using ChaKi.Entity.Corpora;
using ChaKi.Service.Search;
using NHibernate;
using System.Collections;
using ChaKi.Entity;


namespace ChaKi.Service.Database
{
    public abstract class DBService
    {
        public DBParameter DBParam { get; set; }

        /// <summary>
        /// DBServiceをパラメータから作成するシンプルファクトリ
        /// </summary>
        /// <param name="dbms"></param>
        /// <returns></returns>
        static public DBService Create(DBParameter dbparam)
        {
            DBService svc = null;
            if (dbparam.DBType.Equals("MySQL"))
            {
                svc = new MySQLDBService();
            }
            else if (dbparam.DBType.Equals("SQLServer"))
            {
                svc = new MSSQLDBService();
            }
            else if (dbparam.DBType.Equals("SQLite"))
            {
                svc = new SQLiteDBService();
            }
            else if (dbparam.DBType.Equals("PostgreSQL"))
            {
                svc = new PgSQLDBService();
            }
            else 
            {
                svc = new MySQLDBService();    // Default server
            }
            svc.DBParam = dbparam;
            return svc;
        }

        protected DBService()
        {
        }

        /// <summary>
        /// DBに接続して、データベースの一覧を取得する
        /// </summary>
        /// <param name="server"></param>
        /// <param name="user"></param>
        /// <param name="password"></param>
        /// <param name="dblist"></param>
        public abstract void GetDatabaseList(ref List<string> dblist);

        /// <summary>
        /// NHibernate用の設定を得る
        /// </summary>
        /// <returns></returns>
        public NHibernate.Cfg.Configuration GetConnection()
        {
            NHibernate.Cfg.Configuration cfg = SearchConfiguration.GetInstance().NHibernateConfig;
            SetupConnection(cfg);
            return cfg;
        }

        /// <summary>
        /// NHibernateの接続設定を行う
        /// </summary>
        /// <param name="cfg"></param>
        public abstract void SetupConnection(NHibernate.Cfg.Configuration cfg);

        /// <summary>
        /// ADO.NETの接続文字列を得る
        /// </summary>
        /// <returns></returns>
        public abstract string GetConnectionString();

        /// <summary>
        /// Insert文のデフォルト文字列を得る
        /// </summary>
        /// <returns></returns>
        public virtual string GetDefault()
        {
            return "DEFAULT";
        }

        /// <summary>
        /// コーパス名に基づきデータベースを作成する
        /// </summary>
        public abstract void CreateDatabase();

        /// <summary>
        /// データベースを削除する
        /// </summary>
        public abstract void DropDatabase();

        /// <summary>
        /// コーパスを構成するテーブルを全削除する
        /// </summary>
        public virtual void DropAllTables()
        {
            List<string> statements = new List<string>();
            statements.Add(Resources.DropTableCorpusAttributesStatement);
            statements.Add(Resources.DropTablePartsOfSpeechStatement);
            statements.Add(Resources.DropTableCTypesStatement);
            statements.Add(Resources.DropTableCFormsStatement);
            statements.Add(Resources.DropTableCFormsCTypesStatement);
            statements.Add(Resources.DropTableLexemesStatement);
            statements.Add(Resources.DropTableSentencesStatement);
            statements.Add(Resources.DropTableWordsStatement);
            statements.Add(Resources.DropTableBunsetsusStatement);
            statements.Add(Resources.DropTableDocumentTagsStatement);
            statements.Add(Resources.DropTableSentencesDocumentTagsStatement);

            statements.Add(Resources.DropTableProjects);
            statements.Add(Resources.DropTableProjectsTagsets);
            statements.Add(Resources.DropTableDocuments);
            statements.Add(Resources.DropTableDocumentsets);
            statements.Add(Resources.DropTableTagsetVersions);
            statements.Add(Resources.DropTableTagsetDefinitions);
            statements.Add(Resources.DropTableTagDefinitions);
            statements.Add(Resources.DropTableSegmentsStatement);
            statements.Add(Resources.DropTableLinksStatement);
            statements.Add(Resources.DropTableGroupsStatement);
            statements.Add(Resources.DropTableSegmentsGroupsStatement);

            DoStringCommands(statements);
        }

        /// <summary>
        /// コーパスに必要なテーブルを作成する
        /// </summary>
        public virtual void CreateAllTables()
        {
            List<string> statements = new List<string>();
            statements.Add(Resources.CreateTableCorpusAttributesStatement);
            statements.Add(Resources.CreateTablePartsOfSpeechStatement);
            statements.Add(Resources.CreateTableCTypesStatement);
            statements.Add(Resources.CreateTableCFormsStatement);
            statements.Add(Resources.CreateTableCFormsCTypesStatement);
            statements.Add(Resources.CreateTableLexemesStatement);
            statements.Add(Resources.CreateTableSentencesStatement);
            statements.Add(Resources.CreateTableWordsStatement);
            statements.Add(Resources.CreateTableBunsetsusStatement);
            statements.Add(Resources.CreateTableDocumentTagsStatement);
            statements.Add(Resources.CreateTableSentencesDocumentTagsStatement);

            statements.Add(Resources.CreateTableProjects);
            statements.Add(Resources.CreateTableProjectsTagsets);
            statements.Add(Resources.CreateTableDocuments);
            statements.Add(Resources.CreateTableDocumentsets);
            statements.Add(Resources.CreateTableTagsetVersions);
            statements.Add(Resources.CreateTableTagsetDefinitions);
            statements.Add(Resources.CreateTableTagDefinitions);
            statements.Add(Resources.CreateTableSegmentsStatement);
            statements.Add(Resources.CreateTableLinksStatement);
            statements.Add(Resources.CreateTableGroupsStatement);
            statements.Add(Resources.CreateTableSegmentsGroupsStatement);

            DoStringCommands(statements);
        }

        /// <summary>
        /// テーブルにインデックスを作成する
        /// </summary>
        public virtual void CreateAllIndices()
        {
            List<string> statements = new List<string>();
            statements.Add(Resources.CreateIndexCorpusAttributesStatement);
            statements.Add(Resources.CreateIndexPartsOfSpeechStatement);
            statements.Add(Resources.CreateIndexCTypesStatement);
            statements.Add(Resources.CreateIndexCFormsStatement);
            statements.Add(Resources.CreateIndexCFormsCTypesStatement);
            statements.Add(Resources.CreateIndexLexemesStatement1);
            statements.Add(Resources.CreateIndexLexemesStatement2);
            statements.Add(Resources.CreateIndexLexemesStatement3);
            statements.Add(Resources.CreateIndexLexemesStatement4);
            statements.Add(Resources.CreateIndexLexemesStatement5);
            statements.Add(Resources.CreateIndexLexemesStatement6);
            statements.Add(Resources.CreateIndexLexemesStatement7);
            statements.Add(Resources.CreateIndexSentencesStatement1);
            statements.Add(Resources.CreateIndexSentencesStatement2);
            //            statements.Add(Resources.CreateIndexWordsStatement1);
            //            statements.Add(Resources.CreateIndexWordsStatement2);
            statements.Add(Resources.CreateIndexBunsetsusStatement);
            statements.Add(Resources.CreateIndexSegmentsStatement1);
            statements.Add(Resources.CreateIndexSegmentsStatement2);
            statements.Add(Resources.CreateIndexSegmentsStatement3);
            statements.Add(Resources.CreateIndexLinksStatement1);
            statements.Add(Resources.CreateIndexLinksStatement2);
            statements.Add(Resources.CreateIndexLinksStatement3);
            statements.Add(Resources.CreateIndexLinksStatement4);
            statements.Add(Resources.CreateIndexGroupsStatement);
            statements.Add(Resources.CreateIndexSegmentsGroupsStatement1);
            statements.Add(Resources.CreateIndexSegmentsGroupsStatement2);
            statements.Add(Resources.CreateIndexSentencesDocumenttagsStatement1);

            DoStringCommands(statements);
        }

        public virtual void CreateFinalIndices()
        {
            List<string> statements = new List<string>();
            statements.Add(Resources.CreateIndexWordsStatement1);
            statements.Add(Resources.CreateIndexWordsStatement2);

            DoStringCommands(statements);
        }

        /// <summary>
        /// 単純なSQLコマンド(複数)を順次発行する
        /// </summary>
        /// <param name="statements"></param>
        protected void DoStringCommands(List<string> statements)
        {
            using (DbConnection cnn = this.GetDbConnnection())
            using (DbCommand cmd = cnn.CreateCommand())
            {
                cnn.Open();
                foreach (string statement in statements)
                {
                    if (statement.Length == 0)
                    {
                        continue;
                    }
                    cmd.CommandText = statement;
                    cmd.CommandTimeout = 600;   // 10 minutes
                    cmd.ExecuteNonQuery();
                }
            }
        }

        /// <summary>
        /// ADO.NETのデータベース接続を得る。
        /// </summary>
        /// <returns></returns>
        public abstract DbConnection GetDbConnnection();

        /// <summary>
        /// コーパスを使用するのに必須の情報（POSList等）をロードする.
        /// Corpus選択時に実行する.
        /// </summary>
        /// <param name="cps"></param>
        public virtual void LoadMandatoryCorpusInfo(Corpus cps)
        {
            NHibernate.Cfg.Configuration cfg = SearchConfiguration.GetInstance().NHibernateConfig;
            // Corpus(DB)の種類に合わせてConfigurationをセットアップする
            this.SetupConnection(cfg);

            cps.Lex.Reset();

            using (ISessionFactory factory = cfg.BuildSessionFactory())
            using (ISession session = factory.OpenSession())
            {
                IList lst = session.CreateCriteria(typeof(PartOfSpeech)).List();
                foreach (object obj in lst)
                {
                    PartOfSpeech pos = (PartOfSpeech)obj;
                    cps.Lex.PartsOfSpeech.Add(pos);
                }
                lst = session.CreateCriteria(typeof(CType)).List();
                foreach (object obj in lst)
                {
                    CType ctype = (CType)obj;
                    cps.Lex.CTypes.Add(ctype);
                }
                lst = session.CreateCriteria(typeof(CForm)).List();
                foreach (object obj in lst)
                {
                    CForm cform = (CForm)obj;
                    cps.Lex.CForms.Add(cform);
                }
            }
        }

        //@todo Service/Search/CorpusServiceを作ってService/Lexicon/SearchLexiconServiceと統合するべき
        /// <summary>
        /// コーパスの基本情報をロードする
        /// </summary>
        /// <param name="cps"></param>
        public virtual void LoadCorpusInfo(Corpus cps)
        {
            NHibernate.Cfg.Configuration cfg = SearchConfiguration.GetInstance().NHibernateConfig;
            // Corpus(DB)の種類に合わせてConfigurationをセットアップする
            this.SetupConnection(cfg);

            cps.Lex.Reset();

            using (ISessionFactory factory = cfg.BuildSessionFactory())
            using (ISession session = factory.OpenSession())
            using (ITransaction transaction = session.BeginTransaction())
            {
                object o;
                int nAttrs = (int)(long)session.CreateQuery("select count(*) from CorpusAttribute").UniqueResult();

                if ((o = session.CreateQuery("from CorpusAttribute where Name='NLexemes'").UniqueResult()) == null)
                {
                    cps.NLexemes = (int)(long)(session.CreateQuery("select count(*) from Lexeme").UniqueResult());
                    session.Save(new CorpusAttribute(nAttrs++, "NLexemes", cps.NLexemes.ToString()));
                }
                else
                {
                    cps.NLexemes = Int32.Parse(((CorpusAttribute)o).Value);
                }
                if ((o = session.CreateQuery("from CorpusAttribute where Name='NWords'").UniqueResult()) == null)
                {
                    cps.NWords = (int)(long)(session.CreateQuery("select count(*) from Word").UniqueResult());
                    session.Save(new CorpusAttribute(nAttrs++, "NWords", cps.NWords.ToString()));
                }
                else
                {
                    cps.NWords = Int32.Parse(((CorpusAttribute)o).Value);
                }
                if ((o = session.CreateQuery("from CorpusAttribute where Name='NSentences'").UniqueResult()) == null)
                {
                    cps.NSentences = (int)(long)(session.CreateQuery("select count(*) from Sentence").UniqueResult());
                    session.Save(new CorpusAttribute(nAttrs++, "NSentences", cps.NSentences.ToString()));
                }
                else
                {
                    cps.NSentences = Int32.Parse(((CorpusAttribute)o).Value);
                }
                cps.NSegments = (int)(long)(session.CreateQuery("select count(*) from Segment").UniqueResult());
                cps.NLinks = (int)(long)(session.CreateQuery("select count(*) from Link").UniqueResult());
                cps.NGroups = (int)(long)(session.CreateQuery("select count(*) from Group").UniqueResult());

                transaction.Commit();
            }
        }

        //@todo Service/Search/CorpusServiceを作ってService/Lexicon/SearchLexiconServiceと統合するべき
        /// <summary>
        /// コーパスのLexicon情報をロードする
        /// </summary>
        /// <param name="cps"></param>
        public virtual void LoadLexicon(Corpus cps)
        {
            NHibernate.Cfg.Configuration cfg = SearchConfiguration.GetInstance().NHibernateConfig;
            // Corpus(DB)の種類に合わせてConfigurationをセットアップする
            this.SetupConnection(cfg);

            cps.Lex.Reset();

            using (ISessionFactory factory = cfg.BuildSessionFactory())
            using (ISession session = factory.OpenSession())
            {
                IList lst = session.CreateCriteria(typeof(PartOfSpeech)).List();
                foreach (object obj in lst)
                {
                    PartOfSpeech pos = (PartOfSpeech)obj;
                    cps.Lex.PartsOfSpeech.Add(pos);
                }
                lst = session.CreateCriteria(typeof(CType)).List();
                foreach (object obj in lst)
                {
                    CType ctype = (CType)obj;
                    cps.Lex.CTypes.Add(ctype);
                }
                lst = session.CreateCriteria(typeof(CForm)).List();
                foreach (object obj in lst)
                {
                    CForm cform = (CForm)obj;
                    cps.Lex.CForms.Add(cform);
                }
                lst = session.CreateCriteria(typeof(Lexeme)).List();
                foreach (object obj in lst)
                {
                    Lexeme lex = (Lexeme)obj;
                    try
                    {
                        cps.Lex.Entries.Add(lex);
                    }
                    catch
                    {
                        //@todo
                    }
                }
            }
        }

        public virtual string GetBooleanString(bool b)
        {
            return b ? "true" : "false";
        }
    }
}
