﻿using System;
using System.Collections.Generic;
using System.Text;
using ChaKi.Entity.Corpora;
using NHibernate;
using System.Collections;
using NHibernate.Criterion;
using ChaKi.Service.Search;
using ChaKi.Service.Database;
using System.IO;
using ChaKi.Entity.Corpora.Annotations;

namespace ChaKi.Service.DependencyEdit
{
    public class DepEditService : IDisposable
    {
        protected Corpus m_Corpus;
        protected ISession m_Session;
        protected ITransaction m_Transaction;
        protected Sentence m_Sentence;
        internal DepOperationHistory m_History;


        public DepEditService()
        {
            m_Corpus = null;
            m_Session = null;
            m_Transaction = null;
            m_Sentence = null;
            m_History = new DepOperationHistory();
        }

        /// <summary>
        /// 新しい文をDependencyTree Editの対象としてオープンする。
        /// </summary>
        /// <param name="cps">編集対象のコーパス</param>
        /// <param name="sid">編集対象の文番号</param>
        /// <returns></returns>
        /// <exception cref="InvalidOperationException">他の編集が完了していないうちに、二重に編集を行おうとした。</exception>
        public Sentence Open(Corpus cps, int sid)
        {
            if (m_Transaction != null || m_Session != null)
            {
                throw new InvalidOperationException("Active transaction exists. Close or Commit it first.");
            }

            if (m_Corpus == null || cps != m_Corpus)
            {
                m_Corpus = cps;
            }
            m_Sentence = StartTransaction(m_Corpus, sid);
            m_History.Reset();
            return m_Sentence;
        }

        /// <summary>
        /// Close直後に再度同じ文の編集を開始する。
        /// </summary>
        /// <param name="cps"></param>
        /// <param name="sid"></param>
        /// <returns></returns>
        public Sentence ReOpen()
        {
            if (m_Transaction != null || m_Session != null)
            {
                throw new InvalidOperationException("Active transaction exists. Close or Commit it first.");
            }

            if (m_Corpus == null)
            {
                throw new InvalidOperationException("Unknown corpus.");
            }
            if (m_Sentence == null)
            {
                throw new InvalidOperationException("Unknown sentence.");
            }
            int sid = m_Sentence.ID;

            m_Sentence = StartTransaction(m_Corpus, sid);
            m_History.Reset();
            return m_Sentence;
        }

        /// <summary>
        /// m_Sentenceに関連するSegment, Linkのリストをすべて取得する.
        /// </summary>
        /// <param name="segs"></param>
        /// <param name="links"></param>
        public void GetBunsetsuTags(out List<Segment> segs, out List<Link> links)
        {
            if (m_Sentence == null)
            {
                throw new InvalidOperationException("Unknown sentence.");
            }
            segs = new List<Segment>();
            links = new List<Link>();

            //TODO: seg.Tag.IDの比較対象はTagSetからBunsetsu Segment TagのIDを引っ張ってくる必要がある。
            IQuery segquery = m_Session.CreateQuery(
                string.Format("from Segment seg where seg.EndChar > {0} and seg.EndChar <= {1} and seg.Tag.ID = 0 order by seg.StartChar",
                    m_Sentence.StartChar, m_Sentence.EndChar ));
            IList sresults = segquery.List();
            foreach (Object o in sresults)
            {
                segs.Add(o as Segment);
            }

            ICriteria linkMatching = m_Session.CreateCriteria(typeof(Link))
                .Add(Restrictions.Or(
                    Restrictions.In("From", sresults),
                    Restrictions.In("To", sresults)));
            IList lresults = linkMatching.List();
            foreach (Object o in lresults)
            {
                links.Add(o as Link);
            }
        }

        private Sentence StartTransaction(Corpus cps, int sid)
        {
            // Corpus(DB)の種類に合わせてConfigurationをセットアップする
            DBService dbs = DBService.Create(cps.DBParam);
            NHibernate.Cfg.Configuration cfg = dbs.GetConnection();
            ISessionFactory factory = cfg.BuildSessionFactory();
            m_Session = factory.OpenSession();
            m_Transaction = m_Session.BeginTransaction();

            // sidからSentenceオブジェクトを検索
            ICriteria senMatching = m_Session.CreateCriteria(typeof(Sentence))
                .Add(Restrictions.Eq("ID", sid));
            IList results = senMatching.List();
            if (results.Count != 1)
            {
                Dispose();
                return null;
            }
            return (Sentence)results[0];
        }

        /// <summary>
        /// 現在の状態をDBにコミットする。
        /// </summary>
        /// <exception cref="InvalidOperationException"></exception>
        public void Commit()
        {
            if (m_Transaction == null)
            {
                throw new InvalidOperationException("No active transaction exists.");
            }
#if false
            // まだセーブされていない（persistent化されていないBunsetsu）をセーブする。
            if (m_Sentence != null)
            {
                foreach (Bunsetsu ph in m_Sentence.Bunsetsus)
                {
                    if (!m_Session.Contains(ph))
                    {
                        m_Session.Save(ph);
                    }
                }
            }
#endif
            // コミットする
            m_Transaction.Commit();
            Dispose();
        }

        /// <summary>
        /// 現在の状態を捨てて、編集状態から抜ける。
        /// </summary>
        public void Close()
        {
            if (m_Transaction != null)
            {
                m_Transaction.Rollback();
            }
            Dispose();
        }

        /// <summary>
        /// 現在の状態を捨てて、編集状態から抜ける。
        /// </summary>
        public void Dispose()
        {
            try
            {
                if (m_Transaction != null)
                {
                    m_Transaction.Dispose();
                }
                if (m_Session != null)
                {
                    m_Session.Dispose();
                }
            }
            finally
            {
                m_Transaction = null;
                m_Session = null;
            }
        }

        /// <summary>
        /// 指定した文節を指定した語位置で分離する。
        /// </summary>
        /// <param name="bpos">文節のPos</param>
        /// <param name="wpos">語のPos（文節内での位置ではなく、文内の位置）</param>
        public void SplitBunsetsu(int bpos, int wpos)
        {
            DepOperation op = new DepOperation(DOType.Split, new object[] { bpos, wpos });
            op.Execute(m_Sentence, m_Session);
            m_History.Record(op);
        }

        /// <summary>
        /// 指定した文節を次の文節と併合する。
        /// </summary>
        /// <param name="bpos">文節のPos</param>
        /// <param name="wpos">語のPos（文節内での位置ではなく、文内の位置）</param>
        public void MergeBunsetsu(int bpos, int wpos)
        {
            DepOperation op = new DepOperation(DOType.Merge, new object[] { bpos, wpos });
            op.Execute(m_Sentence, m_Session);
            m_History.Record(op);
        }

        public void ChangeDependency(int bunsetsuId, int dep_org, int dep_new)
        {
            DepOperation op = new DepOperation(DOType.MoveArrow, new object[] { bunsetsuId, dep_org, dep_new });
            op.Execute(m_Sentence, m_Session);
            m_History.Record(op);
        }

        public void ChangeDependencyTag(int bunsetsuId, string oldtag, string newtag)
        {
            DepOperation op = new DepOperation(DOType.ChangeTag, new object[] { bunsetsuId, oldtag, newtag });
            op.Execute(m_Sentence, m_Session);
            m_History.Record(op);
        }

        public void ChangeLexeme(int wpos, Lexeme newlex)
        {
            DepOperation op = new DepOperation(DOType.ChangeLexeme, new object[] { wpos, newlex, null });
            op.Execute(m_Sentence, m_Session);
            m_History.Record(op);
        }

        public bool Undo()
        {
            DepOperation op = m_History.Back();
            if (op != null)
            {
                op.UnExecute(m_Sentence, m_Session);
                return true;
            }
            return false;
        }

        public bool Redo()
        {
            DepOperation op = m_History.Forward();
            if (op != null)
            {
                op.Execute(m_Sentence, m_Session);
                return true;
            }
            return false;
        }

        public bool CanUndo()
        {
            return m_History.CanUndo();
        }

        public bool CanRedo()
        {
            return m_History.CanRedo();
        }

        public bool CanSave()
        {
            return m_History.CanSave();
        }

        public Corpus GetCorpus()
        {
            return m_Corpus;
        }

        public void WriteToDotFile(TextWriter wr)
        {
            if (m_Sentence == null)
            {
                throw new InvalidOperationException("Unknown sentence.");
            }
            List<Segment> segs;
            List<Link> links;
            GetBunsetsuTags(out segs, out links);
            Dictionary<long, int> segIdMap = new Dictionary<long, int>();

            wr.WriteLine(string.Format("digraph \"{0}.{1}\" {{", m_Corpus.Name, m_Sentence.ID));
            wr.WriteLine("graph [charset = \"utf-8\"];");
            wr.WriteLine("node [fontname = \"sans\"];");
            for (int i = 0; i < segs.Count;i++)
            {
                Segment b = segs[i];
                segIdMap[b.ID] = i+1;
                string str = string.Format("{0} [label = \"{0}:{1}\"];", i+1, SegmentToString(b));
                wr.WriteLine(str);
            }
            foreach (Link lnk in links)
            {
                int fromid;
                int toid;
                if (!segIdMap.TryGetValue(lnk.From.ID, out fromid))
                {
                    continue;
                }
                if (!segIdMap.TryGetValue(lnk.To.ID, out toid))
                {
                    continue;
                }
                string str = string.Format("{0} -> {1} [label=\"{2}\"];", fromid, toid, lnk.Tag.Name);
                wr.WriteLine(str);
            }

            wr.WriteLine("}");
        }

        public string SegmentToString(Segment seg)
        {
            int start = seg.StartChar - m_Sentence.StartChar;
            int len = seg.EndChar - seg.StartChar;
            string s = m_Sentence.GetText();
            if (start < 0 || len <= 0 || len > s.Length)
            {
                return string.Empty;
            }
            return s.Substring(start,len);
        }
    }
}
