package charactermanaj;

import java.lang.reflect.Method;
import java.util.logging.Level;
import java.util.logging.Logger;

import javax.swing.UIManager;

import charactermanaj.model.AppConfig;
import charactermanaj.model.util.StartupSupport;
import charactermanaj.ui.MainFrame;
import charactermanaj.ui.ProfileListManager;
import charactermanaj.util.ErrorMessageHelper;
import charactermanaj.util.ApplicationLoggerConfigurator;

/**
 * エントリポイント用クラス
 * @author seraphy
 *
 */
public final class Main {
	
	/**
	 * ロガー.<br>
	 */
	private static Logger logger;

	/**
	 * Mac OS Xであるか?
	 */
	private static final boolean isMacOSX;
	
	/**
	 * クラスイニシャライザ
	 */
	static {
		// Mac OS Xでの実行判定
		// システムプロパティos.nameは、すべてのJVM実装に存在する.
		// 基本ディレクトリの位置の決定に使うため、
		// なによりも、まず、これを判定しないとダメ.(順序が重要)
		String lcOS = System.getProperty("os.name").toLowerCase();
		isMacOSX = lcOS.startsWith("mac os x");

		// ロガーの準備
		try {
			// ローカルファイルシステム上のユーザ定義ディレクトリから
			// ログの設定を読み取る.(OSにより、設定ファイルの位置が異なることに注意)
			ApplicationLoggerConfigurator.configure();
			logger = Logger.getLogger(Main.class.getName());

		} catch (Throwable ex) {
			ex.printStackTrace();
			logger = null;
		}

		if (logger != null) {
			logger.log(Level.INFO, "os.name=" + lcOS + "/isMacOSX=" + isMacOSX);
		}
	}
	
	/**
	 * エントリポイント.<br>
	 * 最初のメインフレームを開いたときにMac OS Xであればスクリーンメニューの登録も行う.<br>
	 * @param args 引数(未使用)
	 */
	public static void main(String[] args) {
		try {
			//System.setProperty("swing.aatext", "true");
			//System.setProperty("awt.useSystemAAFontSettings", "on");
			
			// MacOSXであれば、スクリーンメニューを有効化
			if (isMacOSX()) {
				System.setProperty("apple.laf.useScreenMenuBar", "true");
				System.setProperty(
						"com.apple.mrj.application.apple.menu.about.name",
						"CharacterManaJ");
			}

			// 実行プラットフォームのネイティブな外観にする.
			UIManager.setLookAndFeel(UIManager.getSystemLookAndFeelClassName());
			
			// アプリケーション設定の読み込み
			AppConfig appConfig = AppConfig.getInstance();
			appConfig.loadConfig();
	
			// スタートアップ時の初期化
			StartupSupport.getInstance().doStartup();
			
			// デフォルトのプロファイルを開く.
			// (最後に使ったプロファイルがあれば、それが開かれる.)
			MainFrame mainFrame = ProfileListManager.openDefaultProfile();
			if (isMacOSX()) {
				// MacOSXであればスクリーンメニューからのイベントをハンドルできるようにする.
				// OSXにしか存在しないクラスを利用するためリフレクションとしている.
				Class<?> clz = Class.forName("charactermanaj.ui.MainFramePartialForMacOSX");
				Method mtd = clz.getMethod("setupScreenMenu", MainFrame.class);
				mtd.invoke(null, mainFrame);
			}
			
			// 表示(および位置あわせ)
			mainFrame.showMainFrame();

		} catch (Throwable ex) {
			// なんらかの致命的な初期化エラーがあった場合、ログとコンソールに表示
			// ダイアログが表示されるかどうかは状況次第.
			ex.printStackTrace();
			if (logger != null) {
				logger.log(Level.SEVERE, "Application initiation failed.", ex);
			}
			ErrorMessageHelper.showErrorDialog(null, ex);

			// メインフレームを破棄します.
			MainFrame.closeAllProfiles();
		}
	}
	
	/**
	 * Mac OS Xで動作しているか?
	 * @return Max OS X上であればtrue
	 */
	public static boolean isMacOSX() {
		return isMacOSX;
	}
	
}
