package charactermanaj.model.util;

import java.io.File;
import java.net.URI;
import java.net.URL;
import java.util.HashMap;
import java.util.Map;
import java.util.UUID;
import java.util.logging.Level;
import java.util.logging.Logger;

import charactermanaj.model.AppConfig;
import charactermanaj.util.UserDataFactory;

/**
 * 開始前の事前準備するためのサポートクラス 
 * @author seraphy
 */
public abstract class StartupSupport {
	
	private static StartupSupport inst;
	
	/**
	 * インスタンスを取得する.
	 * @return シングルトンインスタンス
	 */
	public static synchronized StartupSupport getInstance() {
		if (inst == null) {
			inst = new StartupSupport() {
				private final Logger logger = Logger.getLogger(StartupSupport.class.getName());

				@Override
				public void doStartup() {
					StartupSupport[] startups = {
							new PurgeOldLogs(),
							new UpgradeCache(),
							new UpgradeFavoritesXml(),
							new PurgeUnusedCache(),
					};
					for (StartupSupport startup : startups) {
						logger.log(Level.FINE, "startup operation start. class="
								+ startup.getClass().getSimpleName());
						try {
							startup.doStartup();
							logger.log(Level.INFO, "startup operation is done.");

						} catch (Exception ex) {
							logger.log(Level.WARNING, "startup operation failed.", ex);
						}
					}
				}
			};
		}
		return inst;
	}
	
	/**
	 * ユーザディレクトリをアップグレードします.
	 */
	public abstract void doStartup();
}

/**
 * 古い形式のCacheを移動する.
 * @author seraphy
 *
 */
class UpgradeCache extends StartupSupport {
	
	/**
	 * ロガー
	 */
	private final Logger logger = Logger.getLogger(getClass().getName());

	@Override
	public void doStartup() {
		UserDataFactory userDataFactory = UserDataFactory.getInstance();
		File appData = userDataFactory.getSpecialDataDir(null);

		// ver0.94まではユーザディレクトリ直下に*.serファイルを配置していたが、
		// ver0.95以降はcachesに移動したため、旧ファイルがのこっていれば移動する.
		for (File file : appData.listFiles()) {
			try {
				String name = file.getName();
				if (file.isFile() && name.endsWith(".ser")) {
					File toDir = userDataFactory.getSpecialDataDir(name);
					if ( !appData.equals(toDir)) {
						String convertedName = convertName(name);
						File toFile = new File(toDir, convertedName);
						boolean ret = file.renameTo(toFile);
						logger.log(Level.INFO, "move file " + file + " to " + toFile + " ;successed=" + ret);
					}
				}
			} catch (Exception ex) {
				logger.log(Level.WARNING, "file move failed. " + file, ex);
			}
		}
	}
	
	protected String convertName(String name) {
		if (name.endsWith("-workingset.ser") || name.endsWith("-favorites.ser")) {
			// {UUID}-CharacterID-workingset.ser もしくは、-favorites.serの場合は、
			// {UUID}-workingset.ser / {UUID}-favorites.serに変更する.
			String[] splitedName = name.split("-");
			if (splitedName.length >= 7) {
				StringBuilder buf = new StringBuilder();
				for (int idx = 0; idx < 5; idx++) {
					if (idx > 0) {
						buf.append("-");
					}
					buf.append(splitedName[idx]);
				}
				buf.append("-");
				buf.append(splitedName[splitedName.length - 1]);
				return buf.toString();
			}
		}
		return name;
	}
}

/**
 * DocBaseのハッシュ値をもとにしたデータをサポートする抽象クラス.
 * @author seraphy
 *
 */
abstract class StartupSupportForDocBasedData extends StartupSupport {
	
	/**
	 * character.xmlのファイル位置を示すUUID表現を算定するためのアルゴリズムの選択肢.<br>
	 * @author seraphy
	 */
	protected enum DocBaseSignatureStoratage {
		
		/**
		 * 新形式のcharacter.xmlのUUIDを取得する.
		 * charatcer.xmlファイルのURIを文字列にしたもののタイプ3-UUID表現.<br>
		 */
		NEW_FORMAT() {
			@Override
			public String getDocBaseSignature(File characterXmlFile) {
				URI docBase = characterXmlFile.toURI();
				UserDataFactory userDataFactory = UserDataFactory.getInstance();
				return userDataFactory.getMangledNamedPrefix(docBase);
			}
		},
		
		/**
		 * 旧形式のcharacter.xmlのUUIDを取得する.<br>
		 * charatcer.xmlファイルのURLを文字列にしたもののタイプ3-UUID表現.<br>
		 */
		OLD_FORMAT() {
			@Override
			public String getDocBaseSignature(File characterXmlFile) {
				try {
					@SuppressWarnings("deprecation")
					URL url = characterXmlFile.toURL();
					return UUID.nameUUIDFromBytes(url.toString().getBytes()).toString();

				} catch (Exception ex) {
					logger.log(Level.WARNING,
							"character.xmlのファイル位置をUUID化できません。:"
									+ characterXmlFile, ex);
					return null;
				}
			}
		},
		;
		
		/**
		 * ロガー
		 */
		private static Logger logger = Logger.getLogger(
				DocBaseSignatureStoratage.class.getName());
		
		/**
		 * character.xmlからuuid表現のプレフィックスを算定する.
		 * @param characterXmlFile character.xmlのファイル
		 * @return UUID
		 */
		public abstract String getDocBaseSignature(File characterXmlFile);
	}
	
	/**
	 * すべてのユーザおよびシステムのキャラクターデータのDocBaseをもととしたハッシュ値(Prefix)の文字列をキーとし、
	 * キャラクターディレクトリを値とするマップを返す.<br>
	 * @return DocBaseをもととしたハッシュ値の文字列表記をキー、キャラクターディレクトリを値とするマップ
	 */
	protected Map<String, File> getDocBaseMap(DocBaseSignatureStoratage storatage) {
		if (storatage == null) {
			throw new IllegalArgumentException();
		}
		AppConfig appConfig = AppConfig.getInstance();

		File[] charactersDirs = {
				appConfig.getSystemCharactersDir(),
				appConfig.getUserCharactersDir()
		};
		
		// キャラクターデータディレクトリを走査しdocBaseの識別子の一覧を取得する
		Map<String, File> docBaseSignatures = new HashMap<String, File>();
		for (File charactersDir : charactersDirs) {
			if (charactersDir == null || !charactersDir.exists()
					|| !charactersDir.isDirectory()) {
				continue;
			}
			for (File characterDir : charactersDir.listFiles()) {
				if ( !characterDir.isDirectory()) {
					continue;
				}
				File characterXml = new File(characterDir, "character.xml");
				if ( !characterXml.exists()) {
					continue;
				}
				String docBaseSig = storatage.getDocBaseSignature(characterXml);
				if (docBaseSig != null) {
					docBaseSignatures.put(docBaseSig, characterDir);
				}
			}
		}
		return docBaseSignatures;
	}
	
	/**
	 * 指定したディレクトリ直下にあるDocBaseのUUIDと推定される文字列で始まり、suffixで終わる
	 * ファイルのUUIDの部分文字列をキーとし、そのファイルを値とするマップを返す.
	 * @param dataDir 対象ディレクトリ
	 * @param suffix  対象となるファイル名の末尾文字列、nullまたは空文字の場合は全て
	 * @return DocBaseのUUID表現と思われる部分文字列をキーとし、そのファイルを値とするマップ
	 */
	protected Map<String, File> getUUIDMangledNamedMap(File dataDir, String suffix) {
		if (dataDir == null || !dataDir.exists() || !dataDir.isDirectory()) {
			throw new IllegalArgumentException();
		}

		Map<String, File> uuidMangledFiles = new HashMap<String, File>();
		for (File file : dataDir.listFiles()) {
			String name = file.getName();
			if (file.isFile() && (suffix == null || suffix.length() == 0 || name.endsWith(suffix))) {
				String[] sigParts = name.split("-");
				if (sigParts.length >= 5) {
					// UUIDはa-b-c-d-eの5パーツになり、更に末尾に何らかの文字列が付与されるので
					// 区切り文字は5つ以上になる.
					// UUIDの部分だけ結合し直して復元する.
					StringBuilder sig = new StringBuilder();
					for (int idx = 0; idx < 5; idx++) {
						if (idx != 0) {
							sig.append("-");
						}
						sig.append(sigParts[idx]);
					}
					uuidMangledFiles.put(sig.toString(), file);
				}
			}
		}
		return uuidMangledFiles;
	}
	
}

/**
 * 古い形式のFavorites.xmlを移動する.
 * @author seraphy
 */
class UpgradeFavoritesXml extends StartupSupportForDocBasedData {
	
	/**
	 * ロガー
	 */
	private final Logger logger = Logger.getLogger(getClass().getName());

	@Override
	public void doStartup() {
		UserDataFactory userDataFactory = UserDataFactory.getInstance();
		File appData = userDataFactory.getSpecialDataDir(null);

		// キャラクターデータディレクトリを走査しdocBaseの識別子の一覧を取得する
		Map<String, File> docBaseSignatures = getDocBaseMap(
				DocBaseSignatureStoratage.OLD_FORMAT);

		// ver0.94までは*.favorite.xmlはユーザディレクトリ直下に配備していたが
		// ver0.95以降は各キャラクターディレクトリに移動するため、旧docbase-id-favorites.xmlが残っていれば移動する
		// ユーザディレクトリ直下にある*-facotites.xmlを列挙する
		Map<String, File> favorites = getUUIDMangledNamedMap(appData, "-favorites.xml");

		// 旧式の*-favorites.xmlを各キャラクターディレクトリに移動する.
		for (Map.Entry<String, File> favoritesEntry : favorites.entrySet()) {
			String sig = favoritesEntry.getKey();
			File file = favoritesEntry.getValue();
			try {
				File characterDir = docBaseSignatures.get(sig);
				if (characterDir != null) {
					File toFile = new File(characterDir, "favorites.xml");
					boolean ret = file.renameTo(toFile);
					logger.log(Level.INFO, "move file " + file + " to " + toFile + " ;successed=" + ret);
				}

			} catch (Exception ex) {
				logger.log(Level.INFO, "move file failed." + file, ex);
			}
		}
	}
}

/**
 * 存在しないDocBaseを参照するキャッシュ(*.ser)を削除する.
 * @author seraphy
 */
class PurgeUnusedCache extends StartupSupportForDocBasedData {
	
	/**
	 * ロガー
	 */
	private final Logger logger = Logger.getLogger(getClass().getName());

	@Override
	public void doStartup() {
		// キャラクターデータディレクトリを走査しdocBaseの識別子の一覧を取得する
		Map<String, File> docBaseSignatures = getDocBaseMap(
				DocBaseSignatureStoratage.NEW_FORMAT);
		
		// キャッシュの保存先を取得する.
		UserDataFactory userDataFactory = UserDataFactory.getInstance();
		File cacheDir = userDataFactory.getSpecialDataDir("*.ser");
		
		// キャッシュ上にあるDocBaseのUUID表現で始まる*.serを列挙
		String[] suffixes = {
				"-character.xml-cache.ser", // character.xmlのキャッシュ
				"-workingset.ser", // 作業状態のキャッシュ
				"-favorites.ser" // お気に入りのキャッシュ
				};
		for (String suffix : suffixes) {
			Map<String, File> caches = getUUIDMangledNamedMap(cacheDir, suffix);
			for (Map.Entry<String, File> cacheEntry : caches.entrySet()) {
				String mangledUUID = cacheEntry.getKey();
				File cacheFile = cacheEntry.getValue();
				try {
					if ( !docBaseSignatures.containsKey(mangledUUID)) {
						boolean result = cacheFile.delete();
						logger.log(Level.INFO, "purge unused cache: " + cacheFile
								+ "/succeeded=" + result);
					}
					
				} catch (Exception ex) {
					logger.log(Level.WARNING, "remove file failed. " + cacheFile, ex);
				}
			}
		}
	}
}

/**
 * 古いログファイルを消去する.
 * @author seraphy
 */
class PurgeOldLogs extends StartupSupport {

	/**
	 * ロガー
	 */
	private final Logger logger = Logger.getLogger(getClass().getName());

	@Override
	public void doStartup() {
		UserDataFactory userDataFactory = UserDataFactory.getInstance();
		File logsDir = userDataFactory.getSpecialDataDir("*.log");
		if (logsDir.exists()) {
			AppConfig appConfig = AppConfig.getInstance();
			long purgeOldLogsMillSec = appConfig.getPurgeLogDays() * 24L * 3600L * 1000L;
			if (purgeOldLogsMillSec > 0) {
				long purgeThresold = System.currentTimeMillis() - purgeOldLogsMillSec;
				for (File file : logsDir.listFiles()) {
					try {
						String name = file.getName();
						if (file.isFile() && file.canWrite() && name.endsWith(".log")) {
							long lastModified = file.lastModified();
							if (lastModified > 0 && lastModified < purgeThresold) {
								boolean result = file.delete();
								logger.log(Level.INFO, "remove file " + file
										+ "/succeeded=" + result);
							}
						}

					} catch (Exception ex) {
						logger.log(Level.WARNING, "remove file failed. " + file, ex);
					}
				}
			}
		}
	}
}
