// Chattr - Twitter client on .NET
//
// Copyright (c) 2007 Katsuhiko Ichinose <ichi@users.sourceforge.jp>
//
// Chattr is Free Software released under GNU General Public License.
//
// $Id: FormBrowser.cs 10 2007-09-30 14:34:09Z ichi $

using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Data;
using System.Drawing;
using System.Text;
using System.Windows.Forms;

namespace Chattr
{
    public partial class FormBrowser : Form
    {
        private FormMain m_FormMain;

        private double m_DeltaWidth;
        private double m_CurrWidth;

        public FormBrowser(FormMain main)
        {
            m_FormMain = main;
            InitializeComponent();
            webBrowser.StatusTextChanged += new EventHandler(webBrowser_StatusTextChanged);
        }

        public void Navigate(string url, bool animation)
        {
            toolStripStatusLabel.Text = "";
            if (!webBrowser.Created)
            {
                webBrowser.CreateControl();
            }
            if (!this.Visible)
            {
                ShowWindow(animation);
                this.Show();
            }
            else if (this.WindowState == FormWindowState.Minimized)
            {
                this.WindowState = FormWindowState.Normal;
                this.Activate();
            }
            webBrowser.Navigate(url);
        }

        private Size GetDefaultWindowSize()
        {
            if (Screen.AllScreens.Length > 1)
            {
                return Properties.Settings.Default.FormBrowserSize2;
            }
            return Properties.Settings.Default.FormBrowserSize;
        }

        private void SetWindowSize()
        {
            Point location = m_FormMain.Location;
            if (Screen.AllScreens.Length > 1)
            {
                location.Offset(Properties.Settings.Default.FormBrowserLocation2);
                this.Size = Properties.Settings.Default.FormBrowserSize2;
            }
            else
            {
                location.Offset(Properties.Settings.Default.FormBrowserLocation);
                this.Size = Properties.Settings.Default.FormBrowserSize;
            }
            this.Location = location;
        }

        private void SaveWindowSize()
        {
            if ((!timer.Enabled) && this.Visible && (this.WindowState == FormWindowState.Normal))
            {
                if (Screen.AllScreens.Length > 1)
                {
                    Properties.Settings.Default.FormBrowserSize2 = this.Size;
                }
                else
                {
                    Properties.Settings.Default.FormBrowserSize = this.Size;
                }
            }
        }

        private void SaveWindowPosition()
        {
            if ((!timer.Enabled) && this.Visible && (this.WindowState == FormWindowState.Normal))
            {
                Point location = this.Location;
                location.Offset(-m_FormMain.Location.X, -m_FormMain.Location.Y);
                if (Screen.AllScreens.Length > 1)
                {
                    Properties.Settings.Default.FormBrowserLocation2 = location;
                }
                else
                {
                    Properties.Settings.Default.FormBrowserLocation = location;
                }
            }
        }

        private void ShowWindow(bool animation)
        {
            timer.Interval = 1000;
            timer.Enabled = true;
            SetWindowSize();
            if (animation)
            {
                m_DeltaWidth = this.Width / 5.0;
                this.Width = 0;
                m_CurrWidth = 0;
                timer.Interval = 33;
                timer.Enabled = true;
            }
            else
            {
                timer.Enabled = false;
            }
        }

        private void FormBrowser_Load(object sender, EventArgs e)
        {
            this.Text = "Chattr";
            toolStripProgressBar.ProgressBar.Style = ProgressBarStyle.Continuous;
            toolStripStatusLabel.Text = "";
        }

        private void FormBrowser_SizeChanged(object sender, EventArgs e)
        {
            SaveWindowSize();
        }

        private void FormBrowser_Move(object sender, EventArgs e)
        {
            SaveWindowPosition();
        }

        private void FormBrowser_FormClosing(object sender, FormClosingEventArgs e)
        {
            webBrowser.Stop();
            webBrowser.Navigate("about:blank");
            this.Hide();
            e.Cancel = true;
        }

        private void timer_Tick(object sender, EventArgs e)
        {
            Size size = GetDefaultWindowSize();
            m_CurrWidth += m_DeltaWidth;
            if (m_CurrWidth >= size.Width)
            {
                m_CurrWidth = size.Width;
                timer.Enabled = false;
            }
            this.Width = Convert.ToInt32(m_CurrWidth);
        }

        private void webBrowser_Navigating(object sender, WebBrowserNavigatingEventArgs e)
        {
            toolStripProgressBar.Visible = true;
            toolStripButtonBack.Enabled = webBrowser.CanGoBack;
            toolStripButtonForward.Enabled = webBrowser.CanGoForward;
        }

        private void webBrowser_ProgressChanged(object sender, WebBrowserProgressChangedEventArgs e)
        {
            this.Text = webBrowser.DocumentTitle;
            toolStripProgressBar.ProgressBar.Maximum = 100;
            toolStripProgressBar.ProgressBar.Value = (int)(e.CurrentProgress * 100 / e.MaximumProgress);
        }
        void webBrowser_StatusTextChanged(object sender, EventArgs e)
        {
            toolStripStatusLabel.Text = webBrowser.StatusText;
        }

        private void webBrowser_DocumentCompleted(object sender, WebBrowserDocumentCompletedEventArgs e)
        {
            this.Text = webBrowser.DocumentTitle;
            toolStripProgressBar.Visible = false;
            toolStripButtonBack.Enabled = webBrowser.CanGoBack;
            toolStripButtonForward.Enabled = webBrowser.CanGoForward;
        }

        private void webBrowser_NewWindow(object sender, CancelEventArgs e)
        {
            //e.Cancel = true;
        }

        private void toolStripButtonBack_Click(object sender, EventArgs e)
        {
            webBrowser.GoBack();
        }

        private void toolStripButtonForward_Click(object sender, EventArgs e)
        {
            webBrowser.GoForward();
        }

        private void toolStripButtonRefresh_Click(object sender, EventArgs e)
        {
            webBrowser.Refresh();
        }

        private void toolStripButtonStop_Click(object sender, EventArgs e)
        {
            webBrowser.Stop();
        }
    }
}