﻿using System;
using System.IO;
using System.Text;
using System.Threading;
using System.Threading.Tasks;
using System.Collections.Generic;
using System.Windows.Forms;
using System.Diagnostics;
using System.Linq;
using CompleteEraser.Properties;

namespace CompleteEraser
{
    public partial class MainForm : Form
    {
        CancellationTokenSource tokenSource;
        Task task;
        SharedInfo info;
        IEnumerable<string> files;
        IOperation op;

        public MainForm()
        {
            InitializeComponent();
        }

        public MainForm(string[] args) : this()
        {
            if (args.Length == 0)
            {
                try
                {
                    this.info = new SharedInfo();
                    this.files = this.info;
                    this.op = new ShrredFiles();
                }
                catch (IOException)
                {
                    MessageBox.Show("ファイルかフォルダーを指定してください");
                    return;
                }
            }
            else if (args[0] == "/recylebin")
            {
                this.op = new BreakRecyleBin();
                this.files = new RecyleBinCollection();
            }else{
                this.op = new ShrredFiles();
                this.files = args;
            }
            this.progressBar1.Maximum = this.files.Count();
            this.op.Progressing += new ProgressingEventHandler(op_Progressing);
            this.op.Progressed += new ProgressedEventHandler(op_Progressed);
        }

        private void MainForm_Shown(object sender, EventArgs e)
        {
            if (op == null)
            {
                this.Close();
                return;
            }

            this.tokenSource = new CancellationTokenSource();

            this.task = new Task(() =>
            {
                op.Execute(this.files);
            });
            this.task.ContinueWith((t) =>
            {
                this.BeginInvoke(new Action(() =>
                {
                    this.Close();
                }));
            });
            this.task.Start();
        }

        void  op_Progressing(object sender, ProgressingEventArgs e)
        {
            this.BeginInvoke(new Action(() =>
            {
                if (e.fileName == null)
                    this.label2.Text = Resources.FINAL_PROCESS;
                else
                    this.label2.Text = string.Format(Resources.FILE_PROCESS, this.TrimFullPath(e.fileName));
            }));
        }

        void op_Progressed(object sender, ProgressedEventArgs e)
        {
            CancellationToken token = this.tokenSource.Token;
            if (token.IsCancellationRequested)
                token.ThrowIfCancellationRequested();

            if (e.ex == null)
            {
                this.BeginInvoke(new Action(() => { if (this.progressBar1.Value < this.progressBar1.Maximum) this.progressBar1.Value++; }));
            }else if(e.ex is IOException){
                DialogResult result = MessageBox.Show(e.ex.Message, "", MessageBoxButtons.AbortRetryIgnore);
                switch (result)
                {
                    case System.Windows.Forms.DialogResult.Abort:
                        e.breaked = true;
                        break;
                    case System.Windows.Forms.DialogResult.Retry:
                        e.retry = true;
                        break;
                }
            }
            else if (e.ex is UnauthorizedAccessException)
            {
                if (MessageBox.Show(string.Format(Resources.CONFIRMRUNAS, e.fileName), "", MessageBoxButtons.YesNo) == DialogResult.Yes)
                {
                    ProcessStartInfo info = new ProcessStartInfo(Application.ExecutablePath);
                    info.Verb = "runas";
                    info.UseShellExecute = true;
                    Process.Start(info);
                    this.Hide();
                    Thread.Sleep(1000);
                    e.breaked = true;
                }
            }
            else
            {
                MessageBox.Show(e.ex.Message);
                e.breaked = true;
            }
        }

        private void button1_Click(object sender, EventArgs e)
        {
            tokenSource.Cancel();
        }

        private void MainForm_FormClosed(object sender, FormClosedEventArgs e)
        {
            if(tokenSource != null)
                tokenSource.Cancel();
            if (this.info != null)
                this.info.Dispose();
        }

        public string TrimFullPath(string filepath)
        {
            if (filepath == null || filepath == "")
                return string.Empty;
            string DirectoryPart = Path.GetDirectoryName(filepath);
            string FilenamePart = Path.GetFileName(filepath);
            string[] slice = DirectoryPart.Split('\\');
            if (slice.Length > 3)
            {
                DirectoryPart = slice[0] + "\\..\\" + slice[slice.Length - 1];
                return DirectoryPart + "\\" + FilenamePart;
            }
            else
                return filepath;
        }

    }
}
