// -*-Mode: C++;-*-
//
// Calculate protein secondary structures
// based on the Kabsh&Sander's paper Biopolymers (1983) 22, 2577-
//
// $Id: Prot2ndry.cpp,v 1.2 2009/11/08 16:42:14 rishitani Exp $

#include <common.h>

#include <qlib/LExceptions.hpp>
#include <qlib/LQuat.hpp>
#include <qlib/Vector4D.hpp>

#include "MolCoord.hpp"
#include "MolChain.hpp"
#include "MolResidue.hpp"
#include "MolAtom.hpp"
#include "AtomIterator.hpp"
#include "ResidIterator.hpp"

#define PEPDIST_MAX  3.0

namespace {

  using qlib::Vector4D;
  using namespace molstr;

  struct HydrogenBond
  {
    int peer;
    double energy;
  };

  bool operator<(const HydrogenBond &arg1,const HydrogenBond &arg2)
  {
    return (arg1.energy<arg2.energy);
  }

  typedef std::set<HydrogenBond> HbonList;

  struct Backbone
  {
    /** residue name */
    LString resn;

    /** sequential chain index */
    int chind;

    /** atom positions */
    Vector4D ca,c,n,o,h;

    bool bHasH;

    /** secondary structure */
    LString ss;

    /** turn type */
    char turn[3];

    /** hydrogen bond link */
    HbonList acceptors, donors;

    /** ptr to the original residue */
    MolResiduePtr pRes;

    //Backbone() : nOrgResid(-1), bHasH(false) {}
  };

  enum {
    nobridge,
    parallel,
    antiparallel
  };

  struct Bridge
  {
    // char sheetname, laddername;
    // bridgeset linkset;

    int btyp;
    int ib, ie, jb, je;

    bool bIsolated;
    //, from, towards;
  };
  
  typedef std::list<Bridge> BridgeList;

  class Prot2ndry
  {
  public:

    typedef std::vector<Backbone *> data_t;

    data_t m_chains;

    BridgeList m_bridges;

    Prot2ndry() {}
    ~Prot2ndry() { fini(); }

    //////////////////////////////////

    /**
       Build the m_chains member from the pMol object
     */
    void init(MolCoordPtr pMol)
    {
      int nelem = 0;
      LString tagnmN("N"), tagnmC("C"), tagnmO("O"), tagnmCA("CA");
      MolAtomPtr pCurrN, pCurrC, pCurrO, pCurrCA;
      MolAtomPtr pPrevC, pPrevO;
      Backbone *pBB;
      
      // check the pMol's structure
      ResidIterator riter(pMol);
      MolResiduePtr pPrev; // = NULL;
      MolResiduePtr pCurr; // = NULL;
      int nres=0, ci=0;
      std::list<Backbone *> chain_list;
      
      for (riter.first(); riter.hasMore(); riter.next()) {
	pCurr = riter.get();
        MB_ASSERT(!pCurr.isnull());
	
	pCurrN = pCurr->getAtom(tagnmN);
	pCurrO = pCurr->getAtom(tagnmO);
	pCurrC = pCurr->getAtom(tagnmC);
	pCurrCA = pCurr->getAtom(tagnmCA);
	if (pCurrN.isnull() ||
	    pCurrC.isnull() ||
	    pCurrO.isnull() ||
            pCurrCA.isnull()) {
	  // Incomplete backbone atoms !! (skip this residue)
          //MB_DPRINTLN("Incomplete/nonprotein residue: %s%d",
          //pCurr->getChainName().c_str(), pCurr->getIndex());
	  pPrev = MolResiduePtr(); //NULL;
	  continue;
	}

	pBB = new Backbone;

	pBB->resn = pCurr->getName().c_str();
	pBB->ca = pCurrCA->getPos();
	pBB->n = pCurrN->getPos();
	pBB->c = pCurrC->getPos();
	pBB->o = pCurrO->getPos();
	pBB->pRes = pCurr;
	pBB->turn[0] = pBB->turn[1] = pBB->turn[2] = ' ';

	if (pPrev.isnull()) {
	  // pCurr is start residue of a segment
	  if (chain_list.size()>0) {
	    // insert chain break mark
	    ci++;
	  }
	  pBB->chind = ci;
	  chain_list.push_back(pBB);
	  pPrev = pCurr;
	  continue;
	}

	//
	// Check linkage between the previous residue
	//  to determin segment breakage
	//

	bool bchbrk = false;
	if (!pPrev->getChainName().equals(pCurr->getChainName())) {
	  // chain break
	  bchbrk = true;
	}
	else {
	  pPrevC = pPrev->getAtom(tagnmC);
	  pPrevO = pPrev->getAtom(tagnmO);
      
	  if (pCurrN.isnull() ||
	      pPrevC.isnull() ||
	      pPrevO.isnull()) {
	    // Incomplete backbone atoms !!
	    bchbrk = true;
	  }
	  else {
	    const double pepdist = (pCurrN->getPos() - pPrevC->getPos()).length();
	    if (pepdist > PEPDIST_MAX) {
	      bchbrk = true;
	    }
	    else {
	      // estimate position of the amide hydrogen atom
	      Vector4D vecCO = pPrevO->getPos() - pPrevC->getPos();
	      vecCO /= vecCO.length();
	      pBB->h = pBB->n - vecCO;
	      pBB->bHasH = true;
	    }
	  }
	}

	if (bchbrk) {
	  // insert chain break mark
	  ci++;
	}

	pBB->chind = ci;
	chain_list.push_back(pBB);

	// next
	pPrev = pCurr;

      } // for (riter.first(); riter.hasMore(); riter.next())...


      //
      // copy chain_list to array
      //
      MB_ASSERT(m_chains.size()==0);
      int i, nEntry = chain_list.size();
      m_chains.resize(nEntry);
      std::list<Backbone *>::const_iterator iter = chain_list.begin();
      for (i=0; iter!=chain_list.end() && i<nEntry; ++iter, ++i) {
	m_chains[i] = *iter;
    
	// Backbone *pB = *iter;
	// MB_DPRINTLN("%s%d\t%s: %d",
	// pB->sOrgChain.c_str(), pB->nOrgResid,
	// pB->resn.c_str(), pB->bHasH);
	// delete pB;
      }

      return;
    }

    void fini()
    {
      int i;
      for (i=0; i<m_chains.size(); ++i)
	delete m_chains.at(i);
    }

    //////////////////////////////////////////////////////////////////////

    void calcHbon()
    {
      // CA<->CA distance cutoff
      const double CADIST = 9.0;
      int i, j;

      for (i = 0; i < m_chains.size(); ++i) {
	// skip the chain break mark
	if (!noChainBrk(i, i))
	  continue;

	Backbone &WITH = *m_chains[i];
	for (j = i + 1; j < m_chains.size(); ++j) {
	  // skip the chain break mark
	  if (!noChainBrk(j, j))
	    continue;

	  // skip the distant residue pair
	  const double cadist = (WITH.ca - m_chains[j]->ca).length();
	  if (cadist >= CADIST)
	    continue;

	  checkHbonPair(i, j);
	  if (j != i + 1)
	    checkHbonPair(j, i);

	  /* j==i+1̂Ƃ:
	     (i, i+1)̃yA͌vZ邪Cii: donor, i+1: acceptor)
	     (i+1, i)̃yA͌vZĂȂDpeptide bond̂
	     hbond`͍݂̂蓾Ȃ*/
	} // for (j)
      } // for (i)

#if 0
      for (i=0; i<m_chains.size(); ++i) {
	MB_DPRINTLN("%s %s%d ---", m_chains[i]->sOrgChain.c_str(),
		    m_chains[i]->resn.c_str(), m_chains[i]->nOrgResid);
	HbonList &accs = m_chains[i]->acceptors;
	HbonList &dons = m_chains[i]->donors;

	MB_DPRINTLN("acceptors:");
	HbonList::const_iterator iter = accs.begin();
	for (; iter!=accs.end(); ++iter) {
	  MB_DPRINTLN("  %d %f", iter->peer, iter->energy);
	}
	
	MB_DPRINTLN("donors:");
	iter = dons.begin();
	for (; iter!=dons.end(); ++iter) {
	  MB_DPRINTLN("  %d %f", iter->peer, iter->energy);
	}
      }
#endif
    }

  private:

    ///////////////////////////////////////////
    // Utility routines for calcHbon

    bool noChainBrk(int i, int j)
    {
      if (i>j) return false;
      if (i<0 || j>=m_chains.size()) return false;

      return m_chains[i]->chind == m_chains[j]->chind;
    }

    void checkHbonPair(int i, int j)
    {
      // HBHIGH - HIGHEST ALLOWED ENERGY OF A HYDROGEN BOND IN CAL/MOL
      const double HBHIGH = -500.0;
      HydrogenBond hb;

      /* I IS >N-H, J IS >C=O */
      hb.energy = calcHbonEnergy(i, j);
      if (hb.energy>=HBHIGH)
	return;
      
      // MB_DPRINTLN("hbon %d %d  = %f kcal/mol", i, j, hb.energy);

      /* CO(J) IS ACCEPTOR OF NH(I) */
      
      // Update the chain-i's acceptor (>C=O) list
      hb.peer = j;
      m_chains[i]->acceptors.insert(hb);
      
      // Update the chain-j's donor (>N-H) list
      hb.peer = i;
      m_chains[j]->donors.insert(hb);
    }

    double calcHbonEnergy(int i, int j)
    {
      // HBLOW - LOWEST ALLOWED  ENERGY OF A HYDROGEN BOND IN CAL/MOL
      const double HBLOW = -9900.0;
      //const double HBHIGH= -500.0;

      // DIST - SMALLEST ALLOWED DISTANCE BETWEEN ANY ATOMS
      const double DIST = 0.5;

      // Q - COUPLING CONSTANT FOR ELECTROSTATIC ENERGY
      const double Q = -27888.0;

      double dho, dhc, dnc, dno;
      double hbe;
      
      hbe = 0;
      Backbone &WITH = *m_chains[i];
      if (WITH.resn.equals("pro"))
	return hbe;

      dho = (WITH.h - m_chains[j]->o).length();
      dhc = (WITH.h - m_chains[j]->c).length();
      dnc = (WITH.n - m_chains[j]->c).length();
      dno = (WITH.n - m_chains[j]->o).length();
      if (dho < DIST || dhc < DIST || dnc < DIST || dno < DIST)
	hbe = HBLOW;
      else
	hbe = Q/dho - Q/dhc + Q/dnc - Q/dno + 0.5;
      if (hbe > HBLOW)
	return hbe;

      LOG_DPRINTLN("CalcHbon> !!! Contact between residues %d, %d too close !!!", i, j);
      //Writeresidue(chain[i]);
      //fprintf(stderr," and ");
      //Writeresidue(chain[j]);
      //fprintf(stderr,"  too close !!!\n");
      hbe = HBLOW;
      return hbe;
    }

    ///////////////////////////////////////////////////////////////
    
    /**
       TESTBOND IS TRUE IF I IS DONOR[=NH] TO J, OTHERWISE FALSE
       (I ==> J)
    */
    bool isHbon(int i, int j)
    {

      Backbone &WITH = *m_chains[i];

      HbonList::const_iterator iter = WITH.acceptors.begin();

      for (int i=0; iter!=WITH.acceptors.end() && i<2; ++iter, ++i) {
	const HydrogenBond &hb = *iter;
	// if (hb.peer==j && hb.energy<HBHIGH)
	if (hb.peer==j)
	  return true;
      }
      return false;
    }
    
    //////////////////////////////////////////////////////////////////////

  public:

    void calcBridge()
    {
      int i;
      for (i=0; i<m_chains.size(); ++i) {
	checkBridge(i);
      }

      linkStrands();

#if 0
      for (i=0; i<m_chains.size(); ++i) {
	MolResiduePtr pres = m_chains[i]->pRes;
	MB_DPRINTLN("==%s %s%d %s", pres->getChainName().c_str(),
		    pres->getName().c_str(), pres->getIndex(),
		    m_chains[i]->ss.c_str());
	
      }
#endif

#if 0
      BridgeList::const_iterator iter = m_bridges.begin();
      for (; iter!=m_bridges.end(); ++iter) {
	const Bridge &with = *iter;
	MolResiduePtr pr_ib = m_chains[with.ib]->pRes;
	MolResiduePtr pr_ie = m_chains[with.ie]->pRes;
	MolResiduePtr pr_jb = m_chains[with.jb]->pRes;
	MolResiduePtr pr_je = m_chains[with.je]->pRes;

	MB_DPRINTLN("%s sheet", (with.btyp==parallel)?"parallel":"antiparallel");

	MB_DPRINTLN("  %s%d (%d) <--> %s%d (%d)",
		    pr_ib->getChainName().c_str(), pr_ib->getIndex(), with.ib,
		    pr_jb->getChainName().c_str(), pr_jb->getIndex(), with.jb);

	MB_DPRINTLN("  %s%d (%d) <--> %s%d (%d)",
		    pr_ie->getChainName().c_str(), pr_ie->getIndex(), with.ie,
		    pr_je->getChainName().c_str(), pr_je->getIndex(), with.je);
      }
#endif
    }

  private:

    void checkBridge(int i)
    {
      int j1, j2, j;
      int b;

      j1 = 0;
      j2 = 0;
      j = i + 3;
      
      // ȋOchain breakꍇ͉Ȃ
      if (!noChainBrk(i - 1, i + 1))
	return;

      for (j=i+3; j<m_chains.size() && j2==0; ++j) {
	if (!noChainBrk(j - 1, j + 1))
	  continue;
	
	if ((isHbon(i + 1, j) && isHbon(j, i - 1)) ||
	    (isHbon(j + 1, i) && isHbon(i, j - 1))) {
	  b = parallel;
	}
	else if ((isHbon(i + 1, j - 1) && isHbon(j + 1, i - 1)) ||
		 (isHbon(j, i) && isHbon(i, j))) {
	  b = antiparallel;
	}
	else
	  b = nobridge;

	if (b != nobridge) {
	  /* beta sheet̏ꍇ́DDD*/
	  if (j1 == 0) {
	    j1 = j; /* --> j1: iƏ߂bridgecԍɂȂ */
	    buildBridgeList(i, j, b);
	    //markExtended(i, j);
	  } else if (j != j1) {
	    /* --> j2: iƎbridgecԍɂȂ */
	    j2 = j; /* --> j2!=0ƂȂ̂ŁCŃ[v𔲂邱ƂɂȂ */
	    buildBridgeList(i, j, b);
	    //markExtended(i, j);
	  }
	}

      }
    }

    /**
       Build the bridge list (m_bridges)
     */
    void buildBridgeList(int i, int j, int btyp)
    {
      int k;
      bool found;
      
      found = false;
      k = 1;
      if (btyp == nobridge || i >= j)
	return;

      BridgeList::iterator iter = m_bridges.begin();
      for (; iter!=m_bridges.end(); ++iter) {
	Bridge &with = *iter;
	if (with.btyp!=btyp) continue;
	if (i != with.ie+1 || !noChainBrk(with.ie,i)) continue;
	
	if (btyp == parallel && j == with.je+1 && noChainBrk(with.je,j)) {
	  // append to the paralle sheet
	  with.ie++;
	  with.je++;
	  return;
	}
	else if (j == with.jb-1 && noChainBrk(j, with.jb)) {
	  // append to the antiparalle sheet
	  with.ie++;
	  with.jb--;
	  return;
	}
      }

      // (i,j) is a new bridge entry: register new object
      Bridge nb;
      nb.ib = i;
      nb.ie = i;
      nb.jb = j;
      nb.je = j;
      nb.btyp = btyp;
      nb.bIsolated = true;
      m_bridges.push_back(nb);
      return;
    }

    /*
      link the two bridges connected by a buldge
     */
    void linkBuldge(Bridge &Abr, Bridge &Bbr)
    {
      if (Abr.btyp==parallel) {
	MB_DPRINTLN("linking par: (%d:%d) --> (%d:%d)",
		    Abr.ie, Abr.je,
		    Bbr.ib, Bbr.jb);
	Bbr.ib = Abr.ie+1;
	Bbr.jb = Abr.je+1;
      }
      else {
	// antiparallel
	MB_DPRINTLN("linking antipar: (%d:%d) --> (%d:%d)",
		    Abr.ie, Abr.jb,
		    Bbr.ib, Bbr.je);
	Bbr.ib = Abr.ie+1;
	Abr.jb = Bbr.je+1;
      }
    }

    /**
       Check and link bridges connected by a buldge.
       If two the same b-type bridges are separated 1 and 4 residues,
       they are defined to be connected by a buldge region.
    */
    bool checkBuldge(Bridge &Abr, Bridge &Bbr)
    {
      int idiff, jdiff;

      if (Abr.btyp!=Bbr.btyp) return false;
      if (Abr.btyp==parallel) {
	idiff = Bbr.ib - Abr.ie;
	jdiff = Bbr.jb - Abr.je;
	if (1<=idiff && 1<=jdiff) {
	  if ((idiff<=2 && jdiff<=5) ||
	      (idiff<=5 && jdiff<=2)) {
	    linkBuldge(Abr, Bbr);
	    return true;
	  }
	}

	idiff = Abr.ib - Bbr.ie;
	jdiff = Abr.jb - Bbr.je;

	if (1<=idiff && 1<=jdiff) {
	  if ((idiff<=2 && jdiff<=5) ||
	      (idiff<=5 && jdiff<=2)) {
	    linkBuldge(Bbr, Abr);
	    return true;
	  }
	}
      }
      else {
	idiff = Bbr.ib - Abr.ie;
	jdiff = Abr.jb - Bbr.je;

	if (1<=idiff && 1<=jdiff) {
	  if ((idiff<=2 && jdiff<=5) ||
	      (idiff<=5 && jdiff<=2)) {
	    linkBuldge(Abr, Bbr);
	    return true;
	  }
	}

	idiff = Abr.ib - Bbr.ie;
	jdiff = Bbr.jb - Abr.je;

	if (1<=idiff && 1<=jdiff) {
	  if ((idiff<=2 && jdiff<=5) ||
	      (idiff<=5 && jdiff<=2)) {
	    linkBuldge(Bbr, Abr);
	    return true;
	  }
	}
      }

      return false;
    }

    /*
      Link and mark strands.
     */
    void linkStrands()
    {
      int i;

      BridgeList::iterator iter = m_bridges.begin();
      for (; iter!=m_bridges.end(); ++iter) {
	Bridge &iwith = *iter;
	if (iwith.ib!=iwith.ie)
	  iwith.bIsolated = false;
	  
	BridgeList::iterator jter = iter; //m_bridges.begin();
	++jter;
	for (; jter!=m_bridges.end(); ++jter) {
	  Bridge &jwith = *jter;
	  if (!checkBuldge(iwith, jwith)) continue;
	  iwith.bIsolated = false;
	  jwith.bIsolated = false;
	}	
      }

      iter = m_bridges.begin();
      for (; iter!=m_bridges.end(); ++iter) {
	Bridge &with = *iter;
	const char *cc = with.bIsolated?"B":"E";
	for (i=with.ib; i<=with.ie; ++i)
	  if (!m_chains[i]->ss.equals("E"))
	    m_chains[i]->ss = cc;

	for (i=with.jb; i<=with.je; ++i)
	  if (!m_chains[i]->ss.equals("E"))
	    m_chains[i]->ss = cc;
      }
    }

    //////////////////////////////////////////////////////////////////////

  public:
    void calcHelices()
    {
      int i, k, j, imax;

      //
      // Mark 3,4,5-turn structures
      //
      for (k = 3; k <= 5; ++k) {
	char cc = '0'+char(k);
	int idx = k-3;
        imax = qlib::max<int>(0, m_chains.size()-k);
	for (i=0; i<imax; ++i) {
	  //
	  // check the combination: i <== i+k
	  //
	  if (!noChainBrk(i, i + k)) continue;
	  if (!isHbon(i + k, i)) continue;

	  // i+k <-- set the '<' mark
          m_chains[i + k]->turn[idx] = '<';

	  //
	  // i+1`i+k-1̎cɁC}[Nccݒ
	  // ccɂ̓^[̕Ă
	  //
          for (j=i+1; j<i+k; ++j) {
            Backbone &with = *m_chains[j];
	    // łɐݒ肳ĂꍇoverwriteȂ悤
            if (with.turn[idx]==' ')
              with.turn[idx] = cc;
          }

	  //
	  // i <-- '>'}[Nݒ
	  // C'<'}[Nݒ肳Ăꍇ
	  // 'X'}[Nŏ(X=><̈ӖH)
	  //
          Backbone &with = *m_chains[i];
          if (with.turn[idx] == '<')
            with.turn[idx] = 'X';
          else
            with.turn[idx] = '>';
	}
      }

      /////////////////////////////
      // check alpha helices
      imax = qlib::max<int>(1, m_chains.size()-4);
      for (i = 1; i < imax; ++i) {
	//
	//  i-1  turn4  'X' or '>'@
	//  i    turn4  'X' or '>'
	//
	if (!isHxStart(i, 4)) continue;
		    
	// i, i+1, ..., i+3 alpha helix mark 'H'ݒ
	  for (j = i; j <= i + 3; j++)
	    m_chains[j]->ss = "H";
      }

      /////////////////////////////
      //  check 3-10 helices
      //
      imax = qlib::max<int>(1, m_chains.size()-3);
      for (i = 1; i < imax; ++i) {
	//
	//  i-1  turn3  'X'() or '>'(Jn)@
	//  i    turn3  'X'() or '>'(Jn)
	//

	if (!isHxStart(i, 3)) continue;

	//
	//  Ԃ̎cЂƂłC
	//  uss  'G'(3-10 helix)ł' '(蓖)łȂvC
	//   --> 'G'ɐݒ肵Ȃ
	//   iԂ͑S'G'蓖ĂłȂ΂ȂȂj	
	//
	bool bempty = true;
	for (j = i; j <= i + 2; j++) {
	  Backbone &with = *m_chains[j];
	  if (!with.ss.equals("G") && !with.ss.isEmpty())
	    bempty = false;
	}
	if (bempty) {
	  for (j = i; j <= i + 2; ++j)
	    m_chains[j]->ss = "G";
	}
      }

      /////////////////////////////
      //  check Pi helices
      //  (procedure is the same as the case of 3-10 helices)
      //
      imax = qlib::max<int>(1, m_chains.size()-5);
      for (i = 1; i < imax; ++i) {
	if (!isHxStart(i, 5)) continue;

	bool bempty = true;
	for (j = i; j <= i + 4; j++) {
	  Backbone &with = *m_chains[j];
	  if (!with.ss.equals("I") && !with.ss.isEmpty())
	    bempty = false;
	}
	if (bempty) {
	  for (j = i; j <= i + 4; ++j)
	    m_chains[j]->ss = "I";
	}
      }

      /////////////////////////////
      //  check turns
      for (i = 0; i < m_chains.size(); ++i) {
	Backbone &with = *m_chains[i];
	if (!with.ss.isEmpty()) continue;

	char cc = ' ';
	int turn;
	for (turn = 3; turn <= 5; ++turn) {
	  for (j = i-(turn-1); j <= i-1; j++) {
	    if (j<0) continue;
	    // MB_DPRINTLN("turn %d", j);
	    if (m_chains[j]->turn[turn-3]=='X' ||
		m_chains[j]->turn[turn-3]=='>')
	      cc = 'T';
	  }
	}

	with.ss = LString(cc);
      }
    }

  private:
    bool isHxStart(int i, int k) const
    {
      if (m_chains[i-1]->turn[k-3]!='>' &&
	  m_chains[i-1]->turn[k-3]!='X')
	return false;
      
      if (m_chains[i]->turn[k-3]!='>' &&
	  m_chains[i]->turn[k-3]!='X')
	return false;

      return true;
    }

  public:

    /**
       Set the calculated results to the property of residues.
    */
    void applyToMol()
    {
      int i;
      for (i=0; i<m_chains.size(); ++i) {
	Backbone &with = *(m_chains[i]);
	MolResiduePtr pres = with.pRes;
	if (with.ss.isEmpty() || with.ss.equals(" ")) continue;
        if (with.ss.equals("H") || with.ss.equals("G") || with.ss.equals("I"))
          pres->setPropStr("secondary", "helix");
        else if (with.ss.equals("E"))
          pres->setPropStr("secondary", "sheet");
      }
    }

    void doit()
    {
      calcHbon();
      calcBridge();
      calcHelices();
    }

  }; // class Prot2ndry

} // anonymous namespace

//////////////////////////////////////

using namespace molstr;

void MolCoord::calcProt2ndry()
{
  MolCoordPtr pMol(this);
  Prot2ndry ps;
  ps.init(pMol);
  if (ps.m_chains.size()<=0) {
    MB_DPRINTLN("calcProt2ndry> no amino acid residues in %d/%s",
                getUID(), getName().c_str());
    return;
  }
  ps.doit();
  ps.applyToMol();

#if 0
  int i;
  for (i=0; i<ps.m_chains.size(); ++i) {
    Backbone &with = *(ps.m_chains[i]);
    MolResiduePtr pres = with.pRes;
    MB_DPRINTLN("==%s %s%d %s %c%c%c(%d)", pres->getChainName().c_str(),
		pres->getName().c_str(), pres->getIndex(),
		with.ss.c_str(), with.turn[0], with.turn[1], with.turn[2],
		i);
    
  }
#endif

  MB_DPRINTLN("done.");
}

#if 0
void MolModule::calcProtDisulf(MolCoordPtr pMol)
{
}
#endif
    
  
