// -*-Mode: C++;-*-
//
// Abstract class for QDF writer classes
//
// $Id: QdfAbsWriter.cpp,v 1.3 2011/04/03 08:08:46 rishitani Exp $
//

#include <common.h>

#include "QdfAbsWriter.hpp"

#include <qlib/BinStream.hpp>

using namespace qsys;

QdfAbsWriter::QdfAbsWriter()
     : m_pBinOut(NULL)
{
}

QdfAbsWriter::~QdfAbsWriter()
{
  if (m_pBinOut!=NULL)
    delete m_pBinOut;
}


void QdfAbsWriter::start(qlib::OutStream &outs)
{
  if (m_pBinOut!=NULL)
    delete m_pBinOut;

  m_pBinOut = new qlib::BinOutStream(outs);

  // write QDF signature
  m_pBinOut->write("QDF0", 0, 4);

  // write endian info
  m_pBinOut->writeInt8(m_pBinOut->getIntByteOrder());

  // write BOM
  m_pBinOut->writeInt32(12345678);

  // write endian info
  m_pBinOut->writeInt8(m_pBinOut->getFloatByteOrder());

  // write FBOM
  m_pBinOut->writeFloat32(1.2345678f);

  // write QDF version no
  m_pBinOut->writeInt32(QDF_VERSION);

  // write file type string (any length)
  m_pBinOut->writeStr(m_strFileType);
}

void QdfAbsWriter::end()
{
  if (m_pBinOut==NULL)
    return;
  m_pBinOut->flush();
  delete m_pBinOut;
  m_pBinOut = NULL;
}

void QdfAbsWriter::setFileType(const LString &type)
{
  m_strFileType = type;
}

void QdfAbsWriter::defineData(const LString &name, int nrec)
{
  // write data name
  m_pBinOut->writeStr(name);

  // write record count in the data
  m_pBinOut->writeInt32(nrec);

  m_recdefs.clear();
  m_recmap.clear();
}
  
void QdfAbsWriter::defineRecord(const LString &name, int ntype)
{
  int nid = m_recdefs.size();
  m_recdefs.push_back(RecElem(name, ntype));
  m_recmap.set(name, nid);
}

void QdfAbsWriter::startData()
{
  int nreclen = m_recdefs.size();
  // write record size
  m_pBinOut->writeInt32(nreclen);

  BOOST_FOREACH (const RecElem &elem, m_recdefs) {
    m_pBinOut->writeStr(elem.first);
    m_pBinOut->writeInt8(elem.second);
  }
}

void QdfAbsWriter::endData()
{
  m_recdefs.clear();
  m_recmap.clear();
}

void QdfAbsWriter::startRecord()
{
  m_nRecInd = 0;
}

void QdfAbsWriter::endRecord()
{
}

void QdfAbsWriter::setRecValStr(const LString &name, const LString &value)
{
  const RecElem &elem = m_recdefs[m_nRecInd];
  if (!elem.first.equals(name) || elem.second!=QDF_TYPE_UTF8STR) {
    MB_THROW(qlib::FileFormatException, "setRecValStr inconsistent record order");
    return;
  }

  m_pBinOut->writeStr(value);
  ++m_nRecInd;
}

void QdfAbsWriter::setRecValInt32(const LString &name, int value)
{
  const RecElem &elem = m_recdefs[m_nRecInd];
  if (!elem.first.equals(name) || elem.second!=QDF_TYPE_INT32) {
    MB_THROW(qlib::FileFormatException, "setRecValInt inconsistent record order");
    return;
  }

  m_pBinOut->writeInt32(value);
  ++m_nRecInd;
}

void QdfAbsWriter::setRecValInt8(const LString &name, qint8 value)
{
  const RecElem &elem = m_recdefs[m_nRecInd];
  if (!elem.first.equals(name) || elem.second!=QDF_TYPE_INT8) {
    MB_THROW(qlib::FileFormatException, "setRecValInt8 inconsistent record order");
    return;
  }

  m_pBinOut->writeInt8(value);
  ++m_nRecInd;
}

void QdfAbsWriter::setRecValFloat32(const LString &name, qfloat32 value)
{
  const RecElem &elem = m_recdefs[m_nRecInd];
  if (!elem.first.equals(name) || elem.second!=QDF_TYPE_FLOAT32) {
    MB_THROW(qlib::FileFormatException, "setRecValFloat32 inconsistent record order");
    return;
  }

  m_pBinOut->writeFloat32(value);
  ++m_nRecInd;
}

