// -*-Mode: C++;-*-
//
// Style file reader/writer
//
// $Id: StyleFile.cpp,v 1.1 2011/04/16 17:30:39 rishitani Exp $

#include <common.h>

#include "StyleFile.hpp"
#include "StyleMgr.hpp"
#include "StyleSet.hpp"
#include "AutoStyleCtxt.hpp"

#include <gfx/SolidColor.hpp>
#include <gfx/NamedColor.hpp>

#include <qlib/FileStream.hpp>
#include <qlib/LDOM2Stream.hpp>

#define DELIM STYLEMGR_DB_DELIM

using namespace qsys;

using qlib::LDom2Node;
using gfx::SolidColor;
using gfx::SolidColorPtr;
using gfx::AbstractColor;
using gfx::NamedColor;

StyleFile::StyleFile()
{
  m_pTarg = StyleMgr::getInstance();
}

void StyleFile::loadNodes(LDom2Node *pRoot, qlib::uid_t nScopeID, const LString &src)
{
  if (!pRoot->getTagName().equals("styles")) {
    LString msg = LString::format("Invalid tag <%s> in style file", pRoot->getTagName().c_str());
    MB_THROW(qlib::FileFormatException, msg);
    return;
  }

  LString root_id = pRoot->getStrAttr("id");

  StyleList *pSL = m_pTarg->getCreateStyleList(nScopeID);
  MB_ASSERT(pSL!=NULL);

  StyleSet *pSet = new StyleSet;
  pSet->setContextID(nScopeID);
  pSet->setSource(src);
  pSet->setName(root_id);
  pSL->push_front(pSet);

  // define the loading style in the nScopeID context
  AutoStyleCtxt asc(nScopeID);

  for (pRoot->firstChild(); pRoot->hasMoreChild(); pRoot->nextChild()) {
    LDom2Node *pChNode = pRoot->getCurChild();
    if (pChNode->getTagName().equals("id")) continue;

    if (pChNode->getTagName().equals("color")) {
      loadPalette(pChNode, pSet);
      continue;
    }
    if (pChNode->getTagName().equals("material")) {
      loadMaterial(pChNode, pSet);
      continue;
    }
    if (pChNode->getTagName().equals("setting")) {
      loadSetting(pChNode, pSet);
      continue;
    }

    if (pChNode->getTagName().equals("style")) {
      loadStyle(pChNode, pSet);
      pRoot->detachCurChild();
      continue;
    }

    // default: string data node
    loadStrData(pChNode, pSet);
  }

  // Loading completed --> reset modified flag
  pSet->setModified(false);
}

void StyleFile::loadStream(qlib::InStream &ins, qlib::uid_t scope, const LString &src)
{
  //
  // Setup streams
  //
  qlib::LDom2InStream ois(ins);

  //
  // Construct data structure from the file
  //
  qlib::LDom2Tree tree;
  ois.read(tree);
  LDom2Node *pRoot = tree.top();
  // pRoot->dump();

  loadNodes(pRoot, scope, src);
  return;
}

bool StyleFile::loadFile(const LString &path, qlib::uid_t scope)
{
  qlib::FileInStream fis;
  try {
    fis.open(path);
  }
  catch (qlib::LException &e) {
    LOG_DPRINT("LoadStyle> cannot open file %s\n",path.c_str());
    LOG_DPRINT("LoadStyle>   (reason: %s)\n", e.getMsg().c_str());
    return false;
  }
    
  try {
    loadStream(fis, scope, path);
  }
  catch (qlib::LException &e) {
    LOG_DPRINT("LoadStyle> cannot read file %s\n",path.c_str());
    LOG_DPRINT("LoadStyle>   (reason: %s)\n", e.getMsg().c_str());
    return false;
  }

  return true;
}

void StyleFile::saveToNode(LDom2Node *pNode, qlib::uid_t nScopeID, const LString &basedir)
{
  StyleList *pSL = m_pTarg->getCreateStyleList(nScopeID);
  if (pSL==NULL || pSL->empty()) return;

  BOOST_FOREACH(StyleList::value_type pSet, *pSL) {
    if ( !pSet->getSource().isEmpty() ) {
      // external style reference is serialized as external reference node.
      LString relpath = qlib::makeRelativePath(pSet->getSource(), basedir);
      qlib::LDom2Node *pChNode = pNode->appendChild();
      pChNode->setTagName("styles");
      pChNode->appendStrAttr("src", relpath);
      continue;
    }
    // TO DO: serialize internal style definitions
  }  
}

//////////////////////////////////////////////////////////////////////////

void StyleFile::loadStyle(LDom2Node *pNode, StyleSet *pSet)
{
  LString id = pNode->getStrAttr("id");
  if (id.isEmpty()) {
    LString msg = "style tag does not have ID attr";
    MB_THROW(qlib::FileFormatException, msg);
    return;
  }
  pNode->removeChild("id");

  //LString key = id + DELIM + "style";
  pSet->putData(StyleMgr::makeStyleKey(id), pNode);
}

void StyleFile::loadStrData(LDom2Node *pChNode, StyleSet *pSet)
{
  LString tagname = pChNode->getTagName();
  LString id = pChNode->getStrAttr("id");
  if (id.isEmpty()) {
    LString msg = LString::format("Invalid tag <%s> in style palette file",
                                  pChNode->getTagName().c_str());
    MB_THROW(qlib::FileFormatException, msg);
    return;
  }

  LString value = pChNode->getValue();
  LString contents = pChNode->getContents();
  if (value.isEmpty() && !contents.isEmpty())
    value = contents;

  LString key = id + DELIM + "string" +DELIM + tagname;
  if (!pSet->putString(key, value)) {
    LString msg = LString::format("Invalid strData entry (key=%s, value=%s)",
                                  key.c_str(), value.c_str());
    MB_THROW(qlib::FileFormatException, msg);
    return;
  }
}

void StyleFile::loadPalette(LDom2Node *pNode, StyleSet *pSet)
{
  LString id = pNode->getStrAttr("id");
  LString mat = pNode->getStrAttr("material");
  LString value = pNode->getValue();
  
  if (id.isEmpty() || value.isEmpty()) {
    LString msg = LString::format("Invalid tag <%s> in style palette file",
                                  pNode->getTagName().c_str());
    MB_THROW(qlib::FileFormatException, msg);
    return;
  }
  
  ColorPtr pACol = ColorPtr(AbstractColor::fromStringS(value));
  if (pACol.isnull()) {
    LString msg = LString::format("Invalid value <%s> in style palette file",
                                  value.c_str());
    MB_THROW(qlib::FileFormatException, msg);
    return;
  }

  qlib::LScrSp<NamedColor> pNCol(pACol, qlib::no_throw_tag());
  if (!pNCol.isnull()) {
    // named color
    LString refname = pNCol->getName();
    //ColorPtr pRefCol = m_pTarg->getColor(refname, pSet->getContextID());
    ColorPtr pRefCol = m_pTarg->getColor(refname);
    if (pRefCol.isnull()) {
      LString msg = LString::format("Undefined named color <%s> in style palette file",
                                    value.c_str());
      MB_THROW(qlib::FileFormatException, msg);
      return;
    }
    //pACol = ColorPtr(static_cast<AbstractColor *>(pRefCol->clone()));
    pACol = procModCol(pNode, pRefCol);
  }

  // Only solid color can be registered to the pallete
  // this possibly throws
  SolidColorPtr pSCol(pACol);

  if (!mat.isEmpty())
    pSCol->setMaterial(mat);


  if (!pSet->putColor(id, pSCol)) {
    LString msg = LString::format("Invalid palette entry (id=%s, value=%s)",
                                  id.c_str(), value.c_str());
    MB_THROW(qlib::FileFormatException, msg);
    return;
  }
}

ColorPtr StyleFile::procModCol(LDom2Node *pNode, const ColorPtr &pRefCol)
{
  LString attr;
  int nr = pRefCol->r();
  int ng = pRefCol->g();
  int nb = pRefCol->b();
  int na = pRefCol->a();
  double hue, sat, bri, alp;
  AbstractColor::RGBtoHSB(nr, ng, nb, hue, sat, bri);
  alp = double(na)/255.0;
  double d;
  bool bChg = false;

  attr = pNode->getStrAttr("set_h");
  if (!attr.isEmpty()) {
    if (!attr.toDouble(&d))
      MB_THROW(qlib::FileFormatException, "set_h");
    // hue of UI is in degree unit
    hue = d/360.0;
    bChg = true;
  }

  attr = pNode->getStrAttr("set_s");
  if (!attr.isEmpty()) {
    if (!attr.toDouble(&d))
      MB_THROW(qlib::FileFormatException, "set_s");
    sat = d;
    bChg = true;
  }

  attr = pNode->getStrAttr("set_b");
  if (!attr.isEmpty()) {
    if (!attr.toDouble(&d))
      MB_THROW(qlib::FileFormatException, "set_b");
    bri = d;
    bChg = true;
  }

  attr = pNode->getStrAttr("set_a");
  if (!attr.isEmpty()) {
    if (!attr.toDouble(&d))
      MB_THROW(qlib::FileFormatException, "set_a");
    alp = d;
    bChg = true;
  }

  ///

  attr = pNode->getStrAttr("mod_h");
  if (!attr.isEmpty()) {
    if (!attr.toDouble(&d))
      MB_THROW(qlib::FileFormatException, "mod_h");
    // hue of UI is in degree unit
    hue += d/360.0;
    bChg = true;
  }

  attr = pNode->getStrAttr("mod_s");
  if (!attr.isEmpty()) {
    if (!attr.toDouble(&d))
      MB_THROW(qlib::FileFormatException, "mod_s");
    sat += d;
    bChg = true;
  }

  attr = pNode->getStrAttr("mod_b");
  if (!attr.isEmpty()) {
    if (!attr.toDouble(&d))
      MB_THROW(qlib::FileFormatException, "mod_b");
    bri += d;
    bChg = true;
  }

  attr = pNode->getStrAttr("mod_a");
  if (!attr.isEmpty()) {
    if (!attr.toDouble(&d))
      MB_THROW(qlib::FileFormatException, "mod_a");
    alp += d;
    bChg = true;
  }

  if (!bChg)
    return pRefCol;

  double ddr,ddg,ddb;
  AbstractColor::HSBtoRGB(hue, sat, bri, ddr, ddg, ddb);
  return ColorPtr(new SolidColor(ddr, ddg, ddb, alp));
}

void StyleFile::loadMaterial(LDom2Node *pNode, StyleSet *pSet)
{
  LString mat_id = pNode->getStrAttr("id");
  if (mat_id.isEmpty()) {
    LString msg = LString::format("Tag <%s> requires id attribute",
                                  pNode->getTagName().c_str());
    MB_THROW(qlib::FileFormatException, msg);
    return;
  }

  for (pNode->firstChild(); pNode->hasMoreChild(); pNode->nextChild()) {
    LDom2Node *pChNode = pNode->getCurChild();

    if (pChNode->getTagName().equals("id")) continue;

    if (!pChNode->getTagName().equals("def")) {
      LString msg = LString::format("Invalid tag <%s> in style material section",
                                    pChNode->getTagName().c_str());
      MB_THROW(qlib::FileFormatException, msg);
      return;
    }

    LString rend_type = pChNode->getStrAttr("type");
    LString value;
    if (!rend_type.isEmpty()) {
      // renderer-type-dependent string material definition
      value = pChNode->getValue();
      LString contents = pChNode->getContents();
      if (value.isEmpty() && !contents.isEmpty())
      value = contents;
      
      // LString key = mat_id+DELIM+rend_type+DELIM+"mat";
      // if (!pSet->putString(key, value)) {
      // MB_DPRINTLN("Cannot put material (%s,%s)", mat_id.c_str(), rend_type.c_str());
      // }
      if (!pSet->putMaterial(mat_id, rend_type, value))
        MB_DPRINTLN("Cannot put material (%s,%s)", mat_id.c_str(), rend_type.c_str());
    }
    else {
      /// Material definition for internal (OpenGL) renderer
      double dvalue;
      value = pChNode->getStrAttr("ambient");
      if (value.toRealNum(&dvalue))
        if (!pSet->putMaterial(mat_id, Material::MAT_AMBIENT, dvalue))
          MB_DPRINTLN("Cannot put material (%s)", mat_id.c_str());
      value = pChNode->getStrAttr("diffuse");
      if (value.toRealNum(&dvalue))
        if (!pSet->putMaterial(mat_id, Material::MAT_DIFFUSE, dvalue))
          MB_DPRINTLN("Cannot put material (%s)", mat_id.c_str());
      value = pChNode->getStrAttr("specular");
      if (value.toRealNum(&dvalue))
        if (!pSet->putMaterial(mat_id, Material::MAT_SPECULAR, dvalue))
          MB_DPRINTLN("Cannot put material (%s)", mat_id.c_str());

      value = pChNode->getStrAttr("shininess");
      if (value.toRealNum(&dvalue))
        if (!pSet->putMaterial(mat_id, Material::MAT_SHININESS, dvalue))
          MB_DPRINTLN("Cannot put material (%s)", mat_id.c_str());
      value = pChNode->getStrAttr("emission");
      if (value.toRealNum(&dvalue))
        if (!pSet->putMaterial(mat_id, Material::MAT_EMISSION, dvalue))
          MB_DPRINTLN("Cannot put material (%s)", mat_id.c_str());
    }
  }
  
}

void StyleFile::loadSetting(LDom2Node *pNode, StyleSet *pSet)
{
  LString type = pNode->getStrAttr("type");
  if (type.isEmpty()) {
    LString msg = LString::format("Tag <%s> requires type attribute",
                                  pNode->getTagName().c_str());
    MB_THROW(qlib::FileFormatException, msg);
    return;
  }

  for (pNode->firstChild(); pNode->hasMoreChild(); pNode->nextChild()) {
    LDom2Node *pChNode = pNode->getCurChild();
    const LString &tagname = pChNode->getTagName();
    if (tagname.equals("type")) continue;
    LString value = pChNode->getValue();
    LString contents = pChNode->getContents();
    if (value.isEmpty() && !contents.isEmpty())
      value = contents;
    
    LString key = tagname + DELIM + type + DELIM + "cfg";
    if (!pSet->putString(key, value)) {
      MB_DPRINTLN("Cannot put setting (%s,%s)", tagname.c_str(), type.c_str());
    }
  }
}

