//
// Text rendering routine using Thebes API
//
// $Id: ThebesTextRender.cpp,v 1.16 2011/03/04 18:34:46 rishitani Exp $
//

#include <common.h>
#include <boost/multi_array.hpp>
#include <boost/shared_ptr.hpp>

#include "xpcom.hpp"

//#include <nsIInterfaceRequestor.h>
//#include <nsIInterfaceRequestorUtils.h>
//#include <docshell/nsIDocShell.h>
//#include <docshell/nsIWebNavigation.h>
//#include <widget/nsIBaseWindow.h>
//#include <widget/nsIWidget.h>
//#include <layout/nsIPresShell.h>
//#include <gfx/nsIRenderingContext.h>
//#include <gfx/nsIDeviceContext.h>
//#include <gfx/nsIFontMetrics.h>
//#include <gfx/nsFont.h>

//#include <thebes/gfxContext.h>
//#include <thebes/gfxImageSurface.h>
//#include <thebes/gfxPlatform.h>

#include <gfxContext.h>
#include <gfxImageSurface.h>
#include <gfxPlatform.h>

#define nsString_h___
//#include <thebes/gfxFont.h>
//#include <thebes/gfxTextRunCache.h>

#include <gfxFont.h>
#include <gfxTextRunCache.h>

//////////

#include "XPCCueMol.hpp"
#include "XPCObjWrapper.hpp"
#include <gfx/TextRenderManager.hpp>
#include <gfx/PixelBuffer.hpp>

//////////

#include <qlib/Utils.hpp>

using qlib::LString;

#ifdef WIN32

class gfxMySurface : public gfxImageSurface
{
public:
  gfxMySurface(const gfxIntSize& size, gfxImageFormat format)
       :gfxImageSurface(size, format) {}

  gfxMySurface(cairo_surface_t *csurf)
       :gfxImageSurface(csurf){}

  virtual ~gfxMySurface()
    {
      //MB_DPRINTLN("surface %p destructed.", this);
    }

  nsresult BeginPrinting(const nsAString& aTitle, const nsAString& aPrintToFileName)
    {
      return NS_OK;
    }

  nsresult EndPrinting()
    {
      return NS_OK;
    }

  nsresult AbortPrinting()
    {
      return NS_OK;
    }
};

#else

typedef gfxImageSurface gfxMySurface;

#endif


class ThebesTextRender : public gfx::TextRenderImpl
{
private:

  gfxContext *mThebesContext;
  gfxASurface *mThebesSurface;
  gfxFontGroup *mFontGroup;

  nsString mTextStyle;

  std::vector<QUE_BYTE> m_bytes;

  typedef boost::const_multi_array_ref<QUE_BYTE, 3> ConstArrayRef;
  typedef boost::multi_array_ref<QUE_BYTE, 3> ArrayRef;

  QUE_BYTE bit_reverse(QUE_BYTE bData)
  {
    QUE_BYTE lookup[] =
    { 0, 8,  4, 12,  
      2, 10, 6, 14 ,  
      1, 9,  5, 13, 
      3, 11, 7, 15 }; 
    QUE_BYTE ret_val = (QUE_BYTE)(((lookup[(bData & 0x0F)]) << 4) + lookup[((bData & 0xF0) >> 4)]); 
    return ret_val; 
  }

  /// Convert Thebes image surface data to qlib PixelBuffer object
  void convertToPixelBuffer(gfx::PixelBuffer &bytes, int w, int h)
  {
    int i, j;

    gfxImageSurface *pImgSurf = static_cast<gfxImageSurface *>(mThebesSurface);
    QUE_BYTE *psf = pImgSurf->Data();
    int stride = pImgSurf->Stride();
    int ntype = pImgSurf->Format();
    
    //ConstArrayRef source(psf, boost::extents[h][w][4]);

    if (ntype==gfxASurface::ImageFormatA8) {
      int nsize = w*h;
      if (bytes.size()<nsize)
        bytes.resize(nsize);

      ConstArrayRef source(psf, boost::extents[h][w][1]);
      ArrayRef dest(bytes.data(), boost::extents[h][w][1]);
      for (j=0; j<h; j++) {
        for (i=0; i<w; i++) {
          //MB_DPRINTLN("(x=%d,y=%d,c=%d) = %d", i,j,0,xxx);
          dest[h-j-1][i][0] = source[j][i][0];
        }
      }
      
      bytes.setDepth(8);
      bytes.setWidth(w);
      bytes.setHeight(h);
    }
    else if (ntype==gfxASurface::ImageFormatA1) {
      int w2 = w/8;
      if (w2%4!=0)
        w2 += (4-w2%4);
      int nsize = w2*h;

      if (bytes.size()<nsize)
        bytes.resize(nsize);

      QUE_BYTE *dest = bytes.data();
      const QUE_BYTE *source = psf;
      for (j=0; j<h; ++j) {
        int mj = h-j-1;
        for (i=0; i<stride; i++) {
          QUE_BYTE val = source[j*stride+i];
          val = bit_reverse(val);
          dest[mj*w2+i] = val;
        }
        for (; i<w2; i++) {
          dest[mj*w2+i] = 0;
        }
      }

      bytes.setDepth(1);
      bytes.setWidth(w);
      bytes.setHeight(h);
    }

  }

  int m_n;

  double m_dFontSize;
  LString m_strFontName;
  PRUint8 m_nFontStyle;
  PRUint16 m_nFontWeight;
  bool m_bUseAA;

public:
  ThebesTextRender()
  {
    //m_pBytes = NULL;
    //m_nByteSize = 0;
    mThebesSurface = new gfxMySurface(gfxIntSize(1, 1), gfxASurface::ImageFormatARGB32);
    mThebesSurface->AddRef();
    mThebesContext = new gfxContext(mThebesSurface);
    mThebesContext->AddRef();
    mFontGroup = NULL;
    m_dFontSize = 0.0;
    m_nFontStyle = 0;
    m_nFontWeight = 0;
    //m_bUseAA = false;
    m_bUseAA = true;
  }
  virtual ~ThebesTextRender() {
    delete mThebesContext;
    //delete mThebesSurface;

    //mThebesContext->Release();
    mThebesSurface->Release();
    if (mFontGroup != NULL)
      mFontGroup->Release();
  }

  bool setupFont(double fontsize, const LString &fontname,
                 const LString &font_style,
                 const LString &font_wgt) {
    gfxFontStyle style;

    if (font_style.equalsIgnoreCase("italic")) {
      style.style = FONT_STYLE_ITALIC;
    }
    else if (font_style.equalsIgnoreCase("oblique")) {
      style.style = FONT_STYLE_OBLIQUE;
    }
    //else if (font_style.equalsIgnoreCase("normal")) {
    else {
      // default is normal
      style.style = FONT_STYLE_NORMAL;
    }

    if (font_wgt.equalsIgnoreCase("bold")) {
      style.weight = FONT_WEIGHT_BOLD;
    }
    //if (font_wgt.equalsIgnoreCase("normal")) {
    else {
      // default is normal
      style.weight = FONT_WEIGHT_NORMAL;
    }

    if (qlib::isNear4(m_dFontSize, fontsize) &&
        m_strFontName.equals(fontname) &&
        m_nFontStyle==style.style &&
        m_nFontWeight==style.weight)
      return true; // already has the same fontface

    style.size = gfxFloat(fontsize); //12.0;
    style.sizeAdjust = 0.0;
    style.systemFont = PR_FALSE;
    style.familyNameQuirks = PR_FALSE;

    nsCAutoString nsstr(fontname.c_str());
    nsAutoString ucs16;
    ::CopyUTF8toUTF16(nsstr, ucs16);

    mFontGroup = gfxPlatform::GetPlatform()->
      CreateFontGroup(ucs16
                      , &style
		      , nsnull
                      );
    mFontGroup->AddRef();

    m_dFontSize = fontsize;
    m_strFontName = fontname;
    m_nFontStyle = style.style;
    m_nFontWeight = style.weight;
    return true;
  }

  bool makeSurface(int w, int h)
  {
    //mThebesContext->Release();
    delete mThebesContext;
    int n = (int) mThebesSurface->Release();
    //m_n = n;
    //delete mThebesSurface;
    
    gfxASurface::gfxImageFormat nfmt;
    if (m_bUseAA)
      nfmt = gfxASurface::ImageFormatA8;
    else
      nfmt = gfxASurface::ImageFormatA1;
    mThebesSurface = new gfxMySurface(gfxIntSize(w, h), nfmt);
    mThebesSurface->AddRef();

    mThebesContext = new gfxContext(mThebesSurface);
    mThebesContext->AddRef();

    return true;
  }

  bool measureText(const LString &str, gfxTextRun::Metrics &rval)
  {

    const PRUint8 *pstr = reinterpret_cast<const PRUint8 *>(str.c_str());

    // XXX
    PRUint32 aupdp = 1, aupcp = 1;
    PRUint32 textrunflags = 0;

    //gfxTextRunCache::AutoTextRun textRun;
    gfxTextRun *textRun;
    textRun = gfxTextRunCache::MakeTextRun(pstr,
                                           str.length(),
                                           mFontGroup,
                                           mThebesContext,
                                           aupdp,
                                           textrunflags);

    if(!textRun)
      return false;

    PRBool tightBoundingBox = PR_FALSE;
    rval =
      textRun->MeasureText(/* offset = */ 0, str.length(),
                           gfxFont::TIGHT_INK_EXTENTS, mThebesContext,
                           nsnull);

    gfxTextRunCache::ReleaseTextRun(textRun);
    return true;
  }

  virtual bool renderText(const qlib::LString &str, gfx::PixelBuffer &buf) {
    gfxTextRun::Metrics mtx;
    if (!measureText(str, mtx)) {
      LOG_DPRINTLN("measure text failed");
      return false;
    }
    
    //MB_DPRINTLN("RenderText (w,h) = %f,%f", mtx.mBoundingBox.Width(), mtx.mBoundingBox.Height());
    //MB_DPRINTLN("RenderText (x,y) = %f,%f", mtx.mBoundingBox.X(), mtx.mBoundingBox.Y());
    
    int width = (int) mtx.mBoundingBox.Width();
    int height = (int) mtx.mBoundingBox.Height();

    if (width%4!=0)
      width += (4-width%4);

    //return false;
    
    if (!makeSurface(width, height))
      return false;

    const PRUint8 *pstr = reinterpret_cast<const PRUint8 *>(str.c_str());
    PRUint32 textrunflags = 0;

    // XXX
    PRUint32 aupdp = 1;

    gfxTextRunCache::AutoTextRun textRun;
    textRun = gfxTextRunCache::MakeTextRun(pstr,
                                           str.length(),
                                           mFontGroup,
                                           mThebesContext,
                                           aupdp,
                                           textrunflags);

    if(!textRun.get())
      return false;

    gfxPoint pt(0.0f, -mtx.mBoundingBox.Y());

    mThebesContext->SetColor(gfxRGBA(0.0, 0.0, 0.0, 0.0));
    mThebesContext->Paint();
    mThebesContext->SetColor(gfxRGBA(1.0, 1.0, 1.0, 1.0));

    textRun->Draw(mThebesContext,
                  pt,
                  0, // offset
                  str.length(),
                  nsnull,
                  nsnull,
                  nsnull);

    convertToPixelBuffer(buf, width, height);
    return true;
  }

  virtual void setMouseCursor(int ncursor) {
  }


};

using namespace xpcom;

bool XPCCueMol::initTextRender()
{
  ThebesTextRender *pTTR = new ThebesTextRender;
  gfx::TextRenderManager *pTRM = gfx::TextRenderManager::getInstance();
  pTRM->setImpl(pTTR);
  pTTR->setupFont(12.0, "sans-serif", "normal", "normal");
  //pTTR->setupFont(20.0, "Times New Roman", FONT_STYLE_NORMAL, FONT_WEIGHT_NORMAL);

  m_pTR = pTTR;
  return true;
}

void XPCCueMol::finiTextRender()
{
  ThebesTextRender *pTTR = static_cast<ThebesTextRender *>(m_pTR);
  delete pTTR;
}

