// -*-Mode: C++;-*-
//
//  Tube section class
//
//  $Id: TubeSection.cpp,v 1.4 2011/01/02 13:11:02 rishitani Exp $

#include <common.h>

#include "TubeSection.hpp"
#include <qlib/Vector4D.hpp>

#include <gfx/DisplayContext.hpp>

using namespace molvis;
//using namespace molstr;

TubeSection::TubeSection()
{
  m_pSectTab = NULL;

  //m_nSectType = TS_ELLIPTICAL;
  //m_lw = 0.35;
  //m_tuber = 1.0;
  //m_alpha = 0.4;
  //m_nSectDetail = 16;

  resetAllProps();
}

TubeSection::~TubeSection()
{
  if (m_pSectTab!=NULL)
    delete [] m_pSectTab;
}

void TubeSection::setupEllipticalSection()
{
  int i;

  m_nSectTabSz = m_nSectDetail;
  m_pSectTab = MB_NEW Vector4D[m_nSectTabSz];
  
  double dth = M_PI/double(m_nSectTabSz);
  
  for (i=0; i<m_nSectTabSz; i++) {
    const double th = double(i)*2.0*M_PI/double(m_nSectTabSz);
    const double si = ::sin(th+dth);
    const double co = ::cos(th+dth);
    const double nlen = ::sqrt(co*co*m_tuber*m_tuber + si*si);
    m_pSectTab[i].x() = co*m_lw;
    m_pSectTab[i].y() = si*m_lw*m_tuber;
    m_pSectTab[i].z() = co*m_tuber/nlen;
    m_pSectTab[i].w() = si/nlen;
  }
}

void TubeSection::setupSquareSection()
{
  double Rx, Ax, Ay;
  if (m_tuber>1.0) {
    Rx = m_lw*(1-m_alpha);
    Ax = m_lw*m_alpha;
    Ay = m_lw*m_tuber - Rx;
  }
  else {
    Rx = m_lw*m_tuber*(1-m_alpha);
    Ax = m_lw - Rx;
    Ay = m_lw*m_tuber*m_alpha;
  }

  const double len = (Ax+Ay)*2.0 + Rx*M_PI*2.0;
  const double ddel = len/double(m_nSectDetail);
  const int Nx = qlib::max(1, int(::floor(Ax*2.0/ddel)));
  const int Ny = qlib::max(1, int(::floor(Ay*2.0/ddel)));
  const int Nr = qlib::max(1, int(::floor(Rx*M_PI*0.5/ddel))) * 2;
  m_nSectTabSz = Nx*2 + Ny*2 + Nr*4;
  m_pSectTab = MB_NEW Vector4D[m_nSectTabSz];

  int i,j;
  double th, si, co;


  for (j=0, i=0; j<Nr; i++, j++) {
    th = double(j)*M_PI/(2.0*double(Nr));
    si = ::sin(th); co = ::cos(th);
    m_pSectTab[i].x() = Ax + co*Rx;
    m_pSectTab[i].y() = Ay + si*Rx;
    m_pSectTab[i].z() = co;
    m_pSectTab[i].w() = si;
  }
  for (j=0; j<Nx; i++, j++) {
    const double d = 2.0*Ax * double(j)/double(Nx);
    m_pSectTab[i].x() = Ax - d;
    m_pSectTab[i].y() = Ay + Rx;
    m_pSectTab[i].z() = 0.0;
    m_pSectTab[i].w() = 1.0;
  }
  for (j=0; j<Nr; i++, j++) {
    th = double(j)*M_PI/(2.0*double(Nr)) + M_PI*0.5;
    si = ::sin(th); co = ::cos(th);
    m_pSectTab[i].x() = - Ax + co*Rx;
    m_pSectTab[i].y() = Ay + si*Rx;
    m_pSectTab[i].z() = co;
    m_pSectTab[i].w() = si;
  }
  for (j=0; j<Ny; i++, j++) {
    const double d = 2.0*Ay * double(j)/double(Ny);
    m_pSectTab[i].x() = -Ax - Rx;
    m_pSectTab[i].y() = Ay - d;
    m_pSectTab[i].z() = -1.0;
    m_pSectTab[i].w() =  0.0;
  }
  for (j=0; j<Nr; i++, j++) {
    th = double(j)*M_PI/(2.0*double(Nr)) + M_PI;
    si = ::sin(th); co = ::cos(th);
    m_pSectTab[i].x() = - Ax + co*Rx;
    m_pSectTab[i].y() = - Ay + si*Rx;
    m_pSectTab[i].z() = co;
    m_pSectTab[i].w() = si;
  }
  for (j=0; j<Nx; i++, j++) {
    const double d = 2.0*Ax * double(j)/double(Nx);
    m_pSectTab[i].x() = -Ax + d;
    m_pSectTab[i].y() = -Ay - Rx;
    m_pSectTab[i].z() =  0.0;
    m_pSectTab[i].w() = -1.0;
  }
  for (j=0; j<Nr; i++, j++) {
    th = double(j)*M_PI/(2.0*double(Nr)) + M_PI*1.5;
    si = ::sin(th); co = ::cos(th);
    m_pSectTab[i].x() = Ax + co*Rx;
    m_pSectTab[i].y() = - Ay + si*Rx;
    m_pSectTab[i].z() = co;
    m_pSectTab[i].w() = si;
  }
  for (j=0; j<Ny; i++, j++) {
    const double d = 2.0*Ay * double(j)/double(Ny);
    m_pSectTab[i].x() =  Ax + Rx;
    m_pSectTab[i].y() = -Ay + d;
    m_pSectTab[i].z() =  1.0;
    m_pSectTab[i].w() =  0.0;
  }
}

void TubeSection::setupRectSection()
{
  const double Ax = m_lw;
  const double Ay = m_lw*m_tuber;
  const double Wx = Ax*2.0;
  const double Wy = Ay*2.0;

  const double delta = (2.0*Wx+2.0*Wy)/double(m_nSectDetail);
  //MB_DPRINTLN("TubSec> delta = %f", delta);
  const int Nx = qlib::max(1, int(::floor(Wx/delta)));
  const int Ny = qlib::max(1, int(::floor(Wy/delta)));
  //MB_DPRINTLN("TubSec> Wx(%f)/del = Nx = %d", Wx, Nx);
  //MB_DPRINTLN("TubSec> Wy(%f)/del = Ny = %d", Wy, Ny);
  m_nSectTabSz = 8 + (Nx-1)*2 + (Ny-1)*2;
  m_pSectTab = MB_NEW Vector4D[m_nSectTabSz];

  // const double Rx = m_lw*(1-m_alpha);
  int i = 0, j;

  m_pSectTab[i] = Vector4D(Ax, Ay, 1.0, 0.0);
  ++i;
  m_pSectTab[i] = Vector4D(Ax, Ay, 0.0, 1.0);
  ++i;

  for (j=0; j<Nx-1; ++j, ++i) {
    m_pSectTab[i] = Vector4D(Ax - (Wx*(j+1.0))/double(Nx), Ay, 0.0, 1.0);
  }

  m_pSectTab[i] = Vector4D(-Ax, Ay, 0.0, 1.0);
  ++i;
  m_pSectTab[i] = Vector4D(-Ax, Ay, -1.0, 0.0);
  ++i;

  for (j=0; j<Ny-1; ++j, ++i) {
    //m_pSectTab[i] = Vector4D(-Ax, 0, -1.0, 0.0);
    m_pSectTab[i] = Vector4D(-Ax, Ay - (Wy*(j+1.0))/double(Ny), -1.0, 0.0);
  }

  m_pSectTab[i] = Vector4D(-Ax, -Ay, -1.0, 0.0);
  ++i;
  m_pSectTab[i] = Vector4D(-Ax, -Ay, 0.0, -1.0);
  ++i;

  for (j=0; j<Nx-1; ++j, ++i) {
    //m_pSectTab[i] = Vector4D(0, -Ay, 0.0, -1.0);
    m_pSectTab[i] = Vector4D(-Ax + (Wx*(j+1.0))/double(Nx), -Ay, 0.0, -1.0);
  }

  m_pSectTab[i] = Vector4D(Ax, -Ay, 0.0, -1.0);
  ++i;
  m_pSectTab[i] = Vector4D(Ax, -Ay, 1.0, 0.0);
  ++i;

  for (j=0; j<Ny-1; ++j, ++i) {
    //m_pSectTab[i] = Vector4D(Ax, 0, 1.0, 0.0);
    m_pSectTab[i] = Vector4D(Ax, -Ay + (Wy*(j+1.0))/double(Ny), 1.0, 0.0);
  }
}

void TubeSection::setupSectionTable()
{
  if (m_pSectTab!=NULL)
    delete [] m_pSectTab;

  switch (m_nSectType) {
  default:
  case TS_ELLIPTICAL: {
    setupEllipticalSection();
    break;
  }
    
  case TS_SQUARE: {
    setupSquareSection();
    break;
  }

  case TS_RECT: {
    setupRectSection();
    break;
  }
    
  } // switch

  return;
}

void TubeSection::invalidate()
{
  if (m_pSectTab!=NULL)
    delete [] m_pSectTab;
  m_pSectTab = NULL;
}

bool TubeSection::resetProperty(const LString &propnm)
{
  bool res = StyleResetPropImpl::resetProperty(propnm, this);
  if (!res) {
    // stylesheet value is not found --> default behaviour
    return super_t::resetProperty(propnm);
  }

  return true;
}

///////////////////////////////////////////////////////
// Cap rendering routine

void TubeSection::makeCap(DisplayContext *pdl,
                           bool fStart, int nType,
                           const Vector4D &f, const Vector4D &vpt,
                           const Vector4D &e1, const Vector4D &e2)
{
  switch (nType) {
  case 2:
  default:
    // no cap (transparent)
    break;

  case 1:
    // Flat cap
    makeFlatCap(pdl, fStart, f, vpt, e1, e2);
    break;

  case 0:
    // Spherical cap
    makeSpherCap(pdl, fStart, f, vpt, e1, e2);
    break;
  }
}

void TubeSection::makeFlatCap(DisplayContext *pdl,
                               bool fStart,
                               const Vector4D &f, const Vector4D &vpt,
                               const Vector4D &e1, const Vector4D &e2)
{
  int j;
  Vector4D norm = vpt.normalize();

  // std::deque<Vector4D> tmpv;

  pdl->startTriangleFan();
  if (fStart)
    pdl->normal(-norm);
  else
    pdl->normal(norm);
  
  pdl->vertex(f);

  if (!fStart) {
    for (j=0; j<=getSize(); j++) {
      Vector4D g2 = getVec(j, e1, e2);
      pdl->vertex(f+g2);
      //tmpv.push_back(f+g2);
      //tmpv.push_back(f+g2+norm);
    }
  }
  else {
    for (j=getSize(); j>=0; j--) {
      Vector4D g2 = getVec(j, e1, e2);
      pdl->vertex(f+g2);
      //tmpv.push_back(f+g2);
      //tmpv.push_back(f+g2-norm);
    }
  }
  pdl->end();

/*
  pdl->startLines();
  BOOST_FOREACH (const Vector4D &elem, tmpv) {
    pdl->vertex(elem);
  }
  pdl->end();
 */
}

void TubeSection::makeSpherCap(DisplayContext *pdl,
                                bool fStart,
                                const Vector4D &f, const Vector4D &vpt,
                                const Vector4D &e1, const Vector4D &e2)
{
  int i,j;

  const int detail = 5;
  double sign, e1len = getVec(0, e1, e2).length();
  sign = (fStart) ? -1.0 : 1.0;
  
  Vector4D v = vpt.scale(1.0/vpt.length());
  v = v.scale(sign*e1len/double(detail));
  
  double gpar2, t2;
  Vector4D f2, e21, e22;
  
  //  int stab_sz =
  for (i=0; i<=detail; i++) {
    double t = double(i)/double(detail);
    double gpar = ::sqrt(1.0-t*t);
    //double dgp = -t/gpar;
    Vector4D f1 = f+v.scale(double(i));
    Vector4D e11 = e1.scale(gpar);
    Vector4D e12 = e2.scale(gpar);
    
    if (i==0) {
      t2 = t;
      gpar2 = gpar;
      f2 = f1;
      e21 = e11;
      e22 = e12;
      continue;
    }
    
    // render tube body
    pdl->startTriangleStrip();
    for (j=0; j<=getSize(); j++) {
      Vector4D stab = getSectTab(j);
      
      Vector4D g1 = e11.scale(stab.x()) + e12.scale(stab.y());
      Vector4D dg1 = e11.scale(stab.z()) + e12.scale(stab.w());
      Vector4D dgp1 = e11.scale(-stab.w()) + e12.scale(stab.z());
      if (i==detail)
        dg1 = v;
      else
        dg1 = dg1.scale(e1len*gpar) - (dgp1.cross(g1)).scale(t*sign);
      
      Vector4D g2 = e21.scale(stab.x()) + e22.scale(stab.y());
      Vector4D dg2 = e21.scale(stab.z()) + e22.scale(stab.w());
      Vector4D dgp2 = e21.scale(-stab.w()) + e22.scale(stab.z());
      dg2 = dg2.scale(e1len*gpar2) - (dgp2.cross(g2)).scale(t2*sign);
      
      // pdl->color(col);
      if (fStart) {
        pdl->normal(dg2);
        pdl->vertex(f2+g2);
        pdl->normal(dg1);
        pdl->vertex(f1+g1);
      }
      else {
        pdl->normal(dg1);
        pdl->vertex(f1+g1);
        pdl->normal(dg2);
        pdl->vertex(f2+g2);
        }
    }
    
    pdl->end();
    t2 = t;
    gpar2 = gpar;
      f2 = f1;
    e21 = e11;
    e22 = e12;
    
  }
}

///////////////////////////////
// Tube tesselation routines

//#define DEBUG_SHOW_NORMAL 1

void TubeSection::startTess()
{
  const int nsize = getSize();
  m_vtess.resize(nsize+1);
  m_ntess.resize(nsize+1);
  m_bTessEmpty = true;
}

void TubeSection::doTess(DisplayContext *pdl,
                         const Vector4D &f1,
                         const gfx::ColorPtr &pCol, bool bSmoothCol,
                         const Vector4D &e11, const Vector4D &e12,
                         const Vector4D &norm_shift)
{
  const int nsize = getSize();
  std::vector<Vector4D> vts(nsize+1), nts(nsize+1);

  for (int k=0; k<=nsize; k++) {
    vts[k] = f1 + getVec(k, e11, e12);
    nts[k] = ( getNormVec(k, e11, e12) + norm_shift ).normalize();
  }

  if (!m_bTessEmpty) {
    pdl->startTriangleStrip();
    for (int k=0; k<=nsize; k++) {
      pdl->normal(nts[k]);
      pdl->color(pCol);
      pdl->vertex(vts[k]);

      pdl->normal(m_ntess[k]);
      if (bSmoothCol)
        pdl->color(m_pPrevCol);
      pdl->vertex(m_vtess[k]);
    }
    pdl->end();
  }

  for (int k=0; k<=nsize; k++) {
    m_vtess[k] = vts[k];
    m_ntess[k] = nts[k];
  }
  m_pPrevCol = pCol;
  m_bTessEmpty = false;

}

void TubeSection::doTess(DisplayContext *pdl,
                         const Vector4D &f1,
                         const gfx::ColorPtr &pCol, bool bSmoothCol,
                         const Vector4D &e11, const Vector4D &e12,
                         const Vector4D &escl, const Vector4D &vpt)
{
  const int nsize = getSize();
  std::vector<Vector4D> vts(nsize+1), nts(nsize+1);

  for (int k=0; k<=nsize; k++) {
    vts[k] = f1 + getVec(k, e11, e12);
    nts[k] = ( getNormVec(k, e11, e12).normalize() + calcDnorm(k, escl, vpt) ).normalize();
    //nts[k] = ( getNormVec(k, e11, e12) ).normalize();
  }

  if (!m_bTessEmpty) {
    pdl->startTriangleStrip();
    for (int k=0; k<=nsize; k++) {
      pdl->normal(nts[k]);
      pdl->color(pCol);
      pdl->vertex(vts[k]);

      pdl->normal(m_ntess[k]);
      if (bSmoothCol)
        pdl->color(m_pPrevCol);
      pdl->vertex(m_vtess[k]);
    }
    pdl->end();
  }

#ifdef DEBUG_SHOW_NORMAL
  {
    pdl->startLines();
    for (int k=0; k<=nsize; k++) {
      pdl->vertex(vts[k]);
      pdl->vertex(vts[k]+nts[k].scale(0.25));
    }
    pdl->end();
  }
#endif

  for (int k=0; k<=nsize; k++) {
    m_vtess[k] = vts[k];
    m_ntess[k] = nts[k];
  }
  m_pPrevCol = pCol;
  m_bTessEmpty = false;

}

void TubeSection::endTess()
{
}

Vector4D TubeSection::calcDnorm(int index, const Vector4D &escl, const Vector4D &vpt)
{
  const Vector4D &sect = m_pSectTab[index%m_nSectTabSz];

  const double sz = qlib::abs(sect.z());
  const double sw = qlib::abs(sect.w());

  const double ez = escl.z() * getWidth();
  const double ew = escl.w() * getWidth() * getTuber();

  double dn = -(sz*ez + sw*ew);
  double vlen2 = vpt.sqlen();
  /*double dn =
    sect.z()*escl.y() * sect.x()*escl.z() -
      sect.w()*escl.x() * sect.y()*escl.w();
*/
  // dn /= vpt.sqlen();
  
  return vpt.scale(dn/vlen2);
}

