// -*-Mode: C++;-*-
//
//  Povray display context implementation
//
//  $Id: PovDisplayContext.cpp,v 1.17 2011/04/11 11:37:29 rishitani Exp $

#include <common.h>

#include "PovDisplayContext.hpp"
#include "RendIntData.hpp"
#include <qlib/PrintStream.hpp>
#include <qlib/Utils.hpp>
#include <gfx/SolidColor.hpp>
#include <qsys/SceneManager.hpp>
#include <qsys/style/StyleMgr.hpp>

using namespace render;

using qlib::PrintStream;
using qlib::Matrix4D;
using qlib::Matrix3D;
using qsys::StyleMgr;
using qsys::SceneManager;

PovDisplayContext::PovDisplayContext()
     : FileDisplayContext()
{
  // m_fPerspective = true;
  // m_bUnitary = true;
  // m_nDetail = 3;
  // m_dUniTol = 1e-3;
  m_pPovOut = NULL;
  m_pIncOut = NULL;
  m_bPostBlend = false;
}

PovDisplayContext::~PovDisplayContext()
{
}


void PovDisplayContext::startSection(const LString &name)
{
  // start of rendering
  super_t::startSection(name);
  m_pIntData->start(m_pPovOut, m_pIncOut, name);

  if (m_bPostBlend) {
    double defalpha = getAlpha();
    MB_DPRINTLN("PovDC> %s default alpha = %f", name.c_str(), defalpha);
    if (!qlib::Util::isNear4(defalpha, 1.0)) {
      MB_DPRINTLN("PovDC> blend tab %s => %f", name.c_str(), defalpha);
      m_blendTab.insert(std::pair<LString, double>(name, defalpha));
    }
  }
}

bool PovDisplayContext::isPostBlend() const
{
  return m_bPostBlend;
}

LString PovDisplayContext::getPostBlendTableJSON() const
{
  LString rval = "{";
  bool bFirst = true;
  BOOST_FOREACH (const BlendTab::value_type &elem, m_blendTab) {
    if (!bFirst)
      rval += ",\n";
    rval += "\""+elem.first+"\":"+LString::format("%f", elem.second);
    bFirst = false;
  }
  rval += "}";
  return rval;
}

//////////////////////////////

void PovDisplayContext::init(qlib::OutStream *pPovOut, qlib::OutStream *pIncOut)
{
  m_matstack.erase(m_matstack.begin(), m_matstack.end());

  pushMatrix();
  loadIdent();
  m_linew = 1.0;
  m_pColor = gfx::SolidColor::createRGB(1,1,1);
  m_nDrawMode = POV_NONE;
  m_fPrevPosValid = false;
  m_nTriIndex = 0;
  m_dZoom = 100;
  m_dViewDist = 100;
  m_dSlabDepth = 100;

  if (m_pIntData!=NULL)
    delete m_pIntData;
  m_pIntData = NULL;

  m_pPovOut = pPovOut;
  m_pIncOut = pIncOut;
}

void PovDisplayContext::startRender()
{
  writeHeader();
}

void PovDisplayContext::endRender()
{
  if (m_pPovOut!=NULL) {
    writeTailer();
    m_pPovOut->close();
  }
  if (m_pIncOut!=NULL) {
    m_pIncOut->close();
  }
}

void PovDisplayContext::writeHeader()
{
  MB_DPRINTLN("povwh: zoom=%f", m_dZoom);
  MB_DPRINTLN("povwh: dist=%f", m_dViewDist);

  SceneManager *pmod = SceneManager::getInstance();
  LString ver = LString::format("Version %d.%d.%d.%d (build %s)",
                                pmod->getMajorVer(),pmod->getMinorVer(),
                                pmod->getRevision(),pmod->getBuildNo(),
                                pmod->getBuildID().c_str());

  PrintStream ps(*m_pPovOut);
  PrintStream ips(*m_pIncOut);

  Vector4D bgcolor;
  if (!m_bgcolor.isnull()) {
    bgcolor.x() = m_bgcolor->fr();
    bgcolor.y() = m_bgcolor->fg();
    bgcolor.z() = m_bgcolor->fb();
  }
  
  ps.println("/*");
  ps.formatln("  POV-Ray output from CueMol %s", ver.c_str());
  ps.format(" */\n");
  ps.format("\n");

  ps.format("#version 3.5;\n");

  StyleMgr *pSM = StyleMgr::getInstance();
  //LString preamble = pSM->getConfig("preamble", "pov").trim(" \r\t\n");
  LString preamble = pSM->getConfig("pov", "preamble").trim(" \r\t\n");
  if (!preamble.isEmpty())
    ps.println(preamble);

  ps.format("\n");
  ps.format("background {color rgb <%f,%f,%f>}\n", bgcolor.x(), bgcolor.y(), bgcolor.z());
  ps.format("\n");
  ps.format("#declare _distance = %f;\n", m_dViewDist);

  ps.format("\n");
  ps.format("// _stereo ... 0:none, 1:for right eye, -1:for left eye\n");
  ps.format("// _perspective ... 0:orthogonal projection, 1:perspective projection\n");
  ps.format("// _iod ... inter-ocullar distance\n");
  ps.format("\n");
  ps.format("#ifndef (_perspective)\n");
  ps.format("  #declare _perspective = %d;\n", m_fPerspective);
  ps.format("#end\n");
  ps.format("#ifndef (_stereo)\n");
  ps.format("  #declare _stereo = 0;\n");
  ps.format("#end\n");
  ps.format("#ifndef (_iod)\n");
  ps.format("  #declare _iod = 0.03;\n");
  ps.format("#end\n");

  ps.format("#ifndef (_shadow)\n");
  ps.format("  #declare _shadow = 0;\n");
  ps.format("#end\n");

  ps.format("#declare _zoomy = %f;\n", m_dZoom);
  ps.format("#declare _zoomx = _zoomy * image_width/image_height;\n");
  ps.format("#declare _fovx = 2.0*degrees( atan2(_zoomx, 2.0*_distance) );\n");

  ps.format("\n");
  ps.format("camera {\n");
  ps.format(" #if (_perspective)\n");
  ps.format(" perspective\n");
  ps.format(" direction <0,0,-1>\n");
  ps.format(" up <0,1,0> * image_height/image_width\n");
  ps.format(" right <1,0,0>\n");
  ps.format(" angle _fovx\n");
  ps.format(" location <_stereo*_distance*_iod,0,_distance>\n");
  ps.format(" look_at <0,0,0>\n");
  ps.format(" #else\n");
  ps.format(" orthographic\n");
  ps.format(" direction <0,0,-1>\n");
  ps.format(" up <0, _zoomy, 0>\n");
  ps.format(" right <_zoomx, 0, 0>\n");
  ps.format(" location <_stereo*_distance*_iod,0,_distance>\n");
  ps.format(" look_at <0,0,0>\n");
  ps.format(" #end\n");
  ps.format("}\n");
  ps.format("\n");

  //ps.format("light_source {<_stereo*_distance*_iod,0,_distance> color rgb 1}\n");
  //ps.format("light_source {<-1,1,1>*10000 color rgb 0.5 shadowless}\n");

  ps.format("light_source {\n");
  ps.format("   <_stereo*_distance*_iod,0,_distance>\n");
  ps.format("   color rgb 0.8 \n");
  ps.format("#if (!_perspective)\n");
  ps.format("   parallel point_at <0,0,0>\n");
  ps.format("#end\n");
  ps.format("}\n");
  ps.format("light_source {\n");
  ps.format("   <1,1,1>*10000\n");
  ps.format("   color rgb 0.5 parallel point_at <0,0,0> \n");
  ps.format("#if (!_shadow)\n");
  ps.format("   shadowless\n");
  ps.format("#end\n");
  ps.format("}\n");
  
  ps.format("\n");

  ps.format("#ifndef (_no_fog)\n");
  ps.format("fog {\n");
  ps.format("  distance %f/3\n", m_dSlabDepth);
  ps.format("  color rgbf <%f,%f,%f,0>\n", bgcolor.x(), bgcolor.y(), bgcolor.z());
  ps.format("  fog_type 2\n");
  ps.format("  fog_offset 0\n");
  ps.format("  fog_alt 1.0e-10\n");
  ps.format("  up <0,0,1>\n");
  ps.format("}\n");
  ps.format("#end\n");
  ps.format("\n");

  ps.format("\n");
  ps.format("/////////////////////////////////////////////\n");

  ips.format("/*\n");
  ips.format("  POV-Ray output from CueMol (%s)\n", ver.c_str());
  ips.format(" */\n");
  ips.format("\n");
  ips.format("\n");

  ips.format("union {\n");
}

void PovDisplayContext::writeTailer()
{
  PrintStream ps(*m_pPovOut);
  PrintStream ips(*m_pIncOut);

  ps.format("\n");
  ps.format("//////////////////////////////////////////////\n");
  ps.format("\n");
  ps.format("#declare _scene = #include \"%s\"\n", m_incFileName.c_str());
  ps.format("\n");
  ps.format("object{\n");
  ps.format("  _scene\n");
  /*
  // clipping plane (slow!!)
  ps.format("#ifdef (_clipping_plane)\n");
  ps.format("bounded_by {\n");
  ps.format("  plane {z, %f}\n", m_dSlabDepth/2.0);
  ps.format("}\n");
  ps.format("clipped_by { bounded_by }\n");
  ps.format("#end\n");
   */
  ps.format("}\n");
  ps.format("\n");

  ips.println("} // union");
  ips.println("");
}

