// -*-Mode: C++;-*-
//
// Scene: root object of the scene document
//
// $Id: Scene.cpp,v 1.101 2011/04/16 17:30:56 rishitani Exp $
//

#include <common.h>

#include "Scene.hpp"
#include "SceneEventCaster.hpp"
#include "ObjLoadEditInfo.hpp"
#include "ScrEventManager.hpp"
#include "PropEditInfo.hpp"
#include "SysConfig.hpp"
#include "StreamManager.hpp"
#include "style/AutoStyleCtxt.hpp"
#include "CameraEditInfo.hpp"

#include <qlib/FileStream.hpp>
#include <qlib/StringStream.hpp>
#include <qlib/LVarArgs.hpp>

// for serialization
#include <qlib/LDOM2Stream.hpp>
#include <qlib/FileStream.hpp>
#include <qlib/LByteArray.hpp>

#include <gfx/DisplayContext.hpp>
#include "style/StyleMgr.hpp"
#include "style/StyleFile.hpp"
#include "style/StyleSet.hpp"

#include "anim/AnimMgr.hpp"

#define NO_SCRIPT 1
#ifndef NO_SCRIPT
// for JS interpreter
#include <jsbr/jsbr.hpp>
#include <jsbr/Interp.hpp>
#endif

#include <boost/filesystem/operations.hpp>
#include <boost/filesystem/path.hpp>

using namespace qsys;
namespace fs = boost::filesystem;

namespace {
class ScrScEvtLsnr : public SceneEventListener
{
private:
  //qlib::LScrCallBack *m_pCb;
  qlib::LSCBPtr m_pCb;
  
public:
  ScrScEvtLsnr(qlib::LSCBPtr pcb) : m_pCb(pcb) {}
  
  virtual ~ScrScEvtLsnr()
  {
    MB_DPRINTLN("ScrScEvtLsnr destr %p", this);
    m_pCb = qlib::LSCBPtr();
  }
  
  void set(qlib::LSCBPtr pcb) { m_pCb = pcb; }

  virtual void sceneChanged(SceneEvent &ev)
  {
    // avoid frequent event from script calling (TO DO: configurable)
    int ntype = ev.getType();
    //if (ntype==SceneEvent::SCE_VIEW_PROPCHG_DRG) return;
    //if (ntype==SceneEvent::SCE_VIEW_SIZECHG) return;
    //if (ntype==SceneEvent::SCE_VIEW_ACTIVATED) return;

    qlib::uid_t scid = ev.getSource();
    qlib::uid_t objid = ev.getTarget();
    qlib::LVarArgs args(5);
    
    // Method name
    args.at(0).setStringValue("sceneChanged");
    // type ID of the event
    args.at(1).setIntValue(ntype);
    // target scene ID
    args.at(2).setIntValue(scid);
    // target object/renderer/view ID
    args.at(3).setIntValue(objid);
    // target message/property name
    args.at(4).setStringValue(ev.getDescr());
    // // event object (in JSON format)
    // args.at(5).setStringValue(ev.getJSON());


    m_pCb->invoke(args);
  }
      
};
}

Scene::Scene()
{
  m_pQscOpts = NULL;

  m_nUID = qlib::ObjectManager::sRegObj(this);
  m_pEvtCaster = MB_NEW SceneEventCaster;
  m_pStyleMgr = StyleMgr::getInstance();
  m_pInterp = NULL;
  // m_nModified = 0;
  addPropListener(this);

  m_pAnimMgr = qlib::LScrSp<AnimMgr>( MB_NEW AnimMgr() );
  m_pAnimMgr->setTgtSceneID(m_nUID);

  m_pEvtCaster->add(m_pAnimMgr.get());

  m_bLoading = false;

  m_nActiveViewID = qlib::invalid_uid;

  MB_DPRINTLN("Scene (%d) created.", m_nUID);
}

void Scene::init()
{
  resetAllProps();

#ifndef NO_SCRIPT
  if (m_pInterp!=NULL) delete m_pInterp;
  ScenePtr rThis(this);
  m_pInterp = jsbr::createInterp(rThis.copy());
  if (m_pInterp==NULL)
    return;

  // setup system default script path
  SysConfig *pconf = SysConfig::getInstance();
  LString scrdir = pconf->get("script_dir");
  if (!scrdir.isEmpty())
    m_pInterp->setScriptPath("system", pconf->convPathName(scrdir));
#endif

  if (m_pQscOpts!=NULL)
    delete m_pQscOpts;
  m_pQscOpts = NULL;
}

Scene::~Scene()
{
   removePropListener(this);
  clearAll();

  m_pAnimMgr = qlib::LScrSp<AnimMgr>();

 delete m_pEvtCaster;
  qlib::ObjectManager::sUnregObj(m_nUID);

  MB_DPRINTLN("Scene (%d/%p) destructed", m_nUID, this);
}

void Scene::dump() const
{
  MB_DPRINTLN("SCENE DUMP:");
  MB_DPRINTLN("Scene : {");
  MB_DPRINTLN("  name = <%s>", m_name.c_str());
  MB_DPRINTLN("  uid = <%d>", m_nUID);

  data_t::const_iterator iter = m_data.begin();
  for (; iter!=m_data.end(); ++iter) {
    MB_DPRINT("%p/%d (nref=%d): ",
	      iter->second.get(),
	      iter->second->getUID(),
	      iter->second.use_count());
    iter->second->dump();
  }

  MB_DPRINTLN("}");
  MB_DPRINTLN("SCENE DUMP END.");

}

void Scene::clearAll()
{
  clearAllData();
  m_viewtab.clear();
}

void Scene::clearAllData()
{
  m_undomgr.clearAllInfo();
  m_rendtab.clear();
  m_data.clear();
  m_camtab.clear();
  m_pStyleMgr->destroyContext(m_nUID);

  if (m_pQscOpts!=NULL)
    delete m_pQscOpts;
  m_pQscOpts = NULL;

  m_pAnimMgr->clear();
}

void Scene::clearAllDataScr()
{
  clearAllData();

  {
    SceneEvent ev;
    ev.setTarget(getUID());
    ev.setType(SceneEvent::SCE_SCENE_CLEARALL);
    fireSceneEvent(ev);
  }
}

void Scene::unloading()
{
  {
    SceneEvent ev;
    ev.setTarget(getUID());
    ev.setType(SceneEvent::SCE_SCENE_REMOVING);
    fireSceneEvent(ev);
  }

  MB_DPRINTLN("Scene.unloading> *** Broadcast unloading ***");
  rendtab_t::const_iterator riter = m_rendtab.begin();
  for (; riter!=m_rendtab.end(); ++riter) {
    RendererPtr rrend = riter->second;
    MB_DPRINTLN("Scene.unloading()> rrend %d unloading().", riter->first);
    rrend->unloading();
  }
  
  data_t::const_iterator oiter = m_data.begin();
  for (; oiter!=m_data.end(); ++oiter) {
    ObjectPtr obj = oiter->second;
    MB_DPRINTLN("Scene.unloading()> robj %d unloading().", oiter->first);
    obj->unloading();
  }

  viewtab_t::const_iterator viter = m_viewtab.begin();
  for (; viter!=m_viewtab.end(); ++viter) {
    ViewPtr rvi = viter->second;
    MB_DPRINTLN("Scene.unloading()> rview %d unloading().", viter->first);
    rvi->unloading();
  }

#ifndef NO_SCRIPT
  delete m_pInterp;
  m_pInterp = NULL;
#endif
}

/// set source path of this scene
void Scene::setSource(const LString &name)
{
  m_source = name;

#ifndef NO_SCRIPT
  if (m_pInterp!=NULL)
    m_pInterp->setScriptPath("scene_base_path", getBasePath());
#endif
}

LString Scene::getBasePath() const
{
  fs::path srcpath(m_source);
  return srcpath.parent_path().directory_string();
}

LString Scene::resolveBasePath(const LString &aLocalFile) const
{
/*
  fs::path inpath(aLocalFile);
  if (inpath.is_complete())
    return inpath.file_string();

  fs::path base_path(getBasePath());

  // convert to the absolute representation
  inpath = fs::complete(inpath, base_path);
  return inpath.file_string();
*/

  return qlib::makeAbsolutePath(aLocalFile, getBasePath());
}

//////////////////////////////

/*ObjectPtr Scene::createObject(const LString &type_name)
{
  ObjectPtr robj(MB_NEW Object());

  if (!registerObjectImpl(robj)) {
    MB_THROW(qlib::RuntimeException, "Cannot create new object");
    return ObjectPtr();
  }

  return robj;
}*/

bool Scene::addObject(ObjectPtr pobj)
{
  qlib::uid_t uid = pobj->getUID();
  data_t::const_iterator i = m_data.find(uid);
  if (i!=m_data.end()) {
    MB_DPRINTLN("Scene::addObject> object %d is already registered!!", int(uid));
    return false;
  }

  if (!registerObjectImpl(pobj)) {
    MB_THROW(qlib::RuntimeException, "Cannot add new object");
    return false;
  }

  return true;
}

bool Scene::destroyObject(qlib::uid_t uid)
{
  data_t::iterator i = m_data.find(uid);
  if (i==m_data.end()) {
    return false;
  }
  
  ObjectPtr robj = i->second;
  qlib::ensureNotNull(robj);

  // Detach this scene from the prop event source
  bool res = robj->removeListener(this);
  MB_ASSERT(res);

  //
  // purge the rendering cache of this scene
  //
  while (robj->getRendCount()>0) {
    Object::RendIter ri = robj->beginRend();
    robj->destroyRenderer(ri->first);
    //RendererPtr rrend = ri->second;
    //robj->destroyRenderer(rrend->getUID());
    //removeRendCache(rrend);
  }

  //
  // Fire the SCE_OBJ_REMOVING Event, before removing
  //
  {
    MB_DPRINTLN("Scene> Firing SCE_OBJ_REMOVING event...");
    SceneEvent ev;
    ev.setType(SceneEvent::SCE_OBJ_REMOVING);
    ev.setSource(getUID());
    ev.setTarget(robj->getUID());
    fireSceneEvent(ev);
  }

  // Here, actually remove the item.
  m_data.erase(i);

  {
    // Record undo/redo info
    UndoManager *pUM = getUndoMgr();
    if (pUM->isOK()) {
      ObjLoadEditInfo *pPEI = MB_NEW ObjLoadEditInfo;
      pPEI->setupObjDestroy(getUID(), robj);
      pUM->addEditInfo(pPEI);
    }
  }

  return true;
}

void Scene::destroyAllObjects()
{
  while (m_data.size()>0) {
    data_t::iterator oiter = m_data.begin();
    qlib::uid_t uid = oiter->first;
    destroyObject(uid);
  }
}


ObjectPtr Scene::getObject(qlib::uid_t uid) const
{
  data_t::const_iterator i = m_data.find(uid);
  if (i==m_data.end()) {
    // LString msg = LString::format("Object %d not found", uid);
    // MB_THROW(qlib::RuntimeException, msg);
    return ObjectPtr();
  }

  return i->second;
}

ObjectPtr Scene::getObjectByName(const LString &name) const
{
  BOOST_FOREACH(const data_t::value_type &i, m_data) {
    // if (pObj.isnull) continue;
    if (i.second->getName().equals(name))
      return i.second;
  }
  return ObjectPtr();
}

namespace {
  bool objsort_less(const ObjectPtr &pObj1, const ObjectPtr &pObj2)
  {
    return (pObj1->getUIOrder()) < (pObj2->getUIOrder());
  }
}

int Scene::getAllObjectUIDs(std::list<qlib::uid_t> &uids) const
{
  if (m_data.empty())
    return 0;

  // sort by ui_order
  std::vector<ObjectPtr> tmpvec;
  {
    BOOST_FOREACH (const data_t::value_type &i, m_data) {
      tmpvec.push_back(i.second);
    }
    std::sort(tmpvec.begin(), tmpvec.end(), objsort_less);
  }

  int nret = 0;
  BOOST_FOREACH (const ObjectPtr &i, tmpvec) {
    uids.push_back(i->getUID());
    ++nret;
  }

  return nret;
}

LString Scene::getObjUIDList() const
{
  LString rval;
  if (m_data.empty())
    return rval;
  
  qlib::UIDList uids;
  getAllObjectUIDs(uids);

  bool bFirst = true;
  BOOST_FOREACH (qlib::uid_t i, uids) {
    if (!bFirst)
      rval += ",";
    rval += LString::format("%d", i);
    bFirst = false;
  }

  return rval;
}

// private

/// Common implementation for the object registration
bool Scene::registerObjectImpl(ObjectPtr robj)
{
  robj->setSceneID(m_nUID);
  bool res = m_data.insert(data_t::value_type(robj->getUID(), robj)).second;
  if (!res)
    return false;

  // observe object events from the new object
  robj->addListener(this);

  //////////////////////////////////////////////
  // Re-setup renderers already attached to robj
  {
    Object::RendIter ri = robj->beginRend();
    for (; ri!=robj->endRend(); ++ri) {
      RendererPtr pRend = ri->second;
      addRendCache(pRend);
      // Update scene ID of the renderer
      pRend->setSceneID(m_nUID);
      // Scene observes events from the renderers
      pRend->addListener(this);
      // Reapply styles
      pRend->reapplyStyle();
    }
  }

  ///////////////////////////////
  // Fire the SCE_OBJ_ADDED Event
  if (!m_bLoading) {
    MB_DPRINTLN("Scene> Firing SCE_OBJ_ADDED event...");
    SceneEvent ev;
    ev.setType(SceneEvent::SCE_OBJ_ADDED);
    ev.setSource(getUID());
    ev.setTarget(robj->getUID());
    fireSceneEvent(ev);
  }


  {
    // Record undo/redo info
    UndoManager *pUM = getUndoMgr();
    if (!pUM->isOK())
      return true; // undo manager is disabled now.
    
    ObjLoadEditInfo *pPEI = MB_NEW ObjLoadEditInfo;
    pPEI->setupObjCreate(getUID(), robj);
    pUM->addEditInfo(pPEI);
  }

  return true;
}

///////////////////////////////////////
// renderer cachelist management

bool Scene::addRendCache(RendererPtr rrend)
{
  bool res = m_rendtab.insert(rendtab_t::value_type(rrend->getUID(), rrend)).second;
  if (res) {
    // scene redrawing is required
    setUpdateFlag();
  }
  return res;
}

bool Scene::removeRendCache(RendererPtr rrend)
{
  qlib::uid_t uid = rrend->getUID();
  rendtab_t::iterator i = m_rendtab.find(uid);
  if (i==m_rendtab.end())
    return false;

  m_rendtab.erase(i);

  // scene redrawing is required
  setUpdateFlag();

  return true;
}

RendererPtr Scene::getRendByName(const LString &nm) const
{
  rendtab_t::const_iterator riter = m_rendtab.begin();
  rendtab_t::const_iterator reiter = m_rendtab.end();
  for (; riter!=reiter; ++riter) {
    RendererPtr rrend = riter->second;
    if (nm.equals(rrend->getName()))
      return rrend;
  }

  return RendererPtr();
}

///////////////////////////////////////
// Rendering of the scene

static
LString makeSectionName(ObjectPtr robj, RendererPtr rrend)
{
/*
  LString name = LString::format("_%s_%d_%s_%d",
                                 robj->getName().c_str(), robj->getUID(),
                                 rrend->getName().c_str(), rrend->getUID());
  name.replace('.', '_');
  name.replace('-', '_');
  name.replace('(', '_');
  name.replace(')', '_');
*/

  LString name = LString::format("_%d_%d", robj->getUID(), rrend->getUID());

  return name;
}

///
/// Display renderers in the scene to the frame buffer
///
void Scene::display(DisplayContext *pdc)
{
  // StyleMgr is held in the member variable (at the ctor)
  m_pStyleMgr->pushContextID(getUID());
  pdc->startRender();

  Matrix4D xform;
  bool bmat = false;
  std::vector<RendererPtr> transps;
  {
    rendtab_t::const_iterator i = m_rendtab.begin();
    rendtab_t::const_iterator ie = m_rendtab.end();
    for (; i!=ie; ++i) {
      RendererPtr rrend = i->second;
      ObjectPtr robj = rrend->getClientObj();
      if (!robj.isnull() && robj->isVisible()) {
        if (rrend->isVisible()) {
          if (rrend->isTransp()) {
            transps.push_back(rrend);
          }
          else {
            pdc->setMaterial(rrend->getDefaultMaterial());
            pdc->setAlpha(rrend->getDefaultAlpha());
            pdc->setStyleNames(rrend->getStyleNames());

            bmat = false;
            xform = rrend->getXformMatrix();
            if (!xform.isIdent()) {
              pdc->pushMatrix();
              pdc->multMatrix(xform);
              bmat = true;
            }

            // alpha should be set before startSection,
            // since startSection() refers to the alpha value (in Pov rendering)
            pdc->startSection(makeSectionName(robj, rrend));

            rrend->display(pdc);
            pdc->endSection();

            if (bmat)
              pdc->popMatrix();

          }
        }
      }
    }
  }

  // Render transparent objects
  if (transps.size()>0) {
    // pdc->enableDepthTest(false);

    std::vector<RendererPtr>::const_iterator i = transps.begin();
    std::vector<RendererPtr>::const_iterator ie = transps.end();
    for (; i!=ie; ++i) {
      RendererPtr rrend = *i;
      ObjectPtr robj = rrend->getClientObj();

      // alpha should be set before startSection,
      // since startSection() refers to the alpha value (in Pov rendering)
      pdc->setMaterial(rrend->getDefaultMaterial());
      pdc->setAlpha(rrend->getDefaultAlpha());
      pdc->setStyleNames(rrend->getStyleNames());

      bmat = false;
      xform = rrend->getXformMatrix();
      if (!xform.isIdent()) {
        pdc->pushMatrix();
        pdc->multMatrix(xform);
        bmat = true;
      }

      pdc->startSection(makeSectionName(robj, rrend));

      rrend->display(pdc);
      pdc->endSection();

      if (bmat)
        pdc->popMatrix();
    }
    // pdc->enableDepthTest(true);
  }

  // Display 2D labels (UI elements /w depth)
  {
    rendtab_t::const_iterator i = m_rendtab.begin();
    rendtab_t::const_iterator ie = m_rendtab.end();
    for (; i!=ie; ++i) {
      RendererPtr rrend = i->second;
      ObjectPtr robj = rrend->getClientObj();
      if (!robj.isnull() &&
          robj->isVisible() &&
          rrend->isVisible()) {
        pdc->setAlpha(rrend->getDefaultAlpha());
        
        bmat = false;
        xform = rrend->getXformMatrix();
        if (!xform.isIdent()) {
          pdc->pushMatrix();
          pdc->multMatrix(xform);
            bmat = true;
        }
        
        rrend->displayLabels(pdc);
        if (bmat)
          pdc->popMatrix();
        
      }
    }
  }

  pdc->endRender();
  m_pStyleMgr->popContextID();
}

void Scene::processHit(DisplayContext *pdc)
{
  m_pStyleMgr->pushContextID(getUID());
  pdc->startRender();

  rendtab_t::const_iterator i = m_rendtab.begin();
  for (; i!=m_rendtab.end(); ++i) {
    RendererPtr rrend = i->second;
    ObjectPtr robj = rrend->getClientObj();
    if (robj.isnull() || (robj->isVisible() && !robj->isUILocked())) {
      if (rrend->isVisible() && !rrend->isUILocked()) {
        rrend->processHit(pdc);
      }
    }
  }

  pdc->endRender();
  m_pStyleMgr->popContextID();
}

//////////////////////////////

ViewPtr Scene::createView()
{
  ViewPtr robj(View::createView());
  if (robj.isnull()) {
    LOG_DPRINTLN("Fatal error: View::createView() returned NULL!!");
    MB_THROW(qlib::NullPointerException, "");
    return ViewPtr();
  }

  // reset to default values
  robj->resetAllProps();

  robj->setSceneID(m_nUID);
  bool res = m_viewtab.insert(viewtab_t::value_type(robj->getUID(), robj)).second;
  if (!res)
    return ViewPtr();

  {
    SceneEvent ev;
    ev.setType(SceneEvent::SCE_VIEW_ADDED);
    ev.setSource(getUID());
    ev.setTarget(robj->getUID());
    fireSceneEvent(ev);
  }

  return robj;
}

ViewPtr Scene::getView(qlib::uid_t uid) const
{
  viewtab_t::const_iterator i = m_viewtab.find(uid);
  if (i==m_viewtab.end())
    return ViewPtr();

  return i->second;
}

ViewPtr Scene::getViewByName(const LString &name) const
{
  BOOST_FOREACH(const viewtab_t::value_type &i, m_viewtab) {
    if (i.second->getName().equals(name))
      return i.second;
  }

  return ViewPtr();
}

bool Scene::destroyView(qlib::uid_t uid)
{
  viewtab_t::iterator i = m_viewtab.find(uid);
  if (i==m_viewtab.end())
    return false;

  ViewPtr pView = i->second;
  qlib::ensureNotNull(pView);
  
  {
    MB_DPRINTLN("Scenek> Firing SCE_VIEW_REMOVING event...");
    SceneEvent ev;
    ev.setType(SceneEvent::SCE_VIEW_REMOVING);
    ev.setSource(getUID());
    ev.setTarget(pView->getUID());
    fireSceneEvent(ev);
  }

  if (m_viewtab.size()==1) {
    // the last view is destructing...
    //  --> all renderers should detach from view-related resources.
    rendtab_t::const_iterator riter = m_rendtab.begin();
    for (; riter!=m_rendtab.end(); ++riter) {
      RendererPtr pRend = riter->second;
      if (!pRend.isnull())
        pRend->unloading();
    }

    data_t::const_iterator oiter = m_data.begin();
    for (; oiter!=m_data.end(); ++oiter) {
      ObjectPtr pObj = oiter->second;
      if (!pObj.isnull())
        pObj->unloading();
    }
  }

  pView->unloading();

  m_viewtab.erase(i);

  return true;
}

void Scene::setActiveViewID(qlib::uid_t uid)
{
  ViewPtr pView = getView(uid);
  if (pView.isnull()) {
    MB_THROW(qlib::IllegalArgumentException, "Unknown view ID");
    return;
  }
    
  m_nActiveViewID = uid;
}

ViewPtr Scene::getActiveView() const
{
  return getView(m_nActiveViewID);
}

void Scene::checkAndUpdate()
{
  // MB_DPRINTLN("scene %d update %d", m_nUID, m_bUpdateRequired);
  if (m_bUpdateRequired) {
    // Force to redraw all views
    viewtab_t::const_iterator viter = m_viewtab.begin();
    for (; viter!=m_viewtab.end(); ++viter) {
      View *pV = viter->second.get();
      if (pV->isActive())
        pV->forceRedraw();
    }
    clearUpdateFlag();
  }
  else {
    // Redraw views if required
    viewtab_t::const_iterator viter = m_viewtab.begin();
    for (; viter!=m_viewtab.end(); ++viter) {
      View *pV = viter->second.get();
      if (pV->getUpdateFlag() && pV->isActive()) {
        //MB_DPRINTLN("Scene::checkAndUpdate() drawScene");
        pV->drawScene();
        pV->clearUpdateFlag();
      }
    }
  }
}

LString Scene::getViewUIDList() const
{
  LString rval;
  if (m_viewtab.empty())
    return rval;
  
  viewtab_t::const_iterator i = m_viewtab.begin();
  viewtab_t::const_iterator end = m_viewtab.end();
  for (int j=0; i!=end; ++i, ++j) {
    if (j>0)
      rval += ",";
    rval += LString::format("%d", i->first);
  }

  return rval;
}

////////////////////////////////////////////////////////////

qlib::uid_t Scene::getRootUID() const
{
  return getUID();
}

bool Scene::isModified() const
{
  StyleMgr *pSM = StyleMgr::getInstance();
  
  if (m_undomgr.getUndoSize()==0 &&
      !pSM->isModified(getUID()))
    return false;
  else
    return true;
}

bool Scene::isJustCreated() const
{
  if (isModified())
    return false;

  // scene is not modified
  
  if (getObjectCount()==0 &&
      getCameraCount()==0) {
      //getCameraCount()==1 &&
      //!getCamera("__current").isnull()) {
    // not modified but just created
    return true;
  }

  return false;
}

void Scene::setName(const LString &name)
{
  // special case; scene's name is readonly property
  // but changing by setName fires propChanged event.
  // (this is required to change UI labels of scene name...)

  // LString oldname = m_name;
  m_name = name;
  
  {
    qlib::LPropEvent ev("name");
    propChanged(ev);
  }
}

int Scene::addListener(SceneEventListener *pL)
{
  return m_pEvtCaster->add(pL);
}

int Scene::addListener(qlib::LSCBPtr scb)
{
  ScrScEvtLsnr *pLn = MB_NEW ScrScEvtLsnr(scb);
  // pLn->m_pCb = scb; // .get();
  return m_pEvtCaster->add(pLn);
}

bool Scene::removeListener(SceneEventListener *pL)
{
  return m_pEvtCaster->remove(pL);
}

bool Scene::removeListener(int nid)
{
  ScrScEvtLsnr *pSLn = dynamic_cast<ScrScEvtLsnr *>(m_pEvtCaster->remove(nid));
  MB_DPRINTLN("Scene::removeListener(%d) remove %p", nid, pSLn);
  if (pSLn==NULL)
    return false;
  delete pSLn;
  return true;
}

void Scene::fireSceneEvent(SceneEvent &ev)
{
  m_pEvtCaster->replicaFire(ev);

  ScrEventManager *pSEM = ScrEventManager::getInstance();
  ev.setSource(m_nUID);
  pSEM->fireEvent(ev);

#ifndef NO_SCRIPT
  // call the script event handlers
  if (ev.getType()==SceneEvent::SCE_SCENE_ONLOADED) {
    if (m_pInterp!=NULL && !m_scrOnLoadEvent.isEmpty())
      m_pInterp->eval(m_scrOnLoadEvent);
  }
#endif
}

void Scene::propChanged(qlib::LPropEvent &ev)
{
  MB_DPRINTLN("!!! Scene::propChanged");

  // record undo/redo info
  if (!ev.isIntrDataChanged()) {
    UndoManager *pUM = getUndoMgr();
    if (pUM->isOK()) {
      // record property changed undo/redo info
      PropEditInfo *pPEI = MB_NEW PropEditInfo;
      pPEI->setup(getUID(), ev);
      pUM->addEditInfo(pPEI);
    }
  }

  // propagate prop event to scene event
  if (!m_bLoading) {
    SceneEvent sev;
    sev.setTarget(getUID());
    sev.setSource(getUID());
    sev.setType(SceneEvent::SCE_SCENE_PROPCHG);
    sev.setDescr(ev.getName());
    sev.setPropEvent(&ev);
    fireSceneEvent(sev);
  }
}

void Scene::objectChanged(ObjectEvent &ev)
{
  if (ev.getType()==ObjectEvent::OBE_CHANGED) {
    // objchg may require redrawing
    setUpdateFlag();
  }
  else if (ev.getType()==ObjectEvent::OBE_PROPCHG) {
    // propchg may require redrawing
    setUpdateFlag();
  }
}

void Scene::rendererChanged(RendererEvent &ev)
{
  if (ev.getType()==RendererEvent::RNE_CHANGED) {
    // objchg may require redrawing
    setUpdateFlag();
  }
  else if (ev.getType()==RendererEvent::RNE_PROPCHG) {
    // propchg may require redrawing
    setUpdateFlag();
  }
}

/////////////////////////////////////////////////

LString Scene::getObjectTreeJSON() const
{
  LString rval = "[\n";

  // this scene
  rval += "{";
  rval += "\"name\":\""+m_name+"\", ";
  rval += "\"parent\": -1, ";
  rval += "\"open\": \"true\", ";
  rval += "\"type\":\"\", ";
  rval += "\"rends\": [],\n";
  rval += LString::format("\"ID\": %d", getUID());
  rval += "}";

  std::list<qlib::uid_t> obj_uids;
  getAllObjectUIDs(obj_uids);
  //data_t::const_iterator iter = m_data.begin();
  // int nrow = 1, nparent = -1;
  // for (; iter!=m_data.end(); ++iter) {

  BOOST_FOREACH (qlib::uid_t objid, obj_uids) {
    // if (iter!=m_data.begin())
    rval += ",\n";
    rval += "{";
    // ObjectPtr obj = iter->second;
    // qlib::uid_t objid = iter->first;
    ObjectPtr obj = getObject(objid);
    
    rval += "\"name\":\""+(obj->getName())+"\", ";
    rval += "\"parent\": -1, ";
    //rval += "\"ui_collapsed\": true, ";
    rval += LString("\"ui_collapsed\": ") + LString::fromBool(obj->isUICollapsed()) + ", ";
    rval += LString::format("\"ui_order\": %d, ", obj->getUIOrder());
    rval += LString("\"visible\": ") + LString::fromBool(obj->isVisible()) + ", ";
    rval += LString("\"locked\": ") + LString::fromBool(obj->isUILocked()) + ", ";
    rval += "\"type\":\""+LString(obj->getClassName())+"\", ";
    rval += LString::format("\"ID\": %d, ", objid);

    // nparent = nrow;
    // nrow ++;

    rval += "\"rends\": [\n";

    qlib::UIDList rend_uids;
    obj->getRendUIDs(rend_uids);
    bool bfirst=true;
    BOOST_FOREACH (qlib::uid_t rendid, rend_uids) {

      //rendtab_t::const_iterator riter = m_rendtab.begin();
      //for (bool bfirst=true; riter!=m_rendtab.end(); ++riter) {
      //RendererPtr rrend = riter->second;
      //if (rrend->getClientObjID()!=objid)
      //continue;
      
      RendererPtr pRend = obj->getRenderer(rendid);

      if (!bfirst)
        rval += ",\n";

      rval += "{";
      rval += "\"name\":\""+(pRend->getName())+"\", ";
      rval += "\"type\":\""+LString(pRend->getTypeName())+"\", ";
      rval += LString::format("\"ui_order\": %d, ", pRend->getUIOrder());
      rval += LString("\"visible\": ") + LString::fromBool(pRend->isVisible()) + ", ";
      rval += LString("\"locked\": ") + LString::fromBool(pRend->isUILocked()) + ", ";
      rval += LString::format("\"ID\": %d", rendid);
      rval += "}";
      // nrow ++;
      bfirst = false;
    }

    rval += "]\n";
    rval += "}";
  }

  rval += "\n]";

  // MB_DPRINTLN("Scene> built JSON=%s", rval.c_str());
  return rval;
}

////////////////////////////////////////////////////////////
// Camera manager

// private
/// set/overwrite the camera
void Scene::setCameraImpl(const LString &name, CameraPtr r)
{
  r->m_name = name;

  camtab_t::iterator i = m_camtab.find(name);
  if (i!=m_camtab.end()) {
    // remove existing camera with the same name
    m_camtab.erase(i);
  }

  bool res = m_camtab.insert(camtab_t::value_type(name, r)).second;
  MB_ASSERT(res);
}

// private
CameraPtr Scene::loadCameraImpl(const LString &aLocalFile) const
{
  LString path = resolveBasePath(aLocalFile);

  qlib::FileInStream fis;
  fis.open(path);
  qlib::LDom2InStream ois(fis);
  
  qlib::LDom2Tree tree;
  ois.read(tree);
  //tree.top()->dump();

  CameraPtr rcam(MB_NEW Camera);
  tree.deserialize(&rcam);

  // set the source property
  rcam->setSource(aLocalFile);

  return rcam;
}

//////////
// Interfaces

/// Set camera with name (overwrite if the camera with same name exists)
void Scene::setCamera(const LString &name, CameraPtr pCam)
{
  bool bCreate = !hasCamera(name);

  // setup undo txn
  UndoUtil uu(this);
  if (uu.isOK()) {
    if (bCreate) {
      CameraCreateEditInfo *pPEI = MB_NEW CameraCreateEditInfo;
      pPEI->setupCreate(getUID(), pCam);
      uu.add(pPEI);
    }
    else {
      CameraPropEditInfo *pPEI = MB_NEW CameraPropEditInfo;
      CameraPtr pBef = getCameraRef(name);
      pPEI->setup(getUID(), name, pBef, pCam );
      uu.add(pPEI);
    }
  }

  setCameraImpl(name, pCam);

  if (!m_bLoading) {
    CameraEvent ev;
    ev.setSource(m_nUID);
    ev.m_name = name;
    if (bCreate) {
      // fire camera-added event
      ev.setDescr("cameraAdded");
      ev.setType(ScrEventManager::SEM_ADDED);
    }
    else {
      // fire camera-changed event
      ev.setDescr("cameraChanged");
      ev.setType(ScrEventManager::SEM_CHANGED);
    }
    ScrEventManager *pSEM = ScrEventManager::getInstance();
    pSEM->fireEvent(ev);
  }
}

/// get the reference of the camera by name
CameraPtr Scene::getCameraRef(const LString &name) const
{
  camtab_t::const_iterator i = m_camtab.find(name);
  if (i==m_camtab.end())
    return CameraPtr();
  return i->second;
}

/// get the copy of the camera by name
CameraPtr Scene::getCamera(const LString &name) const
{
  CameraPtr pc = getCameraRef(name);
  if (pc.isnull())
    return pc;

  // return a copy
  return CameraPtr(MB_NEW Camera( *pc.get() ));
}

bool Scene::hasCamera(const LString &name) const
{
  camtab_t::const_iterator i = m_camtab.find(name);
  if (i==m_camtab.end())
    return false;
  return true;
}

bool Scene::destroyCamera(const LString &name)
{
  {
    // fire camera-removing event
    CameraEvent ev;
    ev.setDescr("cameraRemoving");
    ev.setSource(m_nUID);
    //ev.m_nEvtType = ScrEventManager::SEM_REMOVING;
    ev.setType(ScrEventManager::SEM_REMOVING);
    ev.m_name = name;
    ScrEventManager *pSEM = ScrEventManager::getInstance();
    pSEM->fireEvent(ev);
  }

  camtab_t::iterator i = m_camtab.find(name);
  if (i==m_camtab.end())
    return false;

  UndoUtil uu(this);
  if (uu.isOK()) {
    CameraCreateEditInfo *pPEI = MB_NEW CameraCreateEditInfo;
    pPEI->setupDestroy(getUID(), i->second);
    uu.add(pPEI);
  }

  m_camtab.erase(i);
  return true;
}


/// Save camera to the local file
bool Scene::saveCameraTo(const LString &aName, const LString &aLocalFile) const
{
  CameraPtr rcam = getCameraRef(aName);
  if (rcam.isnull())
    return false;

  CameraPtr r;
  try {
    LString path = resolveBasePath(aLocalFile);

    qlib::FileOutStream fis;
    fis.open(path);
    qlib::LDom2OutStream ois(fis);
  
    qlib::LDom2Tree tree("camera");
    
    // Camera obj --> LDOM2 tree
    // top node doesn't use type attribute as object's typename
    tree.serialize(rcam.get(), false);

    // LDOM2 tree --> Stream (local file)
    ois.write(&tree);
    // tree.top()->dump();
  }
  catch (...) {
    LOG_DPRINTLN("Fatal Error: Load camera %s is failed", aLocalFile.c_str());
    return false;
  }

  if (aName.equals("__current"))
    return true;

  // Change file linkage
  // get before copy
  CameraPtr pBef = getCamera(aName);

  // convert to file-linked camera
  rcam->setSource(aLocalFile);

  // Setup undo txn
  UndoUtil uu(this);
  if (uu.isOK()) {
    CameraPropEditInfo *pPEI = MB_NEW CameraPropEditInfo;
    CameraPtr pAft = getCameraRef(aName);
    pPEI->setup(getUID(), aName, pBef, pAft );
    uu.add(pPEI);
  }

  return true;
}

/// save from View --> Camera (by name)
bool Scene::saveViewToCam(qlib::uid_t viewid, const LString &name)
{
  CameraPtr pOldCam = getCameraRef(name);
  LString srcpath;

  if (!pOldCam.isnull()) {
    // overwrite existing camera (preserve the srcpath of old one)
    srcpath = pOldCam->getSource();
  }
  
  // Get Copy of view's camera
  ViewPtr pView = getView(viewid);
  if (pView.isnull())
    return false;
  CameraPtr pCam = pView->getCamera();

  // source info shouldn't be overwritten by new value (i.e. empty string)
  pCam->setSource(srcpath);

  setCamera(name, pCam);

  return true;
}

/// Camera --> View restore
void Scene::setCamToViewAnim(qlib::uid_t viewid, const LString &name, bool bAnim)
{
  if (viewid==qlib::invalid_uid) {
    BOOST_FOREACH(const viewtab_t::value_type &i, m_viewtab) {
      qlib::uid_t id = i.first;
      if (id!=qlib::invalid_uid)
	Scene::setCamToViewAnim(id, name, bAnim);
    }
    return;
  }

  ViewPtr rv = getView(viewid);
  ensureNotNull(rv);

  CameraPtr rc = getCamera(name);
  ensureNotNull(rc);

  rv->setCameraAnim(rc, bAnim);

  // Redraw of view (rv) is required
  rv->setUpdateFlag();
}

LString Scene::getCameraInfoJSON() const
{
  LString rval = "[";
  
  camtab_t::const_iterator viter = m_camtab.begin();
  camtab_t::const_iterator eiter = m_camtab.end();
  for (; viter!=eiter; ++viter) {
    if (viter!=m_camtab.begin())
      rval += ",";
    CameraPtr obj = viter->second;
    rval += "{\"name\":\""+ viter->first.escapeQuots() +"\",";
    LString src = obj->getSource();
    rval += "\"src\":\""+src.escapeQuots()+"\"}";
  }

  rval += "]";
  return rval;
}

////////////////////////////////////////////////////////////////////
// Undo info

void Scene::commitUndoTxn()
{
  m_undomgr.commitTxn();

  // UndoInfo event
  {
    // Fire the SCE_OBJ_ADDED Event
    MB_DPRINTLN("UndoInfo> Firing SCE_SCENE_UNDOINFO event...");
    SceneEvent ev;
    ev.setType(SceneEvent::SCE_SCENE_UNDOINFO);
    ev.setSource(getUID());
    ev.setTarget(getUID());
    // ev.setTarget(robj->getUID());
    fireSceneEvent(ev);
  }
}

bool Scene::undo(int n)
{
  return m_undomgr.undo(n);
}

bool Scene::redo(int n)
{
  return m_undomgr.redo(n);
}

void Scene::clearUndoDataScr()
{
  if (m_undomgr.getUndoSize()==0&&
      m_undomgr.getRedoSize()==0)
    return; // --> already empty

  m_undomgr.clearAllInfo();

  // (notify changes of Undo/Redo information)
  {
    // Fire the SCE_SCENE_UNDOINFO event
    MB_DPRINTLN("UndoInfo> Firing SCE_SCENE_UNDOINFO event...");
    SceneEvent ev;
    ev.setType(SceneEvent::SCE_SCENE_UNDOINFO);
    ev.setSource(getUID());
    ev.setTarget(getUID());
    // ev.setTarget(robj->getUID());
    fireSceneEvent(ev);
  }
}

////////////////////////////////////////////////////////////////////
// Serialization

// #include <qlib/ObjStream.hpp>
#include <qlib/LDOM2Stream.hpp>

void Scene::writeTo2(qlib::LDom2Node *pNode) const
{
  // Write properties of the scene
  super_t::writeTo2(pNode);

  // Write reader options
  if (m_pQscOpts!=NULL) {
    qlib::LDom2Node *pChNode = MB_NEW qlib::LDom2Node(*m_pQscOpts);
    pNode->appendChild(pChNode);
  }

  // Write styles of the scene
  stylesWriteTo(pNode);

  // Write objects of the scene
  //data_t::const_iterator oiter = m_data.begin();
  //for (; oiter!=m_data.end(); ++oiter) {
  qlib::UIDList obj_uids;
  getAllObjectUIDs(obj_uids);
  BOOST_FOREACH (qlib::uid_t objid, obj_uids) {
    ObjectPtr obj = getObject(objid);
    qlib::LDom2Node *pChNode = pNode->appendChild();
    pChNode->setTagName("object");
    pChNode->setTypeNameByObj(obj.get());
    obj->writeTo2(pChNode);
  }

  // Write camera of the scene
  camerasWriteTo(pNode);

  // Write animation settings
  if (m_pAnimMgr->getSize()>0 ||
      m_pAnimMgr->getLength()>0) {
    qlib::LDom2Node *pChNode = pNode->appendChild();
    pChNode->setTagName("animation");
    m_pAnimMgr->writeTo2(pChNode);
  }

}

/// WriteTo2() impl for style settings
void Scene::stylesWriteTo(qlib::LDom2Node *pNode) const
{
  // StyleFile sfile;
  // sfile.saveToNode(pNode, getUID(), getBasePath());

  qlib::uid_t nScopeID = getUID();
  LString basedir = getBasePath();
  
  StyleList *pSL = m_pStyleMgr->getCreateStyleList(nScopeID);
  if (pSL==NULL || pSL->empty())
    return;

  // Iterate reversed order:
  //  The first node is higest priority, so is defined last in the file!!
  BOOST_REVERSE_FOREACH(StyleList::value_type pSet, *pSL) {

    // make child "styles" node
    qlib::LDom2Node *pChNode = pNode->appendChild();
    pChNode->setTagName("styles");

    // style set name (id)
    LString id = pSet->getName();
    LString src = pSet->getSource();

    if ( src.isEmpty() ) {
      // Internal style description
      pSet->writeToDataNode(pChNode);
    }
    else {
      // External style reference is serialized as external reference node.
      LString relpath = qlib::makeRelativePath(src, basedir);
      pChNode->appendStrAttr("src", relpath);
      if (pSet->isOverrideID()) {
        pChNode->appendStrAttr("id", id);
      }

      // Write style to the external file
      // TO DO: update external source, if the loaded style is modified.
      LString abspath;
      if (qlib::isAbsolutePath(src))
        abspath = src;
      else
        abspath = qlib::makeAbsolutePath(relpath, basedir);
      LOG_DPRINTLN("Scene> write external style file %s", abspath.c_str());
      qlib::uid_t nStyleSetID = pSet->getUID();
      bool bRes = m_pStyleMgr->saveStyleSetToFile(nScopeID, nStyleSetID, abspath);
      if (!bRes) {
        LOG_DPRINTLN("Scene> write external style file %s failed.", abspath.c_str());
      }
    }

  } // BOOST_FOREACH

}

/// WriteTo2() impl for camera settings
void Scene::camerasWriteTo(qlib::LDom2Node *pNode) const
{
  camtab_t::const_iterator viter = m_camtab.begin();
  for (; viter!=m_camtab.end(); ++viter) {
    CameraPtr obj = viter->second;
    LString src = obj->getSource();
    if (src.isEmpty()) {
      // embeded camera
      qlib::LDom2Node *pChNode = pNode->appendChild();
      pChNode->setTagName("camera");
      obj->writeTo2(pChNode);
    }
    else {
      // file-linked camera
      LString relpath = qlib::makeRelativePath(src, getBasePath());
      qlib::LDom2Node *pChNode = pNode->appendChild();
      pChNode->setTagName("camera");
      pChNode->appendStrAttr("src", relpath);

      // write to the linked file
      saveCameraTo(viter->first, src);
      LOG_DPRINTLN("Scene> camera %s is saved to external src %s",
                   viter->first.c_str(), src.c_str());
    }
  }
}

void Scene::objectReadFrom(qlib::LDom2Node *pNode)
{
  // create object by node's type name attr
  ObjectPtr robj = pNode->createObjByTypeNameT<Object>();

  // Object's properties should be built before registration to the scene,
  //   to prevent the event generation
  robj->readFrom2(pNode);

  // Register the built object
  registerObjectImpl(robj);

}

void Scene::cameraReadFrom(qlib::LDom2Node *pNode)
{
  CameraPtr rcam;
  
  LString cam_src = pNode->getStrAttr("src");
  // LString cam_name = pNode->getStrAttr("name");
  if (!cam_src.isEmpty()) {
    rcam = loadCameraImpl(cam_src);
    ensureNotNull(rcam);
  }
  else {
    rcam = CameraPtr(MB_NEW Camera);
  }
  
  // contents in pNode will overwrite the external source contents
  rcam->readFrom2(pNode);
  
  if (!pNode->isChildrenConsumed()) {
      // TO DO: report error (unknown element)
    //LOG_DPRINTLN("Scene::readFrom2> Warning: some Camera nodes are not consumed");
    //pNode->dump();
  }
  
  const LString &nm = rcam->m_name;
  setCamera(nm, rcam);
}

void Scene::readFrom2(qlib::LDom2Node *pNode)
{
  m_bLoading = true;

  try {

    // reader options
    qlib::LDom2Node *pRopts = pNode->findChild("qsc_opts");
    if (pRopts!=NULL) {
      qlib::LDom2Node *pRoptsCopy = MB_NEW qlib::LDom2Node(*pRopts);
      setQscOpts(pRoptsCopy);
    }

    // read other properties of scene
    super_t::readFrom2(pNode);

    StyleFile sfile;

    for (pNode->firstChild(); pNode->hasMoreChild(); pNode->nextChild()) {
      qlib::LDom2Node *pChNode = pNode->getCurChild();
      LString tag = pChNode->getTagName();
      LString type_name = pChNode->getTypeName();

      if (tag.equals("object") && !type_name.isEmpty()) {
        objectReadFrom(pChNode);
      }
      else if (tag.equals("camera")) {
        try {
          cameraReadFrom(pChNode);
        }
        catch (qlib::LException &e) {
          pNode->appendErrMsg("Scene> Load camera error (ignored).");
          pNode->appendErrMsg("Scene> Reason: %s", e.getMsg().c_str());
        }
        catch (...) {
          pNode->appendErrMsg("Scene> Load camera error (ignored).");
          pNode->appendErrMsg("Scene> Reason: unknown");
        }
      }
      else if (tag.equals("styles")) {
        LString style_src = pChNode->getStrAttr("src");
        LString style_id = pChNode->getStrAttr("id");
        if (!style_src.isEmpty()) {
          LString path = resolveBasePath(style_src);
          sfile.loadFile(path, getUID(), style_id);
        }
        else {
          // load local style nodes
          sfile.loadNodes(pChNode, getUID());
        }
      }
      else if (tag.equals("script")) {

        LString value = pChNode->getValue();
        LString contents = pChNode->getContents();
        if (value.isEmpty() && !contents.isEmpty())
          value = contents;

#ifndef NO_SCRIPT
        // TO DO: set file/lineno info for eval()
        // TO DO: save the original script for serialization
        if (m_pInterp!=NULL)
          m_pInterp->eval(value);
#endif
      }
      else if (tag.equals("animation")) {
        m_pAnimMgr->readFrom2(pChNode);
      }
      else {
        // Ignore unknown tags (for backward compatibility)
        continue;
      }

      pChNode->setConsumed(true);
    }

  }
  catch (...) {
    m_bLoading = false;
    throw;
  }

  m_bLoading = false;
}

bool Scene::execJSFile(const LString &scr)
{
#ifndef NO_SCRIPT
  LString path = resolveBasePath(scr);
  return m_pInterp->execFile(path);
#endif

  return false;
}

void Scene::setQscOpts(qlib::LDom2Node *ptree)
{
  if (m_pQscOpts!=NULL) {
    MB_DPRINTLN("Scene> Warning: previous reader options is deleted");
    delete m_pQscOpts;
  }

  m_pQscOpts = ptree;
}


