/*
 * Aipo is a groupware program developed by Aimluck,Inc.
 * Copyright (C) 2004-2011 Aimluck,Inc.
 * http://www.aipo.com
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

package com.aimluck.eip.addressbook;

import java.util.List;
import java.util.jar.Attributes;

import org.apache.cayenne.exp.Expression;
import org.apache.cayenne.exp.ExpressionFactory;
import org.apache.jetspeed.services.logging.JetspeedLogFactoryService;
import org.apache.jetspeed.services.logging.JetspeedLogger;
import org.apache.turbine.util.RunData;
import org.apache.velocity.context.Context;

import com.aimluck.eip.addressbook.beans.AddressBookGroupMemberListBean;
import com.aimluck.eip.addressbook.util.AddressBookUtils;
import com.aimluck.eip.cayenne.om.portlet.EipMAddressGroup;
import com.aimluck.eip.common.ALAbstractSelectData;
import com.aimluck.eip.common.ALDBErrorException;
import com.aimluck.eip.common.ALPageNotFoundException;
import com.aimluck.eip.modules.actions.common.ALAction;
import com.aimluck.eip.orm.Database;
import com.aimluck.eip.orm.query.ResultList;
import com.aimluck.eip.orm.query.SelectQuery;
import com.aimluck.eip.services.accessctl.ALAccessControlConstants;
import com.aimluck.eip.util.ALEipUtils;

/**
 * アドレス帳グループの検索データクラスです。
 * 
 */
public class AddressBookGroupSelectData extends
    ALAbstractSelectData<EipMAddressGroup, EipMAddressGroup> {

  /** logger */
  private static final JetspeedLogger logger =
    JetspeedLogFactoryService.getLogger(AddressBookGroupSelectData.class
      .getName());

  // add start
  /** ログインユーザーID */
  private int loginuserid;

  // add end

  /**
   * 初期処理
   * 
   * @param action
   *          アクションクラス
   * @param rundata
   *          JetSpeedランデータ
   * @param context
   *          JetSpeedコンテキスト
   * @throws ALPageNotFoundException
   * @throws ALDBErrorException
   */
  @Override
  public void init(ALAction action, RunData rundata, Context context)
      throws ALPageNotFoundException, ALDBErrorException {

    String sort = ALEipUtils.getTemp(rundata, context, LIST_SORT_STR);
    if (sort == null || sort.equals("")) {
      ALEipUtils.setTemp(rundata, context, LIST_SORT_STR, "group_name");

    }

    // add start
    loginuserid = ALEipUtils.getUserId(rundata);
    // add end
    context.put("loginuserid", loginuserid);
    super.init(action, rundata, context);
  }

  /**
   * 一覧データを取得します。
   * 
   * @param rundata
   *          JetSpeedランデータ
   * @param context
   *          JetSpeedコンテキスト
   * @return 社外グループ情報リスト
   */
  @Override
  protected ResultList<EipMAddressGroup> selectList(RunData rundata,
      Context context) {
    try {

      SelectQuery<EipMAddressGroup> query = getSelectQuery(rundata, context);
      buildSelectQueryForListView(query);
      buildSelectQueryForListViewSort(query, rundata, context);

      return query.getResultList();
    } catch (Exception ex) {
      // change start
      // logger.error("Exception", ex);
      String msg =
        "会社情報の一覧取得に失敗しました。["
          + ALEipUtils.getBaseUser(loginuserid).getUserName()
          + "]";
      logger.error(msg, ex);
      // change end
      return null;
    }
  }

  /**
   * 詳細データを取得します。
   * 
   * @param rundata
   *          JetSpeedランデータ
   * @param context
   *          JetSpeedコンテキスト
   * @return DBから取得した役職情報
   * @throws ALPageNotFoundException
   */
  @Override
  protected EipMAddressGroup selectDetail(RunData rundata, Context context)
      throws ALPageNotFoundException {
    // change start
    // return AddressBookUtils.getEipMAddressGroup(rundata, context);
    EipMAddressGroup group;
    try {
      group = AddressBookUtils.getEipMAddressGroup(rundata, context);
      if (group == null) {
        throw new Exception();
      }
    } catch (Exception e) {
      String msg =
        "会社情報の詳細取得に失敗しました。["
          + ALEipUtils.getBaseUser(loginuserid).getUserName()
          + "]";
      logger.error(msg, e);
      throw new ALPageNotFoundException();
    }
    return group;
    // change end
  }

  /**
   * ResultData に値を格納して返します。（一覧データ） <BR>
   * 
   * @param obj
   *          DBから取得し社外グループ情報
   * @return 画面表示用データ
   * @see com.aimluck.eip.common.ALAbstractSelectData#getListData(java.lang.Object)
   */
  @Override
  protected Object getResultData(EipMAddressGroup record) {
    AddressBookGroupResultData rd = new AddressBookGroupResultData();
    rd.initField();
    rd.setGroupId(record.getGroupId().longValue());
    rd.setGroupName(record.getGroupName());
    rd.setPublicFlag(record.getPublicFlag());
    // add start
    rd.setOwnerId(record.getOwnerId().longValue());
    // add end
    return rd;
  }

  /**
   * 詳細データを取得します。
   * 
   * @param obj
   * @return
   */
  @Override
  protected Object getResultDataDetail(EipMAddressGroup obj) {
    return getResultData(obj);
  }

  /**
   * 
   * @return
   */
  @Override
  protected Attributes getColumnMap() {
    Attributes map = new Attributes();
    map.putValue("group_name", EipMAddressGroup.GROUP_NAME_PROPERTY);
    return map;
  }

  /**
   * 検索条件を設定した SelectQuery を返します。
   * 
   * @param rundata
   * @param context
   * @return
   */
  private SelectQuery<EipMAddressGroup> getSelectQuery(RunData rundata,
      Context context) {
    SelectQuery<EipMAddressGroup> query =
      Database.query(EipMAddressGroup.class);

    Expression exp =
      ExpressionFactory.matchExp(EipMAddressGroup.OWNER_ID_PROPERTY, Integer
        .valueOf(ALEipUtils.getUserId(rundata)));
    query.setQualifier(exp);

    return buildSelectQueryForFilter(query, rundata, context);
  }

  /**
   * 社外グループに所属している個人アドレスを取得する。
   * 
   * @param gid
   * @return 「個人アドレス.姓 + " " 個人アドレス.名 + "(" + 個人アドレス.役職名 + ")"」のリスト
   * 
   */
  // change start
  // public List<String> getGroupMemberList(String gid) {
  public List<AddressBookGroupMemberListBean> getGroupMemberList(String gid) {
    // change end
    // change start
    // return AddressBookUtils.getGroupMember(gid);
    List<AddressBookGroupMemberListBean> list = null;
    try {
      list = AddressBookUtils.getGroupMember(gid);
    } catch (ALDBErrorException e) {
      String msg =
        "会社情報の一覧取得に失敗しました。["
          + ALEipUtils.getBaseUser(loginuserid).getUserName()
          + "]";
      logger.error(msg, e);
    }
    return list;
    // change end
  }

  /**
   * アクセス権限チェック用メソッド。 アクセス権限の機能名を返します。
   * 
   * @return
   */
  @Override
  public String getAclPortletFeature() {
    return ALAccessControlConstants.POERTLET_FEATURE_ADDRESSBOOK_COMPANY_GROUP;
  }
}
