/*
 * To change this template, choose Tools | Templates
 * and open the template in the editor.
 */

package jp.sourceforge.damstation_dl;

import java.io.IOException;
import java.io.Reader;
import java.text.ParseException;
import java.util.HashMap;
import java.util.Map;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import javax.swing.text.MutableAttributeSet;
import javax.swing.text.html.HTML;
import javax.swing.text.html.HTML.Tag;
import javax.swing.text.html.HTMLEditorKit.ParserCallback;
import javax.swing.text.html.parser.ParserDelegator;
import jp.sourceforge.damstation_dl.data.ResultData2;
import jp.sourceforge.damstation_dl.data.ResultDate;
import jp.sourceforge.damstation_dl.data.SongData;
import jp.sourceforge.damstation_dl.data.SongId;

/**
 *
 * @author ｄ
 */
public class Seimitsu2PageParser {

    /** 正常時のTD要素の出現回数 */
    private static final int TD_ELEMENT_COUNT = 25;
    /** 日付取得用のパターン */
    private static final Pattern DATE_PATTERN = Pattern.compile("\\d{4}/\\d{2}/\\d{2} \\d{2}:\\d{2}:\\d{2}");
    /** ファイル名(拡張子なし)取得用のパターン */
    private static final Pattern FILENAME_PATTERN = Pattern.compile("/(.)+?[.]gif$");
    /** ビブラート時間取得用のパターン */
    private static final Pattern VIBRATO_TIME_PATTERN = Pattern.compile("^[0-9.]+");
    /** 必要な情報の領域の始まりを見分けるためのクラス名 */
    private static final String SEARCH_CLASSNAME = new String("list_marking");
    
    private Map<SongId, SongData> songDataList = new HashMap<SongId, SongData>();;
    private Map<ResultDate, ResultData2> resultData2List = new HashMap<ResultDate, ResultData2>();
    
    /**
     * 精密データのパーサ
     */
    private class Seimitsu2PageParserCallback extends ParserCallback {
        
        /** matcher 正規表現の結果 */
        private Matcher matcher = null;
        /** img単独タグ検出回数 */
        private int imgCount = 0;
        /** TD開始タグ検出回数 */
        private int tdCount = 0;
        /** テキストノード出現回数(TD毎にリセット) */
        private int textCount = 0;
        /** 現在の読み込み位置が曲名かどうか */
        private boolean isTitle = false;
        /** 現在の読み込み位置が歌手名かどうか */
        private boolean isArtist = false;
        /** チェック中かどうか */
        private boolean isChecking = false;
        /** 1番目のテーブルの検査が終わったかどうか(テーブル2個で1曲分のデータ) */
        private boolean isChecked1stTable = false;
        
        /** 現在読み取り中のデータ(完了したらリストに追加する) */
        private String id;
        private String title;
        private String artist;
        private String date;
        private String score;
        private String vibratoType;
        private String vibratoTime;
        private String shakuri;
        private String kobushi;
        private String fall;
        private String interval;
        private String lowTone;
        private String highTone;
        private String vibratoSkill;
        private String longTone;
        private String modulation;
        private String rhythm;


        /**
         * パース完了時のコールバック関数
         * @param eol
         */
        @Override
        public void handleEndOfLineString(String eol) {
            super.handleEndOfLineString(eol);
            this.matcher = null;
        }

        /**
         * 終了タグ検出時のコールバック関数
         * @param t タグの種類
         * @param pos 開始位置
         */
        @Override
        public void handleEndTag(Tag t, int pos) {
            super.handleEndTag(t, pos);
            if (!this.isChecking) {
                return;
            }

            if (Tag.HTML.TABLE.equals(t)) {
                if (!this.isChecked1stTable) {
                    endCheck1stTable();
                } else {
                    endCheck2ndTable();
                }
            } else if (Tag.HTML.A.equals(t) && this.isTitle) {
                this.isTitle = false;
            } else if (Tag.HTML.P.equals(t) && this.isArtist) {
                this.isArtist = false;
            } else if (Tag.HTML.TD.equals(t)) {
                this.textCount = 0;
            }
        }

        /**
         * 単独タグ検出時のコールバック関数
         * @param t タグの種類
         * @param a 属性
         * @param pos 開始位置
         */
        @Override
        public void handleSimpleTag(Tag t, MutableAttributeSet a, int pos) {
            super.handleSimpleTag(t, a, pos);
            if (!this.isChecking) {
                return;
            }
            
            // 2個目のテーブルの時に検査
            if (isChecked1stTable) {
                if (Tag.HTML.IMG.equals(t)) {
                    this.imgCount++;

                    switch (this.imgCount) {
                        case 1:
                            // 低音
                            this.matcher = FILENAME_PATTERN.matcher((String) a.getAttribute(HTML.Attribute.SRC));
                            if (this.matcher.find() && (this.matcher.groupCount() >= 1)) {
                                this.lowTone = matcher.group(1);
                            }
                            break;
                        case 2:
                            // 高音
                            this.matcher = FILENAME_PATTERN.matcher((String) a.getAttribute(HTML.Attribute.SRC));
                            if (this.matcher.find() && (this.matcher.groupCount() >= 1)) {
                                this.highTone = matcher.group(1);
                            }
                            break;
                        case 3:
                            // ビブラートの上手さ
                            this.matcher = FILENAME_PATTERN.matcher((String) a.getAttribute(HTML.Attribute.SRC));
                            if (this.matcher.find() && (this.matcher.groupCount() >= 1)) {
                                this.vibratoSkill = matcher.group(1);
                            }
                            break;
                        case 4:
                            // ロングトーン
                            this.matcher = FILENAME_PATTERN.matcher((String) a.getAttribute(HTML.Attribute.SRC));
                            if (this.matcher.find() && (this.matcher.groupCount() >= 1)) {
                                this.longTone = matcher.group(1);
                            }
                            break;
                        case 5:
                            // 抑揚
                            this.matcher = FILENAME_PATTERN.matcher((String) a.getAttribute(HTML.Attribute.SRC));
                            if (this.matcher.find() && (this.matcher.groupCount() >= 1)) {
                                this.modulation = matcher.group(1);
                            }
                            break;
                        case 6:
                            // リズム
                            this.matcher = FILENAME_PATTERN.matcher((String) a.getAttribute(HTML.Attribute.SRC));
                            if (this.matcher.find() && (this.matcher.groupCount() >= 1)) {
                                this.rhythm = this.matcher.group(1);
                            }
                            break;
                        default:
                            break;
                    }
                }
            }
        }

        /**
         * 開始タグ検出時のコールバック関数
         * @param t タグの種類
         * @param a 属性
         * @param pos 開始位置
         */
        @Override
        public void handleStartTag(Tag t, MutableAttributeSet a, int pos) {
            super.handleStartTag(t, a, pos);
            if (Tag.HTML.TABLE.equals(t) && SEARCH_CLASSNAME.equals(a.getAttribute(HTML.Attribute.CLASS))) {
                if (!this.isChecked1stTable) {
                    startCheck1stTable();
                } else {
                    startCheck2ndTable();
                }
            }

            if (this.isChecking) {
                if (Tag.HTML.TD.equals(t)) {
                    this.tdCount++;
                } else if (Tag.HTML.A.equals(t) && (this.tdCount == 2)) {
                    this.isTitle = true;
                } else if (Tag.HTML.P.equals(t) && (this.tdCount == 2)) {
                    this.isArtist = true;
                }
            }
        }

        /**
         * テキストノード検出時のコールバック関数(前後の空白などは無視)
         * @param data テキストノードの文字列
         * @param pos 開始位置
         */
        @Override
        public void handleText(char[] data, int pos) {
            super.handleText(data, pos);
            if (!this.isChecking) {
                return;
            }
            
            this.textCount++;
            switch (this.tdCount) {
                case 1:
                    // 日時
                    this.matcher = DATE_PATTERN.matcher(new String(data));
                    if (this.matcher.find()) {
                        try {
                            this.date = ResultDate.damStationDateToXmlDateTime(this.matcher.group(0));
                        } catch(ParseException e) {
                            this.date = null;
                        }
                    }
                    break;
                case 2:
                    // 新曲の場合は曲名と歌手名が存在しないケースがある
                    // その場合に対処するため、isArtistとisTitleを使用している
                    if (this.isTitle) {
                        // 曲名
                        this.title = new String(data);
                    } else if (this.isArtist) {
                        // 歌手
                        this.artist = new String(data);
                    } else {
                        // ID
                        this.id = new String(data).replace("(", "").replace(")", "");
                    }
                    break;
                case 3:
                    if (this.textCount == 2) {
                        // 得点
                        this.score = new String(data).replace("点", "");
                    }
                    break;
                case 9:
                    switch (this.textCount) {
                        case 1:
                            // ビブラート時間
                            this.matcher = VIBRATO_TIME_PATTERN.matcher(new String(data));
                            if (this.matcher.find()) {
                                this.vibratoTime = this.matcher.group(0);
                            }
                            break;
                        case 2:
                            // ビブラートタイプ
                            this.vibratoType = new String(data);
                            break;
                    }
                    break;
                case 10:
                    // しゃくり
                    this.shakuri = new String(data).replace("回", "");
                    break;
                case 11:
                    // こぶし
                    this.kobushi = new String(data).replace("回", "");
                    break;
                case 12:
                    // フォール
                    this.fall = new String(data).replace("回", "");
                    break;
                case 13:
                    // 音程
                    this.interval = new String(data).replace("%", "");
                    break;
                default:
                    break;
            }
        }

        /**
         * 読み込み位置が必要な情報の領域に入ったときの処理
         */
        private void startCheck1stTable() {
            this.imgCount = 0;
            this.tdCount = 0;
            this.textCount = 0;
            this.isTitle = false;
            this.isArtist = false;
            this.isChecking = true;
            
            this.id = null;
            this.title = null;
            this.artist = null;
            this.date = null;
            this.score = null;
            this.vibratoType = null;
            this.vibratoTime = null;
            this.shakuri = null;
            this.kobushi = null;
            this.fall = null;
            this.interval = null;
            this.lowTone = null;
            this.highTone = null;
            this.vibratoSkill = null;
            this.longTone = null;
            this.modulation = null;
            this.rhythm = null;
        }
        
        private void startCheck2ndTable() {
            this.isChecking = true;
        }

        private void endCheck1stTable() {
            this.isChecked1stTable = true;
            this.isChecking = false;
        }
        
        /**
         * 読み込み位置が必要な情報の領域から抜けたときの処理
         */
        private void endCheck2ndTable() {
            /*
            // テストコード
            System.out.println("tdCount: " + tdCount);
            System.out.println("id: " + this.id);
            System.out.println("title: " + this.title);
            System.out.println("artist: " + this.artist);
            System.out.println("date: " + this.date);
            System.out.println("score: " + this.score);
            System.out.println("vibratoType: " + this.vibratoType);
            System.out.println("vibratoTime: " + this.vibratoTime);
            System.out.println("shakuri: " + this.shakuri);
            System.out.println("kobushi: " + this.kobushi);
            System.out.println("fall: " + this.fall);
            System.out.println("interval: " + this.interval);
            System.out.println("lowTone: " + this.lowTone);
            System.out.println("highTone: " + this.highTone);
            System.out.println("vibratoSkill: " + this.vibratoSkill);
            System.out.println("longTone: " + this.longTone);
            System.out.println("modulation: " + this.modulation);
            System.out.println("rhythm: " + this.rhythm);
             */
            
            this.isChecking = false;
            this.isChecked1stTable = false;
            
            // TD要素の数をチェックする
            if (this.tdCount == TD_ELEMENT_COUNT) {
                // NULLチェック
                if ((this.id == null) || (this.date == null) || (this.score == null)
                        || (this.vibratoType == null) || (this.vibratoTime == null)
                        || (this.shakuri == null) || (this.kobushi == null)
                        || (this.fall == null) || (this.interval == null)
                        || (this.lowTone == null) || (this.highTone == null)
                        || (this.vibratoSkill == null) || (this.longTone == null)
                        || (this.rhythm == null) || (this.modulation == null)) {
                    return;
                }
                
                // IDと日付の妥当性チェックを行う
                if (!SongId.isValid(this.id) || !ResultDate.isValid(this.date)) {
                    return;
                }
                
                SongId songId = SongId.getInstance(this.id);
                
                // 重複データでなければ曲データに追加するための処理を開始する
                if (!songDataList.containsKey(songId)) {
                    // 曲名かアーティストが取得できなかった場合(新曲の場合)は曲データに追加しない
                    if ((this.title != null) && (this.artist != null)) {
                        // データを追加する
                        songDataList.put(songId, new SongData(this.title, this.artist));
                    }
                }

                ResultDate date = ResultDate.getInstance(this.date);
                ResultData2 data = null;
                
                // 重複データでなければ結果データを追加するための処理を開始する
                if (!resultData2List.containsKey(date)) {
                    try {
                        // 文字列から数値に変換する
                        double score = Double.parseDouble(this.score);
                        int vibratoType = SeimitsuPageParser.parseVibratoTypeInteger(this.vibratoType);
                        double vibratoTime = Double.parseDouble(this.vibratoTime);
                        int shakuri = Integer.parseInt(this.shakuri);
                        int kobushi = Integer.parseInt(this.kobushi);
                        int fall = Integer.parseInt(this.fall);
                        int interval = Integer.parseInt(this.interval);
                        int lowTone = Integer.parseInt(this.lowTone);
                        int highTone = Integer.parseInt(this.highTone);
                        int vibratoSkill = Integer.parseInt(this.vibratoSkill);
                        int longTone = Integer.parseInt(this.longTone);
                        int modulation = Integer.parseInt(this.modulation);
                        int rhythm = Integer.parseInt(this.rhythm);

                        // 各値の妥当性チェックを行う
                        if (ResultData2.isValidScore(score) && ResultData2.isValidVibratoType(vibratoType) && ResultData2.isValidVibratoTime(vibratoTime) && ResultData2.isValidShakuri(shakuri) && ResultData2.isValidKobushi(kobushi) && ResultData2.isValidFall(fall) && ResultData2.isValidInterval(interval) && ResultData2.isValidLowTone(lowTone) && ResultData2.isValidHighTone(highTone) && ResultData2.isValidVibratoSkill(vibratoSkill) && ResultData2.isValidLongTone(longTone) && ResultData2.isValidRhythm(rhythm) && ResultData2.isValidModulation(modulation)) {
                            data = new ResultData2(songId, score, vibratoType, vibratoTime, shakuri, kobushi, fall, interval, lowTone, highTone, vibratoSkill, longTone, modulation, rhythm);
                            
                            // 結果データを追加する
                            resultData2List.put(date, data);
                        }
                    } catch (NumberFormatException e) {
                    } catch (ParseException e) {
                    }
                }
            }
        }
    }
    
    /**
     * A-1等のビブラートの種類を表す文字列を数値に変換する
     * @param vibratoType
     * @return
     * @throws java.text.ParseException
     */
    public static int parseVibratoTypeInteger(String vibratoType) throws ParseException {
        if ("無し".equals(vibratoType)) {
            return 0;
        } else if ("A-1".equals(vibratoType)) {
            return 1;
        } else if ("A-2".equals(vibratoType)) {
            return 2;
        } else if ("A-3".equals(vibratoType)) {
            return 3;
        } else if ("B-1".equals(vibratoType)) {
            return 4;
        } else if ("B-2".equals(vibratoType)) {
            return 5;
        } else if ("B-3".equals(vibratoType)) {
            return 6;
        } else if ("C-1".equals(vibratoType)) {
            return 7;
        } else if ("C-2".equals(vibratoType)) {
            return 8;
        } else if ("C-3".equals(vibratoType)) {
            return 9;
        } else {
            // 意図しない文字列の場合は例外を返す
            throw new ParseException("Seimitsu2PageParser.parseVibratoTypeInteger\n\tvibratoType=" + vibratoType, 0);
        }
    }
    
    /**
     * コンストラクタ
     * @param reader
     * @throws java.io.IOException
     */
    public Seimitsu2PageParser(Reader reader) throws IOException {
        ParserDelegator parserDelegator = new ParserDelegator();
        ParserCallback parserCallback = new Seimitsu2PageParserCallback();
        
        try {
            parserDelegator.parse(reader, parserCallback, true);
        } catch (IOException e) {
            throw e;
        }
    }
    
    /**
     * 解析した曲データリストを取得する
     * @return
     */
    public Map<SongId, SongData> getSongDataList() {
        return songDataList;
    }
    
    /**
     * 解析した結果データリストを取得する
     * @return
     */
    public Map<ResultDate, ResultData2> getResultData2List() {
        return resultData2List;
    }
}
