/*
 * To change this template, choose Tools | Templates
 * and open the template in the editor.
 */
package jp.sourceforge.damstation_dl;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.PrintWriter;
import java.net.InetSocketAddress;
import java.net.MalformedURLException;
import java.net.SocketTimeoutException;
import java.net.URL;
import java.net.URLConnection;
import java.util.Iterator;
import java.util.Map;
import java.util.Set;
import jp.sourceforge.damstation_dl.data.Proxy;

/**
 * URLからテキスト形式のファイルをダウンロードする
 * @author d
 */
public class TextDownloader {
    
    private final int connectTimeout;
    private final int readTimeout;
    private final java.net.Proxy proxy;
    
    /**
     * コンストラクタ
     * @param connectTimeout
     * @param readTimeout
     * @param proxy
     */
    public TextDownloader(int connectTimeout, int readTimeout, Proxy proxy) {
        if((connectTimeout < 0) || (readTimeout < 0)) {
            throw new IllegalArgumentException("TextDownloader.TextDownloader\n\tconnectTimeout: " + connectTimeout + "\n\treadTimeout: " + readTimeout);
        }
        
        this.connectTimeout = connectTimeout;
        this.readTimeout = readTimeout;
        
        if (proxy == null) {
            this.proxy = java.net.Proxy.NO_PROXY;
        } else {
            this.proxy = new java.net.Proxy(java.net.Proxy.Type.HTTP, new InetSocketAddress(proxy.getHost(), proxy.getPort()));
        }
    }
    
    /**
     * ファイルをダウンロードする(POST変数無し)
     * @param urlString URL文字列(GET変数付きでも可能)
     * @throws NullPointerException
     * @throws IOException     */
    public String download(String urlString) throws IOException {
        return download(urlString, null);
    }

    
    /**
     * ファイルをダウンロードする(POST変数有り)
     * @param urlString URL文字列(GET変数付きでも可能)
     * @param postValues POSTで送信するキーと値の組み合わせ
     * @throws NullPointerException
     * @return ダウンロードしたファイル
     */
    public String download(String urlString, Map<String, String> postValues) throws IOException {
        // null チェック
        if (urlString == null) {
            throw new NullPointerException("TextDownloader.download");
        }
        
        // コネクション作成
        URLConnection conn;
        try {
            conn = new URL(urlString).openConnection(this.proxy);
        } catch(MalformedURLException e) {

            throw new IOException("TextDownloader.download\n\turlString=" + urlString + "\n\te=" + e.toString());
        } catch (IOException e) {
            throw new IOException("TextDownloader.download\n\turlString=" + urlString + "\n\te=" + e.toString());
        }

        // コネクションの設定
        conn.setDoInput(true);
        conn.setDoOutput(true);
        conn.setUseCaches(false);
        conn.setConnectTimeout(this.connectTimeout);
        conn.setReadTimeout(this.readTimeout);

        // POSTパラメータの設定
        if (postValues != null) {
            PrintWriter pw = null;
            try {
                pw = new PrintWriter(conn.getOutputStream());
                pw.print(postValuesToString(postValues));
            } catch (IOException e) {
                throw new IOException("TextDownloader.download\n\turlString=" + urlString + "\n\tpostValues=" + postValues.toString() + "\n\te=" + e.toString());
            } finally {
                if(pw != null) {
                    pw.close();
                }
            }
        }
        
        // 接続
        try {
            conn.connect();
        } catch(SocketTimeoutException e) {
            // 接続タイムアウト
            throw new IOException("TextDownloader.download\n\te=" + e.toString());
        } catch (IOException e) {
            // 接続のオープン中に入出力エラーが発生
            throw e;
        }

        // ダウンロードして、ダウンロードしたファイルを返す
        try {
            return inputStreamToString(conn.getInputStream());
        } catch (IOException e) {
            // ダウンロード失敗
            throw e;
        }
    }

    /**
     * InputStreamをStringに変換する
     * @param stream InputStreamインスタンス
     * @return Stringインスタンス
     * @throws IOException
     */
    private static String inputStreamToString(InputStream stream) throws IOException {
        if (stream == null) {
            throw new NullPointerException("TextDownloader.inputStreamToString");
        }

        BufferedReader reader = null;
        StringBuffer text = new StringBuffer();
        try {
            // 1行ずつ読み出してhtmlの後ろに追加していく
            reader = new BufferedReader(new InputStreamReader(stream));
            String line;
            while ((line = reader.readLine()) != null) {
                text.append(line);
            }
        } catch (IOException e) {
            throw e;
        } finally {
            if (reader != null) {
                try {
                    reader.close();
                } catch (IOException e) {
                }
            }
        }

        return text.toString();
    }

    /**
     * POSTで送信するキーと値の組み合わせを文字列に変換する
     * @param postValues
     * @return
     */
    public static String postValuesToString(Map<String, String> postValues) {
        if(postValues == null) {
            throw new NullPointerException("TextDonwloader.postValuesToString");
        }
        
        StringBuffer postValuesString = new StringBuffer();
        Set<String> keys = postValues.keySet();
        Iterator<String> iterator = keys.iterator();
        String key;
        boolean isNotFirst = false;
        while (iterator.hasNext()) {
            if (isNotFirst) {
                postValuesString.append("&");
            } else {
                isNotFirst = true;
            }
            key = iterator.next();
            
            if ((key == null) || (postValues.get(key) == null)) {
                throw new NullPointerException("TextDonwloader.postValuesToString\n\tkey: " + key + "\n\tvalue: " + postValues.get(key));
            }
            else if((key.isEmpty()) || postValues.get(key).isEmpty()) {
                throw new IllegalArgumentException("TextDonwloader.postValuesToString\n\tkey: " + key + "\n\tvalue: " + postValues.get(key));
            }
            
            postValuesString.append(key + "=" + postValues.get(key));
        }

        return postValuesString.toString();
    }
}
